/* This is a re-write of Alexander G. M. Smith's AGMSDeviceTest program.
   Basicly this is a stripped down version of his AGMSDeviceTest program with only the IOCTL functions that are needed to control
   the Parallel Port Sound Drivers I have written.  In the past I wrote diffirent versions of the drivers for each option I wanted
   to test.  However this caused problems in keeping track of diffirent versions, and when I found a global improvement to the code,
   it was a problem updating all the diffirent sources.

   With this program, the timing options can be set by sending IOCTL commands to the driver. */

/* BeOS headers. */
#include <Application.h>
#include <Alert.h>
#include <Bitmap.h>
#include <Button.h>
#include <CheckBox.h>
#include <Drivers.h>
#include <FilePanel.h>
#include <MenuBar.h>
#include <MenuItem.h>
#include <Path.h>
#include <PopUpMenu.h>
#include <RadioButton.h>
#include <ScrollView.h>
#include <String.h>
#include <TextControl.h>
#include <View.h>
#include <Window.h>

/* Posix headers. */
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

/* User Header. */
#include "audio_user_IOCTL_defines.h" //

/******************************************************************************
 * Some globals.  These store the settings that the various suboptions select
 * so that the final complete device command can be given, without having to
 * pick through a hierarchy of objects to get the portions of the command. */

/* Data storage for DeviceChoiceView. */
static BString g_DeviceName = "/dev/audio/old/sound";
static bool g_OpenWithWriteEnabled = false;
static bool g_CreateRatherThanOpen = false;
static int g_DeviceFileDescriptor = -1; /* Non-negative when open. */

/* Data storage for ReadWriteIoctlChoiceView. */
static enum ReadWriteIOCtlModeEnum
  {RWIO_READ = 0, RWIO_WRITE, RWIO_IOCTL, RWIO_MAX}
  g_ReadWriteIOCtlMode = RWIO_IOCTL; /* Operation mode to use. */

/* Data storage for ReadAndWriteView. */
static signed long g_DesiredDataBufferSize; /* Amount user wants to R/W. */
static signed long g_DataBufferSize = 0; /* Current size of the buffer. */
static char *g_DataBufferPntr = NULL; /* NULL if not allocated. */
static off_t g_SeekPosition = 0; /* A 64 bit file position under BeOS. */

/* Data storage for IoctlOperationView. */
static long g_IOCtlOperationNumber = USER_PLAIN_AUDIO; /* IO Control operation. */
static int g_ErrorCodeFromLastOperation = B_OK;

/* Data storage for the various suboptions of the IOCtl feature. */
static void *g_IOCtlDataPntr; /* Points to current IOCtlData global or NULL. */
static size_t g_IOCtlDataSize; /* Number of bytes of data for that IOCtl op. */

static bool g_IOCtlDataBool;
static uint8 g_IOCtlDataByte;
static size_t g_IOCtlDataSize_t;
static status_t g_IOCtlDataStatus_t;
static device_geometry g_IOCtlDataDevice_geometry;
static driver_path g_IOCtlDataPathname;
static partition_info g_IOCtlDataPartition_info;
static device_icon g_IOCtlDataDevice_icon = {32, NULL};
static BBitmap *g_IOCtlDataDevice_iconBitmapPntr;
static open_device_iterator g_IOCtlDataOpen_device_iterator;

/* Miscellaneous globals. */
static float g_MarginBetweenControls; /* Space of a letter "M" between them. */
static float g_LineOfTextHeight;      /* Height of text the current font. */
static float g_ButtonHeight;          /* How many pixels tall buttons are. */
static float g_CheckBoxHeight;        /* Same for check boxes. */
static float g_RadioButtonHeight;     /* Also for radio buttons. */
static float g_PopUpMenuHeight;       /* Again for pop-up menus. */
static float g_TextBoxHeight;         /* Ditto for text controls. */

/* Global string tables. */
typedef struct CodeDescriptionStruct {
  const long  code;
  const char *name;
  const int   sizeOfData;
  const char *description; } CodeDescriptionRecord;

static CodeDescriptionRecord CodeDescriptions [] = {

  { USER_PLAIN_AUDIO, "PLAIN AUDIO", 0, "Outputs audio sample data directly to the DAC.  Data is unprocessed."},

  { USER_INTERPOLATOR_x2, "2x AUDIO", 0, "Interpolates the audio data x2 times.  Data is outputted two(2) times faster to the sound port but "
  "the value for the extra time slot is pre-calculated first."},

  { USER_INTERPOLATOR_x3, "3x AUDIO", 0, "Interpolates the audio data x2 times.  Data is outputted three(3) times faster to the sound port "
  "but the value for the extra time slot is pre-calculated first."},

  { USER_INTERPOLATOR_x4, "4x AUDIO", 0, "Interpolates the audio data x4 times.  Data is outputted four(4) times faster to the sound port "
  "but the values for extra time slots are pre-calculated first."},

  { USER_NULL, "", 0, "" },
  
  { USER_INTERRUPTS_YES, "ENABLE INTERRUPTS", 0, "Enables system interrupts during play-back of sound samples.  This allows the OS threads "
  "to operate normally and thus improves the general OS enviroment, but some noise will be caused by the interrupts occurring just as an "
  "audio sample is going to be outputted.  This is less of a problem when using snooze_until() timing as many of the interrupts will get "
  "serviced while the Snooze_Until() function is sleeping."},

  { USER_INTERRUPTS_NO, "DISABLE_INTERRUPTS", 0, "Disables system interrupts during play-back of sound.  This should improve sound quality "
  "by preventing interrupts occurring (try moving your mouse while playing a sample) just as an audio sample is going to be outputted.  "
  "This does not seem to help much if you are using SNOOZE_TIMING or SKIP_DUPLICATES timing."},

  { USER_NULL, "", 0, "" },

  { USER_BUSY_WAIT_TIMING, "BUSY_WAIT_TIMING", 0, "Uses a very tight busy wait loop to control the timing of audio sample outputs.  This "
  "results in a heavy CPU load from the driver but is very simple to code (debug).  Because system interrupts can interfer with the timing, "
  "turning off the system interrupts will improve the sound quality, but the machine may become very unreponsive to the user while playing "
  "out sounds."},

  { USER_SNOOZE_UNTIL_TIMING, "SNOOZE_UNTIL_TIMING", 0, "Uses Snooze_Until() function to control timing of sample outputs.  Since this means "
  "the driver is sleeping most of the time, it helps to reduce the driver's CPU load on the OS by a large factor."},

  { USER_SKIP_DUP_TIMING, "SKIP_DUPLICATES_TIMING", 0, "Uses variable length Snooze_Until() function to control timing but the timing is "
  "also adjust to skip the outputting of duplicates audio sample outputs.  This allows the driver to be in the sleep mode more often and "
  "reduces the driver's CPU LOAD on the OS the most of any options."},

  { USER_LATENCY_TIMING, "SNOOZE_LATENCY_TIMING", 0, "Uses variable length Snooze_Until() function to control timing but the timing is also "
  "adjust to skip the outputting of duplicates audio sample outputs.  This allows the driver to be in the sleep mode more often and reduces "
  "the driver's CPU LOAD on the OS the most of any options."},

  { USER_FAST_PERIODIC_TIMER, "FAST_PERIODIC_TIMING", 0, "The BeOS's Add_Timer() function is used to control the timing of audio sample "
  "outputs.  The Add_Timer() function is initialized with a B_PERIODIC_TIMER flag.  Thus it sets the BeOS Add_Timer() function to call the "
  "driver's interrupt handler at regular time periods.  The handler is the simplest code possible, it just outputs the next valid "
  "ring_buffer sound sample byte to the parallel port."},

  { USER_FILTER_PERIODIC_TIMER, "FILTER_PERIODIC_TIMER", 0, "The BeOS's Add_Timer() function is used to control the timing of audio "
  "sample outputs.  The Add_Timer() function is initialized with a B_PERIODIC_TIMER flag.  Thus it sets the BeOS Add_Timer() function to "
  "call the driver's interrupt handler at regular time periods.  The handler checks if there is anything in the ring_buffer to output, it "
  "then checks to make sure that it is not already the same value that is presently on the output port before sending it.  "
  "This saves I/O time."},

  { USER_TIMED_PERIODIC_TIMER, "TIMED_PERIODIC_TIMER", 0, "The BeOS's Add_Timer() function is used to control the timing of audio sample "
  "outputs.  The Add_Timer() function is initialized with a B_PERIODIC_TIMER flag.  Thus it sets the BeOS Add_Timer() function to call the "
  "driver's interrupt handler at regular time periods.  The handler checks if there is anything in the ring_buffer to output, it then checks "
  "to make sure that it is not already the same value that is presently on the output port before sending it.  This saves I/O time.  If "
  "there is valid data to output the handler then goes into a busy wait loop to control the timing of the output."},

  { USER_NULL, "", 0, "" },
  
  { USER_LATENCY_00, "Set Latency to 0", 0, "Latency is set to 0 Mircoseconds.  This means when the Snooze_Until() function is called, it "
  "is set to end at the time that the next sound sample needs to be outputed.  This puts the mininal CPU load onto the system from the "
  "driver.  However, if the system is already heavily loaded then the Snooze_Until() function may end up exiting late causing some phase "
  "shift noise in audio output from the driver.  Note: This choice has no effect if you has selected BUSY_WAIT_TIMING as your main timing "
  "function of the driver."},

  { USER_LATENCY_05, "Set Latency to 5", 0, "Latency is set to 5 Mircoseconds.  The Snooze_Until() function is set to end 5 microseconds "
  "early, then the driver will enter a tight busy wait loop to time the output of the audio sample.  This is a larger CPU load, but better "
  "timing of the audio samples is gained.  On a heavily loaded system the loop can still be interrupted if system interrupts are still "
  "enabled.  But as interrupts tend to get serviced while the Snooze_Until() function sleeps there are few problems.  Note: This choice has "
  "no effect if you have the driver using BUSY_WAIT_TIMING."},

  { USER_LATENCY_10, "Set Latency to 10", 0, "Latency is set to 10 Mircoseconds.  The Snooze_Until() function is set to end 10 microseconds "
  "early, then the driver will enter a tight busy wait loop to time the output of the audio sample.  This is a larger CPU load, but better "
  "timing of the audio samples is gained.  On a heavily loaded system the loop can still be interrupted if system interrupts are still "
  "enabled.  But as interrupts tend to get serviced while the Snooze_Until() function sleeps there are few problems.  Note: This choice has "
  "no effect if you have the driver using BUSY_WAIT_TIMING."},

  { USER_LATENCY_20, "Set Latency to 20", 0, "Latency is set to 20 Mircoseconds.  The Snooze_Until() function is set to end 20 microseconds "
  "early, then the driver will enter a tight busy wait loop to time the output of the audio sample.  This is a larger CPU load, but better "
  "timing of the audio samples is gained.  On a heavily loaded system the loop can still be interrupted if system interrupts are still "
  "enabled.  But as interrupts tend to get serviced while the Snooze_Until() function sleeps the are few problems. Note: this choice has no "
  "effect if you have the driver using BUSY_WAIT_TIMING."},

  { USER_NULL, "", 0, "" },
  
  { USER_DEBUG_OFF, "User Debug Info Off", 0, "USER_DEBUG_OFF.  Disables dprint() of debugging info."},

  { USER_DEBUG_ON, "User Debug Info On", 0, "USER_DEBUG_ON.  Enables dprint() of debugging info for details of driver operation."},

  { SOUND_DEBUG_OFF, "Sound Debug Info Off", 0, "USER_DEBUG_OFF.  Disables dprint() of debugging info about sound output."},

  { SOUND_DEBUG_ON, "Sound Debug Info On", 0, "USER_DEBUG_ON.  Enables dprint() of debugging info for details of sound output."},

  { USER_NULL, "", 0, "" },

  { B_DEVICE_OP_CODES_END, "B_DEVICE_OP_CODES_END", 0,
    "User defined control codes follow this one."}
}; // End of CodeDescriptions array initialisation.

static const char * ErrorDescriptions [] =
{ /* status_t values.  Same order as in the Errors.h Device Kit Errors enum. */
  "B_DEV_INVALID_IOCTL",
  "B_DEV_NO_MEMORY",
  "B_DEV_BAD_DRIVE_NUM",
  "B_DEV_NO_MEDIA",
  "B_DEV_UNREADABLE",
  "B_DEV_FORMAT_ERROR",
  "B_DEV_TIMEOUT",
  "B_DEV_RECALIBRATE_ERROR",
  "B_DEV_SEEK_ERROR",
  "B_DEV_ID_ERROR",
  "B_DEV_READ_ERROR",
  "B_DEV_WRITE_ERROR",
  "B_DEV_NOT_READY",
  "B_DEV_MEDIA_CHANGED",
  "B_DEV_MEDIA_CHANGE_REQUESTED",
  "B_DEV_RESOURCE_CONFLICT",
  "B_DEV_CONFIGURATION_ERROR",
  "B_DEV_DISABLED_BY_USER",
  "B_DEV_DOOR_OPEN" };

static const char * DeviceTypeDescriptions [] = { /* Same order as in the Drivers.h enum. */
  "B_DISK",
  "B_TAPE",
  "B_PRINTER",
  "B_CPU",
  "B_WORM",
  "B_CD",
  "B_SCANNER",
  "B_OPTICAL",
  "B_JUKEBOX",
  "B_NETWORK" };

#if 0 /* Unused code. */
/******************************************************************************
 * Global utility function to convert a 32 bit number into a 4 character
 * string so that you can see printable message codes etc.
 */

static const char *U32toString (uint32 Number)
{
  union ConverterUnion
  {
    uint32 number;
    char string [4];
  } Converter;

  static char OutputBuffer [5];

  Converter.number = Number;

  OutputBuffer[0] = Converter.string[3];
  OutputBuffer[1] = Converter.string[2];
  OutputBuffer[2] = Converter.string[1];
  OutputBuffer[3] = Converter.string[0];
  OutputBuffer[4] = 0;

  return OutputBuffer;
}
#endif /* Unused code. */



/******************************************************************************
 * Global utility function to display an error message and return.  The message
 * part describes the error, and if ErrorNumber is non-zero, gets the string
 * ", error code $X (standard description)." appended to it.  If the message
 * is NULL then it gets defaulted to "Something went wrong".  The title part
 * doesn't get displayed (no title bar in the dialog box, but you can see it
 * in the debugger as the window thread name), and defaults to "Error Message"
 * if you didn't specify one.
 */

static void DisplayErrorMessage (
  const char *MessageString = NULL,
  int ErrorNumber = 0,
  const char *TitleString = NULL)
{
  BAlert *AlertPntr;
  char ErrorBuffer [B_PATH_NAME_LENGTH + 80 /* error message */ + 80];

  if (TitleString == NULL)
    TitleString = "Error Message";

  if (MessageString == NULL)
  {
    if (ErrorNumber == 0)
      MessageString = "No error, no message, why bother?";
    else
      MessageString = "Something went wrong";
  }

  if (ErrorNumber != 0)
  {
    sprintf (ErrorBuffer, "%s, error code $%X/%d (%s) has occured.",
      MessageString, ErrorNumber, ErrorNumber, strerror (ErrorNumber));
    MessageString = ErrorBuffer;
  }

  AlertPntr = new BAlert (TitleString, MessageString,
    "Acknowledge", NULL, NULL, B_WIDTH_AS_USUAL, B_STOP_ALERT);
  if (AlertPntr != NULL)
    AlertPntr->Go ();
}



/******************************************************************************
 * Global utility function to allocate memory for the global data buffer.
 * If the size doesn't match the desired size, it will free it and allocate
 * the correct size, plus one extra for a NUL character past the end.
 * The new buffer is initialised with printable characters so that the user
 * can edit it in text mode.  If it runs out of memory then g_DataBufferPntr
 * will be NULL (and you get a pop-up message).
 */

static void AllocateGlobalDataBuffer (void)
{
  if (g_DesiredDataBufferSize != g_DataBufferSize)
  {
    free (g_DataBufferPntr); /* Safe to free NULL pointers. */
    g_DataBufferPntr = NULL;
    g_DataBufferSize = 0;
  }

  if (g_DataBufferPntr == NULL && g_DesiredDataBufferSize > 0)
  {
    g_DataBufferPntr = (char *) malloc (g_DesiredDataBufferSize +
      1 /* space for the NUL at the end of a string that GetText() adds */);

    if (g_DataBufferPntr == NULL)
      DisplayErrorMessage ("Unable to allocate data buffer", errno);
    else
    {
      g_DataBufferSize = g_DesiredDataBufferSize;
      memset (g_DataBufferPntr, '.', g_DataBufferSize);
      g_DataBufferPntr [g_DataBufferSize] = 0; /* for NUL past end of buffer */
    }
  }
}



/******************************************************************************
 * Global utility function to free the memory from the global data buffer.
 */

static void FreeGlobalDataBuffer (void)
{
  free (g_DataBufferPntr); /* Safe to free NULL pointers. */
  g_DataBufferPntr = NULL;
  g_DataBufferSize = 0;
}



/******************************************************************************
 * This is an abstract base class for the various suboption views.  Each
 * option has a few child views for controls arranged at the top and an empty
 * space under the controls down to the bottom of the window for more suboption
 * views to be nested into.  The main feature is that this view can add and
 * remove its children on command, either controls and suboptions or just
 * the suboptions.  It can also be resized.
 */

class GenericOptionsView : public BView
{
public:
  GenericOptionsView (BRect FrameSize, const char *ViewName);

  virtual void AddControls () = 0; /* Things like buttons and text boxes. */
  virtual void AddSubOptions () = 0; /* Adds relevant suboption view. */
  virtual void CopyFromGlobalToDisplay () = 0; /* Updates from global vars. */
  virtual void CopyFromDisplayToGlobal () = 0; /* Copies inputs to globals. */

  void RemoveControls ();
  void RemoveSubOptions ();
  void RemoveSubOptionsAndControls ();

  /* Pointers to the various subviews.  If they are non-NULL then they exist
  and have been added as children of this view. */

  static const unsigned int MAX_CONTROLS_PER_OPTION = 10;
  BView *m_ControlViews[MAX_CONTROLS_PER_OPTION];
  GenericOptionsView *m_SubOptionsView;

  float m_HeightOfControlsArea; /* Used when adding subviews under controls. */

  /* Various message codes used by buttons etc. */

  typedef enum MessageCodesEnum
  {
    DLG_DEVICE_NAME_EDIT = 'EdDv',
    DLG_OPEN_CLOSE_BUTTON = 'OpCl',
    DLG_BROWSE_DEVICES_BUTTON = 'BrDv',
    DLG_WRITE_CHECK_BOX = 'WrBx',
    DLG_CREATE_CHECK_BOX = 'CrBx',
    DLG_RWIOCTL_READ = 'RWIR',
    DLG_RWIOCTL_WRITE = 'RWIW',
    DLG_RWIOCTL_CONTROL = 'RWIC',
    DLG_SEEK_POSITION = 'Seek',
    DLG_BUFFER_SIZE = 'BuSz',
    DLG_DO_READWRITE = 'DoRW',
    DLG_OPCODE_TEXT = 'OpCd',
    DLG_IOCTL_EXECUTE = 'IOEx',
    DLG_IOCTL_BOOL = 'IBol',
    DLG_IOCTL_BYTE = 'IByt',
    DLG_IOCTL_SIZET = 'ISzT',
    DLG_IOCTL_STATUST = 'IStT',
    DLG_IOCTL_GEOMETRY = 'IGeo',
    DLG_IOCTL_PATHNAME = 'IPth',
    DLG_IOCTL_PARTITION = 'IPtn',
    DLG_IOCTL_ICON = 'IIcn',
    DLG_IOCTL_DEVICE_ITERATOR = 'IDIt',
    DLG_MAX
  } MessageCodes;

private: /* Disable the default copy and assignment operators. */
  GenericOptionsView (const GenericOptionsView&);
  GenericOptionsView& operator= (const GenericOptionsView&);
};


GenericOptionsView::GenericOptionsView (
  BRect FrameSize,
  const char *ViewName)
: BView (FrameSize, ViewName,
  B_FOLLOW_LEFT_RIGHT | B_FOLLOW_TOP_BOTTOM,
  B_WILL_DRAW | B_FRAME_EVENTS | B_FULL_UPDATE_ON_RESIZE | B_NAVIGABLE_JUMP),
  m_SubOptionsView (NULL),
  m_HeightOfControlsArea (0)
{
  unsigned int i;

  for (i = 0; i < MAX_CONTROLS_PER_OPTION; i++)
    m_ControlViews[i] = NULL;
}


void GenericOptionsView::RemoveControls ()
{
  unsigned int i;

  for (i = 0; i < MAX_CONTROLS_PER_OPTION; i++)
  {
    if (m_ControlViews[i] != NULL)
    {
      RemoveChild (m_ControlViews[i]);
      delete m_ControlViews[i];
      m_ControlViews[i] = NULL;
    }
  }
}


void GenericOptionsView::RemoveSubOptions ()
{
  if (m_SubOptionsView != NULL)
  {
    RemoveChild (m_SubOptionsView);
    delete m_SubOptionsView; /* Also automagically deletes its children. */
    m_SubOptionsView = NULL;
  }
}


void GenericOptionsView::RemoveSubOptionsAndControls ()
{
  RemoveSubOptions ();
  RemoveControls ();
}



/******************************************************************************
 * This suboption displays the buffer as a text view with scroll bars.
 * You can also transfer data in both directions between the global data
 * buffer and this view, if you need to update the buffer or displayed text.
 */

class PlainTextView : public GenericOptionsView
{
public:
  PlainTextView (BRect FrameSize);

  void FrameResized (float width, float height); /* Override the default. */

  void AddControls ();
  void AddSubOptions () {/* no suboptions */};
  void CopyFromGlobalToDisplay ();
  void CopyFromDisplayToGlobal ();

  typedef enum ControlsEnum /* Indices into m_ControlViews for this view. */
  {
    SCROLL_BAR_VIEW_INDEX = 0,
    TEXT_VIEW_INDEX
  } ControlIndices;
};


PlainTextView::PlainTextView (
  BRect FrameSize)
: GenericOptionsView (FrameSize, "PlainTextView")
{
  SetViewColor (255, 255, 80);
}


/* Override the virtual function so that the text view's text rectangle also
gets resized so that it wraps the text to the new window size. */

void PlainTextView::FrameResized (float width, float height)
{
  BRect        TempRect;
  BTextView   *TextPntr;

  /* Call the parent FrameResized, in case it actually does something. */

  GenericOptionsView::FrameResized (width, height);

  /* Tell the text view about the new margin size. */

  TextPntr = (BTextView *) m_ControlViews[TEXT_VIEW_INDEX];
  if (TextPntr != NULL)
  {
    TempRect = Bounds ();
    TempRect.right -= B_V_SCROLL_BAR_WIDTH;
    if (TempRect.right - TempRect.left < 20)
      TempRect.right = TempRect.left + 20; // Avoid crashing with small widths.
    TextPntr->SetTextRect (TempRect);
  }
}


/* The plain text view just has a BScrollView and BTextView as regular
subviews, and no suboptions.  The buffer is copied from g_DataBufferPntr to
the BTextView when this view is created. */

void PlainTextView::AddControls ()
{
  BScrollView *ScrollPntr;
  BRect        TempRect;
  BTextView   *TextPntr;

  RemoveControls ();

  TempRect = Bounds ();
  TempRect.right -= B_V_SCROLL_BAR_WIDTH; /* Need to leave space for bar. */

  TextPntr = new BTextView (TempRect,
    "PlainTextView - Text subview",
    TempRect,
    B_FOLLOW_ALL_SIDES,
    B_WILL_DRAW | B_FRAME_EVENTS | B_FULL_UPDATE_ON_RESIZE | B_NAVIGABLE);
  TextPntr->SetViewColor (200, 255, 150);
  m_ControlViews[TEXT_VIEW_INDEX] = TextPntr;

  ScrollPntr = new BScrollView (
    "PlainTextView - Scroll subview",
    TextPntr,
    B_FOLLOW_ALL_SIDES,
    0 /* flags */,
    false /* horizontal scroll bar */,
    true /* vertical scroll bar */,
    B_PLAIN_BORDER);
  m_ControlViews[SCROLL_BAR_VIEW_INDEX] = ScrollPntr;

  AddChild (ScrollPntr);

  if (TextPntr != NULL)
  {
    TextPntr->SetWordWrap (true);
    CopyFromGlobalToDisplay (); /* Initialise with our data buffer. */
  }
}


/* Like the name says, copy the data from g_DataBufferPntr into the text. */

void PlainTextView::CopyFromGlobalToDisplay (void)
{
  BTextView *TextPntr;

  TextPntr = (BTextView *) m_ControlViews[TEXT_VIEW_INDEX];

  if (TextPntr != NULL)
  {
    if (g_DataBufferPntr != NULL && g_DataBufferSize > 0)
      TextPntr->SetText (g_DataBufferPntr, g_DataBufferSize);
    else
      TextPntr->SetText ("");
  }
}


/* Copy the data in the text view back into our global data buffer.  It doesn't
resize the buffer to match the amount of text; extra text is ignored. */

void PlainTextView::CopyFromDisplayToGlobal (void)
{
  BTextView *TextPntr;

  TextPntr = (BTextView *) m_ControlViews[TEXT_VIEW_INDEX];

  /* Note that GetText adds a NUL at the end of the data, so our memory
  buffer was allocated 1 byte longer to leave space for it. */

  if (TextPntr != NULL && g_DataBufferPntr != NULL && g_DataBufferSize > 0)
  {
    TextPntr->GetText (0 /* offset */,
      g_DataBufferSize /* amount */,
      g_DataBufferPntr /* buffer */);
  }
}



/******************************************************************************
 * This suboption chooses the amount to read or write and the file position,
 * and does the operation.  There is a text box for the file position (a 64
 * bit number), a text box for the amount, a read only display of the
 * buffer address, and finally a read or write action button.
 */

class ReadAndWriteView : public GenericOptionsView
{
public:
  ReadAndWriteView (BRect FrameSize, const char *ViewName,
    bool WriteMode = false);

  void MessageReceived (BMessage* MessagePntr);

  void AddControls ();
  void AddSubOptions ();
  void CopyFromGlobalToDisplay ();
  void CopyFromDisplayToGlobal ();

  void AllocateBufferAndFillWithPattern ();
  void DoReadWrite ();

  bool m_WriteMode; /* false for read, true for the write variation. */

  typedef enum ControlsEnum /* Indices into m_ControlViews for this view. */
  {
    AMOUNT_CONTROL_INDEX = 0,
    POSITION_CONTROL_INDEX,
    ADDRESS_CONTROL_INDEX,
    DO_READWRITE_CONTROL_INDEX
  } ControlIndices;
};


ReadAndWriteView::ReadAndWriteView (
  BRect FrameSize, const char *ViewName, bool WriteMode)
: GenericOptionsView (FrameSize, ViewName),
  m_WriteMode (WriteMode)
{
  if (m_WriteMode)
    SetViewColor (255, 255, 0);
  else
    SetViewColor (255, 128, 128);
}


void ReadAndWriteView::AddControls ()
{
  BButton      *ButtonPntr;
  float         HeightUsedByControls;
  float         MarginAboveControl;
  BTextControl *NewControlPntr;
  BRect         TempRect;
  float         WidthOfButtons;

  RemoveControls ();

  /* Figure out general height of the controls area and other sizes. */

  HeightUsedByControls = g_TextBoxHeight;
  if (HeightUsedByControls < g_ButtonHeight)
    HeightUsedByControls = g_ButtonHeight;

  m_HeightOfControlsArea = (int) (HeightUsedByControls * 1.2);
  WidthOfButtons = (int)
    be_plain_font->StringWidth ("MMPosition: 123456789123456789MM");

  /* Add the seek position text box. */

  MarginAboveControl =
    (int) ((m_HeightOfControlsArea - g_TextBoxHeight) / 2);
  TempRect = Bounds ();
  TempRect.top += MarginAboveControl;
  TempRect.bottom = TempRect.top + g_TextBoxHeight;
  TempRect.left += g_MarginBetweenControls;
  TempRect.right = TempRect.left + WidthOfButtons;

  NewControlPntr = new BTextControl (TempRect,
    "Position Text Control",
    "Position:" /* label */,
    NULL /* text */,
    new BMessage (DLG_SEEK_POSITION),
    B_FOLLOW_LEFT | B_FOLLOW_TOP);

  AddChild (NewControlPntr);
  NewControlPntr->SetTarget (this);
  NewControlPntr->SetDivider (be_plain_font->StringWidth (
    "Position: ")); /* Leave more space for the big number. */
  m_ControlViews[POSITION_CONTROL_INDEX] = NewControlPntr;

  /* Add the desired amount / buffer size. */

  WidthOfButtons = (int) /* Back to normal sized holes for numbers. */
    be_plain_font->StringWidth ("MAmount: Out of Memory!M");
  TempRect.left = TempRect.right + g_MarginBetweenControls;
  TempRect.right = TempRect.left + WidthOfButtons;

  NewControlPntr = new BTextControl (TempRect,
    "Amount Text Control",
    "Amount:" /* label */,
    NULL /* text */,
    new BMessage (DLG_BUFFER_SIZE),
    B_FOLLOW_LEFT | B_FOLLOW_TOP);

  AddChild (NewControlPntr);
  NewControlPntr->SetTarget (this);
  NewControlPntr->SetDivider (be_plain_font->StringWidth ("Amount: "));
  m_ControlViews[AMOUNT_CONTROL_INDEX] = NewControlPntr;

  /* Add the buffer address string view. */

  TempRect.left = TempRect.right + g_MarginBetweenControls;
  TempRect.right = TempRect.left + WidthOfButtons;

  NewControlPntr = new BTextControl (TempRect,
    "Address Text Control",
    "Address:" /* label */,
    NULL /* text */,
    NULL /* message */,
    B_FOLLOW_LEFT | B_FOLLOW_TOP,
    B_WILL_DRAW | B_FRAME_EVENTS | B_FULL_UPDATE_ON_RESIZE);

  AddChild (NewControlPntr);
  NewControlPntr->SetTarget (this);
  NewControlPntr->SetDivider (be_plain_font->StringWidth ("Address: "));
  NewControlPntr->SetEnabled (false);
  m_ControlViews[ADDRESS_CONTROL_INDEX] = NewControlPntr;

  /* Add the Read/Write button (contents vary depending on mode). */

  TempRect.left = TempRect.right + g_MarginBetweenControls;
  TempRect.right = TempRect.left +
    (int) be_plain_font->StringWidth (m_WriteMode ? "MWriteM" : "MReadM");
  TempRect.top =
    (int) (Bounds().top + (m_HeightOfControlsArea - g_ButtonHeight) / 2);
  TempRect.bottom = TempRect.top + g_ButtonHeight;

  ButtonPntr = new BButton (TempRect,
    m_WriteMode ? "Write Button" : "Read Button",
    m_WriteMode ? "Write" : "Read",
    new BMessage (DLG_DO_READWRITE),
    B_FOLLOW_TOP | B_FOLLOW_LEFT);

  AddChild (ButtonPntr);
  ButtonPntr->SetTarget (this);
  m_ControlViews[DO_READWRITE_CONTROL_INDEX] = ButtonPntr;

  /* Add a suboptions view showing the text buffer if the buffer exists,
  so that you can see the old buffer if you come back to this mode.  Also
  in write mode, creat the buffer so that the user can type into it. */

  if (m_WriteMode)
    AllocateBufferAndFillWithPattern ();

  if (g_DataBufferPntr != NULL)
    AddSubOptions ();

  CopyFromGlobalToDisplay (); /* Update display, including buffer address. */
}


/* Both read and write options have a PlainTextView displaying the buffer
as the suboption.  I suppose in the future we could have a hex view too,
and a new suboption parent with buttons to choose between hex and text. */

void ReadAndWriteView::AddSubOptions ()
{
  BRect TempRect;

  RemoveSubOptions ();

  TempRect = Bounds ();
  TempRect.top += m_HeightOfControlsArea;

  m_SubOptionsView = new PlainTextView (TempRect);
  AddChild (m_SubOptionsView);
  m_SubOptionsView->AddControls ();
}


/* Display the values in the global variables.  Updates seek offset, amount
to read/write, and the buffer address.  Doesn't update the display of the
buffer contents because that is a separate suboption view. */

void ReadAndWriteView::CopyFromGlobalToDisplay ()
{
  char          TempString [80];
  BTextControl *TextPntr;

  /* Fill in the 64 bit seek offset displayed value. */

  TextPntr = (BTextControl *) m_ControlViews[POSITION_CONTROL_INDEX];
  if (TextPntr != NULL)
  {
    sprintf (TempString, "%qd", g_SeekPosition);
    TextPntr->SetText (TempString);
  }

  /* Fill in the buffer size / amount displayed value. */

  TextPntr = (BTextControl *) m_ControlViews[AMOUNT_CONTROL_INDEX];
  if (TextPntr != NULL)
  {
    sprintf (TempString, "%ld", g_DesiredDataBufferSize);
    TextPntr->SetText (TempString);
  }

  /* Fill in the buffer address text box. */

  TextPntr = (BTextControl *) m_ControlViews[ADDRESS_CONTROL_INDEX];
  if (TextPntr != NULL)
  {
    sprintf (TempString, "$%lX", (unsigned long) g_DataBufferPntr);
    TextPntr->SetText (TempString);
  }
}


void ReadAndWriteView::CopyFromDisplayToGlobal ()
{
  BTextControl *TextPntr;

  /* Get the 64 bit value from the seek offset text box. */

  TextPntr = (BTextControl *) m_ControlViews[POSITION_CONTROL_INDEX];
  if (TextPntr != NULL)
    g_SeekPosition = atoll (TextPntr->Text ());

  /* Get the value from the buffer size / amount text box. */

  TextPntr = (BTextControl *) m_ControlViews[AMOUNT_CONTROL_INDEX];
  if (TextPntr != NULL)
    g_DesiredDataBufferSize = atol (TextPntr->Text ());

  /* The buffer address isn't user editable. */
}


/* Custom message processing to handle our common buttons. */

void ReadAndWriteView::MessageReceived (BMessage* MessagePntr)
{
  switch (MessagePntr->what)
  {
    case DLG_SEEK_POSITION:
      CopyFromDisplayToGlobal ();
      CopyFromGlobalToDisplay ();
      break;

    case DLG_BUFFER_SIZE:
      CopyFromDisplayToGlobal ();
      if (m_WriteMode) /* Let the user edit the buffer when writing. */
      {
        AllocateBufferAndFillWithPattern ();
        AddSubOptions ();
      }
      else /* Avoid delays by not allocating buffer until read done. */
      {
        FreeGlobalDataBuffer ();
        RemoveSubOptions ();
      }
      CopyFromGlobalToDisplay ();
      break;

    case DLG_DO_READWRITE:
        CopyFromDisplayToGlobal ();
        DoReadWrite ();
        CopyFromGlobalToDisplay ();
        break;

    default:
      GenericOptionsView::MessageReceived (MessagePntr);
      break;
  }
}


/* Allocates memory for the buffer, if it isn't already allocated at the
right size.  Fills it with a recognisable pattern based on the file offset,
but only if the buffer changed (so it doesn't trash a previously read in
buffer). */

void ReadAndWriteView::AllocateBufferAndFillWithPattern ()
{
  long long i;
  char     *PreviousBufferPntr;

  PreviousBufferPntr = g_DataBufferPntr;
  AllocateGlobalDataBuffer ();

  if (g_DataBufferPntr != NULL && PreviousBufferPntr != g_DataBufferPntr)
  {
    /* Add a hex position number every 1024 bytes, with a line feed before
    it so that the text view doesn't go nuts trying to wordwrap a single
    line several megabytes long.  Don't print numbers near the start or end
    of the buffer to avoid overruns. */

    for (
    i = ((g_SeekPosition + g_DataBufferSize) & 0xFFFFFFFFFFFFFC00LL) - 1024;
    i >= g_SeekPosition + 1024;
    i -= 1024)
    {
      sprintf (g_DataBufferPntr + (i - g_SeekPosition - 1),
      "\nUninitialised data at offset $%qX or %qd decimal. ", i, i);
    }
  }
}


/* Do the real work, read or write data depending on the mode,
and update the text suboption view. */

void ReadAndWriteView::DoReadWrite ()
{
  long           AmountRead;
  long           AmountWritten;
  int            ErrorCode;
  off_t          SeekedTo;
  char           TempString [80];

  /* Make sure buffer exists, if not already there.  Happens when reading. */

  AllocateBufferAndFillWithPattern ();

  if (g_DeviceFileDescriptor < 0)
    return; // An unlikely error.  This suboption shouldn't exist.

  if (g_DataBufferPntr == NULL)
  {
    DisplayErrorMessage ("Read/write needs a buffer, are you out of memory?");
    return;
  }

  SeekedTo = lseek (g_DeviceFileDescriptor, g_SeekPosition, SEEK_SET);
  if (SeekedTo != g_SeekPosition)
  {
    ErrorCode = errno;
    sprintf (TempString, "lseek() returned %qd, was expecting %qd",
      SeekedTo, g_SeekPosition);
    DisplayErrorMessage (TempString, ErrorCode);
    return;
  }

  if (m_WriteMode)
  {
    if (m_SubOptionsView != NULL)
      m_SubOptionsView->CopyFromDisplayToGlobal (); /* Get the user's text. */

    AmountWritten =
      write (g_DeviceFileDescriptor, g_DataBufferPntr, g_DataBufferSize);
    if (AmountWritten != g_DataBufferSize)
    {
      ErrorCode = errno;
      sprintf (TempString, "write() only wrote %ld bytes of %ld",
        AmountWritten, g_DataBufferSize);
      DisplayErrorMessage (TempString, ErrorCode);
    }
  }
  else
  {
    AmountRead =
      read (g_DeviceFileDescriptor, g_DataBufferPntr, g_DataBufferSize);
    if (AmountRead != g_DataBufferSize)
    {
      ErrorCode = errno;
      sprintf (TempString, "read() only read %ld bytes of %ld",
        AmountRead, g_DataBufferSize);
      DisplayErrorMessage (TempString, ErrorCode);
    }
    if (AmountRead > 0)
      AddSubOptions (); /* Destroy & recreate the text view to show buffer. */
  }
}



/******************************************************************************
 * This suboption displays and edits the boolean argument to an IOCtl.
 * We just use a check box as the user interface to the bool.
 */

class IoctlBoolDataView : public GenericOptionsView
{
public:
  IoctlBoolDataView (BRect FrameSize);

  void AddControls ();
  void AddSubOptions () {/* no suboptions */};
  void CopyFromGlobalToDisplay ();
  void CopyFromDisplayToGlobal ();

  void MessageReceived (BMessage* MessagePntr);

  typedef enum ControlsEnum /* Indices into m_ControlViews for this view. */
  {
    CHECK_BOX_INDEX = 0,
  } ControlIndices;

};


IoctlBoolDataView::IoctlBoolDataView (
  BRect FrameSize)
: GenericOptionsView (FrameSize, "IoctlBoolDataView")
{
  SetViewColor (128, 255, 200);
}


void IoctlBoolDataView::AddControls ()
{
  BCheckBox *CheckBoxPntr;
  float      MarginAboveControl;
  BRect      TempRect;

  RemoveControls ();

  /* We will be modifying this global variable to reflect our value. */

  g_IOCtlDataPntr = &g_IOCtlDataBool;
  g_IOCtlDataSize = sizeof (g_IOCtlDataBool);

  /* Add a check box to display the bool. */

  m_HeightOfControlsArea = (int) (1.2 * g_CheckBoxHeight);
  MarginAboveControl = (int) ((m_HeightOfControlsArea - g_CheckBoxHeight) / 2);
  TempRect = Bounds ();
  TempRect.top += MarginAboveControl;
  TempRect.bottom = TempRect.top + g_CheckBoxHeight;
  TempRect.left += g_MarginBetweenControls;
  TempRect.right = TempRect.left +
    (int) be_plain_font->StringWidth ("MBool: XXXM");

  CheckBoxPntr = new BCheckBox (TempRect,
    "Bool Checkbox",
    "Bool:",
    new BMessage (DLG_IOCTL_BOOL),
    B_FOLLOW_TOP | B_FOLLOW_LEFT);

  AddChild (CheckBoxPntr);
  CheckBoxPntr->SetTarget (this);
  m_ControlViews[CHECK_BOX_INDEX] = CheckBoxPntr;

  CopyFromGlobalToDisplay ();
}


void IoctlBoolDataView::CopyFromGlobalToDisplay ()
{
  BCheckBox *CheckBoxPntr;

  CheckBoxPntr = (BCheckBox *) m_ControlViews[CHECK_BOX_INDEX];
  if (CheckBoxPntr != NULL)
    CheckBoxPntr->SetValue (g_IOCtlDataBool ? B_CONTROL_ON : B_CONTROL_OFF);
}


void IoctlBoolDataView::CopyFromDisplayToGlobal ()
{
  BCheckBox *CheckBoxPntr;

  CheckBoxPntr = (BCheckBox *) m_ControlViews[CHECK_BOX_INDEX];
  if (CheckBoxPntr != NULL)
    g_IOCtlDataBool = (CheckBoxPntr->Value () == B_CONTROL_ON);
}


void IoctlBoolDataView::MessageReceived (BMessage* MessagePntr)
{
  if (MessagePntr->what == DLG_IOCTL_BOOL)
  {
    CopyFromDisplayToGlobal ();
    CopyFromGlobalToDisplay ();
  }
  else
    GenericOptionsView::MessageReceived (MessagePntr);
}



/******************************************************************************
 * This suboption displays and edits the byte argument to an IOCtl.
 * We just use a text box as the user interface to the number.
 */

class IoctlByteDataView : public GenericOptionsView
{
public:
  IoctlByteDataView (BRect FrameSize);

  void AddControls ();
  void AddSubOptions () {/* no suboptions */};
  void CopyFromGlobalToDisplay ();
  void CopyFromDisplayToGlobal ();

  void MessageReceived (BMessage* MessagePntr);

  typedef enum ControlsEnum /* Indices into m_ControlViews for this view. */
  {
    TEXT_BOX_INDEX = 0,
  } ControlIndices;

};


IoctlByteDataView::IoctlByteDataView (
  BRect FrameSize)
: GenericOptionsView (FrameSize, "IoctlByteDataView")
{
  SetViewColor (255, 128, 200);
}


void IoctlByteDataView::AddControls ()
{
  float         MarginAboveControl;
  BRect         TempRect;
  BTextControl *TextBoxPntr;

  RemoveControls ();

  /* We will be modifying this global variable to reflect our value. */

  g_IOCtlDataPntr = &g_IOCtlDataByte;
  g_IOCtlDataSize = sizeof (g_IOCtlDataByte);

  /* Add a text box to display the bool. */

  m_HeightOfControlsArea = (int) (1.2 * g_TextBoxHeight);
  MarginAboveControl = (int) ((m_HeightOfControlsArea - g_TextBoxHeight) / 2);
  TempRect = Bounds ();
  TempRect.top += MarginAboveControl;
  TempRect.bottom = TempRect.top + g_TextBoxHeight;
  TempRect.left += g_MarginBetweenControls;
  TempRect.right -= g_MarginBetweenControls;

  TextBoxPntr = new BTextControl (TempRect,
    "Byte Text Control",
    "Byte:" /* label */,
    NULL /* text */,
    new BMessage (DLG_IOCTL_BYTE),
    B_FOLLOW_LEFT_RIGHT | B_FOLLOW_TOP);
  TextBoxPntr->SetDivider (be_plain_font->StringWidth (
    "Byte: ")); /* Leave space for the number. */

  AddChild (TextBoxPntr);
  TextBoxPntr->SetTarget (this);
  m_ControlViews[TEXT_BOX_INDEX] = TextBoxPntr;

  CopyFromGlobalToDisplay ();
}


void IoctlByteDataView::CopyFromGlobalToDisplay ()
{
  char          TempString [80];
  BTextControl *TextBoxPntr;

  TextBoxPntr = (BTextControl *) m_ControlViews[TEXT_BOX_INDEX];
  sprintf (TempString, "%d  $%X", (int) g_IOCtlDataByte,
    (int) g_IOCtlDataByte);
  if (TextBoxPntr != NULL) TextBoxPntr->SetText (TempString);
}


void IoctlByteDataView::CopyFromDisplayToGlobal ()
{
  BTextControl *TextBoxPntr;

  TextBoxPntr = (BTextControl *) m_ControlViews[TEXT_BOX_INDEX];

  if (TextBoxPntr != NULL)
    g_IOCtlDataByte = atoi (TextBoxPntr->Text ());
}


void IoctlByteDataView::MessageReceived (BMessage* MessagePntr)
{
  if (MessagePntr->what == DLG_IOCTL_BYTE)
  {
    CopyFromDisplayToGlobal ();
    CopyFromGlobalToDisplay ();
  }
  else
    GenericOptionsView::MessageReceived (MessagePntr);
}



/******************************************************************************
 * This suboption displays and edits the size_t argument to an IOCtl.
 * We just use a text box as the user interface to the number.
 */

class IoctlSizetDataView : public GenericOptionsView
{
public:
  IoctlSizetDataView (BRect FrameSize);

  void AddControls ();
  void AddSubOptions () {/* no suboptions */};
  void CopyFromGlobalToDisplay ();
  void CopyFromDisplayToGlobal ();

  void MessageReceived (BMessage* MessagePntr);

  typedef enum ControlsEnum /* Indices into m_ControlViews for this view. */
  {
    TEXT_BOX_INDEX = 0,
  } ControlIndices;
};


IoctlSizetDataView::IoctlSizetDataView (
  BRect FrameSize)
: GenericOptionsView (FrameSize, "IoctlSizetDataView")
{
  SetViewColor (128, 200, 255);
}


void IoctlSizetDataView::AddControls ()
{
  BTextControl *TextBoxPntr;
  float         MarginAboveControl;
  BRect         TempRect;

  RemoveControls ();

  /* We will be modifying this global variable to reflect our value. */

  g_IOCtlDataPntr = &g_IOCtlDataSize_t;
  g_IOCtlDataSize = sizeof (g_IOCtlDataSize_t);

  /* Add the text box displaying the size_t number. */

  m_HeightOfControlsArea = (int) (1.2 * g_TextBoxHeight);
  MarginAboveControl = (int) ((m_HeightOfControlsArea - g_TextBoxHeight) / 2);
  TempRect = Bounds ();
  TempRect.top += MarginAboveControl;
  TempRect.bottom = TempRect.top + g_TextBoxHeight;
  TempRect.left += g_MarginBetweenControls;
  TempRect.right = TempRect.left +
    (int) be_plain_font->StringWidth ("Msize_t: 8888888888M");

  TextBoxPntr = new BTextControl (TempRect,
    "size_t Text Control",
    "size_t:" /* label */,
    NULL /* text */,
    new BMessage (DLG_IOCTL_SIZET),
    B_FOLLOW_LEFT | B_FOLLOW_TOP);

  TextBoxPntr->SetDivider (be_plain_font->StringWidth (
    "size_t: ")); /* Leave space for the number. */

  AddChild (TextBoxPntr);
  TextBoxPntr->SetTarget (this);
  m_ControlViews[TEXT_BOX_INDEX] = TextBoxPntr;

  CopyFromGlobalToDisplay ();
}


void IoctlSizetDataView::CopyFromGlobalToDisplay ()
{
  char          TempString [80];
  BTextControl *TextBoxPntr;

  TextBoxPntr = (BTextControl *) m_ControlViews[TEXT_BOX_INDEX];
  sprintf (TempString, "%ld", g_IOCtlDataSize_t);

  if (TextBoxPntr != NULL)
    TextBoxPntr->SetText (TempString);
}


void IoctlSizetDataView::CopyFromDisplayToGlobal ()
{
  BTextControl *TextBoxPntr;

  TextBoxPntr = (BTextControl *) m_ControlViews[TEXT_BOX_INDEX];

  if (TextBoxPntr != NULL)
    g_IOCtlDataSize_t = atol (TextBoxPntr->Text ());
}


void IoctlSizetDataView::MessageReceived (BMessage* MessagePntr)
{
  if (MessagePntr->what == DLG_IOCTL_SIZET)
  {
    CopyFromDisplayToGlobal ();
    CopyFromGlobalToDisplay ();
  }
  else
  {
    GenericOptionsView::MessageReceived (MessagePntr);
  }
}



/******************************************************************************
 * This suboption displays and edits the status_t argument to an IOCtl.
 * We use a text box as the user interface to the number and another text
 * box to display the equivalent symbolic code / error message.
 */

class IoctlStatustDataView : public GenericOptionsView
{
public:
  IoctlStatustDataView (BRect FrameSize);

  void AddControls ();
  void AddSubOptions () {/* no suboptions */};
  void CopyFromGlobalToDisplay ();
  void CopyFromDisplayToGlobal ();

  void MessageReceived (BMessage* MessagePntr);

  typedef enum ControlsEnum /* Indices into m_ControlViews for this view. */
  {
    TEXT_BOX_INDEX = 0,
    SYMBOLIC_CODE_INDEX
  } ControlIndices;
};


IoctlStatustDataView::IoctlStatustDataView (
  BRect FrameSize)
: GenericOptionsView (FrameSize, "IoctlStatustDataView")
{
  SetViewColor (255, 160, 200);
}


void IoctlStatustDataView::AddControls ()
{
  BTextControl *TextBoxPntr;
  float         MarginAboveControl;
  BRect         TempRect;

  RemoveControls ();

  /* We will be modifying this global variable to reflect our value. */

  g_IOCtlDataPntr = &g_IOCtlDataStatus_t;
  g_IOCtlDataSize = sizeof (g_IOCtlDataStatus_t);

  /* Add the text box displaying the status_t number. */

  m_HeightOfControlsArea = (int) (1.2 * g_TextBoxHeight);
  MarginAboveControl = (int) ((m_HeightOfControlsArea - g_TextBoxHeight) / 2);
  TempRect = Bounds ();
  TempRect.top += MarginAboveControl;
  TempRect.bottom = TempRect.top + g_TextBoxHeight;
  TempRect.left += g_MarginBetweenControls;
  TempRect.right = TempRect.left +
    (int) be_plain_font->StringWidth ("Mstatus_t: 8888888888M");

  TextBoxPntr = new BTextControl (TempRect,
    "status_t Text Control",
    "status_t:" /* label */,
    NULL /* text */,
    new BMessage (DLG_IOCTL_STATUST),
    B_FOLLOW_LEFT | B_FOLLOW_TOP);

  TextBoxPntr->SetDivider (be_plain_font->StringWidth (
    "status_t: ")); /* Leave space for the number. */

  AddChild (TextBoxPntr);
  TextBoxPntr->SetTarget (this);
  m_ControlViews[TEXT_BOX_INDEX] = TextBoxPntr;

  TempRect.left = TempRect.right + g_MarginBetweenControls;
  TempRect.right = Bounds().right - g_MarginBetweenControls;

  TextBoxPntr = new BTextControl (TempRect,
    "status_t Symbol Control",
    NULL /* label */,
    NULL /* text */,
    NULL /* message */,
    B_FOLLOW_LEFT_RIGHT | B_FOLLOW_TOP);

  AddChild (TextBoxPntr);
  TextBoxPntr->SetTarget (this);
  m_ControlViews[SYMBOLIC_CODE_INDEX] = TextBoxPntr;

  CopyFromGlobalToDisplay ();
}


void IoctlStatustDataView::CopyFromGlobalToDisplay ()
{
  const char   *ErrorString;
  char          TempString [1024];
  BTextControl *TextBoxPntr;

  /* Update the number in the text box to show the code. */

  TextBoxPntr = (BTextControl *) m_ControlViews[TEXT_BOX_INDEX];
  sprintf (TempString, "%ld", g_IOCtlDataStatus_t);

  if (TextBoxPntr != NULL)
    TextBoxPntr->SetText (TempString);

  /* Find the error message string to go with the code. */

  const int ErrorBase =
    B_DEVICE_ERROR_BASE; /* Avoid macro expansion problem. */
  const int MaxError =
    ErrorBase + sizeof (ErrorDescriptions) / sizeof (const char *);

  if (g_IOCtlDataStatus_t >= ErrorBase &&
  g_IOCtlDataStatus_t < MaxError)
    ErrorString =
    ErrorDescriptions [g_IOCtlDataStatus_t - ErrorBase];
  else
    ErrorString = "?";

  sprintf (TempString, "%ld/$%lX/%s  %s.",
    g_IOCtlDataStatus_t,
    g_IOCtlDataStatus_t,
    ErrorString,
    strerror (g_IOCtlDataStatus_t));

  TextBoxPntr = (BTextControl *) m_ControlViews[SYMBOLIC_CODE_INDEX];

  if (TextBoxPntr != NULL)
    TextBoxPntr->SetText (TempString);
}


void IoctlStatustDataView::CopyFromDisplayToGlobal ()
{
  BTextControl *TextBoxPntr;

  TextBoxPntr = (BTextControl *) m_ControlViews[TEXT_BOX_INDEX];

  if (TextBoxPntr != NULL)
    g_IOCtlDataStatus_t = atol (TextBoxPntr->Text ());
}


void IoctlStatustDataView::MessageReceived (BMessage* MessagePntr)
{
  if (MessagePntr->what == DLG_IOCTL_STATUST)
  {
    CopyFromDisplayToGlobal ();
    CopyFromGlobalToDisplay ();
  }
  else
  {
    GenericOptionsView::MessageReceived (MessagePntr);
  }
}



/******************************************************************************
 * This suboption displays the device_geometry argument to an IOCtl.
 * There are text boxes for the numbers bytes_per_sector, sectors_per_track,
 * cylinder_count, head_count and total space.  There are a pair of boxes for
 * device_type - one for the number and another for the description.  Finally
 * there are check boxes for removable, read_only and write_once.  Most are
 * seem editable, but that isn't implemented since geometry data is read only.
 */

class IoctlGeometryDataView : public GenericOptionsView
{
public:
  IoctlGeometryDataView (BRect FrameSize);

  void AddControls ();
  void AddSubOptions () {/* no suboptions */};
  void CopyFromGlobalToDisplay ();
  void CopyFromDisplayToGlobal () {/* not implemented since unnecessary */};

  void MessageReceived (BMessage* MessagePntr);

  typedef enum ControlsEnum /* Indices into m_ControlViews for this view. */
  {
    BYTES_PER_SECTOR_TEXT_BOX_INDEX = 0,
    SECTORS_PER_TRACK_TEXT_BOX_INDEX,
    CYLINDERS_TEXT_BOX_INDEX,
    HEADS_TEXT_BOX_INDEX,
    TOTAL_SPACE_TEXT_BOX_INDEX,
    DEVICE_TYPE_NUMBER_TEXT_BOX_INDEX,
    DEVICE_TYPE_DESCRIPTION_TEXT_BOX_INDEX,
    REMOVEABLE_CHECK_BOX_INDEX,
    READ_ONLY_CHECK_BOX_INDEX,
    WRITE_ONCE_CHECK_BOX_INDEX
  } ControlIndices;
};


IoctlGeometryDataView::IoctlGeometryDataView (
  BRect FrameSize)
: GenericOptionsView (FrameSize, "IoctlGeometryDataView")
{
  SetViewColor (200, 200, 200);
}


void IoctlGeometryDataView::AddControls ()
{
  BCheckBox    *CheckBoxPntr;
  float         MarginAboveControl;
  BRect         TempRect;
  BTextControl *TextBoxPntr;
  float         WidthOfControls;

  RemoveControls ();

  /* We will be modifying this global variable to reflect our value. */

  g_IOCtlDataPntr = &g_IOCtlDataDevice_geometry;
  g_IOCtlDataSize = sizeof (g_IOCtlDataDevice_geometry);

  /* Do the display as 5 rows, with the size related items on the left
  and all the others on the right. */

  m_HeightOfControlsArea = g_TextBoxHeight;
  if (m_HeightOfControlsArea < g_CheckBoxHeight)
    m_HeightOfControlsArea = g_CheckBoxHeight;
  m_HeightOfControlsArea = (int) (1.2 * m_HeightOfControlsArea);

  WidthOfControls = (int) be_plain_font->StringWidth (
    "MSectors per Track: 8888888888M");

  /* The bytes per sector box.  Row 1, left side. */

  MarginAboveControl = (int) ((m_HeightOfControlsArea - g_TextBoxHeight) / 2);
  TempRect = Bounds ();
  TempRect.top += MarginAboveControl;
  TempRect.bottom = TempRect.top + g_TextBoxHeight;
  TempRect.left += g_MarginBetweenControls;
  TempRect.right = TempRect.left + WidthOfControls;

  TextBoxPntr = new BTextControl (TempRect,
    "Bytes per sector Control",
    "Bytes per Sector:" /* label */,
    NULL /* text */,
    new BMessage (DLG_IOCTL_GEOMETRY),
    B_FOLLOW_LEFT | B_FOLLOW_TOP);

  TextBoxPntr->SetDivider (be_plain_font->StringWidth (
    "Bytes per Sector: ")); /* Leave space for the number. */

  AddChild (TextBoxPntr);
  TextBoxPntr->SetTarget (this);
  m_ControlViews[BYTES_PER_SECTOR_TEXT_BOX_INDEX] = TextBoxPntr;

  /* The device type number box.  Row 1, right side. */

  TempRect.left = TempRect.right + g_MarginBetweenControls;
  TempRect.right = TempRect.left + WidthOfControls;

  TextBoxPntr = new BTextControl (TempRect,
    "Device Type Control",
    "Device Type:" /* label */,
    NULL /* text */,
    new BMessage (DLG_IOCTL_GEOMETRY),
    B_FOLLOW_LEFT | B_FOLLOW_TOP);

  TextBoxPntr->SetDivider (be_plain_font->StringWidth (
    "Device Type: ")); /* Leave space for the number. */

  AddChild (TextBoxPntr);
  TextBoxPntr->SetTarget (this);
  m_ControlViews[DEVICE_TYPE_NUMBER_TEXT_BOX_INDEX] = TextBoxPntr;

  /* The sectors per track box.  Row 2, left side. */

  TempRect = Bounds ();
  TempRect.top += MarginAboveControl + 1 * m_HeightOfControlsArea;
  TempRect.bottom = TempRect.top + g_TextBoxHeight;
  TempRect.left += g_MarginBetweenControls;
  TempRect.right = TempRect.left + WidthOfControls;

  TextBoxPntr = new BTextControl (TempRect,
    "Sectors per track Control",
    "Sectors per Track:" /* label */,
    NULL /* text */,
    new BMessage (DLG_IOCTL_GEOMETRY),
    B_FOLLOW_LEFT | B_FOLLOW_TOP);

  TextBoxPntr->SetDivider (be_plain_font->StringWidth (
    "Sectors per Track: ")); /* Leave space for the number. */

  AddChild (TextBoxPntr);
  TextBoxPntr->SetTarget (this);
  m_ControlViews[SECTORS_PER_TRACK_TEXT_BOX_INDEX] = TextBoxPntr;

  /* The device type description box.  Row 2, right side. */

  TempRect.left = TempRect.right + g_MarginBetweenControls;
  TempRect.right = TempRect.left + WidthOfControls;

  TextBoxPntr = new BTextControl (TempRect,
    "Device Description Control",
    NULL /* label */,
    NULL /* text */,
    NULL /* message */,
    B_FOLLOW_LEFT | B_FOLLOW_TOP);

  AddChild (TextBoxPntr);
  TextBoxPntr->SetTarget (this);
  TextBoxPntr->SetEnabled (false);
  m_ControlViews[DEVICE_TYPE_DESCRIPTION_TEXT_BOX_INDEX] = TextBoxPntr;

  /* The cylinders box.  Row 3, left side. */

  TempRect = Bounds ();
  TempRect.top += MarginAboveControl + 2 * m_HeightOfControlsArea;
  TempRect.bottom = TempRect.top + g_TextBoxHeight;
  TempRect.left += g_MarginBetweenControls;
  TempRect.right = TempRect.left + WidthOfControls;

  TextBoxPntr = new BTextControl (TempRect,
    "Cylinders Control",
    "Cylinders:" /* label */,
    NULL /* text */,
    new BMessage (DLG_IOCTL_GEOMETRY),
    B_FOLLOW_LEFT | B_FOLLOW_TOP);

  TextBoxPntr->SetDivider (be_plain_font->StringWidth (
    "Cylinders: ")); /* Leave space for the number. */

  AddChild (TextBoxPntr);
  TextBoxPntr->SetTarget (this);
  m_ControlViews[CYLINDERS_TEXT_BOX_INDEX] = TextBoxPntr;

  /* The Removeable checkbox.  Row 3, right side. */

  MarginAboveControl = (int) ((m_HeightOfControlsArea - g_CheckBoxHeight) / 2);
  TempRect = Bounds ();
  TempRect.top += MarginAboveControl + 2 * m_HeightOfControlsArea;
  TempRect.bottom = TempRect.top + g_CheckBoxHeight;
  TempRect.left += 2 * g_MarginBetweenControls + WidthOfControls;
  TempRect.right = TempRect.left + WidthOfControls;

  CheckBoxPntr = new BCheckBox (TempRect,
    "Removeable Checkbox",
    "Removeable",
    new BMessage (DLG_IOCTL_GEOMETRY),
    B_FOLLOW_TOP | B_FOLLOW_LEFT);

  AddChild (CheckBoxPntr);
  CheckBoxPntr->SetTarget (this);
  m_ControlViews[REMOVEABLE_CHECK_BOX_INDEX] = CheckBoxPntr;

  /* The Heads box.  Row 4, left side. */

  MarginAboveControl = (int) ((m_HeightOfControlsArea - g_TextBoxHeight) / 2);
  TempRect = Bounds ();
  TempRect.top += MarginAboveControl + 3 * m_HeightOfControlsArea;
  TempRect.bottom = TempRect.top + g_TextBoxHeight;
  TempRect.left += g_MarginBetweenControls;
  TempRect.right = TempRect.left + WidthOfControls;

  TextBoxPntr = new BTextControl (TempRect,
    "Heads Control",
    "Heads:" /* label */,
    NULL /* text */,
    new BMessage (DLG_IOCTL_GEOMETRY),
    B_FOLLOW_LEFT | B_FOLLOW_TOP);

  TextBoxPntr->SetDivider (be_plain_font->StringWidth (
    "Heads: ")); /* Leave space for the number. */

  AddChild (TextBoxPntr);
  TextBoxPntr->SetTarget (this);
  m_ControlViews[HEADS_TEXT_BOX_INDEX] = TextBoxPntr;

  /* The Read Only checkbox.  Row 4, right side. */

  MarginAboveControl = (int) ((m_HeightOfControlsArea - g_CheckBoxHeight) / 2);
  TempRect = Bounds ();
  TempRect.top += MarginAboveControl + 3 * m_HeightOfControlsArea;
  TempRect.bottom = TempRect.top + g_CheckBoxHeight;
  TempRect.left += 2 * g_MarginBetweenControls + WidthOfControls;
  TempRect.right = TempRect.left + WidthOfControls;

  CheckBoxPntr = new BCheckBox (TempRect,
    "Read Only Checkbox",
    "Read Only",
    new BMessage (DLG_IOCTL_GEOMETRY),
    B_FOLLOW_TOP | B_FOLLOW_LEFT);

  AddChild (CheckBoxPntr);
  CheckBoxPntr->SetTarget (this);
  m_ControlViews[READ_ONLY_CHECK_BOX_INDEX] = CheckBoxPntr;

  /* The Total Space box.  Row 5, left side. */

  MarginAboveControl = (int) ((m_HeightOfControlsArea - g_TextBoxHeight) / 2);
  TempRect = Bounds ();
  TempRect.top += MarginAboveControl + 4 * m_HeightOfControlsArea;
  TempRect.bottom = TempRect.top + g_TextBoxHeight;
  TempRect.left += g_MarginBetweenControls;
  TempRect.right = TempRect.left + WidthOfControls;

  TextBoxPntr = new BTextControl (TempRect,
    "Total Space Control",
    "Total Space:" /* label */,
    NULL /* text */,
    new BMessage (DLG_IOCTL_GEOMETRY),
    B_FOLLOW_LEFT | B_FOLLOW_TOP);

  TextBoxPntr->SetDivider (be_plain_font->StringWidth (
    "Total Space: ")); /* Leave space for the number. */

  AddChild (TextBoxPntr);
  TextBoxPntr->SetTarget (this);
  m_ControlViews[TOTAL_SPACE_TEXT_BOX_INDEX] = TextBoxPntr;

  /* The Write Once checkbox.  Row 5, right side. */

  MarginAboveControl = (int) ((m_HeightOfControlsArea - g_CheckBoxHeight) / 2);
  TempRect = Bounds ();
  TempRect.top += MarginAboveControl + 4 * m_HeightOfControlsArea;
  TempRect.bottom = TempRect.top + g_CheckBoxHeight;
  TempRect.left += 2 * g_MarginBetweenControls + WidthOfControls;
  TempRect.right = TempRect.left + WidthOfControls;

  CheckBoxPntr = new BCheckBox (TempRect,
    "Write Once Checkbox",
    "Write Once",
    new BMessage (DLG_IOCTL_GEOMETRY),
    B_FOLLOW_TOP | B_FOLLOW_LEFT);

  AddChild (CheckBoxPntr);
  CheckBoxPntr->SetTarget (this);
  m_ControlViews[WRITE_ONCE_CHECK_BOX_INDEX] = CheckBoxPntr;

  m_HeightOfControlsArea *= 5; /* For 5 rows. */
  CopyFromGlobalToDisplay ();
}


void IoctlGeometryDataView::CopyFromGlobalToDisplay ()
{
  BCheckBox    *CheckBoxPntr;
  char          TempString [80];
  BTextControl *TextBoxPntr;

  /* Copy all fields from the global g_IOCtlDataDevice_geometry into the
  appropriate text boxes and check boxes. */

  TextBoxPntr = (BTextControl *)
    m_ControlViews[BYTES_PER_SECTOR_TEXT_BOX_INDEX];
  sprintf (TempString, "%lu", g_IOCtlDataDevice_geometry.bytes_per_sector);
  if (TextBoxPntr != NULL) TextBoxPntr->SetText (TempString);

  TextBoxPntr = (BTextControl *)
    m_ControlViews[SECTORS_PER_TRACK_TEXT_BOX_INDEX];
  sprintf (TempString, "%lu", g_IOCtlDataDevice_geometry.sectors_per_track);
  if (TextBoxPntr != NULL) TextBoxPntr->SetText (TempString);

  TextBoxPntr = (BTextControl *)
    m_ControlViews[CYLINDERS_TEXT_BOX_INDEX];
  sprintf (TempString, "%lu", g_IOCtlDataDevice_geometry.cylinder_count);
  if (TextBoxPntr != NULL) TextBoxPntr->SetText (TempString);

  TextBoxPntr = (BTextControl *)
    m_ControlViews[HEADS_TEXT_BOX_INDEX];
  sprintf (TempString, "%lu", g_IOCtlDataDevice_geometry.head_count);
  if (TextBoxPntr != NULL) TextBoxPntr->SetText (TempString);

  TextBoxPntr = (BTextControl *)
    m_ControlViews[TOTAL_SPACE_TEXT_BOX_INDEX];
  sprintf (TempString, "%qu",
    g_IOCtlDataDevice_geometry.bytes_per_sector *
    (uint64) g_IOCtlDataDevice_geometry.sectors_per_track *
    g_IOCtlDataDevice_geometry.cylinder_count *
    g_IOCtlDataDevice_geometry.head_count);
  if (TextBoxPntr != NULL) TextBoxPntr->SetText (TempString);

  TextBoxPntr = (BTextControl *)
    m_ControlViews[DEVICE_TYPE_NUMBER_TEXT_BOX_INDEX];
  sprintf (TempString, "%d", (int) g_IOCtlDataDevice_geometry.device_type);
  if (TextBoxPntr != NULL) TextBoxPntr->SetText (TempString);

  TextBoxPntr = (BTextControl *)
    m_ControlViews[DEVICE_TYPE_DESCRIPTION_TEXT_BOX_INDEX];
  if (g_IOCtlDataDevice_geometry.device_type <
  (sizeof (DeviceTypeDescriptions) / sizeof (DeviceTypeDescriptions[0])))
  {
    strcpy (TempString,
      DeviceTypeDescriptions [g_IOCtlDataDevice_geometry.device_type]);
  }
  else
    strcpy (TempString, "Unknown");
  if (TextBoxPntr != NULL) TextBoxPntr->SetText (TempString);

  CheckBoxPntr = (BCheckBox *) m_ControlViews[REMOVEABLE_CHECK_BOX_INDEX];
  if (CheckBoxPntr != NULL)
    CheckBoxPntr->SetValue (g_IOCtlDataDevice_geometry.removable ?
      B_CONTROL_ON : B_CONTROL_OFF);

  CheckBoxPntr = (BCheckBox *) m_ControlViews[READ_ONLY_CHECK_BOX_INDEX];
  if (CheckBoxPntr != NULL)
    CheckBoxPntr->SetValue (g_IOCtlDataDevice_geometry.read_only ?
      B_CONTROL_ON : B_CONTROL_OFF);

  CheckBoxPntr = (BCheckBox *) m_ControlViews[WRITE_ONCE_CHECK_BOX_INDEX];
  if (CheckBoxPntr != NULL)
    CheckBoxPntr->SetValue (g_IOCtlDataDevice_geometry.write_once ?
      B_CONTROL_ON : B_CONTROL_OFF);
}


void IoctlGeometryDataView::MessageReceived (BMessage* MessagePntr)
{
  if (MessagePntr->what == DLG_IOCTL_GEOMETRY)
  {
    CopyFromDisplayToGlobal ();
    CopyFromGlobalToDisplay ();
  }
  else
  {
    GenericOptionsView::MessageReceived (MessagePntr);
  }
}



/******************************************************************************
 * This suboption displays and edits the pathname argument to an IOCtl.
 * We just use a text box as the user interface to the string.
 */

class IoctlPathnameDataView : public GenericOptionsView
{
public:
  IoctlPathnameDataView (BRect FrameSize);

  void AddControls ();
  void AddSubOptions () {/* no suboptions */};
  void CopyFromGlobalToDisplay ();
  void CopyFromDisplayToGlobal ();

  void MessageReceived (BMessage* MessagePntr);

  typedef enum ControlsEnum /* Indices into m_ControlViews for this view. */
  {
    TEXT_BOX_INDEX = 0,
  } ControlIndices;
};


IoctlPathnameDataView::IoctlPathnameDataView (
  BRect FrameSize)
: GenericOptionsView (FrameSize, "IoctlPathnameDataView")
{
  SetViewColor (200, 255, 160);
}


void IoctlPathnameDataView::AddControls ()
{
  BTextControl *TextBoxPntr;
  float         MarginAboveControl;
  BRect         TempRect;

  RemoveControls ();

  /* We will be modifying this global variable to reflect our value. */

  g_IOCtlDataPntr = &g_IOCtlDataPathname;
  g_IOCtlDataSize = sizeof (g_IOCtlDataPathname);

  /* Add the text box displaying the path string. */

  m_HeightOfControlsArea = (int) (1.2 * g_TextBoxHeight);
  MarginAboveControl = (int) ((m_HeightOfControlsArea - g_TextBoxHeight) / 2);
  TempRect = Bounds ();
  TempRect.top += MarginAboveControl;
  TempRect.bottom = TempRect.top + g_TextBoxHeight;
  TempRect.left += g_MarginBetweenControls;
  TempRect.right -= g_MarginBetweenControls;

  TextBoxPntr = new BTextControl (TempRect,
    "Path Text Control",
    "Path:" /* label */,
    NULL /* text */,
    new BMessage (DLG_IOCTL_PATHNAME),
    B_FOLLOW_LEFT_RIGHT | B_FOLLOW_TOP);

  TextBoxPntr->SetDivider (be_plain_font->StringWidth (
    "Path: ")); /* Leave space for the pathname. */

  AddChild (TextBoxPntr);
  TextBoxPntr->SetTarget (this);
  m_ControlViews[TEXT_BOX_INDEX] = TextBoxPntr;

  CopyFromGlobalToDisplay ();
}


void IoctlPathnameDataView::CopyFromGlobalToDisplay ()
{
  BTextControl *TextBoxPntr;

  TextBoxPntr = (BTextControl *) m_ControlViews[TEXT_BOX_INDEX];

  if (TextBoxPntr != NULL)
    TextBoxPntr->SetText (g_IOCtlDataPathname);
}


void IoctlPathnameDataView::CopyFromDisplayToGlobal ()
{
  BTextControl *TextBoxPntr;

  TextBoxPntr = (BTextControl *) m_ControlViews[TEXT_BOX_INDEX];

  if (TextBoxPntr != NULL)
  {
    strncpy (g_IOCtlDataPathname, TextBoxPntr->Text (),
      sizeof (g_IOCtlDataPathname) - 1);
    g_IOCtlDataPathname [sizeof (g_IOCtlDataPathname) - 1] = 0;
  }
}


void IoctlPathnameDataView::MessageReceived (BMessage* MessagePntr)
{
  if (MessagePntr->what == DLG_IOCTL_PATHNAME)
  {
    CopyFromDisplayToGlobal ();
    CopyFromGlobalToDisplay ();
  }
  else
  {
    GenericOptionsView::MessageReceived (MessagePntr);
  }
}



/******************************************************************************
 * This suboption displays and edits the partition_info argument to an IOCtl.
 * The first row contains text boxes for the offset and size.  The next row
 * is the ending offset (calculated) and the block size numbers.  The third
 * row has session and partion IDs.  The fourth row displays the path to the
 * physical device.  All are editable except the ending offset.
 */

class IoctlPartitionDataView : public GenericOptionsView
{
public:
  IoctlPartitionDataView (BRect FrameSize);

  void AddControls ();
  void AddSubOptions () {/* no suboptions */};
  void CopyFromGlobalToDisplay ();
  void CopyFromDisplayToGlobal ();

  void MessageReceived (BMessage* MessagePntr);

  typedef enum ControlsEnum /* Indices into m_ControlViews for this view. */
  {
    OFFSET_TEXT_BOX_INDEX = 0,
    SIZE_TEXT_BOX_INDEX,
    ENDING_OFFSET_TEXT_BOX_INDEX,
    BLOCK_SIZE_TEXT_BOX_INDEX,
    SESSION_ID_TEXT_BOX_INDEX,
    PARTITION_ID_TEXT_BOX_INDEX,
    DEVICE_PATH_TEXT_BOX_INDEX,
  } ControlIndices;
};


IoctlPartitionDataView::IoctlPartitionDataView (
  BRect FrameSize)
: GenericOptionsView (FrameSize, "IoctlPartitionDataView")
{
  SetViewColor (240, 200, 255);
}


void IoctlPartitionDataView::AddControls ()
{
  float         LabelDividerSize;
  float         MarginAboveControl;
  BRect         TempRect;
  BTextControl *TextBoxPntr;
  float         WidthOfControls;

  RemoveControls ();

  /* We will be modifying this global variable to reflect our value. */

  g_IOCtlDataPntr = &g_IOCtlDataPartition_info;
  g_IOCtlDataSize = sizeof (g_IOCtlDataPartition_info);

  /* Do the display as 4 rows, with the size related items on the first
  and second row, ID stuff on the third, and the device path on the fourth. */

  m_HeightOfControlsArea = g_TextBoxHeight;
  m_HeightOfControlsArea = (int) (1.2 * m_HeightOfControlsArea);

  MarginAboveControl = (int) ((m_HeightOfControlsArea - g_TextBoxHeight) / 2);

  WidthOfControls = (int) be_plain_font->StringWidth (
    "MLogical Block Size: 888888888888888888 $FFFFFFFFM");

  LabelDividerSize = (int) be_plain_font->StringWidth (
    "Logical Block Size: ");

  /* The offset box.  Row 1, left side. */

  TempRect = Bounds ();
  TempRect.top += MarginAboveControl;
  TempRect.bottom = TempRect.top + g_TextBoxHeight;
  TempRect.left += g_MarginBetweenControls;
  TempRect.right = TempRect.left + WidthOfControls;

  TextBoxPntr = new BTextControl (TempRect,
    "Offset Control",
    "Offset (bytes):" /* label */,
    NULL /* text */,
    new BMessage (DLG_IOCTL_PARTITION),
    B_FOLLOW_LEFT | B_FOLLOW_TOP);

  TextBoxPntr->SetDivider (LabelDividerSize);
  AddChild (TextBoxPntr);
  TextBoxPntr->SetTarget (this);
  m_ControlViews[OFFSET_TEXT_BOX_INDEX] = TextBoxPntr;

  /* The size box.  Row 1, right side. */

  TempRect.left = TempRect.right + g_MarginBetweenControls;
  TempRect.right = TempRect.left + WidthOfControls;

  TextBoxPntr = new BTextControl (TempRect,
    "Size Control",
    "Size (bytes):" /* label */,
    NULL /* text */,
    new BMessage (DLG_IOCTL_PARTITION),
    B_FOLLOW_LEFT | B_FOLLOW_TOP);

  TextBoxPntr->SetDivider (LabelDividerSize);
  AddChild (TextBoxPntr);
  TextBoxPntr->SetTarget (this);
  m_ControlViews[SIZE_TEXT_BOX_INDEX] = TextBoxPntr;

  /* The ending offset calculated size box.  Row 2, left. */

  TempRect = Bounds ();
  TempRect.top += MarginAboveControl + m_HeightOfControlsArea;
  TempRect.bottom = TempRect.top + g_TextBoxHeight;
  TempRect.left += g_MarginBetweenControls;
  TempRect.right = TempRect.left + WidthOfControls;

  TextBoxPntr = new BTextControl (TempRect,
    "Ending Offset Control",
    "Ending Offset:" /* label */,
    NULL /* text */,
    NULL /* message */,
    B_FOLLOW_LEFT | B_FOLLOW_TOP);

  TextBoxPntr->SetDivider (LabelDividerSize);
  AddChild (TextBoxPntr);
  TextBoxPntr->SetEnabled (false);
  TextBoxPntr->SetTarget (this);
  m_ControlViews[ENDING_OFFSET_TEXT_BOX_INDEX] = TextBoxPntr;

  /* The block size box.  Row 2, right. */

  TempRect.left = TempRect.right + g_MarginBetweenControls;
  TempRect.right = TempRect.left + WidthOfControls;

  TextBoxPntr = new BTextControl (TempRect,
    "Block Size Control",
    "Logical Block Size:" /* label */,
    NULL /* text */,
    new BMessage (DLG_IOCTL_PARTITION),
    B_FOLLOW_LEFT | B_FOLLOW_TOP);

  TextBoxPntr->SetDivider (LabelDividerSize);
  AddChild (TextBoxPntr);
  TextBoxPntr->SetTarget (this);
  m_ControlViews[BLOCK_SIZE_TEXT_BOX_INDEX] = TextBoxPntr;

  /* The session ID box.  Row 3, left side. */

  TempRect = Bounds ();
  TempRect.top += MarginAboveControl + 2 * m_HeightOfControlsArea;
  TempRect.bottom = TempRect.top + g_TextBoxHeight;
  TempRect.left += g_MarginBetweenControls;
  TempRect.right = TempRect.left + WidthOfControls;

  TextBoxPntr = new BTextControl (TempRect,
    "SessionID Control",
    "Session ID:" /* label */,
    NULL /* text */,
    new BMessage (DLG_IOCTL_PARTITION),
    B_FOLLOW_LEFT | B_FOLLOW_TOP);

  TextBoxPntr->SetDivider (LabelDividerSize);
  AddChild (TextBoxPntr);
  TextBoxPntr->SetTarget (this);
  m_ControlViews[SESSION_ID_TEXT_BOX_INDEX] = TextBoxPntr;

  /* The partition ID box.  Row 3, right. */

  TempRect.left = TempRect.right + g_MarginBetweenControls;
  TempRect.right = TempRect.left + WidthOfControls;

  TextBoxPntr = new BTextControl (TempRect,
    "Partition Control",
    "Partition ID:" /* label */,
    NULL /* text */,
    new BMessage (DLG_IOCTL_PARTITION),
    B_FOLLOW_LEFT | B_FOLLOW_TOP);

  TextBoxPntr->SetDivider (LabelDividerSize);
  AddChild (TextBoxPntr);
  TextBoxPntr->SetTarget (this);
  m_ControlViews[PARTITION_ID_TEXT_BOX_INDEX] = TextBoxPntr;

  /* The device path box.  Row 4, all of row 4. */

  TempRect = Bounds ();
  TempRect.top += MarginAboveControl + 3 * m_HeightOfControlsArea;
  TempRect.bottom = TempRect.top + g_TextBoxHeight;
  TempRect.left += g_MarginBetweenControls;
  TempRect.right -= g_MarginBetweenControls;

  TextBoxPntr = new BTextControl (TempRect,
    "Device Path Control",
    "Device Path:" /* label */,
    NULL /* text */,
    new BMessage (DLG_IOCTL_PARTITION),
    B_FOLLOW_LEFT_RIGHT | B_FOLLOW_TOP);

  TextBoxPntr->SetDivider (LabelDividerSize);
  AddChild (TextBoxPntr);
  TextBoxPntr->SetTarget (this);
  m_ControlViews[DEVICE_PATH_TEXT_BOX_INDEX] = TextBoxPntr;

  CopyFromGlobalToDisplay ();
}


void IoctlPartitionDataView::CopyFromGlobalToDisplay ()
{
  char          TempString [80];
  BTextControl *TextBoxPntr;

  /* Copy all fields from the global g_IOCtlDataPartition_info into the
  appropriate text boxes. */

  TextBoxPntr = (BTextControl *) m_ControlViews[OFFSET_TEXT_BOX_INDEX];
  sprintf (TempString, "%qd $%qX", g_IOCtlDataPartition_info.offset,
    g_IOCtlDataPartition_info.offset);
  if (TextBoxPntr != NULL) TextBoxPntr->SetText (TempString);

  TextBoxPntr = (BTextControl *) m_ControlViews[SIZE_TEXT_BOX_INDEX];
  sprintf (TempString, "%qd $%qX", g_IOCtlDataPartition_info.size,
    g_IOCtlDataPartition_info.size);
  if (TextBoxPntr != NULL) TextBoxPntr->SetText (TempString);

  TextBoxPntr = (BTextControl *) m_ControlViews[ENDING_OFFSET_TEXT_BOX_INDEX];
  sprintf (TempString, "%qd $%qX",
    g_IOCtlDataPartition_info.offset + g_IOCtlDataPartition_info.size,
    g_IOCtlDataPartition_info.offset + g_IOCtlDataPartition_info.size);
  if (TextBoxPntr != NULL) TextBoxPntr->SetText (TempString);

  TextBoxPntr = (BTextControl *) m_ControlViews[BLOCK_SIZE_TEXT_BOX_INDEX];
  sprintf (TempString, "%ld", g_IOCtlDataPartition_info.logical_block_size);
  if (TextBoxPntr != NULL) TextBoxPntr->SetText (TempString);

  TextBoxPntr = (BTextControl *) m_ControlViews[SESSION_ID_TEXT_BOX_INDEX];
  sprintf (TempString, "%ld", g_IOCtlDataPartition_info.session);
  if (TextBoxPntr != NULL) TextBoxPntr->SetText (TempString);

  TextBoxPntr = (BTextControl *) m_ControlViews[PARTITION_ID_TEXT_BOX_INDEX];
  sprintf (TempString, "%ld", g_IOCtlDataPartition_info.partition);
  if (TextBoxPntr != NULL) TextBoxPntr->SetText (TempString);

  TextBoxPntr = (BTextControl *) m_ControlViews[DEVICE_PATH_TEXT_BOX_INDEX];
  if (TextBoxPntr != NULL)
    TextBoxPntr->SetText (g_IOCtlDataPartition_info.device);
}


void IoctlPartitionDataView::CopyFromDisplayToGlobal ()
{
  BTextControl *TextBoxPntr;

  /* Copy the contents of each text box to the appropriate field
  in the g_IOCtlDataPartition_info structure. */

  TextBoxPntr = (BTextControl *) m_ControlViews[OFFSET_TEXT_BOX_INDEX];
  if (TextBoxPntr != NULL)
    g_IOCtlDataPartition_info.offset = atoll (TextBoxPntr->Text ());

  TextBoxPntr = (BTextControl *) m_ControlViews[SIZE_TEXT_BOX_INDEX];
  if (TextBoxPntr != NULL)
    g_IOCtlDataPartition_info.size = atoll (TextBoxPntr->Text ());

  TextBoxPntr = (BTextControl *) m_ControlViews[BLOCK_SIZE_TEXT_BOX_INDEX];
  if (TextBoxPntr != NULL)
    g_IOCtlDataPartition_info.logical_block_size = atol (TextBoxPntr->Text ());

  TextBoxPntr = (BTextControl *) m_ControlViews[SESSION_ID_TEXT_BOX_INDEX];
  if (TextBoxPntr != NULL)
    g_IOCtlDataPartition_info.session = atol (TextBoxPntr->Text ());

  TextBoxPntr = (BTextControl *) m_ControlViews[PARTITION_ID_TEXT_BOX_INDEX];
  if (TextBoxPntr != NULL)
    g_IOCtlDataPartition_info.partition = atol (TextBoxPntr->Text ());

  TextBoxPntr = (BTextControl *) m_ControlViews[DEVICE_PATH_TEXT_BOX_INDEX];
  if (TextBoxPntr != NULL)
  {
    strncpy (g_IOCtlDataPartition_info.device, TextBoxPntr->Text (),
      sizeof (g_IOCtlDataPartition_info.device) - 1);
    g_IOCtlDataPartition_info.device [
      sizeof (g_IOCtlDataPartition_info.device) - 1] = 0;
  }
}


void IoctlPartitionDataView::MessageReceived (BMessage* MessagePntr)
{
  if (MessagePntr->what == DLG_IOCTL_PARTITION)
  {
    CopyFromDisplayToGlobal ();
    CopyFromGlobalToDisplay ();
  }
  else
  {
    GenericOptionsView::MessageReceived (MessagePntr);
  }
}



/******************************************************************************
 * This suboption displays an icon and lets you edit the desired size.
 * Internally, whenever you edit the size, a bitmap is created to match
 * that size.  The view displays that bitmap if it exists.
 */

class IoctlIconDataView : public GenericOptionsView
{
public:
  IoctlIconDataView (BRect FrameSize);

  void AddControls ();
  void AddSubOptions () {/* no suboptions */};
  void CopyFromGlobalToDisplay ();
  void CopyFromDisplayToGlobal ();

  void MessageReceived (BMessage* MessagePntr);
  void Draw (BRect UpdateRect);

  typedef enum ControlsEnum /* Indices into m_ControlViews for this view. */
  {
    SIZE_TEXT_BOX_INDEX = 0,
    BITMAP_ADDRESS_TEXT_BOX_INDEX
  } ControlIndices;
};


IoctlIconDataView::IoctlIconDataView (
  BRect FrameSize)
: GenericOptionsView (FrameSize, "IoctlIconDataView")
{
  SetViewColor (180, 180, 240);
}


void IoctlIconDataView::AddControls ()
{
  BTextControl *TextBoxPntr;
  float         MarginAboveControl;
  BRect         TempRect;

  RemoveControls ();

  /* We will be modifying this global variable to reflect our value. */

  g_IOCtlDataPntr = &g_IOCtlDataDevice_icon;
  g_IOCtlDataSize = sizeof (g_IOCtlDataDevice_icon);

  /* Add the text box displaying the icon size number. */

  m_HeightOfControlsArea = (int) (1.2 * g_TextBoxHeight);
  MarginAboveControl = (int) ((m_HeightOfControlsArea - g_TextBoxHeight) / 2);
  TempRect = Bounds ();
  TempRect.top += MarginAboveControl;
  TempRect.bottom = TempRect.top + g_TextBoxHeight;
  TempRect.left += g_MarginBetweenControls;
  TempRect.right = TempRect.left +
    (int) be_plain_font->StringWidth ("MSize: 8888M");

  TextBoxPntr = new BTextControl (TempRect,
    "Size Text Control",
    "Size:" /* label */,
    NULL /* text */,
    new BMessage (DLG_IOCTL_ICON),
    B_FOLLOW_LEFT | B_FOLLOW_TOP);

  TextBoxPntr->SetDivider (be_plain_font->StringWidth (
    "Size: ")); /* Leave space for the number. */

  AddChild (TextBoxPntr);
  TextBoxPntr->SetTarget (this);
  m_ControlViews[SIZE_TEXT_BOX_INDEX] = TextBoxPntr;

  /* Add the text box displaying the address of the bitmap. */

  TempRect.left = TempRect.right + g_MarginBetweenControls;
  TempRect.right = TempRect.left +
    (int) be_plain_font->StringWidth ("MAddress: $88888888M");

  TextBoxPntr = new BTextControl (TempRect,
    "Address Text Control",
    "Address:" /* label */,
    NULL /* text */,
    NULL /* message */,
    B_FOLLOW_LEFT | B_FOLLOW_TOP);

  TextBoxPntr->SetDivider (be_plain_font->StringWidth (
    "Address: ")); /* Leave space for the number. */

  AddChild (TextBoxPntr);
  TextBoxPntr->SetTarget (this);
  TextBoxPntr->SetEnabled (false);
  m_ControlViews[BITMAP_ADDRESS_TEXT_BOX_INDEX] = TextBoxPntr;

  CopyFromGlobalToDisplay ();
}


void IoctlIconDataView::CopyFromGlobalToDisplay ()
{
  char          TempString [80];
  BTextControl *TextBoxPntr;

  TextBoxPntr = (BTextControl *) m_ControlViews[SIZE_TEXT_BOX_INDEX];
  sprintf (TempString, "%ld", g_IOCtlDataDevice_icon.icon_size);
  if (TextBoxPntr != NULL) TextBoxPntr->SetText (TempString);

  TextBoxPntr = (BTextControl *) m_ControlViews[BITMAP_ADDRESS_TEXT_BOX_INDEX];
  sprintf (TempString, "$%lX", (long) g_IOCtlDataDevice_icon.icon_data);
  if (TextBoxPntr != NULL) TextBoxPntr->SetText (TempString);
}


void IoctlIconDataView::CopyFromDisplayToGlobal ()
{
  BRect         TempRect;
  BTextControl *TextBoxPntr;

  TextBoxPntr = (BTextControl *) m_ControlViews[SIZE_TEXT_BOX_INDEX];
  if (TextBoxPntr != NULL)
    g_IOCtlDataDevice_icon.icon_size = atol (TextBoxPntr->Text ());

  /* Allocate a bitmap for the icon to match the new size. */

  if (g_IOCtlDataDevice_icon.icon_size <= 0)
  {
    /* No icon for zero size, deallocate bitmap. */

    if (g_IOCtlDataDevice_iconBitmapPntr != NULL)
    {
      delete g_IOCtlDataDevice_iconBitmapPntr;
      g_IOCtlDataDevice_iconBitmapPntr = NULL;
    }
    g_IOCtlDataDevice_icon.icon_data = NULL;
  }
  else /* Have a positive size, allocate or reallocate bitmap to match. */
  {
    if (g_IOCtlDataDevice_iconBitmapPntr == NULL ||
    (g_IOCtlDataDevice_iconBitmapPntr->Bounds().right !=
    g_IOCtlDataDevice_icon.icon_size))
    {
      /* Need to (re)allocate the BBitmap. */

      if (g_IOCtlDataDevice_iconBitmapPntr != NULL)
        delete g_IOCtlDataDevice_iconBitmapPntr;

      /* Someone goofed when defining the BeOS coordinate system, with axis
      boundaries going through the center of the pixels and not in the cracks
      between the pixels.  They should have kept it the way it was in the Mac
      QuickDraw system - numbering is a lot more consistent, width of a
      rectangle in pixels would then be right-left, without off-by-one
      problems.  Windows is similar to QuickDraw.  Oh well, at least Be used
      floats for better scalability.  And while I'm at it, why all these fix
      sized system arrays (open files, semaphores, tasks)?  The Amiga did it
      better with linked lists for everything, so you never ran out of file
      handles etc.  Sigh. */

      TempRect.left = TempRect.top = 0;
      TempRect.right = TempRect.bottom = g_IOCtlDataDevice_icon.icon_size - 1;

      g_IOCtlDataDevice_iconBitmapPntr = new BBitmap (TempRect,
        B_CMAP8 /* icons are always 8 bit images with the standard palette */);
    }

    /* Get the address of the bitmap pixel array. */

    if (g_IOCtlDataDevice_iconBitmapPntr == NULL)
      g_IOCtlDataDevice_icon.icon_data = NULL;
    else
      g_IOCtlDataDevice_icon.icon_data =
        g_IOCtlDataDevice_iconBitmapPntr->Bits ();
  }

  Invalidate (); // Draw the bitmap or erase the old one.
}


void IoctlIconDataView::MessageReceived (BMessage* MessagePntr)
{
  if (MessagePntr->what == DLG_IOCTL_ICON)
  {
    CopyFromDisplayToGlobal ();
    CopyFromGlobalToDisplay ();
  }
  else
  {
    GenericOptionsView::MessageReceived (MessagePntr);
  }
}


void IoctlIconDataView::Draw (BRect UpdateRect)
{
  BPoint BitmapCorner;

  GenericOptionsView::Draw (UpdateRect);

  /* Draw the bitmap for the icon below the rest of the controls. */

  if (g_IOCtlDataDevice_iconBitmapPntr != NULL)
  {
    BitmapCorner.x = g_MarginBetweenControls;
    BitmapCorner.y = m_HeightOfControlsArea + g_MarginBetweenControls;
    DrawBitmap (g_IOCtlDataDevice_iconBitmapPntr, BitmapCorner);
  }
}


/******************************************************************************
 * This suboption displays and edits the device iterator argument to an IOCtl.
 * We just use text boxes as the user interface to the cookie and string.
 */

class IoctlIteratorDataView : public GenericOptionsView
{
public:
  IoctlIteratorDataView (BRect FrameSize);

  void AddControls ();
  void AddSubOptions () {/* no suboptions */};
  void CopyFromGlobalToDisplay ();
  void CopyFromDisplayToGlobal ();

  void MessageReceived (BMessage* MessagePntr);

  typedef enum ControlsEnum /* Indices into m_ControlViews for this view. */
  {
    COOKIE_TEXT_INDEX = 0,
    DEVICE_NAME_TEXT_INDEX
  } ControlIndices;
};


IoctlIteratorDataView::IoctlIteratorDataView (
  BRect FrameSize)
: GenericOptionsView (FrameSize, "IoctlIteratorDataView")
{
  SetViewColor (200, 240, 160);
}


void IoctlIteratorDataView::AddControls ()
{
  BTextControl *TextBoxPntr;
  float         MarginAboveControl;
  BRect         TempRect;

  RemoveControls ();

  /* We will be modifying this global variable to reflect our value. */

  g_IOCtlDataPntr = &g_IOCtlDataOpen_device_iterator;
  g_IOCtlDataSize = sizeof (g_IOCtlDataOpen_device_iterator);

  /* Add the text box displaying the cookie. */

  m_HeightOfControlsArea = (int) (1.2 * g_TextBoxHeight);
  MarginAboveControl = (int) ((m_HeightOfControlsArea - g_TextBoxHeight) / 2);
  TempRect = Bounds ();
  TempRect.top += MarginAboveControl;
  TempRect.bottom = TempRect.top + g_TextBoxHeight;
  TempRect.left += g_MarginBetweenControls;
  TempRect.right = TempRect.left +
    (int) be_plain_font->StringWidth ("MCookie: 888888888 $88888888M");

  TextBoxPntr = new BTextControl (TempRect,
    "Cookie Text Control",
    "Cookie:" /* label */,
    NULL /* text */,
    new BMessage (DLG_IOCTL_DEVICE_ITERATOR),
    B_FOLLOW_LEFT | B_FOLLOW_TOP);

  TextBoxPntr->SetDivider (be_plain_font->StringWidth (
    "Cookie: ")); /* Leave space for the cookie. */

  AddChild (TextBoxPntr);
  TextBoxPntr->SetTarget (this);
  m_ControlViews[COOKIE_TEXT_INDEX] = TextBoxPntr;

  /* Add the text box displaying the device name being iterated. */

  TempRect.left = TempRect.right + g_MarginBetweenControls;
  TempRect.right = Bounds().right - g_MarginBetweenControls;

  TextBoxPntr = new BTextControl (TempRect,
    "Device Iter Control",
    "Device:" /* label */,
    NULL /* text */,
    NULL /* message */,
    B_FOLLOW_LEFT_RIGHT | B_FOLLOW_TOP);

  TextBoxPntr->SetEnabled (false);
  TextBoxPntr->SetDivider (be_plain_font->StringWidth (
    "Device: ")); /* Leave space for the device name. */

  AddChild (TextBoxPntr);
  TextBoxPntr->SetTarget (this);
  m_ControlViews[DEVICE_NAME_TEXT_INDEX] = TextBoxPntr;

  CopyFromGlobalToDisplay ();
}


void IoctlIteratorDataView::CopyFromGlobalToDisplay ()
{
  char          TempString [80];
  BTextControl *TextBoxPntr;

  TextBoxPntr = (BTextControl *) m_ControlViews[COOKIE_TEXT_INDEX];
  sprintf (TempString, "%lu  $%lX", g_IOCtlDataOpen_device_iterator.cookie,
    g_IOCtlDataOpen_device_iterator.cookie);
  if (TextBoxPntr != NULL) TextBoxPntr->SetText (TempString);

  TextBoxPntr = (BTextControl *) m_ControlViews[DEVICE_NAME_TEXT_INDEX];
  if (TextBoxPntr != NULL)
    TextBoxPntr->SetText (g_IOCtlDataOpen_device_iterator.device);
}


void IoctlIteratorDataView::CopyFromDisplayToGlobal ()
{
  BTextControl *TextBoxPntr;

  TextBoxPntr = (BTextControl *) m_ControlViews[COOKIE_TEXT_INDEX];
  if (TextBoxPntr != NULL)
    g_IOCtlDataOpen_device_iterator.cookie = atol (TextBoxPntr->Text ());
}


void IoctlIteratorDataView::MessageReceived (BMessage* MessagePntr)
{
  if (MessagePntr->what == DLG_IOCTL_DEVICE_ITERATOR)
  {
    CopyFromDisplayToGlobal ();
    CopyFromGlobalToDisplay ();
  }
  else
  {
    GenericOptionsView::MessageReceived (MessagePntr);
  }
}



/******************************************************************************
 * This suboption chooses the IO control operation to be done.  There is
 * a pop up menu which lists the IOCtl operations possible, beside it is a
 * text box that shows the function code number (changed when you select from
 * the pop up menu).  Beside that is an Execute button.  On the next line is
 * a text box with the error number and description if available.  Under that
 * is a multiline text box with a description of the control operation.  The
 * suboption subview displays the parameters for the control operation,
 * essentially showing the global data buffer as the structure relevant for
 * each control operation (some operations don't have any parameters so no
 * suboption view for them).
 */

class IoctlOperationView : public GenericOptionsView
{
public:
  IoctlOperationView (BRect FrameSize);

  void AddControls ();
  void AddSubOptions ();
  void CopyFromGlobalToDisplay ();
  void CopyFromDisplayToGlobal ();

  void MessageReceived (BMessage* MessagePntr);
  void FrameResized (float width, float height);

  void ExecuteIOCtl ();

  typedef enum ControlsEnum /* Indices into m_ControlViews for this view. */
  {
    IO_CODE_POP_UP_MENU_INDEX = 0,
    IO_CODE_NUMBER_TEXT_INDEX,
    EXECUTE_BUTTON_INDEX,
    RETURN_CODE_TEXT_INDEX,
    IO_CODE_DESCRIPTION_TEXT_INDEX
  } ControlIndices;

}; // end class IoctlOperationView


IoctlOperationView::IoctlOperationView (
  BRect FrameSize)
: GenericOptionsView (FrameSize, "IoctlOperationView")
{
  SetViewColor (200, 0, 255);
}


void IoctlOperationView::AddControls ()
{
  BButton      *ButtonControlPntr;
  int           ControlWidth;
  float         HeightUsedByControls;
  unsigned int  i;
  float         MarginAboveControl;
  BMenuBar     *MenuBarPntr;
  BMenuItem    *MenuItemPntr;
  BPopUpMenu   *PopUpMenuPntr;
  BRect         TempRect;
  char          TempString [80];
  BTextControl *TextControlPntr;
  BTextView    *TextViewPntr;

  RemoveControls ();

  /* Figure out general height of the top half of the controls area,
  which has the pop up menu, a couple of text boxes (not including the
  big multiline one yet) and a button. */

  HeightUsedByControls = g_ButtonHeight;
  if (HeightUsedByControls < g_PopUpMenuHeight)
    HeightUsedByControls = g_PopUpMenuHeight;
  if (HeightUsedByControls < g_TextBoxHeight)
    HeightUsedByControls = g_TextBoxHeight;

  m_HeightOfControlsArea = (int) (HeightUsedByControls * 1.2);

  /* Add the operation pop-up menu. */

  MarginAboveControl =
    (int) ((m_HeightOfControlsArea - g_PopUpMenuHeight) / 2);
  TempRect = Bounds ();
  TempRect.top += MarginAboveControl;
  TempRect.bottom = TempRect.top + g_PopUpMenuHeight;
  TempRect.left += g_MarginBetweenControls;

  MenuBarPntr = new BMenuBar (TempRect, "IOCTL Op MenuBar",
    B_FOLLOW_LEFT | B_FOLLOW_TOP, B_ITEMS_IN_COLUMN,
    false /* true resize to fit currently marked item */);

  PopUpMenuPntr = new BPopUpMenu ("IOCTL Op PopUpMenu",
    true /* radio mode */, true /* take label from marked item */);

  /* Use the names of each IOCTL code as a pop-up menu item. */

  for (i = 0;
  i < sizeof (CodeDescriptions) / sizeof (CodeDescriptionStruct);
  i++)
  {
    MenuItemPntr = new BMenuItem (CodeDescriptions[i].name,
      new BMessage (CodeDescriptions[i].code));
    if (CodeDescriptions[i].code == B_SET_UNINTERRUPTABLE_IO)
      MenuItemPntr->SetMarked (true); // Preselect longest string for measuring.
    PopUpMenuPntr->AddItem (MenuItemPntr);
    MenuItemPntr->SetTarget (this);
  }

  MenuBarPntr->AddItem (PopUpMenuPntr);
  MenuBarPntr->ResizeToPreferred (); // Account for largest item's space.
  AddChild (MenuBarPntr);
  m_ControlViews[IO_CODE_POP_UP_MENU_INDEX] = MenuBarPntr;

  TempRect = Bounds ();
  TempRect.left = MenuBarPntr->Frame().right + g_MarginBetweenControls;

  /* Add the disabled text box displaying the operation code as a number. */

  MarginAboveControl =
    (int) ((m_HeightOfControlsArea - g_TextBoxHeight) / 2);
  ControlWidth = (int) be_plain_font->StringWidth ("MOpcode: 12345M");
  TempRect.top += MarginAboveControl;
  TempRect.bottom = TempRect.top + g_TextBoxHeight;
  TempRect.right = TempRect.left + ControlWidth;

  sprintf (TempString, "%ld", g_IOCtlOperationNumber);

  TextControlPntr = new BTextControl (TempRect,
    "Opcode Number Text Control",
    "Opcode:" /* label */,
    TempString /* text */,
    new BMessage (DLG_OPCODE_TEXT),
    B_FOLLOW_LEFT | B_FOLLOW_TOP);

  AddChild (TextControlPntr);
  TextControlPntr->SetTarget (this);
  TextControlPntr->SetDivider (be_plain_font->StringWidth (
    "Opcode: ")); /* Leave space for the number. */
  m_ControlViews[IO_CODE_NUMBER_TEXT_INDEX] = TextControlPntr;

  TempRect = Bounds ();
  TempRect.left = TextControlPntr->Frame().right + g_MarginBetweenControls;

  /* Add the Execute button. */

  MarginAboveControl = (int) ((m_HeightOfControlsArea - g_ButtonHeight) / 2);
  ControlWidth = (int) be_plain_font->StringWidth ("MExecuteM");
  TempRect.top += MarginAboveControl;
  TempRect.bottom = TempRect.top + g_ButtonHeight;
  TempRect.right = TempRect.left + ControlWidth;

  ButtonControlPntr = new BButton (TempRect, "IOCtl Execute Button",
    "Execute", new BMessage (DLG_IOCTL_EXECUTE),
    B_FOLLOW_TOP | B_FOLLOW_LEFT);

  AddChild (ButtonControlPntr);
  ButtonControlPntr->SetTarget (this);
  m_ControlViews[EXECUTE_BUTTON_INDEX] = ButtonControlPntr;

  /* Add the error code display box, full width, under all the buttons. */

  TempRect = Bounds ();
  TempRect.top += m_HeightOfControlsArea;
  MarginAboveControl =
    (int) ((m_HeightOfControlsArea - g_TextBoxHeight) / 2);
  TempRect.top += MarginAboveControl;
  TempRect.bottom = TempRect.top + g_TextBoxHeight;
  TempRect.left += g_MarginBetweenControls;
  TempRect.right -= g_MarginBetweenControls;

  TextControlPntr = new BTextControl (TempRect,
    "Opcode Error Text Control",
    "Error:" /* label */,
    "Text",
    NULL /* no message */,
    B_FOLLOW_LEFT_RIGHT | B_FOLLOW_TOP,
    B_WILL_DRAW | B_FRAME_EVENTS | B_FULL_UPDATE_ON_RESIZE);

  TextControlPntr->SetEnabled (false);
  AddChild (TextControlPntr);
  TextControlPntr->SetTarget (this);
  TextControlPntr->SetDivider (be_plain_font->StringWidth (
    "Error: ")); /* Leave space for the message. */
  m_ControlViews[RETURN_CODE_TEXT_INDEX] = TextControlPntr;

  /* Add the text box description of the IOCtl operation, full width, six
  lines in height, under all the other stuff so far. */

  TempRect = Bounds ();
  TempRect.top += 2 * m_HeightOfControlsArea;
  MarginAboveControl =
    (int) ((m_HeightOfControlsArea - g_TextBoxHeight) / 2);
  TempRect.top += MarginAboveControl;
  TempRect.bottom = TempRect.top + g_TextBoxHeight + 5 * g_LineOfTextHeight;
  TempRect.left += g_MarginBetweenControls;
  TempRect.right -= g_MarginBetweenControls;

  TextViewPntr = new BTextView (TempRect,
    "Opcode Description Text Control",
    BRect (0, 0, TempRect.right - TempRect.left, 10),
    B_FOLLOW_LEFT_RIGHT | B_FOLLOW_TOP,
    B_WILL_DRAW | B_FRAME_EVENTS | B_FULL_UPDATE_ON_RESIZE);

  TextViewPntr->MakeEditable (false);
  TextViewPntr->SetViewColor (239, 239, 239);
  AddChild (TextViewPntr);
  m_ControlViews[IO_CODE_DESCRIPTION_TEXT_INDEX] = TextViewPntr;

  /* Place the suboptions start point just below the final text box. */

  m_HeightOfControlsArea = 3 * m_HeightOfControlsArea + 5 * g_LineOfTextHeight;

  CopyFromGlobalToDisplay ();
}


/* Add a suboption view for the *Data argument of the particular IOCtl op. */

void IoctlOperationView::AddSubOptions ()
{
  BRect  TempRect;

  RemoveSubOptions ();
  TempRect = Bounds ();
  TempRect.top += m_HeightOfControlsArea;

  /* Default is to have no auxiliary data for that IOCtl, the suboptions
  will set these to point to their particular global data if they have any. */

  g_IOCtlDataPntr = NULL;
  g_IOCtlDataSize = 0;

  switch (g_IOCtlOperationNumber)
  {
    case B_GET_READ_STATUS:
    case B_GET_WRITE_STATUS:
    case B_FORMAT_DEVICE:
      m_SubOptionsView = new IoctlBoolDataView (TempRect);
      break;

    case B_GET_BIOS_DRIVE_ID:
      m_SubOptionsView = new IoctlByteDataView (TempRect);
      break;

    case B_GET_DEVICE_SIZE:
    case B_SET_DEVICE_SIZE:
      m_SubOptionsView = new IoctlSizetDataView (TempRect);
      break;

    case B_GET_MEDIA_STATUS:
      m_SubOptionsView = new IoctlStatustDataView (TempRect);
      break;

    case B_GET_GEOMETRY:
    case B_GET_BIOS_GEOMETRY:
      m_SubOptionsView = new IoctlGeometryDataView (TempRect);
      break;

    case B_GET_DRIVER_FOR_DEVICE:
      m_SubOptionsView = new IoctlPathnameDataView (TempRect);
      break;

    case B_GET_PARTITION_INFO:
    case B_SET_PARTITION:
      m_SubOptionsView = new IoctlPartitionDataView (TempRect);
      break;

    case B_GET_ICON:
      m_SubOptionsView = new IoctlIconDataView (TempRect);
      break;

    case B_GET_NEXT_OPEN_DEVICE:
      m_SubOptionsView = new IoctlIteratorDataView (TempRect);
      break;

    default: /* No suboption available for this kind of IOCtl. */
      break;
  }

  if (m_SubOptionsView != NULL)
  {
    AddChild (m_SubOptionsView);
    m_SubOptionsView->AddControls ();
  }
}


void IoctlOperationView::CopyFromGlobalToDisplay ()
{
  const char   *ErrorString;
  unsigned int  i;
  BMenuBar     *MenuBarPntr;
  BMenuItem    *MenuItemPntr = NULL;
  BMenu        *MenuPntr = NULL;
  char          TempString [1024];
  BTextControl *TextControlPntr;
  BTextView    *TextViewPntr;

  /* Update the pop-up menu's selected item to match the one with the
  matching IOCtl code.  If none, select the B_DEVICE_OP_CODES_END item. */

  MenuBarPntr = (BMenuBar *) m_ControlViews[IO_CODE_POP_UP_MENU_INDEX];
  if (MenuBarPntr != NULL)
    MenuPntr = MenuBarPntr->SubmenuAt (0);
  if (MenuPntr != NULL)
  {
    MenuItemPntr = MenuPntr->FindItem ((uint32) g_IOCtlOperationNumber);
    if (MenuItemPntr == NULL)
      MenuItemPntr = MenuPntr->FindItem ((uint32) B_DEVICE_OP_CODES_END);
  }
  if (MenuItemPntr != NULL)
    MenuItemPntr->SetMarked (true);

  /* Update the text control which is displaying the opcode number. */

  TextControlPntr = (BTextControl *) m_ControlViews[IO_CODE_NUMBER_TEXT_INDEX];
  if (TextControlPntr != NULL)
  {
    sprintf (TempString, "%ld", g_IOCtlOperationNumber);
    TextControlPntr->SetText (TempString);
  }

  /* Update the error code display text box. */

  TextControlPntr = (BTextControl *) m_ControlViews[RETURN_CODE_TEXT_INDEX];
  if (TextControlPntr != NULL)
  {
    /* Find the error message string to go with the code. */

    const int ErrorBase =
      B_DEVICE_ERROR_BASE; /* Avoid macro expansion problem. */
    const int MaxError =
      ErrorBase + sizeof (ErrorDescriptions) / sizeof (const char *);

    if (g_ErrorCodeFromLastOperation >= ErrorBase &&
    g_ErrorCodeFromLastOperation < MaxError)
      ErrorString =
      ErrorDescriptions [g_ErrorCodeFromLastOperation - ErrorBase];
    else
      ErrorString = "?";

    sprintf (TempString, "%d/$%X/%s  %s.",
      g_ErrorCodeFromLastOperation,
      g_ErrorCodeFromLastOperation,
      ErrorString,
      strerror (g_ErrorCodeFromLastOperation));

    TextControlPntr->SetText (TempString);
  }

  /* Update the description of the IOCtl operation. */

  TextViewPntr = (BTextView *) m_ControlViews[IO_CODE_DESCRIPTION_TEXT_INDEX];
  if (TextViewPntr != NULL)
  {
    sprintf (TempString, "Operation #%ld isn't a known one.",
      g_IOCtlOperationNumber);

    /* Search our collection of descriptions to see if it is present. */

    for (i = 0;
    i < sizeof (CodeDescriptions) / sizeof (CodeDescriptionStruct);
    i++)
    {
      if (CodeDescriptions[i].code == g_IOCtlOperationNumber)
      {
        sprintf (TempString, "%s = %s", CodeDescriptions[i].name, CodeDescriptions[i].description);
        break;
      }
    }

    TextViewPntr->SetText (TempString);
  }

  /* Update the suboption to display the correct type and current value. */

  AddSubOptions ();
}


void IoctlOperationView::CopyFromDisplayToGlobal ()
{
  BTextControl *TextControlPntr;

  /* Read out the IOCtl opcode from the user editable text box. */

  TextControlPntr = (BTextControl *) m_ControlViews[IO_CODE_NUMBER_TEXT_INDEX];
  if (TextControlPntr != NULL)
    g_IOCtlOperationNumber = atol (TextControlPntr->Text());
}


void IoctlOperationView::MessageReceived (BMessage* MessagePntr)
{
  if (MessagePntr->what >= 0 && MessagePntr->what < B_DEVICE_OP_CODES_END)
  { // Picked an item from the pop-up menu.
    g_IOCtlOperationNumber = MessagePntr->what;
    CopyFromGlobalToDisplay ();
  }
  else if (MessagePntr->what == B_DEVICE_OP_CODES_END)
  { // Reset menu to the last valid selection if invalid pop-up item chosen.
    CopyFromGlobalToDisplay ();
  }
  else if (MessagePntr->what == DLG_OPCODE_TEXT)
  { // User provided an opcode number in the text box.
    CopyFromDisplayToGlobal ();
    CopyFromGlobalToDisplay ();
  }
  else if (MessagePntr->what == DLG_IOCTL_EXECUTE)
  {
    CopyFromDisplayToGlobal ();
    if (m_SubOptionsView != NULL)
      m_SubOptionsView->CopyFromDisplayToGlobal ();
    ExecuteIOCtl ();
    CopyFromGlobalToDisplay (); // Includes updating suboptions by recreation.
  }
  else
  {
    GenericOptionsView::MessageReceived (MessagePntr);
  }
}


/* Override the virtual function so that the description text view's text
rectangle also gets resized so that it wraps the text to the new window
size. */

void IoctlOperationView::FrameResized (float width, float height)
{
  BRect        TempRect;
  BTextView   *TextPntr;

  /* Call the parent FrameResized, in case it actually does something. */

  GenericOptionsView::FrameResized (width, height);

  /* Tell the text view about the new margin size. */

  TextPntr = (BTextView *) m_ControlViews[IO_CODE_DESCRIPTION_TEXT_INDEX];
  if (TextPntr != NULL)
  {
    TempRect = Bounds ();
    TempRect.right -= g_MarginBetweenControls * 2;
    if (TempRect.right - TempRect.left < 20)
      TempRect.right = TempRect.left + 20; // Avoid crashing with small widths.
    TextPntr->SetTextRect (TempRect);
  }
}


void IoctlOperationView::ExecuteIOCtl ()
{
  int ReturnInt;

  ReturnInt = ioctl (g_DeviceFileDescriptor,
    g_IOCtlOperationNumber, g_IOCtlDataPntr, g_IOCtlDataSize);

  if (ReturnInt != 0) // Usually returns -1 for error.
    g_ErrorCodeFromLastOperation = errno;
  else
    g_ErrorCodeFromLastOperation = 0;
}



/******************************************************************************
 * This suboption chooses between read / write / ioctl operations.  There
 * is a set of radio buttons to select the operation.  One button is always
 * turned on while the others are off.
 */

class ReadWriteIoctlChoiceView : public GenericOptionsView
{
public:
  ReadWriteIoctlChoiceView (BRect FrameSize);

  void AddControls ();
  void AddSubOptions ();
  void CopyFromGlobalToDisplay ();
  void CopyFromDisplayToGlobal () {/* not meaningful */};

  void MessageReceived (BMessage* MessagePntr);

  typedef enum ControlsEnum /* Indices into m_ControlViews for this view. */
  {
    READ_CONTROL_INDEX = 0,
    WRITE_CONTROL_INDEX,
    IOCTL_CONTROL_INDEX
  } ControlIndices;
};


ReadWriteIoctlChoiceView::ReadWriteIoctlChoiceView (
  BRect FrameSize)
: GenericOptionsView (FrameSize, "ReadWriteIoctlChoiceView")
{
  SetViewColor (100, 255, 90);
}


void ReadWriteIoctlChoiceView::AddControls ()
{
  float         HeightUsedByControls;
  float         MarginAboveControl;
  BRadioButton *NewControlPntr;
  BRect         TempRect;
  float         WidthOfButtons;

  RemoveControls ();

  /* Figure out general height of the controls area and other sizes. */

  HeightUsedByControls = g_RadioButtonHeight;

  m_HeightOfControlsArea = (int) (HeightUsedByControls * 1.2);
  WidthOfButtons = (int) be_plain_font->StringWidth ("MMControlMM");

  /* Add the Read radio button. */

  MarginAboveControl =
    (int) ((m_HeightOfControlsArea - g_RadioButtonHeight) / 2);
  TempRect = Bounds ();
  TempRect.top += MarginAboveControl;
  TempRect.bottom = TempRect.top + g_RadioButtonHeight;
  TempRect.left += g_MarginBetweenControls;
  TempRect.right = TempRect.left + WidthOfButtons;

  NewControlPntr = new BRadioButton (TempRect,
    "RWI-Read Radio",
    "Read",
    new BMessage (DLG_RWIOCTL_READ),
    B_FOLLOW_TOP | B_FOLLOW_LEFT);

  AddChild (NewControlPntr);
  NewControlPntr->SetTarget (this);
  m_ControlViews[READ_CONTROL_INDEX] = NewControlPntr;

  /* Add the Write radio button. */

  TempRect.left = TempRect.right + g_MarginBetweenControls;
  TempRect.right = TempRect.left + WidthOfButtons;

  NewControlPntr = new BRadioButton (TempRect,
    "RWI-Write Radio",
    "Write",
    new BMessage (DLG_RWIOCTL_WRITE),
    B_FOLLOW_TOP | B_FOLLOW_LEFT);

  AddChild (NewControlPntr);
  NewControlPntr->SetTarget (this);
  m_ControlViews[WRITE_CONTROL_INDEX] = NewControlPntr;

  /* Add the Control radio button. */

  TempRect.left = TempRect.right + g_MarginBetweenControls;
  TempRect.right = TempRect.left + WidthOfButtons;

  NewControlPntr = new BRadioButton (TempRect,
    "RWI-Control Radio",
    "Control",
    new BMessage (DLG_RWIOCTL_CONTROL),
    B_FOLLOW_TOP | B_FOLLOW_LEFT);

  AddChild (NewControlPntr);
  NewControlPntr->SetTarget (this);
  m_ControlViews[IOCTL_CONTROL_INDEX] = NewControlPntr;

  CopyFromGlobalToDisplay ();
}


/* Remove old suboption and add the new suboption which corresponds
to the current read / write / control mode. */

void ReadWriteIoctlChoiceView::AddSubOptions ()
{
  BRect TempRect;

  RemoveSubOptions ();
  TempRect = Bounds ();
  TempRect.top += m_HeightOfControlsArea;

  switch (g_ReadWriteIOCtlMode)
  {
    case RWIO_READ:
      m_SubOptionsView = new ReadAndWriteView (TempRect,
        "Read Options View", false);
      break;

    case RWIO_WRITE:
      m_SubOptionsView = new ReadAndWriteView (TempRect,
        "Write Options View", true);
      break;

    case RWIO_IOCTL:
      m_SubOptionsView = new IoctlOperationView (TempRect);
      break;

    default:
      break;
  }

  if (m_SubOptionsView != NULL)
  {
    AddChild (m_SubOptionsView);
    m_SubOptionsView->AddControls ();
  }
}


void ReadWriteIoctlChoiceView::CopyFromGlobalToDisplay ()
{
  BRadioButton *ButtonPntr;

  /* Turn on the currently selected mode's radio button, the other
  ones will automatically turn off since they are in radio button mode. */

  if (g_ReadWriteIOCtlMode >= 0 &&
  g_ReadWriteIOCtlMode <= (int) IOCTL_CONTROL_INDEX)
  {
    ButtonPntr = (BRadioButton *) m_ControlViews [g_ReadWriteIOCtlMode];
    if (ButtonPntr != NULL)
      ButtonPntr->SetValue (B_CONTROL_ON);
  }

  /* Also turn on the suboptions associated with that button, since the
  suboptions are always visible for this type of control. */

  AddSubOptions (); /* Removes old and adds relevant ones. */
}


/* Override the standard message processing to handle our buttons. */

void ReadWriteIoctlChoiceView::MessageReceived (BMessage* MessagePntr)
{
  enum ReadWriteIOCtlModeEnum NewMode;

  NewMode = g_ReadWriteIOCtlMode;

  switch (MessagePntr->what)
  {
    case DLG_RWIOCTL_READ:
      NewMode = RWIO_READ;
      break;

    case DLG_RWIOCTL_WRITE:
      NewMode = RWIO_WRITE;
      break;

    case DLG_RWIOCTL_CONTROL:
      NewMode = RWIO_IOCTL;
      break;

    default:
      GenericOptionsView::MessageReceived (MessagePntr);
      break;
  }

  if (NewMode != g_ReadWriteIOCtlMode)
  {
    g_ReadWriteIOCtlMode = NewMode;
    CopyFromGlobalToDisplay ();
  }
}



/******************************************************************************
 * This suboption asks for the file name to open, and has a browse button
 * to bring up a file panel to do the same.  The open button becomes a close
 * button while the file is open.
 */

class DeviceChoiceView : public GenericOptionsView
{
public:
  DeviceChoiceView (BRect FrameSize);
  ~DeviceChoiceView ();

  void AddControls ();
  void AddSubOptions ();
  void CopyFromGlobalToDisplay ();
  void CopyFromDisplayToGlobal ();

  void MessageReceived (BMessage* MessagePntr);

  void DoOpenClose ();
  void BrowseForDeviceName ();
  void GotFileNameFromPanel (BMessage* MessagePntr);

  typedef enum ControlsEnum /* Indices into m_ControlViews for this view. */
  {
    DEVICE_NAME_CONTROL_INDEX = 0,
    WRITE_ENABLE_CONTROL_INDEX,
    CREATE_ENABLE_CONTROL_INDEX,
    OPEN_CONTROL_INDEX,
    BROWSE_CONTROL_INDEX
  } ControlIndices;

  BFilePanel *m_FilePanelPntr; /* Gets device to open from user. */
};


DeviceChoiceView::DeviceChoiceView (
  BRect FrameSize)
: GenericOptionsView (FrameSize, "DeviceChoiceView"),
  m_FilePanelPntr (NULL)
{
  SetViewColor (100, 90, 255);
}


DeviceChoiceView::~DeviceChoiceView ()
{
  if (m_FilePanelPntr != NULL)
  {
    delete m_FilePanelPntr;
    m_FilePanelPntr = NULL;
  }
}


void DeviceChoiceView::AddControls ()
{
  float HeightUsedByControls;
  float MarginAboveControl;
  float WidthOfButtons;
  BControl *NewControlPntr;
  BRect  TempRect;

  RemoveControls ();

  /* Figure out general height of the controls area and other sizes. */

  HeightUsedByControls = g_ButtonHeight;
  if (g_TextBoxHeight > HeightUsedByControls)
    HeightUsedByControls = g_TextBoxHeight;
  if (g_CheckBoxHeight > HeightUsedByControls)
    HeightUsedByControls = g_CheckBoxHeight;

  m_HeightOfControlsArea = (int) (HeightUsedByControls * 1.2);
  WidthOfButtons = (int) be_plain_font->StringWidth ("MMBrowseMM");

  /* Add the text entry box for the device name. */

  MarginAboveControl = (int) ((m_HeightOfControlsArea - g_TextBoxHeight) / 2);
  TempRect = Bounds ();
  TempRect.top += MarginAboveControl;
  TempRect.bottom = TempRect.top + g_TextBoxHeight;
  TempRect.left += g_MarginBetweenControls;
  TempRect.right -= 4 * WidthOfButtons + 6 * g_MarginBetweenControls;

  NewControlPntr = new BTextControl (TempRect,
    "Device Name Text Control",
    NULL /* label */,
    NULL /* text */,
    new BMessage (DLG_DEVICE_NAME_EDIT),
    B_FOLLOW_LEFT_RIGHT | B_FOLLOW_TOP);

  AddChild (NewControlPntr);
  NewControlPntr->SetTarget (this);
  m_ControlViews[DEVICE_NAME_CONTROL_INDEX] = NewControlPntr;

  /* Add the Create enable check box. */

  MarginAboveControl = (int) ((m_HeightOfControlsArea - g_CheckBoxHeight) / 2);
  TempRect.top = Bounds().top + MarginAboveControl;
  TempRect.bottom = TempRect.top + g_CheckBoxHeight;
  TempRect.left = TempRect.right + g_MarginBetweenControls;
  TempRect.right = TempRect.left + WidthOfButtons;

  NewControlPntr = new BCheckBox (TempRect,
    "Create Enable Checkbox",
    "Create?",
    new BMessage (DLG_CREATE_CHECK_BOX),
    B_FOLLOW_TOP | B_FOLLOW_RIGHT);

  AddChild (NewControlPntr);
  NewControlPntr->SetTarget (this);
  m_ControlViews[CREATE_ENABLE_CONTROL_INDEX] = NewControlPntr;

  /* Add the write enable check box. */

  TempRect.left = TempRect.right + g_MarginBetweenControls;
  TempRect.right = TempRect.left + WidthOfButtons;

  NewControlPntr = new BCheckBox (TempRect,
    "Write Enable Checkbox",
    "Write?",
    new BMessage (DLG_WRITE_CHECK_BOX),
    B_FOLLOW_TOP | B_FOLLOW_RIGHT);

  AddChild (NewControlPntr);
  NewControlPntr->SetTarget (this);
  m_ControlViews[WRITE_ENABLE_CONTROL_INDEX] = NewControlPntr;

  /* Add the Open/Close button. */

  MarginAboveControl = (int) ((m_HeightOfControlsArea - g_ButtonHeight) / 2);
  TempRect.top = Bounds().top + MarginAboveControl;
  TempRect.bottom = TempRect.top + g_ButtonHeight;
  TempRect.left = TempRect.right + g_MarginBetweenControls;
  TempRect.right = TempRect.left + WidthOfButtons;

  NewControlPntr = new BButton (TempRect,
    "Open/Close Button",
    "Open",
     new BMessage (DLG_OPEN_CLOSE_BUTTON),
     B_FOLLOW_TOP | B_FOLLOW_RIGHT);

  AddChild (NewControlPntr);
  NewControlPntr->SetTarget (this);
  m_ControlViews[OPEN_CONTROL_INDEX] = NewControlPntr;

  /* Add the Browse button. */

  TempRect.left = TempRect.right + g_MarginBetweenControls;
  TempRect.right = TempRect.left + WidthOfButtons;

  NewControlPntr = new BButton (TempRect, "Browse Devices Button",
    "Browse", new BMessage (DLG_BROWSE_DEVICES_BUTTON),
    B_FOLLOW_TOP | B_FOLLOW_RIGHT);

  AddChild (NewControlPntr);
  NewControlPntr->SetTarget (this);
  m_ControlViews[BROWSE_CONTROL_INDEX] = NewControlPntr;

  /* Display the initial device name. */

  CopyFromGlobalToDisplay ();
}


void DeviceChoiceView::AddSubOptions ()
{
  BRect  TempRect;

  RemoveSubOptions ();
  TempRect = Bounds ();
  TempRect.top += m_HeightOfControlsArea;

  m_SubOptionsView = new ReadWriteIoctlChoiceView (TempRect);

  AddChild (m_SubOptionsView);
  m_SubOptionsView->AddControls ();
}


void DeviceChoiceView::CopyFromGlobalToDisplay ()
{
  BButton      *ButtonPntr;
  BCheckBox    *CheckBoxPntr;
  BTextControl *TextControlPntr;
  BWindow      *WindowPntr;
  char          WindowTitle [B_PATH_NAME_LENGTH + 20];

  /* Update the text box with the device name. */

  TextControlPntr = (BTextControl *) m_ControlViews[DEVICE_NAME_CONTROL_INDEX];
  if (TextControlPntr != NULL)
  {
    TextControlPntr->SetText (g_DeviceName.String ());
    TextControlPntr->SetEnabled (g_DeviceFileDescriptor < 0);
  }

  /* Update our one and only window's title bar. */

  strcpy (WindowTitle, "Device Test - ");
  strcat (WindowTitle, g_DeviceName.String ());
  WindowPntr = be_app->WindowAt (0);
  if (WindowPntr != NULL)
    WindowPntr->SetTitle (WindowTitle);

  /* Update the write enable check box. */

  CheckBoxPntr = (BCheckBox *) m_ControlViews[WRITE_ENABLE_CONTROL_INDEX];
  if (CheckBoxPntr != NULL)
  {
    CheckBoxPntr->SetValue (g_OpenWithWriteEnabled ?
      B_CONTROL_ON : B_CONTROL_OFF);
    CheckBoxPntr->SetEnabled (g_DeviceFileDescriptor < 0);
  }

  /* Update the create enable check box. */

  CheckBoxPntr = (BCheckBox *) m_ControlViews[CREATE_ENABLE_CONTROL_INDEX];
  if (CheckBoxPntr != NULL)
  {
    CheckBoxPntr->SetValue (g_CreateRatherThanOpen ?
      B_CONTROL_ON : B_CONTROL_OFF);
    CheckBoxPntr->SetEnabled (g_DeviceFileDescriptor < 0);
  }

  /* Update the open/close button's title. */

  ButtonPntr = (BButton *) m_ControlViews[OPEN_CONTROL_INDEX];
  if (ButtonPntr != NULL)
    ButtonPntr->SetLabel ((g_DeviceFileDescriptor < 0) ?
      (g_CreateRatherThanOpen ? "Create" : "Open") : "Close");

  /* Disable the browse button when the device file is open. */

  ButtonPntr = (BButton *) m_ControlViews[BROWSE_CONTROL_INDEX];
  if (ButtonPntr != NULL)
    ButtonPntr->SetEnabled (g_DeviceFileDescriptor < 0);
}


void DeviceChoiceView::CopyFromDisplayToGlobal ()
{
  BCheckBox    *CheckBoxPntr;
  BTextControl *TextControlPntr;

  /* Get the device name from the text box. */

  TextControlPntr = (BTextControl *) m_ControlViews[DEVICE_NAME_CONTROL_INDEX];
  if (TextControlPntr != NULL)
    g_DeviceName = TextControlPntr->Text ();

  /* Get the write enabled status from the check box. */

  CheckBoxPntr = (BCheckBox *) m_ControlViews [WRITE_ENABLE_CONTROL_INDEX];
  if (CheckBoxPntr != NULL)
    g_OpenWithWriteEnabled = (CheckBoxPntr->Value () == B_CONTROL_ON);

  /* Get the create enabled status from the check box. */

  CheckBoxPntr = (BCheckBox *) m_ControlViews [CREATE_ENABLE_CONTROL_INDEX];
  if (CheckBoxPntr != NULL)
    g_CreateRatherThanOpen = (CheckBoxPntr->Value () == B_CONTROL_ON);
}


/* Override the standard message processing to handle our buttons. */

void DeviceChoiceView::MessageReceived (BMessage* MessagePntr)
{
  switch (MessagePntr->what)
  {
    case DLG_DEVICE_NAME_EDIT:
      CopyFromDisplayToGlobal ();
      CopyFromGlobalToDisplay ();
      break;

    case DLG_WRITE_CHECK_BOX:
      CopyFromDisplayToGlobal ();
      CopyFromGlobalToDisplay ();
      break;

    case DLG_CREATE_CHECK_BOX:
      CopyFromDisplayToGlobal ();
      CopyFromGlobalToDisplay ();
      break;

    case DLG_OPEN_CLOSE_BUTTON:
      CopyFromDisplayToGlobal ();
      DoOpenClose ();
      CopyFromGlobalToDisplay ();
      break;

    case DLG_BROWSE_DEVICES_BUTTON:
      BrowseForDeviceName ();
      break;

    case B_REFS_RECEIVED:
      GotFileNameFromPanel (MessagePntr);
      break;

    default:
      GenericOptionsView::MessageReceived (MessagePntr);
      break;
  }
}


/* When the open/close button is pressed, try to open or close the file
handle for the device as appropriate.  Then change the caption on the
button to reflect the next possible action ("close" if the file was
opened).  Also disable the file name and read / write checkbox while open. */

void DeviceChoiceView::DoOpenClose ()
{
  if (g_DeviceFileDescriptor >= 0)
  {
    /* Close the existing file. */
    close (g_DeviceFileDescriptor);
    g_DeviceFileDescriptor = -1;
    RemoveSubOptions ();
  }
  else /* Open the file using the name in the device text box. */
  {
    g_DeviceFileDescriptor = open (g_DeviceName.String (),
      (g_CreateRatherThanOpen ? O_CREAT : 0) |
      (g_OpenWithWriteEnabled ? O_RDWR : O_RDONLY),
      0777 /* access mode flags, rwxrwxrwx is 777 in octal. */);

    if (g_DeviceFileDescriptor < 0)
    {
      char ErrorMessage [B_PATH_NAME_LENGTH + 80];
      int ErrorNumber = errno;

      sprintf (ErrorMessage, "Unable to open file \"%s\"",
        g_DeviceName.String ());
      DisplayErrorMessage (ErrorMessage, ErrorNumber);
    }
    else /* Device sucessfully opened. */
    {
      AddSubOptions ();
    }
  }
}


void DeviceChoiceView::BrowseForDeviceName ()
{
  if (m_FilePanelPntr == NULL)
  {
    /* Create a new file panel.  First set up the entry ref stuff so that
    the file panel can open to show the initial directory.  Note that we
    have to create it after the window and view are up and running, otherwise
    the BMessenger won't point to a valid looper/handler. */

    BEntry DeviceEntry (g_DeviceName.String());
    BEntry DirectoryEntry;
    entry_ref DirectoryEntryRef;

    if (DeviceEntry.GetParent (&DirectoryEntry) == B_OK)
      DirectoryEntry.GetRef (&DirectoryEntryRef);
    else /* Plan B for backup technique */
      DeviceEntry.GetRef (&DirectoryEntryRef);

    BMessenger Messenger (this); /* Send messages to me. */

    m_FilePanelPntr = new BFilePanel (
      B_OPEN_PANEL /* mode */,
      &Messenger /* target for event messages */,
      &DirectoryEntryRef /* starting directory */,
      B_FILE_NODE,
      false /* true for multiple selections */,
      NULL /* canned message */,
      NULL /* ref filter */,
      false /* true for modal */,
      true /* true to hide when done */);
  }

  if (m_FilePanelPntr != NULL)
    m_FilePanelPntr->Show (); /* Answer returned later in a message. */
}


/* The file panel has selected a file, get the corresponding path name and
add it to the device name text box.  Ignore it if the device is open. */

void DeviceChoiceView::GotFileNameFromPanel (BMessage* MessagePntr)
{
  entry_ref EntryRef;

  if (g_DeviceFileDescriptor < 0 /* if file is closed */ &&
  MessagePntr->FindRef ("refs", &EntryRef) == B_OK)
  {
    BEntry MyEntry (&EntryRef, true /* traverse symlinks */);

    if (MyEntry.InitCheck () == B_OK)
    {
      BPath MyPath;

      if (MyEntry.GetPath (&MyPath) == B_OK)
      {
        g_DeviceName = MyPath.Path ();
        CopyFromGlobalToDisplay ();
      }
    }
  }
}



/******************************************************************************
 * Our kind of window doesn't do much at all.
 */

class DevTestWindow : public BWindow
{
public:
  DevTestWindow ();
  bool QuitRequested ();
};


DevTestWindow::DevTestWindow ()
: BWindow (
  BRect (30, 30, 600, 400),
  "AGMS" /* gets quickly replaced with a dynamically generated title */,
  B_TITLED_WINDOW, 0)
{
}


/* Override the virtual function to make the whole application
quit when the window is closed. */

bool DevTestWindow::QuitRequested ()
{
  be_app->PostMessage (B_QUIT_REQUESTED);
  return true;
}



/******************************************************************************
 * The main application class.  It just opens a window and lets the main app
 * message loop run, which doesn't do much other than wait for a quit message
 * since all the window's subviews handle their own button pushes etc.
 */

class DevTestApp : public BApplication
{
public:
  DevTestApp ();
  void ReadyToRun ();

  DevTestWindow *m_DevTestWindowPntr;
  DeviceChoiceView *m_DeviceChoiceViewPntr;
};


DevTestApp::DevTestApp ()
: BApplication ("application/x-vnd.agmsmith.DevTest"),
  m_DevTestWindowPntr (NULL),
  m_DeviceChoiceViewPntr (NULL)
{
}


void DevTestApp::ReadyToRun ()
{
  float         JunkFloat;
  BButton      *TempButtonPntr;
  BCheckBox    *TempCheckBoxPntr;
  font_height   TempFontHeight;
  BMenuBar     *TempMenuBarPntr;
  BMenuItem    *TempMenuItemPntr;
  BPopUpMenu   *TempPopUpMenuPntr;
  BRadioButton *TempRadioButtonPntr;
  BRect         TempRect;
  const char   *TempString = "Temp My Test";
  BTextControl *TempTextPntr;

  /* Create the new window and add the top level options view to it,
  that view will then add subviews for suboptions as they are selected. */

  m_DevTestWindowPntr = new DevTestWindow;
  if (m_DevTestWindowPntr == NULL)
  {
    DisplayErrorMessage ("Unable to create window.");
    Quit (); /* Tell the BApplication object to quit running. */
    return;
  }

  m_DevTestWindowPntr->Show ();

  /* Set the spacing between buttons and other controls to the width of
  the letter "M" in the user's desired font. */

 g_MarginBetweenControls = (int) be_plain_font->StringWidth ("M");

  /* Also find out how much space a line of text uses. */

  be_plain_font->GetHeight (&TempFontHeight);
  g_LineOfTextHeight =
    TempFontHeight.ascent + TempFontHeight.descent + TempFontHeight.leading;

  /* Find the height of a button, which seems to be larger than a text
  control and can make life difficult.  Make a temporary button, which
  is attached to our window so that it resizes to accomodate the font size. */

  TempRect = m_DevTestWindowPntr->Bounds ();
  TempButtonPntr = new BButton (TempRect, TempString, TempString, NULL);
  if (TempButtonPntr != NULL)
  {
    m_DevTestWindowPntr->Lock ();
    m_DevTestWindowPntr->AddChild (TempButtonPntr);
    TempButtonPntr->GetPreferredSize (&JunkFloat, &g_ButtonHeight);
    m_DevTestWindowPntr->RemoveChild (TempButtonPntr);
    m_DevTestWindowPntr->Unlock ();
    delete TempButtonPntr;
  }

  /* Find the height of a text box. */

  TempTextPntr = new BTextControl (TempRect, TempString, NULL /* label */,
    TempString, NULL);
  if (TempTextPntr != NULL)
  {
    m_DevTestWindowPntr->Lock ();
    m_DevTestWindowPntr->AddChild (TempTextPntr);
    TempTextPntr->GetPreferredSize (&JunkFloat, &g_TextBoxHeight);
    m_DevTestWindowPntr->RemoveChild (TempTextPntr);
    m_DevTestWindowPntr->Unlock ();
    delete TempTextPntr;
  }

  /* Find the height of a checkbox control. */

  TempCheckBoxPntr = new BCheckBox (TempRect, TempString, TempString, NULL);
  if (TempCheckBoxPntr != NULL)
  {
    m_DevTestWindowPntr->Lock ();
    m_DevTestWindowPntr->AddChild (TempCheckBoxPntr);
    TempCheckBoxPntr->GetPreferredSize (&JunkFloat, &g_CheckBoxHeight);
    m_DevTestWindowPntr->RemoveChild (TempCheckBoxPntr);
    m_DevTestWindowPntr->Unlock ();
    delete TempCheckBoxPntr;
  }

  /* Find the height of a radio button control. */

  TempRadioButtonPntr =
    new BRadioButton (TempRect, TempString, TempString, NULL);
  if (TempRadioButtonPntr != NULL)
  {
    m_DevTestWindowPntr->Lock ();
    m_DevTestWindowPntr->AddChild (TempRadioButtonPntr);
    TempRadioButtonPntr->GetPreferredSize (&JunkFloat, &g_RadioButtonHeight);
    m_DevTestWindowPntr->RemoveChild (TempRadioButtonPntr);
    m_DevTestWindowPntr->Unlock ();
    delete TempRadioButtonPntr;
  }

  /* Find the height of a pop-up menu. */

  TempMenuBarPntr = new BMenuBar (TempRect, TempString,
    B_FOLLOW_LEFT | B_FOLLOW_TOP, B_ITEMS_IN_COLUMN,
    true /* resize to fit items */);
  TempPopUpMenuPntr = new BPopUpMenu (TempString);
  TempMenuItemPntr = new BMenuItem (TempString, new BMessage (12345), 'g');

  if (TempMenuBarPntr != NULL && TempPopUpMenuPntr != NULL &&
  TempMenuItemPntr != NULL)
  {
    TempPopUpMenuPntr->AddItem (TempMenuItemPntr);
    TempMenuBarPntr->AddItem (TempPopUpMenuPntr);

    m_DevTestWindowPntr->Lock ();
    m_DevTestWindowPntr->AddChild (TempMenuBarPntr);
    TempMenuBarPntr->GetPreferredSize (&JunkFloat, &g_PopUpMenuHeight);
    m_DevTestWindowPntr->RemoveChild (TempMenuBarPntr);
    m_DevTestWindowPntr->Unlock ();
    delete TempMenuBarPntr; // It will delete contents too.
  }

  /* Create the first suboptions view - ask for the device name. */

  m_DeviceChoiceViewPntr = new DeviceChoiceView (
    m_DevTestWindowPntr->Bounds ());

  m_DevTestWindowPntr->Lock ();
  m_DevTestWindowPntr->AddChild (m_DeviceChoiceViewPntr);
  m_DeviceChoiceViewPntr->AddControls ();
  m_DevTestWindowPntr->Unlock ();
}



/******************************************************************************
 * Finally, the main program which drives it all.
 */

int main (int argc, char** argv)
{
  DevTestApp theApp;

  theApp.Run();

  if (g_IOCtlDataDevice_iconBitmapPntr != NULL)
  {
    delete g_IOCtlDataDevice_iconBitmapPntr;
    g_IOCtlDataDevice_iconBitmapPntr = NULL;
  }

  if (g_DataBufferPntr != NULL)
  {
    free (g_DataBufferPntr);
    g_DataBufferPntr = NULL;
    g_DataBufferSize = 0;
  }

  if (g_DeviceFileDescriptor >= 0)
  {
    close (g_DeviceFileDescriptor);
    g_DeviceFileDescriptor = -1;
  }

  return 0;
}
