/*
	undocumented PCI API
*/

#include <PCI.h>
#include <stdio.h>

extern long		get_nth_pci_info (
					long		index,	/* index into pci device table */
					pci_info 	*info	/* caller-supplied buffer for info */
				);
extern uint32		read_pci_config (
					uchar	bus,		/* bus number */
					uchar	device,		/* device # on bus */
					uchar	function,	/* function # in device */
					uchar	offset,		/* offset in configuration space */
					uchar	size		/* # bytes to read (1, 2 or 4) */
				);

/*
	missing PCI register definitions
*/

#define  PCI_STATUS_CAP_LIST	0x10	/* Support Capability List */

#define  PCI_HEADER_TYPE_NORMAL	0
#define  PCI_HEADER_TYPE_BRIDGE 1
#define  PCI_HEADER_TYPE_CARDBUS 2

#define PCI_CAPABILITY_LIST	0x34	/* Offset of first capability list entry */
#define PCI_CB_CAPABILITY_LIST	0x14

#define PCI_CAP_LIST_ID		0	/* Capability ID */
#define  PCI_CAP_ID_PM		0x01	/* Power Management */
#define  PCI_CAP_ID_AGP		0x02	/* Accelerated Graphics Port */
#define  PCI_CAP_ID_VPD		0x03	/* Vital Product Data */
#define  PCI_CAP_ID_SLOTID	0x04	/* Slot Identification */
#define  PCI_CAP_ID_MSI		0x05	/* Message Signalled Interrupts */
#define  PCI_CAP_ID_CHSWP	0x06	/* CompactPCI HotSwap */
#define PCI_CAP_LIST_NEXT	1	/* Next capability in the list */
#define PCI_CAP_FLAGS		2	/* Capability defined flags (16 bits) */
#define PCI_CAP_SIZEOF		4


#define PCI_AGP_STATUS		4	/* Status register */
#define  PCI_AGP_STATUS_RQ_MASK	0xff000000	/* Maximum number of requests - 1 */
#define  PCI_AGP_STATUS_SBA	0x0200	/* Sideband addressing supported */
#define  PCI_AGP_STATUS_64BIT	0x0020	/* 64-bit addressing supported */
#define  PCI_AGP_STATUS_FW	0x0010	/* FW transfers supported */
#define  PCI_AGP_STATUS_RATE4	0x0004	/* 4x transfer rate supported */
#define  PCI_AGP_STATUS_RATE2	0x0002	/* 2x transfer rate supported */
#define  PCI_AGP_STATUS_RATE1	0x0001	/* 1x transfer rate supported */
#define PCI_AGP_COMMAND		8	/* Control register */
#define  PCI_AGP_COMMAND_RQ_MASK 0xff000000  /* Master: Maximum number of requests */
#define  PCI_AGP_COMMAND_SBA	0x0200	/* Sideband addressing enabled */
#define  PCI_AGP_COMMAND_AGP	0x0100	/* Allow processing of AGP transactions */
#define  PCI_AGP_COMMAND_64BIT	0x0020 	/* Allow processing of 64-bit addresses */
#define  PCI_AGP_COMMAND_FW	0x0010 	/* Force FW transfers */
#define  PCI_AGP_COMMAND_RATE4	0x0004	/* Use 4x rate */
#define  PCI_AGP_COMMAND_RATE2	0x0002	/* Use 2x rate */
#define  PCI_AGP_COMMAND_RATE1	0x0001	/* Use 1x rate */
				
				
int find_capability( pci_info *info, int capability )
{
	int try_count = 48;
	uint16 status;
	uint8 pos;
	
	status = read_pci_config( info->bus, info->device, info->function, PCI_status, 2 );
	
	if( (status & PCI_STATUS_CAP_LIST) == 0 ) {
		printf( "No capability list\n" );
		return -1;
	}
		
	switch( info->header_type & PCI_header_type_mask ) {
	case PCI_HEADER_TYPE_NORMAL:
	case PCI_HEADER_TYPE_BRIDGE:
		pos = read_pci_config( info->bus, info->device, info->function, PCI_CAPABILITY_LIST, 1 );
		break;
	case PCI_HEADER_TYPE_CARDBUS:
		pos = read_pci_config( info->bus, info->device, info->function, PCI_CB_CAPABILITY_LIST, 1 );
		break;
	default:
		printf( "Unknown header type (%i)\n", info->header_type );
		return -1;
	}
	
	while( try_count-- && pos >= 0x40 ) {
		uint8 id;
		
		pos &= ~3;
		id = read_pci_config( info->bus, info->device, info->function, pos + PCI_CAP_LIST_ID, 1 );
		if( id == 0xff )
			break;
		if( id == capability )
			return pos;
		pos = read_pci_config( info->bus, info->device, info->function, pos + PCI_CAP_LIST_NEXT, 1 );
	}
	
	printf( "Capability not found\n" );
	
	return -1;
}

void show_agp_caps( pci_info *info, int idx )
{
	uint32 status, command;
	
	status = read_pci_config( info->bus, info->device, info->function, idx + PCI_AGP_STATUS, 4 );

	printf( "Status (%08lx): ", status );	
	if( (status & PCI_AGP_STATUS_SBA) != 0 )
		printf( "Sideband addressing " );
	if( (status & PCI_AGP_STATUS_64BIT) != 0 )
		printf( "64-bit " );
	if( (status & PCI_AGP_STATUS_FW) != 0 )
		printf( "FastWrite " );
	if( (status & PCI_AGP_STATUS_RATE4) != 0 )
		printf( "4x " );
	if( (status & PCI_AGP_STATUS_RATE2) != 0 )
		printf( "2x " );
	if( (status & PCI_AGP_STATUS_RATE1) != 0 )
		printf( "1x " );
		
	command = read_pci_config( info->bus, info->device, info->function, idx + PCI_AGP_COMMAND, 4 );
	printf( "\n" );
	
	printf( "Command (%08lx): ", command );	
	if( (command & PCI_AGP_COMMAND_SBA) != 0 )
		printf( "Sideband-addressing " );
	if( (command & PCI_AGP_COMMAND_AGP) != 0 )
		printf( "AGP-Enabled " );
	else
		printf( "AGP-Disabled " );
	if( (command & PCI_AGP_COMMAND_64BIT) != 0 )
		printf( "64-bit " );
	if( (command & PCI_AGP_COMMAND_FW) != 0 )
		printf( "FastWrite " );
	if( (command & PCI_AGP_COMMAND_RATE4) != 0 )
		printf( "4x " );
	if( (command & PCI_AGP_COMMAND_RATE2) != 0 )
		printf( "2x " );
	if( (command & PCI_AGP_COMMAND_RATE1) != 0 )
		printf( "1x " );
		
	printf( "\n" );
}

int main( int argc, char **argv )
{
	int i;
	
	printf( "AGP settings dump:\n" );
	
	for( i = 0; i < 100; ++i ) {
		long res;
		pci_info info;
		int idx;
		
		res = get_nth_pci_info( i, &info );
		
		if( res != B_OK )
			break;
			
		printf( "Device %d, vendor_id %04x, device_id %04x,\
(bus %i, device %i, function %i) - ", info.device, info.vendor_id, 
			info.device_id, info.bus, info.device, info.function );

		idx = find_capability( &info, PCI_CAP_ID_AGP );			
		if( idx < 0 )
			continue;
			
		printf( "AGP device found:\n" );
		show_agp_caps( &info, idx );
		printf( "\n" );
	}
	
	return 0;
}
