
//		Copyright © 1999		Matthew S. Chartier (mattc@ic.net)
//
//		This program is free software; you can redistribute it and/or modify
//		it under the terms of the GNU General Public License as published by
//		the Free Software Foundation; either version 1, or (at your option)
//		any later version.
//
//		This program is distributed in the hope that it will be useful,
//		but WITHOUT ANY WARRANTY; without even the implied warranty of
//		MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the
//		GNU General Public License for more details.
//
//		You should have received a copy of the GNU General Public License
//		along with this program; if not, write to the Free Software
//		Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//


#include <KernelExport.h>
#include <Drivers.h>
#include <Errors.h>
#include <ISA.h>
#include <BeBuild.h>

#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>

#include "scsi_ppa.h"
#include "icons.h"


isa_module_info* GLB_bus;



/*----------------------------------------------------------*/

void PPA_TRACE(char* s)
	{
	int fd;
return;
	if((fd = open("/boot/home/zip100.log",O_APPEND|O_RDWR)) > 0)
		{
		write(fd,s,strlen(s));
		close(fd);
		}
	}
	
/*----------------------------------------------------------*/

status_t formatDisk(struct parport* cookie)
	{
	char buf[512];
	memset(buf,0,sizeof(buf));

	return(scsi_ppa_Write(cookie,buf,sizeof(buf),0,1));
	}	


/*----------------------------------------------------------*/

status_t
init_hardware (void)
	{
	status_t ret = 1;
	struct parport* port = NULL;

	get_module(B_ISA_MODULE_NAME,(module_info**)&GLB_bus);

	if(parport_init())
		if((port = parport_getport(0)))
			{
			parport_claimport(port);
			ret = scsi_ppa_Init(port);
			parport_releaseport(port);
			}

	put_module(B_ISA_MODULE_NAME);
	return(!ret ? B_OK : B_ERROR);
	}


/*----------------------------------------------------------*/

status_t
init_driver (void)
	{
	get_module(B_ISA_MODULE_NAME,(module_info**)&GLB_bus);
	return(B_NO_ERROR);
	}


/*----------------------------------------------------------*/

void
uninit_driver (void)
	{
	put_module(B_ISA_MODULE_NAME);
	}	


/*----------------------------------------------------------*/
	
static status_t
ppa_device_open (const char *name, uint32 flags, void** cookie)
	{
	static char buf[512],out[80];
	status_t r,ret = B_ERROR;
	InquiryData inq;
	struct parport* port = parport_getport(0);

	parport_claimport(port);

	if((r = scsi_ppa_TestUnitReady(port)))
		if(!(r = scsi_ppa_Inquiry(port,&inq)))
			r = scsi_ppa_Read(port,buf,sizeof(buf),0,1);

	parport_releaseport(port);

	*cookie = !r ? port : NULL;
	return(*cookie ? B_OK : B_ERROR);
	}


/*----------------------------------------------------------*/

static status_t
ppa_device_read(void* cookie,off_t position,void* buf,size_t* num_bytes)
	{
	size_t s = *num_bytes;
	status_t ret = B_IO_ERROR,t;
	ulong block  = position / 512;
	struct parport* port = NULL;

	if((port = (struct parport*)cookie))
    	{      
      	parport_claimport(port);
	
		if(!(t = scsi_ppa_Read(port,buf,s,block,s / 512)))
			ret = B_NO_ERROR;

		parport_releaseport(port);
		}

	return(ret);
	}


/*----------------------------------------------------------*/

static status_t
ppa_device_write(void* cookie,off_t position,const void* buffer,size_t* num_bytes)
	{
	status_t ret = B_IO_ERROR;
	ulong block  = position / 512;
	size_t s = *num_bytes;
	struct parport* port = NULL;

	if((port = (struct parport*)cookie))
    	{      
      	parport_claimport(port);

		if(!scsi_ppa_Write(port,buffer,s,block,s / 512))
			ret = B_NO_ERROR;

		parport_releaseport(port);
		}

	return(ret);
 	}


/*----------------------------------------------------------*/

static status_t
ppa_device_control (void* cookie, uint32 op, void* arg, size_t len)
	{
	status_t ret = B_BAD_VALUE;
	device_geometry* geo;
	device_icon* icon;
	struct parport* port = NULL;
	
	port = (struct parport*)cookie;

	switch(op)
		{
		case B_GET_GEOMETRY:
			geo = (device_geometry*)arg;
			geo->sectors_per_track	= 196608;
			geo->cylinder_count		= 1;
			geo->head_count			= 1;
			geo->bytes_per_sector	= 512;
			geo->removable 			= true;
			geo->read_only			= false;
			geo->device_type		= B_DISK;
			geo->write_once			= false;
			ret						= B_NO_ERROR;	
			break;
		
		case B_FORMAT_DEVICE:
			parport_claimport(port);
			ret = !formatDisk(port) ? B_OK : B_IO_ERROR;		
			parport_releaseport(port);
			break;
			
		case B_EJECT_DEVICE:
			parport_claimport(port);
			scsi_ppa_Eject(port);
			parport_releaseport(port);
			break;
			
		case B_GET_DEVICE_SIZE:
			*(size_t*)arg 			= 196608 * 512;
			ret						= B_NO_ERROR;
			break;	

		case B_GET_ICON:
			icon = (device_icon*)arg;
			if(icon->icon_size == 32)
				memcpy(icon->icon_data,&iconLarge,sizeof(iconLarge));
			else
				memcpy(icon->icon_data,&iconSmall,sizeof(iconSmall));
				
			ret = B_NO_ERROR;
			break;
		}
	return(ret);
	}


/*----------------------------------------------------------*/

static status_t
ppa_device_close (void* cookie)
	{
	return B_OK;
	}


/*----------------------------------------------------------*/

static status_t
ppa_device_free (void* cookie)
	{
	return B_OK;
	}


/*----------------------------------------------------------*/

static const char *my_device_name[] = 
	{
	"disk/ppa/zip100/raw",
	NULL
	};


/*----------------------------------------------------------*/

device_hooks my_device_hooks = 
	{
	ppa_device_open, 			/* -> open entry point */
	ppa_device_close, 			/* -> close entry point */
	ppa_device_free,			/* -> free cookie */
	ppa_device_control, 		/* -> control entry point */
	ppa_device_read,			/* -> read entry point */
	ppa_device_write			/* -> write entry point */
	};


/*----------------------------------------------------------*/

const char**
publish_devices()
	{
	return my_device_name;
	}


/*----------------------------------------------------------*/

device_hooks*
find_device(const char* name)
	{
	return &my_device_hooks;
	}

