
//		Copyright © 1999		Matthew S. Chartier (mattc@ic.net)
//
//		This program is free software; you can redistribute it and/or modify
//		it under the terms of the GNU General Public License as published by
//		the Free Software Foundation; either version 1, or (at your option)
//		any later version.
//
//		This program is distributed in the hope that it will be useful,
//		but WITHOUT ANY WARRANTY; without even the implied warranty of
//		MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the
//		GNU General Public License for more details.
//
//		You should have received a copy of the GNU General Public License
//		along with this program; if not, write to the Free Software
//		Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
//		Portions of this file Copyright Grant R. Guenther (grant@torque.net)
//										David Cambell (cambell@torque.net)		


#include <stdio.h>
#include "parport.h"
#include "ppa_lowlevel.h"



/*----------------------------------------------------------*/
static void udelay(bigtime_t delay)
	{
	snooze(delay);
	}


/*----------------------------------------------------------*/
void outsl(int addr,long* buffer,int len)
	{
	int l;
	for(l = 0;l < len;l++,buffer++)
		write_io_32(addr,*buffer);
	}

/*----------------------------------------------------------*/
void outsb(int addr,char* buffer,int len)
	{
	int l;
	for(l = 0;l < len;l++,buffer++)
		write_io_8(addr,*buffer);
	}

/*----------------------------------------------------------*/
void insl(int addr,long* buffer,int len)
	{
	int l;
	for(l = 0;l < len;l++,buffer++)
		*buffer = read_io_32(addr);
	}

/*----------------------------------------------------------*/
void insb(int addr,char* buffer,int len)
	{
	int l;
	for(l = 0;l < len;l++,buffer++)
		*buffer = read_io_8(addr);
	}


/*----------------------------------------------------------*/
static void ppa_d_pulse(struct parport* port,unsigned char b)
	{
	w_dtr(port->base,b);
	w_ctr(port->base,0xc);
	w_ctr(port->base,0xe);
	w_ctr(port->base,0xc);
	w_ctr(port->base,0x4);
	w_ctr(port->base,0xc);
	}

/*----------------------------------------------------------*/
static void ppa_c_pulse(struct parport* port,unsigned char b)
	{
	w_dtr(port->base,b);
	w_ctr(port->base,0x4);
	w_ctr(port->base,0x6);
	w_ctr(port->base,0x4);
	w_ctr(port->base,0xc);
	}

/*----------------------------------------------------------*/
static void ppa_reset_pulse(struct parport* port)
	{
	w_dtr(port->base,0x40);
	w_ctr(port->base,0x8);
	udelay(30);
	w_ctr(port->base,0xc);
	}

/*----------------------------------------------------------*/
static int ppa_select(struct parport* port,int target)
	{
	int k = PPA_SELECT_TMO;
	do
		{
		k--;
		}
	while((r_str(port->base) & 0x40) && (k));
	
	if(!k)
		return(0);
		
	w_dtr(port->base,(1 << target));
	w_ctr(port->base,0xe);
	w_ctr(port->base,0xc);
	w_dtr(port->base,0x80);
	w_ctr(port->base,0x8);
	
	k = PPA_SELECT_TMO;		
	do
		{
		k--;
		}
	while(!(r_str(port->base) & 0x40) && (k));

	if(!k)
		return(0);
		
	return(1);	
	}
	

/*----------------------------------------------------------*/
static void epp_reset(struct parport* port)
	{
	int i;
	
	i = r_str(port->base);
	w_str(port->base,i);
	w_str(port->base,i & 0xfe);
	}	

/*----------------------------------------------------------*/
static void ecp_sync(struct parport* port)
	{
	int i;
	
	if((r_ecr(port->base) & 0xe0) != 0x80)
		return;
	for(i=0;i<100;i++)
		{
		if(r_ecr(port->base) & 0x01)
			return;
		udelay(5);
		}
#ifdef __USERSPACE
	printf("ppa: ECP symc failed\n");
#endif
	}
	

/*----------------------------------------------------------*/
static void ppa_disconnect(struct parport* port)
	{
	ppa_d_pulse(port,0);
	ppa_d_pulse(port,0x3c);
	ppa_d_pulse(port,0x20);
	ppa_d_pulse(port,0xf);
	}

/*----------------------------------------------------------*/
static void ppa_connect(struct parport* port,int flag)
	{
	ppa_c_pulse(port,0);
	ppa_c_pulse(port,0x3c);
	ppa_c_pulse(port,0x20);
	
	if((flag == CONNECT_EPP_MAYBE) && IN_EPP_MODE(port))
		ppa_c_pulse(port,0xcf);
	else
		ppa_c_pulse(port,0x8f);
	}

/*----------------------------------------------------------*/
static int ppa_wait(struct parport* port)
	{
	int k;
	unsigned char r;
	
	k = PPA_SPIN_TMO;
	do
		{
		r = r_str(port->base);
		k--;
		udelay(1);
		}
	while(!(r & 0x80) && (k));
	
	if(k)
		return(r & 0xf0);

#ifdef __USERSPACE
	printf("ppa: timeout in ppa_wait\n");
#endif
	return(0);
	}


/*----------------------------------------------------------*/
static int ppa_byte_out(struct parport* port,const char* buffer,int len)
	{
	int i;
	for(i = len;i;i--)
		{
		w_dtr(port->base,*buffer++);
		w_ctr(port->base,0xe);
		w_ctr(port->base,0xc);
		}
		
	return(1);	
	}

	
/*----------------------------------------------------------*/
static int ppa_nibble_in(struct parport* port,char* buffer,int len)
	{
	unsigned char h;
	for(;len;len--)
		{
		w_ctr(port->base,0x4);
		h = r_str(port->base) & 0xf0;
		w_ctr(port->base,0x6);
		
		*buffer++ = h | ((r_str(port->base) & 0xf0) >> 4);
		}
		
	return(1);
	}


/*----------------------------------------------------------*/
static int ppa_out(struct parport* port,char* buffer,int len)
	{
	int r;
	r = ppa_wait(port);
	
	if((r & 0x50) != 0x40)
		return(0);

	if(port->modes & PARPORT_MODE_PCEPP)
		{
		epp_reset(port);
		w_ctr(port->base,0x4);
		
		if(!(((long)buffer | len) & 0x03))
			outsl(port->base + 4,(long*)buffer,len >> 2);	
		else
			outsb(port->base + 4,buffer,len);

		w_ctr(port->base,0xc);
		r = !(r_str(port->base) & 0x01);
		w_ctr(port->base,0xc);
		ecp_sync(port);
		}	
	else
		r = ppa_byte_out(port,buffer,len);

	return(r);
	}

/*----------------------------------------------------------*/
static int ppa_in(struct parport* port,char* buffer,int len)
	{
	int r = ppa_wait(port);
	
	if((r & 0x50) != 0x50)
		return(0);
			
	if(port->modes & PARPORT_MODE_PCEPP)
		{
		epp_reset(port);
		w_ctr(port->base,0x24);
		
		if(!(((long)buffer | len) & 0x03))
			insl(port->base + 4,(long*)buffer,len >> 2);
		else
			insb(port->base + 4,buffer,len);

		w_ctr(port->base,0x2c);
		r = !(r_str(port->base) & 0x01);
		w_ctr(port->base,0x2c);
		ecp_sync(port);
		}
	else
		{
		r = ppa_nibble_in(port,buffer,len);
		w_ctr(port->base,0xc);
		}
	
	return(r);
	}


/*----------------------------------------------------------*/
static int device_check(struct parport* port)
	{
	int loop, status, k,ret = 1;
	unsigned char j;
	static char cmd[6] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
	
	for(loop = 0; loop < 8; loop++)
		{
		ppa_connect(port,CONNECT_EPP_MAYBE);
		if(ppa_select(port,loop))
			{
#ifdef __USERSPACE		
			printf("ppa: Found device at ID %d\n",loop);			
#endif
			status = 1;
			w_ctr(port->base,0x0c);
			for(j = 0;(j < 6) && (status); j++)
				status = ppa_out(port,cmd,1);

			if(status)
				{
				w_ctr(port->base,0x0c);
				k = 1000000;
				do
					{
					j = r_str(port->base);
					k--;
					udelay(1);
					}
				while(!(j & 0x80) && (k));
				
				j &= 0xf0;
				if(j != 0xf0)
					{
#ifdef __USERSPACE					
					printf("Unable to establish communication\n");
#endif
					}
				else
					ret = 0;
				}
			}
		ppa_disconnect(port);
		ppa_connect(port,CONNECT_EPP_MAYBE);
		ppa_reset_pulse(port);
		udelay(1000);
		ppa_disconnect(port);
		udelay(1000);
		}

	return(ret);	
	}

/*----------------------------------------------------------*/
static int ppa_send_command(struct parport* port,Scsi_Cmnd* cmd)
	{
	int k;
	w_ctr(port->base,0x0c);

	for(k=0;k < cmd->cmd_len;k++)
		{
		if(!ppa_out(port,&cmd->cmnd[k],1))
			return(0);
		}

	return(1);
	}


/*----------------------------------------------------------*/
static int ppa_completion(struct parport* port,Scsi_Cmnd* cmd)
	{
	bool needDelay = false;
	unsigned char r,v,*ptr;
	int bulk,status,fast,size;

	v = cmd->cmnd[0];
	bulk = ((v == READ_6) || (v == READ_10) || (v == WRITE_6) || (v == WRITE_10));
	size = cmd->request_bufflen;
	ptr = cmd->request_buffer;

	needDelay = (size >= 1024);

	r   = (r_str(port->base) & 0xf0);

	while(r != (unsigned char)0xf0)
		{
		if(((r & 0xc0) != 0xc0))
			return(-1);
	
		fast = (bulk && (size >= PPA_BURST_SIZE)) ? PPA_BURST_SIZE : 1;

		if(r == (unsigned char)0xc0)
			status = ppa_out(port,ptr,fast);
		else
			status = ppa_in(port,ptr,fast);

		ptr  += fast;
		size -= fast;
		
		if(!status)
			return(-1);

		if(needDelay)
			udelay(5000);		
		
		r = (r_str(port->base) & 0xf0);
		if(!(r & 0x80))
			return(0);
		}

	return(1);	
	}


/*----------------------------------------------------------*/
static int ppa_engine(struct parport* port,Scsi_Cmnd* cmd)
	{
	int retv, ret = 0;
	unsigned char j = 0,h = 0;

	if(cmd->status != STATUS_FAILED)
		{
		switch(cmd->phase)
			{
			case 0:			/* Phase 0 - wait for device */
				cmd->phase++;
				ret = 1;
				break;
				
			case 1:			/* Phase 1 - connect */
				retv = 2;
				ppa_connect(port,CONNECT_EPP_MAYBE);
				w_ctr(port->base,0xe);
				
				if((r_str(port->base) & 0x08) == 0x08)
					retv--;
					
				w_ctr(port->base,0xc);
				if((r_str(port->base) & 0x08) == 0x00)
					retv--;
					
				if(retv)
					cmd->status = STATUS_FAILED;		
				else
					{
					cmd->phase++;
					ret = 1;
					}
				break;
				
			case 2:			/* Phase 2 - now talking to scsi bus */
				if(!ppa_select(port,cmd->target))
					cmd->status = STATUS_FAILED;
				else
					{
					cmd->phase++;
					ret = 1;
					}
				break;
				
			case 3:			/* Phase 3 - ready to accept a command */
				w_ctr(port->base,0xc);
				if(!(r_str(port->base) & 0x80))
					ret = 1;
				else
					if(ppa_send_command(port,cmd))
						{
 						cmd->phase++;
						ret = 1;
						}
				break;
				
			case 4:			/* Phase 4 - setup scatter/gather buffers */
				cmd->phase++;
				ret = 1;
				break;
				
			case 5:			/* Phase 5 - data transfer stage */
				w_ctr(port->base,0x0c);
				if(!(r_str(port->base) & 0x80))
					ret = 1;
				else
					{
					retv = ppa_completion(port,cmd);
					ret  = retv == -1 ? 0 : 1;
					cmd->phase += ret;
					}
				break;
				
			case 6:			/* Phase 6 - read status/message */
				cmd->result = DID_OK << 16;
				if(ppa_wait(port) == (unsigned char)0xf0)
					if(ppa_in(port,&j,1))
						if(ppa_wait(port) == (unsigned char)0xf0)
							{
							ppa_in(port,&h,1);
							cmd->result = (DID_OK << 16) + (h << 8) + (j & STATUS_MASK);
							ret = 0;
							}
				break;
			}
		}
		
	return(ret);	
	}


/*----------------------------------------------------------*/
int ppa_command(struct parport* port,Scsi_Cmnd* cmd)
	{
	cmd->result = DID_ERROR << 16;
	cmd->status = 0;
	cmd->phase  = 0;

	while(ppa_engine(port,cmd))
		;
				
	if(cmd->phase)
		ppa_disconnect(port);

	return(cmd->result);
	}


/*----------------------------------------------------------*/
int ppa_init(struct parport* port)
	{
	int retv;
	
	ppa_disconnect(port);
	ppa_connect(port,CONNECT_NORMAL);	
	
	retv = 2;
	
	w_ctr(port->base,0xe);
	if((r_str(port->base) & 0x08) == 0x08)
		retv--;
		
	w_ctr(port->base,0xc);
	if((r_str(port->base) & 0x08) == 0x00)
		retv--;
		
	if(!retv)
		ppa_reset_pulse(port);
	
	udelay(1000);
	ppa_disconnect(port);
	udelay(1000);
	
	if(!retv)
		retv = device_check(port);
		
	return(retv);
	}

	
/*----------------------------------------------------------*/
int ppa_SendCommand(struct parport* port,void* cmd,uint cmdlen,void* data,uint datalen)
	{
   	Scsi_Cmnd internalCmd;

   	memset(&internalCmd,0,sizeof(internalCmd));

   	internalCmd.target          = 6;
   	internalCmd.cmnd            = (char*)cmd;
   	internalCmd.cmd_len         = cmdlen;
   	internalCmd.request_buffer  = (char*)data;
   	internalCmd.request_bufflen = datalen;

   	return(ppa_command(port,&internalCmd));
   	}

