/* Copyright (C) 1998, Synack Systems Corporation.  Created 1998. All
   rights reserved.

   License to copy and use this software is granted provided that it
   is identified as the "Synack Systems Corp., BeProtected IP Packet
   Filter" in all material mentioning or referencing this software.

   License is also granted to make and use derivative works provided
   that such works are identified as "derived from the Synack Systems
   Corp., BeProtected IP packet filter" in all material
   mentioning or referencing the derived work.

   Synack Systems Corp.  makes no representations concerning either
   the merchantability of this software or the suitability of this
   software for any particular purpose. It is provided "as is"
   without express or implied warranty of any kind.

   These notices must be retained in any copies of any part of this
   documentation and/or software.
 */       

#include <NetDevice.h>
#include <NetProtocol.h>
#include <List.h>
#include "headers.h"
#include "filter.h"
#include "FirewallPacketHandler.h"
#include "socket.h"
#include "firewall.h"

#define whycat(x)		strncat(why,x,sizeof(why)-strlen(why))

FirewallPacketHandler::FirewallPacketHandler()
{
}

bool FirewallPacketHandler::PacketReceived(BNetPacket *p, BNetDevice *bdevice)
{
  if(!isIpPacket(p, bdevice))
    return false;
  return IpPacketHandler();
}

// Returns true if the received packet is an IP packet
// otherwise it returns false
inline bool FirewallPacketHandler::isIpPacket(BNetPacket *p, BNetDevice *bdevice)
{
  device = bdevice->Type();
  
  if(device == B_ETHER_NET_DEVICE) {
    BStandardPacket *sp = (BStandardPacket*)p;
    unsigned size = sp->Size();
    packet = sp->DataBlock(0, &size);
    
    struct ether_header ether;
    memcpy(&ether, packet, ETHER_HEADER_SIZE);
    ether.ether_type = ntohs(ether.ether_type);
    if(ether.ether_type == ETHER_HEADER_IP) {
      // Fast forward to the IP header
      packet+=ETHER_HEADER_SIZE;
      bpacket = p;
      return true;
     }
  } else if(device == B_PPP_NET_DEVICE) {
     BStandardPacket *sp = (BStandardPacket*)p;
     unsigned size = sp->Size();
     packet = sp->DataBlock(0, &size);
     
     struct ppp_header ppp;
     memcpy(&ppp, packet, PPP_HEADER_SIZE);
     ppp.protocol = ntohs(ppp.protocol);

     if(ppp.protocol == PPP_HEADER_IP) {
       // Fast forward to the IP header
       packet+=PPP_HEADER_SIZE;
       bpacket = p;
       return true;
      }
  }
  
  return false;
}

// This function returns "FALSE" if a packet is allowed to pass
// It returns true and deletes the BNetPacket if it isn't
inline bool FirewallPacketHandler::IpPacketHandler() 
{
  int allow;
  int cnt = filters->CountItems();
  
  for(int i=0; i<cnt;i++) {
    allow = allowPacket((struct FirewallFilter*)filters->ItemAt(i));
    
    switch(allow) {
      case ALLOW_YES:
      	return false;
      case ALLOW_NO:
        delete bpacket;
        return true;
    }
   }
   
   delete bpacket;
   return true;
}

inline int FirewallPacketHandler::allowPacket(struct FirewallFilter *f) {
  struct ip ip;
  char *ptr = packet, why[512] = "";
  memcpy(&ip, ptr, IP_HEADER_SIZE);
  ptr+=ip.ip_hl*4;
  
  // These are the default filters
  //Deny small IP fragments
  if((ip.ip_off &0x1fff) == 1) {
	strcpy(why, "Tiny IP fragment");
    goto DENY_IT;
  }
  
  //Deny ICMP bomb packets
  if((ip.ip_off & 0x1fff) > 2048) {
    strcpy(why,"ICMP datagram bomb");
	goto DENY_IT;
  }
   
  // Deny packets that have fragmented TCP headers
  if(ip.ip_p == IP_TCP) {
    if(ip.ip_off == 0) 
      if((ip.ip_len-ip.ip_hl*4) < 40) {
      	strcpy(why, "Fragmented TCP header");
      	goto DENY_IT;
      }
  }
  
  // Check to see if the current filter applies to packets from this interface    
  if((f->flags & FILTER_INTERFACE) == FILTER_INTERFACE) {
    switch(device) {
      case B_PPP_NET_DEVICE:
        if(f->ppp == 0) goto NOMATCH;
        break;
      case B_ETHER_NET_DEVICE:
        if(f->eth == 0) goto NOMATCH;
        break;
    }
  }
  
  if((f->flags & FILTER_IP_OPTIONS) == FILTER_IP_OPTIONS) {
    if(ip.ip_hl == 5)
	  goto NOMATCH;
  }
      
  if((f->flags & FILTER_PROTOCOL) == FILTER_PROTOCOL) {
    if(ip.ip_p != f->protocol)
      goto NOMATCH;
  }
  
  if((f->flags & FILTER_SRC_ADDRESS) == FILTER_SRC_ADDRESS) {	
    if(!((ntohl(ip.ip_src) >= f->ip_src[0]) && (ntohl(ip.ip_src) <= f->ip_src[1])))
      goto NOMATCH;
  }
  
  if((f->flags & FILTER_DST_ADDRESS) == FILTER_DST_ADDRESS) {
    if(!((ntohl(ip.ip_dst) >= f->ip_dst[0]) && (ntohl(ip.ip_dst) <= f->ip_dst[1])))
       goto NOMATCH;
  }
  
  if((f->flags & FILTER_DST_PORT) == FILTER_DST_PORT) {
    unsigned short port = 0;

    if(ip.ip_p == IP_TCP) {
      struct tcphdr tcp;
      memcpy(&tcp, ptr, TCP_HEADER_SIZE);
      port = ntohs(tcp.th_dport);
    } else if(ip.ip_p == IP_UDP) {
      struct udphdr udp;
      memcpy(&udp, ptr, UDP_HEADER_SIZE);
      port = ntohs(udp.uh_dport);
    }
   
    if(!((port >= f->dport[0]) && (port <= f->dport[1])))
      goto NOMATCH;
  }
  
  if((f->flags & FILTER_SRC_PORT) == FILTER_SRC_PORT) {
    unsigned short port = 0;

    if(ip.ip_p == IP_TCP) {
      struct tcphdr tcp;
      memcpy(&tcp, ptr, TCP_HEADER_SIZE);
      port = ntohs(tcp.th_sport);
    } else if(ip.ip_p == IP_UDP) {
      struct udphdr udp;
      memcpy(&udp, ptr, UDP_HEADER_SIZE);
      port = ntohs(udp.uh_sport);
    }
   
    if(!((port >= f->sport[0]) && (port <= f->sport[1])))
      goto NOMATCH;
  }
  
  if((f->flags & FILTER_FLAGS) == FILTER_FLAGS) {
    if(ip.ip_p == IP_TCP) {
      struct tcphdr tcp;
      memcpy(&tcp, ptr, TCP_HEADER_SIZE);
      
      if((tcp.th_flags & f->th_flags) != tcp.th_flags)
        goto NOMATCH;
     }
  }
  
  // We had a match
  if((f->flags & FILTER_ALLOW) == FILTER_ALLOW)
    return ALLOW_YES;
  
  DENY_IT:
    fprintf(flog, "packet denied (%s)\n", why);
    return ALLOW_NO;
  
  NOMATCH:
    return ALLOW_SHRUG;
}

// Just here cause the docs say for it to be
void FirewallPacketHandler::ProcessPacket(BNetPacket *p)
{ // this func never called
  p;	// this line here to avoid warning
}