#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <ctype.h>
#include <Application.h>
#include <Window.h>
#include <View.h>
#include <ScrollView.h>
#include <TextView.h>
#include <Bitmap.h>
#include <Cursor.h>
#include <Mime.h>
#include <NodeInfo.h>
//#include <Roster.h>
#include <Path.h>

#include "str.h"
#include "widgets.h"
#include "dump_wav.h"
#include "bitmaps.h"
#include "amc.h"
#include "colors.h"
#include "sound.h"
#include "chords.h"
#include "ndist.h"
#include "midi-out.h"
#include "read-wav-data.h"

const int
  sect_max=40,       // max sections in ScLine
  sc_max=50,         // max tunes
  sclin_disp_max=28, // max displayed lines per score
  max_alerts=5,
  sect_mus_max=90,   // max sections in music ScLine
  sclin_dist=4,      // line distance
  y_off=14,          // y offset of score lines
  x_off=14,          // x offset
  x_gap=2,           // gap at right side
  sect_len=6,
  m_act_max=8,       // mouse action buttons
  nop=-99,
  eo_arr=-98;        // end of array

const float
  top_left=100,      // main view
  top_top=24,
  view_vmax=684,     // total hight
  view_hmax=x_off+x_gap+sect_mus_max*sect_len,   // = 420, x-width of music view
  scview_vmax=2+y_off+sclin_disp_max*sclin_dist+B_H_SCROLL_BAR_HEIGHT, // height of score view
  musview_vmax=2+y_off+sclin_max*sclin_dist,     // height of music view
  rbutview_left=229, // distance left side of radio-button views and right side
  but_dist=20,
  slider_width=54,   // slider width
  slider_height=42,  // horizontal slider height
  ictrl_height=84,   // instrument control view
  radiob_dist=13,    // radio button distance
  score_info_len=65, // at right side of scores
  checkbox_height=15,// checkbox height
  scview_text=x_off+x_gap+sect_max*sect_len; // text and controls in score view

enum {
  eScore_start,  // saving and restoring from file
  eScore_end,
  ePlayNote,
  eSetMeter,

  eAdd, eTake, eTake_nc,  // read script
  eFile, eString,

  eText,     // ScInfo tags
  eTempo,
  ePurple_a_harm, ePurple_s_harm, ePurple_attack, ePurple_loc,
  eRed_att_timbre, eRed_timbre, eRed_attack, eRed_decay, eRed_soft, eRed_loc,
  eGreen_tone, eGreen_attack, eGreen_decay, eGreen_loc,
  eBlack_mod, eBlack_attack, eBlack_decay, eBlack_subband, eBlack_loc,
  eBlue_attack, eBlue_decay, eBlue_piano, eBlue_chorus, eBlue_rich, eBlue_loc,
  eBrown_mod, eBrown_attack, eBrown_decay, eBrown_subband, eBrown_detune, eBrown_loc,

  eScView, eMusic, eScBuffer,  // score types

  eColorValue, eAmplValue,     // section drawing mode

  eIdle, eTracking, eErasing,  // mouse states
  eMoving, eMoving_hor, eMoving_vert,
  eCopying, eCopying_hor, eCopying_vert, 
  eCollectSel,
  ePortaStart,
  eToBeReset
};

rgb_color  // exported, so cannot be const
    cWhite	= {255,255,255,255},
    cBlack	= {0,0,0,255},
    cGrey	= {220,220,220,255},
    cGhost	= {180,180,180,150},
    cLightGrey	= {245,245,245,255},
    cDarkGrey	= {100,100,100,255},
    cTintedGrey	= {205,220,205,255},
    cPurple	= {210,0,210,255},
    cRed	= {255,0,0,255},
    cBlue	= {0,0,255,255},
    cLightBlue	= {150,255,255,255},
    cDarkBlue   = {0,0,160,255},
    cGreen	= {0,170,0,255},
    cLightGreen	= {0,200,0,255},
    cDarkGreen	= {0,140,0,255},
    cYellow	= {255,235,190,255},
    cBrown	= {200,130,0,255},
    cPink	= {255,210,255,255};

const rgb_color ampl_color[ampl_max+1] =  // index 1 to 6
              { cPink,
                {240,240,240,255},
                {220,220,220,255},
                {190,190,190,255},
                {160,160,160,255},
                {130,130,130,255},
                {0,0,0,255}
              };


AppWindow *appWin;
BFont *appFont;
const char *wave_out_file="out.wav",
           *midi_out_file="out.mid",
           *tunes_mime="audio/AMC-tunes",
           *script_mime="text/AMC-script",
           *app_mime="application/AMC-app";
bool debug=false;
char textbuf[100];
int alerts;
bool dumpwav_okay,
     midiout_okay;

void say(char *form,...) {   // debugging
  va_list ap;
  va_start(ap,form);
  printf("say: "); vprintf(form,ap); putchar('\n');
  va_end(ap);
  fflush(stdout);
}

int linenr(float y) {   // y -> line nr
  return float2int((y-y_off)/sclin_dist);
}

int sectnr(float x) {   // x -> section nr
  return float2int((x-x_off)/sect_len);
}

int ypos(int lnr) {   // linenr -> y
  return y_off + sclin_dist*lnr;
}

rgb_color color(int col) {
  switch (col) {
    case eBlack:  return cBlack;
    case eGrey:   return cGrey;
    case ePurple: return cPurple;
    case eRed:    return cRed;
    case eBlue:   return cBlue;
    case eGreen:  return cGreen;
    case eBrown:  return cBrown;
    case eWhite:  return cWhite;
    case eLightGreen:  return cLightGreen;
    default: alert("color %d?",col); return cBlack;
  }
}

Bitmaps *bitmaps;
BCursor *square_cursor,
        *point_cursor;

struct Scores:Array<Score*,sc_max> {
  int lst_score;
  Scores();
  Score *new_score(const char*);
  Score *exist_name(const char *nam);
} scores;

void alert(const char *form,...) {
  if (alerts>max_alerts) return;
  if (++alerts>max_alerts) sprintf(textbuf,"> %d warnings",max_alerts);
  else {
    va_list ap;
    va_start(ap,form);
    vsprintf(textbuf,form,ap);
    va_end(ap);
  }
  int top=(alerts-1)*20;
  if (!be_app) { puts(textbuf); exit(1); }
  else {
    BPoint pt(10,10);
    if (appWin) pt.Set(appWin->Frame().left-30,appWin->Frame().top);
    call_alert(BRect(pt.x,pt.y+top,pt.x+250,pt.y+top+40),textbuf,&alerts);
  }
}

struct SectData {
  int lnr, snr;
  SectData(int lnr1,int snr1):
    lnr(lnr1),snr(snr1) { }
  bool operator==(SectData &sd) { return lnr==sd.lnr && snr==sd.snr; }
  bool operator<(SectData &sd) { return lnr==sd.lnr ? snr<sd.snr : lnr<sd.lnr; }
  //bool operator>(SectData &sd) { return lnr==sd.lnr ? snr>sd.snr : lnr>sd.lnr; }
};

struct Selected {
  struct ScoreView *sv;
  SLinkedList<SectData> sd_list; 
  bool inv; // list inverted?
  Selected():sv(0),inv(false) { }
  void insert(int lnr,int snr) {
    sd_list.insert(SectData(lnr,snr),!inv);
  }
  void remove(int lnr,int snr) {
    sd_list.remove(SectData(lnr,snr));
  }
  void check_direction(int delta_lnr,int delta_snr) {
    if (!inv && (delta_lnr>0 || delta_snr>0) ||
        inv && (delta_lnr<0 || delta_snr<0)) {
      sd_list.invert();
      inv=!inv;
    }
  }
  void reset() {
    sd_list.reset();
    inv=false;
  }
  void restore_sel();
} selected;

ScSection::ScSection(int col=eBlack,int amp=0):
  note_col(col),cat(eSilent),sign(0),note_ampl(amp),
  stacc(false),sampled(false),sel(false),
  port_dlnr(0),dlnr_sign(0),port_dsnr(0),nxt_note(0) {
}

void ScSection::reset() {  // supposed: sel = false
  static ScSection sec;
  *this=sec;
}

struct ScInfo {
  uint tag;
  ScInfo *next;
  union {
    char *text;
    bool b;
    Array<int8,2>n2;
    Array<int8,harm_max>n5;
    int n;
  };
  ScInfo():tag(0),text(0),next(0) { }
  ScInfo(uint t):tag(t),text(0),next(0) { }
  ~ScInfo() { delete next; }
  void add(ScInfo& info) {
    ScInfo *sci;
    if (!tag) *this=info;
    else {
      for (sci=this;sci->next;sci=sci->next);
      sci->next=new ScInfo(info);
    }
  }
  void reset() {
    delete next;
    tag=0; text=0; next=0;
  }
};

void Score::drawEnd(BView* theV,int d_off) {
  if (end_sect==nop) return;
  theV->SetHighColor(cBlack);
  theV->SetPenSize(1);
  int x=x_off+end_sect*sect_len;
  theV->StrokeLine(BPoint(x,ypos(17-d_off)),BPoint(x,ypos(25-d_off)));
}

ScSection* Score::get_section(int lnr,int snr) {
  return lin[lnr]->sect + snr;
}

void Score::time_mark(BView *theV,int snr) {
  int meter=appWin->act_meter;
  if (snr % meter == 0) {
    Str str;
    theV->SetHighColor(cBlack);
    theV->DrawString(str.tos(snr/meter),BPoint(x_off+snr*sect_len,y_off-3));
  }
}

void Score::drawText(BView* theV,int snr) {
  int x=x_off+snr*sect_len,
      y;
  ScInfo* sci;
  theV->SetHighColor(cBlack);
  for (sci=scInfo+snr,y=ypos(sclin_max)+6;sci;sci=sci->next)
    if (sci->tag==eText) { theV->DrawString(sci->text,BPoint(x,y)); y+=10; }
}

struct ViewScore: BView {
  int draw_mode,
      play_start,
      play_stop;
  ViewScore(BRect,uint32,uint32);
  void draw_start_stop(Score *score,int old_snr,int snr,bool begin);
  void enter_start_stop(Score *score,int snr,uint32 mouse_but);
};

void ScSection::drawS_ghost(ViewScore* theV,int snr,int lnr,int d_off,bool erase) {
  int x=x_off+snr*sect_len,
      y=ypos(lnr-d_off);
  BPoint start(x,y),
         end(x+sect_len-1,y);
  if (erase) drawSect(theV,snr,lnr,d_off);
  else {
    theV->SetDrawingMode(B_OP_ALPHA);
    theV->SetPenSize(3);
    theV->SetHighColor(cGhost);
    theV->StrokeLine(start,end);
    theV->SetDrawingMode(B_OP_COPY);
  }
}

struct LineCol {
  int col[sclin_max];
  LineCol() {
    int i;
    for (i=0;i<sclin_max;i+=2) col[i]=eWhite; // default line color
    for (i=1;i<16;i+=2) col[i]=eGrey;
    for (;i<26;i+=2) col[i]=eBlack;
    for (;i<28;i+=2) col[i]=eGrey;
    for (;i<38;i+=2) col[i]=eLightGreen;
    for (;i<sclin_max;i+=2) col[i]=eGrey;
  }
} linecol;

void ScSection::drawSect(ViewScore* theV,int snr,int lnr,int d_off) {
  int x=x_off+snr*sect_len,
      y=ypos(lnr-d_off);
  BPoint start(x,y),
         end(x+sect_len-1,y);

  theV->SetPenSize(5);
  theV->SetHighColor(cWhite);
  theV->StrokeLine(start,end);
  switch (cat) {
    case eSilent:
      theV->SetPenSize(1);
      --end.x;
      theV->SetHighColor(color(linecol.col[lnr]));
      theV->StrokeLine(start,end);
      if (snr%appWin->act_meter==0 && lnr%2==0) { // timing marks between lines
        theV->SetHighColor(cGrey);
        theV->StrokeLine(BPoint(x,y-1),BPoint(x,y+1));
      }
      break;
    case ePlay:
      drawPlaySect(theV,start,end);
      break;
    default: alert("section cat %d?",cat);
  }
}

ScLine::ScLine(Score* sc):
  len(sc->len),
  sect(new ScSection[len](sc->ncol,sc->nampl)),
  note_sign(0) { }

void ScSection::drawPortaLine(ViewScore *theV,int snr,int lnr,int d_off,bool erase) {
  int x=x_off+snr*sect_len,
      y=ypos(lnr-d_off);
  BPoint start(x,y),
         end(x+sect_len-1,y);
  theV->SetPenSize(1);
  if (erase) theV->SetHighColor(cWhite);
  else theV->SetHighColor(color(note_col));
  int dlnr= dlnr_sign ? port_dlnr : -port_dlnr;
  theV->StrokeLine(end,BPoint(start.x+(port_dsnr+1)*sect_len,start.y+dlnr*sclin_dist));
}

void ScSection::drawPlaySect(ViewScore *theV,BPoint &start,BPoint &end) {
  int note_color= nxt_note ? eGrey : note_col;
  switch (theV->draw_mode) {
    case eColorValue:
      theV->SetHighColor(color(note_color));
      break;
    case eAmplValue:
      theV->SetHighColor(ampl_color[note_ampl]);
      break;
  }
  pattern patt=sel ? B_MIXED_COLORS : B_SOLID_HIGH;
  if (sampled) {
    theV->SetPenSize(1);
    theV->StrokeLine(BPoint(start.x,start.y-1),BPoint(end.x,end.y+1),patt);
    theV->StrokeLine(BPoint(start.x,start.y+1),BPoint(end.x,end.y-1),patt);
  }
  else {
    if (stacc) end.x-=2;
    theV->SetPenSize(3);
    theV->StrokeLine(start,end,patt);
  }
  if (sign) {
    theV->SetPenSize(1);
    theV->SetHighColor(cWhite);
    start.x+=2; 
    if (sign==eHi) { --start.y; --end.y; }
    else if (sign==eLo) { ++start.y; ++end.y; }
    theV->StrokeLine(start,end);
  }
}

void ScLine::eraseScLine(ViewScore* theV,int lnr,int d_off,int left=0,int right=0) {
  int nmax= right ? min(len,right) : len,
      x1=x_off+left*sect_len,
      x2=x_off+nmax*sect_len,
      y=ypos(lnr-d_off);
  BPoint start(x1,y),
         end(x2,y);

  theV->SetPenSize(5); // erase old line
  theV->SetHighColor(cWhite);
  theV->StrokeLine(start,end);
}

void ScLine::drawScLine(ViewScore* theV,int lnr,int d_off,int left=0,int right=0) {
  int snr,
      nmax= right ? min(len,right) : len,
      x1=x_off+left*sect_len,
      x2=x_off+nmax*sect_len,
      y=ypos(lnr-d_off),
      note_color;
  ScSection *sec,*sec1;
  BPoint start(x1,y),
         end(x2,y);
  rgb_color line_color=color(linecol.col[lnr]);
  pattern patt;

  for (snr=left;snr<nmax;++snr) {
    sec=sect+snr;
    x1=x_off+snr*sect_len;
    x2=x1+sect_len-1;
    start.Set(x1,y);
    end.Set(x2,y);
    switch (sec->cat) {
      case eSilent:
        --end.x;
        theV->SetPenSize(1);
        theV->SetHighColor(line_color);
        theV->StrokeLine(start,end);
        if (snr%appWin->act_meter==0 && lnr%2==0) { // timing marks between lines
          theV->SetHighColor(cGrey);
          theV->StrokeLine(BPoint(x1,y-1),BPoint(x1,y+1));
        }
        break;
      case ePlay:
        sec->drawPlaySect(theV,start,end);
        for (sec1=sec;;sec1=appWin->mn_buf+sec1->nxt_note) {
          if (sec1->port_dlnr)
            sec1->drawPortaLine(theV,snr,lnr,d_off,false);
          if (!sec1->nxt_note) break;
        }
        break;
      default: alert("section cat %d?",sec->cat);
    }
  }
}

Score::Score(const char *nam,int length,uint sctype,int col=eBlack,int amp=4):
    name(const_cast<char*>(nam)),
    ncol(col),
    nampl(amp),
    len(length),
    lst_sect(-1),
    end_sect(nop),
    signs_mode(0),
    is_scview(sctype==eScView),
    is_music(sctype==eMusic),
    scInfo(sctype==eMusic ? new ScInfo[len]() : 0),
    rbut(0) {
  int i;
  for (i=0;i<sclin_max;i++) lin[i]=new ScLine(this);
}

int Score::to_half(int lnr,int sign) {
  int ind=lnr%7;
                //  b a g f e d c
  static int ar[]={ 0,2,4,6,7,9,11 };
  return ar[ind] + (sign==eHi ? -1 : sign==eLo ? 1 : 0) + (lnr-ind)/7*12;
}

void Score::from_half(int n,int& lnr,int& sign) {
  int ind=n%12,
      *ar, *s;
                   // b   bes a   as  g   fis f   e   dis d   cis c
  static int ar_0[]={ 0,  0,  1,  1,  2,  3,  3,  4,  5,  5,  6,  6 },
             s_0[]= { 0,  eLo,0,  eLo,0,  eHi,0,  0,  eHi,0,  eHi,0 },

                   // b   bes a   as  g   ges f   e   es d   des c
             ar_f[]={ 0,  0,  1,  1,  2,  2,  3,  4,  4,  5,  5,  6 },
             s_f[]= { 0,  eLo,0,  eLo,0,  eLo,0,  0,  eLo,0,  eLo,0 },

                   // b   ais a   gis  g   fis f   e   dis d   cis c
             ar_s[]={ 0,  1,  1,  2,  2,  3,  3,  4,  5,  5,  6,  6 },
             s_s[]= { 0,  eHi,0,  eHi,0,  eHi,0,  0,  eHi,0,  eHi,0 };
  switch (signs_mode) {
    case eFlat: ar=ar_f; s=s_f; break;
    case eSharp: ar=ar_s; s=s_s; break;
    case 0: ar=ar_0; s=s_0; break;
    default: ar=ar_0; s=s_0; alert("signs_mode?");
  }
  lnr=ar[ind]+(n-ind)/12*7;
  sign=s[ind];
}

bool Score::check_len(int required) {
  ScLine *old_lin;
  ScInfo *old_info;
  int n,n2,
      old_len=len;
  if (len>=required) return false;
  if (debug) printf("len incr: len=%d req=%d\n",len,required);
  while (len<required) len+=old_len;
  for (n=0;n<sclin_max;n++) {
    old_lin=lin[n];
    lin[n]=new ScLine(this);
    lin[n]->note_sign=old_lin->note_sign;
    for (n2=0;n2<old_len;n2++)
      lin[n]->sect[n2]=old_lin->sect[n2];
    delete[] old_lin;
  }
  if (scInfo) {
    old_info=scInfo;
    scInfo=new ScInfo[len]();
    for (n=0;n<old_len;n++) {
      scInfo[n]=old_info[n];
      old_info[n].next=0; // to prohibit deletion of *next sections
    }
    delete[] old_info;
  }
  return true;
}

bool eq(ScSection* one,ScSection* two) {  // NOT considered: next
  return one->cat==two->cat &&
         one->sign==two->sign &&
         one->note_col==two->note_col;
}

void Score::copy(Score *from) { // name NOT copied
                                // end_sect, amplitude and color copied
  int snr,lnr,
      stop;
  end_sect=from->end_sect;
  if (end_sect>nop) { check_len(end_sect+1); stop=end_sect; }
  else { stop=from->len; check_len(stop); }
  for (lnr=0;lnr<sclin_max;lnr++) {
    lin[lnr]->note_sign=from->lin[lnr]->note_sign;
    for (snr=0;snr<stop;++snr) {
      ScSection& to=lin[lnr]->sect[snr];
      to=from->lin[lnr]->sect[snr];
      to.sel=0;
    }
  }
  ncol=from->ncol;
  nampl=from->nampl;
  signs_mode=from->signs_mode;
}

void Score::add_copy(Score *sc,
    int start,int stop,int tim,int ampl,int shift,int raise,ViewScore* theV) {
    // name, ncol, nampl NOT copied
  int n,
      snr,snr1,
      lnr,lnr1,
      sign,
      note_num=0;
  ScSection *from,
            *to,*to1;
  if (tim==nop) tim=lst_sect+1;
  check_len(tim+stop-start);
  if (is_music) {
    ScInfo info(eText);
    info.text=sc->name;
    scInfo[tim].add(info);
    time_mark(theV,tim);
    drawText(theV,tim);
    drawEnd(theV,0);
  }
  for (snr=start,snr1=tim-1;snr<stop;++snr) {
    ++snr1;
    for (lnr=0;lnr<sclin_max;lnr++) {
      from=sc->get_section(lnr,snr);
      if (from->cat==ePlay && (!appWin->solo->value || from->note_col==appWin->act_color)) {
        if (snr==start || (from-1)->cat!=ePlay || !eq(from-1,from)) // find start of note
          ++note_num;
        if (shift==0) { lnr1=lnr; sign=from->sign; }
        else from_half(to_half(lnr,from->sign)-shift,lnr1,sign);
        lnr1-=raise;
        if (lnr1>=sclin_max || lnr1<0) continue;
        to1=to=get_section(lnr1,snr1);
        if (to->cat==ePlay)  // multiple note?
          for (;to1->note_col!=from->note_col || to1->sampled!=from->sampled; // equal colors cannot be played together
                to1=appWin->mn_buf+to1->nxt_note) {
            if (!to1->nxt_note) {
              if (appWin->mn_end==mn_max-1) { alert("> %d multi notes",mn_max); break; }
              to1->nxt_note=++appWin->mn_end;
              to1=appWin->mn_buf+appWin->mn_end;
              break;
            }
          }
        *to1=*from;
        to1->sel=false;
        to1->nxt_note=0;
        to1->sign=sign;
        if (ampl>nop) to1->note_ampl=ampl;
        if (is_music) {
          to->drawSect(theV,snr1,lnr1,0);
          if (to1->port_dlnr) to1->drawPortaLine(theV,snr1,lnr1,0,false);
        }
      }
    }
  }
  if (snr1>lst_sect) {
    end_sect=snr1+1;
    check_len(end_sect+1);
    if (is_music) {
      if (lst_sect>=0)
        for (n=0;n<sclin_max;++n)  // erase old end-line
          lin[n]->sect[lst_sect+1].drawSect(theV,lst_sect+1,n,0);
      time_mark(theV,end_sect);
      drawText(theV,end_sect);
      drawEnd(theV,0);
    }
//    else end_sect=sc->end_sect;
    lst_sect=snr1;
  }
}

void Score::reset() {
  ncol=eBlack;
  nampl=4;
  lst_sect=-1;
  end_sect=nop;
  signs_mode=0;
  int lnr,snr;
  if (is_music) {
    for (lnr=0;lnr<sclin_max;lnr++)
      for (snr=0;snr<len;snr++) lin[lnr]->sect[snr].reset();
    for (snr=0;snr<len;snr++) scInfo[snr].reset();
  }
  else if (len>sect_max) {
    len=sect_max;
    for (lnr=0;lnr<sclin_max;lnr++) {
      delete[] lin[lnr]->sect;
      lin[lnr]->note_sign=0;
      lin[lnr]->sect=new ScSection[len](ncol,nampl);
    }
  }
  else {
    for (lnr=0;lnr<sclin_max;lnr++) {
      ScLine *line=lin[lnr];
      line->note_sign=0;
      for (snr=0;snr<len;snr++) line->sect[snr].reset();
    }
  }
}

Scores::Scores():lst_score(-1) {
  for (int i=0;i<sc_max;i++) buf[i]=0;
}

StdView::StdView(BRect rect):BView(rect,0,B_FOLLOW_RIGHT,B_WILL_DRAW) {
  SetFont(appFont);
  SetViewColor(cYellow);
  SetLowColor(cYellow);
}

void ViewScore::draw_start_stop(Score *score,int old_snr,int snr,bool begin) {
    int x,
        y=7;
    if (old_snr>0) {
      x=x_off+old_snr*sect_len;
      SetHighColor(cWhite);  // erase old sign
      if (begin) FillTriangle(BPoint(x,y+5),BPoint(x+sect_len,y),BPoint(x,y-5));
      else FillTriangle(BPoint(x+sect_len,y+5),BPoint(x,y),BPoint(x+sect_len,y-5));
      score->time_mark(this,old_snr);
      score->time_mark(this,old_snr-1); // big number?
    }
    if (snr>0) {
      x=x_off+snr*sect_len;
      SetHighColor(cRed);
      if (begin) FillTriangle(BPoint(x,y+5),BPoint(x+sect_len,y),BPoint(x,y-5));
      else FillTriangle(BPoint(x+sect_len,y+5),BPoint(x,y),BPoint(x+sect_len,y-5));
    }
  }

ViewScore::ViewScore(BRect rect,uint32 rmode,uint32 flags):
    BView(rect,0,rmode,flags),
    draw_mode(eColorValue),
    play_start(0),play_stop(-1) {
}

struct ScoreViewStruct {
  int id;
  struct ScoreSView *scroll_view;
  struct ScoreView *scv_view;
  struct ScoreViewText *scv_text;
  ScoreViewStruct(BRect rect,int ident);
  void assign_score(Score*);
  void invalidate();
};

struct ScoreView: ViewScore {
  ScoreViewStruct *theViewStruct;
  Score *score;
  ScLine *cur_line;
  ScSection *cur_sect;   // set when mouse down
  int d_off,             // display offset
      state,
      cur_lnr, cur_snr,  // set when mouse down
      prev_snr,          // previous visited section nr
      delta_lnr, delta_snr,
      prev_delta_lnr, prev_delta_snr;
  BPoint cur_point,      // set when mouse down
         prev_point;     // set when mouse moved
  ScoreView(BRect rect,ScoreViewStruct*);
  void Draw(BRect rect);
  void redraw(bool meter);
  void upd_endline(int snr);
  void MouseDown(BPoint);
  void MouseMoved(BPoint, uint32 location, const BMessage *);
  void MouseUp(BPoint);
  void drawSigns();
  void select_all();
  void modify_sel(int mes);
  void select_column(int snr);
};

struct ScScrollBar: BScrollBar {
  ScScrollBar(BRect frame,BView *target,float mini,float maxi):
      BScrollBar(frame,"",target,mini,maxi,B_HORIZONTAL) {
    SetResizingMode(B_FOLLOW_LEFT_RIGHT);
  }
};

struct ScoreSView {
  ScoreView *scoreView;
  BScrollBar *scrollbar;
  ScoreSView(BRect rect,ScoreViewStruct *svs) {
    scoreView=new ScoreView(BRect(rect.left,rect.top,rect.right,rect.bottom-B_H_SCROLL_BAR_HEIGHT),svs);
    scrollbar=new ScScrollBar(BRect(rect.left,rect.bottom-B_H_SCROLL_BAR_HEIGHT,rect.right,rect.bottom),
                             scoreView,0,scview_text);
    scrollbar->SetRange(0,0);
  }
};

void ScoreView::drawSigns() {
  int lnr,
      y,
      sign;
  SetDrawingMode(B_OP_OVER);
  for (lnr=d_off;lnr<d_off+sclin_disp_max;++lnr) {
    y=ypos(lnr-d_off)-3;
    sign=score->lin[lnr]->note_sign;
    if (sign)
      DrawBitmap(bitmaps->get(sign==eHi ? eSharp : eFlat),BPoint((lnr%7 & 1)==0 ? 2 : 7,y));
  }
  SetDrawingMode(B_OP_COPY);
  for (lnr=d_off;lnr<=d_off+sclin_disp_max;++lnr) {  // 1 extra
    SetHighColor(lnr%7==4 || lnr%7==0 ? cBlue : cLightBlue); // small lines at left side
    y=ypos(lnr-d_off)-2;
    StrokeLine(BPoint(0,y),BPoint(1,y));
  }
}

void Score::put_chord(ScoreView *sv) {
  int n,
      lnr,snr,
      sign,
      dist=appWin->chordsWin->the_distance;
  Array<int,10> &chord_notes=appWin->chordsWin->chord_notes;
  ScSection *sect;
  for (n=0;chord_notes[n]>nop;++n) {
    // offset in chord_notes[]: note B = 0, note C = 6, middle C = 6 + 3 * 7 = 27 , is equal to 47 semi-tones
    if (n==0) from_half(47-dist,lnr,sign);
    else from_half(47-chord_notes[n]-dist,lnr,sign);
    if (end_sect>nop) {
      snr=end_sect+1;
      check_len(snr+1);
    }
    else
      snr=0;
    selected.insert(lnr,snr);
    sect=get_section(lnr,snr);
    sect->reset();
    sect->cat=ePlay;
    sect->sign=sign;
    sect->note_ampl=sv->score->nampl;
    sect->note_col=sv->score->ncol;
    sect->sel=true;
    sect->drawSect(sv,snr,lnr,sv->d_off);
  }
}
  
void Score::set_signs(int flatn,int sharpn) {
  int n,lnr;
  for (lnr=6;lnr>=0;--lnr) {
    if (flatn & 1) for (n=lnr;n<sclin_max;n+=7) lin[n]->note_sign=eLo;
    else if (sharpn & 1) for (n=lnr;n<sclin_max;n+=7) lin[n]->note_sign=eHi;
    flatn>>=1; sharpn>>=1;
  }
}

void Score::tone2signs() {   // C = 0, B = 12
  const int f=eLo,s=eHi;
  static int signs[][7]= {
    //  B A G F E D C
      { 0,0,0,0,0,0,0 },  // C
      { f,f,f,0,f,f,0 },  // Des
      { 0,0,0,s,0,0,s },  // D
      { f,f,0,0,f,0,0 },  // Es
      { 0,0,s,s,0,s,s },  // E
      { f,0,0,0,0,0,0 },  // F
      { 0,s,s,s,s,s,s },  // Fis
      { f,f,f,f,f,f,0 },  // Ges ( = Fis)
      { 0,0,0,s,0,0,0 },  // G
      { f,f,0,0,f,f,0 },  // As
      { 0,0,s,s,0,0,s },  // A
      { f,0,0,0,f,0,0 },  // Bes
      { 0,s,s,s,0,s,s }   // B
    },          
             // C Des   D      Es    E      F     Fis    Ges   G      As    A      Bes   B
    smode[] = { 0,eFlat,eSharp,eFlat,eSharp,eFlat,eSharp,eFlat,eSharp,eFlat,eSharp,eFlat,eSharp };
  int n,lnr,
      key_nr=appWin->chordsWin->the_key_nr;
  for (lnr=6;lnr>=0;--lnr) {
    for (n=lnr;n<sclin_max;n+=7) lin[n]->note_sign=signs[key_nr][lnr];
  }
  signs_mode=smode[key_nr];
}

struct ScoreViewText: StdView {
  ScoreViewStruct *theViewStruct;
  UpdText *sc_name;
  HSlider *sc_ampl;
  UpDown *ud;
  CheckBox *display_mode;
  ScoreViewText(BRect rect,ScoreViewStruct* v):
      StdView(rect),
      theViewStruct(v),
      sc_name(new UpdText(this,BPoint(2,12))),
      sc_ampl(new HSlider(BRect(2,16,2+slider_width,0),"amplitude",'ampl',1,ampl_max)) {
    sc_ampl->SetLimitLabels("1","6");
    sc_ampl->SetValue(4);
    sc_ampl->theMessage->AddPointer("",theViewStruct);
    AddChild(sc_ampl);

    appWin->active_scoreCtrl->AddButton(this,BPoint(2,62),"active",theViewStruct->id);

    ud=new UpDown(BPoint(2,82),"move",'ud  ');
    ud->theMessage->AddPointer("",theViewStruct->scv_view);
    ud->value=1;
    AddChild(ud);

    display_mode=new CheckBox(BPoint(2,100),"disp ampl",'dmod');
    display_mode->theMessage->AddPointer("",theViewStruct);
    AddChild(display_mode);

    Button *but=new Button(BRect(2,120,0,0),"play",'play');
    but->theMessage->AddPointer("",theViewStruct->scv_view);
    AddChild(but);
  }
};

void ScoreViewStruct::assign_score(Score *sc) {
  scv_view->score=sc;
  scv_text->sc_ampl->SetValue(sc->nampl);
  scv_view->play_start=0; scv_view->play_stop=-1;
  scroll_view->scrollbar->SetRange(0,(scv_view->score->len-sect_max)*sect_len);
}

void ScoreViewStruct::invalidate() {
  if (scv_view->score)
    scroll_view->scrollbar->SetRange(0,(scv_view->score->len-sect_max)*sect_len);
  else
    scroll_view->scrollbar->SetRange(0,0);
  scroll_view->scrollbar->SetValue(0);
  scv_view->Invalidate();
  scv_text->Invalidate();
}

ScoreView::ScoreView(BRect rect,ScoreViewStruct *svs):
    ViewScore(rect,B_FOLLOW_LEFT_RIGHT,B_WILL_DRAW),
    score(0),
    cur_line(0),cur_sect(0),
    theViewStruct(svs),
    d_off(7),
    state(eIdle) {
  SetFont(appFont);
  SetBlendingMode(B_CONSTANT_ALPHA, B_ALPHA_OVERLAY); // for drawing ghost sections
}

void ScoreView::Draw(BRect) {
  int n;
  SetHighColor(cBlack);
  if (score) {
    theViewStruct->scv_text->sc_name->textColor=color(score->ncol);
    theViewStruct->scv_text->sc_name->write(score->name);
    for (n=d_off;n<d_off+sclin_disp_max;++n)
      score->lin[n]->eraseScLine(this,n,d_off);
    for (n=d_off;n<d_off+sclin_disp_max;++n)
      score->lin[n]->drawScLine(this,n,d_off);
    drawSigns();
    for (n=0;n<max(sect_max,score->end_sect+1);++n) score->time_mark(this,n);
    draw_start_stop(score,0,play_start,true);
    draw_start_stop(score,0,play_stop,false);
    score->drawEnd(this,d_off);
  }
}

void ScoreView::redraw(bool clear_meter) {  // less disturbance then regular Invalidate
  int n;
  BRect rect(Bounds());
  SetHighColor(cWhite);
  FillRect(BRect(0,0,x_off,rect.bottom)); // clear signs
  if (clear_meter)
    FillRect(BRect(0,0,rect.right,y_off)); // clear meter numbers
  Draw(Bounds());
}

void ScoreView::upd_endline(int snr) {
  int n,
      old_len;
  ScSection* sect;
  if (score->end_sect>nop || score->end_sect==snr)   // erase old end
    for (n=d_off;n<d_off+sclin_disp_max;++n) {
      sect=score->lin[n]->sect+score->end_sect;
      sect->drawSect(this,score->end_sect,n,d_off);
    }
  score->end_sect=snr;
  old_len=score->len;
  if (score->check_len(snr+1)) {
    for (n=d_off;n<d_off+sclin_disp_max;++n)
      score->lin[n]->drawScLine(this,n,d_off,old_len,score->len);
    theViewStruct->scroll_view->scrollbar->SetRange(0,(score->len-sect_max)*sect_len);
  }
  for (n=0;n<=snr;++n) score->time_mark(this,n);
  score->drawEnd(this,d_off);
}

struct MusicView: ViewScore {
  struct MusicSView* theSView;
  Score *score;
  int musv_len;
  MusicView(BRect rect,MusicSView* sv):
      ViewScore(rect,B_FOLLOW_TOP_BOTTOM|B_FOLLOW_H_CENTER,B_WILL_DRAW),
      theSView(sv),
      score(new Score("music",sect_mus_max,eMusic)),
      musv_len(sect_mus_max) {
    SetFont(appFont);
  }
  void Draw(BRect rect);
  void redraw(bool);
  void reset(bool reset_start) {
    score->reset();
    if (reset_start) { play_start=0; play_stop=-1; }
    appWin->mn_end=0;
    for (int n=0;n<mn_max;++n) appWin->mn_buf[n].reset();
  }
  void upd_info(int t,ScInfo& sci) {
    if (t) score->check_len(t+1);
    score->scInfo[t].add(sci);
  }
  void MouseDown(BPoint);
};

struct Question: StdView {
  struct QuBHandler:BHandler {
    int mode;
    QuBHandler():BHandler() { }
    void MessageReceived(BMessage *mes) { mode=mes->what; }
  } handler;
  TextControl *textControl;
  UpdTextBg *text;
  Question(BRect rect): StdView(rect),
      textControl(new TextControl(BRect(26,16,rect.Width()-2,0),'ctrl')),
      text(new UpdTextBg(this,BPoint(2,10))) {
    AddChild(textControl);
    AddChild(new Button(BRect(2,16,22,0),"ok",'ctrl'));
  }
  void Draw(BRect) {
    text->write(text->buf);
  }
};

struct MeterView: StdView {
  RButtonArrowCtrl *ctrl;
  RadioButton *rbut6,*rbut8,*rbut12;
  BPoint top;
  MeterView(BRect rect):
      StdView(rect),
      top(BPoint(2,12)),
      ctrl(new RButtonArrowCtrl('setm')) {
    rbut6=ctrl->AddButton(this,top,"6",6); top.y+=radiob_dist;
    rbut8=ctrl->AddButton(this,top,"8",8); top.y+=radiob_dist;
    rbut12=ctrl->AddButton(this,top,"12",12);
    redraw();
  }
  void Draw(BRect rect) {
    DrawString("meter",BPoint(2,10));
  }
  void redraw() {
    switch (appWin->act_meter) {
      case 6: rbut6->SetValue(true); break;
      case 8: rbut8->SetValue(true); break;
      case 12: rbut12->SetValue(true); break;
    }
  }
};

struct TunesView: StdView {
  BPoint top;
  RButtonArrowCtrl *ctrl;
  TunesView(BRect rect):
      StdView(rect),
      top(BPoint(2,10)),
      ctrl(new RButtonArrowCtrl('tune'))  {
    Score* sc;
    for (int n=0;scores[n];++n) {
      sc=scores[n];
      sc->rbut=ctrl->AddButton(this,top,sc->name,n);
      top.y+=radiob_dist;
    }
  }
  void reset(int lst_score) {
    Score *sc;
    for (int n=0;n<=lst_score;++n) {
      sc=scores[n];
      if (sc->rbut) {
        RemoveChild(sc->rbut);
        delete(sc->rbut);
        sc->rbut=0;
      }
    }
    top.Set(2,10);
  }
  void Draw(BRect) { DrawString("tunes",BPoint(2,10)); }
};

struct TunesSView: StdView {
  TunesView *tunesView;
  BScrollView *scrollView;
  TunesSView(BRect rect): StdView(rect) {
    rect.right-=B_V_SCROLL_BAR_WIDTH;
    tunesView=new TunesView(rect);
    scrollView=new BScrollView(0,tunesView,B_FOLLOW_RIGHT,0,false,true,B_NO_BORDER);
    scrollView->ScrollBar(B_VERTICAL)->SetRange(0,sc_max*radiob_dist-rect.Height());
  }
};

struct TempoView: StdView {
  HSlider *tempo;
  void show_val() {
    tempo->SetText("%d",10 * tempo->value);
    tempo->UpdateText();
  }
  void redraw_sliders() {
    if (appWin->act_tempo < 6 || appWin->act_tempo > 16) appWin->act_tempo=11;
    tempo->SetValue(appWin->act_tempo);
    show_val();
  }
  TempoView(BRect rect): StdView(rect) {
    tempo=new HSlider(BRect(0,0,slider_width,0),"tempo",0,6,16);
    tempo->SetValue(11);
    tempo->modMessage=new BMessage('tmpo');
    tempo->SetText("110");
    AddChild(tempo);
  }
};

struct MouseAction: BView {
  RButtonAM1KnobCtrl *ctrl;
  BPoint top;
  Array<RadioButton*,m_act_max>buttons;
  int *but_mess;
  MouseAction(BRect rect):
      BView(rect,0,B_FOLLOW_LEFT,B_WILL_DRAW),
      ctrl(new RButtonAM1KnobCtrl('msta')),
      top(BPoint(2,0)) {
    SetFont(appFont);
    SetViewColor(cTintedGrey);
    SetLowColor(cTintedGrey);
    ctrl->min_width=25.;
    static char *labels[m_act_max+5]={
      "select","sel all","move","copy","portando","sharp","flat","normal", // radio buttons
      "unselect","re-color","amp +","amp -","delete"   // normal buttons
    };
    static int mess[m_act_max+5]={
      'sel','all','move','copy','prta','up','do','ud','uns','rcol','amp+','amp-','del '
    };
    but_mess=mess;
    int n;
    for (n=0;n<m_act_max;++n) {
      buttons[n]=ctrl->AddButton(this,top,labels[n],mess[n]);
      if (n<5) buttons[n]->textColor=cRed;
      top.x += buttons[n]->dx()+2;
    }
    for (;n<m_act_max+5;++n) {
      Button *but=new Button(BRect(top.x,top.y,top.x+25,0),labels[n],mess[n]);
      but->textColor=cDarkBlue;
      AddChild(but);
      top.x += but->dx()+2;
    }
  }
  void set_active(int ch) {
    for (int n=0;n<m_act_max;++n)
      if (but_mess[n]==ch) { 
        buttons[n]->SetValue(true);
        return;
      }
  }
  void reset() {
    ctrl->reset();
    appWin->act_action=nop;
  }
};

Rot::Rot(BRect rect,int msg):RotateChoice(rect,msg,3) {
  SetViewColor(cLightGrey);
}

void Rot::draw(int nr) {
  SetHighColor(cBlack);
  switch (nr) {
    case 0: {
        static BPoint p1[] = { BPoint(7,2),BPoint(2,5),BPoint(7,8) };
        static BPoint p2[] = { BPoint(11,2),BPoint(6,5),BPoint(11,8) };
        StrokePolygon(p1,3,false);
        StrokePolygon(p2,3,false);
      }
      break;
    case 1:
      StrokeLine(BPoint(7,2),BPoint(7,8));
      break;
    case 2: {
        static BPoint p1[] = { BPoint(3,2),BPoint(8,5),BPoint(3,8) };
        static BPoint p2[] = { BPoint(7,2),BPoint(12,5),BPoint(7,8) };
        StrokePolygon(p1,3,false);
        StrokePolygon(p2,3,false);
      }
      break;
  }
}

struct ColorView: StdView {
  RButtonArrowCtrl *ctrl;
  BPoint top;
  Rot *stereo[colors_max];
  ColorView(BRect rect):
      StdView(rect),
      top(BPoint(2,12)),
      ctrl(new RButtonArrowCtrl('colr')) {
    static char *names[colors_max]={ "black","red","green","blue","brown","purple" };
    static int colors[colors_max]={ eBlack,eRed,eGreen,eBlue,eBrown,ePurple };
    appWin->colors=colors;
    RadioButton *rbut;
    BRect rot_rect(rect.Width()-15,15,rect.Width()-1,25);
    for (int n=0;n<colors_max;n++) {
      rbut=ctrl->AddButton(this,top,names[n],colors[n]);
      rbut->textColor=color(colors[n]);
      if (n==0) rbut->SetValue(true);

      stereo[n]=new Rot(rot_rect,0);
      stereo[n]->value= n==1 ? 2 :
                        n==2 ? 0 :
                        n==3 ? 2 : 1;
      AddChild(stereo[n]);
      top.y+=radiob_dist;
      rot_rect.OffsetBy(0,radiob_dist);
    }
  }
  void redraw_sliders() {
    for (int n=0;n<colors_max;++n) stereo[n]->Invalidate();
  }
  void Draw(BRect) {
    DrawString("colors",BPoint(2,10));
    DrawString("loc",BPoint(50,10));
  }
};

struct MScrollBar: BScrollBar {
  MScrollBar(BRect frame,BView *target,float mini,float maxi):
      BScrollBar(frame,"",target,mini,maxi,B_HORIZONTAL) {
    SetResizingMode(B_FOLLOW_BOTTOM|B_FOLLOW_H_CENTER);
  }
};

struct MusicSView {
  MusicView *musicView;
  BScrollBar* scrollbar;
  MusicSView(BRect rect) {
    musicView=new MusicView(BRect(rect.left,rect.top,rect.right,rect.bottom-B_H_SCROLL_BAR_HEIGHT),this);
    scrollbar=new MScrollBar(BRect(rect.left,rect.bottom-B_H_SCROLL_BAR_HEIGHT,rect.right-15,rect.bottom),
                             musicView,0,view_hmax);
  }
};

void MusicView::Draw(BRect rect) {
  int n,
      left=static_cast<int>((Bounds().left-x_off)/sect_len);
  if (left<0) left=0;
  if (score->len > musv_len) {
    musv_len=score->len;
    theSView->scrollbar->SetRange(0,(musv_len-sect_mus_max)*sect_len);
  }
  for (n=0;n<sclin_max;n++)
    score->lin[n]->eraseScLine(this,n,0,left,left+sect_mus_max+10);
  for (n=0;n<sclin_max;n++)
    score->lin[n]->drawScLine(this,n,0,left,left+sect_mus_max+10);
  for (n=0;n<score->len;n++) {
    score->time_mark(this,n);
    score->drawText(this,n);
    if (score->end_sect==n) break;
  }
  score->drawEnd(this,0);
  draw_start_stop(score,0,play_start,true);
  draw_start_stop(score,0,play_stop,false);
}

void MusicView::redraw(bool meter) {
  BRect rect(Bounds());
  float y;
  SetHighColor(cWhite);
  if (meter) {
    y=rect.top;
    FillRect(BRect(rect.left,y,rect.right,y+11)); // clear meter numbers
  }
  y=rect.top+ypos(sclin_max)-5;
  FillRect(BRect(rect.left,y,rect.right,rect.bottom)); // clear tune names
  Draw(rect);
}

ScopeBuf::ScopeBuf():occupied(-1),scope_window(0) { }

void ScopeBuf::set_buf(const int dim) {
  scope_window=dim;
  for (int n=0;n<scope_max;++n) buf[n]=new float[scope_window];
  reset();
}

void ScopeBuf::reset() {
  cur_buf=scope_max-1;
  occupied=-1;
  for (int m=0;m<scope_max;m++) for (int n=0;n<scope_window;n++) buf[m][n]=0.0;
}

int ScopeBuf::get_prev() {
  --cur_buf;
  if (occupied<scope_max-1) ++occupied;
  if (cur_buf<0) cur_buf=scope_max-1;
  return cur_buf;
}

struct ScopeView: BView {
  ScopeView(BRect rect): BView(rect,0,B_FOLLOW_RIGHT,B_WILL_DRAW) {
    SetViewColor(cTintedGrey);
  }
  void Draw(BRect rect) {
    ScopeBuf &scbuf=appWin->sco_buf;
    if (!scbuf.scope_window) return;
    int n,m,offset;
    const int scope_gap=4;     // gap between scope windows
    float *buf;
    const float half=rect.bottom/2,
                hi=half-20,
                lo=half+20;
    BRect area;
    for (m=0; m<scope_max && m<=scbuf.occupied; ++m) {
      buf=scbuf.buf[(scbuf.cur_buf+m)%scope_max];
      offset=(scbuf.occupied-m)*(scbuf.scope_window+scope_gap);

      area.Set(offset+1,1,offset+scbuf.scope_window-1,rect.bottom-1);
      SetHighColor(cLightBlue);
      FillRect(area);

      SetPenSize(1);  // dotted line
      SetHighColor(cBlue);
      for (n=5;n<scbuf.scope_window;n+=20) {
        StrokeLine(BPoint(offset+n,hi),BPoint(offset+n+2,hi));
        StrokeLine(BPoint(offset+n,lo),BPoint(offset+n+2,lo));
      }
      // if (debug) printf("m=%d occ=%d\n",m,scbuf.occupied);

      SetHighColor(cBlack);
      MovePenTo(offset,half-20*buf[0]);
      for (n=1;n<scbuf.scope_window;++n) StrokeLine(BPoint(n+offset,half-20*buf[n]));

      area.InsetBy(-1,-1);  // boarder
      StrokeRect(area);
    }
  }
};

struct ScopeSView: StdView {
  ScopeView *scopeView;
  BScrollView *scrollView;
  ScopeSView(BRect rect): StdView(rect) {
    rect.bottom-=B_H_SCROLL_BAR_HEIGHT;
    scopeView=new ScopeView(rect);
    scrollView=new BScrollView("scope",scopeView,B_FOLLOW_RIGHT,0,true,false,B_NO_BORDER);
  }
  void draw() {
    static bool init;
    if (!init) {
      init=true;
      scrollView->ScrollBar(B_HORIZONTAL)->SetRange(0,appWin->sco_buf.scope_window * scope_max);
    }
    scopeView->Draw(scopeView->Bounds());
  }
};

struct AppView: BView {
  Array<ScoreViewStruct*,3>scViews;
  MouseAction *mouseAction;
  Question *textView;
  MeterView *meterView;
  TempoView *tempoView;
  TunesSView *tunesSView;
  ColorView *colorView;
  MusicSView *musicSView;
  ScopeSView *scopeSView;
  AppView(BRect rect);
};

Score* Scores::exist_name(const char *nam) {
  if (!nam || !nam[0]) { alert("null name"); return 0; }
  for (int i=0;i<=lst_score;++i) {
    Score* sp=buf[i];
    if (!strcmp(sp->name,nam)) return sp;
  }
  return 0;
}


RedCtrl::RedCtrl(BRect rect): StdView(rect) {
  start_timbre=new HVSlider(BRect(0,0,70,ictrl_height-14),"diff/nrsin",'radn','radn',20.0,1,5,2,4);
  start_timbre->SetLimitLabels("1","5","2","4");
  start_timbre->SetValue(4);
  start_timbre->SetValueY(3);
  AddChild(start_timbre);

  timbre=new HVSlider(BRect(70,0,140,ictrl_height-14),"diff/nrsin",'rsdn','rsdn',20.0,1,5,2,4);
  timbre->SetLimitLabels("1","5","2","4");
  timbre->SetValue(3);
  timbre->SetValueY(3);
  AddChild(timbre);

  soft_attack=new CheckBox(BPoint(142,0),"soft attack",'soft');
  AddChild(soft_attack);
  
  startup=new HSlider(BRect(152,13,200,0),"startup",'rest',0,5);
  //startup->SetLimitLabels("0","5");
  startup->SetValue(2);
  AddChild(startup);

  decay=new HSlider(BRect(152,42,200,0),"decay",'rede',0,5);
  decay->SetLimitLabels("0","5");
  decay->SetValue(0);
  AddChild(decay);
}

void RedCtrl::Draw(BRect rect) {
  DrawString("start wave",BPoint(0,rect.bottom-5));
  DrawString("sustain wave",BPoint(72,rect.bottom-5));
}

void RedCtrl::redraw_sliders() {
  timbre->Invalidate();
  start_timbre->Invalidate();
  startup->Invalidate();
  soft_attack->Invalidate();
  decay->Invalidate();
}

BlackCtrl::BlackCtrl(BRect rect,BlackCtrl*& bc): StdView(rect) {
  bc=this;
  sub_band=new CheckBox(BPoint(90,60),"sub band",'blsu');
  AddChild(sub_band);

  fm_ctrl=new HVSlider(BRect(0,0,90,ictrl_height),"fm freq/index",0,0,30.0,0,7,0,7);
  fm_ctrl->SetLimitLabels("0","7","0","7");
  fm_ctrl->SetValue(2);
  set_fm(1);
  fm_ctrl->SetText("3.0");
  fm_ctrl->SetValueY(3);
  set_fm(2);
  fm_ctrl->SetTextY("2.0");
  fm_ctrl->modMessage=new BMessage('blfm');
  fm_ctrl->modMessageY=new BMessage('blin');
  AddChild(fm_ctrl);

  attack=new HSlider(BRect(160,0,210,0),"attack",'blaa',0,5);
  attack->SetLimitLabels("0","5");
  attack->SetValue(0);
  AddChild(attack);

  decay=new HSlider(BRect(160,slider_height,210,0),"decay",'blad',0,5);
  decay->SetLimitLabels("0","5");
  decay->SetValue(1);
  AddChild(decay);
}

void BlackCtrl::redraw_sliders() {
  fm_ctrl->SetText("%.2f",act_freq); fm_ctrl->SetTextY("%.1f",display_mod);
  fm_ctrl->Invalidate();
  attack->Invalidate();
  decay->Invalidate();
  sub_band->Invalidate();
}

BrownCtrl::BrownCtrl(BRect rect,BrownCtrl*& bc): StdView(rect) {
  bc=this;
  sub_band=new CheckBox(BPoint(90,60),"sub band",'subb');
  AddChild(sub_band);

  detune=new HSlider(BRect(95,0,145,0),"detune",'bdet',0,5);
  detune->SetLimitLabels("0","5");
  detune->SetValue(0);
  AddChild(detune);

  fm_ctrl=new HVSlider(BRect(0,0,90,ictrl_height),"fm freq/index",0,0,30.0,0,7,0,7);
  fm_ctrl->SetLimitLabels("0","7","0","7");
  fm_ctrl->SetValue(2);
  set_fm(1);
  fm_ctrl->SetText("3.0");
  fm_ctrl->SetValueY(3);
  set_fm(2);
  fm_ctrl->SetTextY("2.0");
  fm_ctrl->modMessage=new BMessage('fmfr');
  fm_ctrl->modMessageY=new BMessage('fmin');
  AddChild(fm_ctrl);

  attack=new HSlider(BRect(160,0,210,0),"attack",'brat',0,5);
  attack->SetLimitLabels("0","5");
  attack->SetValue(0);
  AddChild(attack);

  decay=new HSlider(BRect(160,slider_height,210,0),"decay",'brde',0,5);
  decay->SetLimitLabels("0","5");
  decay->SetValue(1);
  AddChild(decay);
}

void BrownCtrl::redraw_sliders() {
  fm_ctrl->SetText("%.2f",act_freq); fm_ctrl->SetTextY("%.1f",display_mod);
  fm_ctrl->Invalidate();
  attack->Invalidate();
  decay->Invalidate();
  detune->Invalidate();
  sub_band->Invalidate();
}

BlueCtrl::BlueCtrl(BRect rect): StdView(rect) {
  attack=new HSlider(BRect(5,0,55,0),"attack",'blat',0,5);
  attack->SetLimitLabels("0","5");
  attack->SetValue(0);
  AddChild(attack);
  decay=new HSlider(BRect(5,slider_height,55,0),"decay",'blde',0,5);
  decay->SetLimitLabels("0","5");
  decay->SetValue(2);
  AddChild(decay);
  p_attack=new CheckBox(BPoint(70,0),"piano attack",'piat');
  AddChild(p_attack);
  rich=new CheckBox(BPoint(70,15),"rich tone",0);
  AddChild(rich);
  chorus=new CheckBox(BPoint(70,30),"chorus",0);
  AddChild(chorus);
}

void BlueCtrl::redraw_sliders() {
  attack->Invalidate();
  decay->Invalidate();
  p_attack->Invalidate();
  rich->Invalidate();
  chorus->Invalidate();
}

GreenCtrl::GreenCtrl(BRect rect): StdView(rect) {
  tone=new VSlider(BRect(5,0,45,55),"tone",0,0,3);
  tone->SetLimitLabels("0","3");
  tone->SetValue(2);
  AddChild(tone);

  attack=new HSlider(BRect(60,0,110,0),"attack",'grat',0,5);
  attack->SetLimitLabels("0","5");
  attack->SetValue(0);
  AddChild(attack);
  decay=new HSlider(BRect(60,slider_height,110,0),"decay",'grde',0,5);
  decay->SetLimitLabels("0","5");
  decay->SetValue(1);
  AddChild(decay);
}

void GreenCtrl::redraw_sliders() {
  tone->Invalidate();
  attack->Invalidate();
  decay->Invalidate();
}

PurpleCtrl::PurpleCtrl(BRect rect,PurpleCtrl*& pc): StdView(rect) {
  pc=this;
  int n,
      left=0;
  static char *labels[]={ "1","2","3","6","10" };
  static int init[][harm_max]={{ 3,0,0,1,2 },{ 2,3,3,1,1 }};
  for (n=0;n<harm_max;++n) {
    st_harm[n]=new VSlider(BRect(left,0,left+14,55),labels[n],'puah',0,3);
    st_harm[n]->SetValue(init[0][n]);
    AddChild(st_harm[n]);
    left+=14;
  }
  left+=15;
  for (n=0;n<harm_max;++n) {
    harm[n]=new VSlider(BRect(left,0,left+14,55),labels[n],'purp',0,3);
    harm[n]->SetValue(init[1][n]);
    AddChild(harm[n]);
    left+=14;
  }
  set_st_hs_ampl(init[0]);
  set_hs_ampl(init[1]);
  start_dur=new HSlider(BRect(left+10,0,left+60,0),"startup",'pusu',0,5);
  start_dur->SetLimitLabels("0","5");
  start_dur->SetValue(2);
  AddChild(start_dur);
}

void PurpleCtrl::redraw_sliders() {
  for (int n=0;n<harm_max;++n) {
    st_harm[n]->Invalidate();
    harm[n]->Invalidate();
  }
  start_dur->Invalidate();
}

void PurpleCtrl::Draw(BRect rect) {
  DrawString("startup",BPoint(0,rect.bottom-14));
  DrawString("harmonics",BPoint(0,rect.bottom-4));
  DrawString("sustain",BPoint(5*14+15,rect.bottom-14));
  DrawString("harmonics",BPoint(5*14+15,rect.bottom-4));
}

ShowSampled::ShowSampled(BRect rect): StdView(rect) {
}

void ShowSampled::Draw(BRect rect) {
  RawData *raw;
  SetHighColor(cBlack);
  if (raw_data_okay) { 
    DrawString("sampled instruments:",BPoint(2,10));
    for (int n=0;n<colors_max;n++) {
      raw=RAW+col2wav_nr(appWin->colors[n]);
      SetHighColor(color(appWin->colors[n]));
      if (raw->file_name) DrawString(raw->file_name,BPoint(5,12*n+22));
    }
  }
  else
    DrawString("(wave files not yet read)",BPoint(2,10));
}

struct InfoText : StdView {
  UpdText *updTexts[3];
  InfoText(BRect rect):
      StdView(rect) {
    updTexts[0]=new UpdText(this,BPoint(10,22)); // .sco file
    updTexts[1]=new UpdText(this,BPoint(10,46)); // .scr file
    updTexts[2]=new UpdText(this,BPoint(54,58)); // measure nr
  }
  void Draw(BRect rect) {
    DrawString("score file:",BPoint(2,10));
    DrawString("script file:",BPoint(2,34));
    DrawString("measure:",BPoint(2,58));
    for (int n=0;n<3;++n)
      updTexts[n]->write(0);
  }
};

AppView::AppView(BRect frame): BView(frame,0,B_FOLLOW_ALL,0) {
  SetViewColor(cTintedGrey);
  SetLowColor(cTintedGrey);
  BRect rect(0,0,x_off+x_gap+sect_max*sect_len+score_info_len+2,scview_vmax),
        rect2;
  for (int n=0;n<3;n++) {
    scViews[n]=new ScoreViewStruct(rect,n);
    if (n<=scores.lst_score) scViews[n]->assign_score(scores[n]);
    else scViews[n]->scv_view->score=0;
    AddChild(scViews[n]->scroll_view->scrollbar);
    AddChild(scViews[n]->scv_view);
    AddChild(scViews[n]->scv_text);
    rect.top+=scview_vmax+4;
    rect.bottom+=scview_vmax+4;
  }

  rect.Set(0,rect.top,view_hmax,rect.top+18);
  mouseAction=new MouseAction(rect);
  AddChild(mouseAction);

  rect.Set(0,rect.bottom+4,view_hmax,view_vmax);
  musicSView=new MusicSView(rect);
  AddChild(musicSView->scrollbar);
  AddChild(musicSView->musicView);

  rect.Set(view_hmax-54,10,view_hmax-15,0);
  Button *but;
  AddChild(new Button(rect,"save...",'save',B_FOLLOW_RIGHT)); rect.top+=but_dist;

  AddChild(new Button(rect,"load...",'load',B_FOLLOW_RIGHT)); rect.top+=but_dist;

  but=new Button(rect,"new...",'new ',B_FOLLOW_RIGHT);
  but->textColor=cDarkBlue; AddChild(but); rect.top+=but_dist;

  but=new Button(rect,"copy...",'cp_t',B_FOLLOW_RIGHT);
  but->textColor=cDarkBlue; AddChild(but); rect.top+=but_dist;

  but=new Button(rect,"clear",'clr ',B_FOLLOW_RIGHT);
  but->textColor=cDarkBlue; AddChild(but); rect.top+=but_dist;
  
  AddChild(new Button(rect,"script...",'scr ',B_FOLLOW_RIGHT)); rect.top+=but_dist;
  AddChild(new Button(rect,"mod scr",'mods',B_FOLLOW_RIGHT)); rect.top+=but_dist;
  AddChild(new Button(rect,"cmd...",'cmd ',B_FOLLOW_RIGHT)); rect.top+=but_dist;

  but=new Button(rect,"play",'plmu',B_FOLLOW_RIGHT);
  but->textColor=cRed; AddChild(but); rect.top+=but_dist;

  but=new Button(rect,"stop",'stop',B_FOLLOW_RIGHT);
  AddChild(but); rect.top+=but_dist;

  float top=0;

  rect.Set(view_hmax-rbutview_left,top,view_hmax-rbutview_left+100,top+107);
  tunesSView=new TunesSView(rect);
  AddChild(tunesSView->scrollView);

  rect2.Set(rect.right+4,top,rect.right+58,top+3*radiob_dist+15);
  meterView=new MeterView(rect2);
  AddChild(meterView);

  top=rect2.bottom+4;
  rect2.Set(rect2.left,top,rect2.right,top+45);
  tempoView=new TempoView(rect2);
  AddChild(tempoView);

  top=rect2.bottom+4;
  rect2.Set(rect2.left,top,rect2.right+13,top+6*radiob_dist+15);
  colorView=new ColorView(rect2);
  AddChild(colorView);
  connect_stereo(colorView->stereo);

  top=rect.bottom+2;
  appWin->chords=new CheckBox(BPoint(rect.left+2,top),"chords",'chrd',B_FOLLOW_RIGHT);
  AddChild(appWin->chords);

  top+=checkbox_height;
  appWin->note_dist=new CheckBox(BPoint(rect.left+2,top),"note distances",'ndis',B_FOLLOW_RIGHT);
  AddChild(appWin->note_dist);

  top+=checkbox_height;
  appWin->solo=new CheckBox(BPoint(rect.left+2,top),"solo voice",0,B_FOLLOW_RIGHT);
  AddChild(appWin->solo);

  top+=checkbox_height;
  appWin->draw_col=new CheckBox(BPoint(rect.left+2,top),"draw with color",0,B_FOLLOW_RIGHT);
  AddChild(appWin->draw_col);

  top+=checkbox_height;
  appWin->use_raw=new CheckBox(BPoint(rect.left+2,top),"sampled notes",'uraw',B_FOLLOW_RIGHT);
  AddChild(appWin->use_raw);

  top+=checkbox_height;
  appWin->dumpwav=new CheckBox(BPoint(rect.left+2,top),"create wave file",0,B_FOLLOW_RIGHT);
  AddChild(appWin->dumpwav);

  top+=checkbox_height;
  appWin->midi_output=new CheckBox(BPoint(rect.left+2,top),"create midi file",0,B_FOLLOW_RIGHT);
  AddChild(appWin->midi_output);

  top+=checkbox_height;
  appWin->no_set=new CheckBox(BPoint(rect.left+2,top),"ignore set cmd's",0,B_FOLLOW_RIGHT);
  AddChild(appWin->no_set);

  top+=checkbox_height+2;
  rect.Set(rect.left,top,view_hmax,top+38);
  textView=new Question(rect);
  AddChild(textView);

  // instrument control views
  top+=42;
  rect.Set(view_hmax-rbutview_left,top,view_hmax,top+ictrl_height);
  AddChild(new BlackCtrl(rect,appWin->black_control));
  appWin->act_instr_ctrl=appWin->black_control;
  AddChild(appWin->red_control=new RedCtrl(rect));       appWin->red_control->Hide();
  AddChild(appWin->blue_control=new BlueCtrl(rect));     appWin->blue_control->Hide();
  AddChild(appWin->green_control=new GreenCtrl(rect));   appWin->green_control->Hide();
  AddChild(new PurpleCtrl(rect,appWin->purple_control)); appWin->purple_control->Hide();
  AddChild(new BrownCtrl(rect,appWin->brown_control));   appWin->brown_control->Hide();
  AddChild(appWin->show_sampled=new ShowSampled(rect));  appWin->show_sampled->Hide();

  top=rect.bottom+4;
  rect.Set(view_hmax-rbutview_left,top,view_hmax-84,top+70);
  scopeSView=new ScopeSView(rect);
  AddChild(scopeSView->scrollView);

  appWin->scope_range=new CheckBox(BPoint(rect.right+5,top+60),"show all",0,B_FOLLOW_RIGHT);
  AddChild(appWin->scope_range);

  appWin->info_text=new InfoText(BRect(view_hmax-80,top,view_hmax,top+60));
  AddChild(appWin->info_text);
}

struct Encode {
  char save_buf[50];
  char* set_meter(int m) {
    sprintf(save_buf,"%um%d",eSetMeter,m);
    return save_buf;
  }
  char* score_start(int score_nr,int note_col,int score_ampl,int flat_notes,int sharp_notes,int endsect) {
    sprintf(save_buf,"%uc%da%df%ds%d",eScore_start,note_col,score_ampl,flat_notes,sharp_notes);
    if (endsect>0) sprintf(save_buf+strlen(save_buf),"e%d",endsect);
    return save_buf;
  }
  char* play_note(int lnr,int snr,int dur,int sign,int stacc_sampled,int col,int ampl,int dlnr,int dsnr) {
    sprintf(save_buf,"%uL%dN%dd%di%ds%dc%da%d ",ePlayNote,lnr,snr,dur,sign,stacc_sampled,col,ampl);
    if (dlnr) sprintf(save_buf+strlen(save_buf)-1,"p%d,%d ",dlnr,dsnr);
    return save_buf;
  }
  bool decode(FILE *in,Score*& scp) {
    ScSection *sp;
    int res;
    uint opc=0;
    opc=atoi(save_buf);
    switch (opc) {
      case eSetMeter:
        if (sscanf(save_buf,"%*um%d",&appWin->act_meter)!=1) {
          alert("bad code: %s",save_buf); return false;
        }
        break;
      case eScore_start: {
          int n,lnr,
              flatn,sharpn;
          if (!scores.in_range(scores.lst_score+1)) return false;
          scp=scores[++scores.lst_score];
          if (scp) scp->reset();
          else scp=scores[scores.lst_score]=new Score(0,sect_max,eScView);
          res=sscanf(save_buf,"%*uc%da%df%ds%de%d",&scp->ncol,&scp->nampl,&flatn,&sharpn,&scp->end_sect);
          if (res==4||res==5) {
            scp->set_signs(flatn,sharpn);
            scp->signs_mode= flatn ? eFlat : sharpn ? eSharp : 0;
            if (res==5)    // end_sect > nop
              scp->check_len(scp->end_sect+1);
          }
          else if (res!=2) {
            alert("bad code: %s",save_buf);
            return false;
          };
          fscanf(in,"%s",textbuf);
          scp->name=strdup(textbuf);
        }
        break;
      case eScore_end:
        if (sscanf(save_buf,"%*ue%d",&scp->end_sect)!=1) { alert("bad code: %s",save_buf); return false; }
        scp->check_len(scp->end_sect+1);
        break;
      case ePlayNote: {
          int ind,snr,dur,sign,stacc_sampl,col,ampl,dlnr,dsnr;
          res=sscanf(save_buf,"%*uL%dN%dd%di%ds%dc%da%dp%d,%d",
                     &ind,&snr,&dur,&sign,&stacc_sampl,&col,&ampl,&dlnr,&dsnr);
          if (res==7) dlnr=dsnr=0;
          else if (res!=9) {
            alert("bad code: %s",save_buf); return false;
          }
          scp->check_len(snr+dur); // 1 extra
          sp=scp->get_section(ind,snr);
          for (int i=dur;;) {
            sp->cat=ePlay;
            sp->note_col=col;
            sp->sign=sign;
            sp->stacc=0;
            sp->sampled=0;
            sp->note_ampl=ampl;
            if (--i<=0) break;
            ++sp;
          }
          sp->stacc= stacc_sampl & 1;
          sp->sampled= (stacc_sampl>>1) & 1;
          sp->port_dlnr=abs(dlnr);
          sp->dlnr_sign= dlnr>=0;
          sp->port_dsnr=dsnr;
        }
        break;
      default: alert("unknown opcode %d",opc); return false;
    }
    return true;
  }
};

void set_filetype(const char *file,const char *type) {
  BFile bfile(file,0);
  if (bfile.InitCheck()==B_OK) {
    BNodeInfo ni(&bfile);
    if (ni.SetType(type)!=B_OK) alert("filetype %s not set",file);
    bfile.Unset();
  }
  else alert("no icon for %s",file);
}

bool AppWindow::save(const char *file) {
  FILE *tunes;
  int i,n;
  Encode enc;
  Score *scp;
  ScSection *sect,
            *first_sect;
  int scorenr,
      snr,lnr,
      flatn,sharpn;
  if ((tunes=fopen(file,"w"))==0) { alert("file '%s' not writable",file); return false; }
  fputs(enc.set_meter(act_meter),tunes);
  for (scorenr=0;scorenr<=scores.lst_score;++scorenr) {
    scp=scores[scorenr];
    flatn=sharpn=0;
    for (lnr=0;;++lnr) {
      switch (scp->lin[lnr]->note_sign) {
        case eLo: flatn|=1; break;
        case eHi: sharpn|=1; break;
      }
      if (lnr==6) break;
      flatn<<=1; sharpn<<=1;
    }
    fprintf(tunes,"\n%s %s ",
      enc.score_start(scorenr,scp->ncol,scp->nampl,flatn,sharpn,scp->end_sect),scp->name);
    for (lnr=0;lnr<sclin_max;lnr++) {
      for (snr=0;snr<scp->len-1;) {   // 1 extra for end
        if (scp->end_sect==snr) break;
        sect=scp->get_section(lnr,snr);
        switch (sect->cat) {
          case ePlay:
            first_sect=sect;
            for (i=1;;++i,++sect) {
              if (sect->sampled) { n=2; break; }
              if (sect->stacc) { n=1; break; }
              if (!eq(sect+1,first_sect)) { n=0; break; }
            }
            fputs(enc.play_note(lnr,snr,i,first_sect->sign,n,
                                first_sect->note_col,first_sect->note_ampl,
                                sect->dlnr_sign ? sect->port_dlnr : -sect->port_dlnr,sect->port_dsnr),tunes);
            snr+=i;
            break;
          case eSilent:
            ++snr;
            break;
          default:
            alert("save %u?",sect->cat);
            ++snr;
        }
      }
    }
  }
  putc('\n',tunes); // Unix tools need this
  fclose(tunes);
  set_filetype(file,tunes_mime);

  return true;
}

bool AppWindow::restore(const char *file) {
  FILE *in;
  bool result=true;
  Encode enc;
  if ((in=fopen(file,"r"))==0) {
    alert("file '%s' not found",file);
    return false;
  }
  scores.lst_score=-1;
  Score* scp=0;
  for (;;) {
    if (fscanf(in,"%s",enc.save_buf)<1) break;
    if (isdigit(enc.save_buf[0]) && isalpha(enc.save_buf[1])) {
      if (!enc.decode(in,scp)) { result=false; break; }
    }
    else {
      alert("bad .sco code: %s",enc.save_buf);
      result=false;
      break;
    }
  }
  fclose(in);
  return result;
}

void AppWindow::set_ctrl_sliders() {
  black_control->redraw_sliders();
  red_control->redraw_sliders();
  brown_control->redraw_sliders();
  green_control->redraw_sliders();
  purple_control->redraw_sliders();
  blue_control->redraw_sliders();
  appView->tempoView->redraw_sliders();
  appView->colorView->redraw_sliders();
}

int32 start_sampler(void*) {
  new Sampler(); // does not return
}  

AppWindow::AppWindow(AppWindow*& appWindow,char *appf,char *inf):
    BWindow(
      BRect(top_left,top_top,top_left+view_hmax,top_top+view_vmax),
      "the A'dam Music Composer - 1.12",
      B_DOCUMENT_WINDOW,B_ASYNCHRONOUS_CONTROLS|B_OUTLINE_RESIZE),
    app_file(appf),
    act_score(nop),
    act_color(eBlack),
    act_meter(8),
    act_action(nop),
    prev_mess(0),
    act_tempo(11),
    stop_requested(false),
    use_raw_data(false),
    appView(0),   // so crash if used prematurely
    chordsWin(0),
    ndistWin(0),
    input_file(inf),
    sco_buf(),
    mn_end(0),
    active_scoreCtrl(new RButtonAM1LampCtrl('acts')),
    scoreBuf(new Score("sc buf",sect_max,eScBuffer)),
    cur_score(0) {
  appWindow=this;
  if (inf) {
    char *s=input_file.get_ext();
    if (s && !strcmp(s,".scr")) {
      script_file.cpy(input_file.s);
      input_file.new_ext(".sco");
      if (!restore(input_file.s)) inf=0;
    }
    else if (!restore(input_file.s)) inf=0;
  }
  else {
    input_file.cpy("the.sco");
    scores[0]=new Score("tune1",sect_max,eScView);
    scores[1]=new Score("tune2",sect_max,eScView);
    scores[2]=new Score("tune3",sect_max,eScView);
    scores.lst_score=2;
  }
  appView=new AppView(Bounds());
  appView->SetFont(appFont);
  AddChild(appView);
  if (inf) info_text->updTexts[0]->write(input_file.strip_dir());

  if (script_file.s[0]) {
    if (read_script(script_file.s,appView->musicSView->musicView))
      info_text->updTexts[1]->write(script_file.strip_dir());
  }

  AddHandler(&appView->textView->handler);  
  msgr=new BMessenger(&appView->textView->handler);
  sampler_id=spawn_thread(start_sampler,"sampler",B_NORMAL_PRIORITY,0);
  resume_thread(sampler_id);
}

bool AppWindow::QuitRequested() {
  be_app->PostMessage(B_QUIT_REQUESTED);
  return true;
}

int key2num(key_info *key) {
  if (key->key_states[2])
    switch (key->key_states[2]) {
      case 32: return 1; case 16: return 2; case 8: return 3;
      case 4: return 4; case 2: return 5; case 1: return 6;
    }
  else if (key->key_states[3])
    switch (key->key_states[3]) {
      case 128: return 7; case 64: return 8; case 32: return 9;
      case 16: return 0;
    }
  return -1;
}

int key2char(key_info *key) {
  if (key->key_states[10]==1 && key->key_states[12]==32) return 'ud';
  if (key->key_states[10]==1) return 'up';    // up arrow
  if (key->key_states[12]==32) return 'do';   // down arrow
  if (key->key_states[7]==4) return 'sel';    // s
  if (key->key_states[5]==4) return 'uns';    // u
  if (key->key_states[6]==128) return 'prta'; // p
  if (key->key_states[10]==32) return 'move'; // m
  if (key->key_states[9]==2) return 'copy';   // c
  if (key->key_states[9]==16) return 'kp_a';  // shift, keep score active
  if (key->key_states[7]==8) return 'all';    // a
  if (key->key_states[8]==16) return 'keep';  // k
  return 0;
}

BView* col2ctrl(int col) {
  switch (col) {
    case eBlack: return appWin->black_control;
    case eRed: return appWin->red_control;
    case eBlue: return appWin->blue_control;
    case eGreen: return appWin->green_control;
    case ePurple: return appWin->purple_control;
    case eBrown: return appWin->brown_control;
    default: alert("col2ctrl col=%d",col); return 0;
  }
}

void AppWindow::check_reset_as() { // do not reset act_score if 'k' key pressed
  key_info key;
  get_key_info(&key);
  if (key2char(&key)!='kp_a') {
    active_scoreCtrl->reset();
    act_score=nop;
  }
}

void ScoreView::modify_sel(int mes) {
  SLList_elem<SectData> *sd;
  ScSection *sect;
  for (sd=selected.sd_list.lis;sd;sd=sd->nxt) {
    sect=score->lin[sd->d.lnr]->sect + sd->d.snr;
    switch (mes) {
      case 'uns': sect->sel=false; break;
      case 'rcol': sect->note_col=appWin->act_color; break;
      case 'amp+': if (sect->note_ampl<ampl_max) ++sect->note_ampl; break;
      case 'amp-': if (sect->note_ampl>1) --sect->note_ampl; break;
      case 'del ':
        if (sect->port_dlnr) sect->drawPortaLine(this,sd->d.snr,sd->d.lnr,d_off,true);
        sect->reset();
        break;
    }
    sect->drawSect(this,sd->d.snr,sd->d.lnr,d_off);
  }
  switch (mes) {
    case 'del ':
    case 'uns':
      selected.reset();
      score->drawEnd(this,d_off);
      break;
  }
}

bool AppWindow::read_tunes(const char* file) {
  TunesView *tv=appView->tunesSView->tunesView;
  tv->reset(scores.lst_score);
  if (!restore(file)) return false;  // sets scores.lst_score
  int n;
  for (n=0;n<3;++n) appView->scViews[n]->scv_view->score=0;
  for (n=0;n<=scores.lst_score;++n) {
    if (n<3) appView->scViews[n]->assign_score(scores[n]);
    scores[n]->rbut=tv->ctrl->AddButton(tv,tv->top,scores[n]->name,n);
    tv->top.y += radiob_dist;
  }
  for (n=0;n<3;++n)
    appView->scViews[n]->invalidate();
  appView->meterView->redraw();
  return true;
}

void AppWindow::MessageReceived(BMessage* mes) {
  int n,n1,n2;
  bool ok;
  int32 mval,
        cur_mess=mes->what;
  void *ptr;
  const char *txt;
  MusicView *mv;
  ScoreView *sv,*sv1;
  ScoreViewStruct *svs;
  TunesView *tv;
  Score *sc;
  Question *quest=appView->textView;
  static Str command; 
  switch (cur_mess) {
    case 'acts': // from active_scoreCtrl
      mes->FindInt32("",&act_score);
      break;
    case 'setm':  // from meterView
      mes->FindInt32("",&mval);
      if (mval!=act_meter) {
        act_meter=mval;
        for (n=0;n<3;++n) appView->scViews[n]->scv_view->redraw(true);
        appView->musicSView->musicView->redraw(true);
      }
      break;
    case 'tune': // from tunesView
      mes->FindInt32("",&mval);
      act_name=scores[mval]->name;
      if (act_score>nop) {
        sc=scores.exist_name(act_name);
        if (!sc) break;
        for (ok=true,n=0;n<3;++n) {
          sv1=appView->scViews[n]->scv_view;
          if (sv1->score && sv1->score==sc) {
            alert("tune already assigned");
            ok=false;
            break;
          }
        }
        if (ok) {
          svs=appView->scViews[act_score];
          sv=svs->scv_view;
          if (selected.sv==sv)
            selected.restore_sel();
          svs->assign_score(sc);
          svs->invalidate();
        }
        check_reset_as();
      }
      break;
    case 'mods':  // modify script with gap or part deletion
      alerts=0;
      mv=appView->musicSView->musicView;
      if (script_file.s[0])
        modify_script(script_file.s,mv->play_start,mv->play_stop,act_meter);
      else
        alert("no script file");
      break;
    case 'play':  // play a score
      alerts=0;
      mes->FindPointer("",&ptr);
      sco_buf.reset();
      sv=reinterpret_cast<ScoreView*>(ptr);
      cur_score=sv->score;
      sampler->playScore(sv->play_start,sv->play_stop);
      break;
    case 'ud  ':  // score up/down
      mes->FindPointer("",&ptr);
      sv=reinterpret_cast<ScoreView*>(ptr);
      svs=sv->theViewStruct;
      n=sv->d_off + svs->scv_text->ud->value * 7;
      sv->d_off=max(0,min(n,sclin_max-sclin_disp_max));
      sv->redraw(false);
      break;
    case 'puah': {  // update attack harmonics purple instr
        static int st_harmonics[harm_max];
        for(n=0;n<harm_max;++n)
          st_harmonics[n]=purple_control->st_harm[n]->value;
        purple_control->set_st_hs_ampl(st_harmonics);
      }
      break;
    case 'purp': {  // update purple instr
        static int harmonics[harm_max];
        for(n=0;n<harm_max;++n)
          harmonics[n]=purple_control->harm[n]->value;
        purple_control->set_hs_ampl(harmonics);
      }
      break;
    case 'pusu':  // startup duration purple instr
      purple_control->set_start_dur();
      break;
    case 'grat':  // attack green instrument
      green_control->set_attack();
      break;
    case 'grde':  // decay green instrument
      green_control->set_decay();
      break;
    case 'blaa':  // attack black instrument
      black_control->set_attack();
      break;
    case 'brat':  // attack brown instrument
      brown_control->set_attack();
      break;
    case 'bdet':  // detune brown instrument
      brown_control->set_detune();
      break;
    case 'blad':  // decay black instrument
      black_control->set_decay();
      break;
    case 'brde':  // decay brown instrument
      brown_control->set_decay();
      break;
    case 'rest':  // startup red instrument
      red_control->set_startup();
      break;
    case 'reat':  // attack red instrument
      red_control->set_attack();
      break;
    case 'rede':  // decay red instrument
      red_control->set_decay();
      break;
    case 'uraw':  // sampled waves
      use_raw_data=use_raw->value;
      act_instr_ctrl->Hide();
      if (use_raw_data)
        (act_instr_ctrl=show_sampled)->Show();
      else
        (act_instr_ctrl=col2ctrl(act_color))->Show();
      break;
    case 'blat':  // attack blue instrument
      blue_control->set_attack();
      break;
    case 'blde':  // decay blue instrument
      blue_control->set_decay();
      break;
    case 'piat':  // from blue_control, piano attack
      blue_control->set_piano();
      break;
    case 'dmod':  // display mode of scoreview
      mes->FindPointer("",&ptr);
      svs=reinterpret_cast<ScoreViewStruct*>(ptr);
      sv=svs->scv_view;
      sv->draw_mode=svs->scv_text->display_mode->value ? eAmplValue : eColorValue;
      sv->redraw(false);
      break;
    case 'ampl':  // set score amplitude
      mes->FindPointer("",&ptr);
      svs=reinterpret_cast<ScoreViewStruct*>(ptr);
      sv=svs->scv_view;
      sv->score->nampl=svs->scv_text->sc_ampl->value;
      for (n=0;n<sclin_max;++n) 
        for (n1=0;n1<sv->score->len;++n1)
          sv->score->get_section(n,n1)->note_ampl=sv->score->nampl;
      if (sv->draw_mode==eAmplValue) sv->redraw(false);
      break;
    case 'plmu':  // play musicView
      alerts=0;
      mv=appView->musicSView->musicView;
      if (dumpwav->value) {
        if (!(dumpwav_okay=init_dump_wav(wave_out_file)))
          alert("%s not opened",wave_out_file);
      }
      if (midi_output->value) {
        if (!(midiout_okay=midi_out.init(midi_out_file)))
          alert("%s not opened",midi_out_file);
        midi_output->SetValue(false);  
      }
      cur_score=mv->score;
      sampler->playScore(mv->play_start,mv->play_stop);
      break;
    case 'ctrl':  // from quest->handler
      quest->textControl->reset();
      txt=quest->textControl->Text();
      prev_mess=0;
      switch (quest->handler.mode) {
        case 'save':  // quest->handler.mode
          input_file.cpy(txt);
          if (save(txt)) {
            quest->text->write("saved");
            info_text->updTexts[0]->write(input_file.strip_dir()); // maybe txt was changed
          }
          else info_text->updTexts[0]->write("");
          break;
        case 'load':  // quest->handler.mode
          selected.reset();
          input_file.cpy(txt);
          if (read_tunes(txt)) {
            quest->text->write("loaded");
            info_text->updTexts[0]->write(input_file.strip_dir());
          }
          else info_text->updTexts[0]->write("");
          break;
        case 'new ':  // quest->handler.mode
          sc=scores.new_score(txt);
          sc->ncol=act_color;
          tv=appView->tunesSView->tunesView;
          sc->rbut=tv->ctrl->AddButton(tv,tv->top,sc->name,scores.lst_score);
          tv->top.y += radiob_dist;
          quest->text->write("tune added");
          if (act_score>nop) {
            svs=appView->scViews[act_score];
            if (selected.sv==svs->scv_view) selected.restore_sel();
            svs->assign_score(sc);
            svs->invalidate();
            check_reset_as();
          }
          break;
        case 'cp_t': {  // quest->handler.mode
            Score *from,*to;
            from=scores.exist_name(act_name);
            if (!from) break;
            to=scores.exist_name(txt);
            if (to) {
              to->reset();
              to->copy(from);
              for (n=0;n<3;++n) {
                svs=appView->scViews[n];
                sv1=svs->scv_view;
                if (sv1->score==to) {
                  svs->scv_text->sc_ampl->SetValue(to->nampl);
                  svs->invalidate();
                }
              }
            }
            else {
              to=scores.new_score(txt);
              to->ncol=act_color;
              tv=appView->tunesSView->tunesView;
              to->rbut=tv->ctrl->AddButton(tv,tv->top,to->name,scores.lst_score);
              tv->top.y += radiob_dist;
              to->copy(from);
            }
            if (act_score>nop) {
              svs=appView->scViews[act_score];
              if (selected.sv==svs->scv_view) selected.restore_sel();
              svs->assign_score(to);
              svs->invalidate();
              check_reset_as();
            }
          }
          quest->text->write("copied");
          break;
        case 'scr ':  // quest->handler.mode
          alerts=0;
          mv=appView->musicSView->musicView;
          if (mv->score->lst_sect>0) {
            if (script_file==txt) mv->reset(false); else mv->reset(true);
            mv->redraw(false);
          }
          script_file.cpy(txt);
          if (read_script(txt,mv)) {
            quest->text->write("read");
            info_text->updTexts[1]->write(script_file.strip_dir());
          }
          else info_text->updTexts[1]->write("");
          break;
        case 'cmd ':  // quest->handler.mode
          alerts=0;
          command.cpy(txt);
          if (read_script(0,appView->musicSView->musicView)) quest->text->write("read");
          break;
        case 0:
          alert("question mode = 0");
          break;
        default: alert("unk question mode");
      }
      break;
    case 'save':
      txt=input_file.s;
      quest->text->bgColor=cPink;
      quest->text->write("save as:");
      quest->textControl->SetText(txt);
      if (prev_mess==cur_mess) {
        prev_mess=0;
        if (save(txt)) quest->text->write("saved");
      }
      else {
        prev_mess=cur_mess;
        msgr->SendMessage(mes);
      }
      break;
    case 'new ':  // new score
      quest->text->bgColor=cPink;
      quest->text->write("new tune:");
      quest->textControl->SetText("no-name");
      msgr->SendMessage(mes);
      break;
    case 'cp_t':  // copy tune
      quest->text->bgColor=cPink;
      quest->text->write("copy to tune:");
      quest->textControl->SetText("no-name");
      msgr->SendMessage(mes);
      break;
    case 'load':    // load score file
      txt=input_file.s;
      quest->text->bgColor=cPink;
      quest->text->write("load:");
      quest->textControl->SetText(txt);
      if (prev_mess==cur_mess) {
        prev_mess=0;
        if (read_tunes(txt)) {
          quest->text->write("loaded");
          info_text->updTexts[0]->write(input_file.strip_dir());
        }
        else info_text->updTexts[0]->write("");
      }
      else {
        prev_mess=cur_mess;
        msgr->SendMessage(mes);
      }
      break;
    case 'clr ':    // clear a score
      if (act_score>nop) {
        sv=appView->scViews[act_score]->scv_view;
        if (selected.sv==sv)
          selected.reset();
        sv->score->reset();
        sv->score->ncol=act_color;
        for (n=0;n<3;++n) {
          sv1=appView->scViews[n]->scv_view;
          if (sv1->score && sv1->score==sv->score) sv1->Invalidate();
        }
        check_reset_as();
      }
      else {
        mv=appView->musicSView->musicView;
        if (mv->score->lst_sect>=0) {
          mv->reset(true);
          mv->redraw(true); // clear start and stop sign 
        }
      }
      break;
    case 'scr ':    // read script
      if (!script_file.s[0]) {
        script_file.cpy(input_file.s);
        script_file.new_ext(".scr");
      }
      quest->text->bgColor=cPink;
      quest->text->write("script:");
      quest->textControl->SetText(script_file.s);
      if (prev_mess==cur_mess) {
        prev_mess=0;
        alerts=0;
        mv=appView->musicSView->musicView;
        mv->reset(false);
        mv->redraw(false);
        if (read_script(script_file.s,mv)) {
          quest->text->write("read");
          info_text->updTexts[1]->write(script_file.strip_dir());
        }
        else info_text->updTexts[1]->write("");
      }
      else {
        prev_mess=cur_mess;
        msgr->SendMessage(mes);
      }
      break;
    case 'cmd ':  // command
      quest->text->bgColor=cPink;
      quest->text->write("command:");
      quest->textControl->SetText(command.s);
      if (prev_mess==cur_mess) {
        prev_mess=0;
        if (read_script(0,appView->musicSView->musicView)) quest->text->write("read");
      }
      else {
        prev_mess=cur_mess;
        msgr->SendMessage(mes);
      }
      break;
    case 'tmpo':  // from tempoView
      act_tempo=appView->tempoView->tempo->value;
      appView->tempoView->show_val();
      break;
    case 'radn':  // from red_control, attack diff/nrsin
      red_control->set_start_timbre();
      break;
    case 'rsdn':  // from red_control, sustain diff/nsin
      red_control->set_timbre();
      break;
    case 'blfm':  // from black_control, fm freq
      black_control->set_fm(1);
      black_control->fm_ctrl->SetText("%.2f",black_control->act_freq);
      black_control->fm_ctrl->UpdateText();
      break;
    case 'fmfr':  // from brown_control, fm freq
      brown_control->set_fm(1);
      brown_control->fm_ctrl->SetText("%.2f",brown_control->act_freq);
      brown_control->fm_ctrl->UpdateText();
      break;
    case 'blin':  // from black_control, fm index
      black_control->set_fm(2);
      black_control->fm_ctrl->SetTextY("%.1f",black_control->display_mod);
      black_control->fm_ctrl->UpdateText();
      break;
    case 'fmin':  // from brown_control, fm index
      brown_control->set_fm(2);
      brown_control->fm_ctrl->SetTextY("%.1f",brown_control->display_mod);
      brown_control->fm_ctrl->UpdateText();
      break;
    case 'blsu':  // from black_control, sub band freq
      black_control->set_fm(1);
      black_control->fm_ctrl->SetText("%.2f",black_control->act_freq);
      black_control->fm_ctrl->UpdateText();
      break;
    case 'subb':  // from brown_control, sub band freq
      brown_control->set_fm(1);
      brown_control->fm_ctrl->SetText("%.2f",brown_control->act_freq);
      brown_control->fm_ctrl->UpdateText();
      break;
    case 'soft':  // from red_control, soft attack
      red_control->set_attack();
      break;
    case 'msta':  // from mouseAction
      mes->FindInt32("",&act_action);
      break;
    case 'colr':  // from colorView
      mes->FindInt32("",&act_color);
      if (act_score>nop) {
        sv=appView->scViews[act_score]->scv_view;
        if (sv->score) sv->score->ncol=act_color;
        for (n=0;n<3;++n) {
          sv1=appView->scViews[n]->scv_view;
          if (sv1->score && sv1->score==sv->score) {
            for (n1=0;n1<sclin_max;++n1) {
              ScLine *lin=sv1->score->lin[n1];
              for (n2=0;n2<sv1->score->len;++n2) {
                lin->sect[n2].note_col=act_color;
                lin->sect[n2].nxt_note=0;  // needed if 2 tunes are joined
              }
            }
            sv1->redraw(false);
          }
        }
        check_reset_as();
      }
      act_instr_ctrl->Hide();
      if (use_raw->value)
        (act_instr_ctrl=show_sampled)->Show();
      else
        (act_instr_ctrl=col2ctrl(act_color))->Show();
      break;
    case 'csnr': // current section nr
      mes->FindInt32("",&mval);
      info_text->updTexts[2]->write("%d",mval/act_meter);
      break;
    case 'scop':
      appView->scopeSView->draw();
      break;
    case 'upsl':  // from BufferProc()
      set_ctrl_sliders();
      if (dumpwav_okay) {
        if (close_dump_wav()) alert("wave file %s created",wave_out_file);
        else alert("%s not closed",wave_out_file);
        dumpwav_okay=false;
        dumpwav->SetValue(false);  
      }
      info_text->updTexts[2]->write("stop");
      break;
    case 'stop':  // from stop button
      stop_requested=true;
      break;
    case 'chrd':   // chords button
      if (chords->value) {
        BPoint top(appWin->Frame().right+10,appWin->Frame().top);
        if (!chordsWin)
          chordsWin=new ChordsWindow(top);
        else
          chordsWin->MoveTo(top);
        chordsWin->Show();
      }
      else
        if (chordsWin) chordsWin->Hide();
      break;
    case 'ndis':   // note distances button
      if (note_dist->value) {
        mv=appView->musicSView->musicView;
        if (mv->play_start > mv->play_stop) {
          alert("note dist: stop must be > 0 and > start");
          note_dist->SetValue(false);
          break;
        }
        BPoint top(appWin->Frame().right+10,appWin->Frame().top);
        if (!ndistWin) ndistWin=new NoteDistWindow(top);
        else ndistWin->MoveTo(top);
        ndistWin->set(mv->play_start,mv->play_stop,mv->score);
        ndistWin->Show();
      }
      else {
        if (ndistWin) ndistWin->Hide();
      }
      break;
    case 'chkn':   // from chords window
      if (act_score>nop) {
        sv=appView->scViews[act_score]->scv_view;
        if (sv->score) {
          selected.restore_sel();
          selected.sv=sv;
          sv->score->put_chord(sv); // chord will be selected
        }
        check_reset_as();
      }
      break;
    case 'setk':  // from chords window, set_key button
      if (act_score>nop) {
        sv=appView->scViews[act_score]->scv_view;
        if (sv->score) {
          sv->score->tone2signs();
          sv->SetHighColor(cWhite);
          sv->FillRect(BRect(0,0,x_off-1,sv->Bounds().bottom)); // clear old signs
          sv->drawSigns();
        }
        check_reset_as();
      }
      break;
    case 'uns':      // unselect
    case 'rcol':     // re-color selected
    case 'amp+':     // amplitude+1 selected
    case 'amp-':     // amplitude-1 selected
    case 'del ':
      if (selected.sv) selected.sv->modify_sel(cur_mess);
      appView->mouseAction->reset();
      break;
    default:
      BWindow::MessageReceived(mes);
  }
}

Score* Scores::new_score(const char *name) {
  if (!in_range(lst_score+1)) return buf[0];
  ++lst_score;
  Score* scp=buf[lst_score];
  if (!scp) scp=buf[lst_score]=new Score(strdup(name),sect_max,eScView);
  else { scp->reset(); scp->name=strdup(name); }
  return scp;
}

void get_token(Str& str,FILE* in,int mode,char* delim,int& pos) {
  if (debug) printf("get_token: str=[%s] mode=%d delim=[%s] pos=%d\n",str.s,mode,delim,pos);
  switch (mode) {
    case eFile: str.rword(in,delim); return;
    case eString: str.strtok(appWin->appView->textView->textControl->Text(),delim,pos); return;
    default: alert("get_token: unk mode %d",mode);
  }
}

int read_time(Str& str,FILE *in,int mode,int& pos) {   // e.g. time:2.3
  get_token(str,in,mode," .\n;",pos);
  int nr=atoi(str.s) * appWin->act_meter;
  if (str.ch=='.') {
    get_token(str,in,mode," \n;",pos);
    nr+=atoi(str.s);
  }
  return nr;
}

void read_times(FILE *in,Str& str,int mode,int& pos,Array<int,times_max>& tim,const int stop) {   // e.g. time:2.3,4
  int nr;
  for (int n=0;;++n) {
    get_token(str,in,mode," .\n;,",pos);
    nr=atoi(str.s) * appWin->act_meter;
    if (str.ch=='.') {
      get_token(str,in,mode," \n;,",pos);
      nr+=atoi(str.s);
    }
    tim[n]=nr;
    if (str.ch!=',') { tim[n+1]=stop; return; }
  }
}

bool AppWindow::exec_cmd(ScInfo& info) {
  int n;
  switch (info.tag) {
    case eText:
      break;
    case eTempo:
      act_tempo=info.n;
      break;
    case eRed_att_timbre:
      red_control->start_timbre->value=info.n2[0];
      red_control->start_timbre->valueY=info.n2[1];
      red_control->set_start_timbre();
      break;
    case eRed_timbre:
      red_control->timbre->value=info.n2[0];
      red_control->timbre->valueY=info.n2[1];
      red_control->set_timbre();
      break;
    case eRed_attack:
      red_control->startup->value=info.n;
      red_control->set_startup();
      break;
    case eRed_decay:
      red_control->decay->value=info.n;
      red_control->set_decay();
      break;
    case ePurple_a_harm: {
        static int st_harmonics[harm_max];
        for (n=0;n<harm_max;++n) purple_control->st_harm[n]->value = st_harmonics[n] = info.n5[n];
        purple_control->set_st_hs_ampl(st_harmonics);
      }
      break;
    case ePurple_s_harm: {
        static int harmonics[harm_max];
        for (n=0;n<harm_max;++n) purple_control->harm[n]->value = harmonics[n] = info.n5[n];
        purple_control->set_hs_ampl(harmonics);
      }
      break;
    case ePurple_attack:
      purple_control->start_dur->value=info.n;
      purple_control->set_start_dur();
      break;
    case eBlue_attack:
      blue_control->attack->value=info.n;
      blue_control->set_attack();
      break;
    case eBlue_decay:
      blue_control->decay->value=info.n;
      blue_control->set_decay();
      break;
    case eBlue_piano:
      blue_control->p_attack->value=info.b;
      blue_control->set_piano();
      break;
    case eBlue_rich:
      blue_control->rich->value=info.b;
      break;
    case eBlue_chorus:
      blue_control->chorus->value=info.b;
      break;
    case eBlack_mod:
      black_control->fm_ctrl->value=info.n2[0];
      black_control->fm_ctrl->valueY=info.n2[1];
      black_control->set_fm(1);
      black_control->set_fm(2);
      break;
    case eBlack_attack:
      black_control->attack->value=info.n;
      black_control->set_attack();
      break;
    case eBlack_decay:
      black_control->decay->value=info.n;
      black_control->set_decay();
      break;
    case eBrown_mod:
      brown_control->fm_ctrl->value=info.n2[0];
      brown_control->fm_ctrl->valueY=info.n2[1];
      brown_control->set_fm(1);
      brown_control->set_fm(2);
      break;
    case eBrown_detune:
      brown_control->detune->value=info.n;
      brown_control->set_detune();
      break;
    case eBrown_attack:
      brown_control->attack->value=info.n;
      brown_control->set_attack();
      break;
    case eBrown_decay:
      brown_control->decay->value=info.n;
      brown_control->set_decay();
      break;
    case eBlack_subband:
      black_control->sub_band->value=info.b;
      black_control->set_fm(1);
      break;
    case eBrown_subband:
      brown_control->sub_band->value=info.b;
      brown_control->set_fm(1);
      break;
    case eRed_soft:
      red_control->soft_attack->value=info.b;
      red_control->set_attack();
      break;
    case eGreen_tone:
      green_control->tone->value=info.n;
      break;
    case eGreen_attack:
      green_control->attack->value=info.n;
      green_control->set_attack();
      break;
    case eGreen_decay:
      green_control->decay->value=info.n;
      green_control->set_decay();
      break;
    case eBlack_loc:
      appView->colorView->stereo[0]->value=info.n;
      break;
    case eRed_loc:
      appView->colorView->stereo[1]->value=info.n;
      break;
    case eGreen_loc:
      appView->colorView->stereo[2]->value=info.n;
      break;
    case eBlue_loc:
      appView->colorView->stereo[3]->value=info.n;
      break;
    case eBrown_loc:
      appView->colorView->stereo[4]->value=info.n;
      break;
    case ePurple_loc:
      appView->colorView->stereo[5]->value=info.n;
      break;
    default:
      alert("exec_cmd: unknown tag %u",info.tag);
      return false;
  }
  return true;
}

void AppWindow::set_slider(int N,Str &str,FILE *in,int mode,int &pos,const char* col,
                           int en,Array<int,times_max>times,MusicView *musV) {
  int n;
  ScInfo info;
  get_token(str,in,mode," \n;",pos);
  n=atoi(str.s);
  if (!isdigit(str.s[0])) { alert("bad %s value: %s",col,str.s); n=0; }
  if (n<0 || n>N) { alert("bad %s value: %d",col,n); n=0; }
  info.tag=en;
  info.n=n;
  for (n=0;times[n]!=eo_arr;++n) musV->upd_info(times[n],info);
}

void AppWindow::set_bool(Str &str,FILE *in,int mode,int &pos,const char* col,
                         int en,Array<int,times_max>times,MusicView *musV) {
  int n;
  ScInfo info;
  get_token(str,in,mode," \n;",pos);
  if (str=="on") info.b=true;
  else if (str=="off") info.b=false;
  else { alert("bad %s value: %s",col,str.s); info.b=false; }
  info.tag=en;
  for (n=0;times[n]!=eo_arr;++n) musV->upd_info(times[n],info);
}

void AppWindow::set_loc(Str &str,FILE *in,int mode,int &pos,const char* word,
                           int en,Array<int,times_max>times,MusicView *musV) {
  int n;
  ScInfo info;
  get_token(str,in,mode," \n;",pos);
  if (str=="left") info.n=0;
  else if (str=="mid") info.n=1;
  else if (str=="right") info.n=2;
  else { alert("bad %s value: %s",word,str.s); info.n=1; }
  info.tag=en;
  for (n=0;times[n]!=eo_arr;++n) musV->upd_info(times[n],info);
}

bool eq(char *word,char *prefix,char *all,char *&s) {
/* word="attack", prefix="red", all="red-attack"
   or:
   word="red-attack", all="red-attack"
*/
  s=all;
  int plen=strlen(prefix);
  if (!strcmp(word,all) ||
      plen && !strncmp(prefix,all,plen) && !strcmp(word,all+plen+1))
    return true;
  return false;
}

bool AppWindow::read_script(const char *script,MusicView* musV) {
  // calls add_copy(), which draws sections, meter and tune names in musicView
  FILE *in=0;
  int mode;
  if (!script) mode=eString;
  else {
    mode=eFile;
    if (!(in=fopen(script,"r"))) {
      alert("'%s' unknown",script); return false;
    }
  }
  Score *sc=0;
  Str str;
  int n,n1,
      cmd,
      pos,
      line_nr;
  char *s,
       prefix[10];
  ScInfo info;
  Array<int,times_max> times;
  str.comm_ch='#';
  for (pos=0,line_nr=1;;) {
    get_token(str,in,mode," \n;",pos);
    if (mode==eString) {
      if (!str.s[0] && str.ch==0) break;
    }
    else {
      if (str.ch==EOF) {
        fclose(in);
        set_filetype(script,script_mime);
        break;
      }
      if (!str.s[0]) { ++line_nr; continue; }
    }
    if (debug) printf("cmd:%s\n",str.s);
    if ((cmd=eAdd,str=="add") ||          // from scores to music
        (cmd=eTake,str=="take") ||        // from scores to scoreBuf
        (cmd=eTake_nc,str=="take-nc")) {  // from scores to scoreBuf, no clear
      sc= cmd==eAdd ? scoreBuf : 0;
      times[0]=nop; times[1]=eo_arr;
      int ampl=nop,
          start=0,
          stop=nop,
          shift=0,
          raise=0;
      for (;;) {
        if (strchr("\n;",str.ch)) {
          if (!sc) { alert("error in script"); return false; }
          int act_stop = stop>0 ? stop : sc->end_sect>0 ? sc->end_sect : sc->len;
          if (cmd==eAdd) {
            for (n=0;times[n]!=eo_arr;++n)
              musV->score->add_copy(sc,start,act_stop,times[n],ampl,shift,raise,musV);
          }
          else if (cmd==eTake || cmd==eTake_nc) {
            if (times[1]!=eo_arr) { alert("multiple 'time:' in 'take' command"); return false; }
            if (cmd==eTake) {
              scoreBuf->reset();
              scoreBuf->name=sc->name; // needed for command "add"
              scoreBuf->ncol=sc->ncol;
              scoreBuf->nampl=sc->nampl;
              scoreBuf->signs_mode=sc->signs_mode;
            }
            scoreBuf->add_copy(sc,start,act_stop,times[0],ampl,shift,raise,0);
          }
          break;
        }
        get_token(str,in,mode," :\n;",pos);
        if (debug) printf("add/take cmd: %s ch=[%c]\n",str.s,str.ch);
        if (str.ch==':') {
          if (str=="time") read_times(in,str,mode,pos,times,eo_arr);
          else if (str=="ampl") { get_token(str,in,mode," \n;",pos); ampl=atoi(str.s); }
          else if (str=="from") start=read_time(str,in,mode,pos);
          else if (str=="to") stop=read_time(str,in,mode,pos);
          else if (str=="shift") { get_token(str,in,mode," \n;",pos); shift=atoi(str.s); }
          else if (str=="raise") { get_token(str,in,mode," \n;",pos); raise=atoi(str.s); }
          else { alert("unk option '%s'",str.s); return false; }
        }
        else {
          sc=scores.exist_name(str.s);
          if (!sc) { alert("tune '%s' unknown",str.s); return false; }
        }
      }
    }
    else if (str=="put") {      // from scoreBuf to scores
      if (str.ch!=' ') { alert("tune name missing after put cmd"); return false; }
      get_token(str,in,mode," \n;",pos);
      sc=scores.exist_name(str.s);
      if (sc) {
        sc->reset();
        sc->copy(scoreBuf);
        for (n=0;n<3;++n)
          if (appView->scViews[n]->scv_view->score==sc)
            appView->scViews[n]->invalidate();
      }
      else {
        if (!scores.in_range(scores.lst_score+1)) return false;
        sc=scores[++scores.lst_score];
        if (!sc)
          sc=scores[scores.lst_score]=new Score(strdup(str.s),sect_max,eScView);
        sc->copy(scoreBuf);
        sc->name=strdup(str.s);
        TunesView *tv=appView->tunesSView->tunesView;
        sc->rbut=tv->ctrl->AddButton(tv,tv->top,sc->name,scores.lst_score);
        tv->top.y += radiob_dist;
      }
    }
    else if (str=="set") {
      prefix[0]=0;
      MusicView* musV=appView->musicSView->musicView;
      times[0]=0; times[1]=eo_arr;
      for (;;) {
        get_token(str,in,mode," :\n;",pos);
        if (debug) printf("set cmd:%s\n",str.s);
        if (str.ch==':') {
          if (str=="time")
            read_times(in,str,mode,pos,times,eo_arr);
          else if (str=="tempo") {
            get_token(str,in,mode," \n;",pos);
            info.tag=eTempo;
            info.n=atoi(str.s)/10;
            for (n=0;times[n]!=eo_arr;++n)
              musV->upd_info(times[n],info);
          }
          else if (eq(str.s,prefix,"red-start-wave",s)) {
            get_token(str,in,mode," ,\n;",pos);
            if (str.ch!=',') { alert("bad %s syntax",s); return false; }
            n=atoi(str.s);
            if (n<1 || n>5) { alert("bad %s value: %d",s,n); n=3; }
            info.n2[0]=n;
            get_token(str,in,mode," \n;",pos);
            n=atoi(str.s);
            if (n<2 || n>5) { alert("bad %s value: %d",s,n); n=3; }
            info.n2[1]=n;
            info.tag=eRed_att_timbre;
            for (n=0;times[n]!=eo_arr;++n) musV->upd_info(times[n],info);
          }
          else if (eq(str.s,prefix,"red-sustain-wave",s)) {
            get_token(str,in,mode," ,\n;",pos);
            if (str.ch!=',') { alert("bad %s syntax",s); return false; }
            n=atoi(str.s);
            if (n<1 || n>5) { alert("bad %s value: %d",s,n); n=3; }
            info.n2[0]=n;
            get_token(str,in,mode," \n;",pos);
            n=atoi(str.s);
            if (n<2 || n>5) { alert("bad %s value: %d",s,n); n=3; }
            info.n2[1]=n;
            info.tag=eRed_timbre;
            for (n=0;times[n]!=eo_arr;++n) musV->upd_info(times[n],info);
          }
          else if (eq(str.s,prefix,"red-decay",s))
//          else if (s="red-decay",str==s)
            set_slider(5,str,in,mode,pos,s,eRed_decay,times,musV);
          else if (eq(str.s,prefix,"red-startup",s))
            set_slider(5,str,in,mode,pos,s,eRed_attack,times,musV);
          else if (eq(str.s,prefix,"green-attack",s))
            set_slider(5,str,in,mode,pos,s,eGreen_attack,times,musV);
          else if (eq(str.s,prefix,"green-decay",s))
            set_slider(5,str,in,mode,pos,s,eGreen_decay,times,musV);
          else if (eq(str.s,prefix,"green-tone",s))
            set_slider(3,str,in,mode,pos,s,eGreen_tone,times,musV);
          else if (eq(str.s,prefix,"blue-attack",s))
            set_slider(5,str,in,mode,pos,s,eBlue_attack,times,musV);
          else if (eq(str.s,prefix,"blue-decay",s))
            set_slider(5,str,in,mode,pos,s,eBlue_decay,times,musV);
          else if (eq(str.s,prefix,"blue-piano",s))
            set_bool(str,in,mode,pos,s,eBlue_piano,times,musV);
          else if (eq(str.s,prefix,"blue-rich",s))
            set_bool(str,in,mode,pos,s,eBlue_rich,times,musV);
          else if (eq(str.s,prefix,"blue-chorus",s))
            set_bool(str,in,mode,pos,s,eBlue_chorus,times,musV);
          else if ((eq(str.s,prefix,"black-fm",s))) {
            get_token(str,in,mode," ,\n;",pos);
            if (str.ch!=',') { alert("bad %s syntax",s); return false; }
            n=atoi(str.s);
            if (n<0 || n>7) { alert("bad %s value: %d",s,n); n=0; }
            info.n2[0]=n;
            get_token(str,in,mode," \n;",pos);
            n=atoi(str.s);
            if (n<0 || n>7) { alert("bad %s value: %d",s,n); n=0; }
            info.n2[1]=n;
            info.tag=eBlack_mod;
            for (n=0;times[n]!=eo_arr;++n) musV->upd_info(times[n],info);
          }
          else if (eq(str.s,prefix,"black-decay",s))
            set_slider(5,str,in,mode,pos,s,eBlack_decay,times,musV);
          else if (eq(str.s,prefix,"black-attack",s))
            set_slider(5,str,in,mode,pos,s,eBlack_attack,times,musV);
          else if (eq(str.s,prefix,"black-subband",s))
            set_bool(str,in,mode,pos,s,eBlack_subband,times,musV);
          else if ((eq(str.s,prefix,"brown-fm",s)) || (eq(str.s,prefix,"brown-mod",s))) {
            get_token(str,in,mode," ,\n;",pos);
            if (str.ch!=',') { alert("bad %s syntax",s); return false; }
            n=atoi(str.s);
            if (n<0 || n>7) { alert("bad %s value: %d",s,n); n=0; }
            info.n2[0]=n;
            get_token(str,in,mode," \n;",pos);
            n=atoi(str.s);
            if (n<0 || n>7) { alert("bad %s value: %d",s,n); n=0; }
            info.n2[1]=n;
            info.tag=eBrown_mod;
            for (n=0;times[n]!=eo_arr;++n) musV->upd_info(times[n],info);
          }
          else if (eq(str.s,prefix,"brown-decay",s))
            set_slider(5,str,in,mode,pos,s,eBrown_decay,times,musV);
          else if (eq(str.s,prefix,"brown-detune",s))
            set_slider(5,str,in,mode,pos,s,eBrown_detune,times,musV);
          else if (eq(str.s,prefix,"brown-attack",s))
            set_slider(5,str,in,mode,pos,s,eBrown_attack,times,musV);
          else if (eq(str.s,prefix,"brown-subband",s))
            set_bool(str,in,mode,pos,s,eBrown_subband,times,musV);
          else if (eq(str.s,prefix,"red-soft",s))
            set_bool(str,in,mode,pos,s,eRed_soft,times,musV);
          else if (eq(str.s,prefix,"purple-start-harm",s)) {
            info.tag=ePurple_a_harm;
            for (n=0;n<harm_max;++n) {
              get_token(str,in,mode,", \n;",pos);
              if (str.ch!=',' && n!=4) { alert("%s: no comma",s); return false; }
              n1=atoi(str.s);
              if (n1<0 || n1>3) { alert("bad %s value: %d",s,n1); n1=0; }
              info.n5[n]=n1;
            }
            for (n=0;times[n]!=eo_arr;++n)
              musV->upd_info(times[n],info);
          }
          else if (eq(str.s,prefix,"purple-sustain-harm",s)) {
            info.tag=ePurple_s_harm;
            for (n=0;n<harm_max;++n) {
              get_token(str,in,mode,", \n;",pos);
              if (str.ch!=',' && n!=4) { alert("%s: no comma",s); return false; }
              n1=atoi(str.s);
              if (n1<0 || n1>3) { alert("bad %s value: %d",s,n1); n1=0; }
              info.n5[n]=n1;
            }
            for (n=0;times[n]!=eo_arr;++n)
              musV->upd_info(times[n],info);
          }
          else if (eq(str.s,prefix,"purple-startup",s)) 
            set_slider(5,str,in,mode,pos,s,ePurple_attack,times,musV);
          else if (eq(str.s,prefix,"black-loc",s))
            set_loc(str,in,mode,pos,s,eBlack_loc,times,musV);
          else if (eq(str.s,prefix,"red-loc",s))
            set_loc(str,in,mode,pos,s,eRed_loc,times,musV);
          else if (eq(str.s,prefix,"green-loc",s))
            set_loc(str,in,mode,pos,s,eGreen_loc,times,musV);
          else if (eq(str.s,prefix,"blue-loc",s))
            set_loc(str,in,mode,pos,s,eBlue_loc,times,musV);
          else if (eq(str.s,prefix,"brown-loc",s))
            set_loc(str,in,mode,pos,s,eBrown_loc,times,musV);
          else if (eq(str.s,prefix,"purple-loc",s))
            set_loc(str,in,mode,pos,s,ePurple_loc,times,musV);
          else {
            alert("bad option '%s' at set cmd, line %d",str.s,line_nr);
            return false;
          }
          if (strchr("\n;",str.ch)) break;
        }
        else if (str=="black" || str=="red" || str=="green" ||
                 str=="blue" || str=="brown" || str=="purple") {
          strcpy(prefix,str.s);
        }
        else {
          alert("unk option %s at set cmd, line %d",str.s,line_nr);
          return false;
        }
        if (alerts>max_alerts) return false;
      }
    }
    else if (str=="exit") break;
    else { alert("unk cmd '%s' in script, line %d",str.s,line_nr); return false; }
    if (mode==eString && str.ch==0) break;
    if (str.ch=='\n') ++line_nr;
  }
  return true;
}

void Selected::restore_sel() {
  SLList_elem<SectData> *sd;
  ScSection *sec;
  for (sd=sd_list.lis;sd;sd=sd->nxt) {
    sec=sv->score->get_section(sd->d.lnr,sd->d.snr);
    sec->sel=false;
    sec->drawSect(sv,sd->d.snr,sd->d.lnr,sv->d_off);
  }
  reset();
}

void ScoreView::select_all() {
  int lnr,snr,
      end= score->end_sect>nop ? score->end_sect : score->len;
  ScSection *sect;
  ScLine *lin;
  for (lnr=0;lnr<sclin_max;++lnr) {
    lin=score->lin[lnr];
    for (snr=0;snr<end;++snr) {
      sect=lin->sect+snr;
      if (sect->cat==ePlay) {
        selected.insert(lnr,snr);
        sect->sel=true;
        sect->drawSect(this,snr,lnr,d_off);
      }
    }
  }
}

void ScoreView::select_column(int snr) {
  ScSection *sect;
  for (int lnr=0;lnr<sclin_max;++lnr) {
    sect=score->lin[lnr]->sect+snr;
    if (sect->cat==ePlay) {
      selected.insert(lnr,snr);
      sect->sel=true;
      sect->drawSect(this,snr,lnr,d_off);
    }
  }
}

void ScoreView::MouseDown(BPoint where) {
  state=eIdle;
  if (!score) return;
  SetMouseEventMask(B_POINTER_EVENTS, B_LOCK_WINDOW_FOCUS);
  uint32 mouse_but;
  GetMouse(&where,&mouse_but);
  const int lnr=linenr(where.y)+d_off;
  int snr=sectnr(where.x);
  if (lnr<d_off) {
    enter_start_stop(score,snr,mouse_but);
    return;
  }
  if (lnr>=min(sclin_max,d_off+sclin_disp_max) || snr<0) return;
  if (mouse_but==B_SECONDARY_MOUSE_BUTTON) {  // right mouse button
    upd_endline(snr);  // may update score->len;
    return;
  }
  if (snr>=score->len) return;

  cur_line=score->lin[lnr];
  ScSection *const sect=cur_line->sect+snr;
  ScSection *sec,
            *proto;
  key_info key;
  get_key_info(&key);
  if (debug) {
    for (int n=0;n<16;n++) { printf("%u ",key.key_states[n]); }
    putchar('\n');
  }
  int nr=key2num(&key),
      cat=sect->cat,
      ch=key2char(&key);
  if (nr<1) nr=1;
  prev_snr=snr;
  cur_lnr=lnr;
  cur_snr=snr;
  cur_sect=sect;
  if (ch) {
    appWin->act_action=ch;
    appWin->appView->mouseAction->set_active(ch);
  }
  else
    ch=appWin->act_action;
                       // b    a     g     f     e    d     c
  static bool higher[]={ true,false,false,false,true,false,false },
              lower[]={ false,false,false,true,false,false,true };
  bool to_higher=false,
       to_lower=false;
  switch (ch) {
    case 'up':
    case 'do':
      to_higher=higher[lnr%7];
      to_lower=lower[lnr%7];  // no break
    case 'ud':
      state=eToBeReset;
      for(sec=sect;sec->cat==ePlay;++snr,sec=cur_line->sect+snr) {
        bool stop = snr>=score->len-1 || sec->stacc || sec->sampled;
        switch (ch) {
          case 'up':
            if (to_higher) {
              sec->sign=0;
              *score->get_section(lnr-1,snr)=*sec;
              if (sec->sel) {
                selected.remove(lnr,snr);
                selected.insert(lnr-1,snr);
              }
              if (lnr>d_off) // maybe out-of-range
                sec->drawSect(this,snr,lnr-1,d_off);
              sec->reset();
            }
            else
              sec->sign=eHi;
            break;
          case 'do':
            if (to_lower) {
              sec->sign=0;
              *score->get_section(lnr+1,snr)=*sec;
              if (sec->sel) {
                selected.remove(lnr,snr);
                selected.insert(lnr+1,snr);
              }
              if (lnr-d_off<sclin_disp_max)   // maybe out-of-range
                sec->drawSect(this,snr,lnr+1,d_off);
              sec->reset();
            }
            else
              sec->sign=eLo;
            break;
          case 'ud': sec->sign=0; break;
        }
        sec->drawSect(this,snr,lnr,d_off);
        if (stop) break;
      }
      return;
    case 'all':     // select all
      if (selected.sv!=this) {
        selected.restore_sel();
        selected.sv=this;
      }
      state=eToBeReset;
      select_all();
      return;
    case 'uns':     // unselect via U key
      if (selected.sv==this) {
        selected.restore_sel();
      }
      state=eToBeReset;
      return;
    case 'sel':     // select
      if (selected.sv!=this) {
        selected.restore_sel();
        selected.sv=this;
      }
      if (sect->cat==ePlay) {
        bool b=!sect->sel;
        for(sec=sect;sec->cat==ePlay;++snr,sec=cur_line->sect+snr) {
          if (sec->sel && !b)
            selected.remove(lnr,snr);
          else if (b)
            selected.insert(lnr,snr);  // all selected sections are stored
          sec->sel=b;
          sec->drawSect(this,snr,lnr,d_off);
          if (snr>=score->len-1 || sec->stacc || sec->sampled) return;
        }
      }
      else {
        state=eCollectSel;
        select_column(snr);
      }
      return;
    case 'move':
    case 'copy':
      if (selected.sv==this) {
        state= ch=='copy' ? eCopying :
               ch=='move' ? eMoving : 0;
        cur_point=prev_point=where;
        delta_lnr=delta_snr=prev_delta_lnr=prev_delta_snr=0;
      }
      else
        appWin->appView->mouseAction->reset();
      return;
    case 'prta':
      if (sect->cat!=ePlay) {
        state=eToBeReset;
        return;
      }
      if (sect->port_dlnr) {
        sect->drawPortaLine(this,snr,lnr,d_off,true);
        sect->port_dlnr=sect->port_dsnr=0;
        state=eToBeReset;
      }
      else {
        be_app->SetCursor(square_cursor);
        cur_point=where;
        state=ePortaStart;
      }
      return;
  }
  if (nr==1) {  // MouseMoved will be enabled
    if (cat==ePlay) state=eErasing;
    else if (cat==eSilent) state=eTracking;
  }
  proto= snr>1 && (sect-1)->cat==ePlay && !(sect-1)->stacc && !(sect-1)->sampled ? sect-1 :
         snr<score->len-1 && (sect+1)->cat==ePlay && !(sect+1)->stacc && !(sect+1)->sampled ? sect+1 :
         0;
  for(sec=sect;nr>=1 && sec->cat==cat; --nr, ++snr, sec=cur_line->sect+snr) {
    switch (cat) {
      case ePlay:
        if (mouse_but==B_TERTIARY_MOUSE_BUTTON && !sec->sampled) // middle mouse button
          sec->stacc=!sec->stacc;
        else {
          if (sec->sel) selected.remove(lnr,snr);
          if (sec->port_dlnr) {
            sec->drawPortaLine(this,snr,lnr,d_off,true);
            sec->port_dlnr=sec->port_dsnr=0;
          }
          sec->cat=eSilent;
        }
        break;
      case eSilent:
        if (proto) {
          *sect=*proto;
          sec->sel=false;
          sec->port_dlnr=sec->port_dsnr=0;
        }
        else {
          sec->reset();
          sec->cat=ePlay;
          static bool &draw_col=appWin->draw_col->value;
          sec->note_col= draw_col ? appWin->act_color : score->ncol;
          sec->note_ampl=score->nampl;
          sec->sign=cur_line->note_sign;
          sec->sampled=appWin->use_raw_data;
        }
        sec->stacc= mouse_but==B_TERTIARY_MOUSE_BUTTON;
        break;
      default: alert("sect cat %d?",cat);
    }
    sec->drawSect(this,snr,lnr,d_off);
    if (snr>=score->len-1) return;
  }
}

void ScoreView::MouseMoved(BPoint where, uint32 location, const BMessage *) { 
  if (!score || !cur_line) {
    state=eIdle; return;
  }
  switch (state) {
    case eTracking:
    case eErasing: {
        int snr=sectnr(where.x);
        ScSection *const sect=cur_line->sect+snr;
        if (snr<0 || snr>=score->len) { state=eIdle; return; }
        if (snr==prev_snr) return;
        if (state==eTracking) {
          if (sect->cat==eSilent) {
            *sect=*cur_sect;
            sect->drawSect(this,snr,cur_lnr,d_off);
          }
          else if (sect->cat==ePlay) state=eIdle;
        }
        else {
          if (sect->cat==ePlay) {
            sect->cat=eSilent;
            sect->drawSect(this,snr,cur_lnr,d_off);
            if (sect->sel) selected.remove(cur_lnr,snr);
            if (sect->port_dlnr) {
              sect->drawPortaLine(this,snr,cur_lnr,d_off,true);
              sect->port_dlnr=sect->port_dsnr=0;
            }
            if (sect->stacc || sect->sampled) state=eIdle;
          }
          else if (sect->cat==eSilent) state=eIdle;
        }
        prev_snr=snr;
      }
      break;
    case eMoving:
    case eMoving_hor:
    case eMoving_vert:
    case eCopying:
    case eCopying_hor:
    case eCopying_vert: {
        SLList_elem<SectData> *sd;
        int lnr1, snr1,
            dx=float2int((where.x-prev_point.x)/sect_len),
            dy=float2int((where.y-prev_point.y)/sclin_dist);
        if (dy || dx) {
          int x=abs(dx),
              y=abs(dy);
          if (state==eMoving) {
            if (x>y) state=eMoving_hor;
            else if (x<y) state=eMoving_vert;
            else break;
          }
          else if (state==eCopying) {
            if (x>y) state=eCopying_hor;
            else if (x<y) state=eCopying_vert;
            else break;
          }
          prev_point=where;
          for (sd=selected.sd_list.lis;sd;sd=sd->nxt) { // erase old ghost notes
            lnr1=sd->d.lnr + prev_delta_lnr;
            snr1=sd->d.snr + prev_delta_snr;
            if (lnr1>=d_off && lnr1<d_off+sclin_disp_max && snr1>=0 && snr1<score->len)
              score->get_section(lnr1,snr1)->drawS_ghost(this,snr1,lnr1,d_off,true);
          }
          prev_delta_lnr=delta_lnr;
          prev_delta_snr=delta_snr;
          if (state==eMoving_vert || state==eCopying_vert)
            delta_lnr=float2int((where.y-cur_point.y)/sclin_dist);
          else if (state==eMoving_hor || state==eCopying_hor)
            delta_snr=float2int((where.x-cur_point.x)/sect_len);
          selected.check_direction(delta_lnr - prev_delta_lnr,delta_snr - prev_delta_snr);
          for (sd=selected.sd_list.lis;sd;sd=sd->nxt) { // draw new ghost notes
            lnr1=sd->d.lnr + delta_lnr;
            snr1=sd->d.snr + delta_snr;
            if (lnr1>=d_off && lnr1<d_off+sclin_disp_max && snr1>=0) {
              if (score->check_len(snr1+1)) redraw(false);
              score->get_section(lnr1,snr1)->drawS_ghost(this,snr1,lnr1,d_off,false);
            }
          }
        }
      }
      break;
    case eCollectSel: {
        int snr=sectnr(where.x);
        if (snr<0 || snr>=score->len) { state=eIdle; return; }
        if (snr>prev_snr)
          for (++prev_snr;;++prev_snr) {
            select_column(prev_snr);
            if (prev_snr==snr) break;
          }
        else if (snr<prev_snr)
          for (--prev_snr;;--prev_snr) {
            select_column(prev_snr);
            if (prev_snr==snr) break;
          }
      }
      break;
    case ePortaStart: {
        int snr=sectnr(where.x),
            lnr=linenr(where.y)+d_off;
        ScSection *const sect=score->get_section(lnr,snr);
        if (sect->cat==ePlay || lnr==cur_lnr && snr==cur_snr)
          be_app->SetCursor(square_cursor);
        else
          be_app->SetCursor(point_cursor);
      }
      break;
    case eIdle:
    case eToBeReset:
      break;
    default:
      alert("mouse moving state %d?",state);
      state=eIdle;
  }
}

void ScoreView::MouseUp(BPoint where) {
  key_info key;
  get_key_info(&key);
  switch (state) {
    case eMoving_hor:
    case eMoving_vert:
    case eCopying_hor:
    case eCopying_vert: {
        SLList_elem<SectData> *sd,*lst_sd;
        ScSection *from, *to;
        int lnr1, snr1,
            to_lnr, to_snr;
        for (sd=selected.sd_list.lis;sd;sd=sd->nxt) { // erase ghost notes
          lnr1=sd->d.lnr + prev_delta_lnr;
          snr1=sd->d.snr + prev_delta_snr;
          if (lnr1>=d_off && lnr1<d_off+sclin_disp_max && snr1>=0 && snr1<score->len)
            score->get_section(lnr1,snr1)->drawS_ghost(this,snr1,lnr1,d_off,true);
        }
        if (key2char(&key)!='keep') {
          delta_lnr=delta_snr=0;
          if (state==eMoving_vert || state==eCopying_vert)
            delta_lnr=float2int((where.y-cur_point.y)/sclin_dist);
          if (state==eMoving_hor || state==eCopying_hor)
            delta_snr=float2int((where.x-cur_point.x)/sect_len);
          if (delta_lnr || delta_snr) {
            selected.check_direction(delta_lnr,delta_snr);
            for (lst_sd=0,sd=selected.sd_list.lis;sd;) {
              to_lnr=sd->d.lnr + delta_lnr;
              to_snr=sd->d.snr + delta_snr;
              from=score->get_section(sd->d.lnr,sd->d.snr);
              if (to_lnr>=0 && to_lnr<sclin_max && to_snr>=0) {
                if (score->check_len(to_snr+1)) redraw(false);
                to=score->get_section(to_lnr,to_snr);
                *to=*from;
                if (state==eMoving_hor || state==eMoving_vert) {
                  if (from->port_dlnr) from->drawPortaLine(this,sd->d.snr,sd->d.lnr,d_off,true);
                  from->reset();
                }
                else
                  from->sel=false;
                from->drawSect(this,sd->d.snr,sd->d.lnr,d_off);
                sd->d.lnr=to_lnr;
                sd->d.snr=to_snr;
                if (to_lnr>=d_off && to_lnr<d_off+sclin_disp_max) {
                  to->drawSect(this,to_snr,to_lnr,d_off);
                  if (to->port_dlnr) to->drawPortaLine(this,to_snr,to_lnr,d_off,false);
                }
                lst_sd=sd; sd=sd->nxt;
              }
              else {
                if (state==eMoving_hor || state==eMoving_vert)
                  from->reset();
                else
                  from->sel=false;
                from->drawSect(this,sd->d.snr,sd->d.lnr,d_off);
                selected.remove(sd->d.lnr,sd->d.snr);
                sd=lst_sd ? lst_sd->nxt : selected.sd_list.lis;
              }
            }
          }
          if (state==eMoving_hor)
            score->drawEnd(this,d_off); // moving from nearby endline?
        }
      }
      appWin->appView->mouseAction->reset();
      break;
    case ePortaStart:
      appWin->appView->mouseAction->reset();
      be_app->SetCursor(B_HAND_CURSOR);
      { const int lnr=linenr(where.y)+d_off,
                  snr=sectnr(where.x);
        ScSection *sec=score->get_section(lnr,snr);
        if (sec->cat==ePlay) {
          if (lnr-cur_lnr==0) { alert("portando notes same height"); break; }
          if (cur_sect->note_col!=sec->note_col) { alert("portando notes not same color"); break; }
          if (abs(lnr-cur_lnr)>= 1<<4) { alert("portando notes height difference >= %d",1<<4); break; }
          if (snr-cur_snr<1) { alert("portando notes distance < 0"); break; }
          if (snr-cur_snr>= 1<<4) { alert("portando notes distance >= %d",1<<4); break; }
          cur_sect->port_dlnr= abs(lnr-cur_lnr);
          cur_sect->dlnr_sign= lnr>=cur_lnr;
          cur_sect->port_dsnr= snr-cur_snr-1;
          cur_sect->drawPortaLine(this,cur_snr,cur_lnr,d_off,false);
        }
      }
      break;
    case eCollectSel:
    case eIdle:
    case eTracking:
    case eErasing:
      break;
    case eMoving:
    case eCopying:
    case eToBeReset:
      appWin->appView->mouseAction->reset();
      break;
    default:
      alert("mouse-up state %d?",state);
  }
  state=eIdle;
}

ScoreViewStruct::ScoreViewStruct(BRect rect,int ident):
    id(ident) {
  scroll_view=new ScoreSView(BRect(rect.left,rect.top,rect.left+scview_text-1,rect.bottom),this);
  scv_view=scroll_view->scoreView;
  scv_text=new ScoreViewText(BRect(rect.left+scview_text,rect.top,rect.right,rect.bottom),this);
}

void ViewScore::enter_start_stop(Score *score,int snr,uint32 mouse_but) {
  switch (mouse_but) {
    case B_PRIMARY_MOUSE_BUTTON:
      if (play_start==snr) {
        play_start=0;
        draw_start_stop(score,snr,0,true);
      }
      else {
        draw_start_stop(score,play_start,snr,true);
        play_start=snr;
        if (play_stop>=0 && play_stop<play_start) alert("stop < start");
      }
      break;
    case B_SECONDARY_MOUSE_BUTTON:
      if (play_stop==snr) {
        play_stop=-1;
        draw_start_stop(score,snr,0,false);
      }
      else {
        draw_start_stop(score,play_stop,snr,false);
        play_stop=snr;
        if (play_start && play_stop<play_start) alert("stop < start");
      }
      break;
  }
}

void MusicView::MouseDown(BPoint where) {
  uint32 mouse_but;
  GetMouse(&where,&mouse_but);
  int snr=sectnr(where.x);
//  key_info key; get_key_info(&key); int ch=key2char(&key);
  enter_start_stop(score,snr,mouse_but);
}

bool exec_info(int snr,Score *sc) {
  if (appWin->no_set->value) return true;
  ScInfo* sci=sc->scInfo;
  if (!sci) return true;
  for (sci=sci+snr;sci;sci=sci->next) {
    if (sci->tag && !appWin->exec_cmd(*sci)) return false;
  }
  return true;
}

struct AmcApp: public BApplication {
  char *appf,  // argv[0]
       *inf;   // .sco or .scr file
  AmcApp(char *app,char *in): BApplication(app_mime),appf(app),inf(in) {
    appFont=new BFont(be_plain_font);
    appFont->SetSize(12);

    bitmaps=new Bitmaps();
    square_cursor=new BCursor(sq_cursor_data);
    point_cursor=new BCursor(pnt_cursor_data);
  }
  void ReadyToRun() {
    new AppWindow(appWin,appf,inf);
    appWin->Show();
  }
/*
  void ArgvReceived(int32 argc,char **argv) {
  }
*/
  void RefsReceived(BMessage *msg) {
    if (debug) alert("refs received!");
    if (!IsLaunching()) return;

    uint32 type;
    entry_ref ref;
    static BPath path;

    msg->GetInfo("refs", &type);
    if (type != B_REF_TYPE) return;
    if (msg->FindRef("refs", &ref) == B_OK) {
      BEntry entry(&ref);
      path.SetTo(&entry);
      inf=const_cast<char*>(path.Path());
    }
  }
};

int main(int argc,char **argv) {
  char *inf=0;
  for (int an=1;an<argc;++an) {
    if (!strcmp(argv[an],"-db")) debug=true;
    else if (argv[an][0]=='-') { alert("Unexpected option %s\n",argv[an]); exit(1); }
    else inf=argv[an];
  }
  fputs("\033[34m",stdout); // blue
  AmcApp app(argv[0],inf);
  app.Run();
  fputs("\033[30m",stdout); // black
  return 0;
} 
