/*
Open Tracker License

Terms and Conditions

Copyright (c) 1991-2000, Be Incorporated. All rights reserved.

Permission is hereby granted, free of charge, to any person obtaining a copy of
this software and associated documentation files (the "Software"), to deal in
the Software without restriction, including without limitation the rights to
use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
of the Software, and to permit persons to whom the Software is furnished to do
so, subject to the following conditions:

The above copyright notice and this permission notice applies to all licensees
and shall be included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF TITLE, MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
BE INCORPORATED BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF, OR IN CONNECTION
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of Be Incorporated shall not be
used in advertising or otherwise to promote the sale, use or other dealings in
this Software without prior written authorization from Be Incorporated.

Tracker(TM), Be(R), BeOS(R), and BeIA(TM) are trademarks or registered trademarks
of Be Incorporated in the United States and other countries. Other brand product
names are registered trademarks or trademarks of their respective holders.
All rights reserved.
*/

#include <Debug.h>
#include <Bitmap.h>
#include <Dragger.h>
#include <Menu.h>
#include <MenuItem.h>
#include <Roster.h>

#include "BeMenu.h"
#include "BarApp.h"
#include "BarView.h"
#include "DeskBarUtils.h"
#include "PublicCommands.h"
#include "RecentItems.h"

// from roster_private.h
const uint32 CMD_SHUTDOWN_SYSTEM = 301;
const uint32 CMD_REBOOT_SYSTEM = 302;
const uint32 CMD_SUSPEND_SYSTEM = 304;

#define ROSTER_SIG "application/x-vnd.Be-ROST"

#ifdef B_BEOS_VERSION_5
void run_be_about();
#endif

#ifdef MOUNT_MENU_IN_DESKBAR

class MountMenu : public BMenu {
public:
	MountMenu(const char *name);
	virtual bool AddDynamicItem(add_state s);
};


class MountMenuItem : public BMenuItem {
public:

	MountMenuItem(const char *label, BMessage *message, BBitmap *icon );
	virtual ~MountMenuItem();
	virtual void GetContentSize(float *width, float *height);	
	virtual void DrawContent();
private:
	BBitmap *deviceIcon;
};

#endif

// #define SHOW_RECENT_FIND_ITEMS

namespace BPrivate {

_IMPEXP_TRACKER BMenu *TrackerBuildRecentFindItemsMenu(const char *);

}
using namespace BPrivate;

//********************************************************************************

TBeMenu::TBeMenu(TBarView *barview)
	: BNavMenu("BeMenu", B_REFS_RECEIVED, BMessenger(kTrackerSig)),
		fAddState(kStart),
		fBarView(barview)
{
}

void
TBeMenu::AttachedToWindow()
{
	if (fBarView && fBarView->LockLooper()) {
		if (fBarView->Dragging()) {
			SetTypesList(fBarView->CachedTypesList());
			SetTarget(BMessenger(fBarView));
			SetTrackingHookDeep(this, fBarView->MenuTrackingHook, 
				fBarView->GetTrackingHookData());
			fBarView->DragStart();
		} else {
			SetTypesList(NULL);
			SetTarget(BMessenger(kTrackerSig));
			SetTrackingHookDeep(this, NULL, NULL);		
		}

		fBarView->UnlockLooper();
	}

	BNavMenu::AttachedToWindow();
}

void
TBeMenu::DetachedFromWindow()
{
	if (fBarView) {
		BLooper* looper = fBarView->Looper();
		if (looper && looper->Lock()) {
			fBarView->DragStop();
			looper->Unlock();
		}
	}

	//
	//	don't call BNavMenu::DetachedFromWindow
	//	it sets the TypesList to NULL
	//	
	BMenu::DetachedFromWindow();
}

bool 
TBeMenu::StartBuildingItemList()
{
	int32 count = CountItems()-1;
	for (int32 index = count; index >= 0; index--) {
		BMenuItem *item = ItemAt(index);
		ASSERT(item);

		RemoveItem(index);
		delete item;
	}
	fAddState = kStart;
	return BNavMenu::StartBuildingItemList();
}

void
TBeMenu::DoneBuildingItemList()
{
	if (fItemList->CountItems() <= 0) {
		BMenuItem *item = new BMenuItem("<Be folder is empty>", 0);
		item->SetEnabled(false);
		AddItem(item);
	} else
		BNavMenu::DoneBuildingItemList();
}

bool 
TBeMenu::AddNextItem()
{	
	if (fAddState == kStart)
		return AddStandardBeMenuItems();

	TrackingHookData *data = fBarView->GetTrackingHookData();
	if (fAddState == kAddingRecents) {
		const char *recentTitle[] = {"Recent Documents", "Recent Folders", "Recent Applications"};
		const int recentType[] = {0,2,1};
		int recentTypes = 3;
		TRecentsMenu *recentItem[3];
		int count = 0;
		
		for (int i = 0; i < recentTypes; i++) {
			recentItem[i] = new TRecentsMenu(recentTitle[i], fBarView, recentType[i]);
			if (recentItem[i])
				count += recentItem[i]->RecentsCount();
		}
		if (count > 0) {
			AddSeparatorItem();
			
			for(int i = 0;i < recentTypes;i++) {			
				if (!recentItem[i])
					continue;
					
				if (recentItem[i]->RecentsCount() > 0) {
					recentItem[i]->SetTypesList(TypesList());
					recentItem[i]->SetTarget(Target());
					AddItem(recentItem[i]);
				}
				
				if (data && fBarView && fBarView->Dragging()) {
					recentItem[i]->InitTrackingHook(data->fTrackingHook,
						&data->fTarget, data->fDragMessage);
				}
			}
		}
		
		AddSeparatorItem();
		fAddState = kAddingBeMenu;
		return true;
	}
	
	if (fAddState == kAddingBeMenu) {
		//	keep reentering and adding items
		//	until this returns false
		bool done = BNavMenu::AddNextItem();
		BMenuItem *item = ItemAt(CountItems()-1);
		if (item) {
			BMenu *menu = (BNavMenu *)item->Menu();
			if (menu) {
				BNavMenu* navmenu = dynamic_cast<BNavMenu*>(menu);
				if (navmenu) {
					if (data && fBarView->Dragging()) 
						navmenu->InitTrackingHook(data->fTrackingHook,
							&data->fTarget, data->fDragMessage);
					else 
						navmenu->InitTrackingHook(0, NULL, NULL);
	
				}
			}
		}
		if (!done)
			fAddState = kDone;
		return done;
	}
	
	return false;
}

bool
TBeMenu::AddStandardBeMenuItems()
{
	bool dragging=false;
	if (fBarView)
		dragging = fBarView->Dragging();

	BMenuItem* item = new BMenuItem("About BeOS", new BMessage(kShowSplash));
	item->SetEnabled(!dragging);
	AddItem(item);
	
#ifdef SHOW_RECENT_FIND_ITEMS
	item = new BMenuItem(TrackerBuildRecentFindItemsMenu("Find"B_UTF8_ELLIPSIS),
		new BMessage(kFindButton));
#else
 	item = new BMenuItem("Find"B_UTF8_ELLIPSIS, new BMessage(kFindButton));
#endif
	item->SetEnabled(!dragging);
	AddItem(item);

	item = new BMenuItem("Show Replicants", new BMessage(msg_ToggleDraggers));
	item->SetEnabled(!dragging);
	item->SetMarked(BDragger::AreDraggersDrawn());
	AddItem(item);
	item = new BMenuItem("Configure Be Menu"B_UTF8_ELLIPSIS, new BMessage(msg_config_db));
	item->SetEnabled(!dragging);
	AddItem(item);

#ifdef MOUNT_MENU_IN_DESKBAR
	MountMenu *mountMenu = new MountMenu("Mount Disks");
	mountMenu->SetEnabled(!dragging);
	AddItem(mountMenu);
#endif

	if ((modifiers() & (B_LEFT_SHIFT_KEY|B_LEFT_CONTROL_KEY|B_LEFT_COMMAND_KEY))
		== (B_LEFT_SHIFT_KEY|B_LEFT_CONTROL_KEY|B_LEFT_COMMAND_KEY)) {
		BMenuItem *i;
		BMenu *submenu = new BMenu("Window Decor"B_UTF8_ELLIPSIS);
		submenu->AddItem(i=new BMenuItem("BeOS", new BMessage(msg_Be)));
		i->SetTarget(be_app);
		i->SetEnabled(!dragging);
		submenu->AddItem(i=new BMenuItem("AmigaOS", new BMessage(msg_Amiga)));
		i->SetTarget(be_app);
		i->SetEnabled(!dragging);
		submenu->AddItem(i=new BMenuItem("MacOS 8", new BMessage(msg_Mac)));
		i->SetTarget(be_app);
		i->SetEnabled(!dragging);
		submenu->AddItem(i=new BMenuItem("Windows 95/98", new BMessage(msg_Win95)));
		i->SetTarget(be_app);
		i->SetEnabled(!dragging);

		submenu->SetFont(be_plain_font);
		AddItem(submenu);
	};

	AddSeparatorItem();

	item = new BMenuItem("Restart", new BMessage(CMD_REBOOT_SYSTEM));
	item->SetEnabled(!dragging);
	AddItem(item);

#ifdef APM_SUPPORT
	if (_kapm_control_(APM_CHECK_ENABLED) == B_OK) {
		item = new BMenuItem("Suspend", new BMessage(CMD_SUSPEND_SYSTEM));
		item->SetEnabled(!dragging);
		AddItem(item);
	}
#endif

	item = new BMenuItem("Shut Down", new BMessage(CMD_SHUTDOWN_SYSTEM));
	item->SetEnabled(!dragging);
	AddItem(item);

	fAddState = kAddingRecents;

	return true;
}

void 
TBeMenu::ClearMenuBuildingState()
{
	fAddState = kDone;
	fMenuBuilt = false;
		// force the menu to get rebuilt each time
	BNavMenu::ClearMenuBuildingState();
}

void
TBeMenu::ResetTargets()
{
	BNavMenu::ResetTargets();
	
	//
	//	if we are dragging, set the target to whatever was set
	//	else set it to the default (Tracker)
	//
	if (!fBarView->Dragging())
		SetTarget(BMessenger(kTrackerSig));
	//
	//	now set the target for the menuitems to the currently
	//	set target, which may or may not be tracker
	//
	SetTargetForItems(Target());

	for (int32 i = 0; ; i++) {
		BMenuItem *item = ItemAt(i);
		if (item == NULL)
			break;

		if (item->Message()) {
			switch (item->Message()->what) {
				case kShowSplash:
#ifdef B_BEOS_VERSION_5
#if 0
					run_be_about();
					// about box in libbe in BeOS R5
#endif
#endif
					break;
				case kFindButton:
					// about, find
					item->SetTarget(BMessenger(kTrackerSig));
					break;
	
				case msg_ToggleDraggers:
					// show/hide replicants
					item->SetTarget(be_app);
					break;
					
				case msg_config_db:
					item->SetTarget(be_app);
					break;

				case CMD_REBOOT_SYSTEM:
				case CMD_SUSPEND_SYSTEM:
				case CMD_SHUTDOWN_SYSTEM:
					// restart/shutdown
					item->SetTarget(BMessenger(ROSTER_SIG));
					break;
			}
		}
	}
}

BPoint
TBeMenu::ScreenLocation()
{
	BPoint	pt;
	BRect	r;
	bool	vertical = fBarView->Vertical();
	int32	expando = (fBarView->State() == kExpandoState);

	r = Supermenu()->Bounds();
	Supermenu()->ConvertToScreen(&r);
	if (expando && vertical && fBarView->Left()) {
		PRINT(("Left\n"));
		pt = r.RightTop() + BPoint(0,3);
	} else if (expando && vertical && !fBarView->Left()) {
		PRINT(("Right\n"));
		pt = r.LeftTop() - BPoint(Bounds().Width(), 0) + BPoint(0,3);
	} else {
		pt = BMenu::ScreenLocation();
	}
	return pt;
}

//********************************************************************************
//
//	which == 0 is documents, which ==1 is applications
//
TRecentsMenu::TRecentsMenu(const char* name, TBarView *bar, int32 which)
	: BNavMenu(name, B_REFS_RECEIVED, BMessenger(kTrackerSig)),
		fWhich(which),
		fRecentsCount(0),
		fItemIndex(0),
		fBarView(bar)
{
	TBarApp* app = dynamic_cast<TBarApp*>(be_app);
	if (app) {
		if (fWhich == 0)
			fRecentsCount = app->Settings()->recentDocsCount;
		else if (fWhich == 1)
			fRecentsCount = app->Settings()->recentAppsCount;
		else if (fWhich == 2)
			fRecentsCount = app->Settings()->recentFoldersCount;
	}
}

void
TRecentsMenu::DetachedFromWindow()
{
	//
	//	BNavMenu::DetachedFromWindow sets the TypesList to NULL
	//	
	BMenu::DetachedFromWindow();
}

bool 
TRecentsMenu::StartBuildingItemList()
{
	int32 count = CountItems()-1;
	for (int32 index = count; index >= 0; index--) {
		BMenuItem *item = ItemAt(index);
		ASSERT(item);

		RemoveItem(index);
		delete item;
	}
//
//	!! note: don't call inherited from here
//	the navref is not set for this menu
//	but it still needs to be a draggable navmenu
//	simply return true so that AddNextItem is called
//
//	return BNavMenu::StartBuildingItemList();
	return true;
}

bool 
TRecentsMenu::AddNextItem()
{
	if (fRecentsCount > 0 && AddRecents(fRecentsCount))
		return true;

	fItemIndex = 0;
	return false;
}

bool
TRecentsMenu::AddRecents(int32 count)
{
	if (fItemIndex == 0) {
		fRecentList.MakeEmpty();
		BRoster roster;
		
		switch(fWhich) {
			case 0:
				roster.GetRecentDocuments(&fRecentList, count);
				break;
			case 1:
				roster.GetRecentApps(&fRecentList, count);
				break;
			case 2:
				roster.GetRecentFolders(&fRecentList, count);
				break;
			default:
				return false;
				break;
		}
	}
	for (;;) {
		entry_ref ref;
		if (fRecentList.FindRef("refs", fItemIndex++, &ref) != B_OK) {
			break;
		}
		if (ref.name && strlen(ref.name) > 0) {
			Model model(&ref, true);
			ModelMenuItem* item
				= BNavMenu::NewModelItem(&model,
					new BMessage(B_REFS_RECEIVED),
					Target(), false, NULL, TypesList());
	
			if (item) {
				AddItem(item);
	
				//
				//	return true so that we know to reenter this list
				//
				return true;
			}
			
			return true;
		}
	}

	//
	//	return false if we are done with this list
	//
	return false;
}

void 
TRecentsMenu::DoneBuildingItemList()
{
//
//	!! note: don't call inherited here
//	the object list is not built
//	and this list does not need to be sorted
//	BNavMenu::DoneBuildingItemList();
//
	if (CountItems() <= 0) {
		BMenuItem *item = new BMenuItem("<No Recent Items>", 0);
		item->SetEnabled(false);
		AddItem(item);
	} else
		SetTargetForItems(Target());
}

void 
TRecentsMenu::ClearMenuBuildingState()
{
	fMenuBuilt = false;
	BNavMenu::ClearMenuBuildingState();
}

void
TRecentsMenu::ResetTargets()
{
	BNavMenu::ResetTargets();	
	//
	//	if we are dragging, set the target to whatever was set
	//	else set it to the default (Tracker)
	//
	if (!fBarView->Dragging())
		SetTarget(BMessenger(kTrackerSig));
	//
	//	now set the target for the menuitems to the currently
	//	set target, which may or may not be tracker
	//
	SetTargetForItems(Target());
}

//********************************************************************************

#ifdef MOUNT_MENU_IN_DESKBAR

MountMenu::MountMenu(const char *name)
	:	BMenu(name)
{
	SetFont(be_plain_font);
}

bool MountMenu::AddDynamicItem(add_state s)
{
	BMenuItem *item;
	while ((item = RemoveItem(0L)) != NULL)
		delete item;

	//
	// Send message to tracker to get items.
	//
	BMessage request('gmtv');
	BMessage reply;
	BMessenger(kTrackerSig).SendMessage(&request,
		&reply);

	//	
	// populate menu
	//	
	type_code code;
	int32 countFound;
	reply.GetInfo("DisplayName", &code, &countFound);
	for (int32 vol = 0; vol < countFound; vol++) {
		BBitmap *icon = new BBitmap(BRect(0,0,15,15),B_COLOR_8_BIT);
		get_device_icon(reply.FindString("DeviceName", vol), icon->Bits(), B_MINI_ICON);	
		BMessage *invokeMessage = new BMessage;
		reply.FindMessage("InvokeMessage", vol, invokeMessage);
		AddItem(new MountMenuItem(reply.FindString("DisplayName", vol),
		  invokeMessage, icon));
	}
	
	if (countFound > 0)
		AddSeparatorItem();

	BMenuItem *mountAll = new BMenuItem("All Disks", new BMessage('mntn'));
	AddItem(mountAll);
	mountAll->SetEnabled(countFound > 0);
	BMenuItem *mountSettings = new BMenuItem("Settings", new BMessage('Tram'));
	AddItem(mountSettings);
	
	SetTargetForItems(BMessenger(kTrackerSig));
	
	return false;
}

MountMenuItem::MountMenuItem(const char *label, BMessage *message, BBitmap *icon)
	:	BMenuItem(label,message),
		deviceIcon(icon)
		
{
}

MountMenuItem::~MountMenuItem()
{
	delete deviceIcon;
}

void
MountMenuItem::GetContentSize(float *width, float *height)
{
	BMenuItem::GetContentSize(width, height);
	*width += 20;
	*height += 3;
}

void
MountMenuItem::DrawContent()
{
	BPoint loc = ContentLocation();
	loc.x += 20;
	Menu()->MovePenTo(loc);
	BMenuItem::DrawContent();
	
	BPoint where(ContentLocation());
	where.y = Frame().top;
	Menu()->SetDrawingMode(B_OP_OVER);
	Menu()->DrawBitmapAsync(deviceIcon, where);
}

#endif

