/*
Open Tracker License

Terms and Conditions

Copyright (c) 1991-2000, Be Incorporated. All rights reserved.

Permission is hereby granted, free of charge, to any person obtaining a copy of
this software and associated documentation files (the "Software"), to deal in
the Software without restriction, including without limitation the rights to
use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
of the Software, and to permit persons to whom the Software is furnished to do
so, subject to the following conditions:

The above copyright notice and this permission notice applies to all licensees
and shall be included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF TITLE, MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
BE INCORPORATED BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF, OR IN CONNECTION
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of Be Incorporated shall not be
used in advertising or otherwise to promote the sale, use or other dealings in
this Software without prior written authorization from Be Incorporated.

Tracker(TM), Be(R), BeOS(R), and BeIA(TM) are trademarks or registered trademarks
of Be Incorporated in the United States and other countries. Other brand product
names are registered trademarks or trademarks of their respective holders.
All rights reserved.
*/

#include <Debug.h>
#include <MenuItem.h>
#include <PopUpMenu.h>
#include <Roster.h>

#include <string.h>

#include "TimeView.h"
#include "StatusView.h"

const char* const kMinString = "99:99 AM";

static float
FontHeight(BView* target, bool full)
{
	font_height finfo;		
	target->GetFontHeight(&finfo);
	float h = finfo.ascent + finfo.descent;

	if (full)
		h += finfo.leading;
	
	return h;
}

enum {
	msg_showclock,
	msg_showseconds,
	msg_miltime,
	msg_fulldate,
	msg_changeclock,
	msg_hide
};

TTimeView::TTimeView(bool showSeconds, bool milTime, bool fullDate, bool)
	: 	BView(BRect(-100,-100,-90,-90), "_deskbar_tv_",
			B_FOLLOW_RIGHT | B_FOLLOW_TOP,
			B_WILL_DRAW | B_PULSE_NEEDED | B_FRAME_EVENTS),
			fParent(NULL),
			fShowInterval(true), // defaulting this to true until UI is in place
			fShowSeconds(showSeconds),
			fMilTime(milTime),
			fFullDate(fullDate),
			fOrientation(false)
{
	fShowingDate = false;
	fTime = fLastTime = time(NULL);
	fSeconds = fMinute = fHour = 0;
	fLastTimeStr[0] = 0;
	fLastDateStr[0] = 0;
	fNeedToUpdate = true;
}

#ifdef AS_REPLICANT
TTimeView::TTimeView(BMessage *data)
	: BView(data)
{
	fTime = fLastTime = time(NULL);
	data->FindBool("seconds", &fShowSeconds);
	data->FindBool("miltime", &fMilTime);
	data->FindBool("fulldate", &fFullDate);
	data->FindBool("interval", &fInterval);
	fShowingDate = false;
}
#endif

TTimeView::~TTimeView()
{
}

#ifdef AS_REPLICANT
BArchivable*
TTimeView::Instantiate(BMessage *data)
{
	if (!validate_instantiation(data, "TTimeView"))
		return NULL;
	return new TTimeView(data);
}

status_t
TTimeView::Archive(BMessage *data, bool deep) const
{
	BView::Archive(data, deep);
	data->AddBool("seconds", fShowSeconds);
	data->AddBool("miltime", fMilTime);
	data->AddBool("fulldate", fFullDate);
	data->AddBool("interval", fInterval);
	data->AddInt32("deskbar:private_align", B_ALIGN_RIGHT);
	return B_OK;
}
#endif

void
TTimeView::AttachedToWindow()
{
	fTime = time(NULL);
	
	SetFont(be_plain_font);
	if (Parent()) {
		fParent = Parent();
		SetViewColor(Parent()->ViewColor());
	}
	else
		SetViewColor(216,216,216);
	
	fFontHeight = FontHeight(this, true);		
	ResizeToPreferred();
	CalculateTextPlacement();
}

void
TTimeView::GetPreferredSize(float *width, float *height)
{
	*height = fFontHeight;

	GetCurrentTime();
	GetCurrentDate();
	
	if (ShowingDate())
		*width = 6 + StringWidth(fDateStr);
	else {
		*width = 6 + StringWidth(fTimeStr);
		// Changed this from 10 to 6 so even with interval + seconds, there still
		// is room for two replicants in the default tray.
	}
}

void
TTimeView::ResizeToPreferred()
{
	float w, h;
	float oldw = Bounds().Width(), oldh = Bounds().Height();
	
	GetPreferredSize(&w, &h);
	if (h != oldh || w != oldw) {
		ResizeTo(w, h);
		MoveBy (oldw - w, 0);
		fNeedToUpdate = true;
	}
}

void
TTimeView::FrameMoved(BPoint)
{
	Update();
}

void
TTimeView::MessageReceived(BMessage* message)
{
	switch(message->what) {
		case msg_showseconds:
			ShowSeconds(!ShowingSeconds());
			break;
		case msg_miltime:
			ShowMilTime(!ShowingMilTime());
			break;
		case msg_fulldate:
			ShowFullDate(!ShowingFullDate());
			break;
			
		case msg_changeclock:
			// launch the time prefs app
			be_roster->Launch("application/x-vnd.Be-TIME");
			break;
			
		case 'time':
			Window()->PostMessage(message, Parent());
			break;
			
		default:
			BView::MessageReceived(message);
	}
}

void
TTimeView::GetCurrentTime()
{
	char tmp[64];
	tm time = *localtime(&fTime);
	
	if (fMilTime) {
		if (fShowSeconds)
			strftime(tmp, 64, "%H:%M:%S", &time);
		else	
			strftime(tmp, 64, "%H:%M", &time);
	} else {
		if (fShowInterval) {
			if (fShowSeconds)
				strftime(tmp, 64, "%I:%M:%S %p", &time);
			else {
				strftime(tmp, 64, "%I:%M %p", &time);
			}
		} else {
			if (fShowSeconds)
				strftime(tmp, 64, "%I:%M:%S", &time);
			else {	
				strftime(tmp, 64, "%I:%M", &time);
			}
		}
	}
	
	//	remove leading 0 from time when hour is less than 10
	const char* str = tmp;
	if (str[0] == '0')
		str++;
		
	strcpy(fTimeStr, str);
	
	fSeconds = time.tm_sec;
	fMinute = time.tm_min;
	fHour = time.tm_hour;
}

const char *kShortDateFormat = "%m/%d/%y";
const char *kLongDateFormat = "%a, %B %d, %Y";

void
TTimeView::GetCurrentDate()
{
	char tmp[64];
	tm time = *localtime(&fTime);
	
	strftime(tmp, 64,
		(fFullDate && CanShowFullDate()) ?
		kLongDateFormat : kShortDateFormat, &time);

	//	remove leading 0 from date when month is less than 10
	const char* str = tmp;
	if (str[0] == '0')
		str++;
		
	strcpy(fDateStr, str);
}

void
TTimeView::Draw(BRect )
{
	PushState();

	SetHighColor(ViewColor());
	SetLowColor(ViewColor());
	FillRect(Bounds());			
	SetHighColor(0,0,0,255);		

	
	if (fShowingDate) {
		MovePenTo(fDateLocation);
		DrawString(fDateStr);
	} else {
		MovePenTo(fTimeLocation);
		DrawString(fTimeStr);
	}
	
	PopState();
}

void
TTimeView::MouseDown(BPoint point)
{
	uint32	buttons;

	Window()->CurrentMessage()->FindInt32("buttons", (long*) &buttons);
	if (buttons == B_SECONDARY_MOUSE_BUTTON) {
		ShowClockOptions(ConvertToScreen(point));
		return;
	}

	//	flip to/from showing date or time
	fShowingDate = !fShowingDate;
	if (fShowingDate)
		fLastTime = time(NULL);
		
	Update();
}

void
TTimeView::Pulse()
{
	time_t curTime = time(NULL);
	tm	ct = *localtime(&curTime);
	fTime = curTime;
	
	GetCurrentTime();
	GetCurrentDate();
	if (	(!fShowingDate && strcmp(fTimeStr, fLastTimeStr) != 0)
		|| 	(fShowingDate && strcmp(fDateStr, fLastDateStr) != 0)) {
		strcpy(fLastTimeStr, fTimeStr);
		strcpy(fLastDateStr, fDateStr);
		fNeedToUpdate = true;
	}
	
	if (fShowingDate && (fLastTime + 5 <= time(NULL))) {
		fShowingDate = false;
		Update();	// Needs to happen since size can change here
	}
	
	if (fNeedToUpdate) {
		fSeconds = ct.tm_sec;
		fMinute = ct.tm_min;
		fHour = ct.tm_hour;
		fInterval = ct.tm_hour >= 12;
		
		Draw(Bounds());
		fNeedToUpdate = false;
	}	
}

void
TTimeView::ShowSeconds(bool s)
{
	fShowSeconds = s;
	Update();
}

void
TTimeView::ShowMilTime(bool s)
{
	fMilTime = s;
	Update();
}

void
TTimeView::ShowDate(bool s)
{
	fShowingDate = s;
	Update();
}

void
TTimeView::ShowFullDate(bool s)
{
	fFullDate = s;
	Update();
}

bool
TTimeView::CanShowFullDate() const
{
	bool showFullDate = true;
	TReplicantTray *rt = dynamic_cast <TReplicantTray *> (fParent);
	if (rt && rt->IsMultiRow())
		showFullDate = false;	// This won't fit when the Deskbar isn't horizontally expanded.
	return showFullDate;
}	

void
TTimeView::Update()
{
	GetCurrentTime();
	GetCurrentDate();
	ResizeToPreferred();
	CalculateTextPlacement();
	if (fParent) {
		fParent->MessageReceived (new BMessage ('trfm'));
		fParent->Invalidate();
	}
}

bool
TTimeView::Orientation() const
{
	return fOrientation;
}

void
TTimeView::SetOrientation(bool o)
{
	fOrientation = o;
	CalculateTextPlacement();
	Invalidate();
}

void
TTimeView::CalculateTextPlacement()
{
	if (fOrientation) {		// vertical mode
		fDateLocation.x = Bounds().Width()/2 - StringWidth(fDateStr)/2;
		fTimeLocation.x = Bounds().Width()/2 - StringWidth(fTimeStr)/2;
	} else {
		fTimeLocation.x = Bounds().Width() - StringWidth(fTimeStr) - 5;
		fDateLocation.x = Bounds().Width() - StringWidth(fDateStr) - 5;
	}
	//	center vertically
	fDateLocation.y = fTimeLocation.y = Bounds().Height()/2 + fFontHeight/2;
}
		
void
TTimeView::ShowClockOptions(BPoint point)
{
	BPopUpMenu	*menu = new BPopUpMenu("", false, false);
	menu->SetFont(be_plain_font);
	BMenuItem	*item;

	item = new BMenuItem(fShowSeconds ? "Hide Seconds" : "Show Seconds", new BMessage(msg_showseconds));
	menu->AddItem(item);
	item = new BMenuItem(fMilTime ? "12 Hour Clock" : "24 Hour Clock", new BMessage(msg_miltime));
	menu->AddItem(item);
	item = new BMenuItem(fFullDate ? "Short Date" : "Full Date", new BMessage(msg_fulldate));
	item->SetEnabled ((!fFullDate && CanShowFullDate()) || fFullDate);
	menu->AddItem(item);
	
	menu->AddSeparatorItem();
	
	item = new BMenuItem("Change Time…", new BMessage(msg_changeclock));
	menu->AddItem(item);

	item = new BMenuItem("Hide Time", new BMessage('time'));
	menu->AddItem(item);

	menu->SetTargetForItems(this);
	// Changed to accept screen coord system point;
	// not constrained to this view now
	menu->Go(point, true, true, BRect(point.x - 4, point.y - 4,
		point.x + 4, point.y +4), true);
}
