/*
Open Tracker License

Terms and Conditions

Copyright (c) 1991-2000, Be Incorporated. All rights reserved.

Permission is hereby granted, free of charge, to any person obtaining a copy of
this software and associated documentation files (the "Software"), to deal in
the Software without restriction, including without limitation the rights to
use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
of the Software, and to permit persons to whom the Software is furnished to do
so, subject to the following conditions:

The above copyright notice and this permission notice applies to all licensees
and shall be included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF TITLE, MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
BE INCORPORATED BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF, OR IN CONNECTION
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of Be Incorporated shall not be
used in advertising or otherwise to promote the sale, use or other dealings in
this Software without prior written authorization from Be Incorporated.

Tracker(TM), Be(R), BeOS(R), and BeIA(TM) are trademarks or registered trademarks
of Be Incorporated in the United States and other countries. Other brand product
names are registered trademarks or trademarks of their respective holders.
All rights reserved.
*/

#include <string.h>
#include <stdlib.h>
#include <image.h>

#include <Alert.h>
#include <Application.h>
#include <Debug.h>
#include <Directory.h>
#include <Entry.h>
#include <FindDirectory.h>
#include <MenuItem.h>
#include <MenuBar.h>
#include <Path.h>
#include <PopUpMenu.h>
#include <Screen.h>
#include <Volume.h>
#include <VolumeRoster.h>

#include <fs_attr.h>

#include <memory>

#include "Attributes.h"
#include "AttributeStream.h"
#include "AutoLock.h"
#include "BackgroundImage.h"
#include "Commands.h"
#include "ContainerWindow.h"
#include "DeskWindow.h"
#include "FindPanel.h"
#include "TFSContext.h"
#include "IconMenuItem.h"
#include "OpenWithWindow.h"
#include "Model.h"
#include "MountMenu.h"
#include "NavMenu.h"
#include "PoseView.h"
#include "SelectionWindow.h"
#include "TitleView.h"
#include "Tracker.h"
#include "ThreadMagic.h"

_IMPEXP_BE void		do_minimize_team(BRect zoomRect, team_id team, bool zoom);

namespace BPrivate {

const char *kAddOnsMenuName = "Add-Ons";

}

const int32 kContainerWidthMinLimit = 120;
const int32 kContainerWindowHeightLimit = 85;

const int32 kWindowStaggerBy = 17;

BRect BContainerWindow::fNewWindRect(85, 50, 415, 280);

BContainerWindow::BContainerWindow(LockingList<BWindow> *list,
		uint32 containerWindowFlags,
		window_look look, window_feel feel, uint32 flags, uint32 workspace)
	:	BWindow(InitialWindowRect(feel), "TrackerWindow", look, feel, flags,
			workspace),
		fFileContextMenu(0),
		fWindowContextMenu(0),
		fDropContextMenu(0),
		fVolumeContextMenu(0),
		fDragContextMenu(0),
		fMoveToItem(0),
		fCopyToItem(0),
		fCreateLinkItem(0),
		fOpenWithItem(0),
		fNavigationItem(0),
		fMenuBar(0),
		fPoseView(0),
		fWindowList(list),
		fAttrMenu(0),
		fFileMenu(0),
		fSelectionWindow(NULL),
		fTaskLoop(0),
		fIsTrash(false),
		fInTrash(false),
		fIsPrinters(false),
		fContainerWindowFlags(containerWindowFlags),
		fBackgroundImage(0),
		fSavedZoomRect(0, 0, -1, -1),
		fContextMenu(0),
		fDragMessage(NULL),
		fCachedTypesList(NULL)
{
	TTrackerState::InitIfNeeded();
	
	if (list) {
		ASSERT(list->IsLocked());
		list->AddItem(this);
	}
	Run();
}

BContainerWindow::~BContainerWindow()
{
	ASSERT(IsLocked());
	delete fTaskLoop;
	delete fBackgroundImage;
	delete fDragMessage;
	delete fCachedTypesList;
	
	if (fSelectionWindow && fSelectionWindow->Lock())
		fSelectionWindow->Quit();
	
}

BRect 
BContainerWindow::InitialWindowRect(window_feel feel)
{
	if (feel != kPrivateDesktopWindowFeel)
		return fNewWindRect;
	
	// do not offset desktop window
	BRect result = fNewWindRect;
	result.OffsetTo(0, 0);
	return result;
}

void
BContainerWindow::Minimize(bool minimize)
{
	if (minimize && (modifiers() & B_OPTION_KEY) != 0)
		do_minimize_team(BRect(0, 0, 0, 0), be_app->Team(), true);
	else
		_inherited::Minimize(minimize);
}

bool
BContainerWindow::QuitRequested()
{
	// this is a response to the DeskBar sending us a B_QUIT, when it really
	// means to say close all your windows. It might be better to have it
	// send a kCloseAllWindows message and have windowless apps stay running,
	// which is what we will do for the Tracker
	if (CurrentMessage()
		&& (CurrentMessage()->FindInt32("modifiers") & B_CONTROL_KEY))
		be_app->PostMessage(kCloseAllWindows);

	return true;
}

void
BContainerWindow::Quit()
{
	// get rid of context menus
	if (fNavigationItem) {
		BMenu *menu = fNavigationItem->Menu();
		if (menu)
			menu->RemoveItem(fNavigationItem);
		delete fNavigationItem;
		fNavigationItem = NULL;
	}
	if (fOpenWithItem && !fOpenWithItem->Menu()) {
		delete fOpenWithItem;
		fOpenWithItem = NULL;
	}

	if (fMoveToItem && !fMoveToItem->Menu()) {
		delete fMoveToItem;
		fMoveToItem = NULL;
	}

	if (fCopyToItem && !fCopyToItem->Menu()) {
		delete fCopyToItem;
		fCopyToItem = NULL;
	}

	if (fCreateLinkItem && !fCreateLinkItem->Menu()) {
		delete fCreateLinkItem;
		fCreateLinkItem = NULL;
	}

	if (fAttrMenu && !fAttrMenu->Supermenu()) {
		delete fAttrMenu;
		fAttrMenu = NULL;
	}
		
	delete fFileContextMenu;
	fFileContextMenu = NULL;
	delete fWindowContextMenu;
	fWindowContextMenu = NULL;
	delete fDropContextMenu;
	fDropContextMenu = NULL;
	delete fVolumeContextMenu;
	fVolumeContextMenu = NULL;
	delete fDragContextMenu;
	fDragContextMenu = NULL;
	
	int32 wind_count = 0;
	
	// This is a deadlock code sequence - need to change this
	// to acquire the window list while this container window is unlocked
	if (fWindowList) {
		AutoLock<LockingList<BWindow> > lock(fWindowList);
		if (lock.IsLocked()) {
			fWindowList->RemoveItem(this);
			wind_count = fWindowList->CountItems();
		}
	}

	SaveState();

	if (fWindowList && (wind_count == 0))
		be_app->PostMessage(B_QUIT_REQUESTED);

	_inherited::Quit();
}

BPoseView *
BContainerWindow::NewPoseView(Model *model, BRect rect, uint32 viewMode)
{
	return new BPoseView(model, rect, viewMode);
}

void
BContainerWindow::CreatePoseView(Model *model)
{
	BEntry entry(model->EntryRef());

	if (entry.InitCheck() == B_OK) {
		fIsTrash = TFSContext::IsTrashDir(entry);
		fInTrash = TFSContext::IsInTrash(entry);
		fIsPrinters = TFSContext::IsPrintersDir(entry);
	}

	BRect rect(Bounds());
	rect.right -= B_V_SCROLL_BAR_WIDTH;
	rect.bottom -= B_H_SCROLL_BAR_HEIGHT;
	fPoseView = NewPoseView(model, rect, kListMode);
	AddChild(fPoseView);
}


void 
BContainerWindow::AddContextMenus()
{
	// create context sensitive menus
	fFileContextMenu = new BContextMenu("FileContext", this);
	AddFileContextMenus(fFileContextMenu);
	
	fVolumeContextMenu = new BContextMenu("VolumeContext", this);
	AddVolumeContextMenus(fVolumeContextMenu);

	fWindowContextMenu = new BContextMenu("WindowContext", this);
	AddWindowContextMenus(fWindowContextMenu);

	fDropContextMenu = new BPopUpMenu("DropContext", false, false);
	fDropContextMenu->SetFont(be_plain_font);
	AddDropContextMenus(fDropContextMenu);
	
	fDragContextMenu = new BSlowContextMenu("DragContext");
		//	will get added and built dynamically in ShowContextMenu
}

void
BContainerWindow::Init()
{
	float y_delta;
	BEntry entry;

	ASSERT(fPoseView);
	if (!fPoseView)
		return;

	// deal with new unconfigured folders
	if (NeedsDefaultStateSetup())
		SetUpDefaultState();

	if (ShouldAddScrollBars())
		fPoseView->AddScrollBars();

	fMoveToItem = new BMenuItem(new BNavMenu("Move to", kMoveSelectionTo, this));
	fCopyToItem = new BMenuItem(new BNavMenu("Copy to", kCopySelectionTo, this));
	fCreateLinkItem = new BMenuItem(new BNavMenu("Create Link", kCreateLink, this),
		new BMessage(kCreateLink));

	if (ShouldAddMenus()) {
		// add menu bar, menus and resize poseview to fit
		fMenuBar = new BMenuBar(BRect(0, 0, Bounds().Width() + 1, 1), "MenuBar");
		fMenuBar->SetBorder(B_BORDER_FRAME);
		AddMenus();
		AddChild(fMenuBar);

		y_delta = KeyMenuBar()->Bounds().Height() + 1;
		fPoseView->MoveTo(BPoint(0, y_delta));
		fPoseView->ResizeBy(0, -(y_delta));
		if (fPoseView->VScrollBar()) {
			fPoseView->VScrollBar()->MoveBy(0, KeyMenuBar()->Bounds().Height() + 1);
			fPoseView->VScrollBar()->ResizeBy(0, -(KeyMenuBar()->Bounds().Height() + 1));
		}
	} else
		// add equivalents of the menu shortcuts to the menuless desktop window
		AddShortcuts();


	AddContextMenus();
	AddShortcut('T', B_COMMAND_KEY | B_SHIFT_KEY, new BMessage(kDelete), PoseView());
	AddShortcut('K', B_COMMAND_KEY | B_SHIFT_KEY, new BMessage(kCleanupAll),
		PoseView());
	AddShortcut('Q', B_COMMAND_KEY | B_OPTION_KEY | B_SHIFT_KEY | B_CONTROL_KEY,
		new BMessage(kQuitTracker));
	AddShortcut(B_DOWN_ARROW, B_COMMAND_KEY, new BMessage(kOpenSelection),
		PoseView());
	AddShortcut(B_DOWN_ARROW, B_OPTION_KEY | B_COMMAND_KEY,
		new BMessage(kOpenSelection), PoseView());
		// the command option results in closing the parent window
	AddShortcut(B_UP_ARROW, B_COMMAND_KEY | B_OPTION_KEY,
		new BMessage(kOpenParentDir), PoseView());
	AddShortcut(B_UP_ARROW, B_COMMAND_KEY | B_OPTION_KEY | B_CONTROL_KEY,
		new BMessage(kOpenParentDir), PoseView());
	AddShortcut(B_UP_ARROW, B_COMMAND_KEY | B_CONTROL_KEY,
		new BMessage(kOpenParentDir), PoseView());
		// the command option results in closing the parent window
		// the control is a secret backdoor to get at the Disks menu

#if DEBUG
	// add some debugging shortcuts
	AddShortcut('D', B_COMMAND_KEY | B_CONTROL_KEY, new BMessage('dbug'), PoseView());
	AddShortcut('C', B_COMMAND_KEY | B_CONTROL_KEY, new BMessage('dpcc'), PoseView());
	AddShortcut('F', B_COMMAND_KEY | B_CONTROL_KEY, new BMessage('dpfl'), PoseView());
	AddShortcut('F', B_COMMAND_KEY | B_CONTROL_KEY | B_OPTION_KEY,
		new BMessage('dpfL'), PoseView());
#endif

	RestoreState();

	if (ShouldAddMenus() && PoseView()->ViewMode() == kListMode) {
		// for now only show attributes in list view
		// eventually enable attribute menu to allow users to select
		// using different attributes as titles in icon view modes
		ShowAttributeMenu();
	}
	MarkAttributeMenu(fAttrMenu);
	CheckScreenIntersect();

	if (fBackgroundImage && !dynamic_cast<BDeskWindow *>(this)
		&& PoseView()->ViewMode() != kListMode)
		fBackgroundImage->Show(PoseView(), current_workspace());

	Show();

	// done showing, turn the B_NO_WORKSPACE_ACTIVATION flag off;
	// it was on to prevent workspace jerking during boot
	SetFlags(Flags() & ~B_NO_WORKSPACE_ACTIVATION);
}

void
BContainerWindow::RestoreState()
{
	SetSizeLimits(kContainerWidthMinLimit, 10000, kContainerWindowHeightLimit, 10000);
	
	// set title to full path, if necessary
	TTrackerState trackerState;
	if (trackerState.ShowFullPathInTitleBar()) {
		// use the Entry's full path
		BPath path; 
		TargetModel()->GetPath(&path); 
		SetTitle(path.Path());
	}
	else {
		// use the default look
		SetTitle(TargetModel()->Name());	
	}
	
	WindowStateNodeOpener opener(this, false);
	RestoreWindowState(opener.StreamNode());
	fPoseView->Init(opener.StreamNode());

	if (BootedInSafeMode())
		// don't pick up backgrounds in safe mode
		return;

	bool isDesktop = dynamic_cast<BDeskWindow *>(this);
	if (!TargetModel()->IsRoot() && opener.Node())
		// don't pick up background image for root disks
		// to do this, would have to have a unique attribute for the
		// disks window that doesn't collide with the desktop
		// for R4 this was not done to make things simpler
		// the default image will still work though
		fBackgroundImage = BackgroundImage::GetBackgroundImage(
			opener.Node(), isDesktop);
			// look for background image info in the window's node
	
	BNode defaultingNode;
	if (!fBackgroundImage && !isDesktop
		&& DefaultStateSourceNode(kDefaultFolderTemplate, &defaultingNode)) 
		// look for background image info in the source for defaults
		fBackgroundImage = BackgroundImage::GetBackgroundImage(&defaultingNode, isDesktop);
}

void
BContainerWindow::UpdateBackgroundImage()
{
	if (BootedInSafeMode())
		return;

	bool isDesktop = dynamic_cast<BDeskWindow *>(this) != NULL;
	WindowStateNodeOpener opener(this, false);

	if (!TargetModel()->IsRoot() && opener.Node())
		fBackgroundImage = BackgroundImage::Refresh(fBackgroundImage,
			opener.Node(), isDesktop, PoseView());
	
		// look for background image info in the window's node
	BNode defaultingNode;
	if (!fBackgroundImage && !isDesktop
		&& DefaultStateSourceNode(kDefaultFolderTemplate, &defaultingNode)) 
		// look for background image info in the source for defaults
		fBackgroundImage = BackgroundImage::Refresh(fBackgroundImage,
			&defaultingNode, isDesktop, PoseView());
}

void
BContainerWindow::FrameResized(float, float)
{
	if (PoseView()) {
		PoseView()->UpdateScrollRange();
		PoseView()->ResetPosePlacementHint();
	}
}

void 
BContainerWindow::ViewModeChanged(uint32 oldMode, uint32 newMode)
{
	if (!fBackgroundImage)
		return;

	if (newMode == kListMode)
		fBackgroundImage->Remove();
	else if (oldMode == kListMode)
		fBackgroundImage->Show(PoseView(), current_workspace()); 
}


void
BContainerWindow::CheckScreenIntersect()
{
	BScreen screen(this);
	BRect scrn_frame(screen.Frame());
	BRect frame(Frame());

	if (fNewWindRect.top + 20 > scrn_frame.bottom)
		fNewWindRect.OffsetTo(85, 50);

	if (fNewWindRect.left + 20 > scrn_frame.right)
		fNewWindRect.OffsetTo(85, 50);

	if (!frame.Intersects(scrn_frame))
		MoveTo(fNewWindRect.LeftTop());
}

void
BContainerWindow::SaveState(bool hide)
{
	WindowStateNodeOpener opener(this, true);
	if (opener.StreamNode())
		SaveWindowState(opener.StreamNode());
	if (hide)
		Hide();
	if (opener.StreamNode())
		fPoseView->SaveState(opener.StreamNode());
}

status_t 
BContainerWindow::GetLayoutState(BNode *node, BMessage *message)
{
	// ToDo:
	// get rid of this, use AttrStream instead
	status_t result = node->InitCheck();
	if (result != B_OK)
		return result;
	
	node->RewindAttrs();
	char attrName[256];
	while (node->GetNextAttrName(attrName) == B_OK) {
		attr_info info;
		node->GetAttrInfo(attrName, &info);
		
		// filter out attributes that are not related to window position
		// and column resizing
		// more can be added as needed
		if (strcmp(attrName, kAttrWindowFrame) != 0
			&& strcmp(attrName, kAttrColumns) != 0
			&& strcmp(attrName, kAttrViewState) != 0
			&& strcmp(attrName, kAttrColumnsForeign) != 0
			&& strcmp(attrName, kAttrViewStateForeign) != 0)
			continue;

		char *buffer = new char[info.size];
		if (node->ReadAttr(attrName, info.type, 0, buffer, (size_t)info.size)
			== info.size)
			message->AddData(attrName, info.type, buffer, (ssize_t)info.size);
		delete [] buffer;
	}
	return B_OK;
}

status_t 
BContainerWindow::SetLayoutState(BNode *node, const BMessage *message)
{
	status_t result = node->InitCheck();
	if (result != B_OK)
		return result;
	
	for (int32 globalIndex = 0; ;) {
		char *name;
		type_code type;
		int32 count;
		status_t result = message->GetInfo(B_ANY_TYPE, globalIndex, &name,
			&type, &count);
		if (result != B_OK)
			break;
		
		for (int32 index = 0; index < count; index++) {
			const void *buffer;
			int32 size;
			result = message->FindData(name, type, index, &buffer, &size);
			if (result != B_OK) {
				PRINT(("error reading %s \n", name));
				return result;
			}
			
			if (node->WriteAttr(name, type, 0, buffer, (size_t)size) != size) {
				PRINT(("error writing %s \n", name));
				return result;
			}
			globalIndex++;
		}
	}
	return B_OK;
}

bool
BContainerWindow::ShouldAddMenus() const
{
	return true;
}

bool
BContainerWindow::ShouldAddScrollBars() const
{
	return true;
}

bool
BContainerWindow::ShouldAddCountView() const
{
	return true;
}


Model *
BContainerWindow::TargetModel() const
{
	return fPoseView->TargetModel();
}

void
BContainerWindow::SelectionChanged()
{
}

void
BContainerWindow::Zoom(BPoint, float, float)
{
	BRect old_zoom_rect(fSavedZoomRect);
	fSavedZoomRect = Frame();
	ResizeToFit();

	if (fSavedZoomRect == Frame())
		if (old_zoom_rect.IsValid())
			ResizeTo(old_zoom_rect.Width(), old_zoom_rect.Height());
}

void
BContainerWindow::ResizeToFit()
{
	BScreen screen(this);
	BRect scrn_frame(screen.Frame());

	scrn_frame.InsetBy(5, 5);
	scrn_frame.top += 15;		// keeps title bar of window visible

	BRect frame(Frame());

	// move frame left top on screen
	BPoint left_top(frame.LeftTop());
	left_top.ConstrainTo(scrn_frame);
	frame.OffsetTo(left_top);

	// resize to extent size
	float menuHeight;
	if (KeyMenuBar())
		menuHeight = KeyMenuBar()->Bounds().Height();
	else
		menuHeight = 0;

	BRect extent(PoseView()->Extent());
	frame.right = frame.left + extent.Width() + (float)B_V_SCROLL_BAR_WIDTH + 1.0f;
	frame.bottom = frame.top + extent.Height() + (float)B_H_SCROLL_BAR_HEIGHT + 1.0f
		+ menuHeight;

	if (PoseView()->ViewMode() == kListMode)
		frame.bottom += kTitleViewHeight + 1;  // account for list titles

// ToDo:
// clean this up, move each special case to respective class

	if (!TargetModel())
		frame.bottom += 60;						// Open with window
	else if (TargetModel()->IsQuery())
		frame.bottom += 15;						// account for query string
	
	// make sure entire window fits on screen
	frame = frame & scrn_frame;

	ResizeTo(frame.Width(), frame.Height());
	MoveTo(frame.LeftTop());
	PoseView()->DisableScrollBars();
	PoseView()->ScrollTo(extent.LeftTop());
	PoseView()->UpdateScrollRange();
	PoseView()->EnableScrollBars();
}

void
BContainerWindow::MessageReceived(BMessage *message)
{
	switch (message->what) {
		case kContextMenuDragNDrop:
			//
			//	sent when the SlowContextPopup goes away
			//
			if (fWaitingForRefs && Dragging())
				PostMessage(message, PoseView());
			else
				fWaitingForRefs = false;
			break;
						
		case kRestoreState:
			Init();
			break;

		case kResizeToFit:
			ResizeToFit();
			break;

		case kLoadAddOn:
			LoadAddOn(message);
			break;

		case kCopySelectionTo:
			{
				entry_ref ref;
				if (message->FindRef("refs", &ref) != B_OK)
					break;

				Model model(&ref);
				if (model.InitCheck() != B_OK)
					break;

				if (*model.NodeRef() == *TargetModel()->NodeRef())
					PoseView()->DuplicateSelection();
				else
					PoseView()->MoveSelectionInto(&model, this, true);

				break;
			}
		case kMoveSelectionTo:
			{
				entry_ref ref;
				if (message->FindRef("refs", &ref) != B_OK)
					break;
				
				Model model(&ref);
				if (model.InitCheck() != B_OK)
					break;
	
				PoseView()->MoveSelectionInto(&model, this, false);
				break;
			}

		case kCreateLink:
		case kCreateRelativeLink:
			{
				entry_ref ref;

				if (message->FindRef("refs", &ref) == B_OK) {
					Model model(&ref);
					if (model.InitCheck() != B_OK)
						break;
					PoseView()->MoveSelectionInto(&model, this, false, true,
						(message->what == kCreateRelativeLink));
				} else {
					// no destination specified, create link in same dir as item
					if (!TargetModel()->IsQuery())
						PoseView()->MoveSelectionInto(TargetModel(), this, false, true,
							(message->what == kCreateRelativeLink));
				}
				break;
			}

		case kShowSelectionWindow:
			ShowSelectionWindow();
			break;
		
		case kSelectMatchingEntries:
			PoseView()->SelectMatchingEntries(message);
			break;
					
		case kFindButton:
			(new FindWindow())->Show();
			break;

		case kQuitTracker:
			be_app->PostMessage(B_QUIT_REQUESTED);
			break;

		case kRestoreBackgroundImage:
			UpdateBackgroundImage();
			break;
			
		case B_REFS_RECEIVED:
			if (Dragging()) {
				//
				//	ref in this message is the target,
				//	the end point of the drag
				//
				entry_ref ref;
				if (message->FindRef("refs", &ref) == B_OK) {
					//printf("BContainerWindow::MessageReceived - refs received\n");
					fWaitingForRefs = false;
					BEntry entry(&ref, true);
					//
					//	don't copy to printers dir
					if (TFSContext::IsPrintersDir(entry) == false) {
						if (entry.InitCheck() == B_OK && entry.IsDirectory()) {
							//
							//	build of list of entry_refs from the list
							//	in the drag message
							auto_ptr<TFSContext> tfscontext(new TFSContext(*fDragMessage));

							//
							//	compare the target and one of the drag items' parent
							//	
							entry_ref dragref;
							fDragMessage->FindRef("refs", 0, &dragref);
							
							BEntry item(&dragref, true);
							BEntry itemparent;
							item.GetParent(&itemparent);
							entry_ref parentref;
							itemparent.GetRef(&parentref);
							
							entry_ref targetref;
							entry.GetRef(&targetref);

							//	if they don't match, move/copy
							if (targetref != parentref) {
								//	copy drag contents to target ref in message
								BDirectory target_dir(&entry);
								if (target_dir.InitCheck() == B_OK)
									tfscontext.release()->MoveTo(target_dir, true);	// async
							}

						} else {
							// 	current message sent to apps is only B_REFS_RECEIVED
							fDragMessage->what = B_REFS_RECEIVED;
							FSLaunchItem(&ref, fDragMessage, true, true);
						}
					}
				}
				DragStop();
			}
			break;
		
		default:
			_inherited::MessageReceived(message);
	}
}

void
BContainerWindow::SetCleanUpItem(BMenu *menu)
{
	bool hasPoses = PoseView()->CountItems() > 0;
	BMenuItem *item = menu->FindItem(kCleanup);

	if (item || (item = menu->FindItem(kCleanupAll)) != NULL) {
		item->SetEnabled(hasPoses && (PoseView()->ViewMode() != kListMode));
		if (modifiers() & B_SHIFT_KEY) {
			item->SetLabel("Clean Up All");
			item->SetShortcut('K', B_COMMAND_KEY | B_SHIFT_KEY);
			item->SetMessage(new BMessage(kCleanupAll));
		} else {
			item->SetLabel("Clean Up");
			item->SetShortcut('K', B_COMMAND_KEY);
			item->SetMessage(new BMessage(kCleanup));
		}
	}
}

void
BContainerWindow::SetCloseItem(BMenu *menu)
{
	BMenuItem *item = menu->FindItem(B_CLOSE_REQUESTED);
	
	if (item || (item = menu->FindItem(kCloseAllWindows)) != NULL) {
		if (modifiers() & B_OPTION_KEY) {
			item->SetLabel("Close All");
			item->SetShortcut('W', B_COMMAND_KEY | B_OPTION_KEY);
			item->SetTarget(be_app);
			item->SetMessage(new BMessage(kCloseAllWindows));
		} else {
			item->SetLabel("Close");
			item->SetShortcut('W', B_COMMAND_KEY);
			item->SetTarget(this);
			item->SetMessage(new BMessage(B_CLOSE_REQUESTED));
		}
	}
}

bool
BContainerWindow::IsShowing(const node_ref *node) const
{
	return PoseView()->Represents(node);
}

bool
BContainerWindow::IsShowing(const entry_ref *entry) const
{
	return PoseView()->Represents(entry);
}

static BMenuItem *
MenuItemSetInvokeTimeout(BMenuItem *item)
{
	item->SetTimeout(kSynchMenuInvokeTimeout);
	return NULL;
}

void
BContainerWindow::AddMenus()
{
	fFileMenu = new BMenu("File");
	AddFileMenu(fFileMenu);
	fMenuBar->AddItem(fFileMenu);
	BMenu *menu = new BMenu("Window");
	fMenuBar->AddItem(menu);
	AddWindowMenu(menu);
	// just create the attribute, decide to add it later
	fAttrMenu = new BMenu("Attributes");
	NewAttributeMenu(fAttrMenu);
}

void
BContainerWindow::AddFileMenu(BMenu *menu)
{
	if (!PoseView()->IsFilePanel())
		menu->AddItem(new BMenuItem("Find"B_UTF8_ELLIPSIS,
			new BMessage(kFindButton), 'F'));

	if (!TargetModel()->IsQuery() && !IsTrash() && !InTrash() && !IsPrintersDir())
		menu->AddItem(new BMenuItem("New Folder", new BMessage(kNewFolder), 'N'));
	menu->AddSeparatorItem();
	
	menu->AddItem(new BMenuItem("Open", new BMessage(kOpenSelection), 'O'));
	menu->AddItem(new BMenuItem("Get Info", new BMessage(kGetInfo), 'I'));
	menu->AddItem(new BMenuItem("Edit Name", new BMessage(kEditItem), 'E'));

	if (IsTrash() || InTrash()) {
		menu->AddItem(new BMenuItem("Delete", new BMessage(kDelete)));
		menu->AddItem(new BMenuItem("Restore", new BMessage(kRestoreFromTrash)));
		if (IsTrash()) {
			// add as first item in menu
			menu->AddItem(new BMenuItem("Empty Trash", new BMessage(kEmptyTrash)), 0);
			menu->AddItem(new BSeparatorItem(), 1);
		}
	} else if (IsPrintersDir()) {
		menu->AddItem(new BMenuItem("Add Printer"B_UTF8_ELLIPSIS,
			new BMessage(kAddPrinter), 'N'), 0);
		menu->AddItem(new BSeparatorItem(), 1);
		menu->AddItem(new BMenuItem("Make Active Printer",
			new BMessage(kMakeActivePrinter)));
	} else {
		menu->AddItem(new BMenuItem("Duplicate",
			new BMessage(kDuplicateSelection), 'D'));
		menu->AddItem(new BMenuItem("Move to Trash", new BMessage(kMoveToTrash), 'T'));
		menu->AddSeparatorItem();
		menu->AddSeparatorItem();
		
		menu->AddItem(new BMenuItem("Identify", new BMessage(kIdentifyEntry)));
		BMenu *addOnMenuItem = new BMenu(kAddOnsMenuName);
		addOnMenuItem->SetFont(be_plain_font);
		menu->AddItem(addOnMenuItem);
	}
	menu->SetTargetForItems(PoseView());
}

void
BContainerWindow::AddWindowMenu(BMenu *menu)
{
	BMenuItem *item;

	item = new BMenuItem("Icon View", new BMessage(kIconMode));
	item->SetTarget(PoseView());
	menu->AddItem(item);

	item = new BMenuItem("Mini Icon View", new BMessage(kMiniIconMode));
	item->SetTarget(PoseView());
	menu->AddItem(item);

	item = new BMenuItem("List View", new BMessage(kListMode));
	item->SetTarget(PoseView());
	menu->AddItem(item);

	menu->AddSeparatorItem();

	item = new BMenuItem("Resize to Fit", new BMessage(kResizeToFit), 'Y');
	item->SetTarget(this);
	menu->AddItem(item);

	item = new BMenuItem("Clean Up", new BMessage(kCleanup), 'K');
	item->SetTarget(PoseView());
	menu->AddItem(item);

	item = new BMenuItem("Select"B_UTF8_ELLIPSIS, new BMessage(kShowSelectionWindow), 'A', B_SHIFT_KEY);
	item->SetTarget(PoseView());
	menu->AddItem(item);

	item = new BMenuItem("Select All", new BMessage(B_SELECT_ALL), 'A');
	item->SetTarget(PoseView());
	menu->AddItem(item);
	
	if (!IsTrash()) {
		item = new BMenuItem("Open Parent", new BMessage(kOpenParentDir),
			B_UP_ARROW);
		item->SetTarget(PoseView());
		menu->AddItem(item);
	}

	item = new BMenuItem("Close", new BMessage(B_CLOSE_REQUESTED), 'W');
	item->SetTarget(this);
	menu->AddItem(item);
}

void
BContainerWindow::AddShortcuts()
{
	// add equivalents of the menu shortcuts to the menuless desktop window
	ASSERT(!IsTrash());
	ASSERT(!PoseView()->IsFilePanel());
	ASSERT(!TargetModel()->IsQuery());

	AddShortcut('F', B_COMMAND_KEY, new BMessage(kFindButton), PoseView());
	AddShortcut('N', B_COMMAND_KEY, new BMessage(kNewFolder), PoseView());
	AddShortcut('O', B_COMMAND_KEY, new BMessage(kOpenSelection), PoseView());
	AddShortcut('I', B_COMMAND_KEY, new BMessage(kGetInfo), PoseView());
	AddShortcut('E', B_COMMAND_KEY, new BMessage(kEditItem), PoseView());
	AddShortcut('D', B_COMMAND_KEY, new BMessage(kDuplicateSelection), PoseView());
	AddShortcut('T', B_COMMAND_KEY, new BMessage(kMoveToTrash), PoseView());
	AddShortcut('K', B_COMMAND_KEY, new BMessage(kCleanup), PoseView());
	AddShortcut('A', B_COMMAND_KEY, new BMessage(B_SELECT_ALL), PoseView());
	AddShortcut('A', B_COMMAND_KEY | B_SHIFT_KEY, new BMessage(kShowSelectionWindow), PoseView());
	AddShortcut('G', B_COMMAND_KEY, new BMessage(kEditQuery), PoseView());
		// it is ok to add a global Edit query shortcut here, PoseView will
		// filter out cases where selected pose is not a query
	AddShortcut('U', B_COMMAND_KEY, new BMessage(kUnmountVolume), PoseView());
	AddShortcut(B_UP_ARROW, B_COMMAND_KEY, new BMessage(kOpenParentDir), PoseView());
	AddShortcut('O', B_COMMAND_KEY | B_CONTROL_KEY, new BMessage(kOpenSelectionWith),
		PoseView());
}

void
BContainerWindow::MenusBeginning()
{
	if (!fMenuBar)
		return;

	if (CurrentMessage() && CurrentMessage()->what == B_MOUSE_DOWN)
		// don't commit active pose if only a keyboard shortcut is
		// invoked - this would prevent Cut/Copy/Paste from working
		fPoseView->CommitActivePose();

	// File menu
	int32 selectCount = PoseView()->SelectionList()->CountItems();

	SetUpEditQueryItem(fFileMenu);

	SetupMoveCopyMenus(selectCount
		? PoseView()->SelectionList()->FirstItem()->TargetModel()->EntryRef() : NULL, fFileMenu);

	SetupOpenWithMenu(fFileMenu);

	EnableNamedMenuItem(fMenuBar, kOpenSelection, selectCount > 0);
	EnableNamedMenuItem(fMenuBar, kGetInfo, selectCount > 0);
	EnableNamedMenuItem(fMenuBar, kIdentifyEntry, selectCount > 0);
	EnableNamedMenuItem(fMenuBar, kMoveToTrash, selectCount > 0);
	EnableNamedMenuItem(fMenuBar, kRestoreFromTrash, selectCount > 0);
	EnableNamedMenuItem(fMenuBar, kDelete, selectCount > 0);
	EnableNamedMenuItem(fMenuBar, kEditItem, 
		(selectCount == 1) && !PoseView()->ActivePose());
	EnableNamedMenuItem(fMenuBar, kDuplicateSelection, selectCount > 0);
	EnableNamedMenuItem(fMenuBar, kEmptyTrash, PoseView()->CountItems() > 0);
	EnableNamedMenuItem(fMenuBar, B_SELECT_ALL, PoseView()->CountItems() > 0);

	SetCloseItem(fMenuBar);
	SetCleanUpItem(fMenuBar);

	BuildAddOnMenu(fMenuBar);

	MarkNamedMenuItem(fMenuBar, kIconMode, PoseView()->ViewMode() == kIconMode);
	MarkNamedMenuItem(fMenuBar, kListMode, PoseView()->ViewMode() == kListMode);
	MarkNamedMenuItem(fMenuBar, kMiniIconMode,
		PoseView()->ViewMode() == kMiniIconMode);


	EnableNamedMenuItem(fMenuBar, kOpenParentDir, !TargetModel()->IsRoot());

	AddMimeTypesToMenu(fAttrMenu);
	
	if (IsPrintersDir())
		EnableNamedMenuItem(fFileMenu, "Make Active Printer", selectCount == 1);
}

void
BContainerWindow::MenusEnded()
{
	BMenu *menu;
	BMenuItem *item;

	// when we're done we want to clear nav menus for next time
	if (fNavigationItem) {
		menu = fNavigationItem->Submenu();
		while ((item = menu->RemoveItem((int32)0)) != NULL)
			delete item;
	}

	if (fMoveToItem) {
		menu = fMoveToItem->Submenu();
		while ((item = menu->RemoveItem((int32)0)) != NULL) 
			delete item;
	}

	if (fCopyToItem) {
		menu = fCopyToItem->Submenu();
		while ((item = menu->RemoveItem((int32)0)) != NULL) 
			delete item;
	}

	if (fCreateLinkItem) {
		menu = fCreateLinkItem->Submenu();
		while ((item = menu->RemoveItem((int32)0)) != NULL) 
			delete item;
	}
	
	if (fOpenWithItem) {
		menu = fOpenWithItem->Submenu();
		while ((item = menu->RemoveItem((int32)0)) != NULL) 
			delete item;
	}
}

void
BContainerWindow::SetupNavigationMenu(const entry_ref *ref, BMenu *parent)
{
	// start by removing nav item (and separator) from old menu
	if (fNavigationItem) {
		BMenu *menu = fNavigationItem->Menu();
		if (menu) {
			menu->RemoveItem(fNavigationItem);
			BMenuItem *item = menu->RemoveItem((int32)0);
			ASSERT(item != fNavigationItem);
			delete item;
		}
	}

	// if we weren't passed a ref then we're navigating this window
	if (!ref)
		ref = TargetModel()->EntryRef();

	BEntry entry;
	if (entry.SetTo(ref) != B_OK)
		return;

	// only navigate directories and queries (check for symlink here)
	Model model(&entry);
	entry_ref resolvedRef;

	if (model.InitCheck() != B_OK
		|| (!model.IsContainer() && !model.IsSymLink()))
		return;

	if (model.IsSymLink()) {
		if (entry.SetTo(model.EntryRef(), true) != B_OK)
			return;

		Model resolvedModel(&entry);
		if (resolvedModel.InitCheck() != B_OK || !resolvedModel.IsContainer())
			return;

		entry.GetRef(&resolvedRef);
		ref = &resolvedRef;
	}

	if (!fNavigationItem) 
		// icon menuitem adopts model
		fNavigationItem = new ModelMenuItem(new Model(model),
			new BNavMenu(model.Name(), B_REFS_RECEIVED, be_app, this));

	// setup a navigation menu item which will dynamically load items
	// as menu items are traversed
	BNavMenu *navMenu = dynamic_cast<BNavMenu *>(fNavigationItem->Submenu());
	navMenu->SetNavDir(ref);
	fNavigationItem->SetLabel(model.Name());
	fNavigationItem->SetEntry(&entry);

	parent->AddItem(fNavigationItem, 0);
	parent->AddItem(new BSeparatorItem(), 1);
	
	BMessage *message = new BMessage(B_REFS_RECEIVED);
	message->AddRef("refs", ref);
	fNavigationItem->SetMessage(message);
	fNavigationItem->SetTarget(be_app);
	
	if (!Dragging())
		parent->SetTrackingHook(NULL, NULL);
}

void
BContainerWindow::SetUpEditQueryItem(BMenu *menu)
{
	ASSERT(menu);
	// File menu
	int32 selectCount = PoseView()->SelectionList()->CountItems();

	// add Edit query if appropriate
	bool queryInSelection = false;
	if (selectCount && selectCount < 100) {
		// only do this for a limited number of selected poses

		// if any queries selected, add an edit query menu item
		for (int32 index = 0; index < selectCount; index++) {
			BPose *pose = PoseView()->SelectionList()->ItemAt(index);
			Model model(pose->TargetModel()->EntryRef(), true);
			if (model.InitCheck() != B_OK)
				continue;

			if (model.IsQuery() || model.IsQueryTemplate()) {
				queryInSelection = true;
				break;
			}
		}
	}

	bool poseViewIsQuery = TargetModel()->IsQuery();
		// if the view is a query pose view, add edit query menu item

	BMenuItem *item = menu->FindItem("Edit Query");
	if (!poseViewIsQuery && !queryInSelection && item)
		menu->RemoveItem(item);
		
	else if ((poseViewIsQuery || queryInSelection) && menu && !item) {

		// add edit query item after Open 
		item = menu->FindItem(kOpenSelection);
		if (item) {	
			int32 itemIndex = menu->IndexOf(item);
			item = new BMenuItem("Edit Query", new BMessage(kEditQuery), 'G');
			menu->AddItem(item, itemIndex + 1);
			item->SetTarget(PoseView());
		}
	}
}

void
BContainerWindow::SetupOpenWithMenu(BMenu *parent)
{
	// start by removing nav item (and separator) from old menu
	if (fOpenWithItem) {
		BMenu *menu = fOpenWithItem->Menu();
		if (menu) 
			menu->RemoveItem(fOpenWithItem);

		delete fOpenWithItem;
		fOpenWithItem = 0;
	}

	if (PoseView()->SelectionList()->CountItems() == 0)
		// no selection, nothing to open
		return;
	
	if (TargetModel()->IsRoot())
		// don't add ourselves if we are root
		return;

	// ToDo:
	// check if only item in selection list is the root
	// and do not add if true

	// add after "Open"
	BMenuItem *item = parent->FindItem(kOpenSelection);

	int32 count = PoseView()->SelectionList()->CountItems();
	if (!count)
		return;

	// build a list of all refs to open
	BMessage message(B_REFS_RECEIVED);
	for (int32 index = 0; index < count; index++) {
		BPose *pose = PoseView()->SelectionList()->ItemAt(index);
		message.AddRef("refs", pose->TargetModel()->EntryRef());
	}
	
	// add Tracker token so that refs received recipients can script us
	message.AddMessenger("TrackerViewToken", BMessenger(PoseView()));

	int32 index = parent->IndexOf(item);
	fOpenWithItem = new BMenuItem(
		new OpenWithMenu("Open With"B_UTF8_ELLIPSIS, &message, this, be_app),
		new BMessage(kOpenSelectionWith));
	fOpenWithItem->SetTarget(PoseView());
	fOpenWithItem->SetShortcut('O', B_COMMAND_KEY | B_CONTROL_KEY);

	parent->AddItem(fOpenWithItem, index + 1);
}

void
BContainerWindow::PopulateMoveCopyNavMenu(BNavMenu *navMenu, uint32 what,
	dev_t device, bool addLocalOnly)
{
	BVolume volume;
	BVolumeRoster volumeRoster;
	BDirectory directory;
	BEntry entry;
	BPath path;
	Model model;

	int32 volumeCount = 0;

	// count persistent writable volumes
	volumeRoster.Rewind();
	while (volumeRoster.GetNextVolume(&volume) == B_OK)
		if (!volume.IsReadOnly() && volume.IsPersistent())
			volumeCount++;

	// add Desktop
	TFSContext::GetBootDesktopDir(directory);
	if (directory.InitCheck() == B_OK
		&& directory.GetEntry(&entry) == B_OK
		&& model.SetTo(&entry) == B_OK)
		navMenu->AddNavDir(&model, what, this, true);
			// ask NavMenu to populate submenu for us			

	// add the home dir
	if (find_directory(B_USER_DIRECTORY, &path) == B_OK && 
		entry.SetTo(path.Path()) == B_OK &&
		model.SetTo(&entry) == B_OK)
		navMenu->AddNavDir(&model, what, this, true);

	navMenu->AddSeparatorItem();

	if (addLocalOnly || volumeCount < 2) {
		// add volume this item lives on
		if (volume.SetTo(device) == B_OK
			&& volume.GetRootDirectory(&directory) == B_OK
			&& directory.GetEntry(&entry) == B_OK
			&& model.SetTo(&entry) == B_OK) {
			navMenu->AddNavDir(&model, what, this, false);
				// do not have submenu populated

			navMenu->SetNavDir(model.EntryRef());
		}
	} else {
		// add all persistent writable volumes
		volumeRoster.Rewind();
		while (volumeRoster.GetNextVolume(&volume) == B_OK) {
			if (volume.IsReadOnly() || !volume.IsPersistent())
				continue;

			// add root dir
			if (volume.GetRootDirectory(&directory) == B_OK
				&& directory.GetEntry(&entry) == B_OK
				&& model.SetTo(&entry) == B_OK) 
				navMenu->AddNavDir(&model, what, this, true);
					// ask NavMenu to populate submenu for us
		}
	}
}

void
BContainerWindow::SetupMoveCopyMenus(const entry_ref *item_ref, BMenu *parent)
{
	if (IsTrash() || InTrash() || IsPrintersDir() || !fMoveToItem || !fCopyToItem || !fCreateLinkItem)
		return;

	// Grab the modifiers state since we use it twice
	uint32 modifierKeys = modifiers();
	
	// re-parent items to this menu since they're shared
	int32 index = parent->CountItems() - 3;

	if (fMoveToItem->Menu() != parent) {
		if (fMoveToItem->Menu())
			fMoveToItem->Menu()->RemoveItem(fMoveToItem);

		parent->AddItem(fMoveToItem, index++);
	}

	if (fCopyToItem->Menu() != parent) {
		if (fCopyToItem->Menu())
			fCopyToItem->Menu()->RemoveItem(fCopyToItem);

		parent->AddItem(fCopyToItem, index++);
	}

	if (fCreateLinkItem->Menu() != parent) {
		if (fCreateLinkItem->Menu())
			fCreateLinkItem->Menu()->RemoveItem(fCreateLinkItem);

		parent->AddItem(fCreateLinkItem, index);
	}

    // Set the "Create Link" item label here so it
    // appears correctly when menus are disabled, too.
    if (modifierKeys & B_SHIFT_KEY)
    	fCreateLinkItem->SetLabel("Create Relative Link");
    else
    	fCreateLinkItem->SetLabel("Create Link");
    	
	// only enable once the menus are built
	fMoveToItem->SetEnabled(false);
	fCopyToItem->SetEnabled(false);
	fCreateLinkItem->SetEnabled(false);

	// get ref for item which is selected
	BEntry entry;
	if (entry.SetTo(item_ref) != B_OK)
		return;
			
	Model tempModel(&entry);
	if (tempModel.InitCheck() != B_OK)
		return;

	if (tempModel.IsRoot() || tempModel.IsVolume())
		return;

	// configure "Move to" menu item 
	PopulateMoveCopyNavMenu(dynamic_cast<BNavMenu *>(fMoveToItem->Submenu()),
		kMoveSelectionTo, item_ref->device, true);

	// configure "Copy to" menu item 
	// add all mounted volumes (except the one this item lives on)
	PopulateMoveCopyNavMenu(dynamic_cast<BNavMenu *>(fCopyToItem->Submenu()),
		kCopySelectionTo, item_ref->device, false);

	// Set "Create Link" menu item message and
	// add all mounted volumes (except the one this item lives on)
	if (modifierKeys & B_SHIFT_KEY) {
		fCreateLinkItem->SetMessage(new BMessage(kCreateRelativeLink));
		PopulateMoveCopyNavMenu(dynamic_cast<BNavMenu *>(fCreateLinkItem->Submenu()),
			kCreateRelativeLink, item_ref->device, false);
	} else {
		fCreateLinkItem->SetMessage(new BMessage(kCreateLink));
		PopulateMoveCopyNavMenu(dynamic_cast<BNavMenu *>(fCreateLinkItem->Submenu()),
		kCreateLink, item_ref->device, false);
	}

	fMoveToItem->SetEnabled(true);
	fCopyToItem->SetEnabled(true);
	fCreateLinkItem->SetEnabled(true);
}

uint32
BContainerWindow::ShowDropContextMenu(BPoint loc)
{
	BPoint global(loc);

	PoseView()->ConvertToScreen(&global);
	PoseView()->CommitActivePose();
	BRect mouseRect(global.x, global.y, global.x, global.y);
	mouseRect.InsetBy(-5, -5);

	// Change the "Create Link" item - allow user to
	// create relative links with the Shift key down.
	BMenuItem *item = fDropContextMenu->FindItem(kCreateLink);
	if (item == NULL)
		item = fDropContextMenu->FindItem(kCreateRelativeLink);
	if (item && (modifiers() & B_SHIFT_KEY)) {
		item->SetLabel("Create Relative Link Here");
		item->SetMessage(new BMessage(kCreateRelativeLink));
	} else if (item) {
		item->SetLabel("Create Link Here");
		item->SetMessage(new BMessage(kCreateLink));
	}
	item = fDropContextMenu->Go(global, true, true, mouseRect);
	if (item)
		return item->Command();

	return 0;
}

void
BContainerWindow::ShowContextMenu(BPoint loc, const entry_ref *ref, BView *)
{
	ASSERT(IsLocked());
	BPoint global(loc);
	PoseView()->ConvertToScreen(&global);
	PoseView()->CommitActivePose();
	BRect mouseRect(global.x, global.y, global.x, global.y);
	mouseRect.InsetBy(-5, -5);
	
	if (ref) {
		// clicked on a pose, show file or volume context menu
		Model model(ref);

		bool showAsVolume = false;
		bool filePanel = PoseView()->IsFilePanel();
		
		if (Dragging()) {
			fContextMenu = NULL;

			BEntry entry;
			model.GetEntry(&entry);
			//
			//	only show for directories (directory, volume, root)
			//
			//	don't show a popup for the trash or printers
			//	trash is handled in DeskWindow
			//
			//	since this menu is opened asynchronously
			//	we need to make sure we don't open it more
			//	than once, the IsShowing flag is set in
			//	SlowContextPopup::AttachedToWindow and
			//	reset in DetachedFromWindow
			//	see the notes in SlowContextPopup::AttachedToWindow
			//
			if (!TFSContext::IsPrintersDir(entry) && !fDragContextMenu->IsShowing()) {
				// printf("ShowContextMenu - target is %s %i\n", ref->name, IsShowing(ref));
				fDragContextMenu->ClearMenu();
				//
				//	in case the ref is a symlink, resolve it
				//	only pop open for directories
				BEntry resolvedEntry(ref, true);
				if (!resolvedEntry.IsDirectory())
					return;
				
				entry_ref resolvedRef;
				resolvedEntry.GetRef(&resolvedRef);

				//	use the resolved ref for the menu
				fDragContextMenu->SetNavDir(&resolvedRef);
				
				fDragContextMenu->SetTypesList(fCachedTypesList);
				fDragContextMenu->SetTarget(BMessenger(this));
				
				BPoseView *poseView = PoseView();
				if (poseView) {
					BMessenger tmpTarget(poseView);
					fDragContextMenu->InitTrackingHook(
						&BPoseView::MenuTrackingHook, &tmpTarget, fDragMessage);
				}

				//	this is now asynchronous so that we don't
				//	deadlock in Window::Quit,
				fDragContextMenu->Go(global, true, false, true);
			}
			return;
		} else if (TargetModel()->IsRoot() || model.IsVolume()) {
			fContextMenu = fVolumeContextMenu;
			showAsVolume = true;
		} else
			fContextMenu = fFileContextMenu;

		if (fContextMenu) {
			if (model.InitCheck() == B_OK) { // ??? Do I need this ???
				if (showAsVolume) {
					// non-volume enable/disable copy, move, identify
					EnableNamedMenuItem(fContextMenu, kDuplicateSelection, false);
					EnableNamedMenuItem(fContextMenu, kMoveToTrash, false);
					EnableNamedMenuItem(fContextMenu, kIdentifyEntry, false);
				
					// volume model, enable/disable the Unmount item
					bool ejectableVolumeSelected = false;
					
					BVolume boot;
					BVolumeRoster().GetBootVolume(&boot);
					BVolume volume;
					volume.SetTo(model.NodeRef()->device);
					if (volume != boot)
						ejectableVolumeSelected = true;

					EnableNamedMenuItem(fContextMenu, "Unmount", ejectableVolumeSelected);
				}
			}

			SetupNavigationMenu(ref, fContextMenu);
			if (!showAsVolume && !filePanel) {
				SetupMoveCopyMenus(ref, fContextMenu);
				SetupOpenWithMenu(fContextMenu);
			}
			
			SetUpEditQueryItem(fContextMenu);
			BuildAddOnMenu(fContextMenu);

			EnableNamedMenuItem(fContextMenu, kEditItem, 
				PoseView()->SelectionList()->CountItems() == 1);

			EachMenuItem(fContextMenu, true, MenuItemSetInvokeTimeout);
				// make sure menu items are set to timeout since we do
				// a synch Go and need to prevent deadlocks
							
			fContextMenu->Go(global, true, false, mouseRect);
				// This should be an async Go to make sure I cannot deadlock
				// myself when the window's port is full; Needs a bunch of work to
				// make this happen - the item removal in MenusEnded needs to be redone,
				// etc.
		}
	} else if (fWindowContextMenu) {
		
		// clicked on a window, show window context menu
		
		SetupNavigationMenu(ref, fWindowContextMenu);
		BuildAddOnMenu(fWindowContextMenu);
		MarkNamedMenuItem(fWindowContextMenu, kListMode, false);
		MarkNamedMenuItem(fWindowContextMenu, kIconMode, false);
		MarkNamedMenuItem(fWindowContextMenu, kMiniIconMode, false);
		MarkNamedMenuItem(fWindowContextMenu, PoseView()->ViewMode(), true);

		SetCloseItem(fWindowContextMenu);
		SetCleanUpItem(fWindowContextMenu);
		
		int32 count = PoseView()->CountItems();
		EnableNamedMenuItem(fWindowContextMenu, kOpenParentDir, !TargetModel()->IsRoot());
		EnableNamedMenuItem(fWindowContextMenu, kEmptyTrash, count > 0);
		EnableNamedMenuItem(fWindowContextMenu, B_SELECT_ALL, count > 0);

		EachMenuItem(fWindowContextMenu, true, MenuItemSetInvokeTimeout);
			// make sure menu items are set to timeout since we do
			// a synch Go and need to prevent deadlocks

		fWindowContextMenu->Go(global, true, false, mouseRect);
	}
	MenusEnded();
	fContextMenu = NULL;
}

void 
BContainerWindow::AddFileContextMenus(BMenu *menu)
{
	menu->AddItem(new BMenuItem("Open", new BMessage(kOpenSelection), 'O'));
	menu->AddItem(new BMenuItem("Get Info", new BMessage(kGetInfo), 'I'));
	menu->AddItem(new BMenuItem("Edit Name", new BMessage(kEditItem), 'E'));

	if (!IsTrash() && !InTrash() && !IsPrintersDir()) 
		menu->AddItem(new BMenuItem("Duplicate",
			new BMessage(kDuplicateSelection), 'D'));

	if (!IsTrash() && !InTrash()) {
		menu->AddItem(new BMenuItem("Move to Trash",
			new BMessage(kMoveToTrash), 'T'));
		// add separator for copy to/move to items (navigation items)
		menu->AddSeparatorItem();
	} else {
		menu->AddItem(new BMenuItem("Delete", new BMessage(kDelete), 0));
		menu->AddItem(new BMenuItem("Restore", new BMessage(kRestoreFromTrash), 0));
	}
		
	menu->AddSeparatorItem();
	menu->AddItem(new BMenuItem("Identify", new BMessage(kIdentifyEntry)));
	BMenu *addOnMenuItem = new BMenu(kAddOnsMenuName);
	addOnMenuItem->SetFont(be_plain_font);
	menu->AddItem(addOnMenuItem);
	menu->SetTargetForItems(PoseView());
}

void 
BContainerWindow::AddVolumeContextMenus(BMenu *menu)
{	
	menu->AddItem(new BMenuItem("Open", new BMessage(kOpenSelection), 'O'));
	menu->AddItem(new BMenuItem("Get Info", new BMessage(kGetInfo), 'I'));
	menu->AddItem(new BMenuItem("Edit Name", new BMessage(kEditItem), 'E'));

	menu->AddSeparatorItem();
	menu->AddItem(new MountMenu("Mount"));
	
	BMenuItem *item = new BMenuItem("Unmount", new BMessage(kUnmountVolume), 'U');
	item->SetEnabled(false);
	menu->AddItem(item);

	menu->AddSeparatorItem();
	menu->AddItem(new BMenu(kAddOnsMenuName));

	menu->SetTargetForItems(PoseView());
}


void 
BContainerWindow::AddWindowContextMenus(BMenu *menu)
{
	// create context sensitive menu for empty area of window
	// since we check view mode before display, this should be a radio
	// mode menu

	bool needSeparator = true;
	if (IsTrash()) 
		menu->AddItem(new BMenuItem("Empty Trash", new BMessage(kEmptyTrash)));
	else if (IsPrintersDir())
		menu->AddItem(new BMenuItem("Add Printer"B_UTF8_ELLIPSIS, new BMessage(kAddPrinter), 'N'));
	else if (InTrash())
		needSeparator = false;
	else
		menu->AddItem(new PositionPassingMenuItem("New Folder",
			new BMessage(kNewFolder), 'N'));

	if (needSeparator)
		menu->AddSeparatorItem();

	menu->AddItem(new BMenuItem("Icon View", new BMessage(kIconMode)));
	menu->AddItem(new BMenuItem("Mini Icon View", new BMessage(kMiniIconMode)));
	menu->AddItem(new BMenuItem("List View", new BMessage(kListMode)));
	menu->AddSeparatorItem();
	BMenuItem *resizeItem = new BMenuItem("Resize to Fit",
		new BMessage(kResizeToFit), 'Y');
	menu->AddItem(resizeItem);
	menu->AddItem(new BMenuItem("Clean Up", new BMessage(kCleanup), 'K'));
	menu->AddItem(new BMenuItem("Select"B_UTF8_ELLIPSIS, new BMessage(kShowSelectionWindow), 'A', B_SHIFT_KEY));
	menu->AddItem(new BMenuItem("Select All", new BMessage(B_SELECT_ALL), 'A'));
	if (!IsTrash()) 
		menu->AddItem(new BMenuItem("Open Parent", new BMessage(kOpenParentDir),
			B_UP_ARROW));

	BMenuItem *closeItem = new BMenuItem("Close", new BMessage(B_CLOSE_REQUESTED),
		'W');
	menu->AddItem(closeItem);
	menu->AddSeparatorItem();
	BMenu *addOnMenuItem = new BMenu(kAddOnsMenuName);
	addOnMenuItem->SetFont(be_plain_font);
	menu->AddItem(addOnMenuItem);

#if DEBUG
	menu->AddSeparatorItem();
	BMenuItem *testing = new BMenuItem("Test Icon Cache", new BMessage(kTestIconCache));
	menu->AddItem(testing);
#endif

	// target items as needed
	menu->SetTargetForItems(PoseView());
	closeItem->SetTarget(this);
	resizeItem->SetTarget(this);
}

void 
BContainerWindow::AddDropContextMenus(BMenu *menu)
{
	menu->AddItem(new BMenuItem("Create Link Here", new BMessage(kCreateLink)));
	menu->AddItem(new BMenuItem("Move Here", new BMessage(kMoveSelectionTo)));
	menu->AddItem(new BMenuItem("Copy Here", new BMessage(kCopySelectionTo)));
	menu->AddSeparatorItem();
	menu->AddItem(new BMenuItem("Cancel", new BMessage(kCancelButton)));
}


namespace BPrivate {

static void
StripShortcut(const Model *model, char *result, uint32 &shortcut)
{
	strcpy(result, model->Name());

	// check if there is a shortcut
	uint32 length = strlen(result);
	shortcut = '\0';
	if (result[length - 2] == '-') {
		shortcut = result[length - 1];
		result[length - 2] = '\0';
	}
}

static const Model *
MatchOne(const Model *model, void *castToName)
{
	char buffer[B_FILE_NAME_LENGTH];
	uint32 dummy;
	StripShortcut(model, buffer, dummy);

	if (strcmp(buffer, (const char *)castToName) == 0) 
		// found match, bail out
		return model;

	return 0;
}

int
CompareLabels(const BMenuItem *item1, const BMenuItem *item2)
{
	return strcasecmp(item1->Label(), item2->Label());
}

}

void
BContainerWindow::EachAddon(bool (*eachAddon)(const Model *, const char *,
	uint32 shortcut, void *context), void *passThru)
{
	BObjectList<Model> uniqueList(10, true);
	BPath path;
	bool bail = false;
	if (find_directory(B_BEOS_ADDONS_DIRECTORY, &path) == B_OK)
		bail = EachAddon(path, eachAddon, &uniqueList, passThru);
	
	if (!bail && find_directory(B_USER_ADDONS_DIRECTORY, &path) == B_OK)
		bail = EachAddon(path, eachAddon, &uniqueList, passThru);

	if (!bail && find_directory(B_COMMON_ADDONS_DIRECTORY, &path) == B_OK)
		EachAddon(path, eachAddon, &uniqueList, passThru);
}

bool
BContainerWindow::EachAddon(BPath &path, bool (*eachAddon)(const Model *,
	const char *, uint32 , void *), BObjectList <Model> *uniqueList,
	void *params)
{
	path.Append("Tracker");

	BDirectory dir;
	BEntry entry;

	if (dir.SetTo(path.Path()) != B_OK)
		return false;

	dir.Rewind();
	while (dir.GetNextEntry(&entry) == B_OK) {
	
		if (entry.IsSymLink()) {
			// resolve symlinks if needed
			entry_ref ref;
			entry.GetRef(&ref);
			entry.SetTo(&ref, true);
		}

		Model *model = new Model(&entry);
		if (model->InitCheck() != B_OK || !model->IsExecutable()) {
			delete model;
			continue;
		}

		char name[B_FILE_NAME_LENGTH];
		uint32 key;
		StripShortcut(model, name, key);

		// do a uniqueness check
		if (uniqueList->EachElement(MatchOne, name)) {
			// found one already in the list
			delete model;
			continue;
		}
		uniqueList->AddItem(model);
	
		if ((eachAddon)(model, name, key, params))
			return true;
	}
	return false;
}

struct AddOneAddonParams {
	BObjectList<BMenuItem> *addOnList;
};

static bool
AddOneAddon(const Model *model, const char *name, uint32 shortcut, void *context)
{
	AddOneAddonParams *params = (AddOneAddonParams *)context;

	BMessage *message = new BMessage(kLoadAddOn);
	message->AddRef("refs", model->EntryRef());

	params->addOnList->AddItem(new ModelMenuItem(model, name, message,
		(char)shortcut, B_OPTION_KEY));
	
	return false;
}

void
BContainerWindow::BuildAddOnMenu(BMenu *menu)
{
	BMenuItem *item = menu->FindItem(kAddOnsMenuName);
	if (!item)
		return;

	menu = item->Submenu();
	if (!menu)
		return;

	menu->SetFont(be_plain_font);

	// found the addons menu, empty it first
	while ((item = menu->RemoveItem(0L)) != NULL)
		delete item;

	BObjectList<BMenuItem> addOnList;
	
	AddOneAddonParams params;
	params.addOnList = &addOnList;
	
	EachAddon(AddOneAddon, &params);
	
	addOnList.SortItems(CompareLabels);

	int32 count = addOnList.CountItems();
	for (int32 index = 0; index < count; index++)
		menu->AddItem(addOnList.ItemAt(index));

	menu->SetTargetForItems(this);
}

static int32
AddOnThread(BMessage *refsMessage, const entry_ref &addonRef,
	const entry_ref &dirRef)
{
	auto_ptr<BMessage> refsMessagePtr(refsMessage);

	BEntry entry(&addonRef);
	BPath path;
	status_t result = entry.InitCheck();
	if (result == B_OK)
		result = entry.GetPath(&path);

	if (result == B_OK) {
		image_id addonImage = load_add_on(path.Path());
		if (addonImage >= 0) {
	
			void (*processRefs)(entry_ref, BMessage *, void *);
			result = get_image_symbol(addonImage, "process_refs", 2, (void **)&processRefs);

#ifndef __INTEL__
			if (result < 0) {
				PRINT(("trying old legacy ppc signature\n"));
				// try old-style addon signature
				result = get_image_symbol(addonImage,
					"process_refs__F9entry_refP8BMessagePv", 2, (void **)&processRefs);
			}
#endif

			if (result >= 0) {

				// call add-on code
				(*processRefs)(dirRef, refsMessagePtr.get(), 0);
				
				unload_add_on(addonImage);
				return B_OK;
			} else 
				PRINT(("couldn't find process_refs\n"));

			unload_add_on(addonImage);
		}
	}

	char buffer[1024];
	sprintf(buffer, "Error %s loading Add-On %s.", strerror(result), addonRef.name);

	BAlert *alert = new BAlert("", buffer, "Cancel", 0, 0,
		B_WIDTH_AS_USUAL, B_WARNING_ALERT);
	alert->SetShortcut(0, B_ESCAPE);
	alert->Go();
	
	return result;
}

void
BContainerWindow::LoadAddOn(BMessage *message)
{
	UpdateIfNeeded();

	entry_ref addonRef;
	status_t result = message->FindRef("refs", &addonRef);
	if (result != B_OK) {
		char buffer[1024];
		sprintf(buffer, "Error %s loading Add-On %s.", strerror(result), addonRef.name);
		(new BAlert("", buffer, "Cancel", 0, 0,
			B_WIDTH_AS_USUAL, B_WARNING_ALERT))->Go();
		return;
	}

	// add selected refs to message
	BMessage *refs = new BMessage(B_REFS_RECEIVED);

	BObjectList<BPose> *list = PoseView()->SelectionList();

	int32 index = 0;
	BPose *pose;
	while ((pose = list->ItemAt(index++)) != NULL)
		refs->AddRef("refs", pose->TargetModel()->EntryRef());
	
	refs->AddMessenger("TrackerViewToken", BMessenger(PoseView()));
	
	LaunchInNewThread("Add-on", B_NORMAL_PRIORITY, &AddOnThread, refs, addonRef, 
		*TargetModel()->EntryRef());
}

BMenuItem * 
BContainerWindow::NewAttributeMenuItem (const char *label, const char *attrName, 
	int32 attrType, float attrWidth, int32 attrAlign, bool attrEditable, bool attrStatField)
{
	BMessage *message = new BMessage (kAttributeItem);
	message->AddString ("attr_name", attrName);
	message->AddInt32 ("attr_type", attrType);
	int32 attrHash = (int32)AttrHashString(attrName, (uint32)attrType);	
	message->AddInt32 ("attr_hash", attrHash);
	message->AddFloat ("attr_width", attrWidth);
	message->AddInt32 ("attr_align", attrAlign);
	message->AddBool ("attr_editable", attrEditable);
	message->AddBool ("attr_statfield", attrStatField);
	BMenuItem *menuItem = new BMenuItem(label, message);
	menuItem->SetTarget (PoseView());
	return menuItem;
}

void
BContainerWindow::NewAttributeMenu(BMenu *menu)
{
	ASSERT(PoseView());

	menu->AddItem(NewAttributeMenuItem ("Name", kAttrStatName, B_STRING_TYPE,
		145, B_ALIGN_LEFT, true, true));

	menu->AddItem(NewAttributeMenuItem ("Size", kAttrStatSize, B_OFF_T_TYPE,
		80, B_ALIGN_RIGHT, false, true));

	menu->AddItem(NewAttributeMenuItem ("Modified", kAttrStatModified, B_TIME_TYPE,
		150, B_ALIGN_LEFT, false, true));

	menu->AddItem(NewAttributeMenuItem ("Created", kAttrStatCreated, B_TIME_TYPE,
		150, B_ALIGN_LEFT, false, true));

	menu->AddItem(NewAttributeMenuItem ("Kind", kAttrMIMEType, B_MIME_STRING_TYPE,
		145, B_ALIGN_LEFT, false, false));

	menu->AddItem(NewAttributeMenuItem ("Path", kAttrPath, B_STRING_TYPE,
		225, B_ALIGN_LEFT, false, false));

#ifdef OWNER_GROUP_ATTRIBUTES
	menu->AddItem(NewAttributeMenuItem ("Owner", kAttrStatOwner, B_STRING_TYPE,
		60, B_ALIGN_LEFT, false, true));

	menu->AddItem(NewAttributeMenuItem ("Group", kAttrStatGroup, B_STRING_TYPE,
		60, B_ALIGN_LEFT, false, true));
#endif

	menu->AddItem(NewAttributeMenuItem ("Permissions", kAttrStatMode, B_STRING_TYPE,
		80, B_ALIGN_LEFT, false, true));

	if (IsTrash() || InTrash())
		menu->AddItem(NewAttributeMenuItem ("Original name", kAttrOriginalPath, B_STRING_TYPE,
			225, B_ALIGN_LEFT, false, false));
}

void
BContainerWindow::ShowAttributeMenu()
{
	ASSERT(fAttrMenu);
	fMenuBar->AddItem(fAttrMenu);
}

void
BContainerWindow::HideAttributeMenu()
{
	ASSERT(fAttrMenu);
	fMenuBar->RemoveItem(fAttrMenu);
}

void 
BContainerWindow::MarkAttributeMenu()
{
	MarkAttributeMenu(fAttrMenu);
}

void
BContainerWindow::MarkAttributeMenu(BMenu *menu)
{
	if (!menu)
		return;

	int32 count = menu->CountItems();
	for (int32 index = 0; index < count; index++) {
		BMenuItem *item = menu->ItemAt(index);
		int32 attrHash;
		if (item->Message())
			if (item->Message()->FindInt32("attr_hash", &attrHash) == B_OK)
				item->SetMarked(PoseView()->ColumnFor((uint32)attrHash) != 0);
			else 
				item->SetMarked(false);

		BMenu *submenu = item->Submenu();
		if (submenu) {
			int32 count2 = submenu->CountItems();
			for (int32 subindex = 0; subindex < count2; subindex++) {
				item = submenu->ItemAt(subindex);
				if (item->Message())
					if (item->Message()->FindInt32("attr_hash", &attrHash) == B_OK) 
						item->SetMarked(PoseView()->ColumnFor((uint32)attrHash) != 0);
					else
						item->SetMarked(false);
			}
		}
	}
}

void 
BContainerWindow::AddMimeTypesToMenu()
{
	AddMimeTypesToMenu(fAttrMenu);
}

void
BContainerWindow::AddMimeTypesToMenu(BMenu *menu)
{
	if (!menu)
		return;

	// find start of mime types in menu
	int32 count = menu->CountItems();
	int32 start;

	for (start = 0; start < count; start++) 
		if (menu->ItemAt(start)->Submenu())
			break;

	if (menu->FindItem((uint32)0) == NULL)
		menu->AddSeparatorItem();

 	int32 removeIndex = count - 1;
 	// Remove old mime menu:
 	while (menu->ItemAt(removeIndex)->Submenu() != NULL) {
		delete menu->RemoveItem(removeIndex);
 		removeIndex--;
 	}

	int32 typeCount = PoseView()->CountMimeTypes();

	for (int32 index = 0; index < typeCount; index++) {

		bool shouldAdd = true;
		const char *signature = PoseView()->MimeTypeAt(index);

		for (int32 subindex = start; subindex < count; subindex++) {
			BMenuItem *item = menu->ItemAt(subindex);
			if (!item)
				continue;
			BMessage *message = item->Message();
			if (!message)
				continue;
			const char *str;
			if (message->FindString("mimetype", &str) == B_OK
				&& strcmp(signature, str) == 0) {
				shouldAdd = false;
				break;
			}
		}

		if (shouldAdd) {
			BMessage attr_msg;
			char desc[B_MIME_TYPE_LENGTH];
			const char *nameToAdd = signature;

			BMimeType mimetype(signature);

			if (!mimetype.IsInstalled()) 
				continue;

			// only add things to menu which have "user-visible" data
			if (mimetype.GetAttrInfo(&attr_msg) != B_OK) 
				continue;

			if (mimetype.GetShortDescription(desc) == B_OK && desc[0])
				nameToAdd = desc;

					// go through each field in meta mime and add it to a menu
			BMenu *localMenu = 0;
			int32 index = -1;
			const char *str;

			while (attr_msg.FindString("attr:public_name", ++index, &str) == B_OK) {
				if (!attr_msg.FindBool("attr:viewable", index))
					// don't add if attribute not viewable
					continue;
					
				int32 type;
				int32 align;
				int32 width;
				bool editable;
				
				const char *attrName;
				
				if (attr_msg.FindString("attr:name", index, &attrName) != B_OK)
					continue;

				if (attr_msg.FindInt32("attr:type", index, &type) != B_OK)
					continue;

				if (attr_msg.FindBool("attr:editable", index, &editable) != B_OK)
					continue;

				if (attr_msg.FindInt32("attr:width", index, &width) != B_OK)
					continue;

				if (attr_msg.FindInt32("attr:alignment", index, &align) != B_OK)
					continue;

				if (!localMenu) {
					// do a lazy allocation of the menu
					localMenu = new BMenu(nameToAdd);
					BFont font;
					menu->GetFont(&font);
					localMenu->SetFont(&font);
				}
				localMenu->AddItem(NewAttributeMenuItem (str, attrName, type,
					width, align, editable, false));
			}
			if (localMenu) {
				BMessage *message = new BMessage(kMIMETypeItem);
				message->AddString("mimetype", signature);
				menu->AddItem(new IconMenuItem(localMenu, message, signature, B_MINI_ICON));
			}
		}
	}

	// remove separator if it's the only item in menu
	BMenuItem *item = menu->FindItem((uint32)0);
	if (item && item == menu->ItemAt(menu->CountItems() - 1)) {
		menu->RemoveItem(item);
		delete item;
	}

	MarkAttributeMenu(menu);
}

BHandler *
BContainerWindow::ResolveSpecifier(BMessage *message, int32 index,
	BMessage *specifier, int32 form, const char	*property)
{
	if (strcmp(property, "Poses") == 0) {
//		PRINT(("BContainerWindow::ResolveSpecifier %s\n", property));
		message->PopSpecifier();
		return PoseView();
	}

	return _inherited::ResolveSpecifier(message, index, specifier,
		form, property);
}

BackgroundView::BackgroundView(BRect frame)
	:	BView(frame, "", B_FOLLOW_ALL,
			B_FRAME_EVENTS | B_WILL_DRAW | B_PULSE_NEEDED)
{
}

void
BackgroundView::AttachedToWindow()
{
	SetViewColor(227, 227, 227);
}

void
BackgroundView::FrameResized(float, float)
{
	Invalidate();
}

void 
BackgroundView::PoseViewFocused(bool)
{
	Invalidate();
}

void 
BackgroundView::WindowActivated(bool)
{
	Invalidate();
}

void
BackgroundView::Draw(BRect)
{
	BContainerWindow *window = dynamic_cast<BContainerWindow *>(Window());
	if (!window)
		return;

	BRect frame(window->PoseView()->Frame());

	frame.InsetBy(-1, -1);
	frame.top -= kTitleViewHeight;
	frame.bottom += B_H_SCROLL_BAR_HEIGHT;
	frame.right += B_V_SCROLL_BAR_WIDTH;
	SetHighColor(100, 100, 100);
	StrokeRect(frame);

	// draw the pose view focus  
	if (window->IsActive() && window->PoseView()->IsFocus()) {
		frame.InsetBy(-2, -2);
		SetHighColor(keyboard_navigation_color());
		StrokeRect(frame);
	}
}

void 
BackgroundView::Pulse()
{
	BContainerWindow *window = dynamic_cast<BContainerWindow *>(Window());
	if (window) 
		window->PulseTaskLoop();
}

void 
BContainerWindow::PulseTaskLoop()
{
	if (fTaskLoop)
		fTaskLoop->PulseMe();
}

PiggybackTaskLoop *
BContainerWindow::DelayedTaskLoop()
{
	if (!fTaskLoop)
		fTaskLoop = new PiggybackTaskLoop;

	return fTaskLoop;
}


static bool
NodeHasSavedState(const BNode *node)
{
	attr_info info;
	return node->GetAttrInfo(kAttrWindowFrame, &info) == B_OK;
}

bool 
BContainerWindow::NeedsDefaultStateSetup()
{
	if (!TargetModel())
		return false;

	if (TargetModel()->IsRoot())
		// don't try to set up anything if we are root
		return false;
		
	WindowStateNodeOpener opener(this, false);
	if (!opener.StreamNode())
		// can't read state, give up
		return false;
	
	return !NodeHasSavedState(opener.Node());
}

struct StaggerOneParams {
	bool rectFromParent;
};

static bool
OffsetFrameOne(const char *DEBUG_ONLY(name), uint32 , off_t , void *castToRect,
	void *castToParams)
{
	ASSERT(strcmp(name, kAttrWindowFrame) == 0);
	StaggerOneParams *params = (StaggerOneParams *)castToParams;

	if (!params->rectFromParent)
		return false;
		
	if (!castToRect)
		return false;
	
	((BRect *)castToRect)->OffsetBy(kWindowStaggerBy, kWindowStaggerBy);
	return true;
}


bool
BContainerWindow::DefaultStateSourceNode(const char *name, BNode *result,
	bool createNew, bool createFolder)
{
//	PRINT(("looking for default state in tracker settings dir\n"));
	BPath settingsPath;
	if (TFSContext::GetTrackerSettingsDir(settingsPath) != B_OK)
		return false;

	BDirectory dir(settingsPath.Path());

	BPath path(settingsPath);
	path.Append(name);
	if (!BEntry(path.Path()).Exists()) {

		if (!createNew)
			return false;
	
		BPath tmpPath(settingsPath);
		for (;;) {
			// deal with several levels of folders
			const char *nextSlash = strchr(name, '/');
			if (!nextSlash)
				break;
			
			BString tmp;
			tmp.SetTo(name, nextSlash - name);
			tmpPath.Append(tmp.String());


			mkdir(tmpPath.Path(), 0777);
	
			name = nextSlash + 1;
			if (!name[0]) {
				// can't deal with a slash at end

				return false;
			}
		}
		
		if (createFolder) {
			if (mkdir(path.Path(), 0777) < 0)
				return false;
		} else {
			BFile file;
			if (dir.CreateFile(name, &file) != B_OK)
				return false;
		}
	}
	
// 	PRINT(("using default state from %s\n", path.Path()));
	result->SetTo(path.Path());
	return result->InitCheck() == B_OK;
}

void 
BContainerWindow::SetUpDefaultState()
{
	BNode defaultingNode;
		// this is where we'll ulitimately get the state from
	bool gotDefaultingNode = 0;
	bool shouldStagger = false;
	
	ASSERT(TargetModel());

	PRINT(("folder %s does not have any saved state\n", TargetModel()->Name()));
	
	WindowStateNodeOpener opener(this, true);
		// this is our destination node, whatever it is for this window
	if (!opener.StreamNode())
		return;
	
	if (!TargetModel()->IsRoot()) {
		BDirectory desktop;
		TFSContext::GetDesktopDir(desktop, TargetModel()->EntryRef()->device);

		// try copying state from our parent directory, unless it is the desktop folder
		BEntry entry(TargetModel()->EntryRef());
		BDirectory parent;
		if (entry.GetParent(&parent) == B_OK && parent != desktop) {
			PRINT(("looking at parent for state\n"));
			if (NodeHasSavedState(&parent)) {
				PRINT(("got state from parent\n"));
				defaultingNode = parent;
				gotDefaultingNode = true;
				// when getting state from parent, stagger the window
				shouldStagger = true;
			}
		}
	}

	if (!gotDefaultingNode
		// parent didn't have any state, use the template directory from
		// tracker settings folder for what our state should be
		// For simplicity we are not picking up the most recent
		// changes that didn't get committed if home is still open in
		// a window, that's probably not a problem; would be OK if state got committed
		// after every change
		&& !DefaultStateSourceNode(kDefaultFolderTemplate, &defaultingNode, true))
		return;

	// copy over the attributes
	
	// set up a filter of the attributes we want copied
	const char *allowAttrs[] = {
		kAttrWindowFrame,
		kAttrWindowWorkspace,
		kAttrViewState,
		kAttrViewStateForeign,
		kAttrColumns,
		kAttrColumnsForeign,
		0
	};
	
	// copy over attributes that apply; transform them properly, stripping
	// parts that do not apply, adding a window stagger, etc.

	StaggerOneParams params;
	params.rectFromParent = shouldStagger;
	SelectiveAttributeTransformer frameOffsetter(kAttrWindowFrame, OffsetFrameOne, &params);
	SelectiveAttributeTransformer scrollOriginCleaner(kAttrViewState,
		ClearViewOriginOne, &params);
	
	// do it
	AttributeStreamMemoryNode memoryNode;
	NamesToAcceptAttrFilter filter(allowAttrs);
	AttributeStreamFileNode fileNode(&defaultingNode);

	*opener.StreamNode() << scrollOriginCleaner << frameOffsetter
		<< memoryNode << filter << fileNode;
}

void 
BContainerWindow::RestoreWindowState(AttributeStreamNode *node)
{
	if (!node || dynamic_cast<BDeskWindow *>(this))
		// don't restore any window state if we are a desktop window
		return;
		
	const char *rectAttributeName;
	const char *workspaceAttributeName;
	if (TargetModel()->IsRoot()) {
		rectAttributeName = kAttrDisksFrame;
		workspaceAttributeName = kAttrDisksWorkspace;
	} else {
		rectAttributeName = kAttrWindowFrame;
		workspaceAttributeName = kAttrWindowWorkspace;
	}
	
	BRect frame(Frame());
	if (node->Read(rectAttributeName, 0, B_RECT_TYPE, sizeof(BRect), &frame)
		== sizeof(BRect)) {	
		MoveTo(frame.LeftTop());
		ResizeTo(frame.Width(), frame.Height());
	} else
		fNewWindRect.OffsetBy(kWindowStaggerBy, kWindowStaggerBy);

	uint32 workspace;
	off_t size = node->Read(workspaceAttributeName, 0, B_INT32_TYPE, sizeof(uint32),
		&workspace);

	if (size == sizeof(uint32) && (fContainerWindowFlags & kRestoreWorkspace)) 
		SetWorkspaces(workspace);
}

void 
BContainerWindow::SaveWindowState(AttributeStreamNode *node)
{
	ASSERT(node);
	const char *rectAttributeName;
	const char *workspaceAttributeName;
	if (TargetModel() && TargetModel()->IsRoot()) {
		rectAttributeName = kAttrDisksFrame;
		workspaceAttributeName = kAttrDisksWorkspace;
	} else {
		rectAttributeName = kAttrWindowFrame;
		workspaceAttributeName = kAttrWindowWorkspace;
	}
	
	// node is null if it already got deleted
	BRect frame(Frame());
	node->Write(rectAttributeName, 0, B_RECT_TYPE, sizeof(BRect), &frame);
	uint32 workspaces = Workspaces();
	node->Write(workspaceAttributeName, 0, B_INT32_TYPE, sizeof(uint32),
		&workspaces);
}

status_t
BContainerWindow::DragStart(const BMessage *incoming)
{
	if (!incoming)
		return B_ERROR;	

	//	if already dragging, or
	//	if all the refs match
	if (Dragging() && SpringLoadedFolderCompareMessages(incoming, fDragMessage))
		return B_OK;

	//	cache the current drag message
	//	build a list of the mimetypes in the message
	SpringLoadedFolderCacheDragData(incoming, &fDragMessage, &fCachedTypesList);
	
	fWaitingForRefs = true;
	
	return B_OK;
}

void
BContainerWindow::DragStop()
{
	delete fDragMessage;
	fDragMessage = NULL;

	delete fCachedTypesList;
	fCachedTypesList = NULL;
	
	fWaitingForRefs = false;
}

void
BContainerWindow::ShowSelectionWindow()
{
	if (fSelectionWindow == NULL) {
		fSelectionWindow = new SelectionWindow(this);	
		fSelectionWindow->Show();
	} else if (fSelectionWindow->Lock()) {
		if (fSelectionWindow->IsHidden()) {
			fSelectionWindow->MoveCloseToMouse();
			fSelectionWindow->Show();
		}
		fSelectionWindow->Unlock();
	}
}

WindowStateNodeOpener::WindowStateNodeOpener(BContainerWindow *window, bool forWriting)
	:	fModelOpener(NULL),
		fNode(NULL),
		fStreamNode(NULL)
{
	if (window->TargetModel() && window->TargetModel()->IsRoot()) {
		BDirectory dir;
		if (TFSContext::GetBootDesktopDir(dir) == B_OK) {
			fNode = new BDirectory(dir);
			fStreamNode = new AttributeStreamFileNode(fNode);
		}
	} else if (window->TargetModel()){
		fModelOpener = new ModelNodeLazyOpener(window->TargetModel(), forWriting, false);
		if (fModelOpener->IsOpen(forWriting))
			fStreamNode = new AttributeStreamFileNode(fModelOpener->TargetModel()->Node());
	}
}

WindowStateNodeOpener::~WindowStateNodeOpener()
{
	delete fModelOpener;
	delete fNode;
	delete fStreamNode;
}

void 
WindowStateNodeOpener::SetTo(const BDirectory *node)
{
	delete fModelOpener;
	delete fNode;
	delete fStreamNode;
	
	fModelOpener = NULL;
	fNode = new BDirectory(*node);
	fStreamNode = new AttributeStreamFileNode(fNode);
}

void 
WindowStateNodeOpener::SetTo(const BEntry *entry, bool forWriting)
{
	delete fModelOpener;
	delete fNode;
	delete fStreamNode;
	
	fModelOpener = NULL;
	fNode = new BFile(entry, (uint32)(forWriting ? O_RDWR : O_RDONLY));
	fStreamNode = new AttributeStreamFileNode(fNode);
}

void 
WindowStateNodeOpener::SetTo(Model *model, bool forWriting)
{
	delete fModelOpener;
	delete fNode;
	delete fStreamNode;
	
	fNode = NULL;
	fStreamNode = NULL;
	fModelOpener = new ModelNodeLazyOpener(model, forWriting, false);
	if (fModelOpener->IsOpen(forWriting))
		fStreamNode = new AttributeStreamFileNode(fModelOpener->TargetModel()->Node());
}


AttributeStreamNode *
WindowStateNodeOpener::StreamNode() const
{
	return fStreamNode;
}

BNode *
WindowStateNodeOpener::Node() const
{
	if (!fStreamNode)
		return NULL;
	
	if (fNode)
		return fNode;

	return fModelOpener->TargetModel()->Node();
}
