
#include "FSDialogWindow.h"

#include <Debug.h>
#include <Screen.h>
#include <Autolock.h>
#include <Alert.h>

#include <memory>

namespace BPrivate {

using namespace fs;

//TODO:
// do something about resizing
// take care of the possibility of an infinite loop with default answers

static const float			kColoredRectWidth			= 35;
static const float			kMinDialogWindowHeight		= 100;
#define 					kColoredRectColor			tint_color(ViewColor(), B_DARKEN_1_TINT)

static const float			kTextViewHeight				= 90;	// workaround for the BTextView::TextHeight() bug

//#define PRINT_ERROR_CODE 1		// print the error code number in the error string
#define PRINT_OPERATION_STACK 1		// print the whole operation stack in braces

void
FSDialogWindow :: initialize() {

	mReply = TFSContext::kInvalidCommand;
	*mDefaultAnswerPtr = false;
	mTextControlText = 0;
	
	Minimize(true);
	AddChild(&mMainView);
	SetSizeLimits(20, 1000, 20, 1000);

	AddShortcut(B_ESCAPE, 0, new BMessage(kCancel));	// XXX useless?!
	AddShortcut(B_SPACE, B_COMMAND_KEY, new BMessage(kDontAskKey));

// XXX HERE: without at least one Show() the BAutolocks in Add() and Remove() will end up in a deadlock?! BeOS bug.
	Show();
	Hide();
}


FSDialogWindow :: FSDialogWindow(TFSContext::command *replyptr, bool *defptr, FSDialogWindow **iptr,
								 char *source_name, char *target_name, TFSContext &context,
								 TFSContext::interaction iact) : inherited(BRect(0, 0, 100, 100), "BuG!",
											B_TITLED_WINDOW_LOOK, B_NORMAL_WINDOW_FEEL, // B_FLOATING_APP_WINDOW_FEEL,
											B_NOT_RESIZABLE | B_NOT_ZOOMABLE, B_ALL_WORKSPACES),
									mContainer(iptr),
									mSourceNewName(source_name),
									mTargetNewName(target_name),
									mReplyPtr(replyptr),
									mDefaultAnswerPtr(defptr),
									mContext(context),
									mMainView(),
									mType(kInteraction) {
	initialize();

	if (mTargetNewName  &&  mSourceNewName)
		strncpy(mTargetNewName, mSourceNewName, B_FILE_NAME_LENGTH - 1);

	SetType(kInteraction);
	SetContext(context, iact);

	BAutolock l(this);

	Pack();
}

FSDialogWindow :: FSDialogWindow(TFSContext::command *replyptr, bool *defptr, FSDialogWindow **iptr, TFSContext &context, status_t rc) :
								 inherited(BRect(0, 0, 100, 100), "BuG!",
											B_TITLED_WINDOW_LOOK, B_NORMAL_WINDOW_FEEL, // B_FLOATING_APP_WINDOW_FEEL,
											B_NOT_RESIZABLE | B_NOT_ZOOMABLE, B_ALL_WORKSPACES),
									mContainer(iptr),
									mSourceNewName(0),
									mTargetNewName(0),
									mReplyPtr(replyptr),
									mDefaultAnswerPtr(defptr),
									mContext(context),
									mMainView(),
									mType(kError) {
	initialize();

	SetType(kError);
	SetContext(context, rc);

	BAutolock l(this);

	Pack();
}

FSDialogWindow :: ~FSDialogWindow() {
	mMainView.RemoveSelf();
	
	*mContainer = 0;
}

void
FSDialogWindow :: SetType(type t) {
	mType = t;
	
	if (mType == kError) {
		SetTitle("Tracker Error");
	} else if (mType == kInteraction) {
		SetTitle("Tracker Question");
	} else {
		TRESPASS();
	}
}


void
FSDialogWindow :: SetContext(TFSContext &context, TFSContext::interaction iact) {
	BAutolock l(this);
	
	ASSERT(mMainView.mDialogView == 0);
	mMainView.SetDialogView(new TFSContextDialogView(context, iact));
	mMainView.SetRetryButtonEnabled(mContext.IsAnswerPossible(TFSContext::fRetryOperation));
}

void
FSDialogWindow :: SetContext(TFSContext &context, status_t rc) {
	BAutolock l(this);

	ASSERT(mMainView.mDialogView == 0);
	mMainView.SetDialogView(new TFSContextDialogView(context, rc));
	mMainView.SetRetryButtonEnabled(mContext.IsAnswerPossible(TFSContext::fRetryOperation));
}

void
FSDialogWindow :: Go() {
	Minimize(false);
	Show();
}

void
FSDialogWindow :: MessageReceived(BMessage *msg) {

	switch (msg -> what) {
		case kPack:
			Pack();
			break;

		case kRetry:
			mReply = TFSContext::kRetryOperation;
			Quit();
			break;
			
		case kTextControl: {
			BControl *ptr;
			if (msg -> FindPointer("source", reinterpret_cast<void **>(&ptr)) == B_OK) {
				BTextControl *source = assert_cast<BTextControl *>(ptr);
				mTextControlText = source -> Text();
			}
			break;
		}
		case kDontAskKey:
			mMainView.SwitchDontAsk();
			break;
			
		case kCancel:
			Cancel();
			Quit();
			break;

		case kDefaultAnswerCB: {
			BCheckBox *box;
			if (msg -> FindPointer("source", reinterpret_cast<void **>(&box)) == B_OK)
				*mDefaultAnswerPtr = (box -> Value() == B_CONTROL_ON);
			break;
		}
		case kReplyInvoked:
		case kReplySelected: {
			BListView *list;
			if (msg -> FindPointer("source", reinterpret_cast<void **>(&list)) == B_OK
							&&  list -> CurrentSelection() >= 0) {
							
				CustomItem *item = assert_cast<CustomItem *>(list -> ItemAt(list -> CurrentSelection()));
				mReply = item -> Command();
			}

			if (msg -> what == kReplySelected)
				break;
//			else Fall trough to continue
		}
		case kContinue:
			if (HasReply())
				Quit();
			break;

		default:
			inherited::MessageReceived(msg);
	}
}


void
FSDialogWindow :: ResizeToPreferred() {
	DisableUpdates();

	mMainView.MoveTo(0, 0);
	mMainView.ResizeToPreferred();
	BRect rect = mMainView.Bounds();
	ResizeTo(rect.right - 1, rect.bottom - 1);

	EnableUpdates();
}

void
FSDialogWindow :: Pack() {

	ResizeToPreferred();
	
	BRect rect = Bounds();
	MoveTo(BAlert::AlertPosition(rect.right, rect.bottom));
}


void
FSDialogWindow :: Quit() {
	
	*mReplyPtr = mReply;
	*mContainer = 0;

	if (mReply == TFSContext::kSuppliedNewNameForSource) {
		ASSERT(mTextControlText != 0);
		strncpy(mSourceNewName, mTextControlText, B_FILE_NAME_LENGTH - 1);
	} else if (mReply == TFSContext::kSuppliedNewNameForTarget) {
		ASSERT(mTextControlText != 0);
		strncpy(mTargetNewName, mTextControlText, B_FILE_NAME_LENGTH - 1);
	}

	mContext.HardResume();
	
	inherited::Quit();
}

void
FSDialogWindow :: Cancel() {
	mReply = TFSContext::kCancel;
	mContext.Cancel();
}

//void
//FSDialogWindow :: DispatchMessage(BMessage *msg, BHandler *handler) {
//	if (msg -> what == B_KEY_DOWN) {
//		int32 key = msg -> FindInt32("key");
//		if (key == B_ESCAPE)
//			Quit();
//	}
//
//	inherited::DispatchMessage(msg, handler);
//}

bool
FSDialogWindow :: QuitRequested() {
	Cancel();
	return true;
}






FSDialogWindow::MainView :: MainView() : inherited(BRect(), "MainView", B_FOLLOW_NONE, B_WILL_DRAW),
										mContinueButton(BRect(), "ContinueButton", "Continue", new BMessage(kContinue),
											B_FOLLOW_NONE, B_WILL_DRAW | B_NAVIGABLE | B_NAVIGABLE_JUMP),
										mCancelButton(BRect(), "CancelButton", "Stop", new BMessage(kCancel)),
										mRetryButton(BRect(), "RetryButton", "Retry", new BMessage(kRetry)),
										mDialogView(0) {

	SetViewColor(ui_color(B_PANEL_BACKGROUND_COLOR));

	AddChild(&mCancelButton);
	AddChild(&mRetryButton);
	AddChild(&mContinueButton);
}

FSDialogWindow::MainView :: ~MainView() {
	mContinueButton.RemoveSelf();
	mCancelButton.RemoveSelf();
	mRetryButton.RemoveSelf();
}

void
FSDialogWindow::MainView :: AllAttached() {
	inherited::AllAttached();

	mContinueButton.ResizeToPreferred();
	mContinueButton.SetEnabled(false);
	mContinueButton.MakeDefault(true);
	mCancelButton.ResizeToPreferred();
	mRetryButton.ResizeToPreferred();
}


void
FSDialogWindow::MainView :: Draw(BRect iRect) {
	
	BRect rect = Bounds();
	
	rect.right = kColoredRectWidth;
	SetHighColor(kColoredRectColor);
	if (iRect.Intersects(rect))
		FillRect(rect);
}

void
FSDialogWindow::MainView :: SetDialogView(DialogView *view) {

	ASSERT(mDialogView == 0);
	ASSERT(Looper() -> IsLocked());
	
	mDialogView = view;
	AddChild(view);
//	assert_cast<FSDialogWindow *>(Window()) -> Pack(); // probably useless here...
}

void
FSDialogWindow::MainView :: ResizeToPreferred() {
	BRect rect(0, 0, 0, 0);

	float tw, th = 0;
	
	tw = mContinueButton.Bounds().right + mCancelButton.Bounds().right + mRetryButton.Bounds().right;
	tw *= 1.3;
	tw += kColoredRectWidth;
	
	if (mDialogView) {
		mDialogView -> MoveTo(kColoredRectWidth + 1, 0);
		mDialogView -> ResizeTo(tw, 0);
		mDialogView -> ResizeToPreferred();
		mDialogView -> SetViewColor(ui_color(B_PANEL_BACKGROUND_COLOR));
		rect = mDialogView -> Frame();
	}

	if (tw < rect.right)
		tw = rect.right;
	th += rect.bottom;
	
	mContinueButton.ResizeToPreferred();

	BRect buttonrect = mContinueButton.Bounds();
	
	th += buttonrect.bottom * 1.2;
	buttonrect.right *= 3;				// at least 3 times the button width
	if (tw < buttonrect.right)
		tw = buttonrect.right;

	rect.InsetBy(rect.Width() * 0.07, 0);

	mCancelButton.MoveTo(	rect.left,
							rect.bottom + mCancelButton.Bounds().bottom * 0.1);
	
	mContinueButton.MakeDefault(false);
	buttonrect = mContinueButton.Bounds();
	mContinueButton.MoveTo(	rect.right - buttonrect.right,
							rect.bottom + buttonrect.bottom * 0.1);
	mContinueButton.MakeDefault(true);

	buttonrect = mRetryButton.Bounds();
	mRetryButton.MoveTo(mContinueButton.Frame().left - buttonrect.right * 1.2,
						rect.bottom + buttonrect.bottom * 0.1);

	ResizeTo(tw, th);
}



FSDialogWindow::TFSContextDialogView :: TFSContextDialogView(TFSContext &context, TFSContext::interaction iact) :
											inherited(),
											mContext(context), mType(kInteraction),
											mListView(),
											mScrollView(mListView),
											mDefaultAnswerCB(),
											mTextView(),
											mTextControl() {
	mInteractionCode = iact;
	
	initialize();
}

FSDialogWindow::TFSContextDialogView :: TFSContextDialogView(TFSContext &context, status_t rc) :
											inherited(),
											mContext(context), mType(kError),
											mListView(),
											mScrollView(mListView),
											mDefaultAnswerCB(),
											mTextView(),
											mTextControl() {
	mErrorCode = rc;

	initialize();
}

void
FSDialogWindow::TFSContextDialogView :: initialize() {
	AddChild(&mTextView);
	AddChild(&mScrollView);
	AddChild(&mTextControl);
	AddChild(&mDefaultAnswerCB);
	
	mTextControl.Hide();

	mTextView.MakeEditable(false);
	mTextView.MakeSelectable(false);
	mTextView.SetStylable(true);

	if (mType == kInteraction  &&  mContext.IsExtendedInteraction(mInteractionCode) == false)
		mDefaultAnswerCB.Hide();

	SetViewColor(ui_color(B_PANEL_BACKGROUND_COLOR));

	SetInfoText();
}

FSDialogWindow::TFSContextDialogView :: ~TFSContextDialogView() {
	mScrollView		.RemoveSelf();
	mListView		.RemoveSelf();
	mDefaultAnswerCB.RemoveSelf();
	mTextView		.RemoveSelf();
	mTextControl	.RemoveSelf();
}


void
FSDialogWindow::TFSContextDialogView :: SetInfoText() {
	static const int32 kMaxRunStructs = 20;
	
	// Text styles
	
	struct InfoTextStyle {
		const BFont		**font;
		rgb_color		color;
	};
	
	static const InfoTextStyle sDefault			= { &be_plain_font,		{0, 0, 0, 0}};
	static const InfoTextStyle sStatic			= { &be_plain_font,		{20, 20, 20, 0}};
	static const InfoTextStyle sRootOperation	= { &be_bold_font,		{0, 100, 50, 0}};
	static const InfoTextStyle sEntry			= { &be_bold_font,		{0, 50, 100, 0}};
	static const InfoTextStyle sError			= { &be_bold_font,		{100, 0, 0, 0}};
	static const InfoTextStyle sInteraction		= { &be_bold_font,		{0, 80, 0, 0}};
	static const InfoTextStyle sCurrentOperation= { &be_bold_font,		{0, 0, 100, 0}};
	
#define SET_STYLE(x)							\
	run = &array -> runs[i];					\
	run -> font 	= *s##x.font;				\
	run -> offset	= str.Length();				\
	run -> color	= s##x.color;				\
	++i;										\

// !!! only after a SET_STYLE !!!
#define SCALE_FONT(sizeprop)					\
	run -> font.SetSize(run -> font.Size() * sizeprop);
#define SET_SHEAR								\
	run -> font.SetShear(100);					
//	run -> font.SetSpacing(B_CHAR_SPACING);

	// The function
	
	auto_ptr<text_run_array> array(reinterpret_cast<text_run_array *>(
									new char[sizeof(text_run_array) + sizeof(text_run) * kMaxRunStructs])); 
	BString str;
	int32 i = 0;
	register text_run *run;

	SET_STYLE(Static);
	SET_SHEAR;
	str += "While\n  ";
	
	SET_STYLE(RootOperation);
	str += mContext.AsString(mContext.RootOperation());
	
	if (mContext.HasTargetDir(mContext.RootOperation())) {
		SET_STYLE(Default);
		
		str += ' ';
		str += mContext.TargetDirPrefix(mContext.RootOperation());
		str += ' ';
		
		BPath path;
		mContext.GetRootTargetDir(path);
		str += path.Path();
	}
	str += ":\n";
	
	if (mType == kError  ||  mContext.IsExtendedInteraction(mInteractionCode) == true) {
		const EntryRef &ref(*mContext.CurrentEntry());
		BEntry entry(ref);
		BPath path;
		entry.GetPath(&path);

		SET_STYLE(Static);
		SET_SHEAR;
		str += "With the ";
		
		if (entry.IsFile())
			str += "file ";
		else if (entry.IsSymLink())
			str += "link ";
		else if (entry.IsDirectory())
			str += "folder ";
		else
			TRESPASS();

		str += "\n  ";

		SET_STYLE(Entry);
		str += mContext.CurrentEntryName();
		
		SET_STYLE(Default);
		SCALE_FONT(0.9);
		str += " (from ";
		
		{
			BString tmpstr(path.Path());
			tmpstr.RemoveLast(path.Leaf());
			str += tmpstr;
			str += ")\n";
		}
	
		if (mContext.OperationStackSize() > 1) {
			SET_STYLE(Static);
			SET_SHEAR;
			str += "Actual operation\n  ";
		
			SET_STYLE(CurrentOperation);
			str += mContext.AsString(mContext.CurrentOperation());
		
#if PRINT_OPERATION_STACK
			SET_STYLE(Default);
			SCALE_FONT(0.9);
			str += " (";
			str += mContext.OperationStackAsString();
			str += ')';
			str += '\n';
#else
			str += '\n';
#endif
		}
	}
	
	SET_STYLE(Default);		// Half sized newline
	SCALE_FONT(0.3);
	str += '\n';
	
	if (mType == kError) {
	
//		SET_STYLE(Default);
//		SET_SHEAR;
//		str += "The returned error code:\n";
		
		SET_STYLE(Error);
		SCALE_FONT(1.2);
		str += strerror(mErrorCode);

#if PRINT_ERROR_CODE
		str += " (";
		char buf[128];
		sprintf(buf, "0x%lx", (int32)mErrorCode);
		str += buf;
		str += ")";
#endif
		
	} else if (mType == kInteraction) {
		
		SET_STYLE(Interaction);
		SCALE_FONT(1.2);
		str += mContext.AsString(mInteractionCode);
		
	} else {
		TRESPASS();
	}

	str += '\n';

	ASSERT(i < kMaxRunStructs);

	array -> count = i;
	mTextView.SetText(str.String(), array.get());
#undef SET_STYLE
#undef SCALE_FONT
}

void
FSDialogWindow::TFSContextDialogView :: AllAttached() {
	inherited::AllAttached();

	mDefaultAnswerCB.ResizeToPreferred();
	mDefaultAnswerCB.SetEnabled(false);
	
	mListView.SetViewColor(ui_color(B_PANEL_BACKGROUND_COLOR));
	mListView.SetLowColor(ui_color(B_PANEL_BACKGROUND_COLOR));
	mListView.MakeFocus();
	mListView.SetSelectionMessage(new BMessage(kReplySelected));
	mListView.SetInvocationMessage(new BMessage(kReplyInvoked));
	
	mTextView.SetViewColor(ui_color(B_PANEL_BACKGROUND_COLOR));

	bool separator = false;
	
	if (mType == kInteraction) {
		const TFSContext::command *cmds = mContext.PossibleAnswersFor(mInteractionCode);
		if (cmds[0] != 0) {
			for (int32 i = 0;  i < 6;  ++i) {
				if (cmds[i] == 0)
					break;
				
				mListView.AddItem(new CustomItem(cmds[i]));
			}
	
			mListView.AddItem(new SeparatorItem());
			separator = true;
		}
	}
	
	int32 oldcount = mListView.CountItems();
	
	if (mContext.IsAnswerPossible(TFSContext::fSkipEntry))
		mListView.AddItem(new CustomItem(TFSContext::kSkipEntry));
	if (mContext.IsAnswerPossible(TFSContext::fSkipOperation))
		mListView.AddItem(new CustomItem(TFSContext::kSkipOperation));
	if (mContext.IsAnswerPossible(TFSContext::fRetryEntry))
		mListView.AddItem(new CustomItem(TFSContext::kRetryEntry));
	if (mContext.IsAnswerPossible(TFSContext::fSkipDirectory))
		mListView.AddItem(new CustomItem(TFSContext::kSkipDirectory));
	if (mContext.IsAnswerPossible(TFSContext::fIgnore))
		mListView.AddItem(new CustomItem(TFSContext::kIgnore));
	
	if (separator  &&  mListView.CountItems() == oldcount) {
		BListItem *item = mListView.RemoveItem(oldcount - 1);
		delete item;
	}
	
	assert_cast<FSDialogWindow *>(Window()) -> Pack();
}


void
FSDialogWindow::TFSContextDialogView :: ResizeToPreferred() {

	float tw = Bounds().right + 1;
	BRect rect;

	mScrollView.ResizeToPreferred();
	rect = mScrollView.Bounds();

	rect.right += 10;
	if (tw < rect.right)
		tw = rect.right;

	mTextView.MoveTo(5, 5);
	mTextView.ResizeToPreferred();
	rect = mTextView.Bounds();
	mTextView.SetTextRect(rect);

	rect.right += 10;
	if (tw < rect.right)
		tw = rect.right;

	mScrollView.MoveTo(5, mTextView.Frame().bottom + 6);

	if (mTextControl.IsHidden() == false) {
		mTextControl.ResizeToPreferred();
		mTextControl.ResizeTo(tw - 10, mTextControl.Bounds().bottom);
		mTextControl.MoveTo(5, mScrollView.Frame().bottom + 4);
	}
	
	mDefaultAnswerCB.ResizeToPreferred();
	mDefaultAnswerCB.MoveTo((tw - mDefaultAnswerCB.Bounds().right) / 2,
							(mTextControl.IsHidden()) ? mScrollView.Frame().bottom + 4 :
														mTextControl.Frame().bottom + 4);
	
	if (tw - 10  >  mTextView.Bounds().right) {
		mTextView.ResizeTo(tw - 10, mTextView.Bounds().bottom);
		mTextView.SetTextRect(mTextView.Bounds());
	}
	if (tw - 10  >  mScrollView.Bounds().right) {
		mScrollView.ResizeTo(tw - 10, mScrollView.Bounds().bottom);
		BRect tmp = mScrollView.Bounds();
		tmp.InsetBy(2, 2);
		mScrollView.mListView.ResizeTo(tmp.Width(), tmp.Height());
	}		
	ResizeTo(tw, mDefaultAnswerCB.Frame().bottom);
}

//void
//FSDialogWindow::TFSContextDialogView :: Draw(BRect rect) {
//}

void
FSDialogWindow::TFSContextDialogView::CustomTextView :: ResizeToPreferred() {

	BRegion region;
	BRect frame;
	float width = BScreen(Window()).Frame().right * 0.33;
		
	ResizeTo(width, 200);
	SetTextRect(Bounds());

	GetTextRegion(0, TextLength(), &region);
	frame = region.Frame();
	if (frame.IsValid())
		ResizeTo(width, frame.Height());
	else
		ResizeTo(width, kTextViewHeight);

	SetTextRect(Bounds());

	GetTextRegion(0, TextLength(), &region);
	frame = region.Frame();
	if (frame.IsValid())
		ResizeTo(frame.right, frame.bottom);

	SetTextRect(Bounds());
	
	return;
}

void
FSDialogWindow::TFSContextDialogView::CustomListView :: ResizeToPreferred() {
	ASSERT(Window() != 0);

	float h = 0;

	for (int32 i = 0;  i < CountItems();  ++i)
		h += ceilf(ItemAt(i) -> Height()) + 1;

	h -= 1;
	
	ResizeTo(200, h);
}

void
FSDialogWindow::TFSContextDialogView::CustomScrollView :: ResizeToPreferred() {
	
	mListView.ResizeToPreferred();

	BRect rect = mListView.Bounds();
	rect.InsetBy(-2, -2);

	ResizeTo(rect.Width(), rect.Height());
}

//void
//FSDialogWindow::TFSContextDialogView::CustomListView :: TargetedByScrollView(BScrollView *v) {
//	float w, h;
//	GetPreferredSize(&w, &h);
//
//printf("%f, %f\n", w, h);	// XXX
//
//	v -> ResizeTo(w, h);
//
//	inherited::TargetedByScrollView(v);
//}

void
FSDialogWindow::TFSContextDialogView::CustomListView :: SelectionChanged() {

	FSDialogWindow *win = assert_cast<FSDialogWindow *>(Window());
	TFSContextDialogView *dv = assert_cast<TFSContextDialogView *>(Parent() -> Parent());

	int32 i = CurrentSelection();
	if (i >= 0) {
		win -> SetContinueEnabled(true);

		CustomItem *item = assert_cast<CustomItem *>(ItemAt(i));
		
		dv -> mDefaultAnswerCB.SetEnabled(dv -> mContext.IsDefaultableCommand(item -> Command()));

		if (item -> NeedsTextControl()) {
			if (dv -> mTextControl.IsHidden()) {
				dv -> mTextControl.Show();
				
				if (item -> Command() == TFSContext::kSuppliedNewNameForSource)
					win -> SetTextControlText(win -> mSourceNewName);
				else if (item -> Command() == TFSContext::kSuppliedNewNameForTarget)
					win -> SetTextControlText(win -> mTargetNewName);
				
				win -> ResizeToPreferred();
			}
		} else {
			if (dv -> mTextControl.IsHidden() == false) {
				dv -> mTextControl.Hide();
				win -> ResizeToPreferred();
			}
		}
	} else {
		win -> SetContinueEnabled(false);
		assert_cast<TFSContextDialogView *>(Parent()) -> mDefaultAnswerCB.SetEnabled(false);
		if (dv -> mTextControl.IsHidden() == false) {
			dv -> mTextControl.Hide();
			win -> ResizeToPreferred();
		}
	}
	
	mPreviousSelection = i;
	inherited::SelectionChanged();
}

void
FSDialogWindow::SeparatorItem :: DrawItem(BView *v, BRect rect, bool) {
	v -> SetHighColor(v -> ViewColor());
	v -> FillRect(rect);

	rect.top += 2;
//	rect.InsetBy(0, 1);  //  delme

	v -> SetHighColor(tint_color(v -> ViewColor(), B_DARKEN_2_TINT));
	v -> StrokeLine(rect.LeftTop(), rect.RightTop());

	rect.top += 1;
	v -> SetHighColor(tint_color(v -> ViewColor(), B_LIGHTEN_2_TINT));
	v -> StrokeLine(rect.LeftTop(), rect.RightTop());
}

static const rgb_color			kSkipTextColor			= {120, 120, 0, 0};
static const rgb_color			kAgainTextColor			= {0, 0, 200, 0};
static const rgb_color			kRetryOperationTextColor= {0, 200, 0, 0};
static const rgb_color			kReplaceTextColor		= {200, 0, 0, 0};
static const rgb_color			kDeleteTextColor		= {230, 0, 0, 0};

void
FSDialogWindow::CustomItem :: DrawItem(BView *v, BRect rect, bool full) {

	const char *first = 0, *second = 0, *third = 0;
	
	rgb_color	first_col	= {0, 0, 0, 0},
				second_col	= {0, 0, 0, 0},
				third_col	= {0, 0, 0, 0};
				
	const BFont	*first_font = be_plain_font,
				*second_font = be_plain_font,
				*third_font = be_plain_font;
//	float red_tint = 0, green_tint = 0, blue_tint = 0;
	
	BString str;
	
	switch (static_cast<int32>(mCommand)) {
		case TFSContext::kSkipOperation: {
			TFSContext &context(assert_cast<FSDialogWindow *>(v -> Window()) -> Context());
			
			first = "Skip operation";
			if (context.SkipOperationTarget() != context.kInvalidOperation) {
				str = " (";
				str += context.AsString(context.SkipOperationTarget());
				str += ")";
				second = str.String();
			}
			break;
		}									
		case TFSContext::kRawCopyLink:
											first	= "Copy link ";
											second	= "as is";
											third	= "(raw mode)";
											second_font = be_bold_font;
			break;
		case TFSContext::kSuppliedNewNameForSource:
											first	= "New name";
											second	= " for the source entry";
											first_font = be_bold_font;
			break;
									
		case TFSContext::kSuppliedNewNameForTarget:
											first	= "Rename";
											second	= " entry in the destination";
											first_font = be_bold_font;
			break;
									
		case TFSContext::kSkipEntry:		first	= "Skip";
											second	= " this entry";
											first_col = kSkipTextColor;
											first_font = be_bold_font;
			break;
									
		case TFSContext::kRetryEntry:		first	= "Start ";
											second	= "again";
											third	= " with this entry";
											second_col = kAgainTextColor;
											second_font = be_bold_font;
			break;
									
		case TFSContext::kSkipDirectory:	first	= "Skip the whole directory";
			break;
									
		case TFSContext::kRetryOperation:	first	= "Retry";
											second	= " the operation";
											first_col = kRetryOperationTextColor;
											first_font = be_bold_font;
			break;
									
		case TFSContext::kCreateAbsolute:	first	= "Create absolute link";
			break;
			
		case TFSContext::kCopyEachOneInstead:
											first	= "Copy link targets instead";
			break;
			
		case TFSContext::kCopyInstead:		first	= "Copy";
											second	= " instead";
											first_font = be_bold_font;
			break;
			
		case TFSContext::kCopyInsteadAndDelete:
											first	= "Copy instead and ";
											second	= "delete source";
											second_font = be_bold_font;
//											red_tint = B_LIGHTEN_1_TINT;
			break;
			
		case TFSContext::kCopyInsteadAndTrash:
											first	= "Copy instead and ";
											second	= "move source to Trash";
											second_font = be_bold_font;
			break;
			
		case TFSContext::kReplace:			first	= "Replace";
											second	= " target with the source";
											first_col = kReplaceTextColor;
											first_font = be_bold_font;
//											red_tint = B_LIGHTEN_2_TINT;
			break;
			
		case TFSContext::kIgnore:			first	= "Ignore";
											second	= " this error and go on";
											first_font = be_bold_font;
			break;
			
		case TFSContext::kGoOnAndDelete:	first	= "Go on and ";
											second	= "delete";
											second_col = kDeleteTextColor;
											second_font = be_bold_font;
//											red_tint = B_LIGHTEN_2_TINT;
			break;
			
		case TFSContext::TFSContext::kEnterBoth:
											first	= "Enter";
											second	= " both folders and go on";
											first_font = be_bold_font;
			break;
			
		case TFSContext::kAppend:			first	= "Append";
											second	= " source file to the target";
											first_font = be_bold_font;
			break;
			
		case TFSContext::kMakeUniqueName:	first	= "Find a ";
											second	= "unique name";
											second_font = be_bold_font;
			break;
			
		case TFSContext::kMoveTargetToTrash:first	= "Move target to ";
											second	= "Trash";
											second_font = be_bold_font;
			break;
			
		case TFSContext::kDeleteInstead:	first	= "Delete";
											second	= " instead";
											first_col = kDeleteTextColor;
											first_font = be_bold_font;
			break;
			
		default:							TRESPASS();
	}

	ASSERT(first != 0);

	if (IsSelected()  ||  full == true) {
		rgb_color color = v -> ViewColor();
		if (IsSelected())
			color = tint_color(color, B_DARKEN_1_TINT);

//		if (red_tint != 0)
//			color.red = (uint8)((float)color.red * red_tint);
//		if (green_tint != 0)
//			color.green = (uint8)((float)color.green * green_tint);
//		if (blue_tint != 0)
//			color.blue = (uint8)((float)color.blue * blue_tint);

		v -> SetLowColor(color);
		v -> FillRect(rect, B_SOLID_LOW);
	}

	// for font antialiasing
	v -> SetLowColor(tint_color(v -> ViewColor(), IsSelected() ? B_DARKEN_1_TINT : B_DARKEN_1_TINT / 2));

	font_height fh;
	v -> GetFontHeight(&fh);
	
	v -> MovePenTo(rect.left + 4, floor(rect.bottom) - floor(fh.descent));

	v -> SetHighColor(first_col);
	v -> SetFont(first_font);
	v -> DrawString(first);

	if (second) {
		v -> SetHighColor(second_col);
		v -> SetFont(second_font);
		v -> DrawString(second);
	}

	if (third) {
		v -> SetHighColor(third_col);
		v -> SetFont(third_font);
		v -> DrawString(third);
	}
}


}	// namespace BPrivate



