// CorrectTimeGetter.cpp
// 正しい時間を取得する支援クラスの実装

#include "CorrectTimeGetter.h"

// すべてのフィールドをホストのエンディアンにスワップする
void NtpPacket::Swap()
{
	swap_action defined_type = B_SWAP_BENDIAN_TO_HOST;

	swap_data(B_INT32_TYPE,&control_word,sizeof(int32),defined_type);
	swap_data(B_INT32_TYPE,&root_delay,sizeof(int32),defined_type);
	swap_data(B_INT32_TYPE,&root_dispersion,sizeof(int32),defined_type);
//	swap_data(B_INT32_TYPE,&reference_identifier,sizeof(int32),defined_type); // 文字列
	swap_data(B_INT64_TYPE,&reference_timestamp,sizeof(int64),defined_type);
	swap_data(B_INT64_TYPE,&originate_timestamp,sizeof(int64),defined_type);
	swap_data(B_INT64_TYPE,&receive_timestamp,sizeof(int64),defined_type);
	swap_data(B_INT32_TYPE,&transmit_timestamp_seconds,sizeof(int32),defined_type);
	swap_data(B_INT32_TYPE,&transmit_timestamp_fractions,sizeof(int32),defined_type);
}

void NtpPacket::PrintToStream()
{
#ifdef DEBUG
	cout << "CONTROL_WORD: " << control_word << endl <<
		   "ROOT_DELAY(遅延): " << (root_delay >> 4) << endl <<
		   "ROOT_DISPERSION(分散): " << (root_dispersion >> 4) << endl <<
		   "REFERENCE_IDENTIFIER(基準ID): " << (char*)&reference_identifier << endl <<
		   "REFERENCE_TIMESTAMP: " << (reference_identifier >> 4) << endl << 
		   "ORIGINATE_TIMESTAMP: " << (originate_timestamp >> 4) << endl <<
		   "RECEIVE_TIMESTAMP: " << (receive_timestamp >> 4) << endl << 
		   "TRNSMIT_TIMESTAMP_SECONDS: " << transmit_timestamp_seconds << endl <<
		   "TRNASMIT_TIMESTAMP_FRACTIONS: " << transmit_timestamp_fractions << endl <<		   
		   "TRANSMIT_TIMESTAMP: " << (int64)((transmit_timestamp_seconds << 4) |
		   								 (transmit_timestamp_fractions)) << endl;
#endif
}

/**********************************
 * コンストラクタ
 * 引数
 * _server_name:繋ぎに行くサーバー名の文字列
 * _handler:メッセージを送り付けるハンドラ
 * _timeout:データのタイムアウトの設定。デフォルトは１０秒
 */
TCorrectTimeGetter::TCorrectTimeGetter(const char* _server_name,
									const BHandler* _handler,
									bigtime_t _timeout = GETTER_DEFAULT_TIMEOUT)
			:m_error_code(NO_ERROR),
			 m_timeout(_timeout)
{
	try
	{
		if(_timeout == 0) _timeout = GETTER_DEFAULT_TIMEOUT;

		m_server_name = _server_name;

		status_t err;

		// ソケットの構築
		m_endpoint.reset(new TThreadEndpoint(SOCK_DGRAM));
		err = m_endpoint->InitCheck();
		if (err != B_OK) throw SOCKET_ERROR;
		m_endpoint->SetTimeout(_timeout);  // タイムアウトの設定

		// メッセンジャーの構築
		m_messenger.reset(new BMessenger(_handler,NULL,&err));
		if (err != B_OK) throw OTHER_ERROR;

	} 
	catch(int& err_code)
	{
		m_error_code = err_code;
	}
}

TCorrectTimeGetter::~TCorrectTimeGetter()
{
}

/**********************************
 *	Connect関数
 *	NTPサーバーに繋いでデータを取りに行く
 *   返り値
 *	無事にスレッドが生成されたらtrue
 */
bool TCorrectTimeGetter::Connect()
{
	// スレッド作成
	m_getter_id = spawn_thread(TCorrectTimeGetter::ntpPacketGetFunc,
							"NTP_PAKCET_GET_THREAD",
							B_NORMAL_PRIORITY,this);

	// きちんとスレッドが作成できなかったり
	// スレッドが開始できなかったら終了
	if (m_getter_id < B_NO_ERROR) return false;
	else if (resume_thread(m_getter_id) < B_NO_ERROR) return false;
	
	return true;
}

/**********************************
 *	ntpPacketGetFunc
 * 	NTPパケットをサーバーに取りに行くスレッド関数
 *   引数はこのクラスのインスタンスを取る
 */
int32 TCorrectTimeGetter::ntpPacketGetFunc(void* _data)
{
	TCorrectTimeGetter* getter = reinterpret_cast<TCorrectTimeGetter*>(_data);

	// アドレスの名前解決
	BNetAddress connect_address(getter->m_server_name.c_str(),NTP_PORT);
	if (connect_address.InitCheck() != B_OK)
	{
		getter->errorMessageSend(ADDRESS_NOT_FOUND);
		return B_OK;
	}

	TThreadEndpoint& endpoint = *(getter->m_endpoint); // ソケットのエイリアス作成

	// パケット受信後に送るメッセージ
	int32 data_size = 0;

	// パケットの初期化
	getter->m_packet.control_word = htonl(0x0B000000);
	
	// 送信した時間の記録
	getter->m_local_time_info.send_timestamp = system_time();

	// パケット送信	
	data_size = endpoint.SendTo(&(getter->m_packet),
							  sizeof(NtpPacket),
							  connect_address);
	if(data_size == 0)
	{
		getter->errorMessageSend(NOT_VALID_ERROR);
		return B_OK;
	}
	
	// ちょっと休む
	snooze(1000);

	// 受信すべきデータを待つ
	if(endpoint.IsDataPending(getter->m_timeout) == false)
	{
		getter->errorMessageSend(CONNECT_TIMEOUT_ERROR);
		return B_OK;
	}
	
	// パケット受信
	data_size = endpoint.ReceiveFrom(&getter->m_packet,
								sizeof(NtpPacket),
								connect_address);
	// 受信した時間の記録
	getter->m_local_time_info.receive_timestamp = system_time();

	if(data_size == 0)
	{
		getter->errorMessageSend(NOT_VALID_ERROR);
		return B_OK;
	}

	// 正しいことを伝えるデータを送信
	getter->correctMessageSend();

	return B_OK;					
}

/**********************************
 *  errorMessageSend
 *  エラーであることを伝えるメッセージを送信
 *  
 */

void TCorrectTimeGetter::errorMessageSend(int32 _error_code)
{
	BMessage* send_message = new BMessage(NTP_PACKET_GOT);

	send_message->AddInt32(IS_ERROR_FIELD,_error_code);

	m_messenger->SendMessage(send_message);
}
 
/**********************************
 * correctMessageSend
 * 受信したNtpPacketより正しい時間のパケットを送信
 */
 
void TCorrectTimeGetter::correctMessageSend()
{
	BMessage* send_message = new BMessage(NTP_PACKET_GOT);
	
	send_message->AddInt32(IS_ERROR_FIELD,NO_ERROR);

	send_message->AddPointer(NTP_PACKET_FIELD,&m_packet);
		
	send_message->AddPointer(LOCAL_TIME_FIELD,&m_local_time_info);

	m_messenger->SendMessage(send_message);
}
