/* ++++++++++
	driver.c
	AMD64 PowerNow driver for BeOS
	By Olivier Coursière
	
	Maybe some incompatible license issues here...

	Based on :
	 - linux powernow-k8 driver (CPU states functions and check_supported_cpu function) : */

		/*
		 *   (c) 2003, 2004 Advanced Micro Devices, Inc.
		 *  Your use of this code is subject to the terms and conditions of the
		 *  GNU general public license version 2. See "COPYING" or
		 *  http://www.gnu.org/licenses/gpl.html
		 *
		 *  Support : paul.devriendt@amd.com
		 *
		 *  Based on the powernow-k7.c module written by Dave Jones.
		 *  (C) 2003 Dave Jones <davej@codemonkey.org.uk> on behalf of SuSE Labs
		 *  (C) 2004 Dominik Brodowski <linux@brodo.de>
		 *  (C) 2004 Pavel Machek <pavel@suse.cz>
		 *  Licensed under the terms of the GNU GPL License version 2.
		 *  Based upon datasheets & sample CPUs kindly provided by AMD.
		 *
		 *  Valuable input gratefully received from Dave Jones, Pavel Machek,
		 *  Dominik Brodowski, and others.
		 *  Processor information obtained from Chapter 9 (Power and Thermal Management)
		 *  of the "BIOS and Kernel Developer's Guide for the AMD Athlon 64 and AMD
		 *  Opteron Processors" available for download from www.amd.com
		 */
		 
/*	 - acpica-unix from Intel (ACPI structures) : */

		/******************************************************************************
		 *
		 * 1. Copyright Notice
		 *
		 * Some or all of this work - Copyright (c) 1999 - 2004, Intel Corp.
		 * All rights reserved.
		 *
		 * 2. License
		 *
		 * 2.1. This is your license from Intel Corp. under its intellectual property
		 * rights.  You may have additional license terms from the party that provided
		 * you this software, covering your right to use that party's intellectual
		 * property rights.
		 *
		 * 2.2. Intel grants, free of charge, to any person ("Licensee") obtaining a
		 * copy of the source code appearing in this file ("Covered Code") an
		 * irrevocable, perpetual, worldwide license under Intel's copyrights in the
		 * base code distributed originally by Intel ("Original Intel Code") to copy,
		 * make derivatives, distribute, use and display any portion of the Covered
		 * Code in any form, with the right to sublicense such rights; and
		 *
		 * 2.3. Intel grants Licensee a non-exclusive and non-transferable patent
		 * license (with the right to sublicense), under only those claims of Intel
		 * patents that are infringed by the Original Intel Code, to make, use, sell,
		 * offer to sell, and import the Covered Code and derivative works thereof
		 * solely to the minimum extent necessary to exercise the above copyright
		 * license, and in no event shall the patent license extend to any additions
		 * to or modifications of the Original Intel Code.  No other license or right
		 * is granted directly or by implication, estoppel or otherwise;
		 *
		 * The above copyright and patent license is granted only if the following
		 * conditions are met:
		 *
		 * 3. Conditions
		 *
		 * 3.1. Redistribution of Source with Rights to Further Distribute Source.
		 * Redistribution of source code of any substantial portion of the Covered
		 * Code or modification with rights to further distribute source must include
		 * the above Copyright Notice, the above License, this list of Conditions,
		 * and the following Disclaimer and Export Compliance provision.  In addition,
		 * Licensee must cause all Covered Code to which Licensee contributes to
		 * contain a file documenting the changes Licensee made to create that Covered
		 * Code and the date of any change.  Licensee must include in that file the
		 * documentation of any changes made by any predecessor Licensee.  Licensee
		 * must include a prominent statement that the modification is derived,
		 * directly or indirectly, from Original Intel Code.
		 *
		 * 3.2. Redistribution of Source with no Rights to Further Distribute Source.
		 * Redistribution of source code of any substantial portion of the Covered
		 * Code or modification without rights to further distribute source must
		 * include the following Disclaimer and Export Compliance provision in the
		 * documentation and/or other materials provided with distribution.  In
		 * addition, Licensee may not authorize further sublicense of source of any
		 * portion of the Covered Code, and must include terms to the effect that the
		 * license from Licensee to its licensee is limited to the intellectual
		 * property embodied in the software Licensee provides to its licensee, and
		 * not to intellectual property embodied in modifications its licensee may
		 * make.
		 *
		 * 3.3. Redistribution of Executable. Redistribution in executable form of any
		 * substantial portion of the Covered Code or modification must reproduce the
		 * above Copyright Notice, and the following Disclaimer and Export Compliance
		 * provision in the documentation and/or other materials provided with the
		 * distribution.
		 *
		 * 3.4. Intel retains all right, title, and interest in and to the Original
		 * Intel Code.
		 *
		 * 3.5. Neither the name Intel nor any other trademark owned or controlled by
		 * Intel shall be used in advertising or otherwise to promote the sale, use or
		 * other dealings in products derived from or relating to the Covered Code
		 * without prior written authorization from Intel.
		 *
		 * 4. Disclaimer and Export Compliance
		 *
		 * 4.1. INTEL MAKES NO WARRANTY OF ANY KIND REGARDING ANY SOFTWARE PROVIDED
		 * HERE.  ANY SOFTWARE ORIGINATING FROM INTEL OR DERIVED FROM INTEL SOFTWARE
		 * IS PROVIDED "AS IS," AND INTEL WILL NOT PROVIDE ANY SUPPORT,  ASSISTANCE,
		 * INSTALLATION, TRAINING OR OTHER SERVICES.  INTEL WILL NOT PROVIDE ANY
		 * UPDATES, ENHANCEMENTS OR EXTENSIONS.  INTEL SPECIFICALLY DISCLAIMS ANY
		 * IMPLIED WARRANTIES OF MERCHANTABILITY, NONINFRINGEMENT AND FITNESS FOR A
		 * PARTICULAR PURPOSE.
		 *
		 * 4.2. IN NO EVENT SHALL INTEL HAVE ANY LIABILITY TO LICENSEE, ITS LICENSEES
		 * OR ANY OTHER THIRD PARTY, FOR ANY LOST PROFITS, LOST DATA, LOSS OF USE OR
		 * COSTS OF PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES, OR FOR ANY INDIRECT,
		 * SPECIAL OR CONSEQUENTIAL DAMAGES ARISING OUT OF THIS AGREEMENT, UNDER ANY
		 * CAUSE OF ACTION OR THEORY OF LIABILITY, AND IRRESPECTIVE OF WHETHER INTEL
		 * HAS ADVANCE NOTICE OF THE POSSIBILITY OF SUCH DAMAGES.  THESE LIMITATIONS
		 * SHALL APPLY NOTWITHSTANDING THE FAILURE OF THE ESSENTIAL PURPOSE OF ANY
		 * LIMITED REMEDY.
		 *
		 * 4.3. Licensee shall not export, either directly or indirectly, any of this
		 * software or system incorporating such software without first obtaining any
		 * required license or other approval from the U. S. Department of Commerce or
		 * any other agency or department of the United States Government.  In the
		 * event Licensee exports any such software from the United States or
		 * re-exports any such software from a foreign destination, Licensee shall
		 * ensure that the distribution and export/re-export of the software is in
		 * compliance with all laws, regulations, orders, or other restrictions of the
		 * U.S. Export Administration Regulations. Licensee agrees that neither it nor
		 * any of its subsidiaries will export/re-export any technical data, process,
		 * software, or service, directly or indirectly, to any country for which the
		 * United States government or any agency thereof requires an export license,
		 * other governmental approval, or letter of assurance, without first obtaining
		 * such license, approval or letter.
		 *
		 *****************************************************************************/
		 
/*	- CPU_fix driver by Michael Lotz (BeOS timing fixes) : */

		/*
			(c) 2004, Michael Lotz, mmlr@mlotz.ch
			Corrects timing issues with CPU's faster than 2.1GHz
			and corrects there displayed speed.
			
			Based on the idea of vmware_timer by François Revol.
			Includes CPU detection from Haiku/NewOS kernel.
			
			This file is licensed under the MIT.
		*/
		
/*	- Longrun driver for Transmeta Crusoe by Anders Granlund (ideas) */
		//---------------------------------------------------------------
		// driver.c
		// Longrun driver for Transmeta Crusoe
		//
		// Anders Granlund
		//---------------------------------------------------------------
 
/*	This file is published under the Virtual Universal License, a license that 
	can handle all incompatibilities in the above licenses.
	
	Write this license if you need more detailled license terms...
	
	Good luck !
*/

/*	 
+++++ */

#include <stdlib.h>
#include <drivers/KernelExport.h>
#include <drivers/Drivers.h>
#include <support/Errors.h>
#include <OS.h>
#include <string.h>
#include <inttypes.h>
#include <drivers/driver_settings.h>

#define	DRIVER_NAME	"amd64"
#define	DEVICE_NAME	"misc/amd64"

int32	api_version = B_CUR_DRIVER_API_VERSION;

#define RSDP_SIG "RSD PTR " /* RSDT Pointer signature */

/* ----------
	init_hardware - called once the first time the driver is loaded
----- */

_EXPORT status_t
init_hardware (void)
{
	dprintf (DRIVER_NAME ": init_hardware()\n");
	return B_OK;
}


// from acpica-unix from Intel

/*
 * ACPI Version-independent tables
 *
 * NOTE: The tables that are specific to ACPI versions (1.0, 2.0, etc.)
 * are in separate files.
 */

/*
 * 32-bit type definitions (default)
 */
typedef unsigned char                   UINT8;
typedef unsigned char                   BOOLEAN;
typedef unsigned short                  UINT16;
typedef int                             INT32;
typedef unsigned int                    UINT32;
typedef signed long long		        INT64;
typedef unsigned long long       		UINT64;

/*
 * Miscellaneous common types
 */
typedef UINT16                          UINT16_BIT;
typedef UINT32                          UINT32_BIT;
//typedef ACPI_NATIVE_UINT                ACPI_PTRDIFF;
 
typedef struct rsdp_descriptor /* Root System Descriptor Pointer */
{
    char                    Signature [8];          /* ACPI signature, contains "RSD PTR " */
    UINT8                   Checksum;               /* To make sum of struct == 0 */
    char                    OemId [6];              /* OEM identification */
    UINT8                   Revision;               /* Must be 0 for 1.0, 2 for 2.0 */
    UINT32                  RsdtPhysicalAddress;    /* 32-bit physical address of RSDT */
    UINT32                  Length;                 /* XSDT Length in bytes including hdr */
    UINT64                  XsdtPhysicalAddress;    /* 64-bit physical address of XSDT */
    UINT8                   ExtendedChecksum;       /* Checksum of entire table */
    char                    Reserved [3];           /* Reserved field must be 0 */

} RSDP_DESCRIPTOR;

#define ACPI_TABLE_HEADER_DEF   /* ACPI common table header */ \
    char                    Signature [4];          /* ACPI signature (4 ASCII characters) */\
    UINT32                  Length;                 /* Length of table, in bytes, including header */\
    UINT8                   Revision;               /* ACPI Specification minor version # */\
    UINT8                   Checksum;               /* To make sum of entire table == 0 */\
    char                    OemId [6];              /* OEM identification */\
    char                    OemTableId [8];         /* OEM table identification */\
    UINT32                  OemRevision;            /* OEM revision number */\
    char                    AslCompilerId [4];      /* ASL compiler vendor ID */\
    UINT32                  AslCompilerRevision;    /* ASL compiler revision number */

/*
 * ACPI 1.0 Root System Description Table (RSDT)
 */
typedef struct rsdt_descriptor_rev1
{
    ACPI_TABLE_HEADER_DEF                           /* ACPI common table header */
    UINT32                  TableOffsetEntry [1];   /* Array of pointers to other */
                                                    /* ACPI tables */
} RSDT_DESCRIPTOR_REV1;

/*
 * ACPI 1.0 Fixed ACPI Description Table (FADT)
 */
typedef struct fadt_descriptor_rev1
{
    ACPI_TABLE_HEADER_DEF                           /* ACPI common table header */
    UINT32                  FirmwareCtrl;           /* Physical address of FACS */
    UINT32                  Dsdt;                   /* Physical address of DSDT */
    UINT8                   Model;                  /* System Interrupt Model */
    UINT8                   Reserved1;              /* Reserved */
    UINT16                  SciInt;                 /* System vector of SCI interrupt */
    UINT32                  SmiCmd;                 /* Port address of SMI command port */
    UINT8                   AcpiEnable;             /* Value to write to smi_cmd to enable ACPI */
    UINT8                   AcpiDisable;            /* Value to write to smi_cmd to disable ACPI */
    UINT8                   S4BiosReq;              /* Value to write to SMI CMD to enter S4BIOS state */
    UINT8                   Reserved2;              /* Reserved - must be zero */
    UINT32                  Pm1aEvtBlk;             /* Port address of Power Mgt 1a AcpiEvent Reg Blk */
    UINT32                  Pm1bEvtBlk;             /* Port address of Power Mgt 1b AcpiEvent Reg Blk */
    UINT32                  Pm1aCntBlk;             /* Port address of Power Mgt 1a Control Reg Blk */
    UINT32                  Pm1bCntBlk;             /* Port address of Power Mgt 1b Control Reg Blk */
    UINT32                  Pm2CntBlk;              /* Port address of Power Mgt 2 Control Reg Blk */
    UINT32                  PmTmrBlk;               /* Port address of Power Mgt Timer Ctrl Reg Blk */
    UINT32                  Gpe0Blk;                /* Port addr of General Purpose AcpiEvent 0 Reg Blk */
    UINT32                  Gpe1Blk;                /* Port addr of General Purpose AcpiEvent 1 Reg Blk */
    UINT8                   Pm1EvtLen;              /* Byte Length of ports at pm1X_evt_blk */
    UINT8                   Pm1CntLen;              /* Byte Length of ports at pm1X_cnt_blk */
    UINT8                   Pm2CntLen;              /* Byte Length of ports at pm2_cnt_blk */
    UINT8                   PmTmLen;                /* Byte Length of ports at pm_tm_blk */
    UINT8                   Gpe0BlkLen;             /* Byte Length of ports at gpe0_blk */
    UINT8                   Gpe1BlkLen;             /* Byte Length of ports at gpe1_blk */
    UINT8                   Gpe1Base;               /* Offset in gpe model where gpe1 events start */
    UINT8                   Reserved3;              /* Reserved */
    UINT16                  Plvl2Lat;               /* Worst case HW latency to enter/exit C2 state */
    UINT16                  Plvl3Lat;               /* Worst case HW latency to enter/exit C3 state */
    UINT16                  FlushSize;              /* Size of area read to flush caches */
    UINT16                  FlushStride;            /* Stride used in flushing caches */
    UINT8                   DutyOffset;             /* Bit location of duty cycle field in p_cnt reg */
    UINT8                   DutyWidth;              /* Bit width of duty cycle field in p_cnt reg */
    UINT8                   DayAlrm;                /* Index to day-of-month alarm in RTC CMOS RAM */
    UINT8                   MonAlrm;                /* Index to month-of-year alarm in RTC CMOS RAM */
    UINT8                   Century;                /* Index to century in RTC CMOS RAM */
    UINT8                   Reserved4;              /* Reserved */
    UINT8                   Reserved4a;             /* Reserved */
    UINT8                   Reserved4b;             /* Reserved */
    UINT32_BIT              WbInvd          : 1;    /* The wbinvd instruction works properly */
    UINT32_BIT              WbInvdFlush     : 1;    /* The wbinvd flushes but does not invalidate */
    UINT32_BIT              ProcC1          : 1;    /* All processors support C1 state */
    UINT32_BIT              Plvl2Up         : 1;    /* C2 state works on MP system */
    UINT32_BIT              PwrButton       : 1;    /* Power button is handled as a generic feature */
    UINT32_BIT              SleepButton     : 1;    /* Sleep button is handled as a generic feature, or not present */
    UINT32_BIT              FixedRTC        : 1;    /* RTC wakeup stat not in fixed register space */
    UINT32_BIT              Rtcs4           : 1;    /* RTC wakeup stat not possible from S4 */
    UINT32_BIT              TmrValExt       : 1;    /* The tmr_val width is 32 bits (0 = 24 bits) */
    UINT32_BIT              Reserved5       : 23;   /* Reserved - must be zero */

} FADT_DESCRIPTOR_REV1;

typedef struct ssdt_descriptor
{
    ACPI_TABLE_HEADER_DEF                           /* ACPI common table header */
} SSDT_DESCRIPTOR;

// CPU states change functions (from linux powernow-k8 driver)

/*
 * There are restrictions frequencies have to follow:
 * - only 1 entry in the low fid table ( <=1.4GHz )
 * - lowest entry in the high fid table must be >= 2 * the entry in the
 *   low fid table
 * - lowest entry in the high fid table must be a <= 200MHz + 2 * the entry
 *   in the low fid table
 * - the parts can only step at 200 MHz intervals, so 1.9 GHz is never valid
 * - lowest frequency must be >= interprocessor hypertransport link speed
 *   (only applies to MP systems obviously)
 */

/* fids (frequency identifiers) are arranged in 2 tables - lo and hi */
#define LO_FID_TABLE_TOP     6	/* fid values marking the boundary    */
#define HI_FID_TABLE_BOTTOM  8	/* between the low and high tables    */

#define LO_VCOFREQ_TABLE_TOP    1400	/* corresponding vco frequency values */
#define HI_VCOFREQ_TABLE_BOTTOM 1600

#define MIN_FREQ_RESOLUTION  200 /* fids jump by 2 matching freq jumps by 200 */

#define MAX_FID 0x2a	/* Spec only gives FID values as far as 5 GHz */
#define LEAST_VID 0x1e	/* Lowest (numerically highest) useful vid value */

#define MIN_FREQ 800	/* Min and max freqs, per spec */
#define MAX_FREQ 5000

#define INVALID_FID_MASK 0xffffffc1  /* not a valid fid if these bits are set */
#define INVALID_VID_MASK 0xffffffe0  /* not a valid vid if these bits are set */

#define STOP_GRANT_5NS 1 /* min poss memory access latency for voltage change */

#define PLL_LOCK_CONVERSION (1000/5) /* ms to ns, then divide by clock period */

#define MAXIMUM_VID_STEPS 1  /* Current cpus only allow a single step of 25mV */
#define VST_UNITS_20US 20   /* Voltage Stabalization Time is in units of 20us */


/*
 * Most values of interest are enocoded in a single field of the _PSS
 * entries: the "control" value.
 */
                                                                                                    
#define IRT_SHIFT      30
#define RVO_SHIFT      28
#define PLL_L_SHIFT    20
#define MVS_SHIFT      18
#define VST_SHIFT      11
#define VID_SHIFT       6
#define IRT_MASK        3
#define RVO_MASK        3
#define PLL_L_MASK   0x7f
#define MVS_MASK        3
#define VST_MASK     0x7f
#define VID_MASK     0x1f
#define FID_MASK     0x3f

/* Return a frequency in MHz, given an input fid */
static UINT32 find_freq_from_fid(UINT32 fid)
{
	return 800 + (fid * 100);
}

/* Return a frequency in KHz, given an input fid */
static UINT32 find_khz_freq_from_fid(UINT32 fid)
{
	return 1000 * find_freq_from_fid(fid);
}

/* Return a voltage in miliVolts, given an input vid */
static UINT32 find_millivolts_from_vid(UINT32 vid)
{
	return 1550-vid*25;
}

/* Return the vco fid for an input fid */
static UINT32 convert_fid_to_vco_fid(UINT32 fid)
{
	if (fid < HI_FID_TABLE_BOTTOM) {
		return 8 + (2 * fid);
	} else {
		return fid;
	}
}

static inline int core_voltage_pre_transition(uint32 reqvid);
static inline int core_voltage_post_transition(uint32 reqvid);
static inline int core_frequency_transition(uint32 reqfid);

/* Model Specific Registers for p-state transitions. MSRs are 64-bit. For     */
/* writes (wrmsr - opcode 0f 30), the register number is placed in ecx, and   */
/* the value to write is placed in edx:eax. For reads (rdmsr - opcode 0f 32), */
/* the register number is placed in ecx, and the data is returned in edx:eax. */

#define MSR_FIDVID_CTL      0xc0010041
#define MSR_FIDVID_STATUS   0xc0010042

/* Field definitions within the FID VID Low Control MSR : */
#define MSR_C_LO_INIT_FID_VID     0x00010000
#define MSR_C_LO_NEW_VID          0x00001f00
#define MSR_C_LO_NEW_FID          0x0000002f
#define MSR_C_LO_VID_SHIFT        8

/* Field definitions within the FID VID High Control MSR : */
#define MSR_C_HI_STP_GNT_TO       0x000fffff

/* Field definitions within the FID VID Low Status MSR : */
#define MSR_S_LO_CHANGE_PENDING   0x80000000	/* cleared when completed */
#define MSR_S_LO_MAX_RAMP_VID     0x1f000000
#define MSR_S_LO_MAX_FID          0x003f0000
#define MSR_S_LO_START_FID        0x00003f00
#define MSR_S_LO_CURRENT_FID      0x0000003f

/* Field definitions within the FID VID High Status MSR : */
#define MSR_S_HI_MAX_WORKING_VID  0x001f0000
#define MSR_S_HI_START_VID        0x00001f00
#define MSR_S_HI_CURRENT_VID      0x0000001f

/* fids (frequency identifiers) are arranged in 2 tables - lo and hi */
#define LO_FID_TABLE_TOP     6
#define HI_FID_TABLE_BOTTOM  8

#define LO_VCOFREQ_TABLE_TOP    1400	/* corresponding vco frequency values */
#define HI_VCOFREQ_TABLE_BOTTOM 1600

#define MIN_FREQ_RESOLUTION  200 /* fids jump by 2 matching freq jumps by 200 */

#define MAX_FID 0x2a	/* Spec only gives FID values as far as 5 GHz */

#define LEAST_VID 0x1e	/* Lowest (numerically highest) useful vid value */

#define MIN_FREQ 800	/* Min and max freqs, per spec */
#define MAX_FREQ 5000

/***Paul: masks for register values below ... code uses explicitly sized types
(u32) rather than unsigned int / unsigned long so that these masks remain
correct as the mode changes from 32-bit to 64-bit.*/

#define INVALID_FID_MASK 0xffffffc1  /* not a valid fid if these bits are set */

#define INVALID_VID_MASK 0xffffffe0  /* not a valid vid if these bits are set */

#define STOP_GRANT_5NS 1 /* min poss memory access latency for voltage change */

#define PLL_LOCK_CONVERSION (1000/5) /* ms to ns, then divide by clock period */

#define MAXIMUM_VID_STEPS 1  /* Current cpus only allow a single step of 25mV */

#define VST_UNITS_20US 20   /* Voltage Stabalization Time is in units of 20us */

//---------------------------------------------------------------
// MSR information
//---------------------------------------------------------------
union msrinfo {
	uint64	msr;
	uint32	regs[2];
};

//---------------------------------------------------------------
// Read MSR (Machine specific registers)
//---------------------------------------------------------------
static inline uint64 rdmsr( u_int msr)
{
	uint64 rv;
	asm volatile(".byte 0x0f, 0x032" : "=A" (rv) : "c" (msr));
	return (rv);
}

//---------------------------------------------------------------
// Write MSR (Machine specific registers)
//---------------------------------------------------------------
static inline void wrmsr( uint msr, uint64 newval)
{
	asm volatile(".byte 0x0f, 0x30" : : "A" (newval), "c" (msr));
}

/* Return 1 if the pending bit is set. Unless we are actually just told the */
/* processor to transition a state, seeing this bit set is really bad news. */
static inline int
pending_bit_stuck(void)
{
	union msrinfo data;

	data.msr = rdmsr(MSR_FIDVID_STATUS);
	return data.regs[0] & MSR_S_LO_CHANGE_PENDING ? 1 : 0;
}

static uint32 currvid;	/* keep track of the current fid / vid */
static uint32 currfid;
static uint32 rvo = (0xE0202C80 >> RVO_SHIFT) & RVO_MASK;
static uint32 vidmvs = 1 << ((0xE0202C80 >> MVS_SHIFT) & MVS_MASK);
static uint32 irt = (0xE0202A8A >> IRT_SHIFT) & IRT_MASK;
static uint32 vstable = ((0xE0202A8A >> VST_SHIFT) & VST_MASK);
static uint32 plllock = (0xE020288E >> PLL_L_SHIFT) & PLL_L_MASK;

/* Update the global current fid / vid values from the status msr. Returns 1 */
/* on error.                                                                 */
static int
query_current_values_with_pending_wait(void)
{
	union msrinfo data;
	uint32 i = 0;

	data.regs[0] = MSR_S_LO_CHANGE_PENDING;
	while (data.regs[0] & MSR_S_LO_CHANGE_PENDING) {
		if (i++ > 0x1000000) {
			dprintf(DRIVER_NAME "KERN_ERR PFX detected change pending stuck\n");
			return 1;
		}
		data.msr = rdmsr(MSR_FIDVID_STATUS);
	}

/***Paul: the rdmsr above and the masks below illustrate the hardware
mapping of some of the variables. Note the use of the rdmsr rather than
coding inline assembler to access low level instructions.*/

	currvid = data.regs[1] & MSR_S_HI_CURRENT_VID;
	currfid = data.regs[0] & MSR_S_LO_CURRENT_FID;

	return 0;
}

/* the isochronous relief time */
static inline void
count_off_irt(void)
{
	snooze((1 << irt) * 10);
	return;
}

/* the voltage stabalization time */
static inline void
count_off_vst(void)
{
	snooze(vstable * VST_UNITS_20US);
	return;
}

/* write the new fid value along with the other control fields to the msr */
static int
write_new_fid(uint32 fid)
{
	union msrinfo data;
	uint32 savevid = currvid;

	if ((fid & INVALID_FID_MASK) || (currvid & INVALID_VID_MASK)) {
		dprintf("KERN_ERR PFX internal error - overflow on fid write\n");
		return 1;
	}

	data.regs[0] = fid | (currvid << MSR_C_LO_VID_SHIFT) | MSR_C_LO_INIT_FID_VID;
	data.regs[1] = plllock * PLL_LOCK_CONVERSION;
	
	dprintf(DRIVER_NAME " : writing fid %x, lo %x, hi %x \n",
		(unsigned int)fid, (unsigned int)data.regs[0], (unsigned int)data.regs[1]);

	wrmsr(MSR_FIDVID_CTL, data.msr);

/***Paul: again, the wrmsr above has the correct size variables.*/

	if (query_current_values_with_pending_wait())
		return 1;

	count_off_irt();

	if (savevid != currvid) {
		dprintf("KERN_ERR PFX vid changed on fid transition, save %x, currvid %x\n",
		       (unsigned int)savevid, (unsigned int)currvid);
		return 1;
	}

	if (fid != currfid) {
		dprintf("KERN_ERR PFX fid transition failed, fid %x, currfid %x\n",
		        (unsigned int)fid, (unsigned int)currfid);
		return 1;
	}

	return 0;
}

/* Write a new vid to the hardware */
static int
write_new_vid(uint32 vid)
{
	union msrinfo data;
	uint32 savefid = currfid;

	if ((currfid & INVALID_FID_MASK) || (vid & INVALID_VID_MASK)) {
		dprintf("KERN_ERR PFX internal error - overflow on vid write\n");
		return 1;
	}

	data.regs[0] = currfid | (vid << MSR_C_LO_VID_SHIFT) | MSR_C_LO_INIT_FID_VID;
	data.regs[1] = STOP_GRANT_5NS;
	
//	dprintf("KERN_DEBUG PFX writing vid %x, lo %x, hi %x\n",
//		(unsigned int)vid, (unsigned int)data.regs[0], (unsigned int)data.regs[1]);

	wrmsr(MSR_FIDVID_CTL, data.msr);

	if (query_current_values_with_pending_wait()) {
		return 1;
	}

	if (savefid != currfid) {
		dprintf("KERN_ERR PFX fid changed on vid transition, save %x currfid %x\n",
		       (unsigned int)savefid, (unsigned int)currfid);
		return 1;
	}

	if (vid != currvid) {
		dprintf("KERN_ERR PFX vid transition failed, vid %x, currvid %x\n",
		       (unsigned int)vid, (unsigned int)currvid);
		return 1;
	}

	return 0;
}

/* Reduce the vid by the max of step or reqvid.                   */
/* Decreasing vid codes represent increasing voltages :           */
/* vid of 0 is 1.550V, vid of 0x1e is 0.800V, vid of 0x1f is off. */
static int
decrease_vid_code_by_step(uint32 reqvid, uint32 step)
{
	if ((currvid - reqvid) > step)
		reqvid = currvid - step;

	if (write_new_vid(reqvid))
		return 1;

	count_off_vst();

	return 0;
}

/* Change the fid and vid, by the 3 phases. */
static inline int
transition_fid_vid(uint32 reqfid, uint32 reqvid)
{
	if (core_voltage_pre_transition(reqvid))
		return 1;

	if (core_frequency_transition(reqfid))
		return 1;

	if (core_voltage_post_transition(reqvid))
		return 1;

	if (query_current_values_with_pending_wait())
		return 1;

	if ((reqfid != currfid) || (reqvid != currvid)) {
		dprintf("KERN_ERR PFX failed: req 0x%x 0x%x, curr 0x%x 0x%x\n",
		       (unsigned int)reqfid, (unsigned int)reqvid, (unsigned int)currfid, (unsigned int)currvid);
		return 1;
	}

	dprintf("KERN_INFO PFX transitioned: new fid 0x%x, vid 0x%x\n", (unsigned int)currfid, 
		(unsigned int)currvid);

	return 0;
}

/* Phase 1 - core voltage transition ... setup appropriate voltage for the */
/* fid transition.                                                         */
static inline int
core_voltage_pre_transition(uint32 reqvid)
{
	uint32 rvosteps = rvo;
	uint32 savefid = currfid;

//	dprintf("KERN_DEBUG PFX ph1: start, currfid 0x%x, currvid 0x%x, reqvid 0x%x, rvo %x\n",
//		(unsigned int)currfid, (unsigned int)currvid, (unsigned int)reqvid, (unsigned int)rvo);

	while (currvid > reqvid) {
//		dprintf("KERN_DEBUG PFX ph1: curr 0x%x, requesting vid 0x%x\n",
//			(unsigned int)currvid, (unsigned int)reqvid);
		if (decrease_vid_code_by_step(reqvid, vidmvs))
			return 1;
	}

	while (rvosteps > 0) {
		if (currvid == 0) {
			rvosteps = 0;
		} else {
//			dprintf("KERN_DEBUG PFX ph1: changing vid for rvo, requesting 0x%x\n",
//				(unsigned int)currvid - 1);
			if (decrease_vid_code_by_step(currvid - 1, 1))
				return 1;
			rvosteps--;
		}
	}

	if (query_current_values_with_pending_wait())
		return 1;

	if (savefid != currfid) {
		dprintf("KERN_ERR PFX ph1 err, currfid changed 0x%x\n", (unsigned int)currfid);
		return 1;
	}

//	dprintf("KERN_DEBUG PFX ph1 complete, currfid 0x%x, currvid 0x%x\n",
//		(unsigned int)currfid, (unsigned int)currvid);

	return 0;
}

/* Phase 2 - core frequency transition */
static inline int
core_frequency_transition(uint32 reqfid)
{
	uint32 vcoreqfid;
	uint32 vcocurrfid;
	uint32 vcofiddiff;
	uint32 savevid = currvid;

	if ((reqfid < HI_FID_TABLE_BOTTOM) && (currfid < HI_FID_TABLE_BOTTOM)) {
		dprintf("KERN_ERR PFX ph2 illegal lo-lo transition 0x%x 0x%x\n",
		       (unsigned int)reqfid, (unsigned int)currfid);
		return 1;
	}

	if (currfid == reqfid) {
		dprintf("KERN_ERR PFX ph2 null fid transition 0x%x\n", (unsigned int)currfid);
		return 0;
	}

//	dprintf("KERN_DEBUG PFX ph2 starting, currfid 0x%x, currvid 0x%x, reqfid 0x%x\n",
//		(unsigned int)currfid, (unsigned int)currvid, (unsigned int)reqfid);

	vcoreqfid = convert_fid_to_vco_fid(reqfid);
	vcocurrfid = convert_fid_to_vco_fid(currfid);
	vcofiddiff = vcocurrfid > vcoreqfid ? vcocurrfid - vcoreqfid
	    : vcoreqfid - vcocurrfid;

	while (vcofiddiff > 2) {
		if (reqfid > currfid) {
			if (currfid > LO_FID_TABLE_TOP) {
				if (write_new_fid(currfid + 2)) {
					return 1;
				}
			} else {
				if (write_new_fid
				    (2 + convert_fid_to_vco_fid(currfid))) {
					return 1;
				}
			}
		} else {
			if (write_new_fid(currfid - 2))
				return 1;
		}

		vcocurrfid = convert_fid_to_vco_fid(currfid);
		vcofiddiff = vcocurrfid > vcoreqfid ? vcocurrfid - vcoreqfid
		    : vcoreqfid - vcocurrfid;
	}

	if (write_new_fid(reqfid))
		return 1;

	if (query_current_values_with_pending_wait())
		return 1;

	if (currfid != reqfid) {
		dprintf("KERN_ERR PFX ph2 mismatch, failed fid transition, curr %x, req %x\n",
		       (unsigned int)currfid, (unsigned int)reqfid);
		return 1;
	}

	if (savevid != currvid) {
		dprintf("KERN_ERR PFX ph2 vid changed, save %x, curr %x\n", (unsigned int)savevid,
		       (unsigned int)currvid);
		return 1;
	}

//	dprintf("KERN_DEBUG PFX ph2 complete, currfid 0x%x, currvid 0x%x\n",
//		(unsigned int)currfid, (unsigned int)currvid);

	return 0;
}

/* Phase 3 - core voltage transition flow ... jump to the final vid. */
static inline int
core_voltage_post_transition(uint32 reqvid)
{
	uint32 savefid = currfid;
	uint32 savereqvid = reqvid;

//	dprintf("KERN_DEBUG PFX ph3 starting, currfid 0x%x, currvid 0x%x\n",
//		(unsigned int)currfid, (unsigned int)currvid);

	if (reqvid != currvid) {
		if (write_new_vid(reqvid))
			return 1;

		if (savefid != currfid) {
			dprintf("KERN_ERR PFX ph3: bad fid change, save %x, curr %x\n",
			       (unsigned int)savefid, (unsigned int)currfid);
			return 1;
		}

		if (currvid != reqvid) {
			dprintf("KERN_ERR PFX ph3: failed vid transition\n, req %x, curr %x",
			       (unsigned int)reqvid, (unsigned int)currvid);
			return 1;
		}
	}

	if (query_current_values_with_pending_wait())
		return 1;

	if (savereqvid != currvid) {
		dprintf("KERN_ERR PFX ph3 failed, currvid 0x%x\n", (unsigned int)currvid);
		return 1;
	}

	if (savefid != currfid) {
		dprintf("KERN_ERR PFX ph3 failed, currfid changed 0x%x\n",
			(unsigned int)currfid);
		return 1;
	}

//	dprintf("KERN_DEBUG PFX ph3 complete, currfid 0x%x, currvid 0x%x\n",
//		(unsigned int)currfid, (unsigned int)currvid);

	return 0;
}

/* Pairs of fid/vid values are appended to the version 1.4 PSB table. */
struct pst_s {
	uint8 fid;
	uint8 vid;
	int32 cv_factor;	// OCO BeOS : We cache cv_factor within  each state
};

/* Sort the fid/vid frequency table into ascending order by fid. The spec */
/* implies that it will be sorted by BIOS, but, it only implies it, and I */
/* prefer not to trust when I can check.                                  */
/* Yes, it is a simple bubble sort, but the PST is really small, so the   */
/* choice of algorithm is pretty irrelevant.                              */
static inline void
sort_pst(struct pst_s *ppst, uint32 numpstates)
{
	uint32 i;
	uint8 tempfid;
	uint8 tempvid;
	int swaps = 1;

	while (swaps) {
		swaps = 0;
		for (i = 0; i < (numpstates - 1); i++) {
			if (ppst[i].fid > ppst[i + 1].fid) {
				swaps = 1;
				tempfid = ppst[i].fid;
				tempvid = ppst[i].vid;
				ppst[i].fid = ppst[i + 1].fid;
				ppst[i].vid = ppst[i + 1].vid;
				ppst[i + 1].fid = tempfid;
				ppst[i + 1].vid = tempvid;
			}
		}
	}

	return;
}

// End of CPU states change functions 

/* processor's cpuid instruction support */
#define CPUID_PROCESSOR_SIGNATURE             1	/* function 1               */
#define CPUID_F1_FAM                 0x00000f00	/* family mask              */
#define CPUID_F1_XFAM                0x0ff00000	/* extended family mask     */
#define CPUID_F1_MOD                 0x000000f0	/* model mask               */
#define CPUID_F1_STEP                0x0000000f	/* stepping level mask      */
#define CPUID_XFAM_MOD               0x0ff00ff0	/* xtended fam, fam + model */
#define ATHLON64_XFAM_MOD            0x00000f40	/* xtended fam, fam + model */
#define OPTERON_XFAM_MOD             0x00000f50	/* xtended fam, fam + model */
#define ATHLON64_REV_C0                       8

#define CPUID_GET_MAX_CAPABILITIES   0x80000000
#define CPUID_FREQ_VOLT_CAPABILITIES 0x80000007
#define P_STATE_TRANSITION_CAPABLE            6
#define X86_VENDOR_AMD				"AuthenticAMD"

// Linux 2.6.9 const
#define CPUID_USE_XFAM_XMOD			 0x00000f00
#define CPUID_XFAM					 0x0ff00000	/* extended family */
#define CPUID_XFAM_K8						  0
#define CPUID_XMOD					 0x000f0000	/* extended model */
#define CPUID_XMOD_REV_E			 0x00020000

// This function is not complete : this function only recognize the processor
// in my notebook (aka ATHLON64_XFAM_MOD) and some Opterons (aka OPTERON_XFAM_MOD) 
// Many others AMD CPU IDs should support P-State transition.
// This a bit conservative, but it prevents mistakes. 

// 28/12/2004 Now, this function is based on linux 2.6.9 source code 

// Feel free to add new IDs and check them on your hardware, at your own risks.
// To do this, you should check AMD specifications :
//   - "Revision guide for AMD Athlon 64 and AMD Opteron Processors" (25759.pdf)
static inline int check_supported_cpu(void)
{
	cpuid_info cpuid_info_data;
	
	if (get_cpuid(&cpuid_info_data, 0, 1) == B_OK)
	{
		dprintf(DRIVER_NAME ": %s \n", cpuid_info_data.eax_0.vendorid);
		if (strncmp(cpuid_info_data.eax_0.vendorid, X86_VENDOR_AMD, 12) != 0)
		{
			dprintf(DRIVER_NAME ": Not an AMD processor !\n");
			return 0;
		};
	};
	if (get_cpuid(&cpuid_info_data, CPUID_PROCESSOR_SIGNATURE, 1) == B_OK)
	{
		unsigned int eax;
		eax = cpuid_info_data.regs.eax;
		// detection function based on linux 2.6.9 source code (powernow-k8.c)
		if (((eax & CPUID_USE_XFAM_XMOD) != CPUID_USE_XFAM_XMOD) ||
		    ((eax & CPUID_XFAM) != CPUID_XFAM_K8) ||
		    ((eax & CPUID_XMOD) > CPUID_XMOD_REV_E))
		{
			dprintf(DRIVER_NAME ": Processor cpuid %x not supported\n", eax);
			return 0;
		}
		// Old part from an older powernow-k8.c
/*		else if ((cpuid_info_data.eax_1.family == 0xf)&&(cpuid_info_data.eax_1.model == 0x4))
		{
			// TODO : add others AMD 64 IDs or use Haiku header ? ...
			dprintf(DRIVER_NAME ": This is an Athlon 64 \n");
			if (cpuid_info_data.eax_1.stepping < ATHLON64_REV_C0) 
			{
				dprintf(DRIVER_NAME ": Revision C0 or better "
				       "AMD Athlon 64 processor required\n");
				return 0;
			}	
		}
		else if ((cpuid_info_data.eax_1.family == 0xf)&&(cpuid_info_data.eax_1.model == 0x5))
		{
			dprintf(DRIVER_NAME ": AMD Opteron Processor found\n");
		}
		else
		{
			dprintf(DRIVER_NAME ": Unrecognized CPU\n");
			return 0;
		};*/
	};
	if (get_cpuid(&cpuid_info_data, CPUID_GET_MAX_CAPABILITIES, 1) == B_OK)
	{
		if ((cpuid_info_data.regs.eax < CPUID_FREQ_VOLT_CAPABILITIES))
		{
			dprintf(DRIVER_NAME ": No frequency change capabilities detected\n");
			return 0;
		};
	};
	if (get_cpuid(&cpuid_info_data, CPUID_FREQ_VOLT_CAPABILITIES, 1) == B_OK)
	{
		if ((cpuid_info_data.regs.edx & P_STATE_TRANSITION_CAPABLE) != P_STATE_TRANSITION_CAPABLE)
		{
			dprintf(DRIVER_NAME ": Power state transitions not supported\n");
			return 0;
		};
	};

	dprintf("Found AMD Athlon 64 / Opteron processor "
			"supporting p-state transitions\n");
	return 1;
}

// ACPI _PSS table parser

typedef struct Pkg_Lead_Byte
{
	UINT32_BIT PkgLength : 4;
	UINT8 Reserved : 2;
	UINT8 NbBytesForLength : 2;
} PKG_LEAD_BYTE;

union PkgLeadByte
{
	UINT8 data;
	PKG_LEAD_BYTE LeadByte;
};

typedef struct PackageLengthData
{
	UINT8 LeadByteLength : 4;
	UINT8 SecondByteLo : 4;
	UINT8 SecondByteHi : 4;
	UINT8 ThirdByteLo : 4;
	UINT8 ThirdByteHi : 4;	
	UINT8 QuarterByteLo : 4;
	UINT8 QuarterByteHi : 4;
	UINT8 Reserved : 4;
} PACKAGE_LENGTH_DATA;

union PackageLength
{
	PACKAGE_LENGTH_DATA data;
	UINT32 Length;
};


/* Retrieve the configured state from the driver's config file */
UINT8 GetAskedCPUState()
{
	void *handle;
	const char *AskedCPUState;
	UINT8 AskedCPUStateINT;
	
	// Reading config file
	handle = load_driver_settings("amd64");
	
	AskedCPUState = get_driver_parameter(handle, "cpustate", "0", "0");
	dprintf(DRIVER_NAME ": Asked CPU State (from config file) : %s \n", AskedCPUState);
	AskedCPUStateINT = atoi(AskedCPUState);
	dprintf(DRIVER_NAME ": Asked CPU State (after converting value) : %d \n", AskedCPUStateINT);	
	
	unload_driver_settings(handle);	
	
	return AskedCPUStateINT;
};

/* Static memory allocation for CPU states.
   This avoid dynamic memory allocation (aka bugs) for a non C programmer like me...
   3 because my AMD Athlon 64 3400+ supports four CPU states. */
/* Now, i am more comfortable with dynamic memory allocation... */
static struct pst_s *array_pst_s;
static UINT8 NbCPUStates;
	
struct pst_s ReadCPUState(UINT8 **data)
{
	struct pst_s CPUState;
	UINT8 *my_data;
	UINT8 i = 0;
	union PackageLength CpuStatePkgLength;
	union PkgLeadByte LeadByte;
	void *StartPos;
	UINT32 Pkg_Length;
	UINT32 *Value;
	UINT32 *ArrayData;
	my_data = *data;
	StartPos = my_data;	
	dprintf(DRIVER_NAME ": StartPos : %p \n", StartPos);

	my_data++;

	LeadByte.data = *my_data;
//	dprintf(DRIVER_NAME ": Length : %x ; %x ; %x \n", LeadByte.LeadByte.PkgLength, *my_data, LeadByte.data);
//	dprintf(DRIVER_NAME ": NbBytesForLength : %x \n", LeadByte.LeadByte.NbBytesForLength);	
	CpuStatePkgLength.Length = 0;
	CpuStatePkgLength.data.LeadByteLength = LeadByte.LeadByte.PkgLength;	
//	Pkg_Length = CpuStatePkgLength.Length;
	Pkg_Length = *my_data;
//	dprintf(DRIVER_NAME ": Pkg_Length : %x \n", Pkg_Length);
//	dprintf(DRIVER_NAME ": Length : %x \n", *my_data);

	my_data++;
	ArrayData = malloc(*my_data * sizeof(UINT32));

	while (my_data < (UINT8 *)(StartPos + (Pkg_Length)))
	{
		switch(*my_data)
		{
			case 0x0c:												
				my_data++;
				Value = (UINT32 *) my_data;
				ArrayData[i] = *Value;
//				dprintf(DRIVER_NAME ": Value : %d = 0x%x \n", i, ArrayData[i]);
				i++;				
				my_data++;
				my_data++;
				my_data++;
				my_data++;
				break;
			default: my_data++;
		};
	};
	// set the new parser position (to avoid reading same data again and again)
	*data = my_data;
	CPUState.fid = ArrayData[4] & FID_MASK;;
	CPUState.vid = (ArrayData[4] >> VID_SHIFT) & VID_MASK;
	
	dprintf(DRIVER_NAME ": fid : %d ; vid : %d \n", CPUState.fid, CPUState.vid);
	dprintf(DRIVER_NAME ": frequence : %d \n", find_khz_freq_from_fid(CPUState.fid));
	dprintf(DRIVER_NAME ": voltage : %d \n", find_millivolts_from_vid(CPUState.vid));
	
	return CPUState;
};

status_t ChangeCPUState(UINT8 AskedCPUState)
{
	if (pending_bit_stuck()) {
		dprintf(DRIVER_NAME ": KERN_ERR PFX drv targ fail: change pending bit set\n");
		return -EIO;
	}
	else
	{
		dprintf(DRIVER_NAME ": Ok \n");
	};
	if (query_current_values_with_pending_wait())
		return -EIO;

//	dprintf(DRIVER_NAME ": KERN_DEBUG PFX targ: curr fid 0x%x, vid 0x%x\n",
//		(unsigned int)currfid, (unsigned int)currvid);
	dprintf(DRIVER_NAME ": current frequence : %d \n", find_khz_freq_from_fid(currfid));
	dprintf(DRIVER_NAME ": current voltage : %d \n", find_millivolts_from_vid(currvid));
	
	dprintf(DRIVER_NAME ": Changing CPU speed \n");
	if (AskedCPUState <= (NbCPUStates - 1))
	{
		if ((currfid != array_pst_s[AskedCPUState].fid)&&(currvid != array_pst_s[AskedCPUState].vid))
		{
			transition_fid_vid(array_pst_s[AskedCPUState].fid, array_pst_s[AskedCPUState].vid);								
			dprintf(DRIVER_NAME ": CPU speed Changed \n");
			return B_OK;
		}
		else
		{
			dprintf(DRIVER_NAME ": Already in good state ! \n");
			return B_OK;
		};
	}
	else
	{
		dprintf(DRIVER_NAME ": CPUState exceed limits. \n");
		return B_OK;		
	};								
}

// From CPU_fix driver by Michael Lotz

// variables
extern int32 cv_factor;
extern int64 system_time_base;

// kernel functions
extern void get_cpu_info();

// our functions
extern uint64	rdtsc();
extern void		patch_get_cpu_info();

#define TIMER_CLKNUM_HZ (14318180 / 12)
#define out8(value, port) asm volatile ("outb %%al,%%dx" : : "a" (value), "d" (port))
#define in8(port) ({ uint8 _v; asm volatile ("inb %%dx,%%al" : "=a" (_v) : "d" (port)); _v; })

/* start Haiku extract */

uint32
calculate_cpu_conversion_factor()
{
	uint32 s_low, s_high;
	uint32 low, high;
	uint32 expired;
	uint32 result;
	uint64 t1, t2;
	uint64 p1, p2, p3;
	double r1, r2, r3;
	
	cpu_status former = disable_interrupts();
	
	out8(0x34, 0x43);	/* program the timer to count down mode */
	out8(0xff, 0x40);	/* low and then high */
	out8(0xff, 0x40);

	/* quick sample */
quick_sample:
	do {
		out8(0x00, 0x43); /* latch counter value */
		s_low = in8(0x40);
		s_high = in8(0x40);
	} while(s_high != 255);
	t1 = rdtsc();
	do {
		out8(0x00, 0x43); /* latch counter value */
		low = in8(0x40);
		high = in8(0x40);
	} while (high > 224);
	t2 = rdtsc();

	p1 = t2-t1;
	r1 = (double)(p1) / (double)(((s_high << 8) | s_low) - ((high << 8) | low));

	/* not so quick sample */
not_so_quick_sample:
	do {
		out8(0x00, 0x43); /* latch counter value */
		s_low = in8(0x40);
		s_high = in8(0x40);
	} while (s_high!= 255);
	t1 = rdtsc();
	do {
		out8(0x00, 0x43); /* latch counter value */
		low = in8(0x40);
		high = in8(0x40);
	} while (high> 192);
	t2 = rdtsc();
	p2 = t2-t1;
	r2 = (double)(p2) / (double)(((s_high << 8) | s_low) - ((high << 8) | low));
	if ((r1/r2) > 1.01) {
		//dprintf("Tuning loop(1)\n");
		goto quick_sample;
	}
	if ((r1/r2) < 0.99) {
		//dprintf("Tuning loop(1)\n");
		goto quick_sample;
	}

	/* slow sample */
	do {
		out8(0x00, 0x43); /* latch counter value */
		s_low = in8(0x40);
		s_high = in8(0x40);
	} while (s_high!= 255);
	t1 = rdtsc();
	do {
		out8(0x00, 0x43); /* latch counter value */
		low = in8(0x40);
		high = in8(0x40);
	} while (high > 128);
	t2 = rdtsc();

	p3 = t2-t1;
	r3 = (double)(p3) / (double)(((s_high << 8) | s_low) - ((high << 8) | low));
	if ((r2/r3) > 1.01) {
		//dprintf("Tuning loop(2)\n");
		goto not_so_quick_sample;
	}
	if ((r2/r3) < 0.99) {
		//dprintf("Tuning loop(2)\n");
		goto not_so_quick_sample;
	}

	expired = ((s_high << 8) | s_low) - ((high << 8) | low);
	p3 *= TIMER_CLKNUM_HZ;
	{
		unsigned i;
		unsigned k;
		uint64 C;
		uint64 x0;
		uint64 x1;
		uint64 a, b, c;

		/* first calculate k*/
		k = 0;
		for (i = 12; i < 16; i++) {
			if (expired & (1<<i))
				k = i - 11;
		}

		C = 1000000ULL << 32;
		x0 = expired >> k;
		x1 = expired & ((1 << k) - 1);

		a = ((C * x0) / p3) << k;
		b = (((C * x0) % p3) << k) / p3;
		c = (C * x1) / p3;
		
		result = a + b + c;
	}

	if (p3 / expired / 1000000000LL)
		dprintf("cpu_fix: CPU at %Ld.%03Ld GHz\n", p3 / expired / 1000000000LL, ((p3 / expired) % 1000000000LL) / 1000000LL);
	else
		dprintf("cpu_fix: CPU at %Ld.%03Ld MHz\n", p3 / expired / 1000000LL, ((p3 / expired) % 1000000LL) / 1000LL);

	restore_interrupts(former);
	return result;
}

/* end Haiku extract */

void fix_timer(UINT8 cpu_state)
{
	double temp;
	int64 tsc;
	int32 new_cv_factor;

	dprintf("cpu_fix: fixing cv_factor...\n");

	// when changing the cv_factor we also have to change the system_time_base.
	// otherwise the calculation based on the new cv_factor will be larger/smaller
	// and we jump in time (uptime with new cv_factor would be different).
	
	// OCO to save time with dynamic cpu state change, we try to cache cv_factor
	// Good idea : no more problems with sound in vlc (except the first call for each cpu state) !
	// Workaround : we iterate over each state when opening the driver (see "my_device_open" 
	// function). So, there is only one point where the sound is ugly.
	if (array_pst_s[cpu_state].cv_factor == 0)
	{
		// Not yet calculated for this cpu state
		new_cv_factor = calculate_cpu_conversion_factor();
		array_pst_s[cpu_state].cv_factor = new_cv_factor;
	}
	else
	{
		new_cv_factor = array_pst_s[cpu_state].cv_factor;
	};
	dprintf("cpu_fix: new cv_factor : %ld \n", new_cv_factor);

	tsc = rdtsc();
	temp = tsc;
	temp += system_time_base;
	temp *= cv_factor;
	temp /= new_cv_factor;
	temp -= tsc;
	system_time_base = (int64)temp;
	cv_factor = new_cv_factor;
}

// End cpu_fix

typedef struct daemon_args
{
	UINT8 current_state;
	bigtime_t prev_time;
	bigtime_t prev_active;
} DAEMON_ARGS;

struct daemon_args *mes_args;

static void daemon_func(void *arg, int freq)
{
	system_info sys_info;
	int cpu_time;
	bigtime_t now, part1, part2;
	struct daemon_args *my_args = (struct daemon_args*)arg;

	get_system_info(&sys_info);

	now = system_time();
//	now = real_time_clock_usecs();

	part1 = (sys_info.cpu_infos[0].active_time - mes_args->prev_active);
	part2 = (now - mes_args->prev_time);

	// Floating point values are not well supported it seems... ???
	// So i multiply the numerator to have a usable non floating point value... 
	// The precision is enough for what we want !
	cpu_time = (100 * part1) / (part2);
//	dprintf(DRIVER_NAME ": active_time : %d, %d, %d, %d \n", sys_info.cpu_infos[0].active_time, mes_args->prev_active, now, mes_args->prev_time);	

	my_args->prev_active = sys_info.cpu_infos[0].active_time;	

	if (cpu_time >= 80)
	{		
		if (my_args->current_state < (NbCPUStates - 1))
		{		
			dprintf(DRIVER_NAME ": cpu_time : %d \n", cpu_time);
			dprintf(DRIVER_NAME " more than 80 percent \n");
			my_args->current_state = my_args->current_state + 1;
			ChangeCPUState(my_args->current_state);			
			fix_timer(my_args->current_state);
			// during cpu state change, time is not reported accurately. So, we simply
			// discard cpu activity during this period.
			// We initialize previous activity with current activity (after applying time fixe).
			get_system_info(&sys_info);
			my_args->prev_active = sys_info.cpu_infos[0].active_time;	
			now = system_time();			
		};
	};
	if (cpu_time <= 30)
	{
		if (my_args->current_state > 0)
		{
			dprintf(DRIVER_NAME ": cpu_time : %d \n", cpu_time);
			dprintf(DRIVER_NAME " less than 30 percent \n");
			my_args->current_state = my_args->current_state - 1;		
			ChangeCPUState(my_args->current_state);
			fix_timer(my_args->current_state);
			// during cpu state change, time is not reported accurately (not yet ?). So, we simply
			// discard cpu activity during this period.
			// We initialize previous activity with current activity (after applying time fixe).
			get_system_info(&sys_info);
			my_args->prev_active = sys_info.cpu_infos[0].active_time;	
			now = system_time();
		};
	};

	my_args->prev_time = now;
}

void* find_str(char * str, size_t str_length, void * start_address, void *max_addr)
{
	do
	{
		if (strncmp(start_address, str, str_length) == 0)
		{
			return start_address;
		};
		start_address++;
	}
	while(start_address < max_addr);
	return 0;
}

/* The code below try to find power states supported by the CPU.
   This information is extracted from the BIOS and the ACPI tables.
   Unfortunately, this works only on my laptop (Acer Aspire 1510).
   This is a quick, dirty and hacky micro ACPI parser.
   The right way should be to use a real ACPI parser (ACPI CA for example).
   
   The array that contains cpu state transitions informations (array_pst_s)
   could be initialized using data extracted from AMD datasheets.
   
   hint : 
   		if you write your own ExtractCpuPowerState for your machine,
        you have to initialize this two global variables, with appropriate values :
          
			static struct pst_s *array_pst_s;
			static UINT8 NbCPUStates;
		
		if not, the driver may not work... (you should understand "will crash your machine")
		
		Do this at your own risk !
          
*/
bool ExtractCpuPowerState()
{
	area_id area;
	area_id ACPI_area;
	area_id SSDT_area;
	void *paddr;
	void *paddr2;
	void *paddr_ssdt;
	void *virtual_address;
	void *virtual_address_ssdt; 
	void *virtual_address_ssdt2;
	
	void *virtual_address_tmp;
	size_t	size;
	UINT32 j;
		
	RSDP_DESCRIPTOR* RSDP;
	RSDT_DESCRIPTOR_REV1* RSDT;
	SSDT_DESCRIPTOR* SSDT;

	void *pss_addr;
	UINT8 *data;	
	
	UINT32 PSS_Length;
	void *CurrentPos;
	
	union PackageLength PSSPkgLength;
	union PkgLeadByte LeadByte;
	
	area_info my_area_info;
	bool result = false;	
	
	area = find_area("rom bios");
	if (area > 0)
	{
		dprintf(DRIVER_NAME ": Found ROM BIOS area \n");
		if (get_area_info(area, &my_area_info) == B_OK)
		{
			paddr = my_area_info.address;
			do
			{
				if (strncmp(paddr, RSDP_SIG, (size_t) 8)  == 0)
				{
					dprintf(DRIVER_NAME ": found RSDP_SIG at %p, %p, %p \n", 
						paddr, my_area_info.address, (void *)(paddr - my_area_info.address));
					RSDP = (RSDP_DESCRIPTOR*) paddr;
					dprintf(DRIVER_NAME ": %p \n", (void *)RSDP->RsdtPhysicalAddress);
					dprintf(DRIVER_NAME ": %s \n", RSDP->Signature);
					paddr2 = (void *) (RSDP->RsdtPhysicalAddress & ~((long) B_PAGE_SIZE - 1));
					dprintf(DRIVER_NAME ": paddr2 : %p \n", paddr2);
					size = ( B_PAGE_SIZE/* * 32*/ + B_PAGE_SIZE - 1) & ~((size_t) B_PAGE_SIZE /** 32*/ - 1);
					ACPI_area = map_physical_memory("ACPI_area", paddr2, 
					                              size, B_ANY_KERNEL_ADDRESS, 
					                              B_READ_AREA, &virtual_address);
					if (ACPI_area > 0)
					{
						virtual_address_tmp = virtual_address + 4096;
						do
						{
							if (strncmp(virtual_address, "RSDT", (size_t) 4) == 0)
							{
								dprintf(DRIVER_NAME ": RSDT trouvé \n");
								RSDT = (RSDT_DESCRIPTOR_REV1*) virtual_address;
								dprintf(DRIVER_NAME ": %s \n", RSDT->OemTableId);
								dprintf(DRIVER_NAME ": Nombre d'entrées : %d \n", (RSDT->Length - 36)/4);
								dprintf(DRIVER_NAME ": %p \n", (void *)RSDT->TableOffsetEntry[0]);
								dprintf(DRIVER_NAME ": %p \n", (void *)RSDT->TableOffsetEntry[1]);
								dprintf(DRIVER_NAME ": %p \n", (void *)RSDT->TableOffsetEntry[2]);
								dprintf(DRIVER_NAME ": %p \n", (void *)RSDT->TableOffsetEntry[3]);
								
								paddr_ssdt = (void *) (RSDT->TableOffsetEntry[2] & ~((long) B_PAGE_SIZE - 1));
								virtual_address_ssdt = paddr_ssdt;
								size = ( B_PAGE_SIZE + B_PAGE_SIZE - 1) & ~((size_t) B_PAGE_SIZE - 1);
								SSDT_area = map_physical_memory("SSDT", paddr_ssdt,
								                                20 * B_PAGE_SIZE, B_ANY_KERNEL_ADDRESS,
								                                B_READ_AREA, &virtual_address_ssdt);
								if (SSDT_area > 0)
								{
									dprintf(DRIVER_NAME ": Mapping SSDT Ok \n");
									// Calculate the position of SSDT table in the SSDT_area
									virtual_address_ssdt2 = (void *)((void *)RSDT->TableOffsetEntry[2] - (void *)paddr_ssdt);
									dprintf(DRIVER_NAME ": %p \n", virtual_address_ssdt2);
									dprintf(DRIVER_NAME ": %p \n", virtual_address_ssdt);
									virtual_address_ssdt = virtual_address_ssdt + (int)virtual_address_ssdt2;
									dprintf(DRIVER_NAME ": %p \n", virtual_address_ssdt);
									SSDT = (SSDT_DESCRIPTOR*) virtual_address_ssdt;
									if (strncmp(SSDT->Signature, "SSDT", (size_t) 4) == 0)
									{
										dprintf(DRIVER_NAME ": SSDT found \n");
									};
									dprintf(DRIVER_NAME ": Signature : %s \n", SSDT->Signature);
									dprintf(DRIVER_NAME ": OemId : %s \n", SSDT->OemId);
									dprintf(DRIVER_NAME ": OemTableId : %s \n", SSDT->OemTableId);
									dprintf(DRIVER_NAME ": %p \n", SSDT);

									// I search _PSS table near SSDT start (in the next 2000 bytes)
									// I don't know if it is always a good guess...
									pss_addr = find_str("_PSS", 4, virtual_address_ssdt, (void *) virtual_address_ssdt + 2000);
									if (pss_addr > 0)
									{										
										dprintf(DRIVER_NAME ": ACPI _PSS table found !\n");										
										pss_addr++;
										pss_addr++;
										pss_addr++;
										pss_addr++;										
										
										dprintf(DRIVER_NAME ": %p \n", pss_addr);
										data = (UINT8 *) pss_addr;

										dprintf(DRIVER_NAME ": Start of package : %x \n", *data);										
										data++; 
										dprintf(DRIVER_NAME ": LeadByte data: %x \n", *data);	
										LeadByte.data = *data;
										dprintf(DRIVER_NAME ": NbBytesForLength :%x \n", LeadByte.LeadByte.NbBytesForLength);
										dprintf(DRIVER_NAME ": PkLength : %x \n", LeadByte.LeadByte.PkgLength);
										PSSPkgLength.Length = 0;
										PSSPkgLength.data.LeadByteLength = LeadByte.LeadByte.PkgLength;
										dprintf(DRIVER_NAME ": Reserved : %x \n", LeadByte.LeadByte.Reserved);
										dprintf(DRIVER_NAME ": Length : %x \n", LeadByte.LeadByte.PkgLength);										
										data++; 
										dprintf(DRIVER_NAME ": SecondByteLength : %x \n", *data);
										PSSPkgLength.data.SecondByteLo = *data;
										dprintf(DRIVER_NAME ": PSSPkgLength : %x \n", PSSPkgLength.Length);
										dprintf(DRIVER_NAME ": %x \n", *data);
										PSSPkgLength.data.SecondByteHi = *data << 4;
										dprintf(DRIVER_NAME ": PSSPkgLength : %x \n", PSSPkgLength.Length);

										CurrentPos = data;										
										data++; 
										NbCPUStates = *data;
										dprintf(DRIVER_NAME ": Number of CPU states : %x \n", NbCPUStates);
										// Alloc enough memory for each state data 
										array_pst_s = malloc(*data * (sizeof(struct pst_s)));

										data++; 
										PSS_Length = PSSPkgLength.Length;
										j = 0;
										while(data < (UINT8 *)(CurrentPos + PSS_Length))
										{
											dprintf(DRIVER_NAME ": %x \n", *data);
											switch(*data)
											{
												case 0x12:
													array_pst_s[j] = ReadCPUState(&data);
													j++;
													break;
												default: data++;
											};
										};
										sort_pst(array_pst_s, NbCPUStates);	
										dprintf(DRIVER_NAME ": Dynamic after sorting pst : \n");
										for (j = 0; j < NbCPUStates; j++) 
										{
											// Initialize cv_factor to 0 to check if the cv_factor
											// was already calculated
											array_pst_s[j].cv_factor = 0;
											dprintf(DRIVER_NAME ": KERN_INFO PFX"
											       "   %d : fid 0x%x, vid 0x%x\n", j,
											       array_pst_s[j].fid, array_pst_s[j].vid);
										};
										// cpu state informations are loaded in the array!
										result = true;
									}
									else
									{
										dprintf(DRIVER_NAME ": _PSS not found !\n");
									};
									//delete_area(SSDT_area);
								};
								break;
							};
							virtual_address++;
						}
						while (virtual_address < ((void*) virtual_address_tmp));
						
						delete_area(ACPI_area);
					};
					break;
				};
				paddr++;
			}
			while (paddr < ((void*) my_area_info.address + 0x20000));
		};
	};

	dprintf(DRIVER_NAME ": End parsing ROM BIOS area \n");	
	return result;
}

/* ----------
	init_driver - optional function - called every time the driver
	is loaded.
----- */

_EXPORT status_t
init_driver (void)
{

	UINT8 AskedCPUState;

	dprintf (DRIVER_NAME ": init_driver()\n");
	
	if (check_supported_cpu() == 0)
	{
		return B_ERROR;
	};
	
	// Get default cpu_state stored in the config file.
	AskedCPUState = GetAskedCPUState();	
	
	if (ExtractCpuPowerState())
	{
		// Now, we have all informations needed to initiate the
		// CPU State transition.
		// change the CPU state if not already in good state
		ChangeCPUState(AskedCPUState);
		fix_timer(AskedCPUState);
		return B_OK;		
	}
	else
	{
		return B_ERROR;
	};	
}


/* ----------
	uninit_driver - optional function - called every time the driver
	is unloaded
----- */

_EXPORT void
uninit_driver (void)
{
	dprintf (DRIVER_NAME ": uninit_driver()\n");
	free(array_pst_s);
}

	
/* ----------
	my_device_open - handle open() calls
----- */

static status_t
my_device_open (const char *name, uint32 flags, void** cookie)
{
	UINT8 j;

	dprintf (DRIVER_NAME ": open(\"%s\")\n", name);

	// Cache cv_factor for each cpu state
	// So timing problem occurs only time when opening the driver
	dprintf(DRIVER_NAME ": cv_factor caching \n");
	for (j = 0; j < NbCPUStates; j++) 
	{
		// Initialize cv_factor to 0 to check if the cv_factor
		// was already calculated
		array_pst_s[j].cv_factor = 0;
		ChangeCPUState(j);
		fix_timer(j);
		dprintf(DRIVER_NAME ": KERN_INFO PFX"
		       "   %d : fid 0x%x, vid 0x%x\n", j,
		       array_pst_s[j].fid, array_pst_s[j].vid);
		dprintf(DRIVER_NAME ": KERN_INFO PFX"
		       "   %d : cv factor %ld\n", j,
		       array_pst_s[j].cv_factor);
	};

	dprintf(DRIVER_NAME ": end cv_factor caching \n");

	mes_args = (struct daemon_args *)malloc(sizeof(struct daemon_args));	
	memset(&(mes_args[0]), 0, sizeof(struct daemon_args));

	*cookie = &mes_args[0];
	// we call the daemon function about every second to check if a cpu state change is needed
	register_kernel_daemon(daemon_func, &mes_args[0], 1);

	return B_OK;
}


/* ----------
	my_device_read - handle read() calls
----- */

static status_t
my_device_read (void* cookie, off_t position, void *buf, size_t* num_bytes)
{
	dprintf (DRIVER_NAME ": read():\n");
	return B_OK;
}


/* ----------
	my_device_write - handle write() calls
----- */

static status_t
my_device_write (void* cookie, off_t position, const void* buf, size_t* num_bytes)
{
	dprintf (DRIVER_NAME ": read():\n");
	return B_OK;
}


/* ----------
	my_device_control - handle ioctl calls
----- */

static status_t
my_device_control (void* cookie, uint32 op, void* arg, size_t len)
{
	dprintf (DRIVER_NAME ": ioctl(%d)\n", (int)op);
	return B_ERROR;
}


/* ----------
	my_device_close - handle close() calls
----- */

static status_t
my_device_close (void* cookie)
{
	dprintf (DRIVER_NAME ": close()\n");

	return B_OK;
}

/* -----
	my_device_free - called after the last device is closed, and after
	all i/o is complete.
----- */

static status_t
my_device_free (void* cookie)
{
	UINT8 AskedCPUState;
	struct daemon_args *my_args = (struct daemon_args*)cookie;

	dprintf (DRIVER_NAME ": free()\n");
	dprintf(DRIVER_NAME " unregistering daemon test\n"); 
	unregister_kernel_daemon(daemon_func, &my_args[0]);	
	dprintf(DRIVER_NAME " End unregistering daemon\n");

	// When freeing, we return in the default CPU state
	AskedCPUState = GetAskedCPUState();
	ChangeCPUState(AskedCPUState);
	fix_timer(AskedCPUState);

	free(mes_args);
	return B_OK;
}


/* -----
	null-terminated array of device names supported by this driver
----- */

static const char *my_device_name[] = {
	DEVICE_NAME,
	NULL
};

/* -----
	function pointers for the device hooks entry points
----- */

static device_hooks my_device_hooks = {
	my_device_open, 			/* -> open entry point */
	my_device_close, 			/* -> close entry point */
	my_device_free,			/* -> free cookie */
	my_device_control, 		/* -> control entry point */
	my_device_read,			/* -> read entry point */
	my_device_write			/* -> write entry point */
};

/* ----------
	publish_devices - return a null-terminated array of devices
	supported by this driver.
----- */

_EXPORT const char**
publish_devices()
{
	dprintf (DRIVER_NAME ": publish_devices()\n");
	return my_device_name;
}

/* ----------
	find_device - return ptr to device hooks structure for a
	given device name
----- */

static int
lookup_device_name (const char *name)
{
	int	i;

	for (i = 0; my_device_name [i] != NULL; i++)
		if (strcmp (name, my_device_name [i]) == 0)
			return i;
	return -1;
}

_EXPORT device_hooks*
find_device(const char* name)
{
	dprintf (DRIVER_NAME ": find_device(\"%s\")\n", name);
	if (lookup_device_name (name) < 0)
		return NULL;
	return &my_device_hooks;
}
