#include <stdlib.h>
#include <stdio.h>

#include "Globals.h"
#include "PeakFile.h"
#include "VMSystem.h"

CPeakFile Peak;

#define PEAK_BUFFER_SIZE		64*256	// 64Kb

// ============================================================
CPeakFile::CPeakFile()
{
	buffer_left = NULL;
	buffer_right = NULL;
	buffer = NULL;
}

// ============================================================
CPeakFile::~CPeakFile()
{
	if (buffer)			delete[] buffer;
	if (buffer_left)	free(buffer_left);
	if (buffer_right)	free(buffer_right);
}

// ============================================================
void CPeakFile::Init(int32 size, bool mono)
{
	if (buffer)
		delete[] buffer;
	buffer = new float[PEAK_BUFFER_SIZE];

	m_size = size;
	size = (size >> 7) + 1;
	m_mono = mono;

	int64 mem = size * 4 +16;	// 2 int16's for each channel
	if (!mono)	mem *= 2;

	int16 *p = (int16*)realloc(buffer_left, mem);
	if (p){
		buffer_left = p;				// new block
		memset( buffer_left, 0, mem);	// wipe buffer
	}else{
		(new BAlert(NULL,Language.get("MEM_ERROR"),Language.get("OK")))->Go();
		be_app->Quit();
	}
	
	if (!mono){
		int16 *p = (int16*)realloc(buffer_right, mem);
		if (p){
			buffer_right = p;				// new block
			memset( buffer_right, 0, mem);	// wipe buffer
		}else{
			(new BAlert(NULL,Language.get("MEM_ERROR"),Language.get("OK")))->Go();
			be_app->Quit();
		}
	}else{
		if (buffer_right)	free(buffer_right);
		buffer_right = NULL;
	}
}

// ============================================================
void CPeakFile::CreatePeaks(int32 start, int32 end, int32 progress)
{
	float min, max, max_r, min_r;
	int32 to, ii;

	start &= 0xfffffff8;	// mask off 1st 7 bits to round on 128 bytes
	int32 p_add = 0, p_count = 0, count = 0;
	if (progress){	// init progress process
		p_count = (end-start)/(100*128);
		p_add = progress/100;
		if (!m_mono)	p_add <<=1;
	}

	if (m_mono)		// mono
	{
		float *p = buffer;
		int32 index = 0;
		VM.ReadBlockAt(start, p, PEAK_BUFFER_SIZE);

		for (int32 i=start; i<=end; i+=128){
			min = max = 0.0;
			to = i+127;
			if (to>Pool.size)	to = Pool.size;
			for (int32 x=i; x<=to; x++){
				if (p[index]>max)	max = p[index];
				if (p[index]<min)	min = p[index];
				index++;
				if (index == PEAK_BUFFER_SIZE){
					index = 0;
					VM.ReadBlock(p, PEAK_BUFFER_SIZE);
				}
			}
			ii = i>>6;
			buffer_left[ii] = (int16)(min * 32767);
			buffer_left[ii+1] = (int16)(max * 32767);

			if (progress && count--<0){	count = p_count;	Pool.ProgressUpdate( p_add ); }
		}
	}
	else	// Stereo
	{
		float *p = buffer;
		int32 index = 0;
		VM.ReadBlockAt(start*2, p, PEAK_BUFFER_SIZE);

		for (int32 i=start; i<=end; i+=128){
			min = max = 0.0;
			min_r = max_r = 0.0;
			to = i+127;
			if (to>Pool.size)	to = Pool.size;
			for (int32 x=i; x<=to; x++){
				if (p[index]>max)		max = p[index];
				if (p[index]<min)		min = p[index];
				if (p[index+1]>max_r)	max_r = p[index+1];
				if (p[index+1]<min_r)	min_r = p[index+1];
				index+=2;
				if (index >= PEAK_BUFFER_SIZE){
					index = 0;
					VM.ReadBlock(p, PEAK_BUFFER_SIZE);
				}
			}
			ii = i>>6;
			buffer_left[ii] = (int16)(min * 32767);
			buffer_left[ii+1] = (int16)(max * 32767);
			buffer_right[ii] = (int16)(min_r * 32767);
			buffer_right[ii+1] = (int16)(max_r * 32767);

			if (progress && count--<0){	count = p_count;	Pool.ProgressUpdate( p_add ); }
		}
	}
	Pool.update_peak = true;
}

// ============================================================
void CPeakFile::MonoBuffer(float *out, int32 start, int32 end, float w)
{
	if (!buffer_left || !m_mono)	return;
	
	float step = (end - start)/w;
	int32 iStep = (int32)step;
	int32 index, to;
	
	int32 nBufferSize = MIN( PEAK_BUFFER_SIZE, end-start);

	if ( iStep < 1 )
	{
		float *p = buffer;
		VM.ReadBlockAt(start, p, nBufferSize);

		for (int32 x = 0; x<w; x++){
			index = (int32)(x * step);
			float fTemp = p[index];

			if (fTemp>1.0f)	fTemp = 1.0f;
			else if (fTemp<-1.0f)	fTemp = -1.0f;
			*out++ = fTemp;
			*out++ = 0.0f;
		}
	}else
	if ( iStep < 8 )
	{	float min, max;
		float *p = buffer;
		int32 current_buffer = start/nBufferSize;
		int32 idx = 0;
		VM.ReadBlockAt( current_buffer * nBufferSize, p, nBufferSize);

		for (int32 x = 0; x<w; x++){
			index = start + (int32)(x * step);
			to = index + iStep;	if (to>m_size)	to = m_size;

			min = max = 0;
			for (int32 i=index; i<=to; i++){
				if (i/nBufferSize > current_buffer){
					current_buffer++;
					VM.ReadBlock(p, nBufferSize);
				}
				idx = i % nBufferSize;
				if (p[idx]>max)	max = p[idx];
				if (p[idx]<min)	min = p[idx];
			}
			if (max > -min)	*out++ = MIN(max, 1);
			else			*out++ = MAX(min, -1);
			*out++ = 0.0f;
		}
	}else
	if ( iStep < 128 )
	{	float min, max;
		float *p = buffer;
		int32 current_buffer = start/nBufferSize;
		int32 idx = 0;
		VM.ReadBlockAt( current_buffer * nBufferSize, p, nBufferSize);

		for (int32 x = 0; x<w; x++){
			index = start + (int32)(x * step);
			to = index + iStep;	if (to>m_size)	to = m_size;

			min = 1.0f;
			max = -1.0f;
			for (int32 i=index; i<=to; i++){
				if (i/nBufferSize > current_buffer){
					current_buffer++;
					VM.ReadBlock(p, nBufferSize);
				}
				idx = i % nBufferSize;
				if (p[idx]>max)	max = p[idx];
				if (p[idx]<min)	min = p[idx];
			}

			*out++ = min;
			*out++ = max;
		}
	}
	else
	{	int16 min, max;
		for (int32 x = 0; x<w; x++){
			index = start + (int32)(x * step);
			to = index + iStep;	if (to>m_size)	to = m_size;
			index >>= 6;	index &= 0xfffffffe;
			to >>= 6;		to &= 0xfffffffe;

			min = max = 0;
			for (int32 i=index; i<=to; i+=2){
				if (buffer_left[i]<min)		min = buffer_left[i];
				if (buffer_left[i+1]>max)	max = buffer_left[i+1];
			}

			*out++ = min/32767.0;
			*out++ = max/32767.0;
		}
	}
}

// ============================================================
void CPeakFile::StereoBuffer(float *out, float *out_r, int32 start, int32 end, float w)
{
	if (!buffer_left ||!buffer_right || m_mono)
		return;
	
	float step = (end - start)/w;
	int32 iStep = (int32)step;
	int32 index, to;
	
	int32 nBufferSize = MIN( PEAK_BUFFER_SIZE, (end-start)*2);
	if ( iStep < 1 )
	{
		float *p = buffer;
		VM.ReadBlockAt(start*2, p, nBufferSize);

		for (int32 x = 0; x<w; x++){
			index = (int32)(x * step);
			float fTemp = p[index*2];
			float fTempR = p[index*2+1];

			if (fTemp>1.0f)			fTemp = 1.0f;
			else if (fTemp<-1.0f)	fTemp = -1.0f;
			if (fTempR>1.0f)		fTempR = 1.0f;
			else if (fTempR<-1.0f)	fTempR = -1.0f;
			*out++ = fTemp;
			*out++ = 0.0f;
			*out_r++ = fTempR;
			*out_r++ = 0.0f;
		}
	}else
	if ( iStep < 8 )
	{	float min, max, min_r, max_r;
		float *p = buffer;
		int32 current_buffer = start*2/nBufferSize;
		int32 idx = 0;
		VM.ReadBlockAt( current_buffer * nBufferSize, p, nBufferSize);

		for (int32 x = 0; x<w; x++){
			index = start + (int32)(x * step);
			to = index + iStep;	if (to>m_size)	to = m_size;

			index *= 2;
			to *= 2;

			min = max = min_r = max_r = 0;
			for (int32 i=index; i<=to; i+=2){
				if (i/nBufferSize > current_buffer){
					current_buffer++;
					VM.ReadBlock(p, nBufferSize);
				}
				idx = i % nBufferSize;
				if (p[idx]>max)	max = p[idx];
				if (p[idx]<min)	min = p[idx];
				idx++;
				if (p[idx]>max_r)	max_r = p[idx];
				if (p[idx]<min_r)	min_r = p[idx];
			}
			if (max > -min)	*out++ = MIN(max, 1);
			else			*out++ = MAX(min, -1);
			*out++ = 0.0f;

			if (max_r > -min_r)	*out_r++ = MIN(max_r, 1);
			else				*out_r++ = MAX(min_r, -1);
			*out_r++ = 0.0f;
		}
	}else
	if (iStep <128)
	{	float min, max, min_r, max_r;
		float *p = buffer;
		int32 current_buffer = start*2/nBufferSize;
		int32 idx = 0;
		VM.ReadBlockAt( current_buffer * nBufferSize, p, nBufferSize);

		for (int32 x = 0; x<w; x++){
			index = start + (int32)(x * step);
			to = index + iStep;	if (to>m_size)	to = m_size;

			index *= 2;
			to *= 2;

			min = min_r = 1.0f;
			max = max_r = -1.0f;
			for (int32 i=index; i<=to; i+=2){
				if (i/nBufferSize > current_buffer){
					current_buffer++;
					VM.ReadBlock(p, nBufferSize);
				}
				idx = i % nBufferSize;
				if (p[idx]>max)	max = p[idx];
				if (p[idx]<min)	min = p[idx];
				idx++;
				if (p[idx]>max_r)	max_r = p[idx];
				if (p[idx]<min_r)	min_r = p[idx];
			}
			*out++ = min;
			*out++ = max;
			*out_r++ = min_r;
			*out_r++ = max_r;
		}
	}
	else
	{	int16 min, max, min_r, max_r;
		for (int32 x = 0; x<w; x++){
			index = start + (int32)(x * step);
			to = index + iStep;	if (to>m_size)	to = m_size;
			index >>= 6;	index &= 0xfffffffe;
			to >>= 6;		to &= 0xfffffffe;

			min = max = min_r = max_r = 0;
			for (int32 i=index; i<=to; i+=2){
				if (buffer_left[i]<min)		min = buffer_left[i];
				if (buffer_left[i+1]>max)	max = buffer_left[i+1];
				if (buffer_right[i]<min_r)	min_r = buffer_right[i];
				if (buffer_right[i+1]>max_r)	max_r = buffer_right[i+1];
			}
			*out++ = min/32767.0;
			*out++ = max/32767.0;
			*out_r++ = min_r/32767.0;
			*out_r++ = max_r/32767.0;
		}
	}
}

