#include "TCPListener.h"
#include <errno.h>

const char *kListenerPtr = "Listener Ptr";

TCPListener::TCPListener( const char *name, int32 maxConnections, int32 port )
{
	this->name = (char *)malloc( strlen( name )+1 );
	strcpy( this->name, name );
	this->maxConnections = maxConnections;
	this->localPort = port;
	
	threadList = new BList( maxConnections );
	listAccessBen = new Benaphore( 1, "list_sem" );
	
	totalConnections = 0;
	status = B_NO_ERROR;
}

TCPListener::~TCPListener( void )
{
	delete threadList;
	delete listAccessBen;
	free( name );
}

thread_id TCPListener::Run( void )
{
	// Setup listen port
	if( (listenSocket = socket( AF_INET, SOCK_STREAM, 0 )) < 0 )
		return errno;
	
	// We will listen on all interfaces
	serverAddr.sin_family = AF_INET;
	serverAddr.sin_port = htons( localPort );
	serverAddr.sin_addr.s_addr = htonl( INADDR_ANY );
	memset(serverAddr.sin_zero, 0, sizeof(serverAddr.sin_zero));
	
	if( bind( listenSocket, (sockaddr *)&serverAddr, sizeof( serverAddr ) ) < 0 ) {
		status = errno; closesocket( listenSocket ); return status; }
	if( listen( listenSocket, maxConnections ) < 0 ) {
		status = errno; closesocket( listenSocket ); return status; }
	
	//Start listen thread
	char		tName[32];
	listenStat = true;
	
	sprintf( tName, "Listen on port %ld", localPort );
	listenThread = spawn_thread( thread_entry_point, tName, B_NORMAL_PRIORITY, this );
	if( listenThread < 0 ) {
		closesocket( listenSocket ); return listenThread; }
	
	resume_thread( listenThread );
	return listenThread;
}

void TCPListener::Quit( void )
{
	status_t			exitValue;
	
	// Wake the thread up if it's blocking
	listenStat = false;
	suspend_thread( listenThread );
	resume_thread( listenThread );
	
	// Wait for listen thread to terminate
	wait_for_thread( listenThread, &exitValue );
	// Terminate connection threads
	TCPConnection		*theConnection;
	
	for( int32 i = 0; theConnection = (TCPConnection *)threadList->ItemAt( i ); i++ )
		theConnection->Quit();
	threadList->MakeEmpty();
	delete this;
}

int32 TCPListener::thread_entry_point( void *arg )
{
	TCPListener *obj = (TCPListener *)arg;
	return( obj->ListenThread() );
}

int32 TCPListener::ListenThread( void )
{
	int32			acceptSocket;
	int				clientLen;
	sockaddr_in		clientAddr;
	TCPConnection	*theConnection;
	status_t		status;
	
	while( listenStat )
	{
		if( (acceptSocket = accept( listenSocket, (sockaddr *)&clientAddr, &clientLen )) < 0 )
		{
			// quitting
			closesocket( listenSocket );
			if( listenStat ) // If terminating because of error, tell be_app
			{
				BMessage		msg( MSG_LISTENER_TERMINATED );
				msg.AddPointer( kListenerPtr, this );
				be_app->PostMessage( &msg );
			}
			return B_NO_ERROR;
		}
		else if( listAccessBen->Acquire() == B_NO_ERROR )
		{
			// create new connection
			theConnection = NewConnection( acceptSocket );
			status = theConnection->Run();
			if( status == B_NO_ERROR )
			{
				threadList->AddItem( theConnection );
				totalConnections++;
			}
			listAccessBen->Release();
		}
	}
	closesocket( listenSocket );
	return B_NO_ERROR;
}

int32 TCPListener::DeleteConnection( TCPConnection *theConnection )
{
	delete theConnection;
	threadList->RemoveItem( theConnection );
	return true;
}

int32 TCPListener::GetTotalConnections( void )
{
	return totalConnections;
}

int32 TCPListener::GetCurrentConnections( void )
{
	return threadList->CountItems();
}

uint16 TCPListener::GetLocalPort( void )
{
	return localPort;
}

int32 TCPListener::GetMaxConnections( void )
{
	return maxConnections;
}

const char *TCPListener::GetName( void )
{
	return name;
}