#include "HTTPProxy.h"

// Internal HTTP Proxy error codes
#define HTTPP_URI_PARSE_ERROR	-10001
#define HTTPP_MISSING_NET_PATH	-10002

const char *kHPPointer = "CTPointer";
const char *kSimple_Response = 
"<HTML>\n"
"<HEAD>\n"
"<TITLE> %s </TITLE>\n"
"</HEAD>\n"
"<BODY BGCOLOR=\"#FFFFFF\">\n"
"%s\n"
"</BODY>\n"
"</HTML>\n";

const char *kError_Body =
"<CENTER><H1>HTTP Proxy Error</H1><HR><P>%s</P></CENTER>";

HTTPProxy::HTTPProxy( int32 socket, DNSResolver *nameResolver, TCPListener *threadManager, int32 bufSize )
	: TCPProxy( socket, nameResolver, threadManager, bufSize )
{
	
}

HTTPProxy::~HTTPProxy( void )
{
	
}

status_t HTTPProxy::Connect( void )
{
	int32		size;
	status_t	status;
	char		*uri;
	sockaddr_in	serverAddr;
	
	// Retrieve HTTP Request from client
	recvSocket = clientSocket;
	sendSize = 0;
	do
	{
		if( (bufSize - sendSize - 1) <= 0 ) // buffer size overflow
			return B_ERROR;
		if( (status = can_read_datagram( recvSocket, 0, 15 )) == true )
		{
			if( (size = recv( recvSocket, recvBuf + sendSize, bufSize - sendSize - 1, 0 )) <= 0 )
				return errno;
		}
		else if( status == 0 )
			return B_TIMEOUT;
		else
			return errno;
		sendSize += size;
		recvBuf[sendSize] = 0; // Terminate buffer
	}while( strstr( recvBuf, "\r\n\r\n" ) == NULL ); // end of request header
	
	uri = recvBuf;
	uri[sendSize] = 0; // append NULL character
	if( (status = ResolveURI( uri, &serverAddr )) != B_NO_ERROR )
		return status;
	
	// Remove host address
	if( (status = StripNetPath( recvBuf, sendSize )) != B_NO_ERROR )
		return status;
		
	// Create socket and connect to HTTP server
	if( (serverSocket = socket( AF_INET, SOCK_STREAM, 0 )) < 0 )
		return errno;
	if( (connect( serverSocket, (sockaddr *)&serverAddr, sizeof(serverAddr) )) < 0 )
		return errno;
	
	// Forward HTTP Request to server
	sendSocket = serverSocket;
	SwapBuffers();
	size = send( sendSocket, sendBuf, sendSize, 0 );
	
	// Make sure the rest of the request is retrieved...
	recvSocket = clientSocket;
	sendSocket = serverSocket;
	
	return B_NO_ERROR;
}

status_t HTTPProxy::ResolveURI( const char *uri, sockaddr_in *addr )
{
	char 		hostStr[256], portStr[16];
	uint16		hostPort;
	int32		ip_addr;
	int32		status;
	
	if( (status = GetHostStr( uri, hostStr, &hostPort )) != B_NO_ERROR )
		return status;
	
	if( (status = nameResolver->ResolveName( hostStr, &ip_addr )) != B_NO_ERROR )
		return status;
	
	// Init address
	addr->sin_family = AF_INET;
	addr->sin_port = htons( hostPort );
	addr->sin_addr.s_addr = ip_addr;
	memset(addr->sin_zero, 0, sizeof(addr->sin_zero));
	
	return B_NO_ERROR;
}

status_t HTTPProxy::GetHostStr( const char *uri, char *hostStr, uint16 *port )
{
	int32		length;
	char		*src;
	
	// Find net path
	if( (src = strstr( uri, "//" )) == NULL )
		return HTTPP_MISSING_NET_PATH;
	
	src += 2;
	
	// Find end of host name
	if( (length = strcspn( src, ":/ " )) == 0 )
		return HTTPP_URI_PARSE_ERROR;
	
	// Is there a port number?
	if( src[length] == ':' )
	{
		// Get the port number
		char		*portSrc = src + length + 1;
		int32		portLength = strcspn( portSrc, "/ " );
		char		portStr[16], *end;
		
		if( portLength > 15 )
			*port = 80; // It's probably not a port number...
		else
		{
			strncpy( portStr, portSrc, portLength );
			*port = strtoul( portStr, &end, 0 );
		}
	}
	else
		*port = 80;
	
	strncpy( hostStr, src, length ); // copy to hostStr
	hostStr[length] = 0; // Append NULL
	
	return B_NO_ERROR;
}

status_t HTTPProxy::StripNetPath( char *httpRequest, int32 &size )
{
	char	*src, *dst;
	int32	length;
	
	// Find beginning of URI
	dst = httpRequest + strcspn( httpRequest, " " ) + 1;
	
	// Advance src to end of network path
	if( (src = strstr( dst, "//" )) == NULL )
		return HTTPP_URI_PARSE_ERROR;
	src += 2;
	
	// Find end of network path
	src += strcspn( src, "/ " );
	
	// If the browser did not append a '/' do it for it
	if( *src == ' ' )
		*(--src) = '/';
		
	// Strip network path
	memmove( dst, src, httpRequest + size - src );
	size -= src - dst;
	
	return B_NO_ERROR;
}

// Send a Simple HTTP Response
status_t HTTPProxy::SendHTTPMessage( const char *title, const char *msgBody )
{
	char		*response = (char *)malloc( 2048 );
	status_t	status;
	
	sprintf( response, kSimple_Response, title, msgBody );
	status = send( clientSocket, response, strlen( response ), 0 );
	free( response );
	return status;
}

status_t HTTPProxy::SendErrorMsg( const char *errStr )
{
	char		*msgBody = (char *)malloc( 2048 );
	status_t	status;
	
	sprintf( msgBody, kError_Body, errStr );
	status = SendHTTPMessage( "HTTP Proxy Error", msgBody );
	free( msgBody );
	return status;
}

void HTTPProxy::HandleError( status_t error )
{
	switch( int32(error) )
	{
		case B_NO_ERROR:
			return;
			break;
		
		case HOST_NOT_FOUND:
			SendErrorMsg( "Unknown host name." );
			break;
		
		case TRY_AGAIN:
			SendErrorMsg( "Host name server busy." );
			break;
		
		case NO_DATA:
			SendErrorMsg( "No address data available for this host name." );
			break;
		
		case ECONNREFUSED:
			SendErrorMsg( "Connection Refused" );
			break;
		
		case ETIMEDOUT:
			SendErrorMsg( "The connection attempt timed out." );
			break;
		
		case ENETUNREACH:
			SendErrorMsg( "The host's network is unreachable." );
			break;
			
		case HTTPP_MISSING_NET_PATH:
			SendErrorMsg( "HTTP Proxy could not find the host name in the URL." );
			break;
		
		case HTTPP_URI_PARSE_ERROR:
			SendErrorMsg( "HTTP Proxy could not understand the URL." );
			break;
			
		default:
			SendErrorMsg( strerror( error ) );
			break;
	}
}

