#include "HTTPUtils.h"

// HTTP Headers
static const char *kStatus_Line = "HTTP/1.0 %ld %s\r\n";
static const char *kContent_Type = "Content-Type: %s\r\n";
static const char *kContent_Length = "Content-length: %ld\r\n";
static const char *kServer_Type = "Server: %s\r\n";
static const char *kDate = "Date: %s";
static const char *kExpires = "Expires: %s";
static const char *kCRLF = "\r\n";
static const char *kNoCache = "Pragma: no-cache\r\n";
static const char *kRefresh = "Refresh: %s; url=%s\r\n";
static const char *kAuthenticate = "WWW-Authenticate: Basic realm=\"%s\"\r\n";

// Status Codes
static const char *kHTTP_OK = "OK";
static const char *kHTTP_Created = "Created";
static const char *kHTTP_Accepted = "Accepted";
static const char *kHTTP_No_Content = "No Content";
static const char *kHTTP_Moved_Permanently = "Moved Permanently";
static const char *kHTTP_Moved_Temporarily = "Moved Temporarily";
static const char *kHTTP_Not_Modified = "Not Modified";
static const char *kHTTP_Bad_Request = "Bad Request";
static const char *kHTTP_Unauthorized = "Unauthorized";
static const char *kHTTP_Forbidden = "Forbidden";
static const char *kHTTP_Not_Found = "Not Found";
static const char *kHTTP_Internal_Server_Error = "Internal Server Error";
static const char *kHTTP_Not_Implemented = "Not Implemented";
static const char *kHTTP_Bad_Gateway = "Bad Gateway";
static const char *kHTTP_Service_Unavailable = "Service Unavailable";
static const char *kHTTP_Default = "Error";

// Reserved URI Characters
static const char *kReservedChars = ";/?:@&=+ \"#%<>\0";

//Methods
static const char *kHTTP_GET = "GET";
static const char *kHTTP_POST = "POST";
static const char *kHTTP_HEAD = "HEAD";

size_t esc_c_str( char *dst, const char *src, size_t bufSize )
{
	const char	*s = src;
	size_t		size;
	uint16		c;
	BMemoryIO	temp( dst, bufSize );
	char		escStr[4];
	
	do
	{
		size = strcspn( s, kReservedChars );
		
		temp.Write( s, size );
		
		c = s[size];
		s += size+1;
		
		if( c == 0 )
		{
			size = temp.Position();
			if( size >= bufSize )
				size = bufSize - 1;
			dst[size] = 0;
			return size;
		}
		sprintf( escStr, "%%%.2hx", c );
		temp.Write( escStr, 3 );
		
	}while( true );
}

size_t unescape_c_str( char *dst, const char *src, size_t bufSize )
{
	const char	*s = src;
	size_t		size;
	uint16		c;
	BMemoryIO	temp( dst, bufSize );
	uint8		escChar;
	char		*end;
	
	do
	{
		size = strcspn( s, "%\0" );
		temp.Write( s, size );
		c = s[size];
		
		if( c == 0 )
		{
			size = temp.Position();
			if( size >= bufSize )
				size = bufSize - 1;
			dst[size] = 0;
			return size;
		}
		s += size+1;
		escChar = strtoul( s, &end, 16 );
		s = end;
		
		temp.Write( &escChar, 1 );
		
	}while( true );
}

void http_wstatus_line( BPositionIO *io, int32 code )
{
	char		s[256];
	const char	*codeStr;
	
	switch( code )
	{
		case 200:
			codeStr = kHTTP_OK;
			break;
		
		case 201:
			codeStr = kHTTP_Created;
			break;
		
		case 202:
			codeStr = kHTTP_Accepted;
			break;
		
		case 204:
			codeStr = kHTTP_No_Content;
			break;
		
		case 301:
			codeStr = kHTTP_Moved_Permanently;
			break;
		
		case 302:
			codeStr = kHTTP_Moved_Temporarily;
			break;
		
		case 304:
			codeStr = kHTTP_Not_Modified;
			break;
		
		case 400:
			codeStr = kHTTP_Bad_Request;
			break;
		
		case 401:
			codeStr = kHTTP_Unauthorized;
			break;
		
		case 403:
			codeStr = kHTTP_Forbidden;
			break;
		
		case 404:
			codeStr = kHTTP_Not_Found;
			break;
		
		case 500:
			codeStr = kHTTP_Internal_Server_Error;
			break;
		
		case 501:
			codeStr = kHTTP_Not_Implemented;
			break;
		
		case 502:
			codeStr = kHTTP_Bad_Gateway;
			break;
		
		case 503:
			codeStr = kHTTP_Service_Unavailable;
			break;
		
		default:
			codeStr = kHTTP_Default;
			break;
	}
	sprintf( s, kStatus_Line, code, codeStr );
	io->Write( s, strlen( s ) );
}

void http_wcontent_length( BPositionIO *io, int32 length )
{
	char		s[256];
	
	sprintf( s, kContent_Length, length );
	io->Write( s, strlen( s ) );
}

void http_wcontent_type( BPositionIO *io, const char *type )
{
	char		s[256];
	
	sprintf( s, kContent_Type, type );
	io->Write( s, strlen( s ) );
}

void http_wserver_type( BPositionIO *io, const char *type )
{
	char		s[256];
	
	sprintf( s, kServer_Type, type );
	io->Write( s, strlen( s ) );
}

void http_wdate( BPositionIO *io, const tm *time )
{
	char		s[256];
	
	sprintf( s, kDate, asctime( time ) );
	io->Write( s, strlen( s ) );
}

void http_wexpires( BPositionIO *io, const tm *time )
{
	char		s[256];
	
	sprintf( s, kExpires, asctime( time ) );
	io->Write( s, strlen( s ) );
}

void http_wnocache( BPositionIO *io )
{
	io->Write( kNoCache, strlen( kNoCache ) );
}

void http_wrefresh( BPositionIO *io, const char *time, const char *url )
{
	char		s[256];
	
	sprintf( s, kRefresh, time, url );
	io->Write( s, strlen( s ) );
}

void http_wauthenticate( BPositionIO *io, const char *realm )
{
	char		s[256];
	
	sprintf( s, kAuthenticate, realm );
	io->Write( s, strlen( s ) );
}

void http_wcrlf( BPositionIO *io )
{
	io->Write( kCRLF, 2 );
}

char *http_find_header( const char *msg, const char *headName, size_t *length )
{
	char	*s;
	if( !(s = strstr( msg, headName )) )
		return NULL;
	s += strlen( headName );
	*length = strcspn( s, kCRLF );
	return s;
}

char *http_find_uri( const char *msg, size_t *length )
{
	char	*s = (char *)msg + strcspn( msg, " " ) + 1;
	
	*length = strcspn( s, " \r\n" );
	return s;
}

char *http_find_msg_body( const char *msg )
{
	char	*s;
	
	if( !(s = strstr( msg, "\r\n\r\n" )) )
		return NULL;
	return s + 4;
}

http_method http_find_method( const char *msg )
{
	size_t		size = strcspn( msg, " " );
	if( strncmp( msg, kHTTP_GET, size ) == 0 )
		return METHOD_GET;
	else if( strncmp( msg, kHTTP_POST, size ) == 0 )
		return METHOD_POST;
	else if( strncmp( msg, kHTTP_HEAD, size ) == 0 )
		return METHOD_HEAD;
	else
		return METHOD_UNKNOWN;
}

char *traverse_path( const char *path, size_t *length )
{
	size_t		size;
	char		*s = (char *)path;
	
	while( (size = strcspn( s, "/ ?" )) == 0 )
	{
		if( *s == ' ' )
			return NULL;
		s++;
	}
	*length = size;
	return s;
}

char *find_post_element( const char *msg, const char *name, size_t *length )
{
	char		*s;
	
	if( !(s = strstr( msg, name )) )
		return NULL;
	if( !(s = strpbrk( s, "=" )) )
		return NULL;
	s++;
	*length = strcspn( s, "&" );
	return s;
}

status_t get_post_cstring( const char *msg, const char *name, char *buffer, size_t bufSize )
{
	size_t		size;
	char		*ePtr;
	
	if( !(ePtr = find_post_element( msg, name, &size )) )
		return B_ERROR;
	if( size >= bufSize )
		size = bufSize-1;
	
	memmove( buffer, ePtr, size );
	buffer[size] = 0;
	
	// Unescape
	for( char *c = buffer; *c != 0; c++ )
	{
		if( *c == '+' )
			*c = ' ';
	}
	return B_NO_ERROR;
}

