#include "Proxy.h"

TCPProxy::TCPProxy( int32 socket, DNSResolver *nameResolver, TCPListener *threadManager, int32 bufSize, bigtime_t timeout, bigtime_t recvTimeout )
	: TCPConnection( socket, threadManager )
{
	this->bufSize = bufSize;
	this->nameResolver = nameResolver;
	this->timeout = timeout;
	this->recvTimeout = recvTimeout;
	
	// Allocate send and receive buffers
	buffer = (char *)malloc( bufSize * 2 );
	recvBuf = buffer;
	sendBuf = buffer + bufSize;
	
	// Create thread syncronizing semaphores
	recvSem = create_sem( 0, "recv_sem" );
	sendSem = create_sem( 0, "send_sem" );
	lock = new Benaphore( 1, "lock" );
	
	// Init data members
	recvThread = sendThread = -1;
	totalRecv = totalSend = 0;
	serverSocket = -1;
	running = false;
}

TCPProxy::~TCPProxy( void )
{
	free( buffer );
	
	delete_sem( recvSem );
	delete_sem( sendSem );
	delete lock;
}

status_t TCPProxy::Run( void )
{
	lock->Acquire();
	if( running )
	{
		lock->Release();
		return B_ERROR;
	}
	running = true;
	char	threadName[32];
	
	sprintf( threadName, "%ld: proxy_recv", psn );
	recvThread = spawn_thread( recv_entry_point, threadName, B_LOW_PRIORITY, this );
	
	sprintf( threadName, "%ld: proxy_send", psn );
	sendThread = spawn_thread( send_entry_point, threadName, B_LOW_PRIORITY, this );
	resume_thread( recvThread );
	
	lock->Release();
	return B_NO_ERROR;
}

void TCPProxy::Quit( void )
{
	running = true;
	TCPConnection::Quit();
}

int32 TCPProxy::recv_entry_point( void *arg )
{
	TCPProxy *obj = (TCPProxy *)arg;
	return( obj->RecvThreadLoop() );
}

int32 TCPProxy::send_entry_point( void *arg )
{
	TCPProxy *obj = (TCPProxy *)arg;
	return( obj->SendThreadLoop() );
}

int32 TCPProxy::RecvThreadLoop( void )
{
	int32		size;
	status_t	status;
	bool		swap = false;
	
	timeval 	tv; 
	fd_set 		fds; 
	
	tv.tv_sec = 300; 
	tv.tv_usec = 0; 
	
	if( (status = Connect()) != B_NO_ERROR )
		goto bail_out;
	release_sem( recvSem ); // Don't block on first pass ( nothing to send yet )
	resume_thread( sendThread ); // Start send thread
	
	while( running )
	{
		// Is data waiting?; Wait up to 50 ms
		if( (status = can_read_datagram( recvSocket, 50000, 0 )) == 1 )
		{
			if( (size = Receive( recvSocket, recvBuf, bufSize-1 )) < 0 )
			{
				status = errno;
				break;
			}
			totalRecv += size;
		}
		else if( status == -1 ) // Error...
		{
			status = errno;
			break;
		}
		else // Check other socket for data
			swap = true;
		
		// Wait until Send is complete
		if( (status = acquire_sem( recvSem )) != B_NO_ERROR )	
			break;
		// *** It's now safe to check the other socket ***
		
		if( swap )
		{	
			FD_ZERO(&fds);
			FD_SET( clientSocket, &fds );
			FD_SET( serverSocket, &fds );
			
			// Wait up to 300 secs ( 5 minutes )...
			if( (status = select(32, &fds, NULL, NULL, &tv)) == 1 )
			{
				if( FD_ISSET(clientSocket, &fds) )
				{
					recvSocket = clientSocket;
					sendSocket = serverSocket;
				}
				else if( FD_ISSET(serverSocket, &fds) )
				{
					recvSocket = serverSocket;
					sendSocket = clientSocket;
				}
				else
				{
					status = B_ERROR;
					break;
				}
			}
			else if( status == -1 ) // Proabably interupted
			{
				status = errno;
				break;
			}
			else
			{
				status = B_TIMEOUT;
				break;
			}
			
			swap = false;
			release_sem( recvSem );
			continue;
		}
		
		// if size == 0, that's the last segment
		if( size == 0 )
		{
			// That's all folks!
			status = B_NO_ERROR;
			break;
		}
		// Let send thread forward next segment
		// This thread will receive the next segment at the same time
		sendSize = size;
		SwapBuffers();
		release_sem( sendSem );
	}
	bail_out:
	Shutdown( status );
	return status;
}

int32 TCPProxy::SendThreadLoop( void )
{
	int32		size;
	status_t	status;
	
	while( running )
	{
		// Block until data waiting
		if( (status = acquire_sem( sendSem )) != B_NO_ERROR )
		{
			status = errno;
			break;
		}
		
		// Make sure we don't send the same thing twice when exiting
		if( !running )
		{
			status = B_NO_ERROR;
			break;
		}
		
		// Send
		if( (size = send( sendSocket, sendBuf, sendSize, 0 )) < 0 )
		{
			status = errno;
			break;
		}
		totalSend += size;
		
		// Let receive thread continue
		release_sem( recvSem );
	}
	Shutdown( status );
	return status;
}

void TCPProxy::SwapBuffers( void )
{
	char	*buf;
	buf = recvBuf;
	recvBuf = sendBuf;
	sendBuf = buf;
}

// Not used...
void TCPProxy::SwapSockets( void )
{
	int32		theSocket;
	
	theSocket = recvSocket;
	recvSocket = sendSocket;
	sendSocket = theSocket;
}

// Connection is complete
void TCPProxy::Shutdown( status_t error )
{
	lock->Acquire();
	if( running )
	{
		running = false;
		lock->Release();
		
		status_t		exitValue;
		thread_id		thisThread = find_thread( NULL );
		
		// Shutdown threads; but make sure we don't suspend our own thread...
		// ...That would be a "bad thing"
		if( thisThread != sendThread  )
		{
			release_sem( sendSem );
			suspend_thread( sendThread );
			resume_thread( sendThread );
			wait_for_thread( sendThread, &exitValue );
		}
		if( thisThread != recvThread )
		{
			release_sem( recvSem );
			suspend_thread( recvThread );
			resume_thread( recvThread );
			wait_for_thread( recvThread, &exitValue );
		}
		
		// Send message to client to indicate reason for failure
		if( totalSend == 0 )
			HandleError( error );
		
		// Close sockets
		if( clientSocket >= 0 )
		{
			closesocket( clientSocket );
			clientSocket = -1;
		}
		if( serverSocket >= 0 )
		{
			closesocket( serverSocket );
			serverSocket = -1;
		}
		
		TCPConnection::Shutdown( error );
		return;
	}
	else
		lock->Release();
}

void TCPProxy::HandleError( status_t error )
{
	
}

// Hook for derived classes
// Could be used to pre-process the data, cache the data, or whatever
// Returns B_ERROR if connection should be terminated, B_NO_ERROR if it should forward the data as usual, or 1 if it should not forward the data

// NOTE: Not fully implemented yet!
status_t TCPProxy::DataReceived( void **data, int32 *size, int32 socket )
{
	return B_NO_ERROR;
}

// returns true if data is waiting, false if timeout, and -1 if an error occured
// hook would be usefull for cache access
status_t TCPProxy::DataWaiting( int32 socket, bigtime_t timeout )
{
	return can_read_datagram( socket, timeout, 0 );
}

// returns size of data actually read; < 0 if an error occured
// hook would be usefull for cache access
int32 TCPProxy::Receive( int32 socket, void *buffer, int32 size )
{
	return recv( socket, buffer, size, 0 );
}

			