#include "ProxyApp.h"

static const char *kConfigFileName = "proxy config";

ProxyApp::ProxyApp( void )
	: BApplication( "application/x-vnd.KlossSoft-Proxy" )
{
	proxyMList = new BList();
	nameResolver = new DNSResolver();
	go = false;
}


ProxyApp::~ProxyApp( void )
{
	delete proxyMList;
	delete nameResolver;
}


void ProxyApp::MessageReceived( BMessage *message )
{
	switch( message->what )
	{
		case MSG_RESET_PROXY:
			ShutdownProxies();
			StartAllProxies();
			break;
			
		case MSG_STOP_PROXY:
		{
			BMessage		config( MSG_PROXIES );
			StopProxies( message );
			
			GetProxies( &config );
			SaveConfig( &config );
			break;
		}
		case MSG_START_PROXY:
		{
			BMessage		config( MSG_PROXIES );
			StartProxies( message );
			
			GetProxies( &config );
			SaveConfig( &config );
			break;
		}
		case MSG_GET_PROXIES:
		{
			BMessage	reply( MSG_PROXIES );
			
			GetProxies( &reply );
			message->SendReply( &reply );
			break;
		}
		case MSG_LISTENER_TERMINATED:
		{
			TCPListener		*theListener;
			
			if( message->FindPointer( kListenerPtr, 0, &theListener ) == B_OK )
			{
				if( proxyMList->RemoveItem( theListener ) && go )
					theListener->Quit(); // An error occured; clean up
			}
			
			// Startup again after net_server is restarted...
			// We don't want to do this when the system is being shutdown or restarted
			// In this case, the net_server is shutdown before httpproxy
			// The tracker is shutdown before the NetServer, so it's absence may indicate we are shutting down
			if( go && (proxyMList->CountItems() == 0) && 
				be_roster->IsRunning( "application/x-vnd.Be-TRAK" ) )
			{
				snooze( 5000000 ); // Wait for 5 seconds
				BMessenger		messenger( be_app, be_app );
				admin = new ProxyAdmin( &messenger );
				admin->Run();
				
				StartAllProxies();
			}
			break;
		}
		default: 
			BApplication::MessageReceived( message );
			break;
	}
}

void ProxyApp::ReadyToRun( void )
{
	Lock();
	
	// Start HTTP Proxy Admin Server
	BMessenger		messenger( be_app, be_app );
	admin = new ProxyAdmin( &messenger );
	admin->Run();
	
	// Start Proxies
	StartAllProxies();
	Unlock();
	
  	BApplication::ReadyToRun();
}

void ProxyApp::Quit( void )
{
	ShutdownProxies();
	admin->Quit();
	BApplication::Quit();
}

void ProxyApp::StartAllProxies( void )
{
	status_t	status;
	BMessage	config( MSG_PROXIES );
	
	go = true;
	status = LoadConfig( &config );
	if( status != B_NO_ERROR )
		SaveConfig( &config );
	StartProxies( &config );
}

status_t ProxyApp::GetConfigFile( BFile *file, int32 openMode )
{
	status_t	status;
	BPath		path;
	
	status = find_directory( B_USER_SETTINGS_DIRECTORY, &path);
	if( status == B_NO_ERROR )
	{
		BDirectory	dir( path.Path() );
		
		status = file->SetTo( &dir, kConfigFileName, openMode );
	}
	
	return status;
}

status_t ProxyApp::SaveConfig( BMessage *config )
{
	status_t	status;
	BFile		configFile;
	
	status = GetConfigFile( &configFile, B_WRITE_ONLY | B_CREATE_FILE | B_ERASE_FILE  );
	if( status == B_NO_ERROR )
	{
		GetProxies( config );
		status = config->Flatten( &configFile );
	}
	return status;
}

status_t ProxyApp::LoadConfig( BMessage *config )
{
	status_t	status;
	BFile		configFile;
	
	status = GetConfigFile( &configFile, B_READ_WRITE );
	if( status == B_NO_ERROR )
		status = config->Unflatten( &configFile );
	if( status != B_NO_ERROR )
	{
		config->AddInt32( kProxyType, HTTP_PROXY_TYPE );
		config->AddString( kProxyName, "HTTP Proxy" );
		config->AddInt16(kProxyLocalPort, 8080 );
		config->AddInt32( kProxyMaxCon, 32 );
	}
	return status;
}

int32 shutdown_thread( void *arg );

void ProxyApp::ShutdownProxies( void )
{
	if( go )
	{
		go = false;
		BList				shutdownL;
		ProxyManager		*proxyM;
		thread_id			thread;
		status_t			exitValue;
		
		// Swawn a shutdown thread for each proxy; this speeds up the shutdown process
		for( int32 i = 0; proxyM = (ProxyManager *)proxyMList->ItemAt( i ); i++ )
			shutdownL.AddItem( (void *)spawn_thread( shutdown_thread, "Shudown T", B_NORMAL_PRIORITY, proxyM ) );
		proxyMList->MakeEmpty();
		// Wait until all shutdown threads are done
		for( int32 i = 0; thread = (thread_id)shutdownL.ItemAt( i ); i++ )
			wait_for_thread( thread, &exitValue );
	}
}

int32 shutdown_thread( void *arg )
{
	ProxyManager		*proxyM = (ProxyManager *)arg;
	
	proxyM->Quit();
	return B_NO_ERROR;
}

void ProxyApp::StopProxies( BMessage *message )
{
	ProxyManager		*proxyM;
	const char 			*name;
	status_t			status = B_OK;
	
	// Cycle through all names in message
	for( int32 i = 0; status == B_OK; i++ )
	{
		// Get Name
		if( (status = message->FindString( kProxyName, i, &name )) != B_OK )
			continue;
		// Find name in list
		for( int32 i = 0; proxyM = (ProxyManager *)proxyMList->ItemAt( i ); i++ )
		{
			if( strcmp( proxyM->GetName(), name ) == 0 )
			{
				proxyMList->RemoveItem( proxyM );
				proxyM->Quit();
				break;
			}
		}
	}
}

void ProxyApp::StartProxies( BMessage *message )
{
	int32			type;
	const char 		*name;
	uint16			localPort;
	int32			maxCon;
	char			*host;
	uint16			remotePort;
	ProxyManager	*theProxyM;
	status_t		status = B_OK;
	
	int32			total = 0;
	
	for( int32 i = 0, j = 0; status == B_OK; i++ )
	{
		// Get Type
		if( (status = message->FindInt32( kProxyType, i, &type )) != B_OK )
			continue;
		// Get Name
		if( message->FindString( kProxyName, i, &name ) != B_OK )
			continue;
		// Get Local Port
		if( message->FindInt16( kProxyLocalPort, i, (int16 *)&localPort ) != B_OK )
			continue;
		// Get Max Connections
		if( message->FindInt32( kProxyMaxCon, i, &maxCon ) != B_OK )
			maxCon = 32;
		
		switch( type )
		{
			case HTTP_PROXY_TYPE:
				theProxyM = new HTTPProxyManager( nameResolver, name, localPort, maxCon );
				total++;
				break;
				
			case GENERIC_PROXY_TYPE:
				// Get Host
				if( message->FindString( kProxyHost, j, &host ) != B_OK )
					continue;
				// Get Remote Port
				if( message->FindInt16( kProxyRemotePort, j, (int16 *)&remotePort ) != B_OK )
					continue;
				theProxyM = new GenericProxyManager( nameResolver, name, 
					localPort, host, remotePort, maxCon );
				j++;
				total++;
				break;
				
			default:
				continue;
				break;
		} // end switch()
		
		if( theProxyM->Run() >= 0 )
			proxyMList->AddItem( theProxyM );
		else
		{
			delete theProxyM;
			continue;
		}
	} // End for()
}

void ProxyApp::GetProxies( BMessage *message )
{
	ProxyManager	*proxyM;
	int32			total = 0;
	for( int32 i = 0; proxyM = (ProxyManager *)proxyMList->ItemAt( i ); i++ )
	{
		int32		type;
		
		message->AddInt32( kProxyType, type = proxyM->GetType() );
		message->AddString( kProxyName, proxyM->GetName() );
		message->AddInt16(kProxyLocalPort, proxyM->GetLocalPort() );
		message->AddInt32( kProxyMaxCon, proxyM->GetMaxConnections() );
		message->AddInt32( kProxyHits, proxyM->GetTotalConnections() );
		message->AddInt32( kProxyCon, proxyM->GetCurrentConnections() );
		
		switch( type )
		{
			case HTTP_PROXY_TYPE:
				
				break;
			
			case GENERIC_PROXY_TYPE:
			{
				GenericProxyManager		*genProxyM = (GenericProxyManager *)proxyM;
				
				message->AddString( kProxyHost, genProxyM->GetHost() );
				message->AddInt16( kProxyRemotePort, genProxyM->GetRemotePort() );
				break;
			}
			default:
				
				break;
		}
		
		total++;
	}
}

