#include "AdminCon.h"
#include "ProxyAdmin.h"
#include "SocketTools.h"
#include "ProxyAppMsg.h"
#include "HTTPUtils.h"
#include "Base64.h"
#include <unistd.be.h>
#include <errno.h>

extern const char *kSimple_Response;

AdminConnection::AdminConnection( int32 socket, ProxyAdmin *threadManager, 
	const BMessenger *messenger )
	: TCPConnection( socket, threadManager )
{
	pAdmin = threadManager;
	running = false;
	lock = new Benaphore( 1, "lock" );
	this->messenger = new BMessenger( *messenger );
}

AdminConnection::~AdminConnection( void )
{
	delete lock;
	delete messenger;
}

status_t AdminConnection::AdminConnection::Run( void )
{
	lock->Acquire();
	if( running )
	{
		lock->Release();
		return B_ERROR;
	}
	running = true;
	char	threadName[32];
	
	sprintf( threadName, "%ld: http_admin", psn );
	conThread = spawn_thread( thread_entry_point, threadName, B_LOW_PRIORITY, this );
	if( conThread < 0 )
	{
		closesocket( clientSocket );
		lock->Release();
		return conThread;
	}
	
	resume_thread( conThread );
	lock->Release();
	return B_NO_ERROR;
}

void AdminConnection::Shutdown( status_t error )
{
	lock->Acquire();
	if( running || (error == CONNECTION_TERMINATED) )
	{
		running = false;
		lock->Release();
		
		status_t		exitValue;
		
		if( error == CONNECTION_TERMINATED )
		{
			suspend_thread( conThread );
			resume_thread( conThread );
			wait_for_thread( conThread, &exitValue );
		}
		if( clientSocket >= 0 )
		{
			closesocket( clientSocket );
			clientSocket = -1;
		}
		TCPConnection::Shutdown( error );
		return;
	}
	else
		lock->Release();
}

int32 AdminConnection::thread_entry_point( void *arg )
{
	AdminConnection *obj = (AdminConnection *)arg;
	return( obj->ThreadLoop() );
}

int32 AdminConnection::ThreadLoop( void )
{
	status_t		size = 1, status;
	size_t			sendSize = 0;
	size_t			bufSize = 16384;
	char			*buffer = (char *)malloc( bufSize );
	char			*endPtr, *sPtr;
	
	// Get Request Message
	do
	{
		if( (bufSize - sendSize - 1) <= 0 ) // buffer size overflow
			return B_ERROR;
		if( (status = can_read_datagram( clientSocket, 0, 15 )) == true )
		{
			if( (size = recv( clientSocket, buffer + sendSize, bufSize - sendSize - 1, 0 )) <= 0 )
				return errno;
		}
		else if( status == 0 )
			return B_TIMEOUT;
		else
			return errno;
		sendSize += size;
		buffer[sendSize] = 0; // Terminate buffer
	}while( (endPtr = strstr( buffer, "\r\n\r\n" )) == NULL ); // end of request header
	
	// Check for content-length header
	sPtr = http_find_header( buffer, "Content-length: ", (uint32 *)&size );
	if( sPtr != NULL ) // If there is content, we may not have it all
	{
		int32		contentLength;
		int32		bytesRemaining;
		
		contentLength = strtol( sPtr, NULL, 10 );
		bytesRemaining = (contentLength + (endPtr + 4 - buffer))-sendSize;
		while( bytesRemaining > 0 )
		{
			if( (bufSize - sendSize - 1) <= 0 ) // buffer size overflow
				return B_ERROR;
			if( (status = can_read_datagram( clientSocket, 0, 15 )) == true )
			{
				if( (size = recv( clientSocket, buffer + sendSize, bufSize - sendSize - 1, 0 )) <= 0 )
					return errno;
			}
			else if( status == 0 )
				return B_TIMEOUT;
			else
				return errno;
			sendSize += size;
			bytesRemaining -= size;
		}
	}
	buffer[sendSize] = 0;
	
	BMallocIO		reply;
	
	// Is the user authorized?
	if( Authenticate( &reply, buffer ) )
	{
		// Handle request
		switch( http_find_method( buffer ) )
		{
			case METHOD_GET:
				status = HandleGet( &reply, buffer );
				break;
			case METHOD_POST:
				status = HandlePost( &reply, buffer );
				break;
			case METHOD_HEAD:
				status = HandleHead( &reply, buffer );
				break;
			default:
				status = MakeDefaultReply( &reply );
				break;
		}
	}
	
	// Send Reply
	size = send( clientSocket, reply.Buffer(), reply.Position(), 0 );
	
	if( status == B_QUIT_REQUESTED )
		messenger->SendMessage( B_QUIT_REQUESTED );
	// Cleanup and shutdown
	free( buffer );
	Shutdown( B_NO_ERROR );
	return B_NO_ERROR;
}

// Build HTML Reply
size_t AdminConnection::WriteReplyBody( BPositionIO *msgBody )
{
	char			string[1024];
	const char		*strPtr;
	char			hostStr[256], nameStr[256];
	
	size_t			size;
	
	// Write body head
	strPtr = pAdmin->GetBodyHead( &size );
	msgBody->Write( strPtr, size );

	strPtr = pAdmin->GetTableRow( &size );
	
	// Get info about what's running
	BMessage		replyMsg;
	messenger->SendMessage( MSG_GET_PROXIES, &replyMsg );
	
	int32			type;
	const char 		*name;
	uint16			localPort;
	int32			maxCon;
	int32			hits, connections;
	char			*host;
	uint16			remotePort;
	status_t		status = B_OK;
	
	// Make one table row for each proxy
	for( int32 i = 0, j = 0; status == B_OK; i++ )
	{
		// Get Type
		if( (status = replyMsg.FindInt32( kProxyType, i, &type )) != B_OK )
			continue;
		// Get Name
		if( replyMsg.FindString( kProxyName, i, &name ) != B_OK )
			continue;
		// Get Local Port
		if( replyMsg.FindInt16( kProxyLocalPort, i, (int16 *)&localPort ) != B_OK )
			continue;
		// Get Max Connections
		if( replyMsg.FindInt32( kProxyMaxCon, i, &maxCon ) != B_OK )
			maxCon = 32;
		// Gets Hits
		if( replyMsg.FindInt32( kProxyHits, i, &hits ) != B_OK )
			hits = 0;
		// Get Current Connections
		if( replyMsg.FindInt32( kProxyCon, i, &connections ) != B_OK )
			connections = 0;
			
		switch( type )
		{
			case GENERIC_PROXY_TYPE:
				// Get Host
				if( replyMsg.FindString( kProxyHost, j, &host ) != B_OK )
					continue;
				// Get Remote Port
				if( replyMsg.FindInt16( kProxyRemotePort, j, (int16 *)&remotePort ) != B_OK )
					continue;
				sprintf( hostStr, "%s:%d", host, remotePort );
				j++;
				break;
				
			default:
				sprintf( hostStr, "N/A" );
				break;
		}
		esc_c_str( nameStr, name, 256 ); // Make sure white space in name is escaped
		sprintf( string, strPtr, name, localPort, hits, connections, maxCon, hostStr, nameStr );
		msgBody->Write( string, strlen( string ) );
	}
	
	// Write body foot
	strPtr = pAdmin->GetBodyFoot( &size );
	msgBody->Write( strPtr, size );
	return msgBody->Position();
}

status_t AdminConnection::HandleGet( BPositionIO *reply, char *request )
{
	size_t		size;
	char		*uri;
	BMallocIO	body;
	bool		refresh = false;
	
	uri = http_find_uri( request, &size );
	uri = traverse_path( uri, &size );
	if( uri == NULL )
	{
		// Nothing to do
	}
	else if( strncmp( "quit", uri, size ) == 0 )
	{
		char	*body = (char *)malloc( 4096 );
		
		sprintf( body, kSimple_Response, "Quit", "<CENTER><H3>HTTP Proxy Terminated.</H3></CENTER>" );
		size = strlen( body );
		WriteHeader( reply, 200, size );
		reply->Write( body, size );
		free( body );
		return B_QUIT_REQUESTED;
	}
	else if( strncmp( "restart", uri, size ) == 0 )
		messenger->SendMessage( MSG_RESET_PROXY );
	else if( strncmp( "delete", uri, size ) == 0 )
	{
		uri += size;
		uri = traverse_path( uri, &size );
		uri[size] = 0;
		char		dst[256];
		
		unescape_c_str( dst, uri, 256 );
		
		BMessage		msg( MSG_STOP_PROXY );
		msg.AddString( kProxyName, dst );
		messenger->SendMessage( &msg );
	}
	else if( strncmp( "refresh", uri, size ) == 0 )
		refresh = true;
	else
	{
		WriteHeader( reply, 404, 0 ); // Not Found
		return B_NO_ERROR;
	}
	size = WriteReplyBody( &body );
	WriteHeader( reply, 200, size, refresh ); // 200 OK
	reply->Write( body.Buffer(), size );
	return B_NO_ERROR;
}

status_t AdminConnection::HandlePost( BPositionIO *reply, const char *request )
{
	size_t			size;
	status_t		status;
	char			*reqBody;
	BMallocIO		body;
	BMessage		msg( MSG_START_PROXY );
	
	// Get the message body
	reqBody = http_find_msg_body( request );
	
	char			element[256];
	int32			type;
	
	// Retrieve elements and add to message
	if( get_post_cstring( request, "type", element, 256 ) != B_NO_ERROR )
		goto bail_out;
	if( strcmp( element, "HTTP" ) == 0 )
		type = HTTP_PROXY_TYPE;
	else
		type = GENERIC_PROXY_TYPE;
	msg.AddInt32( kProxyType, type );
	
	if( get_post_cstring( request, "name", element, 256 ) != B_NO_ERROR )
		goto bail_out;
	msg.AddString( kProxyName, element );
	
	if( get_post_cstring( request, "local_port", element, 256 ) != B_NO_ERROR )
		goto bail_out;
	msg.AddInt16( kProxyLocalPort, strtol( element, NULL, 0 ) );
	
	if( get_post_cstring( request, "max", element, 256 ) != B_NO_ERROR )
		goto bail_out;
	msg.AddInt32( kProxyMaxCon, strtol( element, NULL, 0 ) );
	
	if( type == GENERIC_PROXY_TYPE )
	{
		if( get_post_cstring( request, "host", element, 256 ) != B_NO_ERROR )
			goto bail_out;
		msg.AddString( kProxyHost, element );
		
		if( get_post_cstring( request, "remote_port", element, 256 ) != B_NO_ERROR )
			goto bail_out;
		msg.AddInt16( kProxyRemotePort, strtol( element, NULL, 0 ) );
	}
	// Tell proxy app to start proxy
	messenger->SendMessage( &msg );
	
	// Send back updated proxy list
	size = WriteReplyBody( &body );
	WriteHeader( reply, 200, size ); // Created
	reply->Write( body.Buffer(), size );
	return B_NO_ERROR;
	
	bail_out:
	WriteHeader( reply, 400, 0 ); // Bad Request
	return B_NO_ERROR;
}

status_t AdminConnection::HandleHead( BPositionIO *reply, const char *request )
{
	size_t		size;
	BMallocIO	body;
	
	size = WriteReplyBody( &body );
	WriteHeader( reply, 200, size ); // OK
	return B_NO_ERROR;
}

status_t AdminConnection::MakeDefaultReply( BPositionIO *reply )
{
	WriteHeader( reply, 400, 0 ); // Bad Request
	return B_NO_ERROR;
}

void AdminConnection::WriteHeader( BPositionIO *reply, int32 statusCode, size_t bodySize, bool refresh )
{
	time_t			now = time( NULL );
	struct tm		*date = localtime( &now );
	
	http_wstatus_line( reply, statusCode );
	http_wcontent_type( reply, "text/html" );
	http_wnocache( reply ); // It's too bad NetPositive ignores this...
	http_wserver_type( reply, "BeOS/HTTPProxy1.0" );
	http_wcontent_length( reply, bodySize );
	if( bodySize != 0 )
	{
		http_wexpires( reply, date );
		http_wdate( reply, date );
		if( refresh )
			http_wrefresh( reply, "15", "refresh" ); // And this too!
	}
	http_wcrlf( reply );
}

bool AdminConnection::Authenticate( BPositionIO *reply, const char *request )
{
	char			*credentials;
	size_t			size;
	status_t		status;
	
	// Were basic credintials sent in the request header?
	credentials = http_find_header( request, "Authorization: Basic ", &size );
	if( credentials != NULL )
	{
		char		credString[80];
		
		if( size >= 80 )
			size = 79; // Leave room for NULL character
		
		// Make a copy of credentials
		memmove( credString, credentials, size );
		credString[size] = 0;
		
		// Decode base64 format
		size = decode_base64( credString, credString, size );
		credString[size] = 0;
		
		// Get username and password strings form credentials
		char	username[32], password[32];
		
		size = strcspn( credString, ":" );
		strncpy( username, credString, size );
		username[size] = 0;
		strcpy( password, credString + size + 1 );
		
		// do the user names match? ...and does the password encrypt to the one stored in the prefs file?
		if( (strcmp( username, pAdmin->GetUser() ) == 0)&&(strcmp( pAdmin->GetPass(), crypt( password, pAdmin->GetPass() ) ) == 0) )
			return true;
		else // Bad username or password: Send Forbidden reply
		{
			http_wstatus_line( reply, 403 );
			http_wauthenticate( reply, "http_proxy" );
			http_wcontent_length( reply, 0 );
			http_wcrlf( reply );
			return false;
		}
	}
	else // No credentials!
	{
		not_authorized:
		
		// Request credentials
		http_wstatus_line( reply, 401 );
		http_wauthenticate( reply, "http_proxy" ); // http_proxy realm...
		http_wcontent_length( reply, 0 );
		http_wcrlf( reply );
		return false;
	}
}
