/******************************************************************************
//
//	File:		ProxyMain.h
//
//	Description:	Generic TCP proxy class; base class for other proxies
//
//	Copyright 1998, Joe Kloss
//
*******************************************************************************/

#ifndef PROXY_H
#define PROXY_H

#include <errno.h>
#include "TCPConnection.h"
#include "SocketTools.h"
#include "benaphore.h"

// Message Stuff ( sent to thread manager when done )
extern const char *kHPPointer; // Pointer to terminating HTTPProxy
#define HP_TERMINATE 'PMtm'

class TCPProxy : public TCPConnection
{
	public:
		TCPProxy( int32 socket, DNSResolver *nameResolver, 
			TCPListener *threadManager, int32 bufSize = 16384, 
			bigtime_t timeout = 60000000, bigtime_t recvTimeout = 500000 );
		~TCPProxy( void );
		
		virtual status_t Run( void ); // Similar to BLooper
		virtual void Quit( void );
		
	protected:
		virtual status_t Connect( void ) = 0; // Connect to remote host
		virtual void HandleError( status_t error ); // Called by Shutdown() if connecting to the host failed
		virtual status_t DataReceived( void **data, int32 *size, int32 socket );
		virtual status_t DataWaiting( int32 socket, bigtime_t timeout );
		virtual int32 Receive( int32 socket, void *buffer, int32 size );
		
		static int32 recv_entry_point( void *arg );
		int32 RecvThreadLoop( void );
		static int32 send_entry_point( void *arg );
		int32 SendThreadLoop( void );
		
		void SwapBuffers( void ); // Swap buffer pointers; sendBuf <--> recvBuf
		void SwapSockets( void ); // Swap send and receive sockets
		virtual void Shutdown( status_t error ); // Called when connection can't continue
		
	protected:
		DNSResolver 	*nameResolver;
		bigtime_t		timeout; // (usecs) no activity timeout period; calls Shutdown() if no activity for specified period.
		bigtime_t		recvTimeout; // (usecs) to watch for data on a Socket before checking the other socket for data
		
		thread_id		recvThread, sendThread;
		sem_id			recvSem, sendSem; // Thread sync semaphores
		int32 			serverSocket;
		int32			bufSize; // size of send and receive buffers
		char 			*buffer; // 2x buffer size
		
		char			*recvBuf, *sendBuf;
		int32			recvSocket, sendSocket;
		
		int32			sendSize; // size of buffer to send
		int32			totalRecv, totalSend; // Tracks total bytes sent and received
		bool			running;
		Benaphore		*lock;
};

#endif
