//
//
//	3DS objects parser
//
//										(C) JoOl 1998


#include "Parser3ds.h"
#include "LAPerrors.h"


#if __POWERPC__
#pragma export on
#endif
char			handlerName[]		= "Autodesk 3D Studio v3.0 (.3ds)";
char			handlerVersion[]	= "1.1.0";
char			handlerAuthor[]		= "Alain GAUTHEROT";
char			handlerEmail[]		= "gauthero@club-internet.fr";
#if __POWERPC__
#pragma export reset
#endif



Parser3ds*
instantiate_parser(int32 bs, LAPtracer* eo)
{
	return new Parser3ds(bs, eo);
}


Parser3ds::Parser3ds(int32 blockSize, LAPtracer* eo)
{
	// input is a big-endian binary file
	SetLexer(new LAPbinLexer(max_c(1024, blockSize), LAP_LITTLE_ENDIAN));

	uint32			ownFlags =	LAP_OWN_TRACE_OUTPUT | LAP_OWN_DISPLAY;
	if (!eo)
	{
		eo = new LAPtracer(256, LAP_BUFFER_OUTPUT_ON_OVERFLOW, LAP_BUFFER_SCREEN_OUTPUT);
		ownFlags |=	LAP_OWN_ERROR_OUTPUT;
	}

	// error output, tracer output, display
	SetTo(	eo,
			new LAPtextBufferedOutput(128, LAP_BUFFER_OUTPUT_ON_OVERFLOW, LAP_BUFFER_NO_OUTPUT/*LAP_BUFFER_SCREEN_OUTPUT*/),
			new LAPdisplay(),
			ownFlags);

	SetLexemes();
}


Parser3ds::~Parser3ds()
{
	if (data)
		delete (pScene* )data;
}


void
Parser3ds::SetLexemes()
{
	lexemes.Insert(new LAPlexeme("MAIN_3DS", MAIN_3DS));
		lexemes.Insert(new LAPlexeme("EDIT3DS", EDIT3DS));
			lexemes.Insert(new LAPlexeme("EDIT_MATERIAL", EDIT_MATERIAL));
				lexemes.Insert(new LAPlexeme("MAT_NAME", MAT_NAME));
				lexemes.Insert(new LAPlexeme("MAT_AMBIENTCOLOR", MAT_AMBIENTCOLOR));
				lexemes.Insert(new LAPlexeme("MAT_DIFFUSECOLOR", MAT_DIFFUSECOLOR));
				lexemes.Insert(new LAPlexeme("MAT_SPECULARCOLOR", MAT_SPECULARCOLOR));
				lexemes.Insert(new LAPlexeme("MAT_SHININESS", MAT_SHININESS));
				lexemes.Insert(new LAPlexeme("MAT_TRANSPARENCY", MAT_TRANSPARENCY));
				lexemes.Insert(new LAPlexeme("MAT_TRANSFALLOFF", MAT_TRANSFALLOFF));
				lexemes.Insert(new LAPlexeme("MAT_REFLECTBLUR", MAT_REFLECTBLUR));
				lexemes.Insert(new LAPlexeme("MAT_TEXTUREMAP", MAT_TEXTUREMAP));
				lexemes.Insert(new LAPlexeme("MAT_OPACITYMAP", MAT_OPACITYMAP));
				lexemes.Insert(new LAPlexeme("MAT_TRANSFOFFENABLE", MAT_TRANSFOFFENABLE));
				lexemes.Insert(new LAPlexeme("MAT_REFLBLURENABLE", MAT_REFLBLURENABLE));
				lexemes.Insert(new LAPlexeme("MAT_BUMPMAP", MAT_BUMPMAP));
				lexemes.Insert(new LAPlexeme("MAT_MAPFILE", MAT_MAPFILE));
			lexemes.Insert(new LAPlexeme("EDIT_CONFIG1", EDIT_CONFIG1));
			lexemes.Insert(new LAPlexeme("EDIT_CONFIG2", EDIT_CONFIG2));
			lexemes.Insert(new LAPlexeme("EDIT_VIEW_P1", EDIT_VIEW_P1));
			lexemes.Insert(new LAPlexeme("EDIT_VIEW_P2", EDIT_VIEW_P2));
			lexemes.Insert(new LAPlexeme("EDIT_VIEW_P3", EDIT_VIEW_P3));
			lexemes.Insert(new LAPlexeme("EDIT_VIEW1", EDIT_VIEW1));
			lexemes.Insert(new LAPlexeme("EDIT_BACKGR", EDIT_BACKGR));
			lexemes.Insert(new LAPlexeme("EDIT_AMBIENT", EDIT_AMBIENT));
			lexemes.Insert(new LAPlexeme("EDIT_OBJECT", EDIT_OBJECT));
				lexemes.Insert(new LAPlexeme("OBJ_TRIMESH", OBJ_TRIMESH));
					lexemes.Insert(new LAPlexeme("TRI_VERTLIST", TRI_VERTLIST));
					lexemes.Insert(new LAPlexeme("TRI_FACELIST2", TRI_FACELIST2));
					lexemes.Insert(new LAPlexeme("TRI_FACELIST", TRI_FACELIST));
					lexemes.Insert(new LAPlexeme("TRI_MATERIAL", TRI_MATERIAL));
					lexemes.Insert(new LAPlexeme("TRI_MAPCOORD", TRI_MAPCOORD));
					lexemes.Insert(new LAPlexeme("TRI_SMOOTH", TRI_SMOOTH));
					lexemes.Insert(new LAPlexeme("TRI_LOCALAXIS", TRI_LOCALAXIS));
					lexemes.Insert(new LAPlexeme("TRI_VISIBLE", TRI_VISIBLE));
					lexemes.Insert(new LAPlexeme("TRI_MAPPINGSTD", TRI_MAPPINGSTD));
				lexemes.Insert(new LAPlexeme("OBJ_LIGHT", OBJ_LIGHT));
					lexemes.Insert(new LAPlexeme("LIGHT_OFF", LIGHT_OFF));
					lexemes.Insert(new LAPlexeme("LIGHT_SPOT", LIGHT_SPOT));
					lexemes.Insert(new LAPlexeme("LIGHT_INNER_RANGE", LIGHT_INNER_RANGE));
					lexemes.Insert(new LAPlexeme("LIGHT_OUTER_RANGE", LIGHT_OUTER_RANGE));
					lexemes.Insert(new LAPlexeme("LIGHT_MULTIPLIER", LIGHT_MULTIPLIER));
				lexemes.Insert(new LAPlexeme("OBJ_CAMERA", OBJ_CAMERA));
					lexemes.Insert(new LAPlexeme("CAM_UNKNOWN1", CAM_UNKNOWN1));
					lexemes.Insert(new LAPlexeme("CAM_UNKNOWN2", CAM_UNKNOWN2));
				lexemes.Insert(new LAPlexeme("OBJ_UNKNOWN1", OBJ_UNKNOWN1));
				lexemes.Insert(new LAPlexeme("OBJ_UNKNOWN2", OBJ_UNKNOWN2));
			lexemes.Insert(new LAPlexeme("EDIT_UNKNOWN01", EDIT_UNKNOWN01));
			lexemes.Insert(new LAPlexeme("EDIT_UNKNOWN02", EDIT_UNKNOWN02));
			lexemes.Insert(new LAPlexeme("EDIT_UNKNOWN03", EDIT_UNKNOWN03));
			lexemes.Insert(new LAPlexeme("EDIT_UNKNOWN04", EDIT_UNKNOWN04));
			lexemes.Insert(new LAPlexeme("EDIT_UNKNOWN05", EDIT_UNKNOWN05));
			lexemes.Insert(new LAPlexeme("EDIT_UNKNOWN06", EDIT_UNKNOWN06));
			lexemes.Insert(new LAPlexeme("EDIT_UNKNOWN07", EDIT_UNKNOWN07));
			lexemes.Insert(new LAPlexeme("EDIT_UNKNOWN08", EDIT_UNKNOWN08));
			lexemes.Insert(new LAPlexeme("EDIT_UNKNOWN09", EDIT_UNKNOWN09));
			lexemes.Insert(new LAPlexeme("EDIT_UNKNOWN10", EDIT_UNKNOWN10));
			lexemes.Insert(new LAPlexeme("EDIT_UNKNOWN11", EDIT_UNKNOWN11));
			lexemes.Insert(new LAPlexeme("EDIT_UNKNOWN12", EDIT_UNKNOWN12));
			lexemes.Insert(new LAPlexeme("EDIT_UNKNOWN13", EDIT_UNKNOWN13));
		lexemes.Insert(new LAPlexeme("KEYF3DS", KEYF3DS));
			lexemes.Insert(new LAPlexeme("KEYF_UNKNOWN1", KEYF_UNKNOWN1));
			lexemes.Insert(new LAPlexeme("KEYF_UNKNOWN2", KEYF_UNKNOWN2));
			lexemes.Insert(new LAPlexeme("KEYF_FRAMES", KEYF_FRAMES));
			lexemes.Insert(new LAPlexeme("KEYF_OBJDESCR", KEYF_OBJDESCR));
	lexemes.Insert(new LAPlexeme("COLOR_RGBFLOAT", COLOR_RGBFLOAT));
	lexemes.Insert(new LAPlexeme("COLOR_RGBBYTE", COLOR_RGBBYTE));
	lexemes.Insert(new LAPlexeme("COLOR_LINE_RGBFLOAT", COLOR_LINE_RGBFLOAT));
	lexemes.Insert(new LAPlexeme("COLOR_LINE_RGBBYTE", COLOR_LINE_RGBBYTE));
	lexemes.Insert(new LAPlexeme("VIEWPORT_TOP", VIEWPORT_TOP));
	lexemes.Insert(new LAPlexeme("VIEWPORT_BOTTOM", VIEWPORT_BOTTOM));
	lexemes.Insert(new LAPlexeme("VIEWPORT_LEFT", VIEWPORT_LEFT));
	lexemes.Insert(new LAPlexeme("VIEWPORT_RIGHT", VIEWPORT_RIGHT));
	lexemes.Insert(new LAPlexeme("VIEWPORT_FRONT", VIEWPORT_FRONT));
	lexemes.Insert(new LAPlexeme("VIEWPORT_BACK", VIEWPORT_BACK));
	lexemes.Insert(new LAPlexeme("VIEWPORT_USER", VIEWPORT_USER));
	lexemes.Insert(new LAPlexeme("VIEWPORT_CAMERA", VIEWPORT_CAMERA));
	lexemes.Insert(new LAPlexeme("VIEWPORT_LIGHT", VIEWPORT_LIGHT));

	lexemes.Insert(new LAPlexeme("INT_PERCENTAGE", INT_PERCENTAGE));
	lexemes.Insert(new LAPlexeme("FLOAT_PERCENTAGE", FLOAT_PERCENTAGE));

	lexemes.Insert(new LAPlexeme("MAIN_MLI", MAIN_MLI));

	lexemes.Insert(new LAPlexeme("MAIN_PRJ", MAIN_PRJ));
}


const char*
Parser3ds::GetLexeme(LAPtoken t)
{
	for (int32 i = 0; i < lexemes.CountItems(); i++)
	{
		LAPlexeme*		l = (LAPlexeme* )lexemes.ItemAt(i);
		if (l->GetToken() == t)
			return l->GetName();
	}
	return NULL;
}


//---------------------------------------------------------------
// a few useful routines
//---------------------------------------------------------------
void
Parser3ds::SkipChunk(uint16 chunkId, uint32 chunkLen)
{
	SkipBytes(chunkLen-6);
	const char*			s = GetLexeme(chunkId);
	*traceOutput << "skip chunk " << chunkId << ' ';
	if (s)
		*traceOutput << s << ' ';
	*traceOutput << int32(chunkLen) << " bytes" << eol;
}


void
Parser3ds::ReadRgbFloatChunk(uint32 len, pSpectra& color)
{
	*traceOutput << "ReadRgbFloatChunk: " << int32(len) << eol;
	*traceOutput << adv;
	uint32		currLen = 6;
	ReadFloat(color.r);
	ReadFloat(color.g);
	ReadFloat(color.b);
	currLen += 3*sizeof(float);
	*traceOutput << ret;
	checkChunk("ReadRgbFloatChunk");
}


void
Parser3ds::ReadRgbByteChunk(uint32 len, pSpectra& color)
{
	*traceOutput << "ReadRgbByteChunk: " << int32(len) << eol;
	*traceOutput << adv;
	uint32			currLen = 6;
	pColor			color24;
	ReadInt8(color24.r);
	ReadInt8(color24.g);
	ReadInt8(color24.b);
	color.Set(color24);
	currLen += 3*sizeof(uint8);

	while (currLen < len)
	{
		ReadChunk;
		skipChunk;
		currLen += chunkLen;
	}

	*traceOutput << ret;
	checkChunk("ReadRgbByteChunk");
}


void
Parser3ds::ReadColorChunk(uint32 len, pSpectra& color)
{
	*traceOutput << "ReadColorChunk: " << int32(len) << eol;
	*traceOutput << adv;
	uint32			currLen = 6;
	while (currLen < len)
	{
		ReadChunk;

		if (chunkId == COLOR_RGBFLOAT)
			ReadRgbFloatChunk(chunkLen, color);

		else if (chunkId == COLOR_RGBBYTE)
			ReadRgbByteChunk(chunkLen, color);

		else
		{
			skipChunk;
		}

		currLen += chunkLen;
	}

	*traceOutput << ret;
	checkChunk("ReadColorChunk");
}


void
Parser3ds::ReadAmountChunk(uint32 len, float& value)
{
	*traceOutput << "ReadAmountChunk: " << int32(len) << eol;
	*traceOutput << adv;
	uint32			currLen = 6;
	while (currLen < len)
	{
		ReadChunk;

		if (chunkId == INT_PERCENTAGE)
		{
			int16			intValue;
			ReadInt16(intValue);
			value = float(intValue);
			*traceOutput << intValue << eol;
		}
		else if (chunkId == FLOAT_PERCENTAGE)
		{
			ReadFloat(value);
			*traceOutput << value << eol;
		}
		else
		{
			skipChunk;
		}

		currLen += chunkLen;
	}

	*traceOutput << ret;
	checkChunk("ReadAmountChunk");
}


//---------------------------------------------------------------
// parsing routines
//---------------------------------------------------------------

void
Parser3ds::_Terminate()
{
}


status_t
Parser3ds::DoAbort()
{
	PierrotParser::Reset();
	return LAP_NO_ERROR;
}


status_t
Parser3ds::_Identify()
{
	ReadChunk;
	if (chunkId != MAIN_3DS)
		return LAP_BAD_INPUT_TYPE;
	return LAP_NO_ERROR;
}


status_t
Parser3ds::_Parse()
{
	PierrotParser::Reset();

	ReadChunk;
	if (chunkId != MAIN_3DS)
		return LAP_BAD_INPUT_TYPE;

	scene = new pScene((char* )GetLexer()->GetInputName());
	scene->ambient.Set(0.0, 0.0, 0.0);

	defaultMaterial = NewMaterial("default material");
	defaultMaterial->ambient.Set(0.1, 0.1, 0.1);
	defaultMaterial->diffuse.Set(0.25, 0.25, 0.25);
	defaultMaterial->specular.Set(0.4, 0.4, 0.4);
	defaultMaterial->shininess = 40.0;

	ReadMainChunk(chunkLen);
	CheckMaterials();

	pSphere				normSphere(pPoint3(0.0, 0.0, 0.0), 1.0);
	scene->Preprocess(&normSphere, appliInterface, 0.0, 1.0);

	// lights
	if (scene->lights.CountItems() == 0)
	{
		pLight*			l = NewDirectionalLight("directional light 1", pVector3(-0.5, 0.5, 0.8));
		l->ambient.Set(0.2, 0.2, 0.2);
		l->diffuse.Set(1.0, 1.0, 1.0);
		l->specular.Set(1.0, 1.0, 1.0);
	}

	Output(*traceOutput, scene, PRINT_LIGHT_ALL | PRINT_MATERIAL_ALL |
			PRINT_OBJECT_BOUNDS /*| PRINT_OBJECT_MATRIX | PRINT_OBJECT_POINTS*/);

	data = scene;
	scene = NULL;

	return LAP_NO_ERROR;
}


// main chunk
void
Parser3ds::ReadMainChunk(uint32 len)
{
	*traceOutput << "ReadMainChunk: " << int32(len) << eol;
	*traceOutput << adv;
	uint32			currLen = 6;
	while (currLen < len)
	{
		ReadChunk;

		if (chunkId == EDIT3DS)
			ReadEditChunk(chunkLen);

		else if (chunkId == KEYF3DS)
			ReadKeyfChunk(chunkLen);

		else
		{
			skipChunk;
		}

		currLen += chunkLen;
	}

	*traceOutput << ret;
	checkChunk("ReadMainChunk");
}


// location of main data
void
Parser3ds::ReadEditChunk(uint32 len)
{
	*traceOutput << "ReadEditChunk: " << int32(len) << eol;
	*traceOutput << adv;
	uint32			currLen = 6;
	while (currLen < len)
	{
		Lock();

		ReadChunk;

		if (chunkId == EDIT_BACKGR)
			ReadColorChunk(chunkLen, scene->background);

		else if (chunkId == EDIT_AMBIENT)
			ReadColorChunk(chunkLen, scene->ambient);

		else if (chunkId == EDIT_OBJECT)
			ReadObjectChunk(chunkLen);

		else if (chunkId == EDIT_MATERIAL)
			ReadEditMaterialChunk(chunkLen);

		else
		{
			skipChunk;
		}

		currLen += chunkLen;

		Unlock();
	}

	*traceOutput << ret;
	checkChunk("ReadEditChunk");
}


void
Parser3ds::ReadMaterialName(uint32 len, char* name)
{
	*traceOutput << "ReadMaterialName: " << int32(len) << eol;
	*traceOutput << adv;
	uint32			currLen = 6;
	ReadString(name, 17);
	currLen += strlen(name) + 1;

	while (currLen < len)
	{
		ReadChunk;
		skipChunk;
		currLen += chunkLen;
	}

	*traceOutput << ret;
	checkChunk("ReadMaterialName");
}


// material chunk
void
Parser3ds::ReadEditMaterialChunk(uint32 len)
{
	pMaterial*			newMaterial = NULL;
	float				tmp;

	*traceOutput << "ReadEditMaterialChunk: " << int32(len) << eol;
	*traceOutput << adv;
	uint32			currLen = 6;
	while (currLen < len)
	{
		ReadChunk;

		if (chunkId == MAT_NAME)
		{
			char		name[18];
			ReadMaterialName(chunkLen, name);
			newMaterial = NewMaterial(name);
			*traceOutput << "material name '" << name << '\'' << eol;
		}
		else if (chunkId == MAT_AMBIENTCOLOR)
			ReadColorChunk(chunkLen, newMaterial->ambient);
		else if (chunkId == MAT_DIFFUSECOLOR)
			ReadColorChunk(chunkLen, newMaterial->diffuse);
		else if (chunkId == MAT_SPECULARCOLOR)
			ReadColorChunk(chunkLen, newMaterial->specular);
		else if (chunkId == MAT_TRANSPARENCY)
		{
			ReadAmountChunk(chunkLen, tmp);
			newMaterial->opacity = 1.0 - float(tmp) * 0.01;
		}
		else if (chunkId == MAT_SHININESS)
		{
			ReadAmountChunk(chunkLen, tmp);
			newMaterial->shininess = float(tmp);
		}

//MAT_TRANSFALLOFF
//MAT_REFLECTBLUR
//MAT_TEXTUREMAP
//MAT_OPACITYMAP
//MAT_TRANSFOFFENABLE
//MAT_REFLBLURENABLE
//MAT_BUMPMAP
//MAT_MAPFILE

		else
		{
			skipChunk;
		}

		currLen += chunkLen;
	}

//	newMaterial->specular.Set(0.6, 0.6, 0.6);

	*traceOutput << ret;
	checkChunk("ReadEditMaterialChunk");
}


void
Parser3ds::ReadObjectChunk(uint32 len)
{
	*traceOutput << "ReadObjectChunk: " << int32(len) << eol;
	*traceOutput << adv;
	uint32			currLen = 6;

	char			name[18];
//	ReadString(name, 16);
	ReadString(name, 17);
	currLen += strlen(name)+1;
	*traceOutput << "object '" << name << "'" << eol;

	while (currLen < len)
	{
		ReadChunk;

		if (chunkId == OBJ_TRIMESH)
			ReadTriMeshChunk(chunkLen, name);

		else if (chunkId == OBJ_LIGHT)
			ReadLightChunk(chunkLen, name);

		else if (chunkId == OBJ_CAMERA)
			ReadCameraChunk(chunkLen);

		else
		{
			skipChunk;
		}

		currLen += chunkLen;
	}

	*traceOutput << ret;
	checkChunk("ReadObjectChunk");
}


void
Parser3ds::ReadTriMeshChunk(uint32 len, char* name)
{
	*traceOutput << "ReadTriMeshChunk: " << int32(len) << eol;
	*traceOutput << adv;
	uint32			currLen = 6;
	pObject*		object = NewObject(name);

	while (currLen < len)
	{
		ReadChunk;

		if (chunkId == TRI_VERTLIST)
			ReadVerticesChunk(chunkLen, object);
		else if (chunkId == TRI_FACELIST)
			ReadFacesChunk(chunkLen, object);
		else if (chunkId == TRI_LOCALAXIS)
			ReadTransfoChunk(chunkLen, object);
		else
		{
			skipChunk;
		}

		currLen += chunkLen;
	}

	*traceOutput << ret;
	checkChunk("ReadTriMeshChunk");
}


void
Parser3ds::ReadVerticesChunk(uint32 len, pObject* object)
{
	*traceOutput << "ReadVerticesChunk: " << int32(len) << eol;
	*traceOutput << adv;
	uint32			currLen = 6;
	uint16			nbVertices;
	ReadInt16(nbVertices);
	*traceOutput << int32(nbVertices) << " vertices" << eol;
	currLen += 2;

	object->vertices->SetAlloc(nbVertices, 0);

	for (uint16 i = 0; i < nbVertices; i++)
	{
		pPoint3				pt;
		ReadFloat(pt.x);
		ReadFloat(pt.y);
		ReadFloat(pt.z);
		object->vertices->AddItem(pt);
	}
	currLen += 3*sizeof(float) * nbVertices;

	*traceOutput << ret;
	checkChunk("ReadVerticesChunk");
}


void
Parser3ds::ReadFacesChunk(uint32 len, pObject* object)
{
	*traceOutput << "ReadFacesChunk: " << int32(len) << eol;
	*traceOutput << adv;
	uint32			currLen = 6;
	uint16			nbFaces;
	ReadInt16(nbFaces);
	*traceOutput << int32(nbFaces) << " faces" << eol;
	currLen += 2;

	object->faces.SetAlloc(nbFaces, 0);

	for (uint16 i = 0; i < nbFaces; i++)
	{
		uint16		i0, i1, i2, flags;
		ReadInt16(i0);
		ReadInt16(i1);
		ReadInt16(i2);
		ReadInt16(flags);

		object->faces.AddItem(new pFace(defaultMaterial, i0, i1, i2));
	}
	currLen += 4*sizeof(uint16)*nbFaces;

	while (currLen < len)
	{
		ReadChunk;

		if (chunkId == TRI_MATERIAL)
			ReadFaceMaterialChunk(chunkLen, object);
//		else if (chunkId == TRI_SMOOTH)
//			ReadFaceSmoothChunk(chunkLen, nbFaces, object);
		else
		{
			skipChunk;
		}

		currLen += chunkLen;
	}
	*traceOutput << ret;
	checkChunk("ReadFacesChunk");
}


void
Parser3ds::ReadTransfoChunk(uint32 len, pObject* /*object*/)
{
	*traceOutput << "ReadTransfoChunk: " << int32(len) << eol;
	*traceOutput << adv;
	uint32			currLen = 6;
	pVector4		u, v, n;
	pVector3		t;

	ReadFloat(u.x);
	ReadFloat(u.y);
	ReadFloat(u.z);
	u.w = 1.0;

	ReadFloat(v.x);
	ReadFloat(v.y);
	ReadFloat(v.z);
	v.w = 1.0;

	ReadFloat(n.x);
	ReadFloat(n.y);
	ReadFloat(n.z);
	n.w = 1.0;

	ReadFloat(t.x);
	ReadFloat(t.y);
	ReadFloat(t.z);

	currLen += 12 * sizeof(float);
/*
	object->matrix.SetX(u);
	object->matrix.SetY(v);
	object->matrix.SetZ(n);
	object->matrix.Translate(t.x, t.y, t.z);
*/
	*traceOutput << ret;
	checkChunk("ReadTransfoChunk");
}


void
Parser3ds::ReadFaceMaterialChunk(uint32 len, pObject* object)
{
	*traceOutput << "ReadFaceMaterialChunk: " << int32(len) << eol;
	*traceOutput << adv;
	uint32			currLen = 6;
	char			name[18];
	ReadString(name, 17);
	currLen += strlen(name) + 1;
	*traceOutput << "material '" << name << '\'' << eol;

	int32			materialIndex = scene->materials.Search(name);
	if (materialIndex < 0)
	{
		SkipBytes(len - currLen);
		return ;
	}
	pMaterial*		material = scene->materials[materialIndex];

	// Number of faces concerned with the current material
	uint16			nbFaces;
	ReadInt16(nbFaces);
	*traceOutput << int32(nbFaces) << " faces" << eol;
	currLen += 2;

	for (uint16 i = 0; i < nbFaces; i++)
	{
		uint16		faceIndex;
		ReadInt16(faceIndex);
		object->faces[faceIndex]->material = material;
	}
	currLen += sizeof(uint16) * nbFaces;

	while (currLen < len)
	{
		ReadChunk;
		skipChunk;
		currLen += chunkLen;
	}

	*traceOutput << ret;
	checkChunk("ReadFaceMaterialChunk");
}

/*
void
Parser3ds::ReadFaceSmoothChunk(uint32 len, uint16 nbFaces, pObject* object)
{
	*traceOutput << "ReadFaceSmoothChunk: " << int32(len) << eol;
	*traceOutput << adv;
	uint32			currLen = 6;
	uint16			tmp = 0;

	for (uint16 i = 0; i < nbFaces; i++)
	{
		// the nth bit is set if the face belongs to the nth smoothing group
		uint32		set;
		ReadInt32(set);
	}
	currLen += sizeof(uint32) * nbFaces;

	while (currLen < len)
	{
		ReadChunk;
		skipChunk;
		currLen += chunkLen;
	}

	*traceOutput << ret;
	checkChunk("ReadFaceSmoothChunk");
}
*/

void
Parser3ds::ReadCameraChunk(uint32 len)
{
	*traceOutput << "ReadCameraChunk: " << int32(len) << eol;
	*traceOutput << adv;
	uint32			currLen = 6;

	pPoint3			from;
	pPoint3			to;
	float			twist;
	float			lens;
	ReadFloat(from.x);	ReadFloat(from.y);	ReadFloat(from.z);
	ReadFloat(to.x);	ReadFloat(to.y);	ReadFloat(to.z);
	ReadFloat(twist);
	ReadFloat(lens);
	currLen += 8*sizeof(float);
/*
	*traceOutput << "from ";	Output(*traceOutput, from);	*traceOutput << eol;
	*traceOutput << "to ";		Output(*traceOutput, to);	*traceOutput << eol;
	*traceOutput << "twist = " << twist << eol;

	scene->viewsys.SetTo(from, to, twist);
*/
	while (currLen < len)
	{
		ReadChunk;
		skipChunk;
		currLen += chunkLen;
	}

	*traceOutput << ret;
	checkChunk("ReadCameraChunk");
}


void
Parser3ds::ReadLightChunk(uint32 len, char* name)
{
	*traceOutput << "ReadLightChunk: " << int32(len) << eol;
	*traceOutput << adv;
	uint32				currLen = 6;

	pPoint3				pos;
	pSpectra			color;

	bool				lightOn = true;
	bool				isSpot = false;
	pPoint3				target;
	float				hotSpot, fallOff;

	// light position
	ReadFloat(pos.x);	ReadFloat(pos.y);	ReadFloat(pos.z);
	currLen += 3*sizeof(float);

	while (currLen < len)
	{
		ReadChunk;

		if (chunkId == COLOR_RGBFLOAT)
			ReadRgbFloatChunk(chunkLen, color);
		else if (chunkId == LIGHT_OFF)
			lightOn = false;
		else if (chunkId == COLOR_RGBBYTE)
		{
			pColor			color24;
			ReadRgbByteChunk(chunkLen, color24);
			color.Set(color24);
		}
		else if (chunkId == LIGHT_SPOT)
		{
			ReadSpotChunk(chunkLen, target, hotSpot, fallOff);
			isSpot = true;
		}
		else
		{
			skipChunk;
		}

		currLen += chunkLen;
	}

	pLight*					newLight;
	if (isSpot)
	{
		pVector3			dir(pos, target);
		newLight = NewSpotlight(name, pos, dir, hotSpot, fallOff);
	}
	else
	{
		newLight = NewPositionalLight(name, pos);
	}
	newLight->diffuse = color;
	//--
	newLight->SetOff();

	*traceOutput << ret;
	checkChunk("ReadLightChunk");
}


void
Parser3ds::ReadSpotChunk(uint32 len, pPoint3& target, float& hotSpot, float& fallOff)
{
	*traceOutput << "ReadSpotChunk: " << int32(len) << eol;
	*traceOutput << adv;
	uint32				currLen = 6;

	ReadFloat(target.x);	ReadFloat(target.y);	ReadFloat(target.z);
	ReadFloat(hotSpot);
	ReadFloat(fallOff);
	currLen += 5*sizeof(float);

	while (currLen < len)
	{
		ReadChunk;
		skipChunk;
		currLen += chunkLen;
	}

	*traceOutput << ret;
	checkChunk("ReadSpotChunk");
}


void
Parser3ds::ReadKeyfChunk(uint32 len)
{
	*traceOutput << "ReadKeyfChunk: " << int32(len) << eol;
	*traceOutput << adv;
	uint32			currLen = 6;
	while (currLen < len)
	{
		Lock();

		ReadChunk;
/*
		if (chunkId == KEYF_OBJDESCR)
			ReadObjDesc(chunkLen);
		else if (chunkId == KEYF_FRAMES)
			ReadFrames(chunkLen);
		else
*/		{
			skipChunk;
		}

		currLen += chunkLen;

		Unlock();
	}

	*traceOutput << ret;
	checkChunk("ReadKeyfChunk");
}

/*
void
Parser3ds::ReadObjDesc(uint32 len)
{
	char			name[18];
	int16			parentIndex = -1;

	*traceOutput << "ReadObjDesc: " << int32(len) << eol;
	*traceOutput << adv;
	uint32			currLen = 6;

	while (currLen < len)
	{
		ReadChunk;

		if (chunkId == ODES_NAMHIER)
			ReadHierarchy(chunkLen, name, parentIndex);
		else if (chunkId == ODES_DUMMYNAME)
			ReadDummyName(chunkLen, name);
		else
		{
			skipChunk;
		}

		currLen += chunkLen;
	}

	int32				childIndex = scene->objects.Search(name);
	if (childIndex >= 0)
	{
		pObject*			child = scene->objects[childIndex];

		if (parentIndex >= 0)
		{
			pObject*			parent = scene->objects[parentIndex];
			parent->children += child;
		}
		else
			scene->roots += child;
	}

	*traceOutput << "object " << name << ", index " << childIndex
				<< ": child of ";
	if (parentIndex >= 0)
		*traceOutput << scene->objects[parentIndex]->name;
	else
		*traceOutput << "root";
	*traceOutput << eol << ret;
	checkChunk("ReadObjDesc");
}


void
Parser3ds::ReadHierarchy(uint32 len, char* name, int16& parentIndex)
{
	*traceOutput << "ReadHierarchy: " << int32(len) << eol;
	*traceOutput << adv;
	uint32			currLen = 6;

	int16			dummy1, dummy2;
	ReadString(name, 17);
	ReadInt16(dummy1);
	ReadInt16(dummy2);
	currLen += strlen(name)+1 + 2*sizeof(int16);

	ReadInt16(parentIndex);
	currLen += sizeof(int16);

	*traceOutput << ret;
	checkChunk("ReadHierarchy");
}


void
Parser3ds::ReadDummyName(uint32 len, char* name)
{
	*traceOutput << "ReadDummyName: " << int32(len) << eol;
	*traceOutput << adv;
	uint32			currLen = 6;

	ReadString(name, 17);
	currLen += strlen(name) + 1;

	*traceOutput << "dummy object name '" << name << '\'' << eol << ret;

	// skip chunks if any
	while (currLen < len)
	{
		ReadChunk;
		skipChunk;
		currLen += chunkLen;
	}

	checkChunk("ReadDummyName");
}
*/

/*
void
Parser3ds::ReadFrames(uint32 len)
{
	*traceOutput << "ReadFrames: " << int32(len) << eol;
	*traceOutput << adv;
	uint32			currLen = 6;

	int32			startFrame, endFrame;
	ReadInt32(startFrame);
	ReadInt32(endFrame);
	*traceOutput << "frames " << startFrame << " to " << endFrame << eol;
	currLen += 2*sizeof(int32);

	while (currLen < len)
	{
		ReadChunk;
		skipChunk;
		currLen += chunkLen;
	}

	*traceOutput << ret;
	checkChunk("ReadFrames");
}
*/


void
Parser3ds::CheckMaterials()
{
	for (int32 i = 0; i < scene->materials.CountItems(); i++)
	{
		pMaterial*			mat = scene->materials[i];
		if ((mat->ambient == pSpectra(0.0, 0.0, 0.0))
			&& (mat->diffuse == pSpectra(0.0, 0.0, 0.0)))
		{
			mat->SetAmbientAndDiffuse(pSpectra(0.3, 0.4, 0.6));
			mat->specular.Set(0.6, 0.6, 0.6);
			mat->shininess = 40.0;
		}
	}
}
