//
//
//	NFF objects parser
//
//												(C) 1998 JoOl


#include "ParserNff.h"


#if __POWERPC__
#pragma export on
#endif
char			handlerName[]		= "SPD's Neutral File Format v3.1 (.nff)";
char			handlerVersion[]	= "1.0.0";
char			handlerAuthor[]		= "Alain GAUTHEROT";
char			handlerEmail[]		= "gauthero@club-internet.fr";
#if __POWERPC__
#pragma export reset
#endif



ParserNff*
instantiate_parser(int32 bs)
{
	return new ParserNff(bs);
}


ParserNff::ParserNff(int32 blockSize)
: LAPtextParser()
{
	SetLexer(new LexerNff(max_c(1024, blockSize), new LAPlexemeBuffer(128, true)));

	SetTo(new LAPtracer(128, LAP_BUFFER_OUTPUT_ON_OVERFLOW, LAP_BUFFER_SCREEN_OUTPUT),
			new LAPtextBufferedOutput(128, LAP_BUFFER_OUTPUT_ON_OVERFLOW, LAP_BUFFER_NO_OUTPUT/*LAP_BUFFER_SCREEN_OUTPUT*/),
			new LAPdisplay(),
			LAP_OWN_ERROR_OUTPUT | LAP_OWN_TRACE_OUTPUT | LAP_OWN_DISPLAY);

	conePrecision = DEFAULT_CONE_PRECISION;
	spherePrecision = DEFAULT_SPHERE_PRECISION;
	torusPrecision = DEFAULT_TORUS_PRECISION;
}


ParserNff::~ParserNff()
{
	if (data)
		delete (pScene* )data;
}


status_t
ParserNff::DoAbort()
{
	PierrotParser::Reset();
	return LAP_NO_ERROR;
}


status_t
ParserNff::_Identify()
{
	bool		silent = traceOutput->IsSilent();
	traceOutput->SetSilent(true);

	ParseView();

	traceOutput->SetSilent(silent);

	return LAP_NO_ERROR;
}


status_t
ParserNff::_Parse()
{
	PierrotParser::Reset();

	scene = new pScene((char* )GetLexer()->GetInputName(),
						2, 2,					// lights
						16, 16,					// materials
						16, 16);				// objects
	scene->ambient.Set(0.5, 0.5, 0.5);
	currentObject = NewObject(scene->name,	POLY_VERTICES_ALLOC,
											POLY_VERTICES_BLOCK,
											POLY_FACES_ALLOC,
											POLY_FACES_BLOCK);

	currentMaterial = NewMaterial("default material");
	currentMaterial->ambient.Set(0.1, 0.1, 0.1);
	currentMaterial->diffuse.Set(0.4, 0.5, 0.6);
	currentMaterial->specular.Set(0.4, 0.4, 0.4);
	currentMaterial->shininess = 40.0;

	ParseView();
	while (!Peek(LAP_TOKEN_EOF))
	{
		switch(currentToken)
		{
			case TOKEN_BACKGROUND:		ParseBackground();		break;
			case TOKEN_LIGHT:			ParseLight();			break;
			case TOKEN_MATERIAL:		ParseMaterial();		break;
			case TOKEN_CONE:			ParseCone();			break;
			case TOKEN_SPHERE:			ParseSphere();			break;
			case TOKEN_POLYGON:			ParsePolygon(false);	break;
			case TOKEN_PATCH:			ParsePolygon(true);		break;
			case TOKEN_TORUS:			ParseTorus();			break;
			case TOKEN_RING:			ParseRing();			break;
			case TOKEN_PRISM:			ParsePrism();			break;

			default:
				*traceOutput << "line " << lexer->GetLineNumber() << ": unknown token ("
					<< currentToken << ' ' << lexer->GetLexeme(currentToken) << ' '
					<< lexer->GetStringLiteral() << ')' << eol;
				SkipLine();
				break;
		}
	}

	pSphere				normSphere(pPoint3(0.0, 0.0, 0.0), 1.0);
	scene->Preprocess(&normSphere, appliInterface, 0.0, 1.0);

	// lights
	scene->ambient.Set(0.6, 0.6, 0.6);
	if (scene->lights.CountItems() == 0)
	{
		pLight*			l1 = NewDirectionalLight("directional light 1", pVector3(-0.5, 0.5, 0.8));
		l1->ambient.Set(0.2, 0.2, 0.2);
		l1->diffuse.Set(1.0, 1.0, 1.0);
		l1->specular.Set(1.0, 1.0, 1.0);
	}

	Output(*traceOutput, scene, PRINT_LIGHT_ALL | PRINT_MATERIAL_ALL |
			PRINT_OBJECT_BOUNDS | PRINT_OBJECT_MATRIX | PRINT_OBJECT_POINTS);

	data = scene;
	scene = NULL;

	return LAP_NO_ERROR;
}


void
ParserNff::ParseColor(pSpectra& c)
{
	ParseFloat(c.r);
	ParseFloat(c.g);
	ParseFloat(c.b);
}


void
ParserNff::ParsePoint(pPoint3& pt)
{
	ParseFloat(pt.x);
	ParseFloat(pt.y);
	ParseFloat(pt.z);
}


void
ParserNff::ParseVector(pVector3& v)
{
	ParseFloat(v.x);
	ParseFloat(v.y);
	ParseFloat(v.z);
}


void
ParserNff::ParseView()
{
	pPoint3				from;
	pPoint3				at;
	pVector3			up;
	float				angle, hither;
	int32				xres, yres;

	Match(TOKEN_VIEW);
		Match(TOKEN_FROM);		ParsePoint(from);
		Match(TOKEN_AT);		ParsePoint(at);
		Match(TOKEN_UP);		ParseVector(up);
		Match(TOKEN_ANGLE);		ParseFloat(angle);
		Match(TOKEN_HITHER);	ParseFloat(hither);
		Match(TOKEN_RESOLUTION);
			LexInt(xres);
			LexInt(yres);
/*
	*traceOutput << "view from ";		Output(*traceOutput, from);
	*traceOutput << adv;
		*traceOutput << eol << "at ";		Output(*traceOutput, at);
		*traceOutput << eol << "up ";		Output(*traceOutput, up);
		*traceOutput << eol << "angle " << angle << ", hither " << hither;
		*traceOutput << eol << "resolution " << xres << " x " << yres << eol;
	*traceOutput << ret;
*/
}


void
ParserNff::ParseBackground()
{
	Match(TOKEN_BACKGROUND);
		ParseColor(scene->background);
/*
	*traceOutput << "background ";		Output(*traceOutput, scene->background);
	*traceOutput << eol;
*/
}


void
ParserNff::ParseLight()
{
	pPoint3				posn;
	pSpectra			color(0.25, 0.3, 0.3);

	Match(TOKEN_LIGHT);
		ParsePoint(posn);
		if (Peek(LAP_FLOAT_LITERAL))
			ParseColor(color);
/*
	*traceOutput << "light at ";				Output(*traceOutput, posn);
	*traceOutput << adv << eol << "color ";		Output(*traceOutput, color);
	*traceOutput << ret << eol;
*/
	char				buf[32];
	sprintf(buf, "light %d", scene->lights.CountItems()+1);
	pLight*				l = NewPositionalLight(buf, posn);
	l->ambient.Set(0.2, 0.2, 0.2);
	l->diffuse.Set(color);
	l->specular.Set(0.7, 0.7, 0.7);
}


void
ParserNff::ParseMaterial()
{
	pSpectra			color;
	float				Kd, Ks, shininess;
	float				T, refractionIndex;

	Match(TOKEN_MATERIAL);
		ParseColor(color);
		ParseFloat(Kd);
		ParseFloat(Ks);
		ParseFloat(shininess);
		ParseFloat(T);
		ParseFloat(refractionIndex);

	Kd = max_c(min_c(Kd, 1.0), 0.0);
	Ks = max_c(min_c(Ks, 1.0), 0.0);
	shininess = max_c(min_c(shininess * 5.0, 128.0), 0.0);
/*
	*traceOutput << "material:" << eol << adv;
		*traceOutput << "color ";		Output(*traceOutput, color);
		*traceOutput << eol << "Kd=" << Kd << ", Ks=" << Ks << ", shininess=" << shininess;
		*traceOutput << eol << "T=" << T << ", refraction index=" << refractionIndex << eol;
	*traceOutput << ret;
*/
	char			buf[32];
	sprintf(buf, "material %d", scene->materials.CountItems());
	currentMaterial = NewMaterial(buf);
	currentMaterial->ambient.Set(0.2, 0.2, 0.2);
	currentMaterial->diffuse.Set(color);
	currentMaterial->specular.Set(Ks, Ks, Ks);
	currentMaterial->shininess = shininess;
}


void
ParserNff::ParseCone()
{
	pCone				cone;

	Match(TOKEN_CONE);
		ParsePoint(cone.base);		ParseFloat(cone.baseRadius);
		ParsePoint(cone.apex);		ParseFloat(cone.apexRadius);
cone.isCapped = false;

// A negative value for both radii means that only the inside of the object is visible
	if (cone.baseRadius < 0.0)
		cone.baseRadius = -cone.baseRadius;
	if (cone.apexRadius < 0.0)
		cone.apexRadius = -cone.apexRadius;

	char				buf[32];
	sprintf(buf, "cone %d", scene->objects.CountItems());
	pObject*			obj = BuildCone(buf, cone, currentMaterial, conePrecision);
	scene->objects.AddItem(obj);

/*
	*traceOutput << "cone:" << eol;
	*traceOutput << adv;
		*traceOutput << "base: ";	Output(*traceOutput, cone.base);
		*traceOutput << ", radius=" << cone.baseRadius << eol;
		*traceOutput << "apex: ";	Output(*traceOutput, cone.apex);
		*traceOutput << ", radius=" << cone.apexRadius << eol;
	*traceOutput << ret;
*/
}


void
ParserNff::ParseSphere()
{
	pSphere			sphere;

	Match(TOKEN_SPHERE);
		ParsePoint(sphere.center);
		ParseFloat(sphere.radius);

// If the radius is negative, then only the sphere's inside is visible
	if (sphere.radius < 0.0)
		sphere.radius = -sphere.radius;

	char				buf[32];
	sprintf(buf, "sphere %d", scene->objects.CountItems());
	pObject*			obj = BuildSphere(buf, sphere, currentMaterial,
											spherePrecision, spherePrecision);
	scene->objects.AddItem(obj);
/*
	*traceOutput << "sphere: " << eol;
		Output(*traceOutput, sphere.center);
		*traceOutput << ", radius=" << sphere.radius << eol;
*/
}


void
ParserNff::ParsePolygon(bool isPatch)
{
	int32			nbVert;

	Match(isPatch ? TOKEN_PATCH : TOKEN_POLYGON);
		LexInt(nbVert);
/*
		if (isPatch)
			*traceOutput << "pp ";
		else
			*traceOutput << "p ";
		*traceOutput << nbVert << eol;
		*traceOutput << adv;
*/
		PierrotArray<int32 >		indexes(nbVert, 1);
		for (int32 i = 0; i < nbVert; i++)
		{
			pPoint3			pt;
			ParsePoint(pt);
			if (isPatch)
				SkipLine();

			int32			vertexIndex = currentObject->vertices->Search(pt);
			if (vertexIndex < 0)
			{
				vertexIndex = currentObject->vertices->CountItems();
				currentObject->vertices->AddItem(pt);
			}
			indexes.AddItem(vertexIndex);

			Output(*traceOutput, pt);
//			*traceOutput << " / " << vertexIndex << eol;
		}

//		*traceOutput << ret;

		currentObject->faces.AddItem(new pFace(currentMaterial, nbVert, indexes.Items()));
}


void
ParserNff::ParseTorus()
{
	pTorus			torus;

	Match(TOKEN_TORUS);
		ParsePoint(torus.center);
		ParseVector(torus.normal);
		ParseFloat(torus.radius);
		ParseFloat(torus.a);
		ParseFloat(torus.b);

	char				buf[32];
	sprintf(buf, "torus %d", scene->objects.CountItems());
	pObject*			obj = BuildTorus(buf, torus, currentMaterial,
										torusPrecision, torusPrecision);
	scene->objects.AddItem(obj);
/*
	*traceOutput << "torus: " << eol;
		...
*/
}


void
ParserNff::ParseRing()
{
	pRing			ring;

	Match(TOKEN_RING);
		ParsePoint(ring.base);
		ParseVector(ring.normal);
		ParseFloat(ring.innerRadius);
		ParseFloat(ring.outerRadius);

	char				buf[32];
	sprintf(buf, "ring %d", scene->objects.CountItems());
	pObject*			obj = BuildRing(buf, ring, currentMaterial,
										spherePrecision);
	scene->objects.AddItem(obj);
/*
	*traceOutput << "ring: " << eol;
		...
*/
}


void
ParserNff::ParsePrism()
{
	PierrotArray<pPoint3 >	base;
	int32					n;
	pVector3				dir;

	Match(TOKEN_PRISM);
		LexInt(n);
		base.SetAlloc(n, 0);
		for (int32 i = 0; i < n; i++)
		{
			pPoint3			pt;
			ParsePoint(pt);
			base.AddItem(pt);
		}
		ParseVector(dir);

	pPrism					prism(base, dir, true);
	char					buf[32];
	sprintf(buf, "prism %d", scene->objects.CountItems());
	pObject*				obj = prism.BuildPrism(buf, currentMaterial, 4);
	scene->objects.AddItem(obj);
/*
	*traceOutput << "prism: " << eol;
		...
*/
}
