//
//
//	Pierrot 3d scene
//
//										(C) JoOl 1998

// La mort n'est rien, mais vivre vaincu et sans gloire,
// c'est mourir tous les jours (Napoleon)


#include "PierrotImpExp.h"

#include <string.h>
#include "PierrotScene.h"



//--------------------------------------------------------
// 3d scene
//--------------------------------------------------------

pScene::pScene(char* n,
				int32 li, int32 lb,
				int32 mi, int32 mb,
				int32 oi, int32 ob)
:lights(li, lb),
 materials(mi, mb),
 objects(oi, ob)
{
	ambient.Set(0.2, 0.2, 0.2);
	name = strdup(n);
	background.Set(0.0, 0.0, 0.0);
	viewsys.SetDefault();
}


pScene::~pScene()
{
	delete[] name;
}


bool
pScene::Lock()
{
	return locker.Lock();
}


void
pScene::Unlock()
{
	locker.Unlock();
}


bool
pScene::IsNormalized()
{
	return sbounds != originalBounds;
}


void
pScene::Unnormalize()
{
	NormalizeTo(originalBounds, NULL, 0.0, 1.0);
}


void
pScene::SetMaterialCounts()
{
	for (int32 i = 0; i < materials.CountItems(); i++)
		materials[i]->refCount = 0;

	for (int32 i = 0; i < objects.CountItems(); i++)
	{
		pObject*			obji = objects[i];
		for (int32 j = 0; j < obji->faces.CountItems(); j++)
			obji->faces[j]->material->refCount++;
	}
}


void
pScene::SetNormals(LAPappliInterface* interface, float start, float end)
{
	if (interface)
		interface->ReportProgress(start, 1.0, 1.0);

	// clear all normals
	for (int32 i = 0; i < objects.CountItems(); i++)
		objects[i]->ClearNormals();

	float			step = ((end - start) * 0.5) / objects.CountItems();
	float			currentRate = start;

	// compute vertices normals
	// vertices owners are unlikely to have faces, but who knows...
	for (int32 i = 0; i < objects.CountItems(); i++)
	{
		objects[i]->SetNormals();

		currentRate += step;
		if (interface)
			interface->ReportProgress(currentRate, 1.0, 1.0);
	}

	currentRate = start + (end - start) * 0.5;
	// normalize the normals
	for (int32 i = 0; i < objects.CountItems(); i++)
	{
		objects[i]->NormNormals();

		currentRate += step;
		if (interface)
			interface->ReportProgress(currentRate, 1.0, 1.0);
	}
}


void
pScene::NormalizeTo(pSphere& s, LAPappliInterface* interface, float start, float end)
{
	MoveTo(s.center, interface, start, start + (end - start) * 0.5);
	ScaleTo(s.radius, interface, start + (end - start) * 0.5, end);
}


void
pScene::MoveTo(pPoint3& o, LAPappliInterface* interface, float start, float end)
{
	if (interface)
		interface->ReportProgress(start, 1.0, 1.0);

	pVector3			translate(sbounds.center, o);

	for (int32 i = 0; i < lights.CountItems(); i++)
		lights[i]->MoveBy(translate);
	viewsys.target += translate;
	viewsys.origin += translate;

	float				step = (end - start) / objects.CountItems();
	float				currentRate = start;
	for (int32 i = 0; i < objects.CountItems(); i++)
	{
		objects[i]->MoveBy(translate);

		currentRate += step;
		if (interface)
			interface->ReportProgress(currentRate, 1.0, 1.0);
	}
}


void
pScene::ScaleTo(float s, LAPappliInterface* interface, float start, float end)
{
	if (interface)
		interface->ReportProgress(start, 1.0, 1.0);

	float			ss = s / sbounds.radius;

	for (int32 i = 0; i < lights.CountItems(); i++)
		lights[i]->ScaleBy(ss);
	viewsys.distance *= ss;

	float				step = (end - start) / objects.CountItems();
	float				currentRate = start;
	for (int32 i = 0; i < objects.CountItems(); i++)
	{
		objects[i]->ScaleBy(ss);

		currentRate += step;
		if (interface)
			interface->ReportProgress(currentRate, 1.0, 1.0);
	}
}


void
pScene::SetBounds(LAPappliInterface* interface, float start, float end)
{
	if (interface)
		interface->ReportProgress(start, 1.0, 1.0);

	float			step = ((end - start) * 0.5) / objects.CountItems();
	float			currentRate = start;
	for (int32 i = 0; i < objects.CountItems(); i++)
	{
		objects[i]->SetBounds();

		currentRate += step;
		if (interface)
			interface->ReportProgress(currentRate, 1.0, 1.0);
	}

	// set box bounds
	currentRate = start + (end - start) * 0.5;
	for (int32 i = 0; i < objects.CountItems(); i++)
	{
		if (i == 0)
			bounds = objects[i]->bounds;
		else
			bounds |= objects[i]->bounds;

		currentRate += step;
		if (interface)
			interface->ReportProgress(currentRate, 1.0, 1.0);
	}

	// set sphere bounds
	pPoint3				center = bounds.Center();
	float				radius = 0.0;
	for (int32 i = 0; i < objects.CountItems(); i++)
	{
		if (objects[i]->HasVertices())
		{
			float			currLen;
			currLen = pVector3(center, objects[i]->sbounds.center).Length()
						+ objects[i]->sbounds.radius;
			if (currLen > radius)
				radius = currLen;
		}
	}
	sbounds.Set(center, radius);
}


void
pScene::Preprocess(pSphere* normSphere, LAPappliInterface* interface,
					float start, float end)
{
	float			curr = start;
	float			step = (end-start) * 0.2;

	SetMaterialCounts();
	SetNormals(interface, curr, curr + step);
	curr += step;

	SetBounds(interface, curr, curr + step);
	originalBounds = sbounds;
	curr += step;

	if (normSphere)
	{
		NormalizeTo(*normSphere, interface, curr, curr + 2*step);
		curr += 2*step;

		SetBounds(interface, curr, curr + step);
	}

	if (interface)
		interface->ReportProgress(end, end, 1.0);
}


void
pScene::ReverseFaces()
{
	for (int32 i = 0; i < objects.CountItems(); i++)
	{
		pObject*			obji = objects[i];
		for (int32 j = 0; j < obji->faces.CountItems(); j++)
			obji->faces[j]->Reverse();
	}
}


bool
pScene::HasTransparentFaces()
{
	for (int32 i = 0; i < objects.CountItems(); i++)
	{
		pObject*			obji = objects[i];

		for (int32 j = 0; j < obji->faces.CountItems(); j++)
			if (obji->faces[j]->material->opacity < 1.0)
				return true;
	}
	return false;
}


void
pScene::SetDefaultView()
{
	viewsys.SetTo(sbounds.radius * 1.5, sbounds.center);
}


void
pScene::Merge(pScene* s2)//, LAPappliInterface* interface, float start, float end)
{
	// add materials
	materials.GrowForItems(s2->materials.CountItems());
	for (int32 i = 0; i < s2->materials.CountItems(); i++)
		materials.AddItem(s2->materials[i]);
	s2->materials.ClearItems();

	// add objects
	objects.GrowForItems(s2->objects.CountItems());
	for (int32 i = 0; i < s2->objects.CountItems(); i++)
		objects.AddItem(s2->objects[i]);
	s2->objects.ClearItems();
}
