//
//
//	Amachi OpenGL configuration view
//
//


#ifndef _AMACHI_OPENGL_CONFIG_H_
#define _AMACHI_OPENGL_CONFIG_H_


#include <View.h>


enum
{
	AMACHI_PERSPECTIVE_PROJECTION	= 0x100,
	AMACHI_ORTHOGRAPHIC_PROJECTION,

	AMACHI_FLAT_SHADING,
	AMACHI_SMOOTH_SHADING,

	AMACHI_NO_TRANSPARENCY,
	AMACHI_USE_TRANSPARENCY,

	AMACHI_NO_ZBUFFER,
	AMACHI_USE_ZBUFFER,

	AMACHI_DOTS_RENDERING,
	AMACHI_LINES_RENDERING,
	AMACHI_FILLED_RENDERING,

	AMACHI_NO_CULLING,
	AMACHI_CW_CULLING,
	AMACHI_CCW_CULLING,

	AMACHI_NO_FULLSCREEN,
	AMACHI_USE_FULLSCREEN,

	AMACHI_MANUAL_MOTION,
	AMACHI_AUTO_MOTION,

	AMACHI_NO_LIGHTING				= 0x00,
	AMACHI_USE_LIGHTING				= 0x01,
	AMACHI_LOCAL_VIEWER_LIGHTING	= 0x02,
	AMACHI_TWO_SIDES_LIGHTING		= 0x04
};


enum
{
	AMACHI_PROJECTION				= 0x00,
	AMACHI_SHADING,
	AMACHI_ZBUFFER,
	AMACHI_TRANSPARENCY,
	AMACHI_LIGHTING,
	AMACHI_FRONT_RENDERING,
	AMACHI_BACK_RENDERING,
	AMACHI_CULLING,
	AMACHI_FULLSCREEN,
	AMACHI_MOTION,

	AMACHI_MODES_COUNT
};


// basic structure
typedef struct GLconfig
{
public:

	int16			modes[AMACHI_MODES_COUNT];

	GLconfig()
	{};
	~GLconfig()
	{};

	void			SetModes(int16 m)
	{
		for (int32 i = 0; i < AMACHI_MODES_COUNT; i++)
			modes[i] = m;
	}

	void			SetDefault()
	{
		modes[AMACHI_PROJECTION]		= AMACHI_PERSPECTIVE_PROJECTION;
		modes[AMACHI_FRONT_RENDERING]	= AMACHI_FILLED_RENDERING;
		modes[AMACHI_BACK_RENDERING]	= AMACHI_LINES_RENDERING;
		modes[AMACHI_SHADING]			= AMACHI_FLAT_SHADING;
		modes[AMACHI_CULLING]			= AMACHI_CW_CULLING;
		modes[AMACHI_ZBUFFER]			= AMACHI_USE_ZBUFFER;
		modes[AMACHI_TRANSPARENCY]		= AMACHI_NO_TRANSPARENCY;
		modes[AMACHI_LIGHTING]			= AMACHI_NO_LIGHTING;
		modes[AMACHI_FULLSCREEN]		= AMACHI_NO_FULLSCREEN;
		modes[AMACHI_MOTION]			= AMACHI_MANUAL_MOTION;
	};

	int32			operator == (GLconfig& c)
	{
		for (int32 i = 0; i < AMACHI_MODES_COUNT; i++)
			if (modes[i] != c.modes[i])
				return 0;
		return 1;
	};
	int32			operator != (GLconfig& c)
	{
		for (int32 i = 0; i < AMACHI_MODES_COUNT; i++)
			if (modes[i] != c.modes[i])
				return 1;
		return 0;
	};
	GLconfig&		operator = (GLconfig& c)
	{
		for (int32 i = 0; i < AMACHI_MODES_COUNT; i++)
			modes[i] = c.modes[i];
		return *this;
	};

	inline int16	GetProjection()
	{ return modes[AMACHI_PROJECTION]; };
	inline int16	GetShading()
	{ return modes[AMACHI_SHADING]; };
	inline int16	GetTransparency()
	{ return modes[AMACHI_TRANSPARENCY]; };
	inline int16	GetZbuffer()
	{ return modes[AMACHI_ZBUFFER]; };
	inline int16	GetLighting()
	{ return modes[AMACHI_LIGHTING]; };
	inline int16	GetFrontFace()
	{ return modes[AMACHI_FRONT_RENDERING]; };
	inline int16	GetBackFace()
	{ return modes[AMACHI_BACK_RENDERING]; };
	inline int16	GetCulling()
	{ return modes[AMACHI_CULLING]; }
	inline int16	GetFullscreen()
	{ return modes[AMACHI_FULLSCREEN]; }
	inline int16	GetMotion()
	{ return modes[AMACHI_MOTION]; }

	inline void		SetProjection(int16 m)
	{ modes[AMACHI_PROJECTION] = m; };
	inline void		SetShading(int16 m)
	{ modes[AMACHI_SHADING] = m; };
	inline void		SetTransparency(int16 m)
	{ modes[AMACHI_TRANSPARENCY] = m; };
	inline void		SetZbuffer(int16 m)
	{ modes[AMACHI_ZBUFFER] = m; };
	inline void		SetLighting(int16 m)
	{ modes[AMACHI_LIGHTING] = m; };
	inline void		SetFrontFace(int16 m)
	{ modes[AMACHI_FRONT_RENDERING] = m; };
	inline void		SetBackFace(int16 m)
	{ modes[AMACHI_BACK_RENDERING] = m; };
	inline void		SetCulling(int16 m)
	{ modes[AMACHI_CULLING] = m; }
	inline void		SetFullscreen(int16 m)
	{ modes[AMACHI_FULLSCREEN] = m; }
	inline void		SetMotion(int16 m)
	{ modes[AMACHI_MOTION] = m; }

	void			ReadFrom(BMessage* msg)
	{
		msg->FindInt16("projection", &modes[AMACHI_PROJECTION]);
		msg->FindInt16("shading", &modes[AMACHI_SHADING]);
		msg->FindInt16("transparency", &modes[AMACHI_TRANSPARENCY]);
		msg->FindInt16("zbuffer", &modes[AMACHI_ZBUFFER]);
		msg->FindInt16("lighting", &modes[AMACHI_LIGHTING]);
		msg->FindInt16("front_face", &modes[AMACHI_FRONT_RENDERING]);
		msg->FindInt16("back_face", &modes[AMACHI_BACK_RENDERING]);
		msg->FindInt16("culling", &modes[AMACHI_CULLING]);
		msg->FindInt16("fullscreen", &modes[AMACHI_FULLSCREEN]);
		msg->FindInt16("motion", &modes[AMACHI_MOTION]);
	}
	void			WriteTo(BMessage* msg)
	{
		msg->AddInt16("projection", modes[AMACHI_PROJECTION]);
		msg->AddInt16("shading", modes[AMACHI_SHADING]);
		msg->AddInt16("transparency", modes[AMACHI_TRANSPARENCY]);
		msg->AddInt16("zbuffer", modes[AMACHI_ZBUFFER]);
		msg->AddInt16("lighting", modes[AMACHI_LIGHTING]);
		msg->AddInt16("front_face", modes[AMACHI_FRONT_RENDERING]);
		msg->AddInt16("back_face", modes[AMACHI_BACK_RENDERING]);
		msg->AddInt16("culling", modes[AMACHI_CULLING]);
		msg->AddInt16("fullscreen", modes[AMACHI_FULLSCREEN]);
		msg->AddInt16("motion", modes[AMACHI_MOTION]);
	}

	void			Print()
	{
		if (modes[AMACHI_PROJECTION] == AMACHI_PERSPECTIVE_PROJECTION)
			printf("projection: perspective\n");
		else if (modes[AMACHI_PROJECTION] == AMACHI_ORTHOGRAPHIC_PROJECTION)
			printf("projection: orthographic\n");
		else
			printf("projection: *** %d\n", modes[AMACHI_PROJECTION]);

		if (modes[AMACHI_SHADING] == AMACHI_FLAT_SHADING)
			printf("shading: flat\n");
		else if (modes[AMACHI_SHADING] == AMACHI_SMOOTH_SHADING)
			printf("shading: smooth\n");
		else
			printf("shading: *** %d\n", modes[AMACHI_SHADING]);

		if (modes[AMACHI_TRANSPARENCY] == AMACHI_USE_TRANSPARENCY)
			printf("transparency: on\n");
		else if (modes[AMACHI_TRANSPARENCY] == AMACHI_NO_TRANSPARENCY)
			printf("transparency: off\n");
		else
			printf("transparency: *** %d\n", modes[AMACHI_TRANSPARENCY]);

		if (modes[AMACHI_ZBUFFER] == AMACHI_USE_ZBUFFER)
			printf("zbuffer: on\n");
		else if (modes[AMACHI_ZBUFFER] == AMACHI_NO_ZBUFFER)
			printf("zbuffer: off\n");
		else
			printf("zbuffer: *** %d\n", modes[AMACHI_ZBUFFER]);

		printf("lighting: ");
		if (modes[AMACHI_LIGHTING] & AMACHI_USE_LIGHTING)
			printf("on");
		else
			printf("off");
		if (modes[AMACHI_LIGHTING] & AMACHI_LOCAL_VIEWER_LIGHTING)
			printf(" / local viewer ");
		if (modes[AMACHI_LIGHTING] & AMACHI_TWO_SIDES_LIGHTING)
			printf(" / 2 sides");
		printf("\n");

		if (modes[AMACHI_FRONT_RENDERING] == AMACHI_DOTS_RENDERING)
			printf("front faces: dots\n");
		else if (modes[AMACHI_FRONT_RENDERING] == AMACHI_LINES_RENDERING)
			printf("front faces: lines\n");
		else if (modes[AMACHI_FRONT_RENDERING] == AMACHI_FILLED_RENDERING)
			printf("front faces: filled\n");
		else
			printf("front faces: *** %d\n", modes[AMACHI_FRONT_RENDERING]);

		if (modes[AMACHI_BACK_RENDERING] == AMACHI_DOTS_RENDERING)
			printf("back faces: ***dots***\n");
		else if (modes[AMACHI_BACK_RENDERING] == AMACHI_LINES_RENDERING)
			printf("back faces: lines\n");
		else if (modes[AMACHI_BACK_RENDERING] == AMACHI_FILLED_RENDERING)
			printf("back faces: filled\n");
		else
			printf("back faces: ***\n");

		if (modes[AMACHI_CULLING] == AMACHI_NO_CULLING)
			printf("culling: off\n");
		else if (modes[AMACHI_CULLING] == AMACHI_CW_CULLING)
			printf("culling: CW\n");
		else if (modes[AMACHI_CULLING] == AMACHI_CCW_CULLING)
			printf("culling: CCW\n");
		else
			printf("culling: *** %d\n", modes[AMACHI_CULLING]);

		if (modes[AMACHI_FULLSCREEN] == AMACHI_NO_FULLSCREEN)
			printf("fullscreen: off\n");
		else if (modes[AMACHI_FULLSCREEN] == AMACHI_USE_FULLSCREEN)
			printf("fullscreen: on\n");
		else
			printf("fullscreen: ***\n");

		if (modes[AMACHI_MOTION] == AMACHI_MANUAL_MOTION)
			printf("motion: manual\n");
		else if (modes[AMACHI_MOTION] == AMACHI_AUTO_MOTION)
			printf("motion: auto\n");
		else
			printf("motion: ***\n");
	};
};


class					AmachiGLConfig: public BView
{
	BCheckBox*			projection;
	BCheckBox*			shading;
	BCheckBox*			transparency;
	BCheckBox*			zbuffer;
	BCheckBox*			lighting;
	BCheckBox*			lighting_local;
	BCheckBox*			lighting_2sides;

	BCheckBox*			fullscreen;
	BCheckBox*			motion;
	BMenuField*			front_faces;
		BMenuItem*			front_dots;
		BMenuItem*			front_lines;
		BMenuItem*			front_filled;
	BMenuField*			back_faces;
		BMenuItem*			back_lines;
		BMenuItem*			back_filled;
	BMenuField*			culling;
		BMenuItem*			culling_off;
		BMenuItem*			culling_cw;
		BMenuItem*			culling_ccw;
	void				BuildView();

public:

						AmachiGLConfig(BRect fr, const char* n, uint32 , uint32 );
	virtual				~AmachiGLConfig();

	void				SetEnabled(bool );
	void				EnableTransparency(bool );
	void				EnableAll();
	void				SetTarget(BHandler* );

	void				ReadProjection(GLconfig& );
	void				ReadShading(GLconfig& );
	void				ReadTransparency(GLconfig& );
	void				ReadZbuffer(GLconfig& );
	void				ReadLighting(GLconfig& );
	void				ReadFrontFace(GLconfig& );
	void				ReadBackFace(GLconfig& );
	void				ReadCulling(GLconfig& );
	void				ReadFullscreen(GLconfig& );
	void				ReadMotion(GLconfig& );
	void				ReadFrom(GLconfig& );
	void				ReadFrom(BMessage* );

	void				WriteProjection(GLconfig& );
	void				WriteShading(GLconfig& );
	void				WriteTransparency(GLconfig& );
	void				WriteZbuffer(GLconfig& );
	void				WriteLighting(GLconfig& );
	void				WriteFrontFace(GLconfig& );
	void				WriteBackFace(GLconfig& );
	void				WriteCulling(GLconfig& );
	void				WriteFullscreen(GLconfig& );
	void				WriteMotion(GLconfig& );
	void				WriteTo(GLconfig& );
	void				WriteTo(BMessage* );

	void				AttachedToWindow();
};


#endif /* _AMACHI_OPENGL_CONFIG_H_ */
