//
//
//	3D2 objects parser
//
//										(C) JoOl 1998


#include "Parser3d2.h"
#include "LAPerrors.h"


#if __POWERPC__
#pragma export on
#endif
char			handlerName[]		= "CAD-3D v2.0 (.3d2)";
char			handlerVersion[]	= "1.1.0";
char			handlerAuthor[]		= "Alain GAUTHEROT";
char			handlerEmail[]		= "gauthero@club-internet.fr";
#if __POWERPC__
#pragma export reset
#endif


Parser3d2*
instantiate_parser(int32 bs, LAPtracer* eo)
{
	return new Parser3d2(bs, eo);
}


Parser3d2::Parser3d2(int32 blockSize, LAPtracer* eo)
{
	// input is a big-endian binary file
	SetLexer(new LAPbinLexer(max_c(1024, blockSize), LAP_BIG_ENDIAN));

	uint32			ownFlags =	LAP_OWN_TRACE_OUTPUT | LAP_OWN_DISPLAY;
	if (!eo)
	{
		eo = new LAPtracer(256, LAP_BUFFER_OUTPUT_ON_OVERFLOW, LAP_BUFFER_SCREEN_OUTPUT);
		ownFlags |=	LAP_OWN_ERROR_OUTPUT;
	}

	// error output, tracer output, display
	SetTo(	eo,
			new LAPtextBufferedOutput(128, LAP_BUFFER_OUTPUT_ON_OVERFLOW, LAP_BUFFER_NO_OUTPUT/*LAP_BUFFER_SCREEN_OUTPUT*/),
			new LAPdisplay(),
			ownFlags);
}


Parser3d2::~Parser3d2()
{
//	lexer deleted in parent classes
//	so are the error output, the tracer and the display (cf. SetTo call)

	if (data)
		delete (pScene* )data;
}


void
Parser3d2::_Terminate()
{
}


status_t
Parser3d2::DoAbort()
{
	PierrotParser::Reset();
	return LAP_NO_ERROR;
}


status_t
Parser3d2::_Identify()
{
	if (!ParseHeader())
		return LAP_BAD_INPUT_TYPE;
	if (header.Sig != SIGNATURE_3D2)
		return LAP_BAD_INPUT_TYPE;
	return LAP_NO_ERROR;
}


status_t
Parser3d2::_Parse()
{
	PierrotParser::Reset();

	if (!ParseHeader())
		return LAP_BAD_INPUT_TYPE;
	if (header.Sig != SIGNATURE_3D2)
		return LAP_BAD_INPUT_TYPE;

	if (Lock())
	{
		ProcessHeader();
		Unlock();
	}

	for (uint16 i = 0; i < header.NbObj; i++)
	{
		if (Lock())
		{
			ParseObject();
			Unlock();
		}
	}

	pSphere				normSphere(pPoint3(0.0, 0.0, 0.0), 1.0);
	scene->Preprocess(&normSphere, appliInterface, 0.0, 1.0);

	if (scene->lights.CountItems() == 0)
	{
		pLight*			l = NewDirectionalLight("directional light 1", pVector3(-0.5, 0.8, -0.5));
		l->ambient.Set(0.2, 0.2, 0.2);
		l->diffuse.Set(1.0, 1.0, 1.0);
		l->specular.Set(1.0, 1.0, 1.0);
	}

	Output(*traceOutput, scene, PRINT_BOUNDS | PRINT_LIGHT_ALL | PRINT_MATERIAL_ALL |
			PRINT_OBJECT_BOUNDS | PRINT_OBJECT_MATRIX);

	data = scene;
	scene = NULL;
	return LAP_NO_ERROR;
}


bool
Parser3d2::ParseHeader()
{
	try
	{
		ReadInt16(header.Sig);
		ReadInt16(header.NbObj);

		ReadInt16(header.LightAon);
		ReadInt16(header.LightBon);
		ReadInt16(header.LightCon);

		ReadInt16(header.BrightA);
		ReadInt16(header.BrightB);
		ReadInt16(header.BrightC);

		ReadInt16(header.BrightAmb);

		ReadInt16(header.Az);
		ReadInt16(header.Ay);
		ReadInt16(header.Ax);

		ReadInt16(header.Bz);
		ReadInt16(header.By);
		ReadInt16(header.Bx);

		ReadInt16(header.Cz);
		ReadInt16(header.Cy);
		ReadInt16(header.Cx);

		for (uint16 i = 0; i < 32; i++)
		{
			ReadInt16(header.ObjColor[i]);
		}
		for (uint16 i = 0; i < 32; i++)
		{
			ReadInt16(header.ColGroup[i]);
		}

		ReadInt16(header.PalType);
		ReadInt16(header.WireColor);
		ReadInt16(header.OutlineColor);

		SkipBytes(86);
		return true;
	}

	catch (...)
	{
		return false;
	}
}


void
Parser3d2::ProcessHeader()
{
	scene = new pScene((char* )GetLexer()->GetInputName(),
						3, 0,					// lights
						16, 0,					// materials
						header.NbObj, 0);		// objects

	pMaterial*			material;
	for (int32 i = 0; i < 16; i++)
	{
		int32			col = header.ObjColor[i];
		char			buf[32];
		sprintf(buf, "material %d", scene->materials.CountItems());
		material = NewMaterial(buf);
		material->SetAmbientAndDiffuse(pSpectra(float((col >> 8) & 0xf) / 15.0,
												float((col >> 4) & 0xf) / 15.0,
												float((col)      & 0xf) / 15.0));
		material->specular.Set(0.5, 0.5, 0.5);
		material->shininess = 10.0;
	}

	// at most 3 positional lights
	float					conv;
	pLight*					light;
	pSpectra				col;
	if (header.LightAon)
	{
		light = NewPositionalLight("light A",
								pPoint3(float(header.Ax) * 0.01,
										float(header.Ay) * 0.01,
										float(header.Az) * 0.01)
								);
		conv = float(header.BrightA) * 0.1;
		col.Set(conv, conv, conv);
		light->SetAmbientAndDiffuse(col);
		light->specular.Set(0.7, 0.7, 0.7);
	}
	if (header.LightBon)
	{
		light = NewPositionalLight("light B",
								pPoint3(float(header.Bx) * 0.01,
										float(header.By) * 0.01,
										float(header.Bz) * 0.01)
								);
		conv = float(header.BrightB) * 0.1;
		col.Set(conv, conv, conv);
		light->SetAmbientAndDiffuse(col);
		light->specular.Set(0.7, 0.7, 0.7);
	}
	if (header.LightCon)
	{
		light = NewPositionalLight("light C",
								pPoint3(float(header.Cx) * 0.01,
										float(header.Cy) * 0.01,
										float(header.Cz) * 0.01)
								);
		conv = float(header.BrightC) * 0.1;
		col.Set(conv, conv, conv);
		light->SetAmbientAndDiffuse(col);
		light->specular.Set(0.7, 0.7, 0.7);
	}
}


void
Parser3d2::ParseObject()
{
	char			objName[10];
	ReadBytes(objName, 9);
	*traceOutput << "obj '" << objName << '\'' << eol;

	pObject*		obj = NewObject(objName);

	ParseVertices(obj);
	ParseFaces(obj);
}


void
Parser3d2::ParseVertices(pObject* obj)
{
	uint16			nbVertices;
	ReadInt16(nbVertices);

	obj->vertices->SetAlloc(nbVertices, 0);

	for (uint16 i = 0; i < nbVertices; i++)
	{
		int16		_x, _y, _z;
		ReadInt16(_x);
		ReadInt16(_y);
		ReadInt16(_z);

		pPoint3			pt(float(_x)*0.01, float(_y)*0.01, float(_z)*0.01);
		obj->vertices->AddItem(pt);
	}

	*traceOutput << adv << nbVertices << " vertices parsed" << eol << ret;
}


void
Parser3d2::ParseFaces(pObject* obj)
{
	uint16			nbFaces;
	ReadInt16(nbFaces);

	obj->faces.SetAlloc(nbFaces, 0);

	for (uint16 i = 0; i < nbFaces; i++)
	{
		int16			face[4];
		ReadInt16(face[0]);
		ReadInt16(face[1]);
		ReadInt16(face[2]);
		ReadInt16(face[3]);
		obj->faces.AddItem(new pFace(scene->materials[face[3] & 0xf], face[0], face[1], face[2]));
	}

	*traceOutput << adv << nbFaces << " faces parsed" << eol << ret;
}
