//
//
//	IOB objects parser
//
//										(C) JoOl 1998

#include "ParserIob.h"
#include "LAPerrors.h"



#if __POWERPC__
#pragma export on
#endif
char			handlerName[]		= "Imagine v3.0 (.iob)";
char			handlerVersion[]	= "0.8.0";
char			handlerAuthor[]		= "Alain GAUTHEROT";
char			handlerEmail[]		= "gauthero@club-internet.fr";
#if __POWERPC__
#pragma export reset
#endif



ParserIob*
instantiate_parser(int32 bs)
{
	return new ParserIob(bs);
}


ParserIob::ParserIob(int32 blockSize)
{
	// input is a big-endian binary file
	SetLexer(new LAPbinLexer(max_c(1024, blockSize),
			/* LAP_BIG_ENDIAN or LAP_LITLE_ENDIAN */
			LAP_BIG_ENDIAN));

	// error output, tracer output, display
	SetTo(new LAPtracer(128, LAP_BUFFER_OUTPUT_ON_OVERFLOW, LAP_BUFFER_SCREEN_OUTPUT),
			new LAPtextBufferedOutput(128, LAP_BUFFER_OUTPUT_ON_OVERFLOW, LAP_BUFFER_NO_OUTPUT/*LAP_BUFFER_SCREEN_OUTPUT*/),
			new LAPdisplay(),
			LAP_OWN_ERROR_OUTPUT | LAP_OWN_TRACE_OUTPUT | LAP_OWN_DISPLAY);
}


ParserIob::~ParserIob()
{
	if (data)
		delete (pScene* )data;
}


void
ParserIob::_Terminate()
{
}


status_t
ParserIob::DoAbort()
{
	PierrotParser::Reset();
	return LAP_NO_ERROR;
}


status_t
ParserIob::_Identify()
{
	ReadChunk;
	if (chunkId != CHUNK_FORM)
		return LAP_BAD_INPUT_TYPE;

	uint32			formType;
	ReadInt32(formType);
	if (formType != CHUNK_TDDD)
		return LAP_BAD_INPUT_TYPE;

	return LAP_NO_ERROR;
}


status_t
ParserIob::_Parse()
{
	PierrotParser::Reset();

	// read file header
	ReadChunk;
	if (chunkId != CHUNK_FORM)
		return LAP_BAD_INPUT_TYPE;

	uint32			formType;
	ReadInt32(formType);
	if (formType != CHUNK_TDDD)
		return LAP_BAD_INPUT_TYPE;

	// new scene
	scene = new pScene((char* )GetLexer()->GetInputName());

	// create default material
	defaultMaterial = NewMaterial("default material");
	defaultMaterial->ambient.Set(0.1, 0.1, 0.1);
	defaultMaterial->diffuse.Set(0.4, 0.5, 0.6);
	defaultMaterial->specular.Set(0.4, 0.4, 0.4);
	defaultMaterial->shininess = 40.0;

	// read data
	ReadMainChunk(chunkLen-sizeof(formType));
	CheckMaterials();

	// normalize
	pSphere				normSphere(pPoint3(0.0, 0.0, 0.0), 1.0);
	scene->Preprocess(&normSphere, appliInterface, 0.0, 1.0);

	// add lights if necessary
	scene->ambient.Set(0.6, 0.6, 0.6);
	if (scene->lights.CountItems() == 0)
	{
		pLight*			l1 = NewDirectionalLight("directional light 1", pVector3(-0.5, 0.5, 0.8));
		l1->ambient.Set(0.2, 0.2, 0.2);
		l1->diffuse.Set(1.0, 1.0, 1.0);
		l1->specular.Set(1.0, 1.0, 1.0);
	}

	Output(*traceOutput, scene, PRINT_LIGHT_ALL | PRINT_MATERIAL_ALL |
			PRINT_OBJECT_BOUNDS | PRINT_OBJECT_MATRIX | PRINT_OBJECT_POINTS);

	data = scene;
	scene = NULL;
	return LAP_NO_ERROR;
}


//---------------------------------------------
void
ParserIob::SkipChunk(uint32 chunkId, uint32 chunkLen)
{
	char		tmp[6];
	tmp[0] = (chunkId >> 24) & 0xff;
	tmp[1] = (chunkId >> 16) & 0xff;
	tmp[2] = (chunkId >>  8) & 0xff;
	tmp[3] = (chunkId      ) & 0xff;
	tmp[4] = 0;

	SkipBytes(chunkLen);
	*traceOutput << "skip chunk " << tmp << ' ' << int32(chunkLen) << " bytes" << eol;
}


void
ParserIob::ReadFract(float& out)
{
	int32			a;
	ReadInt32(a);
	out = float(a) / 65536.0;
}


void
ParserIob::ReadColor(pSpectra& out)
{
	pColor			color;
	ReadInt8(color.r);
	ReadInt8(color.g);
	ReadInt8(color.b);
	out.Set(color);
}


void
ParserIob::ReadPaddedColor(pColor& out)
{
	int8			dummy;
	ReadInt8(dummy);
	ReadInt8(out.r);
	ReadInt8(out.g);
	ReadInt8(out.b);
}


void
ParserIob::ReadPoint(pPoint3& out)
{
	ReadFract(out.x);
	ReadFract(out.y);
	ReadFract(out.z);
}


void
ParserIob::ReadVector(pVector3& out)
{
	ReadFract(out.x);
	ReadFract(out.y);
	ReadFract(out.z);
}


void
ParserIob::ReadMatrix(pMatrix3& out)
{
	ReadVector(out.GetX());
	ReadVector(out.GetY());
	ReadVector(out.GetZ());
}



//---------------------------------------------
void
ParserIob::ReadMainChunk(uint32 len)
{
	*traceOutput << "ReadMainChunk: " << int32(len) << eol;
	*traceOutput << adv;

	uint32			currLen = 0;
	while (currLen < len)
	{
		ReadChunk;

		if (chunkId == CHUNK_OBJ)
			ReadObjectChunk(chunkLen);
		else
		{
			skipChunk;
		}
		currLen += chunkLen+8;
		PadChunk;
	}

	*traceOutput << ret;
}


void
ParserIob::ReadObjectChunk(uint32 len)
{
	*traceOutput << "ReadObjectChunk: " << int32(len) << eol;
	*traceOutput << adv;

	uint32			currLen = 0;
	while (currLen < len)
	{
		ReadChunk;

		if (chunkId == OBJ_DESC)
			ReadDescChunk(chunkLen);

		else if (chunkId == CHUNK_TOBJ)
		{
			*traceOutput << "TOBF" << eol << ret;
		}
		else
		{
			skipChunk;
		}
		currLen += chunkLen+8;
		PadChunk;
	}

//	*traceOutput << ret;
}


void
ParserIob::ReadDescChunk(uint32 len)
{
	pColor				color;
	char				objectName[20];
	pObject*			object = NULL;
	pMaterial*			oldDefaultMaterial = defaultMaterial;

	*traceOutput << "ReadDescChunk: " << int32(len) << eol;
	*traceOutput << adv;

	uint32			currLen = 0;
	while (currLen < len)
	{
		ReadChunk;

		if (chunkId == DESC_NAME)
		{
			ReadNameChunk(chunkLen, objectName);
			if (*objectName == 0)
				sprintf(objectName, "object %d", scene->objects.CountItems());
			object = NewObject(objectName);
		}

		else if ((chunkId == DESC_SHAPE2) || (chunkId == DESC_SHAPE))
			ReadShapeChunk(chunkLen);

		else if (chunkId == DESC_POSITION)
			ReadPositionChunk(chunkLen, object);

		else if (chunkId == DESC_AXIS)
			ReadAxisChunk(chunkLen, object);
/*
		else if (chunkId == DESC_STND)
			ReadStateNameDataChunk(chunkLen);
*/
		else if (chunkId == DESC_POINTS)
			ReadPointsChunk(chunkLen, object);

		else if (chunkId == DESC_EDGES)
			ReadEdgesChunk(chunkLen, object);

		else if (chunkId == DESC_FACES)
			ReadFacesChunk(chunkLen, object);

		else if (chunkId == DESC_OBJ_COLOR)
		{
			ReadPaddedColor(color);
			*traceOutput << "object color ";
			Output(*traceOutput, color);
			*traceOutput << eol;

			char				buf[32];
			sprintf(buf, "%s material", object->name);
			defaultMaterial = NewMaterial(buf);
			defaultMaterial->diffuse.Set(pSpectra(color));
			defaultMaterial->ambient.Set(0.2, 0.2, 0.2);
		}

		else if (chunkId == DESC_OBJ_REFLECTIVITY)
		{
			ReadPaddedColor(color);
			*traceOutput << "object reflectivity ";
			Output(*traceOutput, color);
			*traceOutput << eol;
		}

		else if (chunkId == DESC_OBJ_TRANSPARENCY)
		{
			ReadPaddedColor(color);
			*traceOutput << "object transparency ";
			Output(*traceOutput, color);
			*traceOutput << eol;

			defaultMaterial->opacity = 1.0 - pSpectra(color).GreyLevel();
		}

		else if (chunkId == DESC_OBJ_SPECULARITY)
		{
			ReadPaddedColor(color);
			*traceOutput << "object specularity ";
			Output(*traceOutput, color);
			*traceOutput << eol;

			defaultMaterial->specular.Set(pSpectra(color));
		}

		else if (chunkId == DESC_FACES_COLOR)
			ReadFacesColorChunk(chunkLen, object);

		else if (chunkId == DESC_FACES_REFLECTIVITY)
			ReadFacesReflectivityChunk(chunkLen, object);

		else if (chunkId == DESC_FACES_TRANSPARENCY)
			ReadFacesTransparencyChunk(chunkLen, object);

		else if (chunkId == DESC_FOG_LENGTH)
		{
			float			fogLength;
			ReadFract(fogLength);
			*traceOutput << "fog length = " << fogLength << eol;
		}

		else if (chunkId == DESC_OBJ_PROPERTIES)
			ReadObjectPropertiesChunk(chunkLen);

		else if (chunkId == DESC_INTENSITY)
			ReadIntensityChunk(chunkLen, color);

		else
		{
			skipChunk;
		}
		currLen += chunkLen+8;
		PadChunk;
	}

	defaultMaterial = oldDefaultMaterial;

//	*traceOutput << ret;
}


void
ParserIob::ReadNameChunk(uint32 len, char* name)
{
	*traceOutput << "ReadNameChunk: " << int32(len) << eol;
	Assert("ReadNameChunk", 18);

	name[18] = 0;
	ReadBytes(name, 18);
	*traceOutput << adv << '\'' << name << '\'' << eol << ret;
}


void
ParserIob::ReadShapeChunk(uint32 len)
{
	*traceOutput << "ReadShapeChunk: " << int32(len) << eol;
	Assert("ReadShapeChunk", 4);

	int16			shape;
	int16			lamp;
	ReadInt16(shape);
	ReadInt16(lamp);

	*traceOutput << adv << shape << ", " << lamp << eol << ret;
}


void
ParserIob::ReadPositionChunk(uint32 len, pObject* /*object*/)
{
	*traceOutput << "ReadPositionChunk: " << int32(len) << eol;
	Assert("ReadPositionChunk", 12);

	pVector3			position;
	ReadVector(position);
//	object->matrix.Translate(position.x, position.y, position.z);

	*traceOutput << adv;
		Output(*traceOutput, position);
	*traceOutput << eol << ret;
}


void
ParserIob::ReadAxisChunk(uint32 len, pObject* /*object*/)
{
	*traceOutput << "ReadAxisChunk: " << int32(len) << eol;
	Assert("ReadAxisChunk", 36);
	*traceOutput << adv;
	pVector3			a;
	pVector4			b;

	ReadVector(a);
	b.Set(a.x, a.y, a.z, 0.0);
//	object->matrix.SetX(b);
	*traceOutput << "u = ";	Output(*traceOutput, a);	*traceOutput << eol;

	ReadVector(a);
	b.Set(a.x, a.y, a.z, 0.0);
//	object->matrix.SetY(b);
	*traceOutput << "v = ";	Output(*traceOutput, a);	*traceOutput << eol;

	ReadVector(a);
	b.Set(a.x, a.y, a.z, 0.0);
//	object->matrix.SetZ(b);
	*traceOutput << "n = ";	Output(*traceOutput, a);	*traceOutput << eol;

	*traceOutput << ret;
}

/*
void
ParserIob::ReadStateNameDataChunk(uint32 len)
{
	*traceOutput << "ReadStateNameDataChunk: " << int32(len) << eol;
	*traceOutput << adv;

	uint32			currLen = 0;
	while (currLen < len)
	{
		ReadChunk;

		if (chunkId == STND_STATE_ID)
			ReadStateIdChunk(chunkLen);

		else if (chunkId == STND_STATE_DATA)
			ReadStateDataChunk(chunkLen);

		else
		{
			skipChunk;
		}
		currLen += chunkLen+8;
		PadChunk;
	}

	*traceOutput << ret;
}


void
ParserIob::ReadStateIdChunk(uint32 len)
{
	*traceOutput << "ReadStateIdChunk: " << int32(len) << eol;
	Assert("ReadStateIdChunk", 20);

	char			stateName[20];
	uint16			stateFlags;
	stateName[18] = 0;
	ReadBytes(stateName, 18);
	ReadInt16(stateFlags);

	*traceOutput << adv;
		*traceOutput << "state name = " << stateName << eol;
		*traceOutput << "state flags = " << stateFlags << eol;
	*traceOutput << ret;
}


void
ParserIob::ReadStateDataChunk(uint32 len)
{
	*traceOutput << "ReadStateDataChunk: " << int32(len) << eol;
	*traceOutput << adv;
	int16				tag;
	int16				flags;
	ReadInt16(tag);
	ReadInt16(flags);
	*traceOutput << "tag = " << tag << ", " << flags << eol;

	if (tag == STDT_FACE_COLOR)
	{
		*traceOutput << "faces color, ";
		ReadFacesProperty(len-4);
	}
	else if (tag == STDT_FACE_TRANSPARENCY)
	{
		*traceOutput << "faces transparency, ";
		ReadFacesProperty(len-4);
	}
	else if (tag == STDT_FACE_REFLECTIVITY)
	{
		*traceOutput << "faces reflectivity, ";
		ReadFacesProperty(len-4);
	}
	else if (tag == STDT_ATTRIBUTE)
	{
		*traceOutput << "object properties:" << eol;
		ReadObjectProperties(len-4);
	}

	*traceOutput << ret;
}


void
ParserIob::ReadFacesProperty(uint32 len)
{
	int32			nbFaces = len / 3;
	*traceOutput << nbFaces << " faces" << eol;
	*traceOutput << adv;

	pColor			color;
	for (int32 i = 0; i < nbFaces; i++)
	{
		ReadInt8(color.r);
		ReadInt8(color.g);
		ReadInt8(color.b);
		Output(*traceOutput, color);
		*traceOutput << eol;
	}

	*traceOutput << ret;
}


void
ParserIob::ReadObjectProperties(uint32 len)
{
	*traceOutput << "ReadObjectProperties: " << int32(len) << eol;
	Assert("ReadObjectProperties", 44);

	char			props[10];
	props[8] = 0;
	uint16			lamp;
	uint16			flags;
	pSpectra		intensity;
	float			fogLength;
	pColor			objectColor;
	pColor			objectReflectivity;
	pColor			objectTransparency;
	pColor			objectSpecularity;

	ReadBytes(props, 8);
	ReadInt16(lamp);
	ReadInt16(flags);
	ReadFract(intensity.r);	ReadFract(intensity.g);	ReadFract(intensity.b);
	intensity.Clamp(0.0, 1.0);
	ReadFract(fogLength);
	ReadPaddedColor(objectColor);
	ReadPaddedColor(objectReflectivity);
	ReadPaddedColor(objectTransparency);
	ReadPaddedColor(objectSpecularity);

	*traceOutput << adv;
		*traceOutput << "props = " << props << eol;
		*traceOutput << "lamp = " << lamp << eol;
		*traceOutput << "intensity = ";
			Output(*traceOutput, intensity);
			*traceOutput << eol;
		*traceOutput << "fog length = " << fogLength << eol;
		*traceOutput << "object color = ";
			Output(*traceOutput, objectColor);
			*traceOutput << eol;
		*traceOutput << "object reflectivity = ";
			Output(*traceOutput, objectReflectivity);
			*traceOutput << eol;
		*traceOutput << "object transparency = ";
			Output(*traceOutput, objectTransparency);
			*traceOutput << eol;
		*traceOutput << "object specularity = ";
			Output(*traceOutput, objectSpecularity);
			*traceOutput << eol;
	*traceOutput << ret;
}
*/

void
ParserIob::ReadPointsChunk(uint32 len, pObject* object)
{
	*traceOutput << "ReadPointsChunk: " << int32(len) << eol;
	int16				nbVertices;
	ReadInt16(nbVertices);
	*traceOutput << adv << nbVertices << " vertices" << eol;
	*traceOutput << adv;

	object->vertices->SetAlloc(nbVertices, 0);

	for (int16 i = 0; i < nbVertices; i++)
	{
		pPoint3				pt;
		ReadPoint(pt);
		object->vertices->AddItem(pt);
//		Output(*traceOutput, pt);
//		*traceOutput << eol;
	}
	*traceOutput << ret << ret;
}


void
ParserIob::ReadEdgesChunk(uint32 len, pObject* object)
{
	*traceOutput << "ReadEdgesChunk: " << int32(len) << eol;
	int16				nbEdges;
	ReadInt16(nbEdges);
	*traceOutput << adv << nbEdges << " edges" << eol;
	*traceOutput << adv;

	object->lines.SetAlloc(nbEdges, 0);

	for (int16 i = 0; i < nbEdges; i++)
	{
		int16			start, end;
		ReadInt16(start);
		ReadInt16(end);
		object->lines.AddItem(pSegment(start, end));
//		*traceOutput << start << '-' << end << eol;
	}
	*traceOutput << ret << ret;
}


void
ParserIob::ReadFacesChunk(uint32 len, pObject* object)
{
	*traceOutput << "ReadFacesChunk: " << int32(len) << eol;
	int16				nbFaces;
	ReadInt16(nbFaces);
	*traceOutput << adv << nbFaces << " faces" << eol;
	*traceOutput << adv;

	object->faces.SetAlloc(nbFaces, 0);

	for (int16 i = 0; i < nbFaces; i++)
	{
		int16			e0, e1, e2;
		ReadInt16(e0);
		ReadInt16(e1);
		ReadInt16(e2);

		int32			i0 = object->lines[e0].start;
		int32			i1 = object->lines[e0].end;
		int32			i2 = object->lines[e1].start;
		if ((i0 == i2) || (i1 == i2))
			i2 = object->lines[e1].end;

		object->faces.AddItem(new pFace(defaultMaterial, i0, i1, i2));
//		*traceOutput << i0 << ' ' << i1 << ' ' << i2 << ": " << (void* )defaultMaterial << eol;
//		*traceOutput << i0 << ' ' << i1 << ' ' << i2 << eol;
	}

	// trash the 'lines', used to store Imagine's edges
	object->lines.DelAlloc();

	*traceOutput << ret << ret;
}


void
ParserIob::ReadFacesColorChunk(uint32 len, pObject* object)
{
	if (object->vertices->CountItems() == 0)
	{
		SkipBytes(len);
		*traceOutput << "ReadFacesColorChunk: '" << object->name << "' is an object of unsupported type" << eol;
		return ;
	}

	*traceOutput << "ReadFacesColorChunk: " << int32(len) << eol;
	int16			nbFaces;
	ReadInt16(nbFaces);
	Assert("ReadFacesColorChunk", 2 + 3*nbFaces);

	pSpectra			spectra;
	*traceOutput << adv << nbFaces << " faces" << eol;
	*traceOutput << adv;
	for (int32 i = 0; i < nbFaces; i++)
	{
		pColor			color;
		ReadInt8(color.r);
		ReadInt8(color.g);
		ReadInt8(color.b);
		spectra.Set(color);

		object->faces[i]->material = SearchMaterial(spectra);

//		Output(*traceOutput, spectra);
//		*traceOutput << eol;
	}
	*traceOutput << ret << ret;
}


void
ParserIob::ReadFacesReflectivityChunk(uint32 len, pObject* object)
{
	if (object->vertices->CountItems() == 0)
	{
		SkipBytes(len);
		return ;
	}

	*traceOutput << "ReadFacesReflectivityChunk: " << int32(len) << eol;
	int16			nbFaces;
	ReadInt16(nbFaces);
	Assert("ReadFacesReflectivityChunk", 2 + 3*nbFaces);

	pSpectra			spectra;
	*traceOutput << adv << nbFaces << " faces" << eol;
	*traceOutput << adv;
	for (int32 i = 0; i < nbFaces; i++)
	{
		pColor			color;
		ReadInt8(color.r);
		ReadInt8(color.g);
		ReadInt8(color.b);
		spectra.Set(color);

		// search material with such diffuse and specular properties
		pMaterial*			mat;
		mat = SearchMaterial(object->faces[i]->material->diffuse, color);
		object->faces[i]->material = mat;

//		Output(*traceOutput, spectra);
//		*traceOutput << eol;
	}
	*traceOutput << ret << ret;
}


void
ParserIob::ReadFacesTransparencyChunk(uint32 len, pObject* object)
{
	if (object->vertices->CountItems() == 0)
	{
		SkipBytes(len);
		return ;
	}

	*traceOutput << "ReadFacesTransparencyChunk: " << int32(len) << eol;
	int16			nbFaces;
	ReadInt16(nbFaces);
	Assert("ReadFacesTransparencyChunk", 2 + 3*nbFaces);

	pSpectra			spectra;
	*traceOutput << adv << nbFaces << " faces" << eol;
	*traceOutput << adv;
	for (int32 i = 0; i < nbFaces; i++)
	{
		pColor			color;
		ReadInt8(color.r);
		ReadInt8(color.g);
		ReadInt8(color.b);
		spectra.Set(color);

		pMaterial*			currMat = object->faces[i]->material;
		pMaterial*			mat = SearchMaterial(currMat->diffuse, currMat->specular, color);
		object->faces[i]->material = mat;

//		Output(*traceOutput, spectra);
//		*traceOutput << eol;
	}
	*traceOutput << ret << ret;
}


void
ParserIob::ReadObjectPropertiesChunk(uint32 len)
{
	*traceOutput << "ReadObjectPropertiesChunk: " << int32(len) << eol;
	Assert("ReadObjectPropertiesChunk", 8);
	*traceOutput << adv;

	float			tmp;
	uint8			properties[8];
	ReadBytes((char* )properties, 8);

	tmp = float(properties[0]) / 255.0;
	*traceOutput << "dither " << tmp << eol;
	tmp = float(properties[1]) / 255.0;
	*traceOutput << "hardness " << tmp << eol;
	tmp = float(properties[2]) / 255.0;
	*traceOutput << "roughness " << tmp << eol;
	tmp = float(properties[3]) / 255.0;
	*traceOutput << "shininess " << tmp << eol;
	tmp = float(properties[4]) / 100.0 + 1.0;
	*traceOutput << "refraction index " << tmp << eol;

	// quick draw
	*traceOutput << "quick draw:";
	if (properties[5] == 0)
		*traceOutput << " none";
	else
	{
		if (properties[5] & 0x01)
			*traceOutput << " bbox";
		if (properties[5] & 0x02)
			*traceOutput << " quick edges";
	}
	*traceOutput << eol;
	*traceOutput << "Phong shading " << (properties[6] ? "on":"off") << eol;
	*traceOutput << "Genlock " << (properties[7] ? "on":"off") << eol;
	*traceOutput << ret;
}


void
ParserIob::ReadIntensityChunk(uint32 len, pSpectra& intensity)
{
	*traceOutput << "ReadIntensityChunk: " << int32(len) << eol;
	Assert("ReadIntensityChunk", 12);

	ReadFract(intensity.r);
	ReadFract(intensity.g);
	ReadFract(intensity.b);
	intensity.Clamp(0.0, 1.0);

	*traceOutput << adv;
	Output(*traceOutput, intensity);
	*traceOutput << eol << ret;
}


pMaterial*
ParserIob::SearchMaterial(pSpectra& col)
{
	int32			matIndex = scene->materials.Search(col);
	if (matIndex >= 0)
		return scene->materials[matIndex];

	char				buf[32];
	sprintf(buf, "material %d", scene->materials.CountItems());
	pMaterial*			newMaterial = NewMaterial(buf);
	newMaterial->SetAmbientAndDiffuse(col);
	newMaterial->shininess = 40.0;
	// values expected to be set with the contents of chunks yet to come
	newMaterial->specular.Set(-1.0, -1.0, -1.0);
	newMaterial->opacity = -1.0;
	return newMaterial;
}


pMaterial*
ParserIob::SearchMaterial(pSpectra& col, pSpectra& spec)
{
	for (int32 i = 0; i < scene->materials.CountItems(); i++)
	{
		pMaterial*			currentMaterial = scene->materials[i];
		if (currentMaterial->diffuse == col)
		{
			if (currentMaterial->specular == pSpectra(-1.0, -1.0, -1.0))
			{
				currentMaterial->specular = spec;
				return currentMaterial;
			}
			if (currentMaterial->specular == spec)
				return currentMaterial;
		}
	}

	char				buf[32];
	sprintf(buf, "material %d", scene->materials.CountItems());
	pMaterial*			material = NewMaterial(buf);
	material->SetAmbientAndDiffuse(col);
	material->specular = spec;
	material->shininess = 40.0;
	material->opacity = -1.0;
	return material;
}


pMaterial*
ParserIob::SearchMaterial(pSpectra& col, pSpectra& spec, pSpectra& trans)
{
	float				opacity = 1.0 - trans.GreyLevel();

	for (int32 i = 0; i < scene->materials.CountItems(); i++)
	{
		pMaterial*			currentMaterial = scene->materials[i];
		if ((currentMaterial->diffuse == col)
		&& (currentMaterial->specular == spec))
		{
			if (currentMaterial->opacity == -1.0)
			{
				currentMaterial->opacity = opacity;
				return currentMaterial;
			}
			if (currentMaterial->opacity == opacity)
				return currentMaterial;
		}
	}


	char				buf[32];
	sprintf(buf, "material %d", scene->materials.CountItems());
	pMaterial*			material = NewMaterial(buf);
	material->SetAmbientAndDiffuse(col);
	material->specular = spec;
	material->shininess = 40.0;
	material->opacity = opacity;
	return material;
}


void
ParserIob::CheckMaterials()
{
	for (int32 i = 0; i < scene->objects.CountItems(); i++)
	{
		pObject*		object = scene->objects[i];
		char			buf[32];
		sprintf(buf, "%s material", object->name);
		int32			matIndex = scene->materials.Search(buf);
		if (matIndex < 0)
			matIndex = scene->materials.Search("default material");
		pMaterial*		defMaterial = scene->materials[matIndex];

		for (int32 j = 0; j < object->faces.CountItems(); j++)
		{
			pMaterial*	mat = object->faces[j]->material;
			if ((mat->diffuse == pSpectra(0.0, 0.0, 0.0))
			&& (mat->specular == pSpectra(0.0, 0.0, 0.0))
			&& (mat->opacity == 1.0))
				object->faces[j]->material = defMaterial;
		}
	}
}
