//
//
//	PLY objects parser
//
//										(C) JoOl 1998


#include <ctype.h>
#include "ParserPly.h"
#include "LAPerrors.h"



#if __POWERPC__
#pragma export on
#endif
char			handlerName[]		= "PoLYgonal v1.0 (.ply)";
char			handlerVersion[]	= "1.0.0";
char			handlerAuthor[]		= "Alain GAUTHEROT";
char			handlerEmail[]		= "gauthero@club-internet.fr";
#if __POWERPC__
#pragma export reset
#endif



ParserPly*
instantiate_parser(int32 bs)
{
	return new ParserPly(bs);
}


ParserPly::ParserPly(int32 blockSize)
{
	// input is a big-endian binary file
	SetLexer(new LAPbinLexer(max_c(1024, blockSize), LAP_BIG_ENDIAN));

	// error output, tracer output, display
	SetTo(new LAPtracer(128, LAP_BUFFER_OUTPUT_ON_OVERFLOW, LAP_BUFFER_SCREEN_OUTPUT),
			new LAPtextBufferedOutput(128, LAP_BUFFER_OUTPUT_ON_OVERFLOW, LAP_BUFFER_NO_OUTPUT/*LAP_BUFFER_SCREEN_OUTPUT*/),
			new LAPdisplay(),
			LAP_OWN_ERROR_OUTPUT | LAP_OWN_TRACE_OUTPUT | LAP_OWN_DISPLAY);


	lexemes.Insert(new LAPlexeme("ply", TOKEN_PLY));
	lexemes.Insert(new LAPlexeme("format", TOKEN_FORMAT));
		lexemes.Insert(new LAPlexeme("ascii", TOKEN_ASCII));
		lexemes.Insert(new LAPlexeme("binary_big_endian", TOKEN_BIG_ENDIAN));
		lexemes.Insert(new LAPlexeme("binary_little_endian", TOKEN_LITTLE_ENDIAN));
	lexemes.Insert(new LAPlexeme("comment", TOKEN_COMMENT));
	lexemes.Insert(new LAPlexeme("element", TOKEN_ELEMENT));
	lexemes.Insert(new LAPlexeme("property", TOKEN_PROPERTY));
		lexemes.Insert(new LAPlexeme("list", TOKEN_LIST));
		lexemes.Insert(new LAPlexeme("char", TOKEN_CHAR));
		lexemes.Insert(new LAPlexeme("uchar", TOKEN_UCHAR));
		lexemes.Insert(new LAPlexeme("short", TOKEN_INT16));
		lexemes.Insert(new LAPlexeme("ushort", TOKEN_UINT16));
		lexemes.Insert(new LAPlexeme("int", TOKEN_INT32));
		lexemes.Insert(new LAPlexeme("uint", TOKEN_UINT32));
		lexemes.Insert(new LAPlexeme("float", TOKEN_FLOAT));
		lexemes.Insert(new LAPlexeme("double", TOKEN_DOUBLE));
	lexemes.Insert(new LAPlexeme("end_header", TOKEN_END_HEADER));

	typeString[0] = "char";
	typeString[1] = "uchar";
	typeString[2] = "int16";
	typeString[3] = "uint16";
	typeString[4] = "int32";
	typeString[5] = "uint32";
	typeString[6] = "float";
	typeString[7] = "double";

	elements = NULL;
	currentElement = NULL;
}


ParserPly::~ParserPly()
{
	if (elements)
		delete elements;

	if (data)
		delete (pScene* )data;
}


void
ParserPly::_Terminate()
{
}


status_t
ParserPly::DoAbort()
{
	PierrotParser::Reset();
	return LAP_NO_ERROR;
}


status_t
ParserPly::_Identify()
{
	bool		silent = traceOutput->IsSilent();
	traceOutput->SetSilent(true);
	elements = new PierrotPtrArray<plyElement* >(8,4);

	status_t		st = (ParseHeader() == LAP_NO_ERROR) ? LAP_NO_ERROR : LAP_BAD_INPUT_TYPE;

	traceOutput->SetSilent(silent);
	delete elements;
	elements = NULL;
	return st;
}


status_t
ParserPly::_Parse()
{
	PierrotParser::Reset();

	elements = new PierrotPtrArray<plyElement* >(8,4);
	if (ParseHeader() != LAP_NO_ERROR)
		return LAP_BAD_INPUT_TYPE;

	scene = new pScene((char* )GetLexer()->GetInputName());
	scene->ambient.Set(0.0, 0.0, 0.0);

	// create default material
	defaultMaterial = NewMaterial("default material");
	defaultMaterial->ambient.Set(0.1, 0.1, 0.1);
	defaultMaterial->diffuse.Set(0.4, 0.5, 0.6);
	defaultMaterial->specular.Set(0.4, 0.4, 0.4);
	defaultMaterial->shininess = 40.0;

	ParseObject();

	pSphere				normSphere(pPoint3(0.0, 0.0, 0.0), 1.0);
	scene->Preprocess(&normSphere, appliInterface, 0.0, 1.0);

	scene->ambient.Set(0.5, 0.5, 0.5);
	if (scene->lights.CountItems() == 0)
	{
		pLight*			l1 = NewDirectionalLight("directional light 1", pVector3(-0.5, 0.5, 0.8));
		l1->ambient.Set(0.2, 0.2, 0.2);
		l1->diffuse.Set(1.0, 1.0, 1.0);
		l1->specular.Set(1.0, 1.0, 1.0);
	}

	Output(*traceOutput, scene, PRINT_LIGHT_ALL | PRINT_MATERIAL_ALL |
			PRINT_OBJECT_BOUNDS | PRINT_OBJECT_MATRIX | PRINT_OBJECT_POINTS);

	data = scene;
	scene = NULL;
	return LAP_NO_ERROR;
}


int32
ParserPly::ReadWord()
{
	*buffer = 0;
	char*			out = buffer;
	char			currentCharacter;

	ReadBytes((char* )&currentCharacter, 1);
	while (!Eof() && isspace(currentCharacter))
	{
		ReadBytes((char* )&currentCharacter, 1);
	}

	if (!Eof())
	{
		while (!Eof() && !isspace(currentCharacter))
		{
			*out++ = currentCharacter;
			ReadBytes((char* )&currentCharacter, 1);
		}
	}
	*out++ = 0;

	LAPlexeme*		lexm = lexemes.Search(buffer);
	if (lexm)
		return lexm->GetToken();

	for (int32 i = 0; i < strlen(buffer); i++)
		if (((buffer[i] >= 'a') && (buffer[i] <= 'z'))
		|| ((buffer[i] >= 'A') && (buffer[i] <= 'Z')))
			return TOKEN_IDENTIFIER;
	return TOKEN_NUMBER;
}


void
ParserPly::SkipLine()
{
	char			currentCharacter;

	ReadBytes((char* )&currentCharacter, 1);
	while (!Eof() && currentCharacter != '\n')
		ReadBytes((char* )&currentCharacter, 1);
}


int32
ParserPly::SizeOf(int32 type)
{
	if (type == TOKEN_FLOAT)
		return 4;
	return (1 << ((type-TOKEN_CHAR)/2));
}


int32
ParserPly::SizeOf(plyProperty* prop, int32 listLen)
{
	if (!prop->IsList())
		return -1;
	return listLen * SizeOf(prop->type);
}


void
ParserPly::SkipElement(plyElement* element)
{
//	*traceOutput << "skip element " << element->name << eol;
	for (int32 i = 0; i < element->properties.CountItems(); i++)
		SkipProperty(element->properties[i]);
}


void
ParserPly::SkipProperty(plyProperty* prop)
{
	if (format == TOKEN_ASCII)
	{
		SkipLine();
//		*traceOutput << "  skip property " << prop->name << eol;
	}
	else
	{
		int32			siz = 0;
		if (prop->IsList())
		{
			int32			listLength;
			ReadProperty(prop->countType, listLength);
			siz = SizeOf(prop, listLength);
		}
		else
			siz = SizeOf(prop->type);
		SkipBytes(siz);
//		*traceOutput << "  skip property " << prop->name << ' ' << siz << " bytes" << eol;
	}
}


void
ParserPly::ReadProperty(plyProperty* prop, int32& out)
{
	if (prop->IsList())
	{
		printf("*** error ParserPly::ReadProperty int32\n");
		return ;
	}

	if (format == TOKEN_ASCII)
	{
		ReadWord();
		if ((prop->type == TOKEN_FLOAT) || (prop->type == TOKEN_DOUBLE))
		{
			double			v;
			sscanf(buffer, "%g", &v);
			out = int32(v);
		}
		else
		{
			sscanf(buffer, "%d", &out);
		}
	}
	else
	{
		int32			size = 1 << ((prop->type-TOKEN_CHAR)/2);
		if (size == 1)
		{
			int8		v;
			ReadInt8(v);
			out = int32(v);
		}
		else if (size == 2)
		{
			int16		v;
			ReadInt16(v);
			out = int32(v);
		}
		else if (size == 4)
		{
			ReadInt32(out);
		}
		else if (size == 8)
		{
			if (prop->type == TOKEN_FLOAT)
			{
				float		v;
				ReadFloat(v);
				out = int32(v);
			}
			else
			{
				double		v;
				ReadDouble(v);
				out = int32(v);
			}
		}
	}
}


void
ParserPly::ReadProperty(plyProperty* prop, float& out)
{
	if (prop->IsList())
	{
		printf("*** error ParserPly::ReadProperty float\n");
		return ;
	}

	if (format == TOKEN_ASCII)
	{
		ReadWord();
		if ((prop->type == TOKEN_FLOAT) || (prop->type == TOKEN_DOUBLE))
		{
			double			v;
			sscanf(buffer, "%g", &v);
			out = float(v);
		}
		else
		{
			int32			v;
			sscanf(buffer, "%d", &v);
			out = float(v);
		}
	}
	else
	{
		int32			size = 1 << ((prop->type-TOKEN_CHAR)/2);
		if (size == 1)
		{
			int8		v;
			ReadInt8(v);
			out = float(v);
		}
		else if (size == 2)
		{
			int16		v;
			ReadInt16(v);
			out = float(v);
		}
		else if (size == 4)
		{
			int32		v;
			ReadInt32(v);
			out = float(v);
		}
		else if (size == 8)
		{
			if (prop->type == TOKEN_FLOAT)
			{
				float		v;
				ReadFloat(v);
				out = float(v);
			}
			else
			{
				double		v;
				ReadDouble(v);
				out = float(v);
			}
		}
	}
}


void
ParserPly::ReadProperty(int32 type, int32& out)
{
	if (format == TOKEN_ASCII)
	{
		ReadWord();
		if ((type == TOKEN_FLOAT) || (type == TOKEN_DOUBLE))
		{
			double			v;
			sscanf(buffer, "%g", &v);
			out = int32(v);
		}
		else
		{
			sscanf(buffer, "%d", &out);
		}
	}
	else
	{
		int32			size = 1 << ((type-TOKEN_CHAR)/2);
		if (size == 1)
		{
			int8		v;
			ReadInt8(v);
			out = int32(v);
		}
		else if (size == 2)
		{
			int16		v;
			ReadInt16(v);
			out = int32(v);
		}
		else if (size == 4)
		{
			int32		v;
			ReadInt32(v);
			out = int32(v);
		}
		else if (size == 8)
		{
			if (type == TOKEN_FLOAT)
			{
				float		v;
				ReadFloat(v);
				out = int32(v);
			}
			else
			{
				double		v;
				ReadDouble(v);
				out = int32(v);
			}
		}
	}
}


status_t
ParserPly::ParseHeader()
{
	if (ReadWord() != TOKEN_PLY)
		return LAP_BAD_INPUT_TYPE;

	int32			currentToken = ReadWord();
	while(currentToken != TOKEN_END_HEADER)
	{
		if (currentToken == TOKEN_FORMAT)
			ParseFormat();
		else if (currentToken == TOKEN_COMMENT)
			SkipLine();
		else if (currentToken == TOKEN_ELEMENT)
			ParseElement();
		else if (currentToken == TOKEN_PROPERTY)
			ParseProperty();

		currentToken = ReadWord();
	}
	return LAP_NO_ERROR;
}


void
ParserPly::ParseFormat()
{
	format = ReadWord();
	ReadWord();
	sscanf(buffer, "%f", &version);

	if (format == TOKEN_BIG_ENDIAN)
		lexer->GetInput()->SetInputEndianness(LAP_BIG_ENDIAN);
	else if (format == TOKEN_LITTLE_ENDIAN)
		lexer->GetInput()->SetInputEndianness(LAP_LITTLE_ENDIAN);

	*traceOutput << "format: ";
	if (format == TOKEN_ASCII)
		*traceOutput << "ascii";
	else if (format == TOKEN_BIG_ENDIAN)
		*traceOutput << "bin big endian";
	else if (format == TOKEN_LITTLE_ENDIAN)
		*traceOutput << "bin little endian";
	*traceOutput << ", version " << version << eol;
}


void
ParserPly::ParseElement()
{
	ReadWord();
	currentElement = new plyElement(buffer);
	elements->AddItem(currentElement);
	ReadWord();
	sscanf(buffer, "%d", &currentElement->nb);

	*traceOutput << currentElement->nb << ' ' << currentElement->name << eol;
}


void
ParserPly::ParseProperty()
{
	int32			countType = -1;
	int32			type = ReadWord();
	if (type == TOKEN_LIST)
	{
		countType = ReadWord();
		type = ReadWord();
	}
	ReadWord();
	plyProperty*	prop = new plyProperty(countType, buffer, type);
	currentElement->properties.AddItem(prop);

	*traceOutput << "  ";
	if (prop->IsList())
		*traceOutput << "list, count:" << typeString[prop->countType-TOKEN_CHAR] << ", ";
	*traceOutput << prop->name << ':' << typeString[prop->type-TOKEN_CHAR];
	*traceOutput << eol;
}


void
ParserPly::ParseObject()
{
	pObject*			object = NewObject("main");

	for (int32 i = 0; i < elements->CountItems(); i++)
	{
		plyElement*		elem = (*elements)[i];
		*traceOutput << "parse " << elem->nb << ' ' << elem->name << eol;

		if (!strcmp(elem->name, "vertex"))
			ParseVertices(object, elem);
		else if (!strcmp(elem->name, "face"))
			ParseFaces(object, elem);
		else
			SkipElement(elem);
	}
}


void
ParserPly::ParseVertices(pObject* object, plyElement* element)
{
	object->vertices->SetAlloc(element->nb, 0);
	for (int32 i = 0; i < element->nb; i++)
	{
		pPoint3			pt;
		ParseVertex(element, pt);
		object->vertices->AddItem(pt);
/*
		Output(*traceOutput, pt);
		*traceOutput << eol;
*/	}
}


void
ParserPly::ParseVertex(plyElement* element, pPoint3& vrt)
{
	for (int32 i = 0; i < element->properties.CountItems(); i++)
	{
		plyProperty*		property = element->properties[i];
		if (!strcmp(property->name, "x"))
			ReadProperty(property, vrt.x);
		else if (!strcmp(property->name, "y"))
			ReadProperty(property, vrt.y);
		else if (!strcmp(property->name, "z"))
			ReadProperty(property, vrt.z);
//		else if (!strcmp(property->name, "u"))
//			ReadProperty();
//		else if (!strcmp(property->name, "v"))
//			ReadProperty();
		else
			SkipProperty(property);
	}
}


void
ParserPly::ParseFaces(pObject* object, plyElement* element)
{
	object->faces.SetAlloc(element->nb, 0);

	for (int32 i = 0; i < element->nb; i++)
	{
		pFace*				face = NULL;
		ParseFace(element, face);
		if (face)
			object->faces.AddItem(face);
	}
}


void
ParserPly::ParseFace(plyElement* element, pFace*& face)
{
	for (int32 i = 0; i < element->properties.CountItems(); i++)
	{
		plyProperty*		property = element->properties[i];

		if (!strcmp(property->name, "vertex_indices"))
		{
			int32			vertCount;
			ReadProperty(property->countType, vertCount);
//*traceOutput << vertCount << ": ";
			PierrotArray<int32 >	indexes(vertCount, 0);
			for (int32 i = 0; i < vertCount; i++)
			{
				int32			currentIndex;
				ReadProperty(property->type, currentIndex);
				indexes.AddItem(currentIndex);
//*traceOutput << ' ' << currentIndex;
			}
//*traceOutput << eol;
			face = new pFace(defaultMaterial, vertCount, indexes.Items());
		}
		else
			SkipProperty(property);
	}
}
