//
//
//	LAP Parser class
//
//										(C) JoOl 1998


#include "LAPimpexp.h"

#include "LAPparser.h"
#include "LAPerrors.h"
#include "LAPexception.h"



LAPparser::LAPparser()
{
	status = LAP_NO_ERROR;
	_lexer = NULL;
	display = NULL;
	errorOutput = NULL;
	traceOutput = NULL;
	appliInterface = NULL;
	owned = 0;
	data = NULL;
}


LAPparser::~LAPparser()
{
	Terminate();
	delete _lexer;
	if (owned & LAP_OWN_ERROR_OUTPUT)
		delete errorOutput;
	if (owned & LAP_OWN_TRACE_OUTPUT)
		delete traceOutput;
	if (owned & LAP_OWN_DISPLAY)
		delete display;
}


void
LAPparser::SetAppliInterface(LAPappliInterface* ai)
{
	appliInterface = ai;
	_lexer->SetAppliInterface(appliInterface);
}


LAPappliInterface*
LAPparser::GetAppliInterface() const
{
	return appliInterface;
}


LAPdisplay*
LAPparser::GetDisplay() const
{
	return display;
}


void
LAPparser::SetLexer(LAPlexer* l)
{
	if (!l)
		throw LAPbadInit("LAPparser::SetLexer", "_lexer");

	_lexer = l;
	if (errorOutput)
		_lexer->SetErrorOutput(errorOutput);
	if (traceOutput)
		_lexer->SetTraceOutput(traceOutput);
	if (display)
		_lexer->SetDisplay(display);
}


LAPlexer*
LAPparser::GetLexer() const
{
	return _lexer;
}


void
LAPparser::SetTo(	LAPtracer* eo,
					LAPtextBufferedOutput* to,
					LAPdisplay* d,
					uint8 o)
{
	if (!eo)
		throw LAPbadInit("LAPparser::SetTo", "errorOutput");
	if (!d)
		throw LAPbadInit("LAPparser::SetTo", "display");
	errorOutput = eo;
	traceOutput = to;
	display = d;
	owned = o;

	if (display->IsEmpty())
		display->EnterScope();

	if (_lexer)
	{
		_lexer->SetErrorOutput(errorOutput);
		_lexer->SetTraceOutput(traceOutput);
		_lexer->SetDisplay(display);
	}
}


bool
LAPparser::LockNoThrow()
{
	return Locker.Lock();
}


void
LAPparser::UnlockNoThrow()
{
	Locker.Unlock();
}


bool
LAPparser::Lock()
{
	if ((Locker.CountLocks() == 0) && abortRequested)
		throw LAPloadAborted("LAPparser::Lock");
	return Locker.Lock();
}


void
LAPparser::Unlock()
{
	Locker.Unlock();
	if ((Locker.CountLocks() == 0) && abortRequested)
		throw LAPloadAborted("LAPparser::Unlock");
}


bool
LAPparser::IsLocked()
{
	return Locker.IsLocked();
}


bool
LAPparser::CheckStatus() const
{
	return status == LAP_NO_ERROR;
}


status_t
LAPparser::GetStatus() const
{
	return status;
}


bool
LAPparser::Eof() const
{
	return _lexer->Eof();
}


LAPtracer*
LAPparser::GetErrorOutput() const
{
	return errorOutput;
}


void
LAPparser::SetVerboseError(bool b)
{
	if (errorOutput)
		errorOutput->SetSilent(~b);
}



LAPtextBufferedOutput*
LAPparser::GetTraceOutput() const
{
	return traceOutput;
}


void
LAPparser::SetVerboseTrace(bool b)
{
	if (traceOutput)
		traceOutput->SetSilent(~b);
}


status_t
LAPparser::SetTraceFile(entry_ref ref, uint32 flags)
{
	if (!traceOutput)
		return LAP_BAD_INIT;

	BEntry			e(&ref, false);
	BPath			p;
	e.GetPath(&p);
	p.GetParent(&p);
	char*			str = new char[strlen(ref.name)+6+1];
	strcpy(str, ref.name);
	strcat(str, ".trace");
	p.Append(str);
	delete[] str;
	return traceOutput->SetFileOutput((char* )p.Path(), flags);
}



status_t
LAPparser::SetMessagesTraceFile(entry_ref ref, uint32 flags)
{
	if (!errorOutput)
		return LAP_BAD_INIT;

	BEntry			e(&ref, false);
	BPath			p;
	e.GetPath(&p);
	p.GetParent(&p);
	char*			str = new char[strlen(ref.name)+7+1];
	strcpy(str, ref.name);
	strcat(str, ".errors");
	p.Append(str);
	delete[] str;
	return errorOutput->SetFileOutput((char* )p.Path(), flags);
}


bool
LAPparser::CheckTraceFile() const
{
	if (!traceOutput)
		return false;
	return traceOutput->GetOutputs() & LAP_BUFFER_FILE_OUTPUT;
}


bool
LAPparser::CheckErrorsTraceFile() const
{
	return errorOutput->GetOutputs() & LAP_BUFFER_FILE_OUTPUT;
}


void*
LAPparser::DetachData()
{
	void*			out = data;
	data = NULL;
	return out;
}


LAPsymbol*
LAPparser::Search(char* n) const
{
	return display->Search(n);
}


LAPsymbol*
LAPparser::SearchGlobal(char* n) const
{
	LAPsymbolTable*		st = display->GetGlobalScope();
	if (!st)
		return NULL;
	return st->Search(n);
}


LAPsymbol*
LAPparser::SearchLocal(char* n) const
{
	return display->SearchLocal(n);
}


status_t
LAPparser::Insert(LAPsymbol* s)
{
	return display->Insert(s);
}


status_t
LAPparser::Identify()
{
	try
	{
		return _Identify();
	}
	catch (LAPexception )
	{
		return LAP_BAD_INPUT_TYPE;
	}
}


status_t
LAPparser::Parse()
{
	try
	{
		status_t		out = _Parse();
		Terminate();

		if (abortRequested)
			throw LAPloadAborted("LAPparser::Parse");
		return out;
	}

	catch (LAPexception& e)
	{
		e.Output(*errorOutput);
		DoAbort();
		return LAP_LOAD_ABORTED;
	}
	catch (...)
	{
		DoAbort();
		return LAP_SYSTEM_EXCEPTION;
	}
}


void
LAPparser::Terminate()
{
	_lexer->Terminate();
	if (errorOutput)
		*errorOutput << flush;
	if (traceOutput)
		*traceOutput << flush;
	_Terminate();
}


void
LAPparser::_Terminate()
{
}


void
LAPparser::Abort()
{
	abortRequested = true;
}
