//
//
//	Pierrot 3d geometry
//
//										(C) JoOl 1998


#include "PierrotImpExp.h"

#include <math.h>
#include "PierrotGeometry.h"
#include "PierrotMatrix.h"



//--------------------------------------------------------
// 3d position
//--------------------------------------------------------

pPoint3::pPoint3()
{}


pPoint3::pPoint3(float _x, float _y, float _z)
{ Set(_x, _y, _z); }



pPoint3::pPoint3(pPoint3& p)
{ Set(p.x, p.y, p.z); }


pPoint3::~pPoint3()
{}


void
pPoint3::Set(float _x, float _y, float _z)
{
	x = _x;
	y = _y;
	z = _z;
}


void
pPoint3::Clear()
{
	x = y = z = 0.0;
}


bool
pPoint3::Eq(pPoint3& p, float e)
{
	return (fabs(p.x - x) < e)
		&& (fabs(p.y - y) < e)
		&& (fabs(p.z - z) < e);
}


pPoint3
pPoint3::Project(pLine& l)
{
	pVector3			dir = l.dir;
	dir.Norm();
	float				k = pVector3(l.pt, *this) * dir;
	return l.pt + dir*k;
}


pPoint3
pPoint3::Project(pPlane& p)
{
	pLine			line(*this, p.normal);
	pPoint3			i;
	line.Intersects(p, i);
	return i;
}


pPoint3
pPoint3::ProjectXoY()
{
	return pPoint3(x, y, 0.0);
}


pPoint3
pPoint3::ProjectYoZ()
{
	return pPoint3(0.0, y, z);
}


pPoint3
pPoint3::ProjectXoZ()
{
	return pPoint3(x, 0.0, z);
}


pPoint3
pPoint3::Mirror(pLine& l)
{
	pPoint3			i = this->Project(l);
	return i + pVector3(*this, i);
}


pPoint3
pPoint3::Mirror(pPlane& p)
{
	pPoint3			i = this->Project(p);
	return i + pVector3(*this, i);
}


pPoint3
pPoint3::MirrorXoY()
{
	return pPoint3(x, y, -z);
}


pPoint3
pPoint3::MirrorYoZ()
{
	return pPoint3(-x, y, z);
}


pPoint3
pPoint3::MirrorXoZ()
{
	return pPoint3(x, -y, z);
}


pPoint3
pPoint3::operator*(float s)
{
	return pPoint3(x * s, y * s, z * s);
}


pPoint3
pPoint3::operator/(float s)
{
	return pPoint3(x / s, y / s, z / s);
}


float
pPoint3::operator*(pPoint3& p)
{ return x * p.x + y * p.y + z * p.z; }


float
pPoint3::operator*(pVector3& v)
{ return x * v.x + y * v.y + z * v.z; }


pPoint3
pPoint3::operator*(pMatrix3& m)
{
	return pPoint3(	*this * m.GetX(),
					*this * m.GetY(),
					*this * m.GetZ());
}


pPoint3&
pPoint3::operator*=(float s)
{
	x *= s;
	y *= s;
	z *= s;
	return *this;
}


pPoint3&
pPoint3::operator*=(pMatrix3& m)
{
	pPoint3		res(*this * m.GetX(),
					*this * m.GetY(),
					*this * m.GetZ());
	*this = res;
	return *this;
}


pPoint3&
pPoint3::operator/=(float s)
{
	x /= s;
	y /= s;
	z /= s;
	return *this;
}


pPoint3
pPoint3::operator+(pPoint3& p)
{
	return pPoint3(x + p.x, y + p.y, z + p.z);
}


pPoint3
pPoint3::operator+(pVector3& v)
{
	return pPoint3(x + v.x, y + v.y, z + v.z);
}


pPoint3&
pPoint3::operator+=(pPoint3& p)
{
	x += p.x;
	y += p.y;
	z += p.z;
	return *this;
}


pPoint3&
pPoint3::operator+=(pVector3& v)
{
	x += v.x;
	y += v.y;
	z += v.z;
	return *this;
}


pPoint3
pPoint3::operator-(pPoint3& p)
{
	return pPoint3(x - p.x, y - p.y, z - p.z);
}


pPoint3
pPoint3::operator-(pVector3& v)
{
	return pPoint3(x - v.x, y - v.y, z - v.z);
}


pPoint3&
pPoint3::operator-=(pPoint3& p)
{
	x -= p.x;
	y -= p.y;
	z -= p.z;
	return *this;
}


pPoint3&
pPoint3::operator-=(pVector3& v)
{
	x -= v.x;
	y -= v.y;
	z -= v.z;
	return *this;
}


pPoint3&
pPoint3::operator=(pPoint3& p)
{
	x = p.x;
	y = p.y;
	z = p.z;
	return *this;
}


pPoint3&
pPoint3::operator=(pVector3& v)
{
	x = v.x;
	y = v.y;
	z = v.z;
	return *this;
}


int
pPoint3::operator==(pPoint3& v)
{
	return (x == v.x) && (y == v.y) && (z == v.z);
}


int
pPoint3::operator!=(pPoint3& v)
{
	return (x != v.x) || (y != v.y) || (z != v.z);
}



//--------------------------------------------------------
// 3d vector
//--------------------------------------------------------

pVector3::pVector3()
{}


pVector3::pVector3(float _x, float _y, float _z)
{ Set(_x, _y, _z); }


pVector3::pVector3(pPoint3& a, pPoint3& b)
{ Set(a, b); }


pVector3::pVector3(pVector3& v)
{ Set(v.x, v.y, v.z); }


pVector3::~pVector3()
{}


void
pVector3::Set(float _x, float _y, float _z)
{
	x = _x;
	y = _y;
	z = _z;
}


void
pVector3::Set(pPoint3& a, pPoint3& b)
{
	x = b.x - a.x;
	y = b.y - a.y;
	z = b.z - a.z;
}


void
pVector3::Clear()
{
	x = y = z = 0.0;
}


bool
pVector3::Eq(pVector3& v, float e)
{
	return (fabs(v.x - x) < e)
		&& (fabs(v.y - y) < e)
		&& (fabs(v.z - z) < e);
}


pVector3
pVector3::ProjectXoY()
{
	return pVector3(x, y, 0.0);
}


pVector3
pVector3::ProjectYoZ()
{
	return pVector3(0.0, y, z);
}


pVector3
pVector3::ProjectXoZ()
{
	return pVector3(x, 0.0, z);
}


pVector3
pVector3::MirrorXoY()
{
	return pVector3(x, y, -z);
}


pVector3
pVector3::MirrorYoZ()
{
	return pVector3(-x, y, z);
}


pVector3
pVector3::MirrorXoZ()
{
	return pVector3(x, -y, z);
}


bool
pVector3::Colinear(pVector3& v, float e)
{
	float			k = v.x / x;
	return (fabs(y*k - v.y) < e)
		&& (fabs(z*k - v.z) < e);
}


bool
pVector3::Orthogonal(pVector3& v, float e)
{
	return fabs(*this * v) < e;
}


float
pVector3::Length2() const
{
	return x * x + y * y + z * z;
}


float
pVector3::Length() const
{
	return sqrt(x * x + y * y + z * z);
}


void
pVector3::Norm()
{
	float			n = Length();
	x /= n;
	y /= n;
	z /= n;
}


pVector3
pVector3::operator*(float s)
{
	return pVector3(x * s, y * s, z * s);
}


pVector3
pVector3::operator/(float s)
{
	return pVector3(x / s, y / s, z / s);
}


float
pVector3::operator*(pPoint3& p)
{ return x * p.x + y * p.y + z * p.z; }


float
pVector3::operator*(pVector3& v)
{ return x * v.x + y * v.y + z * v.z; }


pVector3
pVector3::operator*(pMatrix3& m)
{
	return pVector3(*this * m.GetX(),
					*this * m.GetY(),
					*this * m.GetZ());
}


pVector3
pVector3::operator^(pVector3& v)
{
	return pVector3(y*v.z - z*v.y,
					z*v.x - x*v.z,
					x*v.y - y*v.x);
}


pVector3&
pVector3::operator*=(float s)
{
	x *= s;
	y *= s;
	z *= s;
	return *this;
}


pVector3&
pVector3::operator*=(pMatrix3& m)
{
	pPoint3		res(*this * m.GetX(),
					*this * m.GetY(),
					*this * m.GetZ());
	*this = res;
	return *this;
}


pVector3&
pVector3::operator/=(float s)
{
	x /= s;
	y /= s;
	z /= s;
	return *this;
}


pVector3
pVector3::operator+(pPoint3& p)
{
	return pVector3(x + p.x, y + p.y, z + p.z);
}


pVector3
pVector3::operator+(pVector3& v)
{
	return pVector3(x + v.x, y + v.y, z + v.z);
}


pVector3&
pVector3::operator+=(pPoint3& p)
{
	x += p.x;
	y += p.y;
	z += p.z;
	return *this;
}


pVector3&
pVector3::operator+=(pVector3& v)
{
	x += v.x;
	y += v.y;
	z += v.z;
	return *this;
}


pVector3
pVector3::operator-(pPoint3& p)
{
	return pVector3(x - p.x, y - p.y, z - p.z);
}


pVector3
pVector3::operator-(pVector3& v)
{
	return pVector3(x - v.x, y - v.y, z - v.z);
}


pVector3&
pVector3::operator-=(pPoint3& p)
{
	x -= p.x;
	y -= p.y;
	z -= p.z;
	return *this;
}


pVector3&
pVector3::operator-=(pVector3& v)
{
	x -= v.x;
	y -= v.y;
	z -= v.z;
	return *this;
}


pVector3&
pVector3::operator=(pPoint3& p)
{
	x = p.x;
	y = p.y;
	z = p.z;
	return *this;
}


pVector3&
pVector3::operator=(pVector3& v)
{
	x = v.x;
	y = v.y;
	z = v.z;
	return *this;
}


int
pVector3::operator==(pVector3& v)
{
	return (x == v.x) && (y == v.y) && (z == v.z);
}


int
pVector3::operator!=(pVector3& v)
{
	return (x != v.x) || (y != v.y) || (z != v.z);
}



//--------------------------------------------------------
// 3d line
//--------------------------------------------------------

pLine::pLine()
{
}


pLine::pLine(pPoint3& p1, pPoint3& p2)
: pt(p1), dir(p1, p2)
{
}


pLine::pLine(pPoint3& p, pVector3& d)
: pt(p), dir(d)
{
}


pLine::pLine(pLine& l)
: pt(l.pt), dir(l.dir)
{
}


pLine::~pLine()
{
}


void
pLine::Set(pPoint3& p1, pPoint3& p2)
{
	pt = p1;
	dir.Set(p1, p2);
}


void
pLine::Set(pPoint3& p, pVector3& d)
{
	pt = p;
	dir = d;
}


pPoint3
pLine::Pt2()
{
	return pt + dir;
}


bool
pLine::Eq(pLine& l, float e)
{
	return dir.Colinear(l.dir, e) && this->Contains(l.pt, e);
}


bool
pLine::Parallel(pLine& l, float e)
{
	return dir.Colinear(l.dir, e);
}


bool
pLine::Parallel(pPlane& p, float e)
{
	return dir.Orthogonal(p.normal, e);
}


bool
pLine::Orthogonal(pLine& l, float e)
{
	return dir.Orthogonal(l.dir, e);
}


bool
pLine::Orthogonal(pPlane& p, float e)
{
	return dir.Colinear(p.normal, e);
}


bool
pLine::Contains(pPoint3& p, float e)
{
	pVector3		v(pt, p);
	return dir.Colinear(v, e);
}


bool
pLine::Intersects(pPlane& p, pPoint3& i, float e)
{
	float			deno = p.normal * dir;
	if (fabs(deno) < e)
		return false;

	float			t = -(p.normal * (p.pt + pt)) / deno;
	i = pt + dir * t;
	return true;
}


pLine
pLine::Project(pPlane& p)
{
	return pLine(pt.Project(p), Pt2().Project(p));
}


pLine
pLine::ProjectXoY()
{
	return pLine(pt.ProjectXoY(), dir.ProjectXoY());
}


pLine
pLine::ProjectYoZ()
{
	return pLine(pt.ProjectYoZ(), dir.ProjectYoZ());
}


pLine
pLine::ProjectXoZ()
{
	return pLine(pt.ProjectXoZ(), dir.ProjectXoZ());
}


pLine
pLine::Mirror(pPlane& p)
{
	return pLine(pt.Mirror(p), Pt2().Mirror(p));
}


pLine
pLine::MirrorXoY()
{
	return pLine(pt.MirrorXoY(), dir.MirrorXoY());
}


pLine
pLine::MirrorYoZ()
{
	return pLine(pt.MirrorYoZ(), dir.MirrorYoZ());
}


pLine
pLine::MirrorXoZ()
{
	return pLine(pt.MirrorXoZ(), dir.MirrorXoZ());
}


pLine&
pLine::operator=(pLine& l)
{
	pt = l.pt;
	dir = l.dir;
	return *this;
}


int
pLine::operator==(pLine& l)
{
	return (pt == l.pt) && (dir == l.dir);
}


int
pLine::operator!=(pLine& l)
{
	return (pt != l.pt) || (dir == l.dir);
}


//--------------------------------------------------------
// 3d plane
//--------------------------------------------------------

pPlane::pPlane()
{
}


pPlane::pPlane(pPoint3& p, pVector3& n)
: pt(p), normal(n)
{
	normal.Norm();
}


pPlane::pPlane(pPoint3& p1, pPoint3& p2, pPoint3& p3)
: pt(p1), normal(pVector3(p1, p2) ^ pVector3(p1, p3))
{
	normal.Norm();
}


pPlane::pPlane(pPoint3& p, pVector3& v1, pVector3& v2)
: pt(p), normal(v1 ^ v2)
{
	normal.Norm();
}


pPlane::pPlane(pPlane& p)
: pt(p.pt), normal(p.normal)
{
	normal.Norm();
}


pPlane::~pPlane()
{
}


void
pPlane::Set(pPoint3& p, pVector3& n)
{
	pt = p;
	normal = n;
	normal.Norm();
}


void
pPlane::Set(pPoint3& p1, pPoint3& p2, pPoint3& p3)
{
	pt = p1;
	normal = pVector3(p1, p2) ^ pVector3(p1, p3);
	normal.Norm();
}


void
pPlane::Set(pPoint3& p, pVector3& v1, pVector3& v2)
{
	pt = p;
	normal = v1 ^ v2;
	normal.Norm();
}


bool
pPlane::Eq(pPlane& p, float e)
{
	return normal.Colinear(p.normal, e) && this->Contains(p.pt);
}


bool
pPlane::Parallel(pLine& l, float e)
{
	return normal.Orthogonal(l.dir, e);
}


bool
pPlane::Parallel(pPlane& p, float e)
{
	return normal.Colinear(p.normal, e);
}


bool
pPlane::Orthogonal(pLine& l, float e)
{
	return normal.Colinear(l.dir, e);
}


bool
pPlane::Orthogonal(pPlane& p, float e)
{
	return normal.Orthogonal(p.normal, e);
}


bool
pPlane::Contains(pPoint3& p, float e)
{
	return fabs(normal * pVector3(p, pt)) < e;
}


bool
pPlane::Contains(pLine& l, float e)
{
	return this->Contains(l.pt) && normal.Orthogonal(l.dir);
}


pPlane
pPlane::ProjectXoY()
{
	return pPlane(pt.ProjectXoY(), normal.ProjectXoY());
}


pPlane
pPlane::ProjectYoZ()
{
	return pPlane(pt.ProjectYoZ(), normal.ProjectYoZ());
}


pPlane
pPlane::ProjectXoZ()
{
	return pPlane(pt.ProjectXoZ(), normal.ProjectXoZ());
}


pPlane
pPlane::MirrorXoY()
{
	return pPlane(pt.MirrorXoY(), normal.MirrorXoY());
}


pPlane
pPlane::MirrorYoZ()
{
	return pPlane(pt.MirrorYoZ(), normal.MirrorYoZ());
}


pPlane
pPlane::MirrorXoZ()
{
	return pPlane(pt.MirrorXoZ(), normal.MirrorXoZ());
}


pPlane&
pPlane::operator=(pPlane& p)
{
	pt = p.pt;
	normal = p.normal;
	return *this;
}


int
pPlane::operator==(pPlane& p)
{
	return (pt == p.pt) && (normal == p.normal);
}


int
pPlane::operator!=(pPlane& p)
{
	return (pt != p.pt) || (normal != p.normal);
}




//--------------------------------------------------------
// texture point
//--------------------------------------------------------


pTexPoint::pTexPoint()
{}


pTexPoint::pTexPoint(float _s, float _t, float _r, float _q)
{
	s = _s;
	t = _t;
	r = _r;
	q = _q;
}


pTexPoint::pTexPoint(pTexPoint& p)
{
	s = p.s;
	t = p.t;
	r = p.r;
	q = p.q;
}


pTexPoint::~pTexPoint()
{}


void
pTexPoint::Set(float _s, float _t, float _r, float _q)
{
	s = _s;
	t = _t;
	r = _r;
	q = _q;
}


void
pTexPoint::Set(pTexPoint& p)
{
	s = p.s;
	t = p.t;
	r = p.r;
	q = p.q;
}


void
pTexPoint::Clear()
{
	r = s = t = 0.0;
	q = 1.0;
}


pTexPoint&
pTexPoint::operator=(pTexPoint& p)
{
	this->Set(p);
	return *this;
}


int
pTexPoint::operator==(pTexPoint& p)
{
	return (r == p.r) && (s == p.s) && (t == p.t) && (q == p.q);
}


int
pTexPoint::operator!=(pTexPoint& p)
{
	return (r != p.r) || (s != p.s) || (t != p.t) || (q != p.q);
}

