//
//
//	Pierrot parser
//
//										(C) JoOl 1998

#include "PierrotImpExp.h"

#include "PierrotParser.h"
#include "PierrotColor4.h"



PierrotParser::PierrotParser()
{
	scene = NULL;
}


PierrotParser::~PierrotParser()
{
	Reset();
}


void
PierrotParser::Reset()
{
	if (scene)
	{
		delete scene;
		scene = NULL;
	}
}


void
PierrotParser::Output(LAPtextBufferedOutput& out, pMatrix4& m)
{
	out << '[' << m.GetX().x << ' ' << m.GetY().x << ' ' << m.GetZ().x << ' ' << m.GetW().x << ']' << eol;
	out << '[' << m.GetX().y << ' ' << m.GetY().y << ' ' << m.GetZ().y << ' ' << m.GetW().y << ']' << eol;
	out << '[' << m.GetX().z << ' ' << m.GetY().z << ' ' << m.GetZ().z << ' ' << m.GetW().z << ']' << eol;
	out << '[' << m.GetX().w << ' ' << m.GetY().w << ' ' << m.GetZ().w << ' ' << m.GetW().w << ']' << eol;
}


void
PierrotParser::Output(LAPtextBufferedOutput& out, pBox& b)
{
	out << '[';
	PierrotParser::Output(out, b.minPt);
	out << " -> ";
	PierrotParser::Output(out, b.maxPt);
	out << ']';
}


void
PierrotParser::Output(LAPtextBufferedOutput& out, pSphere& s)
{
	out << '[';
	PierrotParser::Output(out, s.center);
	out << ", " << s.radius << ']';
}


void
PierrotParser::Output(LAPtextBufferedOutput& out, pSpectra& s)
{
	out << '[' << s.r << ' ' << s.g << ' ' << s.b << ']';
}


void
PierrotParser::Output(LAPtextBufferedOutput& out, pSpectra4& s)
{
	out << '[' << s.r << ' ' << s.g << ' ' << s.b << ' ' << s.a << ']';
}


void
PierrotParser::Output(LAPtextBufferedOutput& out, pColor& c)
{
	out << '[' << int32(c.r) << ' ' << int32(c.g) << ' ' << int32(c.b) << ']';
}


void
PierrotParser::Output(LAPtextBufferedOutput& out, pColor4& c)
{
	out << '[' << int32(c.r) << ' ' << int32(c.g) << ' ' << int32(c.b) << ' ' << int32(c.a) << ']';
}


void
PierrotParser::Output(LAPtextBufferedOutput& out, pPoint3& p)
{
	out << '(' << p.x << ' ' << p.y << ' ' << p.z << ')';
}


void
PierrotParser::Output(LAPtextBufferedOutput& out, pVector3& v)
{
	out << '{' << v.x << ' ' << v.y << ' ' << v.z << '}';
}


void
PierrotParser::Output(LAPtextBufferedOutput& out, pTexPoint& p)
{
	out << '{' << p.s << ' ' << p.t << ' ' << p.r << ' ' << p.q << '}';
}


void
PierrotParser::Output(LAPtextBufferedOutput& out, pFace* f)
{
	out << '\'' << f->material->name << "': ";
	for (int32 i = 0; i < f->vcount; i++)
	{
		out << f->index[i];
		if (i < f->vcount-1)
			out << ' ';
	}
}


void
PierrotParser::Output(LAPtextBufferedOutput& out, pLight* l)
{
	if ((out.GetOutputs() == LAP_BUFFER_NO_OUTPUT) || out.IsSilent())
		return ;

	if (l->IsSpotlight())
		out << "spotlight: ";
	else
		out << "light: ";
	out << l->name << eol;
	out << adv;
		if (l->IsPositional())
		{
			out << "position: ";
				PierrotParser::Output(out, ((pPositionalLight* )l)->position);
		}
		else
		{
			out << "direction: ";
				PierrotParser::Output(out, ((pDirectionalLight* )l)->direction);
		}
		if (l->IsSpotlight())
		{
			out << "exponent " << ((pSpotlight* )l)->spotExponent << ", ";
			out << "cutoff " << ((pSpotlight* )l)->spotCutOff;
		}
		out << eol;
		out << "ambient = ";	PierrotParser::Output(out, l->ambient);		out << eol;
		out << "diffuse = ";	PierrotParser::Output(out, l->diffuse);		out << eol;
		out << "specular = ";	PierrotParser::Output(out, l->specular);	out << eol;
		out << "attenuation = " << l->constantAttenuation << ", "
								<< l->linearAttenuation << ", "
								<< l->quadraticAttenuation << eol;
	out << ret;
}


void
PierrotParser::Output(LAPtextBufferedOutput& out, pMaterial* m)
{
	if ((out.GetOutputs() == LAP_BUFFER_NO_OUTPUT) || out.IsSilent())
		return ;

	out << "material '" << m->name << '\'' << eol;
	out << adv;
		out << "opacity = " << m->opacity << eol;
		out << "ambient = ";	PierrotParser::Output(out, m->ambient);	out << eol;
		out << "diffuse = ";	PierrotParser::Output(out, m->diffuse);	out << eol;
		out << "specular = ";	PierrotParser::Output(out, m->specular);	out << eol;
		out << "emission = ";	PierrotParser::Output(out, m->emission);	out << eol;
		out << "shininess = " << m->shininess << eol;
	out << ret;
}


void
PierrotParser::Output(LAPtextBufferedOutput& out, pObject* o, uint32 flags)
{
	if ((out.GetOutputs() == LAP_BUFFER_NO_OUTPUT) || out.IsSilent())
		return ;

	out << "object '" << o->name << '\'' << eol;
	out << adv;

		if (flags & PRINT_OBJECT_BOUNDS)
		{
			// bounds
			out << "bounding box: ";
			PierrotParser::Output(out, o->bounds);
			out << eol;
			out << "bounding sphere: ";
			PierrotParser::Output(out, o->sbounds);
			out << eol;
		}

		if (flags & PRINT_OBJECT_MATRIX)
			PierrotParser::Output(out, o->matrix);

		out << o->dlVertices.CountItems() << " stupid vertices" << eol;

		out << o->dots.CountItems() << " dots" << eol;
		if (flags & PRINT_OBJECT_DOTS)
		{
			out << adv;
			for (int32 i = 0; i < o->dots.CountItems(); i++)
			{
				pPoint3			pt = o->dlVertices[o->dots[i]];
				PierrotParser::Output(out, pt);
				out << eol;
			}
			out << ret;
		}

		out << o->lines.CountItems() << " lines" << eol;
		if (flags & PRINT_OBJECT_LINES)
		{
			out << adv;
			for (int32 i = 0; i < o->dots.CountItems(); i++)
			{
				pPoint3			start = o->dlVertices[o->lines[i].start];
				pPoint3			end = o->dlVertices[o->lines[i].end];
				PierrotParser::Output(out, start);
				out << " -> ";
				PierrotParser::Output(out, end);
				out << eol;
			}
			out << ret;
		}

		if (o->OwnVertices())
		{
			out << o->vertices->CountItems() << " vertices" << eol;
			if (flags & PRINT_OBJECT_POINTS)
			{
				out << adv;
				for (int32 i = 0; i < o->vertices->CountItems(); i++)
				{
					PierrotParser::Output(out, (*o->vertices)[i]);
					out << eol;
				}
				out << ret;
			}
		}

		out << o->faces.CountItems() << " faces" << eol;
		if (flags & PRINT_OBJECT_FACES)
		{
			out << adv;
			for (int32 i = 0; i < o->faces.CountItems(); i++)
			{
				PierrotParser::Output(out, o->faces[i]);
				out << eol;
			}
			out << ret;
		}
	out << ret;
}


void
PierrotParser::Output(LAPtextBufferedOutput& out, pScene* s, uint32 printFlags)
{
	if ((out.GetOutputs() == LAP_BUFFER_NO_OUTPUT) || out.IsSilent())
		return ;

	out << "scene '" << s->name << '\'' << eol;
	out << adv;

		if (printFlags & PRINT_VIEW)
		{
			out << "from ";		PierrotParser::Output(out, s->viewsys.origin);		out << eol;
			out << "  to ";		PierrotParser::Output(out, s->viewsys.target);		out << eol;
			out << "   u ";		PierrotParser::Output(out, s->viewsys.u);			out << eol;
			out << "   v ";		PierrotParser::Output(out, s->viewsys.v);			out << eol;
			out << "   n ";		PierrotParser::Output(out, s->viewsys.n);			out << eol << eol;
		}

		// bounds
		if (printFlags & PRINT_BOUNDS)
		{
			out << "bounding box: ";
			PierrotParser::Output(out, s->bounds);
			out << eol;
			out << "bounding sphere: ";
			PierrotParser::Output(out, s->sbounds);
			out << eol;
		}

		// ambient light
		out << "ambient light: ";
		PierrotParser::Output(out, s->ambient);
		out << eol;

		// lights
		out << s->lights.CountItems() << " light";
		if (s->lights.CountItems() > 1)
			out << 's';
		out << eol;
		if (printFlags & PRINT_LIGHT_ALL)
		{
			out << adv;
				for (int32 i = 0; i < s->lights.CountItems(); i++)
				{
					PierrotParser::Output(out, s->lights[i]);
					out << eol;
				};
			out << ret;
		}

		// materials
		out << s->materials.CountItems() << " material";
		if (s->materials.CountItems() > 1)
			out << 's';
		out << eol;
		if (printFlags & PRINT_MATERIAL_ALL)
		{
			out << adv;
				for (int32 i = 0; i < s->materials.CountItems(); i++)
				{
					PierrotParser::Output(out, s->materials[i]);
					out << eol;
				};
			out << ret;
		}

		// objects
		out << s->objects.CountItems() << " object";
		if (s->objects.CountItems() > 1)
			out << 's';
		out << eol;
		if (printFlags & PRINT_OBJECT_ALL)
		{
			out << adv;
				for (int32 i = 0; i < s->objects.CountItems(); i++)
				{
					PierrotParser::Output(out, s->objects[i], printFlags);
					out << eol;
				};
			out << ret;
		}
	out << ret << "done " << s->name << eol;
}


int32
PierrotParser::NewVertex(pObject* object, pPoint3& pt)
{
	int32				vertexIndex = object->vertices->Search(pt);
	if (vertexIndex < 0)
	{
		vertexIndex = object->vertices->CountItems();
		object->vertices->AddItem(pt);
	}
	return vertexIndex;
}


pDirectionalLight*
PierrotParser::NewDirectionalLight(char* n, pVector3& dir)
{
	pDirectionalLight*		light = new pDirectionalLight(n, dir);
	scene->lights.AddItem(light);
	return light;
}


pPositionalLight*
PierrotParser::NewPositionalLight(char* n, pPoint3& pos)
{
	pPositionalLight*		light = new pPositionalLight(n, pos);
	scene->lights.AddItem(light);
	return light;
}


pSpotlight*
PierrotParser::NewSpotlight(char* n, pPoint3& pos, pVector3& dir, float exp, float coff)
{
	pSpotlight*				light = new pSpotlight(n, pos, dir, exp, coff);
	scene->lights.AddItem(light);
	return light;
}


pMaterial*
PierrotParser::NewMaterial(char* n)
{
	pMaterial*				material = new pMaterial(n);
	scene->materials.AddItem(material);
	return material;
}


pMaterial*
PierrotParser::NewMaterial(char* n, pSpectra& diff)
{
	pMaterial*				material = new pMaterial(n);
	material->diffuse = diff;
	scene->materials.AddItem(material);
	return material;
}


pMaterial*
PierrotParser::NewMaterial(char* n, float op, pSpectra& em, float sh)
{
	pMaterial*				material = new pMaterial(n, op, em, sh);
	scene->materials.AddItem(material);
	return material;
}


pObject*
PierrotParser::NewObject(char* n)
{
	pObject*			obj = new pObject(n);
	scene->objects.AddItem(obj);
	return obj;
}


pObject*
PierrotParser::NewObject(char* n,
						int32 vi, int32 vb,
						int32 fi, int32 fb)
{
	pObject*			obj = new pObject(n, vi, vb, fi, fb);
	scene->objects.AddItem(obj);
	return obj;
}


pObject*
PierrotParser::NewObject(char* n, pPointsArray* vrt, pVectorsArray* nrm)
{
	pObject*			obj = new pObject(n, vrt, nrm);
	scene->objects.AddItem(obj);
	return obj;
}


// spin it!
// used for objects whose spinDesc contains the revolution axys
pObject*
PierrotParser::BuildSpinObject(	char* name,
								pPointsArray* spinDesc,
								pMaterial* material,
								pPoint3& center,
								pVector3& dir,
								int32 n)
{
	pObject*			out = new pObject(name);

	// generate vertices
	out->vertices->SetAlloc(2 + (spinDesc->CountItems()-2)*n, 0);
	out->vertices->AddItem((*spinDesc)[0]);
	out->vertices->AddItem((*spinDesc)[spinDesc->CountItems()-1]);
	for (int32 i = 0; i < n; i++)
	{
		float			tmp = i * 2.0 * M_PI / n;
		float			cs = cos(tmp);
		float			sn = sin(tmp);

		for (int32 j = 1; j < spinDesc->CountItems()-1; j++)
		{
			pPoint3&	pt = (*spinDesc)[j];
			out->vertices->AddItem(pPoint3(	cs * pt.x + sn * pt.z,
											pt.y,
											-sn * pt.x + cs * pt.z));
		}
	}

	// vertices transformation
	pMatrix3			rotation = ~pMatrix3::RotateToZ(dir);
	pVector3			translation(center.x, center.y, center.z);
	pMatrix4			mat(rotation, translation);
	out->Transform(mat);

	// generate faces
	out->faces.SetAlloc((spinDesc->CountItems()-1)*n, 0);
	for (int32 i = 0; i < n; i++)
	{
		int32			a = 2 + (spinDesc->CountItems()-2)*i;
		int32			b = 2 + (spinDesc->CountItems()-2)*((i == n-1)?0:i+1);

		out->faces.AddItem(new pFace(material, 0, a, b));
		for (int32 j = 1; j < spinDesc->CountItems()-2; j++)
		{
			out->faces.AddItem(new pFace(material, a, a+1, b+1, b));
			a++;
			b++;
		}
		out->faces.AddItem(new pFace(material, 1, b, a));
	}

	return out;
}


// spin it!
// used by torus
pObject*
PierrotParser::BuildSpinObject2(char* name,
								pPointsArray* spinDesc,
								pMaterial* material,
								pPoint3& center,
								pVector3& dir,
								int32 n)
{
	pObject*			out = new pObject(name);

	// generate vertices
	out->vertices->SetAlloc(spinDesc->CountItems()*n, 0);
	for (int32 i = 0; i < n; i++)
	{
		float			tmp = i * 2.0 * M_PI / n;
		float			cs = cos(tmp);
		float			sn = sin(tmp);

		for (int32 j = 0; j < spinDesc->CountItems(); j++)
		{
			pPoint3&	pt = (*spinDesc)[j];
			out->vertices->AddItem(pPoint3(	cs * pt.x + sn * pt.z,
											pt.y,
											-sn * pt.x + cs * pt.z));
		}
	}

	// vertices transformation
	pMatrix3			rotation = ~pMatrix3::RotateToZ(dir);
	pVector3			translation(center.x, center.y, center.z);
	pMatrix4			mat(rotation, translation);
	out->Transform(mat);

	// generate faces
	out->faces.SetAlloc((spinDesc->CountItems()-1)*n, 0);
	for (int32 i = 0; i < n; i++)
	{
		int32			a = spinDesc->CountItems()*i;
		int32			b = spinDesc->CountItems()*((i == n-1)?0:i+1);

		for (int32 j = 0; j < spinDesc->CountItems()-1; j++)
		{
			out->faces.AddItem(new pFace(material, a, b, b+1, a+1));
			a++;
			b++;
		}
	}

	return out;
}


pObject*
PierrotParser::BuildVolume(char* name, pSpinVolume& spinVolume, pMaterial* material, int32 n1, int32 n2)
{
	pPoint3					center;
	pVector3				direction;
	pPointsArray*			spinDesc;
	pObject*				out;

	spinVolume.GetPosition(center, direction);
	spinDesc = spinVolume.GenerateSpinDesc(n1);
	out = BuildSpinObject(name, spinDesc, material, center, direction, n2);

	delete spinDesc;
	return out;
}


pObject*
PierrotParser::BuildVolume2(char* name, pSpinVolume& spinVolume, pMaterial* material, int32 n1, int32 n2)
{
	pPoint3					center;
	pVector3				direction;
	pPointsArray*			spinDesc;
	pObject*				out;

	spinVolume.GetPosition(center, direction);
	spinDesc = spinVolume.GenerateSpinDesc(n1);
	out = BuildSpinObject2(name, spinDesc, material, center, direction, n2);

	delete spinDesc;
	return out;
}


pObject*
PierrotParser::BuildCylinder(char* name, pCylinder& cyl, pMaterial* material, int32 n)
{
	pPoint3					center;
	pVector3				direction;
	pPointsArray*			spinDesc;
	pObject*				out;

	cyl.GetPosition(center, direction);
	spinDesc = cyl.GenerateSpinDesc(1);

	if (cyl.isCapped)
		out = BuildSpinObject(name, spinDesc, material, center, direction, n);
	else
		out = BuildSpinObject2(name, spinDesc, material, center, direction, n);

	delete spinDesc;
	return out;
}


pObject*
PierrotParser::BuildCone(char* name, pCone& cone, pMaterial* material, int32 n)
{
	pPoint3					center;
	pVector3				direction;
	pPointsArray*			spinDesc;
	pObject*				out;

	cone.GetPosition(center, direction);
	spinDesc = cone.GenerateSpinDesc(1);

	if (cone.isCapped)
		out = BuildSpinObject(name, spinDesc, material, center, direction, n);
	else
		out = BuildSpinObject2(name, spinDesc, material, center, direction, n);

	delete spinDesc;
	return out;
}


pObject*
PierrotParser::BuildSphere(char* name, pSphere& sphere, pMaterial* material, int32 n1, int32 n2)
{
	pPoint3					center;
	pVector3				direction;
	pPointsArray*			spinDesc;
	pObject*				out;

	sphere.GetPosition(center, direction);
	spinDesc = sphere.GenerateSpinDesc(n1);
	out = BuildSpinObject(name, spinDesc, material, center, direction, n2);

	delete spinDesc;
	return out;
}


pObject*
PierrotParser::BuildTorus(char* name, pTorus& torus, pMaterial* material, int32 n1, int32 n2)
{
	pPoint3					center;
	pVector3				direction;
	pPointsArray*			spinDesc;
	pObject*				out;

	torus.GetPosition(center, direction);
	spinDesc = torus.GenerateSpinDesc(n1);
	out = BuildSpinObject2(name, spinDesc, material, center, direction, n2);

	delete spinDesc;
	return out;
}


pObject*
PierrotParser::BuildRing(char* name, pRing& ring, pMaterial* material, int32 n2)
{
	pPoint3					center;
	pVector3				direction;
	pPointsArray*			spinDesc;
	pObject*				out;

	if (ring.innerRadius == 0.0)
		ring.innerRadius = ring.outerRadius * 0.001;

	ring.GetPosition(center, direction);
	spinDesc = ring.GenerateSpinDesc(2);
	out = BuildSpinObject2(name, spinDesc, material, center, direction, n2);

	delete spinDesc;
	return out;
}

