//
//
//	Pierrot 3d volume
//
//										(C) JoOl 1998


#include "PierrotImpExp.h"

#include <math.h>
#include <string.h>
#include "PierrotVolume.h"
#include "PierrotObject.h"



//--------------------------------------------------------
// 3d box
//--------------------------------------------------------

pBox::pBox()
:minPt(0.0, 0.0, 0.0),
 maxPt(0.0, 0.0, 0.0)
{
}


pBox::pBox(pPoint3& mn, pPoint3& mx)
:minPt(mn), maxPt(mx)
{
}


pBox::pBox(pBox& b)
:minPt(b.minPt), maxPt(b.maxPt)
{
}


pBox::~pBox()
{}


void
pBox::Set(pPoint3& mn, pPoint3& mx)
{
	minPt = mn;
	maxPt = mx;
}


void
pBox::SetMin(pPoint3& m)
{ minPt = m; };


void
pBox::SetMax(pPoint3& m)
{ maxPt = m; };


pPoint3
pBox::Center()
{
	return pPoint3(	0.5*(maxPt.x + minPt.x),
					0.5*(maxPt.y + minPt.y),
					0.5*(maxPt.z + minPt.z));
}


bool
pBox::Eq(pBox& b, float e)
{
	return (minPt.Eq(b.minPt, e))
		&& (maxPt.Eq(b.maxPt, e));
}


bool
pBox::Contains(pPoint3& pt) const
{
	return (pt.x >= minPt.x) && (pt.x <= maxPt.x)
		&& (pt.y >= minPt.y) && (pt.y <= maxPt.y)
		&& (pt.z >= minPt.z) && (pt.z <= maxPt.z);
}


bool
pBox::Contains(pBox& b) const
{
	return (b.minPt.x >= minPt.x) && (b.maxPt.x <= maxPt.x)
		&& (b.minPt.y >= minPt.y) && (b.maxPt.y <= maxPt.y)
		&& (b.minPt.z >= minPt.z) && (b.maxPt.z <= maxPt.z);
}


bool
pBox::Excludes(pPoint3& pt) const
{
	return (pt.x < minPt.x) || (pt.x > maxPt.x)
		|| (pt.y < minPt.y) || (pt.y > maxPt.y)
		|| (pt.z < minPt.z) || (pt.z > maxPt.z);
}


bool
pBox::Excludes(pBox& b) const
{
	return (b.maxPt.x < minPt.x) || (b.minPt.x > maxPt.x)
		|| (b.maxPt.y < minPt.y) || (b.minPt.y > maxPt.y)
		|| (b.maxPt.z < minPt.z) || (b.minPt.z > maxPt.z);
}


pBox
pBox::operator | (pPoint3& p)
{
	return pBox(pPoint3(min_c(minPt.x, p.x),
						min_c(minPt.y, p.y),
						min_c(minPt.z, p.z)),
				pPoint3(max_c(maxPt.x, p.x),
						max_c(maxPt.y, p.y),
						max_c(maxPt.z, p.z))
				);
}

pBox
pBox::operator | (pBox& b)
{
	return pBox(pPoint3(min_c(minPt.x, b.minPt.x),
						min_c(minPt.y, b.minPt.y),
						min_c(minPt.z, b.minPt.z)),
				pPoint3(max_c(maxPt.x, b.maxPt.x),
						max_c(maxPt.y, b.maxPt.y),
						max_c(maxPt.z, b.maxPt.z))
				);
}


pBox&
pBox::operator |= (pPoint3& p)
{
	minPt.x = min_c(minPt.x, p.x);
	minPt.y = min_c(minPt.y, p.y);
	minPt.z = min_c(minPt.z, p.z);

	maxPt.x = max_c(maxPt.x, p.x);
	maxPt.y = max_c(maxPt.y, p.y);
	maxPt.z = max_c(maxPt.z, p.z);
	return *this;
}


pBox&
pBox::operator |= (pBox& b)
{
	minPt.x = min_c(minPt.x, b.minPt.x);
	minPt.y = min_c(minPt.y, b.minPt.y);
	minPt.z = min_c(minPt.z, b.minPt.z);

	maxPt.x = max_c(maxPt.x, b.maxPt.x);
	maxPt.y = max_c(maxPt.y, b.maxPt.y);
	maxPt.z = max_c(maxPt.z, b.maxPt.z);
	return *this;
}


pBox&
pBox::operator = (pBox& b)
{
	minPt = b.minPt;
	maxPt = b.maxPt;
	return *this;
}


int
pBox::operator == (pBox& b)
{
	return (minPt == b.minPt) && (maxPt == b.maxPt);
}


int
pBox::operator != (pBox& b)
{
	return (minPt != b.minPt) || (maxPt != b.maxPt);
}



//--------------------------------------------------------
// 3d sphere
//--------------------------------------------------------

pSphere::pSphere()
:center(0.0, 0.0, 0.0),
 radius(0.0)
{
}


pSphere::pSphere(pPoint3& c, float r)
:center(c), radius(r)
{
}


pSphere::pSphere(pSphere& s)
:center(s.center), radius(s.radius)
{
}


pSphere::~pSphere()
{}


void
pSphere::Set(pPoint3& c, float r)
{
	center = c;
	radius = r;
}


bool
pSphere::Eq(pSphere& s, float e)
{
	return (center.Eq(s.center, e))
		&& (fabs(radius - s.radius) < e);
}


bool
pSphere::Contains(pPoint3& pt)
{ return pVector3(center, pt).Length2() <= radius*radius; }


bool
pSphere::Contains(pSphere& s)
{ return pVector3(center, s.center).Length2() <= (radius + s.radius)*(radius + s.radius); }


bool
pSphere::Excludes(pPoint3& pt)
{ return pVector3(center, pt).Length2() > radius*radius; }


bool
pSphere::Excludes(pSphere& s)
{ return pVector3(center, s.center).Length2() > (radius + s.radius)*(radius + s.radius); }


pSphere
pSphere::operator | (pPoint3& p)
{
	pVector3			b(center, p);
	float				len = b.Length();
	if (len <= radius)
		return pSphere(center, radius);
	b.Norm();
	pPoint3				c(	center.x - radius * b.x,
							center.y - radius * b.y,
							center.z - radius * b.z);
	len = 0.5 * (radius + len);
	b *= len;
	return pSphere(pPoint3(c.x + b.x, c.y + b.y, c.z + b.z), len);
}


pSphere&
pSphere::operator |= (pPoint3& p)
{
	pVector3			b(center, p);
	float				len = b.Length();
	if (len > radius)
	{
		b.Norm();
		pPoint3				c(	center.x - radius * b.x,
								center.y - radius * b.y,
								center.z - radius * b.z);
		radius = 0.5 * (radius + len);
		b *= radius;
		center.Set(c.x + b.x, c.y + b.y, c.z + b.z);
	}
	return *this;
}


pSphere
pSphere::operator | (pSphere& s)
{
	pVector3			b(center, s.center);
	float				len = b.Length();
	if (len <= radius - s.radius)
		return pSphere (center, radius);
	b.Norm();
	pPoint3				c(	center.x - radius * b.x,
							center.y - radius * b.y,
							center.z - radius * b.z);
	len = 0.5 * (radius + len + s.radius);
	b *= len;
	return pSphere(pPoint3(c.x + b.x, c.y + b.y, c.z + b.z), len);
}


pSphere&
pSphere::operator |= (pSphere& s)
{
	pVector3			b(center, s.center);
	float				len = b.Length();
	if (len > radius - s.radius)
	{
		b.Norm();
		pPoint3				c(	center.x - radius * b.x,
								center.y - radius * b.y,
								center.z - radius * b.z);
		radius = 0.5 * (radius + len + s.radius);
		b *= radius;
		center.Set(c.x + b.x, c.y + b.y, c.z + b.z);
	}
	return *this;
}


pSphere&
pSphere::operator = (pSphere& s)
{
	center = s.center;
	radius = s.radius;
	return *this;
}


int
pSphere::operator == (pSphere& s)
{
	return (center == s.center) && (radius == s.radius);
}


int
pSphere::operator != (pSphere& s)
{
	return (center != s.center) || (radius != s.radius);
}


pPointsArray*
pSphere::GenerateSpinDesc(int32 nbSteps)
{
	pPointsArray*	spinDesc = new pPointsArray(nbSteps+1, 0);
	for (int32 i = 0; i <= nbSteps; i++)
	{
		float			tmp = M_PI_2 - (i * M_PI / nbSteps);
		pPoint3			pt(radius * cos(tmp), radius * sin(tmp), 0.0);
		spinDesc->AddItem(pt);
	}

	return spinDesc;
}


void
pSphere::GetPosition(pPoint3& c, pVector3& axis)
{
	c = center;
	axis.Set(0.0, 1.0, 0.0);
}


//--------------------------------------------------------
// 3d cylinder
//--------------------------------------------------------

pCylinder::pCylinder()
: isCapped(true)
{
}


pCylinder::pCylinder(pPoint3& b, pPoint3& a, float r, bool capped)
:base(b),
 apex(a),
 radius(r),
 isCapped(capped)
{
}


pCylinder::pCylinder(pCylinder& c)
:base(c.base),
 apex(c.apex),
 radius(c.radius),
 isCapped(c.isCapped)
{
}


pCylinder::~pCylinder()
{}


void
pCylinder::Set(pPoint3& b, pPoint3& a, float r, bool capped)
{
	base = b;
	apex = a;
	radius = r;
	isCapped = capped;
}


bool
pCylinder::Eq(pCylinder& c, float e)
{
	return (base.Eq(c.base, e))
		&& (apex.Eq(c.apex, e))
		&& (fabs(radius - c.radius) < e)
		&& (isCapped == c.isCapped);
}


pCylinder&
pCylinder::operator = (pCylinder& c)
{
	base = c.base;
	apex = c.apex;
	radius = c.radius;
	isCapped = c.isCapped;
	return *this;
}


int
pCylinder::operator == (pCylinder& c)
{
	return (base == c.base) && (apex == c.apex)
			&& (radius == c.radius)
			&& (isCapped == c.isCapped);
}


int
pCylinder::operator != (pCylinder& c)
{
	return (base != c.base) || (apex != c.apex)
			|| (radius != c.radius)
			|| (isCapped != c.isCapped);
}


pPointsArray*
pCylinder::GenerateSpinDesc(int32 nbSteps)
{
	float			cylHeight = pVector3(base, apex).Length();
	pPoint3			start(radius, cylHeight, 0.0);
	pVector3		edge(start, pPoint3(radius, 0.0, 0.0));
	pPointsArray*	spinDesc;

	if (isCapped)
		spinDesc = new pPointsArray(nbSteps+3, 0);
	else
		spinDesc = new pPointsArray(nbSteps+1, 0);

	if (isCapped)
		spinDesc->AddItem(pPoint3(0.0, cylHeight, 0.0));
	for (int32 i = 0; i <= nbSteps; i++)
	{
		pPoint3			pt = start + edge * float(i/nbSteps);
		spinDesc->AddItem(pt);
	}
	if (isCapped)
		spinDesc->AddItem(pPoint3(0.0, 0.0, 0.0));

	return spinDesc;
}


void
pCylinder::GetPosition(pPoint3& c, pVector3& axis)
{
	c = base;
	axis.Set(base, apex);
}


//--------------------------------------------------------
// 3d cone
//--------------------------------------------------------

pCone::pCone()
: isCapped(true)
{
}


pCone::pCone(pPoint3& b, float br, pPoint3& a, float ar, bool capped)
:base(b),
 baseRadius(br),
 apex(a),
 apexRadius(ar),
 isCapped(capped)
{
}


pCone::pCone(pCone& c)
:base(c.base),
 baseRadius(c.baseRadius),
 apex(c.apex),
 apexRadius(c.apexRadius),
 isCapped(c.isCapped)
{
}


pCone::~pCone()
{}


void
pCone::Set(pPoint3& b, float br, pPoint3& a, float ar, bool capped)
{
	base = b;
	baseRadius = br;
	apex = a;
	apexRadius = ar;
	isCapped = capped;
}


bool
pCone::IsCylinder() const
{
	return baseRadius == apexRadius;
}


bool
pCone::IsPureCone() const
{
	return apexRadius == 0.0;
}


bool
pCone::Eq(pCone& c, float e)
{
	return (base.Eq(c.base, e))
		&& (apex.Eq(c.apex, e))
		&& (fabs(apexRadius - c.apexRadius) < e)
		&& (fabs(baseRadius - c.baseRadius) < e)
		&& (isCapped == c.isCapped);
}


pCone&
pCone::operator = (pCone& c)
{
	base = c.base;
	baseRadius = c.baseRadius;
	apex = c.apex;
	apexRadius = c.apexRadius;
	isCapped = c.isCapped;
	return *this;
}


int
pCone::operator == (pCone& c)
{
	return (base == c.base) && (baseRadius == c.baseRadius)
			&& (apex == c.apex) && (apexRadius == c.apexRadius)
			&& (isCapped == c.isCapped);
}


int
pCone::operator != (pCone& c)
{
	return (base != c.base) || (baseRadius != c.baseRadius)
			|| (apex != c.apex) || (apexRadius != c.apexRadius)
			|| (isCapped != c.isCapped);
}


pPointsArray*
pCone::GenerateSpinDesc(int32 nbSteps)
{
	float			coneHeight = pVector3(base, apex).Length();
	pPoint3			start(apexRadius, coneHeight, 0.0);
	pVector3		edge(start, pPoint3(baseRadius, 0.0, 0.0));
	pPointsArray*	spinDesc = new pPointsArray(nbSteps+3, 0);

	if (isCapped && (apexRadius != 0.0))
		spinDesc->AddItem(pPoint3(0.0, coneHeight, 0.0));
	for (int32 i = 0; i <= nbSteps; i++)
	{
		pPoint3			pt = start + edge * float(i/nbSteps);
		spinDesc->AddItem(pt);
	}
	if (isCapped)
		spinDesc->AddItem(pPoint3(0.0, 0.0, 0.0));

	return spinDesc;
}


void
pCone::GetPosition(pPoint3& c, pVector3& axis)
{
	c = base;
	axis.Set(base, apex);
}



//--------------------------------------------------------
// 3d torus
//--------------------------------------------------------

pTorus::pTorus()
{
}


pTorus::pTorus(pPoint3& c, pVector3& n, float r, float _a, float _b)
:center(c),
 normal(n),
 radius(r),
 a(_a),
 b(_b)
{
	normal.Norm();
}


pTorus::pTorus(pTorus& t)
:center(t.center),
 normal(t.normal),
 radius(t.radius),
 a(t.a),
 b(t.b)
{
	normal.Norm();
}


pTorus::~pTorus()
{
}


void
pTorus::Set(pPoint3& c, pVector3& n, float r, float _a, float _b)
{
	center = c;
	normal = n;
	normal.Norm();
	radius = r;
	a = _a;
	b = _b;
}


bool
pTorus::Eq(pTorus& t, float e)
{
	return (center.Eq(t.center, e))
		&& (normal.Eq(t.normal, e))
		&& (fabs(radius - t.radius) < e)
		&& (fabs(a - t.a) < e)
		&& (fabs(b - t.b) < e);
}


pTorus&
pTorus::operator=(pTorus& t)
{
	center = t.center;
	normal = t.normal;
	radius = t.radius;
	a = t.a;
	b = t.b;

	return *this;
}


int
pTorus::operator==(pTorus& t)
{
	return (center == t.center)
		&& (normal == t.normal)
		&& (radius == t.radius)
		&& (a == t.a)
		&& (b == t.b);
}


int
pTorus::operator!=(pTorus& t)
{
	return (center != t.center)
		|| (normal != t.normal)
		|| (radius != t.radius)
		|| (a != t.a)
		|| (b != t.b);
}


pPointsArray*
pTorus::GenerateSpinDesc(int32 nbSteps)
{
	pPointsArray*	spinDesc = new pPointsArray(nbSteps+1, 0);

	for (int32 i = 0; i <= nbSteps; i++)
	{
		float			tmp = i * 2.0 * M_PI / nbSteps;
		pPoint3			pt(a * cos(tmp), b * sin(tmp), 0.0);
		pt.x += radius;
		spinDesc->AddItem(pt);
	}

	return spinDesc;
}


void
pTorus::GetPosition(pPoint3& c, pVector3& axis)
{
	c = center;
	axis = normal;
}



//--------------------------------------------------------
// 3d ring
//--------------------------------------------------------

pRing::pRing()
{
}


pRing::pRing(pPoint3& b, pVector3& n, float irr, float orr)
:base(b),
 normal(n),
 innerRadius(irr),
 outerRadius(orr)
{
}


pRing::pRing(pRing& r)
:base(r.base),
 normal(r.normal),
 innerRadius(r.innerRadius),
 outerRadius(r.outerRadius)
{
}


pRing::~pRing()
{}


void
pRing::Set(pPoint3& b, pVector3& n, float irr, float orr)
{
	base = b;
	normal = n;
	innerRadius = irr;
	outerRadius = orr;
}


bool
pRing::Eq(pRing& r, float e)
{
	return (base.Eq(r.base, e))
		&& (normal.Colinear(r.normal, e))
		&& (fabs(innerRadius - r.innerRadius) < e)
		&& (fabs(outerRadius - r.outerRadius) < e);
}


pRing&
pRing::operator = (pRing& r)
{
	base = r.base;
	normal = r.normal;
	innerRadius = r.innerRadius;
	outerRadius = r.outerRadius;
	return *this;
}


int
pRing::operator == (pRing& r)
{
	return (base == r.base)
			&& (normal == r.normal)
			&& (innerRadius == r.innerRadius)
			&& (outerRadius == r.outerRadius);
}


int
pRing::operator != (pRing& r)
{
	return (base != r.base)
			|| (normal != r.normal)
			|| (innerRadius != r.innerRadius)
			|| (outerRadius != r.outerRadius);
}


pPointsArray*
pRing::GenerateSpinDesc(int32 nbSteps)
{
	pPoint3			start(innerRadius, 0.0, 0.0);
	pVector3		edge(start, pPoint3(outerRadius, 0.0, 0.0));
	pPointsArray*	spinDesc = new pPointsArray(nbSteps+1, 0);

	for (int32 i = 0; i < nbSteps; i++)
	{
		pPoint3			pt = start + edge * float(i/(nbSteps-1));
		spinDesc->AddItem(pt);
	}

	return spinDesc;
}


void
pRing::GetPosition(pPoint3& c, pVector3& axis)
{
	c = base;
	axis = normal;
}



//--------------------------------------------------------
// 3d prism
//--------------------------------------------------------

pPrism::pPrism()
: isCapped(true)
{
}


pPrism::pPrism(PierrotArray<pPoint3 >& b, pVector3& d, bool c)
{
	Set(b, d, c);
}


pPrism::pPrism(pPrism& p)
{
	Set(p.base, p.direction, p.isCapped);
}


pPrism::~pPrism()
{
}


void
pPrism::Set(PierrotArray<pPoint3 >& b, pVector3& d, bool c)
{
	direction = d;
	isCapped = c;

	if (base.IsAllocated())
		base.DelAlloc();
	base.SetAlloc(b.CountItems(), 0);
	for (int32 i = 0; i < b.CountItems(); i++)
		base.AddItem(b[i]);
}


bool
pPrism::Eq(pPrism& p, float e)
{
	if (!direction.Eq(p.direction, e))
		return false;
	if (isCapped != p.isCapped)
		return false;
	if (base.CountItems() != p.base.CountItems())
		return false;

	for (int32 i = 0; i < base.CountItems(); i++)
		if (!base[i].Eq(p.base[i], e))
			return false;
	return true;
}


pPrism&
pPrism::operator = (pPrism& p)
{
	Set(p.base, p.direction, p.isCapped);
}


int
pPrism::operator == (pPrism& p)
{
	if (direction != p.direction)
		return false;
	if (isCapped != p.isCapped)
		return false;
	if (base.CountItems() != p.base.CountItems())
		return false;

	for (int32 i = 0; i < base.CountItems(); i++)
		if (base[i] != p.base[i])
			return false;
	return true;
}


int
pPrism::operator != (pPrism& p)
{
	if (direction != p.direction)
		return true;
	if (isCapped != p.isCapped)
		return true;
	if (base.CountItems() != p.base.CountItems())
		return true;

	for (int32 i = 0; i < base.CountItems(); i++)
		if (base[i] != p.base[i])
			return true;
	return false;
}


pObject*
pPrism::BuildPrism(char* name, pMaterial* mat, int32 n)
{
	pObject*			out = new pObject(name);

	out->vertices->SetAlloc(base.CountItems() * n, 0);

	// generate the slices
	for (int32 i = 0; i < n; i++)
	{
		// generate a "slice" translated by "tr"
		pVector3		tr = direction * float(i / (n-1));
		for (int32 j = 0; j < base.CountItems(); j++)
		{
			pPoint3		pt = base[j] + tr;
			out->vertices->AddItem(pt);
//printf("slice %d, %d: (%f %f %f)\n", i, j, pt.x, pt.y, pt.z);
		}
	}

	// generate the faces
	if (isCapped)
	{
		out->faces.SetAlloc(base.CountItems() * (n-1) + 2, 0);

		PierrotArray<int32 >	cap(base.CountItems(), 0);

		// top cap
		for (int32 i = 0; i < base.CountItems(); i++)
			cap.AddItem(i);
		out->faces.AddItem(new pFace(mat, cap.CountItems(), cap.Items()));

		// bottom cap
		for (int32 i = 0; i < base.CountItems(); i++)
			cap[i] = base.CountItems() * n -i -1;
		out->faces.AddItem(new pFace(mat, cap.CountItems(), cap.Items()));
	}
	else
		out->faces.SetAlloc(base.CountItems() * (n-1), 0);

	// generate prism's body
	for (int32 i = 0; i < n-1; i++)
	{
		int32				in = i*base.CountItems();
		for (int32 j = 0; j < base.CountItems(); j++)
		{
			int32				a = in + j;
			int32				b = (j == base.CountItems()-1) ? in : (a+1);
//printf("(%d %d %d %d)\n", a, b, b+base.CountItems(), a+base.CountItems());
			out->faces.AddItem(new pFace(mat, a, b, b+base.CountItems(),
										a+base.CountItems()));
		}
	}

	return out;
}

