//
//
//	COB objects parser
//
//										(C) JoOl 1998

#include "ParserCob.h"
#include "LAPerrors.h"


#if __POWERPC__
#pragma export on
#endif
char			handlerName[]		= "Caligari's trueSpace v2.0 (.cob)";
char			handlerVersion[]	= "0.9.0";
char			handlerAuthor[]		= "Alain GAUTHEROT";
char			handlerEmail[]		= "gauthero@club-internet.fr";
#if __POWERPC__
#pragma export reset
#endif


ParserCob*
instantiate_parser(int32 bs)
{
	return new ParserCob(bs);
}


ParserCob::ParserCob(int32 blockSize)
{
	// input is a big-endian binary file
	SetLexer(new LAPbinLexer(max_c(1024, blockSize),
			/* LAP_BIG_ENDIAN or LAP_LITLE_ENDIAN */
			LAP_BIG_ENDIAN));

	// error output, tracer output, display
	SetTo(new LAPtracer(128, LAP_BUFFER_OUTPUT_ON_OVERFLOW, LAP_BUFFER_SCREEN_OUTPUT),
			new LAPtextBufferedOutput(128, LAP_BUFFER_OUTPUT_ON_OVERFLOW, LAP_BUFFER_NO_OUTPUT/*LAP_BUFFER_SCREEN_OUTPUT*/),
			new LAPdisplay(),
			LAP_OWN_ERROR_OUTPUT | LAP_OWN_TRACE_OUTPUT | LAP_OWN_DISPLAY);
}


ParserCob::~ParserCob()
{
	if (data)
		delete (pScene* )data;
}


void
ParserCob::_Terminate()
{
}


status_t
ParserCob::DoAbort()
{
	PierrotParser::Reset();
	return LAP_NO_ERROR;
}


status_t
ParserCob::_Identify()
{
	return ParseHeader();
}


status_t
ParserCob::_Parse()
{
	PierrotParser::Reset();

	scene = new pScene((char* )GetLexer()->GetInputName());

	pMaterial*			defaultMaterial = NewMaterial("default material");
	defaultMaterial->ambient.Set(0.1, 0.1, 0.1);
	defaultMaterial->diffuse.Set(0.4, 0.4, 0.4);
	defaultMaterial->specular.Set(0.5, 0.5, 0.5);
	defaultMaterial->shininess = 40.0;

	status_t		st = ParseHeader();
	if (st != LAP_NO_ERROR)
		return st;
	ReadMainChunk();

	pSphere				normSphere(pPoint3(0.0, 0.0, 0.0), 1.0);
	scene->Preprocess(&normSphere, appliInterface, 0.0, 1.0);

	scene->ambient.Set(0.4, 0.4, 0.4);
	if (scene->lights.CountItems() == 0)
	{
		pLight*			l1 = NewDirectionalLight("directional light 1", pVector3(-0.5, 0.5, 0.8));
		l1->ambient.Set(0.2, 0.2, 0.2);
		l1->diffuse.Set(1.0, 1.0, 1.0);
		l1->specular.Set(1.0, 1.0, 1.0);
	}

	Output(*traceOutput, scene, PRINT_LIGHT_ALL | PRINT_MATERIAL_ALL |
			PRINT_OBJECT_BOUNDS /*| PRINT_OBJECT_MATRIX | PRINT_OBJECT_POINTS*/);

	data = scene;
	scene = NULL;
	return LAP_NO_ERROR;
}


status_t
ParserCob::ParseHeader()
{
	char			id[10];
	char			version[8];
	char			binText;	// 'A' for ASCII, 'B' for binary
	char			lbe[2];		// little or big endian
	id[9] = 0;
	version[6] = 0;

	ReadBytes(id, 9);
	ReadBytes(version, 6);
	ReadBytes(&binText, 1);
	ReadBytes((char* )&lbe, 2);
	SkipBytes(32-9-6-1-2);

	if (strcmp(id, "Caligari ") != 0)
		return LAP_BAD_INPUT_TYPE;
	if (binText != 'B')
		return LAP_BAD_INPUT_TYPE;
	lexer->GetInput()->SetInputEndianness((lbe[0] == 'L') ? LAP_LITTLE_ENDIAN : LAP_BIG_ENDIAN);

	return LAP_NO_ERROR;
}


void
ParserCob::ParseChunkHeader(CobChunk& chunk)
{
	ReadBytes((char* )&chunk.chunkType, 4);
	ReadInt16(chunk.majorVersion);
	ReadInt16(chunk.minorVersion);
	ReadInt32(chunk.chunkId);
	ReadInt32(chunk.parentId);
	ReadInt32(chunk.chunkLen);
}


void
ParserCob::OutputChunk(CobChunk& chunk)
{
	char*			str = (char* )&chunk.chunkType;
	*traceOutput << "('" << str[0] << str[1] << str[2] << str[3] << "', "
				<< chunk.majorVersion << ", " << chunk.minorVersion
				<< ", chunkId=" << uint32(chunk.chunkId) << ", parentId="
				<< uint32(chunk.parentId) << ", length=" << chunk.chunkLen
				<< ')' << eol << flush;
}


void
ParserCob::Match(CobChunk& chunk, int32 type, int32 majvers, int32 minvers)
{
	if ((type != chunk.chunkType) || (majvers > chunk.majorVersion)
			|| (minvers < chunk.minorVersion))
		throw CobMismatchChunk("ParserCob::Match", type, majvers, minvers, chunk);

	*traceOutput << "match ";
	OutputChunk(chunk);
}


void
ParserCob::SkipChunk(CobChunk& chunk)
{
	char*			str = (char* )&chunk.chunkType;
	*traceOutput << "skip ";
	OutputChunk(chunk);
	SkipBytes(chunk.chunkLen);
}


char*
ParserCob::ParseString()
{
	int16			len;
	ReadInt16(len);
	char*			out = new char[len+1];
	ReadBytes(out, len);
	out[len] = 0;
	return out;
}


char*
ParserCob::ParseName()
{
	char			tmp[32];
	int16			dupecount;
	ReadInt16(dupecount);
	sprintf(tmp, "%d", dupecount);

	int16			len;
	ReadInt16(len);
	char*			out = new char[len+1+strlen(tmp)+1];
	ReadBytes(out, len);
	out[len] = 0;
	strcat(out, ",");
	strcat(out, tmp);
	return out;
}


void
ParserCob::ParsePoint(pPoint3& pt)
{
	ReadFloat(pt.x);
	ReadFloat(pt.y);
	ReadFloat(pt.z);
}


void
ParserCob::ParseVector(pVector3& vec)
{
	ReadFloat(vec.x);
	ReadFloat(vec.y);
	ReadFloat(vec.z);
}


void
ParserCob::ParseVector(pVector4& vec)
{
	ReadFloat(vec.x);
	ReadFloat(vec.y);
	ReadFloat(vec.z);
	ReadFloat(vec.w);
}


void
ParserCob::ParseColor(pSpectra& col)
{
	ReadFloat(col.r);
	ReadFloat(col.g);
	ReadFloat(col.b);
}


void
ParserCob::ParseColor(pSpectra4& col)
{
	ReadFloat(col.r);
	ReadFloat(col.g);
	ReadFloat(col.b);
	ReadFloat(col.a);
}


void
ParserCob::ParseLocalAxis(pMatrix4& matrix)
{
	*traceOutput << "ParseLocalAxis" << eol;
	*traceOutput << adv;

	pVector3			t, u, v, n;
	ParseVector(t);
	ParseVector(u);
	ParseVector(v);
	ParseVector(n);
/*
	matrix.SetX(pVector4(u.x, u.y, u.z, t.x));
	matrix.SetY(pVector4(v.x, v.y, v.z, t.y));
	matrix.SetZ(pVector4(n.x, n.y, n.z, t.z));
	matrix.SetW(pVector4(0.0, 0.0, 0.0, 1.0));
*/
	matrix.SetX(pVector4(u.x, v.x, n.x, t.x));
	matrix.SetY(pVector4(u.y, v.y, n.y, t.y));
	matrix.SetZ(pVector4(u.z, v.z, n.z, t.z));
	matrix.SetW(pVector4(0.0, 0.0, 0.0, 1.0));
/*
	*traceOutput << "t=";	Output(*traceOutput, t);	*traceOutput << eol;
	*traceOutput << "u=";	Output(*traceOutput, u);	*traceOutput << eol;
	*traceOutput << "v=";	Output(*traceOutput, v);	*traceOutput << eol;
	*traceOutput << "n=";	Output(*traceOutput, n);	*traceOutput << eol;
*/
	*traceOutput << ret;
}


void
ParserCob::ParsePosition(pMatrix4& matrix)
{
	*traceOutput << "ParsePosition" << eol;
	*traceOutput << adv;

	ParseVector(matrix.GetX());
	ParseVector(matrix.GetY());
	ParseVector(matrix.GetZ());
	matrix.SetW(pVector4(0.0, 0.0, 0.0, 1.0));

	Output(*traceOutput, matrix);
	*traceOutput << ret;
}


void
ParserCob::ReadMainChunk()
{
	*traceOutput << "ReadMainChunk" << eol;
	*traceOutput << adv;

	for (;;)
	{
		CobChunk		currentChunk;
		ParseChunkHeader(currentChunk);

		if (currentChunk.chunkType == CHUNK_END)
		{
			Match(currentChunk, CHUNK_END, 0, 1);
			break ;
		}

		if (currentChunk.chunkType == CHUNK_GROUP)
			ReadGroupChunk(currentChunk);

		else if (currentChunk.chunkType == CHUNK_UNIT)
			ReadUnitChunk(currentChunk);

		else if (currentChunk.chunkType == CHUNK_POLYGONAL_DATA)
			ReadPolyDataChunk(currentChunk);

		else if (currentChunk.chunkType == CHUNK_MATERIAL)
			ReadMaterialChunk(currentChunk);

/*		else if (currentChunk.chunkType == CHUNK_PROC_TEXTURE)
			ReadProcTextureChunk(currentChunk);
*/
		else if (currentChunk.chunkType == CHUNK_CAMERA)
			ReadCameraChunk(currentChunk);

		else if (currentChunk.chunkType == CHUNK_CALIMAGE)
			ReadCalImageChunk(currentChunk);

		else if (currentChunk.chunkType == CHUNK_LIGHT)
			ReadLightChunk(currentChunk);

		else
			SkipChunk(currentChunk);
	}

	FixUpFacesMaterial();

	*traceOutput << ret;
}

void
ParserCob::ReadGroupChunk(CobChunk& chunk)
{
	*traceOutput << "ReadGroupChunk" << eol;
	*traceOutput << adv;

	Match(chunk, CHUNK_GROUP, GROUP_MAJOR, GROUP_MINOR);

	char*				name = ParseName();

	pMatrix4			matrix1;
	pMatrix4			matrix2;
	ParseLocalAxis(matrix1);
	ParsePosition(matrix2);

	*traceOutput << '\'' << name << '\'' << eol;
	delete[] name;
	*traceOutput << ret;
}



void
ParserCob::ReadUnitChunk(CobChunk& chunk)
{
	*traceOutput << "ReadUnitChunk" << eol;
	*traceOutput << adv;
	Match(chunk, CHUNK_UNIT, UNIT_MAJOR, UNIT_MINOR);
	Assert("ParserCob::ReadUnitChunk", 2);

	int16			dummy;
	ReadInt16(dummy);

	*traceOutput << dummy << eol;
	*traceOutput << ret;
}


void
ParserCob::ReadPolyDataChunk(CobChunk& chunk)
{
	*traceOutput << "ReadPolyDataChunk" << eol;
	*traceOutput << adv;
	Match(chunk, CHUNK_POLYGONAL_DATA, POLYGONAL_DATA_MAJOR, POLYGONAL_DATA_MINOR);

	char*				name = ParseName();
	*traceOutput << '\'' << name << '\'' << eol;

	pObject*			object = NewObject(name);
	delete[] name;

	pMatrix4			matrix1;
	pMatrix4			matrix2;
	ParseLocalAxis(matrix1);
	ParsePosition(matrix2);
	ReadVerticesChunk(object);
	ReadMapVerticesChunk(object);
	ReadFacesChunk(object);

	pMatrix4			currMatrix = matrix2;
	object->Transform(currMatrix);

	*traceOutput << ret;
}


void
ParserCob::ReadVerticesChunk(pObject* object)
{
	*traceOutput << "ReadVerticesChunk" << eol;
	*traceOutput << adv;

	int32			nbVertices;
	ReadInt32(nbVertices);
	*traceOutput << nbVertices << " vertices" << eol << adv;

	object->vertices->SetAlloc(nbVertices, 0);

	for (int32 i = 0; i < nbVertices; i++)
	{
		pPoint3			pt;
		ParsePoint(pt);

		object->vertices->AddItem(pt);
//		Output(*traceOutput, pt);
//		*traceOutput << eol;
	}

	*traceOutput << ret << ret;
}


void
ParserCob::ReadMapVerticesChunk(pObject* /*object*/)
{
	*traceOutput << "ReadMapVerticesChunk" << eol;
	*traceOutput << adv;

	int32			nbMapVertices;
	ReadInt32(nbMapVertices);
	*traceOutput << nbMapVertices << " map vertices" << eol << adv;

//	object->mapVertices.SetAlloc(nbMapVertices);

	for (int32 i = 0; i < nbMapVertices; i++)
	{
		float			u, v;
		ReadFloat(u);
		ReadFloat(v);

//		object->mapVertices.AddItem(pTexPoint(u, v, 0.0));
//		*traceOutput << u << ' ' << v << eol;
	}

	*traceOutput << ret << ret;
}


void
ParserCob::ReadFacesChunk(pObject* object)
{
	*traceOutput << "ReadFacesChunk" << eol;
	*traceOutput << adv;

	int32			nbFaces;
	ReadInt32(nbFaces);
	*traceOutput << nbFaces << " faces" << eol << adv;

	object->faces.SetAlloc(nbFaces, 0);

	for (int32 i = 0; i < nbFaces; i++)
	{
		uint8			flag;
		int16			nbVert;
		int16			matIndex = -1;
		ReadInt8(flag);
		ReadInt16(nbVert);
		PierrotArray<int32 >	indexes(nbVert, 0);
//		PierrotArray<int32 >	mapIndexes(nbVert, 0);

		if (flag & FACE_HOLE)			// hole
		{
//			*traceOutput << "hole";
		}
		else							// face
		{
			ReadInt16(matIndex);
//			*traceOutput << "mat " << matIndex;
		}
//		*traceOutput << ' ' << nbVert << ':';
		for (int16 j = 0; j < nbVert; j++)
		{
			int32		vertIndex;
			int32		mapIndex;
			ReadInt32(vertIndex);
			ReadInt32(mapIndex);

			indexes.AddItem(vertIndex);
//			mapIndexes.AddItem(mapIndex);
//			*traceOutput << " (" << vertIndex << ' ' << mapIndex << ')';
		}

		if (!(flag & FACE_HOLE))
			object->faces.AddItem(new pFace((pMaterial* ) matIndex, nbVert, indexes.Items()));

//		*traceOutput << eol;
	}

	*traceOutput << ret << ret;
}


void
ParserCob::ReadMaterialChunk(CobChunk& chunk)
{
	*traceOutput << "ReadMaterialChunk" << eol;
	*traceOutput << adv;
	Match(chunk, CHUNK_MATERIAL, MATERIAL_MAJOR, MATERIAL_MINOR);

	int32			currLen = ReadMaterial();
	if (currLen < chunk.chunkLen)
	{
		int16			id;
		ReadBytes((char* )&id, 2);
		if (id == 'e:')
		{
			currLen += ReadEnvMapData()+2;
			if (currLen < chunk.chunkLen)
				ReadBytes((char* )&id, 2);
		}
		if (id == 't:')
		{
			currLen += ReadTextureData()+2;
			if (currLen < chunk.chunkLen)
				ReadBytes((char* )&id, 2);
		}
		if (id == 'b:')
		{
			currLen += ReadBumpMapData()+2;
			if (currLen < chunk.chunkLen)
				ReadBytes((char* )&id, 2);
		}
	}

	*traceOutput << ret;
}


int32
ParserCob::ReadMaterial()
{
	*traceOutput << "ReadMaterial" << eol;
	*traceOutput << adv;

	char			tmp[32];
	int16			matIndex;
	char			shadType;
	char			facetType;
	uint8			autofacetAngle;
	float			ambient;
	float			specular;
	float			highlight;
	float			refractionIndex;
	ReadInt16(matIndex);
	ReadBytes((char* )&shadType, 1);
	ReadBytes((char* )&facetType, 1);
	ReadInt8(autofacetAngle);

//	sprintf(tmp, "material %d", matIndex);
	sprintf(tmp, "material %d", scene->materials.CountItems());
	pMaterial*			mat = NewMaterial(tmp);

	ParseColor(mat->diffuse);
	ReadFloat(mat->opacity);
	ReadFloat(ambient);
	mat->ambient.Set(ambient, ambient, ambient);
	ReadFloat(specular);
//	mat->specular.Set(specular, specular, specular);
	ReadFloat(highlight);
//	mat->shininess = 128.0 * highlight;
	ReadFloat(refractionIndex);

	*traceOutput << "matIndex=" << matIndex << ", '" << shadType << "', '"
				<< facetType << "', angle=" << int32(autofacetAngle) << eol;
	*traceOutput << "color ";
	Output(*traceOutput, mat->diffuse);
	*traceOutput << eol << "opacity=" << mat->opacity << ", ambient=" << ambient
				<< ", specular=" << specular << eol
				<< "highlight=" << highlight << ", refractionIndex="
				<< refractionIndex << eol;

	*traceOutput << ret;

	return 37;
}


int32
ParserCob::ReadEnvMapData()
{
	*traceOutput << "ReadEnvMapData" << eol;
	*traceOutput << adv;

	int8			flag;
	ReadInt8(flag);

	char*			envmapFile = ParseString();
//	*traceOutput << "flag=" << int32(flag) << ", file=" << envmapFile << eol;
	int32			stringLen = strlen(envmapFile)+2;
	delete[] envmapFile;

	*traceOutput << ret;

	return 1+stringLen;
}


int32
ParserCob::ReadTextureData()
{
	*traceOutput << "ReadTextureData" << eol;
	*traceOutput << adv;

	int8			flag;
	ReadInt8(flag);
	char*			textmapFile = ParseString();
	int32			stringLen = strlen(textmapFile)+2;
	float			u, v;
	float			du, dv;
	ReadFloat(u);
	ReadFloat(v);
	ReadFloat(du);
	ReadFloat(dv);
/*
	*traceOutput << "flag=" << int32(flag) << ", file=" << textmapFile << ", " << u
				<< ' ' << v << ' ' << du << ' ' << dv << eol;
*/
	delete[] textmapFile;

	*traceOutput << ret;

	return 1+stringLen+16;
}


int32
ParserCob::ReadBumpMapData()
{
	*traceOutput << "ReadBumpData" << eol;
	*traceOutput << adv;

	int8			flag;
	ReadInt8(flag);
	char*			bumpmapFile = ParseString();
	int32			stringLen = strlen(bumpmapFile)+2;
	float			u, v;
	float			du, dv;
	float			amplitude;
	ReadFloat(u);
	ReadFloat(v);
	ReadFloat(du);
	ReadFloat(dv);
	ReadFloat(amplitude);
/*
	*traceOutput << " flag=" << int32(flag) << ", file=" << bumpmapFile << eol;
	*traceOutput << u << ' ' << v << ' ' << du << ' ' << dv << ' ' << amplitude << eol;
*/
	delete[] bumpmapFile;

	*traceOutput << ret;

	return 1+stringLen+20;
}


void
ParserCob::ReadProcTextureChunk(CobChunk& chunk)
{
	*traceOutput << "ReadProcTextureChunk" << eol;
	*traceOutput << adv;
	Match(chunk, CHUNK_PROC_TEXTURE, PROC_TEXTURE_MAJOR, PROC_TEXTURE_MINOR);

	ReadMaterial();

	int16			id;
	ReadBytes((char* )&id, 2);
	if (id == 'g:')
		ReadGranitData();
	else if (id == 'm:')
		ReadMarbleData();
	else if (id == 'w:')
		ReadWoodData();

	// bump data
	off_t			pos = lexer->GetPosition();
	ReadBytes((char* )&id, 2);
	if (id == 'b:')
		ReadBumpMapData();
	else
	{
		lexer->GetInput()->SetPosition(pos);
	}

	*traceOutput << ret;
}


void
ParserCob::ReadGranitData()
{
	*traceOutput << "ReadGranitData" << eol;
	*traceOutput << adv;

	int8			reserved;
	ReadInt8(reserved);

	pSpectra4		colors[4];
	float			amounts[4];
	float			sharpness;
	float			r, s, t;
	int16			seed;
	ParseColor(colors[0]);
	ParseColor(colors[1]);
	ParseColor(colors[2]);
	ParseColor(colors[3]);
	ReadFloat(amounts[0]);
	ReadFloat(amounts[1]);
	ReadFloat(amounts[2]);
	ReadFloat(amounts[3]);
	ReadFloat(sharpness);
	ReadFloat(r);
	ReadFloat(s);
	ReadFloat(t);
	ReadInt16(seed);
/*
	for (int16 i = 0; i < 4; i++)
	{
		Output(*traceOutput, colors[i]);
		*traceOutput << ' ' << amounts[i] << eol;
	}

	*traceOutput << "sharpness=" << sharpness << ", r=" << r << ", s=" << s
				<< ", t=" << t << ", seed=" << seed << eol;
*/
	*traceOutput << ret;
}


void
ParserCob::ReadMarbleData()
{
	*traceOutput << "ReadMarbleData" << eol;
	*traceOutput << adv;

	int8			reserved;
	ReadInt8(reserved);

	pSpectra4		stone;
	pSpectra4		vein;
	float			sharpness;
	pVector3		rst;
	int16			veinTurbulence;
	int16			rstGrainFlag;
	int16			seed;
	ParseColor(stone);
	ParseColor(vein);
	ReadFloat(sharpness);
	ParseVector(rst);
	ReadInt16(veinTurbulence);
	ReadInt16(rstGrainFlag);
	ReadInt16(seed);
/*
	*traceOutput << "stone=";
	Output(*traceOutput, stone);
	*traceOutput << ", vein=";
	Output(*traceOutput, vein);
	*traceOutput << eol;
	*traceOutput << "rst=";
	Output(*traceOutput, rst);
	*traceOutput << eol;
	*traceOutput << "veinTurbulance=" << veinTurbulence << ", rstGrainFlag="
				<< rstGrainFlag << ", seed=" << seed << eol;
*/
	*traceOutput << ret;
}


void
ParserCob::ReadWoodData()
{
	*traceOutput << "ReadWoodData" << eol;
	*traceOutput << adv;

	int8			reserved;
	ReadInt8(reserved);

	pSpectra4		spring;
	pSpectra4		summer;
	float			springToSummerRatio;
	float			ringDensity, ringWidth, ringShape;
	pPoint3			rstRingCenter;
	pVector3		rstScale;
	int16			rstGrainFlag;
	int16			seed;

	ParseColor(spring);
	ParseColor(summer);
	ReadFloat(springToSummerRatio);
	ReadFloat(ringDensity);
	ReadFloat(ringWidth);
	ReadFloat(ringShape);
	ParsePoint(rstRingCenter);
	ParseVector(rstScale);
	ReadInt16(rstGrainFlag);
	ReadInt16(seed);
/*
	*traceOutput << "spring=";
	Output(*traceOutput, spring);
	*traceOutput << ", summer=";
	Output(*traceOutput, summer);
	*traceOutput << eol;
	*traceOutput << "ring density=" << ringDensity << ", ringWidth=" << ringWidth
				<< ", ringShape=" << ringShape << eol;
	*traceOutput << "rstRingCenter=";
	Output(*traceOutput, rstRingCenter);
	*traceOutput << "rstScale=";
	Output(*traceOutput, rstScale);
	*traceOutput << "rstGrainFlag=" << rstGrainFlag << ", seed=" << seed << eol;
*/
	*traceOutput << ret;
}


void
ParserCob::ReadCameraChunk(CobChunk& chunk)
{
	*traceOutput << "ReadCameraChunk" << eol;
	*traceOutput << adv;
	Match(chunk, CHUNK_CAMERA, CAMERA_MAJOR, CAMERA_MINOR);

	char*			name;
	pMatrix4		matrix1;
	pMatrix4		matrix2;

	name = ParseName();
	ParseLocalAxis(matrix1);
	ParsePosition(matrix2);

	*traceOutput << "camera '" << name << '\'' << eol;
	*traceOutput << ret;

	delete[] name;
}


void
ParserCob::ReadCalImageChunk(CobChunk& chunk)
{
	*traceOutput << "ReadCalImageChunk" << eol;
	*traceOutput << adv;
	Match(chunk, CHUNK_CALIMAGE, CALIMAGE_MAJOR, CALIMAGE_MINOR);

	int32			width, height;
	ReadInt32(width);
	ReadInt32(height);

	for (int32 i = 0; i < height; i++)
	{
		pColor4			currentPixel;
		for (int32 j = 0; j < width; j++)
		{
			ReadInt8(currentPixel.r);
			ReadInt8(currentPixel.g);
			ReadInt8(currentPixel.b);
			ReadInt8(currentPixel.a);
		}
	}

	*traceOutput << "calimage " << width << 'x' << height << eol;
	*traceOutput << ret;
}


void
ParserCob::ReadLightChunk(CobChunk& chunk)
{
	*traceOutput << "ReadLightChunk" << eol;
	*traceOutput << adv;
	Match(chunk, CHUNK_LIGHT, LIGHT_MAJOR, LIGHT_MINOR);

	char*			name;
	pMatrix4		matrix1;
	pMatrix4		matrix2;
	int16			lightType;
	int16			shadowFlag;
	int16			shadowType = -1;
	int16			shadowMapResolution = -1;
	int16			shadowMapSharpness = -1;
	pSpectra		color;
	float			illuminAngle;
	float			hotSpotAngle;
	float			fallOff;

	name = ParseName();
	ParseLocalAxis(matrix1);
	ParsePosition(matrix2);
	ReadInt16(lightType);
	ReadInt16(shadowFlag);
	if (shadowFlag)
		ReadInt16(shadowType);
	if ((shadowType == 2) || (shadowType == 4))
		ReadInt16(shadowMapResolution);
	if (chunk.minorVersion > 3)
		ReadInt16(shadowMapSharpness);
	ParseColor(color);			
	ReadFloat(illuminAngle);
	ReadFloat(hotSpotAngle);
	ReadFloat(fallOff);
/*
	*traceOutput << "light '" << name << '\'' << eol;
	*traceOutput << adv;
		*traceOutput << "type=" << lightType;
		if (shadowType >= 0)
			*traceOutput << ", shadowType=" << shadowType;
		if (shadowMapResolution >= 0)
			*traceOutput << ", shadowMapResolution=" << shadowMapResolution;
		if (shadowMapSharpness >= 0)
			*traceOutput << ", shadowMapSharpness=" << shadowMapSharpness;
		*traceOutput << eol << "color ";
		Output(*traceOutput, color);
		*traceOutput << "illuminAngle=" << illuminAngle << eol
					<< ", hotSpotAngle=" << hotSpotAngle
					<< ", fallOff=" << fallOff << eol;
	*traceOutput << ret << ret;
*/

	delete[] name;
}


void
ParserCob::FixUpFacesMaterial()
{
	for (int32 i = 0; i < scene->objects.CountItems(); i++)
	{
		pObject*		object = scene->objects[i];
		for (int32 j = 0; j < object->faces.CountItems(); j++)
		{
			int32			matIndex = (int32 )object->faces[j]->material;
/*			char			buf[32];
			sprintf(buf, "material %d", matIndex);
			matIndex = scene->materials.Search(buf);
			if (matIndex < 0)
				matIndex = scene->materials.Search("default material");

			object->faces[j]->material = scene->materials[matIndex];
*/
			object->faces[j]->material = scene->materials[matIndex+1];
		}
	}
}



//-------------------------------------------------------------
//
//-------------------------------------------------------------

CobMismatchChunk::CobMismatchChunk(const char* location,
									int32 etype, int32 emajvers, int32 eminvers,
									CobChunk& actualChunk)
: LAPparserException(COB_MISMATCH_CHUNK, location)
{
	expectedType = etype;
	expectedMajorVersion = emajvers;
	expectedMinorVersion = eminvers;
	actualType = actualChunk.chunkType;
	actualMajorVersion = actualChunk.majorVersion;
	actualMinorVersion = actualChunk.minorVersion;
}


CobMismatchChunk::CobMismatchChunk(CobMismatchChunk& e)
: LAPparserException(e.what, e.where)
{
	expectedType = e.expectedType;
	expectedMajorVersion = e.expectedMajorVersion;
	expectedMinorVersion = e.expectedMinorVersion;
	actualType = e.actualType;
	actualMajorVersion = e.actualMajorVersion;
	actualMinorVersion = e.actualMinorVersion;
}


CobMismatchChunk::~CobMismatchChunk()
{
}


void
CobMismatchChunk::Output(LAPtextBufferedOutput& out)
{
	char*			str;
	_Output(out);

	str = (char* )&expectedType;
	out << ": mismatch chunk, expected ('" << str[0] << str[1] << str[2] << str[3]
		<< "', " << expectedMajorVersion << ", " << expectedMinorVersion << ") got ";
	str = (char* )&actualType;
	out << str[0] << str[1] << str[2] << str[3] << "', " << actualMajorVersion
		<< ", " << actualMinorVersion << ')' << eol << flush;
}

