//
//
//	DXF objects parser
//
//											(C) 1998 JoOl


#include "ParserDxf.h"
#include "PierrotParser.h"



#if __POWERPC__
#pragma export on
#endif
char			handlerName[]		= "AutoCAD DXF r14 (.dxf)";
char			handlerVersion[]	= "1.0.0";
char			handlerAuthor[]		= "Alain GAUTHEROT";
char			handlerEmail[]		= "gauthero@club-internet.fr";
#if __POWERPC__
#pragma export reset
#endif



ParserDxf*
instantiate_parser(int32 bs)
{
	return new ParserDxf(bs);
}



ParserDxf::ParserDxf(int32 blockSize)
: LAPtextParser(), layersColor(16, 8)//, layersGruik(16, 8)
{
	SetLexer(new LexerDxf(max_c(1024, blockSize), new LAPlexemeBuffer(128, true)));

	SetTo(new LAPtracer(128, LAP_BUFFER_OUTPUT_ON_OVERFLOW, LAP_BUFFER_SCREEN_OUTPUT),
			new LAPtextBufferedOutput(128, LAP_BUFFER_OUTPUT_ON_OVERFLOW, LAP_BUFFER_NO_OUTPUT/*LAP_BUFFER_SCREEN_OUTPUT*/),
			new LAPdisplay(),
			LAP_OWN_ERROR_OUTPUT | LAP_OWN_TRACE_OUTPUT | LAP_OWN_DISPLAY);
}


ParserDxf::~ParserDxf()
{
	if (data)
		delete (pScene* )data;
}


void
ParserDxf::NextToken()
{
	LAPtextParser::NextToken();
	groupCode = ((LexerDxf* )lexer)->GetGroupCode();
}


status_t
ParserDxf::DoAbort()
{
	PierrotParser::Reset();
	return LAP_NO_ERROR;
}


status_t
ParserDxf::_Identify()
{
	while (Peek(999))
		Match(999);

	if (!Peek(0, TOKEN_SECTION))
		return LAP_BAD_INPUT_TYPE;
	return LAP_NO_ERROR;
}



// A few handy routines
void
ParserDxf::Match(int32 gcode)
{
	NextToken();
	if (gcode == groupCode)
		return ;

	// generate error
	GenerateMismatchGroupCodes("ParserDxf::Match", gcode, groupCode);
}


void
ParserDxf::Match(int32 gcode, LAPtoken expectedToken)
{
	NextToken();
	if (gcode != groupCode)
		GenerateMismatchGroupCodes("ParserDxf::Match", gcode, groupCode);

	if (currentToken != expectedToken)
		GenerateMismatchToken("ParserDxf::Match", expectedToken, NULL,
								currentToken, NULL);
//	PrintGroup(groupCode, currentToken);
}


bool
ParserDxf::Peek(int32 gcode)
{
	if (!peekFlag)
	{
		peekFlag = true;
		currentToken = lexer->NextToken();
		groupCode = ((LexerDxf* )lexer)->GetGroupCode();
	}
	return (gcode == groupCode);
}


bool
ParserDxf::Peek(int32 gcode, LAPtoken expectedToken)
{
	if (!peekFlag)
	{
		peekFlag = true;
		currentToken = lexer->NextToken();
		groupCode = ((LexerDxf* )lexer)->GetGroupCode();
	}
	return (gcode == groupCode) && (currentToken == expectedToken);
}


void
ParserDxf::GenerateMismatchGroupCodes(const char* Location,
									int32 expectedGcode, int32 actualGcode)
{
	char		Buf1[20];
	sprintf(Buf1, "<group code %d>", expectedGcode);
	char		Buf2[20];
	sprintf(Buf2, "<group code %d>", actualGcode);
	GenerateMismatchToken(Location, LAP_TOKEN_NUMBER, Buf1,
							LAP_TOKEN_NUMBER, Buf2);
}


void
ParserDxf::MatchGroup(int32 gcode, int32& out)
{
	Match(gcode, LAP_TOKEN_NUMBER);
	if (lexer->GetLiteralType() == LAP_INTEGER_LITERAL)
	{
		out = lexer->GetInt32Literal();
		return ;
	}

	GenerateMismatchLiteral("ParserDxf::MatchGroup", LAP_INTEGER_LITERAL,
							currentToken);
}


void
ParserDxf::MatchGroup(int32 gcode, float& out)
{
	Match(gcode, LAP_TOKEN_NUMBER);
	if (lexer->GetLiteralType() == LAP_FLOAT_LITERAL)
	{
		out = lexer->GetFloatLiteral();
		return ;
	}

	GenerateMismatchLiteral("ParserDxf::MatchGroup", LAP_FLOAT_LITERAL,
							currentToken);
}


void
ParserDxf::MatchGroupCast(int32 gcode, float& out)
{
	Match(gcode, LAP_TOKEN_NUMBER);
	if (lexer->GetLiteralType() == LAP_FLOAT_LITERAL)
		out = lexer->GetFloatLiteral();
	else
		out = float(lexer->GetInt32Literal());
}


void
ParserDxf::MatchGroup(int32 gcode, char*& out)
{
	Match(gcode, LAP_TOKEN_IDENTIFIER);
	out = lexer->DetachStringLiteral();
}


void
ParserDxf::MatchGroupCast(int32 gcode, char*& out)
{
	if (Peek(gcode, LAP_TOKEN_IDENTIFIER))
	{
		Match(gcode, LAP_TOKEN_IDENTIFIER);
		out = lexer->DetachStringLiteral();
		return ;
	}

	char			Conv[16];
	if (lexer->GetLiteralType() == LAP_INTEGER_LITERAL)
		sprintf(Conv, "%d", lexer->GetInt32Literal());
	else
		sprintf(Conv, "%f", lexer->GetFloatLiteral());
	out = strdup(Conv);
	NextToken();
}


void
ParserDxf::PrintGroup(int32 gcode, LAPtoken token)
{
	*traceOutput << '[' << gcode << ", ";
	if (token == LAP_TOKEN_NUMBER)
	{
		if (lexer->GetLiteralType() == LAP_CHARACTER_LITERAL)
			*traceOutput << lexer->GetCharLiteral();
		else if (lexer->GetLiteralType() == LAP_INTEGER_LITERAL)
			*traceOutput << lexer->GetInt32Literal();
		else if (lexer->GetLiteralType() == LAP_FLOAT_LITERAL)
			*traceOutput << lexer->GetFloatLiteral();
		else if (lexer->GetLiteralType() == LAP_STRING_LITERAL)
			*traceOutput << lexer->GetStringLiteral();
		else
			*traceOutput << "literal" << int32(lexer->GetLiteralType());
	}
	else if (token == LAP_TOKEN_IDENTIFIER)
		*traceOutput << '\'' << lexer->GetStringLiteral() << '\'';
	else
		*traceOutput << (char* )lexer->GetLexeme(token);
	*traceOutput << ']' << eol;
	*traceOutput << flush;
}


void
ParserDxf::ParseTo(int32 gcode)
{
	while (!Peek(0, TOKEN_EOF) && !Peek(gcode))
	{
		NextToken();
	}
	if (groupCode == gcode)
		return ;
	GenerateMismatchGroupCodes("ParserDxf::ParseTo", gcode, groupCode);
}


void
ParserDxf::ParseTo(int32 gcode, LAPtoken expectedToken)
{
	while (!Peek(0, TOKEN_EOF) && !Peek(gcode, expectedToken))
	{
		NextToken();
	}
	if ((groupCode == gcode) && (currentToken == expectedToken))
		return ;

	if (groupCode != gcode)
		GenerateMismatchGroupCodes("ParserDxf::ParseTo", gcode, groupCode);
	if (currentToken != expectedToken)
		GenerateMismatchToken("ParserDxf::ParseTo", expectedToken, NULL,
								currentToken, NULL);
}


void
ParserDxf::ParsePoint(int32 gcode, pPoint3& out)
{
	MatchGroupCast(gcode, out.x);
	MatchGroupCast(gcode+10, out.y);
	MatchGroupCast(gcode+20, out.z);
}


void
ParserDxf::ParseVector(int32 gcode, pVector3& out)
{
	MatchGroupCast(gcode, out.x);
	MatchGroupCast(gcode+10, out.y);
	MatchGroupCast(gcode+20, out.z);
}


int32
ParserDxf::FindOrCreateLayer(char* layerName)
{
	int32			layerIndex = scene->objects.Search(layerName);
	if (layerIndex < 0)
	{
		layerIndex = scene->objects.CountItems();
		NewObject(layerName, POLY_VERTICES_ALLOC, POLY_VERTICES_BLOCK,
							POLY_FACES_ALLOC, POLY_FACES_BLOCK);
		layersColor.AddItem(DEFAULT_COLOR);
//		layersGruik.AddItem(0);
	}

	return layerIndex;
}


int32
ParserDxf::NewVertex(char* layerName, pPoint3& pt)
{
	int32			layerIndex = FindOrCreateLayer(layerName);
	return PierrotParser::NewVertex(scene->objects[layerIndex], pt);
/*	pObject*		obj = scene->objects[layerIndex];
	int32			vertexIndex = obj->vertices->CountItems();
	obj->vertices->AddItem(pt);
	return vertexIndex;
*/
}


void
ParserDxf::NewFace(char* layerName, pMaterial* material,
					pPoint3& p0, pPoint3& p1, pPoint3& p2)
{
	int32			layerIndex = FindOrCreateLayer(layerName);
	pObject*		object = scene->objects[layerIndex];

	if (!material)
	{
		int32			color = int32(layersColor[layerIndex]);
		int32			matIndex = SearchMaterial(palette[color]);
		material = scene->materials[matIndex];
	}

	int32			indexes[3];
	indexes[0] = PierrotParser::NewVertex(object, p0);
	indexes[1] = PierrotParser::NewVertex(object, p1);
	indexes[2] = PierrotParser::NewVertex(object, p2);
	object->faces.AddItem(new pFace(material, indexes[0], indexes[1], indexes[2]));
}


void
ParserDxf::NewFace(char* layerName, pMaterial* material,
					pPoint3& p0, pPoint3& p1, pPoint3& p2, pPoint3& p3)
{
	int32			layerIndex = FindOrCreateLayer(layerName);
	pObject*		object = scene->objects[layerIndex];

	if (!material)
	{
		int32			color = int32(layersColor[layerIndex]);
		int32			matIndex = SearchMaterial(palette[color]);
		material = scene->materials[matIndex];
	}

	int32			indexes[4];
	indexes[0] = PierrotParser::NewVertex(object, p0);
	indexes[1] = PierrotParser::NewVertex(object, p1);
	indexes[2] = PierrotParser::NewVertex(object, p2);
	indexes[3] = PierrotParser::NewVertex(object, p3);

	object->faces.AddItem(new pFace(material, indexes[0], indexes[1], indexes[2], indexes[3]));
}


int32
ParserDxf::NewDot(char* layerName, pPoint3& pt)
{
	int32			layerIndex = FindOrCreateLayer(layerName);
	pObject*		object = scene->objects[layerIndex];

	int32			vertexIndex = NewDotVertex(layerName, pt);
	int32			dotIndex = object->dots.CountItems();
	object->dots.AddItem(vertexIndex);
	return dotIndex;
}


int32
ParserDxf::NewDotVertex(char* layerName, pPoint3& pt)
{
	int32			layerIndex = FindOrCreateLayer(layerName);
	pObject*		object = scene->objects[layerIndex];
	if (!object->dlVertices.IsAllocated())
		object->dlVertices.SetAlloc(PONCT_VERTICES_ALLOC, PONCT_VERTICES_BLOCK);
	if (!object->dots.IsAllocated())
		object->dots.SetAlloc(PONCT_DOTS_ALLOC, PONCT_DOTS_BLOCK);

	int32			vertexIndex = object->dlVertices.Search(pt);
	if (vertexIndex < 0)
	{
		vertexIndex = object->dlVertices.CountItems();
		object->dlVertices.AddItem(pt);
	}
	return vertexIndex;
}


int32
ParserDxf::NewLine(char* layerName, pPoint3& start, pPoint3& end)
{
	int32			layerIndex = FindOrCreateLayer(layerName);
	pObject*		object = scene->objects[layerIndex];
	if (!object->dlVertices.IsAllocated())
		object->dlVertices.SetAlloc(PONCT_VERTICES_ALLOC, PONCT_VERTICES_BLOCK);
	int32			index1 = object->dlVertices.Search(start);
	if (index1 < 0)
	{
		index1 = object->dlVertices.CountItems();
		object->dlVertices.AddItem(start);
	}
	int32			index2 = object->dlVertices.Search(end);
	if (index2 < 0)
	{
		index2 = object->dlVertices.CountItems();
		object->dlVertices.AddItem(end);
	}
	if (!object->lines.IsAllocated())
		object->lines.SetAlloc(PONCT_LINES_ALLOC, PONCT_LINBS_BLOCK);
	int32			lineIndex = object->lines.CountItems();
	object->lines.AddItem(pSegment(index1, index2));
	return lineIndex;
}


//--------------------------------------------------------
// main routines
//--------------------------------------------------------
status_t
ParserDxf::_Parse()
{
	PierrotParser::Reset();

	scene = new pScene((char* )GetLexer()->GetInputName(),
						2, 2,					// lights
						16, 16,					// materials
						16, 16);				// objects

	InitPalette();

	// skip heading comments
	while (Peek(999))
		Match(999);

	while (!Peek(0, TOKEN_EOF))
	{
		Match(0, TOKEN_SECTION);

		if (Peek(2, TOKEN_HEADER))
		{
			ParseHeader();
		}
		else if (Peek(2, TOKEN_TABLES))
		{
			ParseTables();
		}
		else if (Peek(2, TOKEN_ENTITIES))
		{
			ParseEntities();
		}
		else
		{
			// Skip to next section
			ParseTo(0, TOKEN_ENDSEC);
			Match(0, TOKEN_ENDSEC);
		}
	}

	pSphere				normSphere(pPoint3(0.0, 0.0, 0.0), 1.0);
	scene->Preprocess(&normSphere, appliInterface, 0.0, 1.0);

	scene->ambient.Set(0.5, 0.5, 0.5);
	if (scene->lights.CountItems() == 0)
	{
		pLight*			l1 = NewDirectionalLight("directional light 1", pVector3(-0.5, 0.5, 0.8));
		l1->ambient.Set(0.2, 0.2, 0.2);
		l1->diffuse.Set(1.0, 1.0, 1.0);
		l1->specular.Set(1.0, 1.0, 1.0);
	}

	Output(*traceOutput, scene, PRINT_LIGHT_ALL | PRINT_MATERIAL_ALL |
			PRINT_OBJECT_BOUNDS | PRINT_OBJECT_MATRIX | PRINT_OBJECT_POINTS);

	data = scene;
	scene = NULL;
	return LAP_NO_ERROR;
}



//-------------------------------------------------
// Header related routines
//-------------------------------------------------

void
ParserDxf::ParseHeader()
{
	Match(2, TOKEN_HEADER);

	while (!Peek(0, TOKEN_EOF) && !Peek(0, TOKEN_ENDSEC))
	{
		if (Peek(9, TOKEN_S_ACADVER))
			ParseVersion();
		else
			NextToken();
	}
	Match(0, TOKEN_ENDSEC);
}


void
ParserDxf::ParseVersion()
{
	char*				versionStr;
	Match(9, TOKEN_S_ACADVER);
	MatchGroup(1, versionStr);
/*
	*traceOutput << "version " << versionStr;
	if (!strcmp(versionStr, VERSION_R14))
	{ *traceOutput << ": r14"; }
	if (!strcmp(versionStr, VERSION_R13))
	{ *traceOutput << ": r13"; }
	if (!strcmp(versionStr, VERSION_R11_R12))
	{ *traceOutput << ": r11/r12"; }
	else if (!strcmp(versionStr, VERSION_R10))
	{ *traceOutput << ": r10"; }
	*traceOutput << eol;
*/
	delete[] versionStr;
}



//-------------------------------------------------
// Tables related routines
//-------------------------------------------------

void
ParserDxf::ParseTables()
{
	Match(2, TOKEN_TABLES);

	while (!Peek(0, TOKEN_EOF) && !Peek(0, TOKEN_ENDSEC))
	{
		Lock();

		if (Peek(0, TOKEN_TABLE))
			ParseTable();
		else
			NextToken();

		Unlock();
	}
	Match(0, TOKEN_ENDSEC);
}


void
ParserDxf::ParseTable()
{
	int32			tableType;
	int32			flags = 0;

	Match(0, TOKEN_TABLE);

	Match(2);
	tableType = currentToken;
	MatchGroup(70, flags);

	if (tableType == TOKEN_APPID)
		ParseAppId();
	else if (tableType == TOKEN_LAYER)
		ParseLayers();
	else if (tableType == TOKEN_VIEW)
		ParseView();
	else
	{
		ParseTo(0, TOKEN_ENDTAB);
		Match(0, TOKEN_ENDTAB);
	}
}


void
ParserDxf::ParseAppId()
{
	char*			appName;
	int32			flags = 0;

	// Tables can be empty!?!
	if (Peek(0, TOKEN_ENDTAB))
	{
		Match(0, TOKEN_ENDTAB);
		return ;
	}

	Match(0, TOKEN_APPID);
	MatchGroup(2, appName);
	MatchGroup(70, flags);

//	*traceOutput << "app '" << appName << "', flags=" << uint32(flags) << eol;
	delete[] appName;
}


void
ParserDxf::ParseLayers()
{
	while (!Peek(0, TOKEN_EOF) && !Peek(0, TOKEN_ENDTAB))
	{
		Lock();

		if (Peek(0, TOKEN_LAYER))
			ParseLayer();
		else
			NextToken();

		Unlock();
	}
	Match(0, TOKEN_ENDTAB);

	// test layers
/*	*traceOutput << "layers: " << eol << adv;
	for (int32 i = 0; i < scene->objects.CountItems(); i++)
	{
		*traceOutput << scene->objects[i]->name << ", color " << layersColor[i] << eol;
	}
	*traceOutput << ret;
*/
}


void
ParserDxf::ParseLayer()
{
	int32			layerNameInt;
	char*			layerName = NULL;
	int32			flags = 0;
	int32			color;
	char*			layerTypeName = NULL;

	Match(0, TOKEN_LAYER);

	// Names can be integers!!!
	Match(2);
	if (currentToken == LAP_TOKEN_NUMBER)
		layerNameInt = lexer->GetInt32Literal();
	else
		layerName = lexer->DetachStringLiteral();

	MatchGroup(70, flags);
	MatchGroup(62, color);
	MatchGroup(6, layerTypeName);

	if (!layerName)
	{
		char			tmp[32];
		sprintf(tmp, "%d", layerNameInt);
		NewObject(tmp, POLY_VERTICES_ALLOC, POLY_VERTICES_BLOCK,
						POLY_FACES_ALLOC, POLY_FACES_BLOCK);
	}
	else
		NewObject(layerName, POLY_VERTICES_ALLOC, POLY_VERTICES_BLOCK,
							POLY_FACES_ALLOC, POLY_FACES_BLOCK);
	layersColor.AddItem(color);

	if (layerName)
		delete[] layerName;
	if (layerTypeName)
		delete[] layerTypeName;
}


void
ParserDxf::ParseView()
{
	char*			viewName;
	int32			flags = 0;
	float			screenWidth, screenHeight;
	float			centerX, centerY;
	pVector3		viewDirection;
	pPoint3			target;
	float			lensLength;
	float			f, b;				// Front & Back clipping plane distance
	float			twistAngle;
	int32			viewMode;

	// Tables can be empty!?!
	if (Peek(0, TOKEN_ENDTAB))
	{
		Match(0, TOKEN_ENDTAB);
		return ;
	}

	Match(0, TOKEN_VIEW);
	while (!Peek(0))
	{
		if (Peek(2))
			MatchGroup(2, viewName);
		else if (Peek(70))
			MatchGroup(70, flags);
		else if (Peek(40))
			MatchGroup(40, screenWidth);
		else if (Peek(41))
			MatchGroup(41, screenHeight);
		else if (Peek(10))
			MatchGroup(10, centerX);
		else if (Peek(20))
			MatchGroup(20, centerY);
		else if (Peek(11))
			ParseVector(11, viewDirection);
		else if (Peek(12))
			ParsePoint(12, target);
		else if (Peek(42))
			MatchGroup(42, lensLength);
		else if (Peek(43))
			MatchGroup(43, f);
		else if (Peek(44))
			MatchGroup(44, b);
		else if (Peek(50))
			MatchGroup(50, twistAngle);
		else if (Peek(71))
			MatchGroup(71, viewMode);	// No information on bits in "view mode"
		else
			NextToken();
	}
/*
	*traceOutput << "view '" << viewName << "' flags=" << uint16(flags) << eol;
	*traceOutput << adv;
		*traceOutput << "screen=" << screenWidth << " x " << screenHeight << eol;
		*traceOutput << "center=" << centerX << " x " << centerY << eol;
		*traceOutput << "direction ";
			Output(*traceOutput, viewDirection);
		*traceOutput << eol << "to ";
			Output(*traceOutput, target);
		*traceOutput << eol;
		*traceOutput << "lens=" << lensLength << " f=" << f << " b=" << b << eol;
		*traceOutput << "twist=" << twistAngle << " view mode=" << viewMode << eol;
	*traceOutput << ret;
*/
	delete[] viewName;
}


//-------------------------------------------------
// Entities related routines
//-------------------------------------------------


void
ParserDxf::ParseEntities()
{
	Match(2, TOKEN_ENTITIES);

	while (!Peek(0, TOKEN_EOF))
	{
		Lock();

		if (Peek(0, TOKEN_LINE))
			ParseLine();
		else if (Peek(0, TOKEN_POINT))
			ParsePoint();
//		else if (Peek(0, TOKEN_VERTEX))
//			ParseVertex();
		else if (Peek(0, TOKEN_3DFACE))
			ParseFace3d();
		else if (Peek(0, TOKEN_POLYLINE))
			ParsePolyline();
		else
			NextToken();

		Unlock();
	}
}


void
ParserDxf::ParseLine()
{
	char*			layerName = NULL;
	pPoint3			startPt, endPt;

	Match(0, TOKEN_LINE);

	while (!Peek(0))
	{
		if (Peek(10))
			ParsePoint(10, startPt);
		else if (Peek(11))
			ParsePoint(11, endPt);
		else if (Peek(8))
			MatchGroupCast(8, layerName);
		else
			NextToken();
	}

	if (!layerName)
		layerName = strdup("default");
	NewLine(layerName, startPt, endPt);
/*
	*traceOutput << layerName << ": l ";
		Output(*traceOutput, startPt);
		*traceOutput << " to ";
		Output(*traceOutput, endPt);
		*traceOutput << eol;
*/
	delete[] layerName;
}


void
ParserDxf::ParsePoint()
{
	char*			layerName = NULL;
	pPoint3			pt;

	Match(0, TOKEN_POINT);

	while (!Peek(0))
	{
		if (Peek(10))
			ParsePoint(10, pt);
		else if (Peek(8))
			MatchGroupCast(8, layerName);
		else
			NextToken();
	}

	if (!layerName)
		layerName = strdup("default");
	NewDot(layerName, pt);
/*
	*traceOutput << layerName << ": p ";
		Output(*traceOutput, pt);
	*traceOutput << eol;
*/
	delete[] layerName;
}


int32
ParserDxf::ParseVertex()
{
	char*			layerName = NULL;
	pPoint3			pt;
	int32			flags = 0;

	Match(0, TOKEN_VERTEX);

	while (!Peek(0))
	{
		if (Peek(10))
			ParsePoint(10, pt);
		else if (Peek(8))
			MatchGroupCast(8, layerName);
		else if (Peek(70))
			MatchGroup(70, flags);
		else
			NextToken();
	}
	if (!layerName)
		layerName = strdup("default");
/*	*traceOutput << layerName << ": v ";
		Output(*traceOutput, pt);
	*traceOutput << eol;
*/

/*
int32			layerIndex = FindOrCreateLayer(layerName);
if (!(flags & 0x80))
	layersGruik[layerIndex]++;
*/
	int32			out = NewVertex(layerName, pt);
	delete[] layerName;
	return out;
}


int32
ParserDxf::ParseDotVertex()
{
	char*			layerName = NULL;
	pPoint3			pt;
	int32			flags = 0;

	Match(0, TOKEN_VERTEX);

	while (!Peek(0))
	{
		if (Peek(10))
			ParsePoint(10, pt);
		else if (Peek(8))
			MatchGroupCast(8, layerName);
		else if (Peek(70))
			MatchGroup(70, flags);
		else
			NextToken();
	}
	if (!layerName)
		layerName = strdup("default");

	int32			out = NewDotVertex(layerName, pt);
	delete[] layerName;
	return out;
}


void
ParserDxf::ParsePolyfaceVertex(int32& i0, int32& i1, int32& i2, int32& i3)
{
	char*			layerName = NULL;
	pPoint3			pt;
	int32			flags = 0;

	Match(0, TOKEN_VERTEX);

	i0 = i1 = i2 = i3 = 0x80000000;
	while (!Peek(0))
	{
		if (Peek(10))
			ParsePoint(10, pt);
		else if (Peek(8))
			MatchGroupCast(8, layerName);
		else if (Peek(70))
			MatchGroup(70, flags);

		else if (Peek(71))
			MatchGroup(71, i0);
		else if (Peek(72))
			MatchGroup(72, i1);
		else if (Peek(73))
			MatchGroup(73, i2);
		else if (Peek(74))
			MatchGroup(74, i3);

		else
			NextToken();
	}
	if (!layerName)
		layerName = strdup("default");

	if (flags & 0x40)
	{
		NewVertex(layerName, pt);
		i0 = -1;
	}
	delete[] layerName;
}


void
ParserDxf::ParseFace3d()
{
	char*			layerName = NULL;
	pPoint3			pt0, pt1, pt2, pt3;
	int32			flags = 0;
	int32			color = -1;

	Match(0, TOKEN_3DFACE);

	while (!Peek(0))
	{
		if (Peek(10))
			ParsePoint(10, pt0);
		else if (Peek(11))
			ParsePoint(11, pt1);
		else if (Peek(12))
			ParsePoint(12, pt2);
		else if (Peek(13))
			ParsePoint(13, pt3);
		else if (Peek(8))
			MatchGroupCast(8, layerName);
		else if (Peek(62))
			MatchGroup(62, color);
		else if (Peek(70))
			MatchGroup(70, flags);
		else
			NextToken();
	}

	pMaterial*			mat = NULL;
	if (!layerName)
	{
		layerName = strdup("default");
		if (color < 0)
			color = DEFAULT_COLOR;
		int32			matIndex = SearchMaterial(palette[color & 0xff]);
		mat = scene->materials[matIndex];
	}

	if (pt3 == pt2)
		NewFace(layerName, mat, pt0, pt1, pt2);
	else
		NewFace(layerName, mat, pt0, pt1, pt2, pt3);
/*
	*traceOutput << layerName << ": f, flags " << uint16(flags) << eol;
	*traceOutput << adv;
		Output(*traceOutput, pt0);		*traceOutput << eol;
		Output(*traceOutput, pt1);		*traceOutput << eol;
		Output(*traceOutput, pt2);		*traceOutput << eol;
		if (pt3 != pt2)
		{
			Output(*traceOutput, pt3);
			*traceOutput << eol;
		}
	*traceOutput << ret;
*/
	delete[] layerName;
}


void
ParserDxf::ParsePolyline()
{
	char*			layerName = NULL;
	int32			color = -1;
	int32			verticesFollow;
	pPoint3			elevation(0.0, 0.0, 0.0);
	pVector3		extrusion(0.0, 0.0, 0.0);
	int32			flags = 0;
	int32			m = 0;
	int32			n = 0;
	int32			m73 = 0;
	int32			n74 = 0;

	Match(0, TOKEN_POLYLINE);
	while (!Peek(0))
	{
		if (Peek(10))
			ParsePoint(10, elevation);
		else if (Peek(62))
			MatchGroup(62, color);
		else if (Peek(66))
			MatchGroup(66, verticesFollow);
		else if (Peek(8))
			MatchGroupCast(8, layerName);
		else if (Peek(70))
			MatchGroup(70, flags);
		else if (Peek(71))
			MatchGroup(71, m);
		else if (Peek(72))
			MatchGroup(72, n);
		else if (Peek(73))
			MatchGroup(73, m73);
		else if (Peek(74))
			MatchGroup(74, n74);
		else if (Peek(210))
			ParseVector(210, extrusion);
		else
			NextToken();
	}

	if ((m73 != 0) && (n74 != 0))
	{
		m = m73;
		n = n74;
	}

	if (!layerName)
		layerName = strdup("default");
/*
	*traceOutput << layerName << ": pl, flags " << uint16(flags) << " m=" << m << " n=" << n << ": ";
		Output(*traceOutput, elevation);
	*traceOutput << eol << adv;
	if (extrusion != pVector3(0.0, 0.0, 0.0))
	{
		*traceOutput << "extrusion along ";
		Output(*traceOutput, extrusion);
		*traceOutput << eol;
	}
*/

	int32			layerIndex = FindOrCreateLayer(layerName);
	delete[] layerName;
	if (color < 0)
		color = int32(layersColor[layerIndex]);
	int32			matIndex = SearchMaterial(palette[color]);
	pMaterial*		material = scene->materials[matIndex];
	pObject*		object = scene->objects[layerIndex];

	// parse vertices
	PierrotArray<int32 >		indexes(16, 8);
	while (!Peek(0, TOKEN_SEQEND))
	{
		if (Peek(0, TOKEN_VERTEX))
		{
			if (flags & 0x08)
				indexes.AddItem(ParseDotVertex());
			else if (flags & 0x10)
				indexes.AddItem(ParseVertex());
			else
			{
				int32			i0, i1, i2, i3;
				ParsePolyfaceVertex(i0, i1, i2, i3);
/*				bool			isQuad = (i3 != 0x80000000);
				if (i0 != 0x80000000)
				{
					i1 = ((i1 < 0) ? -i1 : i1);// - 1;
					i2 = ((i2 < 0) ? -i2 : i2);// - 1;
					i3 = ((i3 < 0) ? -i3 : i3);// - 1;
					if (isQuad)
						object->faces.AddItem(new pFace(material, i0, i1, i2, i3));
					else
						object->faces.AddItem(new pFace(material, i0, i1, i2));
				}
*/			}
		}
		else
			NextToken();
	}
	Match(0, TOKEN_SEQEND);
//	*traceOutput << ret;

	if (flags & 0x10)
	{
		for (int32 i = 0; i < m - 1; i++)
		{
			int32			ni = n*i;
			for (int32 j = 0; j < n - 1; j++)
			{
				object->faces.AddItem(new pFace(material,
									indexes[ni + j+1], indexes[ni + j],
									indexes[ni+n + j], indexes[ni+n + j+1]));
			}
			// close the mesh in the N direction
			if (flags & 0x20)
			{
				object->faces.AddItem(new pFace(material,
									indexes[ni], indexes[ni + n-1],
									indexes[ni+n + n-1], indexes[ni+n]));
			}
		}
		// close the mesh in the M direction
		if (flags & 0x01)
		{
			int32			nm_1 = n*(m-1);
			for (int32 j = 0; j < n - 1; j++)
			{
				object->faces.AddItem(new pFace(material,
									indexes[j+1], indexes[j],
									indexes[nm_1 + j], indexes[nm_1 + j+1]));
			}
		}
	}
/*	else if (flags & 0x08)		// simple polyline
	{
		pSegment			liline;
		for (int32 i = 0; i < indexes.CountItems(); i++)
		{
			int32			b = i+1;
			if (b >= indexes.CountItems())
				b = 0;
			liline.SetTo(indexes[i], indexes[b]);
			object->lines.AddItem(liline);
		}
	}
*/
}


//----------------------------------------------------

void
ParserDxf::InitPalette()
{
	int16			i = 0;
	palette[i++].Set(0.0, 0.0, 0.0);
	palette[i++].Set(1.0, 0.0, 0.0);
	palette[i++].Set(1.0, 1.0, 0.0);
	palette[i++].Set(0.0, 1.0, 0.0);
	palette[i++].Set(0.0, 1.0, 1.0);
	palette[i++].Set(0.0, 0.0, 1.0);
	palette[i++].Set(1.0, 0.0, 1.0);
	palette[i++].Set(1.0, 1.0, 1.0);
	palette[i++].Set(0.5, 0.5, 0.5);
	palette[i++].Set(0.75, 0.75, 0.75);
	palette[i++].Set(1.0, 0.0, 0.0);
	palette[i++].Set(1.0, 0.5, 0.5);
	palette[i++].Set(0.65, 0.0, 0.0);
	palette[i++].Set(0.65, 0.325, 0.325);
	palette[i++].Set(0.5, 0.0, 0.0);
	palette[i++].Set(0.5, 0.25, 0.25);
	palette[i++].Set(0.3, 0.0, 0.0);
	palette[i++].Set(0.3, 0.15, 0.15);
	palette[i++].Set(0.15, 0.0, 0.0);
	palette[i++].Set(0.15, 0.075, 0.075);
	palette[i++].Set(1.0, 0.25, 0.0);
	palette[i++].Set(1.0, 0.625, 0.5);
	palette[i++].Set(0.65, 0.1625, 0.0);
	palette[i++].Set(0.65, 0.4063, 0.325);
	palette[i++].Set(0.5, 0.125, 0.0);
	palette[i++].Set(0.5, 0.3125, 0.25);
	palette[i++].Set(0.3, 0.075, 0.0);
	palette[i++].Set(0.3, 0.1875, 0.15);
	palette[i++].Set(0.15, 0.0375, 0.0);
	palette[i++].Set(0.15, 0.0938, 0.075);
	palette[i++].Set(1.0, 0.5, 0.0);
	palette[i++].Set(1.0, 0.75, 0.5);
	palette[i++].Set(0.65, 0.325, 0.0);
	palette[i++].Set(0.65, 0.4875, 0.325);
	palette[i++].Set(0.5, 0.25, 0.0);
	palette[i++].Set(0.5, 0.375, 0.25);
	palette[i++].Set(0.3, 0.15, 0.0);
	palette[i++].Set(0.3, 0.225, 0.15);
	palette[i++].Set(0.15, 0.075, 0.0);
	palette[i++].Set(0.15, 0.1125, 0.075);
	palette[i++].Set(1.0, 0.75, 0.0);
	palette[i++].Set(1.0, 0.875, 0.5);
	palette[i++].Set(0.65, 0.4875, 0.0);
	palette[i++].Set(0.65, 0.5688, 0.325);
	palette[i++].Set(0.5, 0.375, 0.0);
	palette[i++].Set(0.5, 0.4375, 0.25);
	palette[i++].Set(0.3, 0.225, 0.0);
	palette[i++].Set(0.3, 0.2625, 0.15);
	palette[i++].Set(0.15, 0.1125, 0.0);
	palette[i++].Set(0.15, 0.1313, 0.075);
	palette[i++].Set(1.0, 1.0, 0.0);
	palette[i++].Set(1.0, 1.0, 0.5);
	palette[i++].Set(0.65, 0.65, 0.0);
	palette[i++].Set(0.65, 0.65, 0.325);
	palette[i++].Set(0.5, 0.5, 0.0);
	palette[i++].Set(0.5, 0.5, 0.25);
	palette[i++].Set(0.3, 0.3, 0.0);
	palette[i++].Set(0.3, 0.3, 0.15);
	palette[i++].Set(0.15, 0.15, 0.0);
	palette[i++].Set(0.15, 0.15, 0.075);
	palette[i++].Set(0.75, 1.0, 0.0);
	palette[i++].Set(0.875, 1.0, 0.5);
	palette[i++].Set(0.4875, 0.65, 0.0);
	palette[i++].Set(0.5688, 0.65, 0.325);
	palette[i++].Set(0.375, 0.5, 0.0);
	palette[i++].Set(0.4375, 0.5, 0.25);
	palette[i++].Set(0.225, 0.3, 0.0);
	palette[i++].Set(0.2625, 0.3, 0.15);
	palette[i++].Set(0.1125, 0.15, 0.0);
	palette[i++].Set(0.1313, 0.15, 0.075);
	palette[i++].Set(0.5, 1.0, 0.0);
	palette[i++].Set(0.75, 1.0, 0.5);
	palette[i++].Set(0.325, 0.65, 0.0);
	palette[i++].Set(0.4875, 0.65, 0.325);
	palette[i++].Set(0.25, 0.5, 0.0);
	palette[i++].Set(0.375, 0.5, 0.25);
	palette[i++].Set(0.15, 0.3, 0.0);
	palette[i++].Set(0.225, 0.3, 0.15);
	palette[i++].Set(0.075, 0.15, 0.0);
	palette[i++].Set(0.1125, 0.15, 0.075);
	palette[i++].Set(0.25,1.0, 0.0);
	palette[i++].Set(0.625,1.0, 0.5);
	palette[i++].Set(0.1625, 0.65, 0.0);
	palette[i++].Set(0.4063, 0.65, 0.325);
	palette[i++].Set(0.125, 0.5, 0.0);
	palette[i++].Set(0.3125, 0.5, 0.25);
	palette[i++].Set(0.075, 0.3, 0.0);
	palette[i++].Set(0.1875, 0.3, 0.15);
	palette[i++].Set(0.0375, 0.15, 0.0);
	palette[i++].Set(0.0938, 0.15, 0.075);
	palette[i++].Set(0.0, 1.0, 0.0);
	palette[i++].Set(0.5, 1.0, 0.5);
	palette[i++].Set(0.0, 0.65, 0.0);
	palette[i++].Set(0.325, 0.65, 0.325);
	palette[i++].Set(0.0, 0.5, 0.0);
	palette[i++].Set(0.25, 0.5, 0.25);
	palette[i++].Set(0.0, 0.3, 0.0);
	palette[i++].Set(0.15, 0.3, 0.15);
	palette[i++].Set(0.0, 0.15, 0.0);
	palette[i++].Set(0.075, 0.15, 0.075);
	palette[i++].Set(0.0, 1.0, 0.25);
	palette[i++].Set(0.5, 1.0, 0.625);
	palette[i++].Set(0, 0.65, 0.1625);
	palette[i++].Set(0.325, 0.65, 0.4063);
	palette[i++].Set(0.0, 0.5, 0.125);
	palette[i++].Set(0.25, 0.5, 0.3125);
	palette[i++].Set(0.0, 0.3, 0.075);
	palette[i++].Set(0.15, 0.3, 0.1875);
	palette[i++].Set(0.0, 0.15, 0.0375);
	palette[i++].Set(0.075, 0.15, 0.0938);
	palette[i++].Set(0.0, 1.0, 0.5);
	palette[i++].Set(0.5, 1.0, 0.75);
	palette[i++].Set(0.0, 0.65, 0.325);
	palette[i++].Set(0.325, 0.65, 0.4875);
	palette[i++].Set(0.0, 0.5, 0.25);
	palette[i++].Set(0.25, 0.5, 0.375);
	palette[i++].Set(0.0, 0.3, 0.15);
	palette[i++].Set(0.15, 0.3, 0.225);
	palette[i++].Set(0.0, 0.15, 0.075);
	palette[i++].Set(0.075, 0.15, 0.1125);
	palette[i++].Set(0.0, 1.0, 0.75);
	palette[i++].Set(0.5, 1.0, 0.875);
	palette[i++].Set(0.0, 0.65, 0.4875);
	palette[i++].Set(0.325, 0.65, 0.5688);
	palette[i++].Set(0.0, 0.5, 0.375);
	palette[i++].Set(0.25, 0.5, 0.4375);
	palette[i++].Set(0.0, 0.3, 0.225);
	palette[i++].Set(0.15, 0.3, 0.2625);
	palette[i++].Set(0.0, 0.15, 0.1125);
	palette[i++].Set(0.075, 0.15, 0.1313);
	palette[i++].Set(0.0, 1.0, 1.0);
	palette[i++].Set(0.5, 1.0, 1.0);
	palette[i++].Set(0.0, 0.65, 0.65);
	palette[i++].Set(0.325, 0.65, 0.65);
	palette[i++].Set(0.0, 0.5, 0.5);
	palette[i++].Set(0.25, 0.5, 0.5);
	palette[i++].Set(0.0, 0.3, 0.3);
	palette[i++].Set(0.15, 0.3, 0.3);
	palette[i++].Set(0.0, 0.15, 0.15);
	palette[i++].Set(0.075, 0.15, 0.15);
	palette[i++].Set(0.0, 0.75, 1.0);
	palette[i++].Set(0.5, 0.875, 1.0);
	palette[i++].Set(0.0, 0.4875, 0.65);
	palette[i++].Set(0.325, 0.5688, 0.65);
	palette[i++].Set(0.0, 0.375, 0.5);
	palette[i++].Set(0.25, 0.4375, 0.5);
	palette[i++].Set(0.0, 0.225, 0.3);
	palette[i++].Set(0.15, 0.2625, 0.3);
	palette[i++].Set(0.0, 0.1125, 0.15);
	palette[i++].Set(0.075, 0.1313, 0.15);
	palette[i++].Set(0.0, 0.5, 1.0);
	palette[i++].Set(0.5, 0.75, 1.0);
	palette[i++].Set(0.0, 0.325, 0.65);
	palette[i++].Set(0.325, 0.4875, 0.65);
	palette[i++].Set(0.0, 0.25, 0.5);
	palette[i++].Set(0.25, 0.375, 0.5);
	palette[i++].Set(0.0, 0.15, 0.3);
	palette[i++].Set(0.15, 0.225, 0.3);
	palette[i++].Set(0.0, 0.075, 0.15);
	palette[i++].Set(0.075, 0.1125, 0.15);
	palette[i++].Set(0.0, 0.25, 1.0);
	palette[i++].Set(0.5, 0.625, 1.0);
	palette[i++].Set(0.0, 0.1625, 0.65);
	palette[i++].Set(0.325, 0.4063, 0.65);
	palette[i++].Set(0.0, 0.125, 0.5);
	palette[i++].Set(0.25, 0.3125, 0.5);
	palette[i++].Set(0.0, 0.075, 0.3);
	palette[i++].Set(0.15, 0.1875, 0.3);
	palette[i++].Set(0.0, 0.0375, 0.15);
	palette[i++].Set(0.075, 0.0938, 0.15);
	palette[i++].Set(0.0, 0.0, 1.0);
	palette[i++].Set(0.5, 0.5, 1.0);
	palette[i++].Set(0.0, 0.0, 0.65);
	palette[i++].Set(0.325, 0.325, 0.65);
	palette[i++].Set(0.0, 0.0, 0.5);
	palette[i++].Set(0.25, 0.25, 0.5);
	palette[i++].Set(0.0, 0.0, 0.3);
	palette[i++].Set(0.15, 0.15, 0.3);
	palette[i++].Set(0.0, 0.0, 0.15);
	palette[i++].Set(0.075, 0.075, 0.15);
	palette[i++].Set(0.25, 0.0, 1.0);
	palette[i++].Set(0.625, 0.5, 1.0);
	palette[i++].Set(0.1625, 0.0, 0.65);
	palette[i++].Set(0.4063, 0.325, 0.65);
	palette[i++].Set(0.125, 0.0, 0.5);
	palette[i++].Set(0.3125, 0.25, 0.5);
	palette[i++].Set(0.075, 0.0, 0.3);
	palette[i++].Set(0.1875, 0.15, 0.3);
	palette[i++].Set(0.0375, 0.0, 0.15);
	palette[i++].Set(0.0938, 0.075, 0.15);
	palette[i++].Set(0.5, 0.0, 1.0);
	palette[i++].Set(0.75, 0.5, 1.0);
	palette[i++].Set(0.325, 0.0, 0.65);
	palette[i++].Set(0.4875, 0.325, 0.65);
	palette[i++].Set(0.25, 0.0, 0.5);
	palette[i++].Set(0.375, 0.25, 0.5);
	palette[i++].Set(0.15, 0.0, 0.3);
	palette[i++].Set(0.225, 0.15, 0.3);
	palette[i++].Set(0.075, 0.0, 0.15);
	palette[i++].Set(0.1125, 0.075, 0.15);
	palette[i++].Set(0.75, 0.0, 1.0);
	palette[i++].Set(0.875, 0.5, 1.0);
	palette[i++].Set(0.4875, 0.0, 0.65);
	palette[i++].Set(0.5688, 0.325, 0.65);
	palette[i++].Set(0.375, 0.0, 0.5);
	palette[i++].Set(0.4375, 0.25, 0.5);
	palette[i++].Set(0.225, 0.0, 0.3);
	palette[i++].Set(0.2625, 0.15, 0.3);
	palette[i++].Set(0.1125, 0.0, 0.15);
	palette[i++].Set(0.1313, 0.075, 0.15);
	palette[i++].Set(1.0, 0.0, 1.0);
	palette[i++].Set(1.0, 0.5, 1.0);
	palette[i++].Set(0.65, 0.0, 0.65);
	palette[i++].Set(0.65, 0.325, 0.65);
	palette[i++].Set(0.5, 0.0, 0.5);
	palette[i++].Set(0.5, 0.25, 0.5);
	palette[i++].Set(0.3, 0.0, 0.3);
	palette[i++].Set(0.3, 0.15, 0.3);
	palette[i++].Set(0.15, 0.0, 0.15);
	palette[i++].Set(0.15, 0.075, 0.15);
	palette[i++].Set(1.0, 0.0, 0.75);
	palette[i++].Set(1.0, 0.5, 0.875);
	palette[i++].Set(0.65, 0.0, 0.4875);
	palette[i++].Set(0.65, 0.325, 0.5688);
	palette[i++].Set(0.5, 0.0, 0.375);
	palette[i++].Set(0.5, 0.25, 0.4375);
	palette[i++].Set(0.3, 0.0, 0.225);
	palette[i++].Set(0.3, 0.15, 0.2625);
	palette[i++].Set(0.15, 0.0, 0.1125);
	palette[i++].Set(0.15, 0.075, 0.1313);
	palette[i++].Set(1.0, 0.0, 0.5);
	palette[i++].Set(1.0, 0.5, 0.75);
	palette[i++].Set(0.65, 0.0, 0.325);
	palette[i++].Set(0.65, 0.325, 0.4875);
	palette[i++].Set(0.5, 0.0, 0.25);
	palette[i++].Set(0.5, 0.25, 0.375);
	palette[i++].Set(0.3, 0.0, 0.15);
	palette[i++].Set(0.3, 0.15, 0.225);
	palette[i++].Set(0.15, 0.0, 0.075);
	palette[i++].Set(0.15, 0.075, 0.1125);
	palette[i++].Set(1.0, 0.0, 0.25);
	palette[i++].Set(1.0, 0.5, 0.625);
	palette[i++].Set(0.65, 0.0, 0.1625);
	palette[i++].Set(0.65, 0.325, 0.4063);
	palette[i++].Set(0.5, 0.0, 0.125);
	palette[i++].Set(0.5, 0.25, 0.3125);
	palette[i++].Set(0.3, 0.0, 0.075);
	palette[i++].Set(0.3, 0.15, 0.1875);
	palette[i++].Set(0.15, 0.0, 0.0375);
	palette[i++].Set(0.15, 0.075, 0.0938);
	palette[i++].Set(0.33, 0.33, 0.33);
	palette[i++].Set(0.464, 0.464, 0.464);
	palette[i++].Set(0.598, 0.598, 0.598);
	palette[i++].Set(0.732, 0.732, 0.732);
	palette[i++].Set(0.866, 0.866, 0.866);
	palette[i++].Set(1.0, 1.0, 1.0);
}


int32
ParserDxf::SearchMaterial(pSpectra& diff)
{
	int32			matIndex = scene->materials.Search(diff);
	if (matIndex < 0)
	{
		matIndex = scene->materials.CountItems();
		char			buf[32];
		sprintf(buf, "material %d", matIndex);

		pMaterial*		mat = NewMaterial(buf/*, diff*/);
		mat->diffuse.Set(diff);
	}
	return matIndex;
}

