//
//
//	LAP buffered-input class
//
//											(C) JoOl 1998


#include "LAPimpexp.h"

#include <new.h>
#include "LAPbufferedInput.h"
#include "LAPfileException.h"
#include "LAPerrors.h"



LAPbufferedInput::LAPbufferedInput(int32 bufSize)
: LAPdynBuffer(bufSize)
{
	input = LAP_BUFFER_NO_INPUT;
	inputMemorySize = 0;
	inputMemoryPtr = 0;
	inputMemory = NULL;
	inputFile = NULL;
	eofFlag = true;
	appliInterface = NULL;
//	status set in LAPbuffer::LAPbuffer
}


LAPbufferedInput::~LAPbufferedInput()
{
	SetNoInput();
}


void
LAPbufferedInput::SetAppliInterface(LAPappliInterface* ai)
{
	appliInterface = ai;
}


LAPappliInterface*
LAPbufferedInput::GetAppliInterface() const
{
	return appliInterface;
}


void
LAPbufferedInput::_badRead(char* str, int32 len)
{
	// read by pieces
	while (len > 0)
	{
		ReadWhatsLeft(str, len);
		if (!canRead())
		{
			ReadInput();
			if (eofFlag)
			{
				LAPbuffer::_badRead(str, len);
			}
		}
	}
}


void
LAPbufferedInput::badRead(char* str, int32 len)
{
	_badRead(str, len);
}


status_t
LAPbufferedInput::SetNoInput()
{
	UnsetFileInput();
	UnsetMemoryInput();
	input = LAP_BUFFER_NO_INPUT;
	eofFlag = true;
	return LAP_NO_ERROR;
}


status_t
LAPbufferedInput::SetFileInput(const char* nam)
{
	try
	{
		SetNoInput();

		inputFile = new BFile(nam, B_READ_ONLY);
		if (inputFile->InitCheck() != B_NO_ERROR)
			return LAP_BAD_VALUE;
		inputFilePath.SetTo(nam);
		input = LAP_BUFFER_FILE_INPUT;
		eofFlag = false;
		ReadInput();
		return LAP_NO_ERROR;
	}
	catch (bad_alloc )
	{
		return LAP_BAD_ALLOC;
	}
}


void
LAPbufferedInput::UnsetFileInput()
{
	if (input == LAP_BUFFER_FILE_INPUT)
	{
		inputFilePath.Unset();
		inputFile->Unset();
		delete inputFile;
		input = LAP_BUFFER_NO_INPUT;
	}
}


status_t
LAPbufferedInput::SetMemoryInput(char* str, int32 siz)
{
	SetNoInput();
	if ((!str) || (siz < 0))
		return LAP_BAD_VALUE;

	try
	{
		inputMemory = new char[siz+1];
		memcpy(inputMemory, str, siz);
		str[siz] = 0;
		inputMemoryPtr = 0;
		inputMemorySize = siz;
		input = LAP_BUFFER_MEMORY_INPUT;
		eofFlag = false;
		ReadInput();
		return LAP_NO_ERROR;
	}
	catch (bad_alloc )
	{
		return LAP_BAD_ALLOC;
	}
}


void
LAPbufferedInput::UnsetMemoryInput()
{
	if (input == LAP_BUFFER_MEMORY_INPUT)
	{
		inputMemorySize = inputMemoryPtr = 0;
		delete inputMemory;
		input = LAP_BUFFER_NO_INPUT;
	}
}


off_t
LAPbufferedInput::GetInputSize() const
{
	off_t			out = 0;
	if (input == LAP_BUFFER_MEMORY_INPUT)
		out = inputMemorySize;
	else if (input == LAP_BUFFER_FILE_INPUT)
		inputFile->GetSize(&out);
	return out;
}


off_t
LAPbufferedInput::GetInputPosition() const
{
	off_t			out = 0;
	if (input == LAP_BUFFER_MEMORY_INPUT)
		out = inputMemoryPtr;
	else if (input == LAP_BUFFER_FILE_INPUT)
		out = inputFile->Position();
	return out;
}


off_t
LAPbufferedInput::GetPosition() const
{
	off_t			out = 0;
	off_t			unreadBytes = writePtr - readPtr;
	if (input == LAP_BUFFER_MEMORY_INPUT)
		out = inputMemoryPtr - unreadBytes;
	else if (input == LAP_BUFFER_FILE_INPUT)
		out = inputFile->Position() - unreadBytes;
	return out;
}


status_t
LAPbufferedInput::SetPosition(off_t newpos)
{
	if (newpos < 0)
		return LAP_BAD_VALUE;

	off_t			is = GetInputSize();
	if (newpos > is)
	{
		eofFlag = true;
		status = LAP_BAD_SEEK;

		char*			src = (input == LAP_BUFFER_FILE_INPUT)
							? (char* )inputFilePath.Leaf()
							: "<memory>";
		throw LAPbadSeek("LAPbufferedInput::SetPosition", src, newpos, is);
	}

	int32			nBlocks = newpos / alloc;
	int32			offset = newpos % alloc;
	if (input == LAP_BUFFER_FILE_INPUT)
		inputFile->Seek(nBlocks * alloc, SEEK_SET);
	else
		inputMemoryPtr = nBlocks * alloc;
	ReadInput();
	readPtr = offset;
	return LAP_NO_ERROR;
}


char*
LAPbufferedInput::GetInputPath() const
{
	return (inputFile) ? (char* )inputFilePath.Path() : (char* )"<memory>";
}


char*
LAPbufferedInput::GetInputLeaf() const
{
	return (inputFile) ? (char* )inputFilePath.Leaf() : (char* )"<memory>";
}


bool
LAPbufferedInput::Eof() const
{
	return eofFlag;
}


status_t
LAPbufferedInput::_ReadInput()
{
	if (input == LAP_BUFFER_FILE_INPUT)
	{
		writePtr = inputFile->Read(buffer, alloc);
	}
	else
	{
		writePtr = min_c(alloc, inputMemorySize - inputMemoryPtr);
		memcpy(buffer, &inputMemory[inputMemoryPtr], writePtr);
		inputMemoryPtr += writePtr;
	}
	readPtr = 0;
	if (writePtr < 1)
		eofFlag = true;
	return LAP_NO_ERROR;
}


status_t
LAPbufferedInput::ReadInput()
{
	status_t			s = _ReadInput();

	if (appliInterface)
	{
		thread_info		tinfo;
		get_thread_info(find_thread(NULL), &tinfo);
		appliInterface->ReportProgress(	float(GetInputPosition()),		// current done
										float(GetInputSize()),			// to be done
										10e-6*float(tinfo.user_time));	// elapsed time in s
	}

	return s;
}


void
LAPbufferedInput::Read(char* str, int32 len)
{
	LAPbuffer::Read(str, len);
	if (!canRead())
		ReadInput();
}


void
LAPbufferedInput::Read(char& c)
{
	LAPbuffer::Read(c);
	if (!canRead())
		ReadInput();
}


char
LAPbufferedInput::Read()
{
	char			c;
	LAPbuffer::Read(c);
	if (!canRead())
		ReadInput();
	return c;
}


void
LAPbufferedInput::ReadString(char* str, int32 maxLen)
{
	char*		ptr = str;
	int32		i = 0;
	while ((i++ < maxLen) && (!eofFlag))
	{
		char		c;
		Read(c);
		if (c == '\0')
			break;
		*ptr++ = c;
	}
	*ptr = 0;
}


void
LAPbufferedInput::SkipBytes(off_t len)
{
	if (len < 0)
		return ;

	if (readPtr + len < writePtr)
	{
		readPtr += len;
		return ;
	}

	off_t			newpos = GetPosition() + len;
	SetPosition(newpos);
}


void
LAPbufferedInput::Skip()
{
	char			c;
	Read(c);
}
