#include <Debug.h>
#include "HistoryIO.h"

#include <Locker.h>
#include <Message.h>
#include <Directory.h>
#include <File.h>
#include <String.h>
#include <Window.h>

#include "GConstants.h"
#include "GManager.h"
#include "Gicq.h"
#include "Locale.h"
#include "Prefs.h"

HistoryIO::HistoryIO(GManager *aManager)
	:BLooper( "History" )
	,mManager( aManager )
{
	
	the_lock   = new BLocker();
	
	Run();	

}

//-----------------------------------------------------------------------------

HistoryIO::~HistoryIO()
{
	delete the_lock;
}

//-----------------------------------------------------------------------------

void HistoryIO::ReadAll(const uint32 uin, const uint32 own_uin, BWindow *history_window)
{
	
	// Don't need to acquire the lock, since one never manipulates a class global variable.
	the_lock -> Lock();
	
	//Lock the window;
	
	BDirectory history_dir;
	BString path, filename;
	BString error_msg;
	char *temp;
		
	path = "history-";
	path << own_uin;

	filename << uin;
	BFile read_file;
	
	if(mManager->prefs->FindConfigFile( path.String(), filename.String(), &read_file, B_READ_ONLY) == B_OK) {
		while(findNextMessage(&read_file)) {
			chatEntry entry;
			
			read_file.Read((void *) &entry.time, sizeof(time_t));
			read_file.Read((void *) &entry.type, sizeof(int32));
			switch(entry.type) {
			
				case CT_OUTGOING_URL:
				case CT_INCOMING_URL:
				{
					int32 size;
					read_file.Read((void *) &size, sizeof(int32));
					temp = new char[size+1];
					read_file.Read((void *) temp, size);
					temp[size] = '\0';
					entry.url = temp;
					delete temp;
				}
				// Goes on to read message.
	
				
				case CT_OUTGOING_MESSAGE:
				case CT_INCOMING_MESSAGE:
				{
					int32 size;
					read_file.Read((void *) &size, sizeof(int32));
					temp = new char[size+1];
					read_file.Read((void *) temp, size);
					temp[size] = '\0';
					entry.message = temp;
					delete temp;
				}
				break;
				
				case CT_OFFLINE_DONE:
				break; 
			} // switch
			
			// Add;
			BMessage *history_msg = new BMessage( entry.type);
			history_msg -> AddInt64("time", entry.time );
			history_msg -> AddString("TEXT", entry.message);
			history_msg -> AddString("URL", entry.url);
			
			history_window -> LockLooper();
			history_window -> PostMessage( history_msg );
			history_window -> UnlockLooper();
			
			delete history_msg;
			
		} // while
	} else {
		error_msg = mManager -> mLocale -> FindString("text190");
		error_msg += " ";
		error_msg += path.String();
		fprintf(stderr,"%s\n", error_msg.String());
	}

	history_window -> LockLooper();
	history_window -> PostMessage( GIMICQ_LAST_MESSAGE );
	history_window -> UnlockLooper();
	the_lock -> Unlock();
}

//-----------------------------------------------------------------------------

void HistoryIO::MessageReceived(BMessage *msg)
{
	chatEntry entry;
	switch(msg -> what) {
	
	case GIMICQ_OUTGOING_URL:
		entry.type = CT_OUTGOING_URL;
		entry.message = msg->FindString("TEXT");
		entry.url = msg->FindString("URL");
		entry.uin = (uint32) msg -> FindInt32("uin");
		entry.own_uin = (uint32) msg -> FindInt32("own_uin");
		entry.time = (time_t) msg -> FindInt64("time");
		Write(entry);
		break;

	case GIMICQ_ATTEMPT_SEND_IM:
		entry.type = CT_OUTGOING_MESSAGE;
		entry.message = msg->FindString("TEXT");
		entry.uin = (uint32) msg -> FindInt32("uin");
		entry.own_uin = (uint32) msg -> FindInt32("own_uin");
		entry.time = (time_t) msg -> FindInt64("time");
		Write(entry);
		break;

	case ICQ_TEXT_MESSAGE:
		entry.type = CT_INCOMING_MESSAGE;
		entry.message = msg->FindString("TEXT");
		entry.uin = (uint32) msg -> FindInt32("uin");
		entry.own_uin = (uint32) msg -> FindInt32("own_uin");
		entry.time = (time_t) msg -> FindInt64("time");
		Write(entry);
		break;

	case ICQ_URL_MESSAGE:
		entry.type = CT_INCOMING_URL;
		entry.message = msg->FindString("TEXT");
		entry.url = msg->FindString("URL");
		entry.uin = (uint32) msg -> FindInt32("uin");
		entry.own_uin = (uint32) msg -> FindInt32("own_uin");
		entry.time = (time_t) msg -> FindInt64("time");
		Write(entry);
		break;
	default:
		BLooper::MessageReceived(msg);
	}
	
}

//-----------------------------------------------------------------------------

void HistoryIO::Write(const chatEntry entry)
{

	the_lock -> Lock();

	BDirectory history_dir;
	BString path, filename;
	BString error_msg;
	status_t s=B_OK;
		
	path = "history-";
	path << entry.own_uin;

	filename << entry.uin;
	BFile write_file;
	s= mManager->prefs->FindConfigFile( path.String(), filename.String(), &write_file, B_WRITE_ONLY | B_CREATE_FILE | B_OPEN_AT_END ); 
	if( s == B_OK) {
		
		if(entry.type != CT_OFFLINE_DONE) {
			write_file.Write("\n::",3);

			write_file.Write(&entry.time, sizeof(time_t));
			write_file.Write(&entry.type, sizeof(int32));
			switch(entry.type) {
			
				case CT_OUTGOING_URL:
				case CT_INCOMING_URL:
				{
					int32 size = entry.url.Length();
					write_file.Write(&size, sizeof(int32));
					write_file.Write(entry.url.String(), size);
				}
				// Goes on to write message.
	
				
				case CT_OUTGOING_MESSAGE:
				case CT_INCOMING_MESSAGE:
				{
					int32 size = entry.message.Length();
					write_file.Write(&size, sizeof(int32));
					write_file.Write(entry.message.String(), size);
				}
				break;

				case CT_OFFLINE_DONE:
				break;
				
			}
		} // switch 
	} else {
		error_msg = mManager -> mLocale -> FindString("text190");
		error_msg += " ";
		error_msg += path.String();
		fprintf(stderr,"%s\n", error_msg.String());
	}

	the_lock -> Unlock();
}

//-----------------------------------------------------------------------------

bool HistoryIO::findNextMessage(BFile *the_file)
{
	
	BString error_msg;
	char c = 0;
	size_t read_mnt;
		
	while(c!=':') {
	
		if((read_mnt = the_file -> Read((void *) &c,1)) <0 ) {
			error_msg = mManager -> mLocale -> FindString("text192");
			fprintf(stderr,"%s\n", error_msg.String());
			return false; 
		} else {
			if( read_mnt == 0)
				return false;
		}
		
		if(c==':') {  // To identify "::"
		
			if((read_mnt = the_file -> Read((void *) &c,1)) <0 ) {
			error_msg = mManager -> mLocale -> FindString("text192");
			fprintf(stderr,"%s\n", error_msg.String());
				return false; 
			} else {
				if( read_mnt == 0)
					return false;
			}
		}
		
	}

	
	return true;
}

//-----------------------------------------------------------------------------

#if 0
void HistoryIO::ReWrite(GenList<chatEntry> list, const uint32 uin, const uint32 own_uin)
{

	the_lock -> Lock();

	BDirectory history_dir;
	BString path;
	BString error_msg;
		
	path = SETTINGS_FOLDER;
	path += "/history-";
	path << own_uin;
	if(history_dir.SetTo( path.String() ) == B_ENTRY_NOT_FOUND)
		history_dir.CreateDirectory( path.String() ,&history_dir);


	if(history_dir.InitCheck() != B_OK) {
		error_msg = mManager -> mLocale -> FindString("text191");
		error_msg += " ";
		error_msg += path.String();
		fprintf(stderr,"%s\n", error_msg.String());
		the_lock -> Unlock();
		return;
	}
	path += "/";
	path << uin;
	

	BFile write_file;
	if(write_file.SetTo( path.String() , B_WRITE_ONLY | B_CREATE_FILE | B_ERASE_FILE) != B_OK) {
		error_msg = mManager -> mLocale -> FindString("text190");
		error_msg += " ";
		error_msg += path.String();
		fprintf(stderr,"%s\n", error_msg.String());
		the_lock -> Unlock();
		return;
	}

	write_file.Unset(); // Closes the file.

	chatEntry entry;	
	bool keepGoing = list.First(entry);
	while( keepGoing ) {
		
		the_lock -> Unlock();
		Write(entry);
		the_lock -> Lock();
		
		keepGoing = list.Next(entry);
	}

	the_lock -> Unlock();

}

#endif
