#include <Debug.h>
#include "Prefs.h"

#include <Autolock.h>
#include <Locker.h>
#include <Directory.h>
#include <FindDirectory.h>
#include <File.h>
#include <Rect.h>
#include <Entry.h>
#include <TypeConstants.h>
#include <String.h>
#include <Path.h>

#include <string.h>
#include <stdlib.h>


#include "MiscStuff.h"

//-------------------------------------------------------------------------

const char *OLD_SETTINGS_FOLDER = "GimICQ";

GPreferences::GPreferences()
	:mUin(0)

{
	PRINT( ("GPreferences::GPreferences\n") );

	status_t error;
	mPrefsDirectory = new BDirectory();
	mPrefsFile = new BFile();
	mGlobalPrefsFile = new BFile();
	
	BPath settingsPath, path;
	
	error = find_directory( B_USER_SETTINGS_DIRECTORY, &settingsPath, true );
	if( error != B_OK ) {
		fprintf(stderr, "Failed to locate config dir. Application might behave strange." );
		return;
	}
	path.SetTo( settingsPath.Path(), SETTINGS_FOLDER );
	error = mPrefsDirectory->SetTo( path.Path() );
	
	// Try and open the Gim-ICQ prefs folder, and create a new one if it doesn't exist
	if( error == B_ENTRY_NOT_FOUND ) {
		path.SetTo( settingsPath.Path(), OLD_SETTINGS_FOLDER );
		error = mPrefsDirectory->SetTo( path.Path() );
		if( error == B_ENTRY_NOT_FOUND ) {
			path.SetTo( settingsPath.Path(), SETTINGS_FOLDER );
			mPrefsDirectory->CreateDirectory( path.Path(), mPrefsDirectory );
		}
	}
	
	// Now verify the global prefs file
	if( FindConfigFile( NULL, "globalprefs", mGlobalPrefsFile, B_READ_WRITE | B_CREATE_FILE ) != B_OK ) {
		fprintf( stderr, "\tError opening global config file: %d\n", (int)mGlobalPrefsFile->InitCheck() );
	}

	// make the locker
	mLock = new BLocker( "prefs_locker" );

	LoadDefaultUser();

}

//-------------------------------------------------------------------------

GPreferences::~GPreferences() {
	PRINT( ("GPreferences::~GPreferences\n") );

	mPrefsFile->Sync();
	mGlobalPrefsFile->Sync();
	delete mPrefsFile;
	delete mGlobalPrefsFile;
	delete mPrefsDirectory;
	delete mLock;
}

//-------------------------------------------------------------------------

void GPreferences::LoadUserFile( uint32 aUin, bool create ) {
	PRINT( ("GPreferences::LoadUserFile\n") );

	BAutolock a( mLock );
 
	mUin = aUin;
	char filename[B_PATH_NAME_LENGTH];

	MakeFilenameFromUIN( filename, mUin );	
	FindConfigFile( NULL, filename, mPrefsFile, B_READ_WRITE | B_CREATE_FILE);
		
}

//-------------------------------------------------------------------------

void GPreferences::LoadDefaultUser() {
	PRINT( ("GPreferences::LoadDefaultUser\n") );

#if 0
	BAutolock a( mLock );
	status_t error;
	BString file;
	
	file = SETTINGS_FOLDER;
	file += "/defaultuser";
	
	// Now verify the prefs file
	error = mPrefsFile->SetTo( file.String(), B_READ_WRITE );
	if( error == B_ENTRY_NOT_FOUND )
		mPrefsDirectory->CreateFile( file.String(), mPrefsFile );
#endif		
}

//-------------------------------------------------------------------------

void GPreferences::WriteBool( const char *aName, bool aValue, bool global ) {
	PRINT( ("GPreferences::WriteBool\n") );
	
	BAutolock a( mLock );	
	if( global )
		mGlobalPrefsFile->WriteAttr( aName, B_BOOL_TYPE, 0, &aValue, sizeof(bool) );
	else
		mPrefsFile->WriteAttr( aName, B_BOOL_TYPE, 0, &aValue, sizeof(bool) );
	
}

//-------------------------------------------------------------------------

bool GPreferences::ReadBool( const char *aName, bool aDefault, bool global ) {
	PRINT( ("GPreferences::ReadBool\n") );

	BAutolock a( mLock );
	bool ret;
	ssize_t res;
	
	ret = aDefault;
	if( global )
		res = mGlobalPrefsFile->ReadAttr( aName, B_BOOL_TYPE, 0, &ret, sizeof(bool) );
	else
		res = mPrefsFile->ReadAttr( aName, B_BOOL_TYPE, 0, &ret, sizeof(bool) );
	
	return ret;
}

//-------------------------------------------------------------------------
		
void GPreferences::WriteString( const char *aName, const char *aValue, bool global ) {
	PRINT( ("GPreferences::WriteString\n") );

	BAutolock a( mLock );
	int gSize;
	int hSize;

	gSize = strlen( aValue ) + 1;
	
	if( global ) {
		mGlobalPrefsFile->RemoveAttr( aName );
		hSize = mGlobalPrefsFile->WriteAttr( aName, B_STRING_TYPE, 0, aValue, gSize );
	} else {
		mPrefsFile->RemoveAttr( aName );
		hSize = mPrefsFile->WriteAttr( aName, B_STRING_TYPE, 0, aValue, gSize );
	}
	
}

//-------------------------------------------------------------------------
		
void GPreferences::ReadString( const char *aName, char *aBuf, const char *aDefault, uint32 aSize, bool global ) {
	PRINT( ("GPreferences::ReadString\n") );

	BAutolock a( mLock );
	ssize_t res;
	
	strcpy( aBuf, aDefault );
	if( global )
		res = mGlobalPrefsFile->ReadAttr( aName, B_STRING_TYPE, 0, aBuf, aSize );
	else
		res = mPrefsFile->ReadAttr( aName, B_STRING_TYPE, 0, aBuf, aSize );		
	
}

//-------------------------------------------------------------------------
		
void GPreferences::WriteInt32( const char *aName, int32 aValue, bool global ) {
	PRINT( ("GPreferences::WriteInt32\n") );

	BAutolock a( mLock );
	if( global )
		mGlobalPrefsFile->WriteAttr( aName, B_INT32_TYPE, 0, &aValue, sizeof(int32) );
	else
		mPrefsFile->WriteAttr( aName, B_INT32_TYPE, 0, &aValue, sizeof(int32) );
	
}

//-------------------------------------------------------------------------
		
int32 GPreferences::ReadInt32( const char *aName, int32 aDefault, bool global ) {
	PRINT( ("GPreferences::ReadInt32\n") );

	BAutolock a( mLock );
	int32 ret;
	ssize_t res;
	
	ret = aDefault;
	if( global )
		res = mGlobalPrefsFile->ReadAttr( aName, B_INT32_TYPE, 0, &ret, sizeof(int32) );
	else
		res = mPrefsFile->ReadAttr( aName, B_INT32_TYPE, 0, &ret, sizeof(int32) );
	
	return ret;
}

//-------------------------------------------------------------------------
		
void GPreferences::WriteFloat( const char *aName, float aValue, bool global ) {
	PRINT( ("GPreferences::WriteFloat\n") );

	BAutolock a( mLock );
	if( global )
		mGlobalPrefsFile->WriteAttr( aName, B_FLOAT_TYPE, 0, &aValue, sizeof(float) );
	else
		mPrefsFile->WriteAttr( aName, B_FLOAT_TYPE, 0, &aValue, sizeof(float) );
	
}

//-------------------------------------------------------------------------
		
float GPreferences::ReadFloat( const char *aName, float aDefault, bool global ) {
	PRINT( ("GPreferences::ReadFloat\n") );

	BAutolock a( mLock );
	float ret;
	ssize_t res;
	
	ret = aDefault;
	if( global )
		res = mGlobalPrefsFile->ReadAttr( aName, B_FLOAT_TYPE, 0, &ret, sizeof(float) );
	else
		res = mPrefsFile->ReadAttr( aName, B_FLOAT_TYPE, 0, &ret, sizeof(float) );
	
	return ret;
}

//-------------------------------------------------------------------------
		
void GPreferences::WriteRect( const char *aName, BRect aValue, bool global ) {
	PRINT( ("GPreferences::WriteRect\n") );

	BAutolock a( mLock );
	if( global )
		mGlobalPrefsFile->WriteAttr( aName, B_RECT_TYPE, 0, &aValue, sizeof(BRect) );
	else
		mPrefsFile->WriteAttr( aName, B_RECT_TYPE, 0, &aValue, sizeof(BRect) );
	
}

//-------------------------------------------------------------------------
		
BRect GPreferences::ReadRect( const char *aName, BRect aDefault, bool global ) {
	PRINT( ("GPreferences::ReadRect\n") );

	BAutolock a( mLock );
	BRect ret;
	ssize_t res;
	
	ret = aDefault;
	if( global )
		res = mGlobalPrefsFile->ReadAttr( aName, B_RECT_TYPE, 0, &ret, sizeof(BRect) );
	else
		res = mPrefsFile->ReadAttr( aName, B_RECT_TYPE, 0, &ret, sizeof(BRect) );
	
	return ret;
}

//-------------------------------------------------------------------------
		
void GPreferences::WritePoint( const char *aName, BPoint aValue, bool global ) {
	PRINT( ("GPreferences::WritePoint\n") );

	BAutolock a( mLock );
	if( global )
		mGlobalPrefsFile->WriteAttr( aName, B_POINT_TYPE, 0, &aValue, sizeof(BPoint) );
	else
		mPrefsFile->WriteAttr( aName, B_POINT_TYPE, 0, &aValue, sizeof(BPoint) );

}

//-------------------------------------------------------------------------
		
BPoint GPreferences::ReadPoint( const char *aName, BPoint aDefault, bool global ) {
	PRINT( ("GPreferences::ReadPoint\n") );

	BAutolock a( mLock );
	BPoint ret;
	ssize_t res;
	
	ret = aDefault;
	if( global )
		res = mGlobalPrefsFile->ReadAttr( aName, B_POINT_TYPE, 0, &ret, sizeof(BPoint) );
	else
		res = mPrefsFile->ReadAttr( aName, B_POINT_TYPE, 0, &ret, sizeof(BPoint) );
	
	return ret;
}

//-------------------------------------------------------------------------

bool GPreferences::GetNextUser( uint32& aUin, bool first ) {
	PRINT( ("GPreferences::GetNextUser\n") );

	BAutolock a( mLock );
	bool keepGoing = true, retval = false;
	char user[256];
	entry_ref fileRef;
	status_t res;

	// if this is the first one, rewind the directory object
	if( first )
		mPrefsDirectory->Rewind();
		
	// Loop through all the entries in the directory
	while( keepGoing ) {
	
		// Get the next one and break if it ain't there
		res = mPrefsDirectory->GetNextRef( &fileRef );		
		if( res == B_ENTRY_NOT_FOUND ) {
			keepGoing = false;
			retval = false;
			break;
		}
		
		if( res != B_OK ) {
			fprintf( stderr, "\tError reading ref in config dir: %d\n", (int)res );
			keepGoing = false;
			retval = false;
			break;
		}
		
		// OK, found a valid user
		if( strncmp( fileRef.name, "user-", 5 ) == 0 ) {

			// load up the user file for that user and get the prefs
			strcpy( user, fileRef.name + 5 );
			aUin = atoi(user);
			if( aUin != 0 )
				LoadUserFile( aUin );
			LoadDefaultUser();
			keepGoing = false;
			retval = true;
		}
	}

	return retval;
}

//-------------------------------------------------------------------------	

void GPreferences::GetUserLoginParameters( uint32 aUin, char *aPassword, int aLen, bool& autoLogin, bool& savePassword )
{
	PRINT( ("GPreferences::GetUserLoginParameters\n") );

	BAutolock a( mLock );
	savePassword = false;	

	// load up the user's file and get the info
	LoadUserFile( aUin );
	savePassword = ReadBool( "savepassword", false );
	autoLogin = savePassword && ReadBool( "autologin", false );
	if( savePassword ) {
		ReadString( "password", aPassword, "", aLen );
	}
	LoadDefaultUser();

}

//-------------------------------------------------------------------------

status_t 
GPreferences::FindConfigDir(const char *aRelativeDir, BDirectory *aDirectory)
{
	PRINT( ("GPreferences::FindConfigDir\n") );

	status_t s;
	if( !aRelativeDir )
		return B_ERROR;
	
	if( !mPrefsDirectory->Contains( aRelativeDir, B_DIRECTORY_NODE ) ) {
		PRINT( ("Creating dir: %s\n", aRelativeDir) );
		s = mPrefsDirectory->CreateDirectory( aRelativeDir, aDirectory );			
	} else
		s=aDirectory->SetTo( mPrefsDirectory, aRelativeDir );	
	return s;
}

//-------------------------------------------------------------------------

status_t 
GPreferences::FindConfigFile(const char *aRelativeDir, const char *aFilename, BFile *aFile, uint32 openMode, BPath *aPath)
{
	PRINT( ("GPreferences::FindConfigFile\n") );

	BDirectory dir;
	status_t s;
	
	if( aRelativeDir ) {
		s=FindConfigDir( aRelativeDir, &dir );
		if( s != B_OK )
			return s;
	} else {
		dir = *mPrefsDirectory;
	}

	if(aPath)
		aPath->SetTo( &dir, aFilename, true );

	s = aFile->SetTo( &dir, aFilename, openMode );
	return s; 
}

//-------------------------------------------------------------------------

BPath
GPreferences::SettingDir()
{
	PRINT( ("GPreferences::SettingDir\n") );

	BPath path;
	BEntry entry;
	mPrefsDirectory->GetEntry( &entry );
	entry.GetPath( &path );
	return path;
}

