// GIMICQ - BuddyList.cpp
// The Buddy List module

#include "BuddyList.h"

#include <Application.h>
#include <Debug.h>
#include <MenuBar.h>
#include <Menu.h>
#include <PopUpMenu.h>
#include <Bitmap.h>	
#include <ScrollView.h>
#include <NodeInfo.h>
#include <Dragger.h>
#include <Locker.h>
#include <Autolock.h>

#include "Say.h"
#include "SearchForUser.h"
#include "MiscStuff.h"
#include "SendURL.h"
#include "UpdateInfo.h"
#include "PasswordChanger.h"
#include "KDeskbar.h"
#include "HistoryWindow.h"
#include "TrayIcon.h"
#include "ChatWindow.h"
#include "GManager.h"
#include "WindowManager.h"
#include "Prefs.h"
#include "Gicq.h"
#include "ProtocolManager.h"
#include "PeopleData.h"
#include "classSoundMaster.h"
#include "History.h"
#include "PeopleData.h"
#include "Locale.h"

//=========================================================================

const int LOGO_HEIGHT = 40;

//=========================================================================

BuddyWindow::BuddyWindow(BRect frame, GManager *the_manager)
						:BWindow(frame, the_manager -> _("title3"), B_DOCUMENT_WINDOW_LOOK, B_NORMAL_WINDOW_FEEL, B_NOT_ZOOMABLE, B_ALL_WORKSPACES )
{
	PRINT(("BuddyWindow::BuddyWindow\n"));
	mManager = the_manager;

	// load the saved rect if there is one
	if( mManager -> prefs ) {
		frame = mManager -> prefs->ReadRect( "blrect", frame, true );
		if( mManager -> windows )
			mManager -> windows->AdjustFrame( frame );
		MoveTo( frame.LeftTop() );
		ResizeTo( frame.Width(), frame.Height() );
	}

	bool allWorkspaces = mManager -> prefs->ReadBool("bl_aw", true);
	if( allWorkspaces )
		SetWorkspaces( B_ALL_WORKSPACES );
	else
		SetWorkspaces( B_CURRENT_WORKSPACE );

	// set up 1-second pulses, to handle the buddy list change updates
	SetPulseRate( 1000000 );
}

BuddyWindow::~BuddyWindow() {
}

bool BuddyWindow::QuitRequested()
{
	if( mManager -> prefs ) {
		mManager -> prefs->WriteRect( "blrect", Frame(), true);
	}

	SetPulseRate(0);
	BMessage* curMsg = CurrentMessage();
	if( !(curMsg->HasBool("killapp") && !curMsg->FindBool("killapp")) )
		be_app->PostMessage(B_QUIT_REQUESTED);
	
	return true;
}

// #pragma mark -


BuddyList::BuddyList(BRect frame, GManager *the_manager)
          : BView(frame, "BuddyView", B_FOLLOW_LEFT_RIGHT | B_FOLLOW_TOP_BOTTOM, B_WILL_DRAW | B_PULSE_NEEDED){
	
	PRINT(("BuddyList::BuddyList\n"));

	mManager = the_manager;	
	is_replicant = false;
	
	buddy_bounds = frame;
	BRect r = buddy_bounds;
	r.OffsetTo(B_ORIGIN);
	r.top = r.bottom - 7;
	r.right = r.left + 7;
	BDragger *bDragger;

	bDragger = new BDragger(r, this, B_FOLLOW_LEFT | B_FOLLOW_BOTTOM,B_WILL_DRAW);

	AddChild(bDragger);

}

BuddyList::BuddyList(BMessage *archive)
					: BView( archive->FindRect("_frame") , "BuddyList" ,B_FOLLOW_LEFT_RIGHT | B_FOLLOW_TOP_BOTTOM, B_WILL_DRAW | B_PULSE_NEEDED) {

	SET_DEBUG_ENABLED(false);
	PRINT(("BuddyList::BuddyList(archive)\n"));

	mManager = new GManager();

	is_replicant = true;

	buddy_bounds            = archive -> FindRect("buddy_bounds"); 
	mManager -> LoggedIn = archive -> FindBool("logged_in");
	//mManager -> LoggedIn = false;
	
	BRect r = buddy_bounds;
	r.OffsetTo(B_ORIGIN);
	r.top = r.bottom - 7;
	r.right = r.left + 7;
	BDragger *bDragger;

	bDragger = new BDragger(r, this, B_FOLLOW_LEFT | B_FOLLOW_BOTTOM,B_WILL_DRAW);

	AddChild(bDragger);

}


//-------------------------------------------------------------------------

void BuddyList::Init(){

	PRINT(("BuddyList::Init\n"));
	BString temp;
	DoGimICQStartupStuff();
	
	mLocale = mManager -> mLocale;
	BRect r; 
	BBitmap* smallLogo;
	smallLogo = NULL;
	r = buddy_bounds;
	

	menubar = new BMenuBar(r, "menu_bar");
	beMenu = new BMenu( _("constant1") );
	BMenuItem* menuItem;

	temp = _("menuitem1");
	temp += B_UTF8_ELLIPSIS;
	beMenu->AddItem(menuItem = new BMenuItem( temp.String() , new BMessage(GIMICQ_SEARCH_BY_EMAIL) ));
	menuItem -> SetEnabled(false);

	temp = _("menuitem2");
	temp += B_UTF8_ELLIPSIS;
	beMenu->AddItem(menuItem = new BMenuItem( temp.String() , new BMessage(GIMICQ_UPDATE_MY_INFO) ));
	menuItem -> SetEnabled(false);

	temp = _("menuitem3");
	temp += B_UTF8_ELLIPSIS;
	beMenu->AddItem(menuItem = new BMenuItem( temp.String() , new BMessage(GIMICQ_CHANGE_PASSWORD) ));
	menuItem -> SetEnabled(false);

	temp = _("menuitem4");
	temp += B_UTF8_ELLIPSIS;
	beMenu->AddSeparatorItem();
	beMenu->AddItem(menuItem = new BMenuItem( temp.String() , new BMessage(GIMICQ_OPEN_PREFS)) );
	menuItem -> SetEnabled(false);

	temp = _("menuitem5");
	temp += B_UTF8_ELLIPSIS;
	beMenu->AddItem(menuItem = new BMenuItem( temp.String() , new BMessage(GIMICQ_ABOUT)) );
	menuItem -> SetEnabled(true);

	beMenu->AddSeparatorItem();
	loginMenuItem = new BMenuItem( _("menuitem6") , new BMessage(GIMICQ_LOGIN));
	beMenu->AddItem( loginMenuItem);

	logoutMenuItem = new BMenuItem( _("menuitem7") , new BMessage(GIMICQ_LOGOUT)); 
	logoutMenuItem -> SetEnabled(false);
	beMenu->AddItem(logoutMenuItem);

	menubar->AddItem(beMenu);
	AddChild(menubar);	

	r.top = menubar -> Bounds().bottom + 1.0;
//	r.top += 20.0;
/*	r.bottom = r.top + LOGO_HEIGHT;
	GetBitmapFromResources( smallLogo, 942 );
	logoView = new BitmapView( r, smallLogo );
	AddChild( logoView );
	r.top = r.bottom - 1;
*/
		
	r.bottom = r.top;
	iView = new BuddyListInfoView( r, mManager );
	AddChild( iView );
	
	BRect q = buddy_bounds;
	q.top = r.bottom;
	q.bottom -= 14;
	onlineview = new BuddyListView(q, mManager);
	AddChild( onlineview );
	
	r = buddy_bounds;
	r.top = q.bottom + 1;
	statView = new BuddyListStatusView( r, mManager );
	AddChild( statView );
	
	iView->doIdleTime = false;
	statView->doIdleTime = false;
}

BuddyList::~BuddyList()
{

	// remove the tray icon
	if( mManager -> deskbarIconInstalled )
		DoDeskbarIcon( false );

	delete mManager;
}

//-------------------------------------------------------------------------


status_t BuddyList::Archive(BMessage *data, bool deep) const{
	BView::Archive(data, deep);

	data -> AddString( "add_on" , APP_SIGNATURE );
	data -> AddRect( "buddy_bounds", Bounds() );
	data -> AddBool( "logged_in", mManager -> LoggedIn );

	return B_OK;
}

BArchivable *BuddyList::Instantiate(BMessage *data) { 

	if ( validate_instantiation(data, "BuddyList") ) {
		return new BuddyList( data ); 
	}
	
	return NULL; 
}

void BuddyList::AttachedToWindow(void) {

	Init();

	/*
	 * ICQ STUFF
	 */	

	mManager -> icq->SetTarget(this);
	mManager -> windows -> setBuddyList(this);

	/*
	 * GUI STUFF
	 */
	
	beMenu -> SetTargetForItems(this);
	beMenu -> ItemAt(4) -> SetTarget(this);
	// tell the list that the display list is there
	onlineview -> online -> SetTarget(this);
	onlineview -> online -> userPopup -> SetTargetForItems(this);
	onlineview -> online -> SetInvocationMessage( new BMessage(GIMICQ_BUDDY_INVOKED) );	
	mManager -> list->SetDisplayList( onlineview -> online );
	
	if(!(mManager -> LoggedIn))
		mManager -> windows -> OpenSignOnWindow();
	
}

void BuddyList::DetachedFromWindow(void) {
	
	DoGimICQShutdownStuff();
}

void BuddyList::DoGimICQStartupStuff() {
	
	last_user_uin = 0;
}

//---------------------------------------------------------------------

void BuddyList::DoGimICQShutdownStuff() {


	// post a logout message and wait a bit
	if(mManager -> LoggedIn) {
		
		Logout();
		snooze(100000);		
	}
	
	mManager -> windows -> CloseSignOnWindow();
}

//---------------------------------------------------------------------

void BuddyList::Logout() {
	
	mManager -> LoggedIn = false;
	last_user_uin = mManager -> own_uin;
	
//	mManager -> windows -> CloseAllWindows();	// LoggedIn MUST BE false BEFORE THIS IS EXECUTED!!!!
	mManager -> windows -> CloseAllGeneralWindows();
	
	statView -> status = IS_OFFLINE;
	statView -> userPopup -> SetEnabled(false);

	Window() -> Lock();
	statView -> Invalidate();
	Window() -> Unlock();

	// remove the tray icon
	if( mManager -> deskbarIconInstalled )
		DoDeskbarIcon( false );

	mManager -> windows -> SignOnStop();
	mManager -> icq -> PostMessage( new BMessage(ICQ_LOGOUT) );
//	mManager -> icq -> PostMessage( new BMessage(ST_DISCONNECT) );
	mManager -> list -> Reset();
}

//-------------------------------------------------------------------------

void BuddyList::LoadPrefs() {
	PRINT(("BuddyList::LoadPrefs\n"));

	if(mManager -> LoggedIn) {
	
		// Load these prefs when you log in.
		statView -> status = (ICQStatus)mManager -> prefs -> ReadInt32( "status", IS_AVAILABLE );
		statView -> userPopup -> SetEnabled(true);
		Window() -> Lock();
		statView -> Invalidate();
		Window() -> Unlock();

		int32 tmp_status = StatusB(statView -> status);
		
		BMessage *sendMessage = new BMessage(ICQ_STATUS_UPDATE);
		sendMessage -> AddInt32( "STATUS", tmp_status);
		mManager -> icq -> PostMessage(sendMessage);

		BMessage *msg = new BMessage(GIMICQ_MY_STATUS_CHANGE);
		msg->AddInt32( "status", statView->status);
		KDeskbar::Instance()->TalkToReplicant( msg );

	}

	// load and implement the idle time setting
	iView->doIdleTime = statView->doIdleTime = mManager -> prefs->ReadBool("gm_id", true);

}

void BuddyList::DoGlobalPrefs() {

	// do sound prefs
	bool soundEnabled = mManager -> prefs->ReadBool("gm_ps", true);
	if( mManager -> sounds->IsEnabled() != soundEnabled )
		mManager -> sounds->SetEnabled( soundEnabled );
	
	if(!is_replicant) {
	
		bool allWorkspaces = mManager -> prefs->ReadBool("bl_aw", true);
		if( allWorkspaces )
			Window() -> SetWorkspaces( B_ALL_WORKSPACES );
		else
			Window() -> SetWorkspaces( B_CURRENT_WORKSPACE );

		// do deskbar icon prefs
		bool deskbarEnabled = mManager -> prefs->ReadBool("gm_di", true);
		if( deskbarEnabled != mManager -> deskbarIconInstalled ) {
			DoDeskbarIcon( deskbarEnabled );
		}
	
		// flashy stuff
		if( deskbarEnabled ) {
			BMessage* msg = new BMessage(GIMICQ_ENABLE_FLASHY_STUFF);
			msg->AddBool( "enabled", mManager -> prefs->ReadBool("gm_fl", true) );
			KDeskbar::Instance()->TalkToReplicant( msg );
		}
	}
	
	// encoding shtuff
	mManager -> defEncoding = (uint32)mManager -> prefs->ReadInt32("def_encoding", (int32)B_MS_WINDOWS_CONVERSION );
}

//---------------------------------------------------------------------

void BuddyList::DoDeskbarIcon( bool install ) {

	if( install ) {
		BBitmap* active;
		BBitmap* inactive;
		GetBitmapFromResources(active, 962, mManager -> mLocale );
		GetBitmapFromResources(inactive, 963, mManager -> mLocale );
		TrayIcon* view = new TrayIcon(BRect(0, 0, 15, 15), active, inactive);
		
		KDeskbar::Instance()->AddReplicant(view);
		delete view;	// You must delete it!		
		
		KDeskbar::Instance()->TalkToReplicant( new BMessage(GIMICQ_FLASH_OFF) );

	} else {
	
		KDeskbar::Instance()->RemoveReplicant(kViewName);
	
	}
	
	mManager -> deskbarIconInstalled = install;

}

//---------------------------------------------------------------------

void BuddyList::MessageReceived( BMessage* aMsg ) {

	// Some variables needed for various messages

	BListItem* currentItem = NULL;
	int32 curSel;
	time_t the_time;

	switch(aMsg->what) {

		case GIMICQ_SEND_MESSAGE:
		case GIMICQ_CANCEL_SIGN_ON:
		case GIMICQ_SIGN_ON:
		case GIMICQ_PEOPLELIST_COMMIT:
		case GIMICQ_NOW_IDLE:
		case GIMICQ_NOW_ACTIVE:
		case GIMICQ_SEND_IDLE_PULSE:
			//net->PostMessage( aMsg );
			break;

		// GENERELLA GUI MEDDELANDEN!

		case GIMICQ_BUDDY_STATUS_CHANGE:
			mManager -> list->SetPersonStatus( (unsigned)aMsg->FindInt32("uin"),
								   	(ICQStatus)aMsg->FindInt32("status") );
			break;
			
		case GIMICQ_MY_STATUS_CHANGE: {
			
			statView -> status = (ICQStatus)aMsg -> FindInt32("status");
			Window() -> Lock();
			statView -> Invalidate();
			Window() -> Unlock();
			BMessage *msg = new BMessage(ICQ_STATUS_UPDATE);
			msg -> AddInt32( "STATUS", StatusB(statView->status) );
			mManager -> icq -> PostMessage(msg);

			KDeskbar::Instance()->TalkToReplicant( aMsg );
			
			if( statView->status != IS_OFFLINE )
				mManager -> prefs -> WriteInt32( "status", statView -> status );
			break;
		}

		case GIMICQ_OPEN_PREFS: 
			mManager -> windows -> OpenPrefsWindow();

			break;
			
		case GIMICQ_RELOAD_PREF_SETTINGS:
			DoGlobalPrefs();
			LoadPrefs();
			break;
			
		case GIMICQ_ADD_PERSON: {
			uint32 encoding = mManager -> defEncoding;
			if( aMsg->HasInt32("encoding") )
				encoding = aMsg->FindInt32("encoding");
			if( aMsg->HasString("NICKNAME") )
				mManager -> list->AddPerson( (unsigned)aMsg->FindInt32("UIN"), aMsg->FindString("NICKNAME"), IS_OFFLINE, encoding );
			else
				mManager -> list->AddPerson( (unsigned)aMsg->FindInt32("UIN"), NULL, IS_OFFLINE, encoding );
			break;
		}
		
		case GIMICQ_DELETE_PERSONORGROUP:
			if( mManager -> list )
				mManager -> list->RemovePerson( (unsigned)aMsg->FindInt32("UIN") );
			break;

		case GIMICQ_LOGOUT:
			Logout();
			break;
			
		case GIMICQ_PREV_CHAT_WINDOW:
		case GIMICQ_NEXT_CHAT_WINDOW:
			mManager -> windows->SwitchUserWindow( aMsg );
			break;
			
		case GIMICQ_SEARCH_BY_EMAIL: {
			BRect rect( 0, 0, 230, 240 );
			mManager -> windows->MakeDialogFrame( rect, this );
			SearchUserWindow* sW = new SearchUserWindow(rect, this, mManager);
			mManager -> windows -> AddGeneralWindow(sW);
			sW->Show();
			break;
		}
		
		case GIMICQ_CHANGE_PASSWORD: {
			BRect rect( 0, 0, 230, 110 );
			mManager -> windows->MakeDialogFrame( rect, this );
			PasswordChanger* pc = new PasswordChanger( rect, mManager );
			mManager -> windows -> AddGeneralWindow(pc);
			pc->Show();
			break;
		}
		
		case GIMICQ_UPDATE_MY_INFO: {
			BRect rect( 0, 0, 330, 210 );
			mManager -> windows->MakeDialogFrame( rect, this );
			UpdateInfoWindow* bWindow = new UpdateInfoWindow( rect, mManager );
			mManager -> windows -> AddGeneralWindow(bWindow);
			bWindow->Show();
			break;
		}
			
		case GIMICQ_BUDDY_INVOKED:{
			PeopleItem* pItem;
			GroupItem* gItem;
			curSel = onlineview->online->CurrentSelection();
			currentItem = onlineview->online->ItemAt(curSel);
			pItem = dynamic_cast<PeopleItem*>(currentItem);
			gItem = dynamic_cast<GroupItem*>(currentItem);
			if( pItem ) {
				BMessage *msg = new BMessage(GIMICQ_OPEN_USER_WINDOW);
				msg->AddInt32( "wtype", (int32)USER_MESSAGE_TYPE );
				msg->AddInt32( "UIN", (int32)pItem->GetUIN() );
				msg->AddInt32( "encoding", (int32)pItem->GetEncoding() );
				msg->AddString( "nick", pItem->GetDisplayName() );
				msg->AddBool( "minimize", false );
				ChatWindow *userWindow = (ChatWindow *) mManager -> windows->OpenUserWindow( msg );
				userWindow -> ForcePopup();
				delete msg;
			}
			break;
		}
			
		case GIMICQ_OPEN_IM_WINDOW:  // isnt used.
		case GIMICQ_OPEN_USER_WINDOW:
			BWindow *userWindow;
			userWindow = mManager -> windows->OpenUserWindow( aMsg );
			if(aMsg -> FindInt32("wtype") == (int32)USER_MESSAGE_TYPE )
				((ChatWindow *) userWindow ) -> ForcePopup();
			else {
				if(userWindow->Lock()) {
					userWindow->Show();
					userWindow->Unlock();
				}
			}

			KDeskbar::Instance()->TalkToReplicant( aMsg );
			break;

		case GIMICQ_CHANGE_ENCODING:
			if( mManager -> list )
				mManager -> list->SetPersonEncoding( (unsigned)aMsg->FindInt32("UIN"),
									 (uint32)aMsg->FindInt32("encoding") );
			break;

		case GIMICQ_IM_WINDOW_OPENED:
			mManager -> windows->UserWindowOpened( aMsg );
			KDeskbar::Instance()->TalkToReplicant( aMsg );
			break;

		case GIMICQ_IM_WINDOW_CLOSED:
			if( mManager -> windows )
				mManager -> windows->UserWindowClosed( aMsg );
			KDeskbar::Instance()->TalkToReplicant( aMsg );
			break;

		case GIMICQ_ESEARCH_RESULTS_CLOSED:
			mManager -> windows->EmailSearchResultsWindowClosed();
			break;

		case GIMICQ_SET_ONLINE_NOTIFY:
			if( mManager -> list )
				mManager -> list->SetPersonNotify( (unsigned)aMsg->FindInt32("UIN"),
									 (bool)aMsg->FindBool("enabled") );
			break;

		case GIMICQ_UPDATE_INFO:
			mManager -> windows->BroadcastMessage( aMsg, false );
			if( mManager -> list )
				mManager -> list->SetPersonNick( (unsigned)aMsg->FindInt32("UIN"),
									 aMsg->FindString("NICKNAME") );
			break;

		case GIMICQ_TOGGLE_DESKBAR_ICON:
			DoDeskbarIcon( aMsg->FindBool("enabled") );
			break;

		case GIMICQ_GET_UIN_HISTORY:
			{
				unsigned int uin = (unsigned int) aMsg -> FindInt32("UIN");
				HistoryWindow *h = new HistoryWindow(uin, mManager -> own_uin, mManager);
				mManager -> windows -> AddGeneralWindow(h);
//				h -> Show();
			}
			break;
		
		case GIMICQ_ABOUT:
			mManager -> windows -> DoAbout();
			break;



		// ICQ MESSAGES!

		case ICQ_RECONNECTING:
	
			beMenu -> ItemAt(0) -> SetEnabled(false);
			beMenu -> ItemAt(1) -> SetEnabled(false);
			beMenu -> ItemAt(2) -> SetEnabled(false);
			beMenu -> ItemAt(4) -> SetEnabled(false);
			loginMenuItem -> SetEnabled(true);
			logoutMenuItem -> SetEnabled(false);

			statView -> userPopup -> SetEnabled(false);
			
			// remove the tray icon
			if( mManager -> deskbarIconInstalled )
				DoDeskbarIcon( false );
			
			last_user_uin = mManager -> own_uin;
			statView -> status = IS_RECONNECTING;
			Window() -> Lock();
			statView -> Invalidate();
			Window() -> Unlock();
		
			break;

		case ICQ_USER_SEARCH_DONE:
		case ICQ_USER_SEARCH:
			mManager -> windows->ForwardEmailSearchResults( aMsg );
			break;

		case GIMICQ_OPEN_EMAIL_SEARCH:
			mManager -> windows->OpenEmailSearchResultsWindow( aMsg );
			break;

		case ICQ_LOGGED_IN:
			mManager -> LoggedIn = true;			

			LoadPrefs();
			mManager -> windows -> CloseSignOnWindow();

			if(last_user_uin != mManager -> own_uin )
				mManager -> windows -> CloseAllIMWindows();

			mManager -> sounds->PlaySound( WS_LOGON );
			beMenu -> ItemAt(0) -> SetEnabled(true);
//			beMenu -> ItemAt(1) -> SetEnabled(true);
			beMenu -> ItemAt(2) -> SetEnabled(true);
			beMenu -> ItemAt(4) -> SetEnabled(true);
			loginMenuItem -> SetEnabled(false);
			logoutMenuItem -> SetEnabled(true);
			break;
			
		case ICQ_USER_ONLINE:
		case ICQ_USER_STATUS:
			if( mManager -> list )
				mManager -> list->SetPersonStatus( (unsigned)aMsg->FindInt32("UIN"), 
									   StatusA((unsigned)aMsg->FindInt32("STATUS")) );
			break;
			
		case ICQ_USER_OFFLINE:
			if( mManager -> list )
				mManager -> list->SetPersonStatus( (unsigned)aMsg->FindInt32("UIN"), IS_OFFLINE );
			break;
		
		case ICQ_LOGIN_ERROR:
		case ICQ_BAD_PASSWORD:
		case GIMICQ_LOGIN:

			beMenu -> ItemAt(0) -> SetEnabled(false);
			beMenu -> ItemAt(1) -> SetEnabled(false);
			beMenu -> ItemAt(2) -> SetEnabled(false);
			beMenu -> ItemAt(4) -> SetEnabled(false);
			loginMenuItem -> SetEnabled(true);
			logoutMenuItem -> SetEnabled(false);
			
			
			mManager -> windows -> CloseSignOnWindow();
			mManager -> windows -> OpenSignOnWindow();
			break;
				
		case ICQ_LOGGED_OUT:
			
			//Just in case.
			mManager -> windows -> SignOnStop();
			mManager -> list -> Reset();
			mManager -> LoggedIn = false;	


			beMenu -> ItemAt(0) -> SetEnabled(false);
			beMenu -> ItemAt(1) -> SetEnabled(false);
			beMenu -> ItemAt(2) -> SetEnabled(false);
			beMenu -> ItemAt(4) -> SetEnabled(false);
			loginMenuItem -> SetEnabled(true);
			logoutMenuItem -> SetEnabled(false);

			// remove the tray icon
			if( mManager -> deskbarIconInstalled )
				DoDeskbarIcon( false );

			statView -> status = IS_OFFLINE;
			statView -> userPopup -> SetEnabled(false);

			Window() -> Lock();
			statView -> Invalidate();
			Window() -> Unlock();
			
			break;

		case ICQ_USER_EXTINFO:
		case ICQ_USER_INFO:
			aMsg->AddInt32( "wtype", (int32)USER_INFO_TYPE );
			mManager -> windows -> ForwardIncomingMessage( aMsg );
			break;

		case GIMICQ_OUTGOING_URL:
			{
				mManager -> windows -> ForwardIncomingMessage(aMsg);

				time(&the_time);
				aMsg -> AddInt32("uin", aMsg->FindInt32("UIN") );
				aMsg -> AddInt32("own_uin", mManager -> own_uin);
				aMsg -> AddInt64("time",the_time);
				mManager -> history -> Add( aMsg );
			}
			break;
			
		case GIMICQ_ATTEMPT_SEND_IM:  // Comes from ChatWindow.
			mManager -> history -> Add( aMsg );
			break;
			
		case ICQ_TEXT_MESSAGE:
		case ICQ_URL_MESSAGE: {
			
			BMessage *history_msg = new BMessage(aMsg -> what);
	
			time(&the_time);
			int32 uin = aMsg->FindInt32("FROM");
			aMsg->AddInt32( "UIN", uin );
			aMsg->AddInt32( "wtype", (int32)USER_MESSAGE_TYPE );

			history_msg -> AddString("TEXT", aMsg -> FindString("TEXT"));
			if(aMsg -> what == ICQ_URL_MESSAGE)
				history_msg -> AddString("URL", aMsg -> FindString("URL"));

			history_msg -> AddInt32("uin", uin);
			history_msg -> AddInt32("own_uin", mManager -> own_uin);
			history_msg -> AddInt64("time",the_time);
			mManager -> history -> Add( history_msg );
			
			mManager -> list -> SetPersonMessage(uin, true);
			mManager -> windows -> ForwardIncomingMessage( aMsg );

			Window() -> LockLooper();
			onlineview -> online -> Invalidate();
			Window() -> UnlockLooper();

			delete history_msg;
			
			break;
		}
		
		// No more offline messages!
		case ICQ_OFFLINE_DONE:
			mManager -> windows -> BroadcastMessage( aMsg, false );
			break;

		// needs to be fixed for thread-safeness?
		case ICQ_CLIENT_INFO:
			mManager -> own_uin = aMsg->FindInt32("UIN");
			mManager -> p_comm -> SetNick( aMsg->FindString("NICKNAME") );
			break;

		case GIMICQ_ONLINE_NOTIFY:
		case ICQ_UPDATE_REPLY:
		case ICQ_AUTH_REQUEST:
			mManager->p_comm->HandleMessage( aMsg );
			break;

#if 0
		case ICQ_NEW_UIN:
			mManager -> windows->ForwardNewUserMessage( aMsg );
			break;
#endif

//		case ICQ_EXTUPDATE_REPLY:
//			break;
//		case ACQ_ADD_CONTACT:
//			break;
//		case ICQ_PASSWORD_REPLY:
//			break;

		case GIMICQ_SET_LOGIN_STEP_COUNT:
		case GIMICQ_LOGIN_STEP:
		case GIMICQ_LOGIN_FAILURE:
			
			mManager -> windows -> SendSignOnMessage( aMsg );
			break;

		default:
			BView::MessageReceived(aMsg);
			break;
	}
}

// #pragma mark -
//-------------------------------------------------------------------------
// BuddyListStatusItem draws the statusBitmap beneath the items in the status menu
//=========================================================================


BuddyListStatusItem::BuddyListStatusItem(const char *label, BBitmap *bitmap, char shortcut, uint32 _modifiers)
			: BMenuItem(label,NULL,shortcut,_modifiers)
{
	fBitmap = bitmap;
}


void BuddyListStatusItem::DrawContent()
{
	// get location and size of the menu item
	BPoint point = ContentLocation();
	float  width,height;
	GetContentSize(&width,&height);
	
	BView *menu = Menu();
	
	if (fBitmap != NULL)
	{
		BRect bounds(fBitmap->Bounds());
		
		menu->PushState();
		menu->SetDrawingMode(B_OP_ALPHA);
		menu->DrawBitmap(fBitmap,BPoint(point.x,point.y + (height - bounds.Height())/2));
		menu->PopState();

		point.x += bounds.Width()+5;
	}
	// let the inherited class handle the text drawing
	menu->MovePenTo(point);
	BMenuItem::DrawContent();
}


// #pragma mark -
//-------------------------------------------------------------------------

//=========================================================================

BuddyListStatusView::BuddyListStatusView(BRect rect, GManager *the_manager)
	   	   : BView(rect, "BuddyListStatus", B_FOLLOW_LEFT_RIGHT | B_FOLLOW_BOTTOM, B_WILL_DRAW | B_PULSE_NEEDED )
{
	PRINT(("BuddyListStatusView::BuddyListStatusView\n"));
	
	mManager = the_manager;
	mLocale = mManager -> mLocale;
	
	SetFontSize(12);
	SetViewColor( 216, 216, 216 );
	//status = (ICQStatus) mManager -> prefs->ReadInt32( "status", IS_AVAILABLE );
	status = IS_OFFLINE;
	wasIdle = false;
	
	userPopup = new BPopUpMenu( "userpopup", false );
	for(int i = 0;i < 7;i++)
	{
		char text[11] = "menuitem2 ";
		
		text[9] = i + '0';
		userPopup->AddItem(new BuddyListStatusItem(_(text),mManager->statBitmaps[i],i + '1'));
	}
	//userPopup -> SetEnabled(false);
}

//-------------------------------------------------------------------------

void BuddyListStatusView::DetachedFromWindow(void) {
	PRINT(("BuddyListStatusView::DetachedFromWindow\n"));
	
}

void BuddyListStatusView::Draw( BRect ) {
//	PRINT(("BuddyListStatusView::Draw\n"));

	BRect frame = Bounds();

	SetHighColor( 156, 154, 156 );		// dark grey
	BPoint start = frame.LeftTop();
	BPoint end = frame.RightTop();
	StrokeLine( start, end );
	
	BString status_text = " ";
/*	status_text += _("text20");
	status_text += ":";
*/
	status_text << _("text20") << ":";
		
	SetHighColor( 0, 0, 0 );
	SetLowColor( 222, 219, 222 );
	MovePenTo(frame.left + 5, frame.bottom-1);
	SetFont(be_bold_font);
	SetFontSize(12);
	DrawString( status_text.String() );
	MovePenTo(frame.left + 53, frame.bottom-1);
	SetFont(be_plain_font);	
	SetFontSize(12);
	
	switch( status ) {
		case IS_AVAILABLE:
			DrawString(	_("menuitem20") );
			break;
		case IS_FREEFORCHAT:
			DrawString( _("menuitem21") );
			break;
		case IS_AWAY:
			DrawString( _("menuitem22") );
			break;
		case IS_EXTENDEDAWAY:
			DrawString( _("menuitem23") );
			break;
		case IS_OCCUPIED:
			DrawString( _("menuitem24") );
			break;
		case IS_DND:
			DrawString( _("menuitem25") );
			break;
		case IS_INVISIBLE:
			DrawString( _("menuitem26") );
			break;
		case IS_OFFLINE:
			DrawString( _("text21") );
			break;
		case IS_RECONNECTING:
			DrawString( _("text22") );
			break;
		case IS_PENDING:
			DrawString( "you shouldn't see this" );
			break;
	}
}

//-------------------------------------------------------------------------

void BuddyListStatusView::MouseDown( BPoint ) {
	PRINT(("BuddyListStatusView::MouseDown\n"));

	BMenuItem* selected;
	BMessage* sndMessage;
	int32 option;

	BPoint cursor( 53, 1 );
	ConvertToScreen( &cursor );
	selected = userPopup->Go( cursor );
	if( !selected )
		return;
	option = userPopup->IndexOf(selected);
	sndMessage = new BMessage(GIMICQ_MY_STATUS_CHANGE);
	sndMessage->AddInt32( "status", option );
	mManager -> windows -> SendBuddyListMessage(sndMessage );
}

//-------------------------------------------------------------------------

void BuddyListStatusView::Pulse() {

	bigtime_t idle;
	
	if( doIdleTime ) {
		
		// get the idle time
		idle = idle_time();
		
		// bigger than 10 minutes? (600 seconds)
		if( idle > 600000000 && idle < 1200000000 ) {
			if( status != IS_AWAY ) {
				oldStatus = status;
				BMessage* msg = new BMessage(GIMICQ_MY_STATUS_CHANGE);
				msg->AddInt32( "status", IS_AWAY );
				mManager -> windows -> SendBuddyListMessage(msg );
			} else if( !wasIdle )
				oldStatus = IS_AWAY;
			wasIdle = true;
		}
		
		// bigger than 20 minutes? (1200 seconds)
		else if( idle >= 1200000000 && status != IS_EXTENDEDAWAY ) {
			wasIdle = true;
			BMessage* msg = new BMessage(GIMICQ_MY_STATUS_CHANGE);
			msg->AddInt32( "status", IS_EXTENDEDAWAY );
			mManager -> windows -> SendBuddyListMessage(msg );
		}
		
		// did we just get "un-idle"?
		else if( wasIdle && idle < 2000000 ) {
			wasIdle = false;
			BMessage* msg = new BMessage(GIMICQ_MY_STATUS_CHANGE);
			msg->AddInt32( "status", oldStatus );
			mManager -> windows -> SendBuddyListMessage(msg );
		}
	}

}


// #pragma mark -
//=========================================================================

BuddyListInfoView::BuddyListInfoView(BRect rect, GManager *the_manager)
	   	   : BView(rect, "BuddyListInfoView", B_FOLLOW_LEFT_RIGHT, B_WILL_DRAW | B_PULSE_NEEDED )
{
	PRINT(("BuddyListInfoView::BuddyListInfoView\n"));
	mManager = the_manager;
	SetFont(be_bold_font);
	SetFontSize(24);
	SetViewColor( 216, 216, 216 );
	
	// reset the idle counter
	idleCount = 0;
}

//-------------------------------------------------------------------------

// update the change list... pulse is used on this particular view because it is visible
//   whenever the buddylist is visible... pulse stops occasionally for the others
void BuddyListInfoView::Pulse() {

	// make any changes that need to be done in the list

}

//-------------------------------------------------------------------------

void BuddyListInfoView::DoIdleStuff() {
	PRINT(("BuddyListInfoView::DoIdleStuff\n"));

}


// #pragma mark -
//=========================================================================

BuddyListView::BuddyListView(BRect rect, GManager *mManager)
	   	   : BView(rect, "BuddyListView", B_FOLLOW_ALL, B_WILL_DRAW | B_PULSE_NEEDED )
{
	PRINT(("BuddyListView::BuddyListView\n"));
	SetViewColor( 216, 216, 216 );
	BFont baseFont; 
	baseFont.SetSize( 12.0 );
	
	// Add the outline control
	BRect r = Bounds();
	//r.InsetBy( 1.0, 1.0 );
	r.right -= B_V_SCROLL_BAR_WIDTH;
	r.top++;
	online = new ClickableListView( r, "online_list", B_SINGLE_SELECTION_LIST, B_FOLLOW_ALL, mManager );
	online->SetFont( &baseFont );
	
	// set up the scroller
	scroller = new BScrollView( "scroller2", online, B_FOLLOW_ALL, 0, false, true, B_NO_BORDER );
	AddChild( scroller );
}

//-------------------------------------------------------------------------

void BuddyListView::Draw( BRect )
{
//	PRINT(("BuddyListView::Draw\n"));
	BRect r = Bounds();
	SetHighColor( 156, 154, 156 );		// dark grey
	StrokeRect( r );
}

//-------------------------------------------------------------------------

void BuddyListView::DoResizeStuff( float width, float height ) {

	PRINT(("BuddyListView::BuddyListView\n"));
	ResizeTo( width, height );
	
	// Ugh... resizing the scroller stuff is a pain!
	online->ResizeTo( width - B_V_SCROLL_BAR_WIDTH - 2, height - 2 );
	scroller->ResizeTo( width - 2, height - 2 );
	scroller->ScrollBar(B_VERTICAL)->MoveTo( width - B_V_SCROLL_BAR_WIDTH - 1, -1 );
	scroller->ScrollBar(B_VERTICAL)->ResizeTo( B_V_SCROLL_BAR_WIDTH, height );
	scroller->ScrollBar(B_VERTICAL)->SetValue(0);
}


// #pragma mark -
//=========================================================================

ClickableListView::ClickableListView( BRect frame, const char* name, list_view_type type, uint32 flags, GManager *the_manager )
				  					: BOutlineListView( frame, name, type, flags )
{
	mManager = the_manager;
	Locale *mLocale = mManager -> mLocale;
	BString temp;
		
	userPopup = new BPopUpMenu( "userpopup", false );
	
	temp = _("menuitem40");
	temp += B_UTF8_ELLIPSIS;
	userPopup->AddItem(new BMenuItem( temp.String() , NULL ) );

	temp = _("menuitem41");
	temp += B_UTF8_ELLIPSIS;
	userPopup->AddItem(new BMenuItem( temp.String() , NULL ) );

	temp = _("menuitem42");
	temp += B_UTF8_ELLIPSIS;
	userPopup->AddItem(new BMenuItem( temp.String() , NULL ) );

	temp = _("menuitem43");
	temp += B_UTF8_ELLIPSIS;
	userPopup->AddItem(new BMenuItem( temp.String() , NULL ) );

	userPopup->AddSeparatorItem();
	userPopup->AddItem(new BMenuItem( _("menuitem44") , NULL) );

	userPopup->AddItem( onItem = new BMenuItem( _("menuitem45") , NULL ) );
	userPopup->AddSeparatorItem();

	userPopup->AddItem(new BMenuItem( _("menuitem46") , NULL ) );
	
}

//-------------------------------------------------------------------------

ClickableListView::~ClickableListView() {
	delete userPopup;
}

//-------------------------------------------------------------------------

void ClickableListView::MessageReceived(BMessage *msg) {

	switch(msg -> what) {
		case B_SIMPLE_DATA: {
			int drop_type = 0;
			BString filetype;
			entry_ref the_ref;
			BPoint point, mess_point;
			
			if(msg -> FindString("be:filetypes", &filetype) == B_OK) {
				if(filetype == "application/x-vnd.Be-bookmark") {
					if(msg -> HasString("be:url"))
						drop_type = BOOKMARK;
				} // if filetype
			} // if FindString			
			else if(msg -> FindRef("refs", &the_ref) == B_OK) {				
				BNode the_node(&the_ref);
				BNodeInfo info(&the_node);
				char type[100];

				if(info.GetType(type) == B_OK) {
					BMimeType mime_type(type);
					if(mime_type == "application/x-vnd.Be-bookmark") {
						drop_type = BOOKMARKFILE;
					}
				}
			}
			
			if(drop_type != 0) {
				int32 clickIndex;
				PeopleItem* dropItem;
	
				if(msg -> FindPoint("_drop_point_", &mess_point)!= B_OK)
					return;
				point = ConvertFromScreen(mess_point);
				clickIndex = IndexOf(point);
				if( clickIndex < 0 )			// not an item?
					return;
				Select(clickIndex);
		
				dropItem = dynamic_cast<PeopleItem*>( ItemAt(clickIndex) );
				if( !dropItem )
					return;
				
				switch(drop_type) {
					case BOOKMARKFILE:
					case BOOKMARK: {
						uint32 enc;
						if( !(mManager -> list && mManager -> list->FindEncodingForUIN(dropItem->GetUIN(),enc)) )
							enc = mManager -> defEncoding;
						BRect r( 0, 0, 256, 158 );
						mManager -> windows->MakeDialogFrame( r, Window() );
						SendURLWindow* su = new SendURLWindow( r, dropItem->GetUIN(),
												dropItem->GetDisplayName(), enc, mManager );
						su -> MessageReceived(msg);
						mManager -> windows -> AddGeneralWindow(su);
						su->Show();
					}
				}
			}
			break;
		}
		default:
				BOutlineListView::MessageReceived(msg);
				break;
	}
}

void ClickableListView::MouseMoved(BPoint point, uint32 transit, const BMessage *msg) {

	if(msg != NULL) {
		if(msg -> what == B_SIMPLE_DATA) {
			bool can_drop = false;
			BString filetype;
			entry_ref the_ref;
			
			if(msg -> FindString("be:filetypes", &filetype) == B_OK) {
				if(filetype == "application/x-vnd.Be-bookmark") {
					can_drop = true;
				} // if filetype
			} // if FindString			
			else if(msg -> FindRef("refs", &the_ref) == B_OK) {				
				BNode the_node(&the_ref);
				BNodeInfo info(&the_node);
				char type[100];

				if(info.GetType(type) == B_OK) {
					BMimeType mime_type(type);
					if(mime_type == "application/x-vnd.Be-bookmark")
						can_drop = true;
//					else if (mime_type == "text/plain")
//						can_drop = true;
				
				}
			}


			if(can_drop) {
				Select(IndexOf(point));
			}
		} // if what
	} // if mManager -> LoggedIn
}

void ClickableListView::MouseDown( BPoint cursor ) {

	uint32 buttons;
	int32 clickIndex;
	PeopleItem* clickItem;
	BMessage* popupMsg;
	BMenuItem* selected;
	int32 option;

	// get the mouse position
	GetMouse( &cursor, &buttons );
		
	// it was a right mouse button click, popup the menu
	if( buttons & B_SECONDARY_MOUSE_BUTTON ) {
		
		// get the index of the item the user clicked on, and select it
		clickIndex = IndexOf(cursor);
		if( clickIndex < 0 )			// not an item?
			return;
		Select(clickIndex);

		// get a pointer to the item, and check to see if it's a person or not
		clickItem = dynamic_cast<PeopleItem*>( ItemAt(clickIndex) );
		if( !clickItem )
			return;

		// it's all good... display the popup menu
		ConvertToScreen( &cursor );
		onItem->SetMarked( clickItem->GetOnlineNotify() );		
		selected = userPopup->Go( cursor );

		// if the user selected something, take action based on the choice
		if( selected ) {
			option = userPopup->IndexOf(selected);
			popupMsg = new BMessage(GIMICQ_OPEN_USER_WINDOW);
			popupMsg->AddInt32( "UIN", (int32)clickItem->GetUIN() );
			popupMsg->AddPointer( "poswindow", Window() );
			popupMsg->AddString( "nick", clickItem->GetDisplayName() );
			switch( option ) {
				case 0: popupMsg->AddInt32( "wtype", (int32)USER_MESSAGE_TYPE );
					mManager -> windows -> SendBuddyListMessage( popupMsg );
					break;
				case 1: {
					delete popupMsg;
					uint32 enc;
					if( !(mManager -> list && mManager -> list->FindEncodingForUIN(clickItem->GetUIN(),enc)) )
						enc = mManager -> defEncoding;
					BRect r( 0, 0, 256, 158 );
					mManager -> windows->MakeDialogFrame( r, Window() );
					SendURLWindow* su = new SendURLWindow( r, clickItem->GetUIN(),
											clickItem->GetDisplayName(), enc, mManager );
					mManager -> windows -> AddGeneralWindow(su);
					su->Show();
					break;
				}
				case 2:
					popupMsg->AddInt32( "wtype", (int32)USER_INFO_TYPE );
					mManager -> windows -> SendBuddyListMessage( popupMsg );
					break;
				case 3:
					{
					delete popupMsg;
					BMessage* msg = new BMessage(GIMICQ_GET_UIN_HISTORY);
					msg->AddInt32( "UIN", clickItem->GetUIN() );
					mManager -> windows -> SendBuddyListMessage( msg );
					}
					break;
				case 5: {
					delete popupMsg;
					BMessage* msg = new BMessage(ICQ_GRANT_AUTH);
					msg->AddInt32( "UIN", clickItem->GetUIN() );
//					mManager -> windows -> SendBuddyListMessage( msg );
					mManager -> icq -> PostMessage( msg );
					break;
				}
				case 6: {
					BMessage* msg = new BMessage(GIMICQ_SET_ONLINE_NOTIFY);
					msg->AddInt32( "UIN", clickItem->GetUIN() );
					msg->AddBool( "enabled", !clickItem->GetOnlineNotify() );
					mManager -> windows -> SendBuddyListMessage( msg );
					break;
				}
				case 8: {
					BMessage* msg = new BMessage(GIMICQ_DELETE_PERSONORGROUP);
					msg->AddInt32( "UIN", clickItem->GetUIN() );
					mManager -> windows -> SendBuddyListMessage( msg );
					break;
				}
			}
		}
	} else
		BOutlineListView::MouseDown( cursor );
}

//=========================================================================
