#include <Debug.h>
#include "ChatWindow.h"

#include <Application.h>
#include <Box.h>
#include <Rect.h>
#include <Point.h>
#include <Message.h>
#include <MessageQueue.h>
#include <Handler.h>
#include <MenuBar.h>
#include <Menu.h>
#include <MenuItem.h>
#include <ScrollView.h>
#include <Font.h>
#include <String.h>

#include <stdlib.h>
#include <string.h>

#include "DataContainer.h"
#include "GButton.h"
#include "FancyView.h"
#include "HTMLView.h"
#include "KDeskbar.h"
#include "SendURL.h"
#include "HTMLStuff.h"
#include "Say.h"
#include "HistoryIO.h"
#include "MiscStuff.h"
#include "GManager.h"
#include "ProtocolManager.h"
#include "WindowManager.h"
#include "Prefs.h"
#include "History.h"
#include "Gicq.h"
#include "classSoundMaster.h"
#include "PeopleData.h"
#include "Locale.h"


#warning To remove a lot of debug messages.
#define PRINT(m)

ChatWindow::ChatWindow( uint32 u, const char* nick, BRect frame, bool minimize, uint32 enc, GManager *the_manager )
			: BWindow(frame, nick, B_TITLED_WINDOW, 0, B_ALL_WORKSPACES )
{

	PRINT(("ChatWindow::ChatWindow\n"));
	mManager = the_manager;
	mLocale = mManager -> mLocale;

	strcpy( displayName, nick );
	uin = u;
	mFinnishedUpdating = true;
	popupOnReceive = false;
	hasNewMessage = false;
	prefixNewMessages = false;
	enterIsNewline = false;
	closeAfterSend = false;
	isFlashing = false;
	
	lastWidth = lastHeight = -1;
	
	_InitWindow();
	srand(unsigned(time(NULL)));
	
	// Tell the main app that the window has been opened
	BMessage* sendMessage = new BMessage( GIMICQ_IM_WINDOW_OPENED );
	sendMessage->AddInt32( "wtype", (int32)USER_MESSAGE_TYPE );
	sendMessage->AddPointer( "new_window", this );
	sendMessage->AddInt32( "uin", (int32)uin );
	mManager -> windows -> SendBuddyListMessage( sendMessage );
	
	// start out minimized?
	if( minimize && closeAfterSend )
		Minimize(true);

	Hide();
	Show();

	SetEncoding( enc );	

	if( !mManager->list->HasPerson( uin ) ) {
		ForcePopup();
	}

}

//-----------------------------------------------------

// Initialize the window.
void ChatWindow::_InitWindow(void) {

	PRINT(("ChatWindow::_InitWindow\n"));
	BRect r;
	BMenu *menu;
	BMenuItem *item;
	float HeightCalc;
	
	BString temp;
	
	BString mess_str;
	mess_str << "(" << _("text40") << ")";
	
	// Add the menu bar
	r = Bounds();
	menubar = new BMenuBar(r, "menu_bar");
	AddChild(menubar);
	
	// add the "last message" view
	BRect lastRect = Bounds();
	lastRect.top = menubar -> Bounds().bottom + 1.0;
	lastRect.bottom = lastRect.top + 13;
	lastView = new PointerStringView( lastRect, "lastmsg", mess_str.String() , B_FOLLOW_TOP | B_FOLLOW_LEFT_RIGHT,
								B_WILL_DRAW | B_FULL_UPDATE_ON_RESIZE );
	lastView->SetViewColor( 125, 125, 125 );
	lastView->SetLowColor( 125, 125, 125 );
	//lastView->SetFont( be_bold_font );
	lastView->SetFontSize(10);
	lastView->SetHighColor( 255, 255, 255 );
	lastView->SetAlignment( B_ALIGN_RIGHT );
	AddChild( lastView );
	
	// Set up the Divider
	divider = 0.73;
	topUIMargin = menubar->Bounds().bottom + lastRect.Height() + 1.0;
	midUIMargin = 9;
	botUIMargin = 37;
	totalUIMargin = topUIMargin + midUIMargin + botUIMargin;

	// Set up the view rectangles
	BRect textframe = Bounds();
	HeightCalc = textframe.bottom - textframe.top - totalUIMargin;
	textframe.right -= B_V_SCROLL_BAR_WIDTH;
	BRect editframe = textframe;		
	textframe.top = topUIMargin;
	textframe.bottom = HeightCalc*divider + topUIMargin;	
	editframe.top = textframe.bottom + midUIMargin;
	editframe.bottom -= botUIMargin;

	// Add the text (conversation) BHTMLView
	BRect textrect = textframe;
	textrect.OffsetTo(B_ORIGIN);
	textrect.InsetBy( 2.0, 2.0 );
	textview = new FancyTextView( textframe, "text_view", mManager , textrect,  B_FOLLOW_ALL,
								  B_PULSE_NEEDED | B_WILL_DRAW | B_NAVIGABLE_JUMP );
	AddChild(textscrollview = new BScrollView("text_scroll_view", textview,
				B_FOLLOW_LEFT_RIGHT, 0, false, true, B_NO_BORDER));
	textview->SetDoesUndo(true);
	textview->SetViewColor( 242, 242, 242 );
	
	// Add the message editing BHTMLView
	BRect editrect = editframe;
	editrect.OffsetTo(B_ORIGIN);
	editrect.InsetBy( 2.0, 2.0 );
	editview = new HTMLView( true, editframe, "edit_view", editrect,
				B_FOLLOW_ALL_SIDES, B_WILL_DRAW | B_PULSE_NEEDED | B_NAVIGABLE );
	AddChild(editscrollview = new BScrollView("edit_scroll_view", editview,
				B_FOLLOW_LEFT_RIGHT, 0, false, true, B_NO_BORDER));
	editview->SetDoesUndo(true);
	editview->SetMaxBytes( 450 );
	editview->ResetFontToBase();
	editview->MakeFocus(true);	

	// Add the divider bar (currently for looks only)
	BRect divframe = Bounds();
	divframe.top = textframe.bottom + 1;
	divframe.bottom = editframe.top - 1;
	divframe.right += 1;
	dragger = new BBox( divframe, "dragger", B_FOLLOW_LEFT_RIGHT, 
					   B_WILL_DRAW | B_FRAME_EVENTS | B_FULL_UPDATE_ON_RESIZE, B_FANCY_BORDER );		
	AddChild( dragger );
	
	// Add the bottom view
	BRect btRect = Bounds();
	btRect.top = btRect.bottom - botUIMargin;
	btView = new BottomView( btRect, mManager );
	AddChild( btView );

	// Add the Edit menu to the menu bar
	menu = new BMenu( _("menu1") );
	menu->AddItem(miUndo=new BMenuItem( _("menuitem60") , new BMessage(B_UNDO), 'Z'));
	miUndo->SetTarget(editview);
	menu->AddSeparatorItem();
	menu->AddItem(miCut=new BMenuItem( _("menuitem61") , new BMessage(B_CUT), 'X'));
	miCut->SetTarget(editview);
	menu->AddItem(miCopy=new BMenuItem( _("menuitem62") , new BMessage(B_COPY), 'C'));
	miCopy->SetTarget(editview);
	menu->AddItem(miPaste=new BMenuItem( _("menuitem63") , new BMessage(B_PASTE), 'V'));
	miPaste->SetTarget(editview);
	menu->AddSeparatorItem();
	menu->AddItem(miSelectAll=new BMenuItem( _("menuitem64") , new BMessage(B_SELECT_ALL), 'A'));
	miSelectAll->SetTarget(editview);
	menubar->AddItem(menu);	
	
	// Add the View menu to the menubar
	menu = new BMenu( _("menu2") );

	BMenu* workspaceView = new BMenu( _("menu3") );
	workspaceView->SetRadioMode( true );
	miThisWorkspace = new BMenuItem( _("menuitem80") , new BMessage(CHATWINDOW_SHOW_IN_THIS_WORKSPACE));
	workspaceView->AddItem( miThisWorkspace );
	workspaceView->AddItem( miAllWorkspaces = new BMenuItem( _("menuitem81") , new BMessage(CHATWINDOW_SHOW_IN_ALL_WORKSPACES)) );
	menu->AddItem(workspaceView);	

	menu->AddItem(miPopupOnReceive = new BMenuItem( _("menuitem90") , new BMessage(CHATWINDOW_POPUP_ON_RECEIVE), 'P'));
	menu->AddItem(miEnableLinks = new BMenuItem( _("menuitem91") , new BMessage(CHATWINDOW_ENABLE_LINKS)));
	//menu->AddItem(new BMenuItem("Previous Chat Window", new BMessage(GIMICQ_PREV_CHAT_WINDOW), ','));
	//menu->AddItem(new BMenuItem("Next Chat Window", new BMessage(GIMICQ_NEXT_CHAT_WINDOW), '.'));
	menu->AddItem(item=new BMenuItem( _("menuitem92") , new BMessage(GIMICQ_OPEN_BUDDY_LIST), '/'));
	menu->AddSeparatorItem();
	menu->AddItem(new BMenuItem( _("menuitem93") , new BMessage(MENU_FILE_CLOSE), 'W'));
	item->SetTarget(APP_SIGNATURE);
	menubar->AddItem(menu);
	
	// Add the <ScreenName> menu to the menubar
	
	
	BMenuItem* buddyItem;
	nameMenu = new BMenu( displayName );

	temp = _("menuitem100");
	temp += B_UTF8_ELLIPSIS;
	nameMenu->AddItem(buddyItem = new BMenuItem( temp.String() ,
							  new BMessage(GIMICQ_TRY_ADD_PERSON) ));

	
	temp = _("menuitem101");
	temp += B_UTF8_ELLIPSIS;
	nameMenu->AddItem(buddyItem = new BMenuItem( temp.String() ,
							  new BMessage(GIMICQ_GET_PERSON_INFO) ));


	temp = _("menuitem102");
	temp += B_UTF8_ELLIPSIS;
	nameMenu->AddItem(buddyItem = new BMenuItem( temp.String() ,
							  new BMessage(GIMICQ_ATTEMPT_SEND_URL) ));
	nameMenu->AddSeparatorItem();
	BMenu *useEncodingMenu = new BMenu( _("menu4") );
	useEncodingMenu ->SetRadioMode( true );
	BMessage* encMsg = new BMessage(GIMICQ_CHANGE_ENCODING);
	encMsg->AddInt32( "encoding", (int32)B_ISO2_CONVERSION );
	useEncodingMenu ->AddItem(e1 = new BMenuItem( _("menuitem110") , encMsg) );
	encMsg = new BMessage(GIMICQ_CHANGE_ENCODING);
	encMsg->AddInt32( "encoding", (int32)B_ISO5_CONVERSION );
	useEncodingMenu ->AddItem(e2 = new BMenuItem( _("menuitem111") , encMsg) );
	encMsg = new BMessage(GIMICQ_CHANGE_ENCODING);
	encMsg->AddInt32( "encoding", (int32)B_KOI8R_CONVERSION );
	useEncodingMenu ->AddItem(e3 = new BMenuItem( _("menuitem112") , encMsg) );
	encMsg = new BMessage(GIMICQ_CHANGE_ENCODING);
	encMsg->AddInt32( "encoding", (int32)B_MS_DOS_866_CONVERSION );
	useEncodingMenu ->AddItem(e4 = new BMenuItem( _("menuitem113") , encMsg) );
	encMsg = new BMessage(GIMICQ_CHANGE_ENCODING);
	encMsg->AddInt32( "encoding", (int32)B_MS_WINDOWS_1251_CONVERSION );
	useEncodingMenu ->AddItem(e5 = new BMenuItem( _("menuitem114") , encMsg) );
	encMsg = new BMessage(GIMICQ_CHANGE_ENCODING);
	encMsg->AddInt32( "encoding", (int32)B_ISO7_CONVERSION );
	useEncodingMenu ->AddItem(e6 = new BMenuItem( _("menuitem115") , encMsg) );
	encMsg = new BMessage(GIMICQ_CHANGE_ENCODING);
	encMsg->AddInt32( "encoding", (int32)B_SJIS_CONVERSION );
	useEncodingMenu ->AddItem(e7 = new BMenuItem( _("menuitem116") , encMsg) );
	encMsg = new BMessage(GIMICQ_CHANGE_ENCODING);
	encMsg->AddInt32( "encoding", (int32)B_EUC_CONVERSION );
	useEncodingMenu ->AddItem(e8 = new BMenuItem( _("menuitem117") , encMsg) );
	encMsg = new BMessage(GIMICQ_CHANGE_ENCODING);
	encMsg->AddInt32( "encoding", (int32)B_UNICODE_CONVERSION );
	useEncodingMenu ->AddItem(e9 = new BMenuItem( _("menuitem118") , encMsg) );
	encMsg = new BMessage(GIMICQ_CHANGE_ENCODING);
	encMsg->AddInt32( "encoding", (int32)B_ISO1_CONVERSION );
	useEncodingMenu ->AddItem(e10 = new BMenuItem( _("menuitem119") , encMsg) );
	encMsg = new BMessage(GIMICQ_CHANGE_ENCODING);
	encMsg->AddInt32( "encoding", (int32)B_MAC_ROMAN_CONVERSION );
	useEncodingMenu ->AddItem(e11 = new BMenuItem( _("menuitem120") , encMsg) );
	encMsg = new BMessage(GIMICQ_CHANGE_ENCODING);
	encMsg->AddInt32( "encoding", (int32)B_MS_WINDOWS_CONVERSION );
	useEncodingMenu ->AddItem(e12 = new BMenuItem( _("menuitem121") , encMsg) );
	nameMenu->AddItem( useEncodingMenu  );
	menubar->AddItem(nameMenu);

	// Set the BTextControl's base font	and attributes
	rgb_color black;
	black.red = black.green = black.blue = 0;
	chatFont.SetSize( 12.0 );
	textview->SetBaseFontAndColor( chatFont, black );
	textview->MakeEditable( false );
	textview->SetStylable( true );
	editview->SetBaseFontAndColor( chatFont, black );
	editview->MakeEditable( true );
	editview->SetStylable( true );
	
	// load the default prefs settings
	if( mManager -> prefs->ReadBool( "cw_aw", false ) ) {
		SetWorkspaces( B_ALL_WORKSPACES );
		miAllWorkspaces->SetMarked( true );
	} else {
		SetWorkspaces( B_CURRENT_WORKSPACE );
		miThisWorkspace->SetMarked( true );
	}
	if( mManager -> prefs->ReadBool( "cw_pu", false ) ) {
		popupOnReceive = true;
		miPopupOnReceive->SetMarked( true );
	}
	if( mManager -> prefs->ReadBool( "cw_cs", false ) ) {
		closeAfterSend = true;
	}
	if( mManager -> prefs->ReadBool( "cw_lk", true ) ) {
		linksEnabled = true;
		textview->SetShowLinks( true );
		miEnableLinks->SetMarked( true );
	} else
		textview->SetShowLinks( false );
	prefixNewMessages = mManager -> prefs->ReadBool( "cw_p>", false );
	enterIsNewline = mManager -> prefs->ReadBool( "cw_en", true );

	editview -> MakeFocus( true );

}

//-----------------------------------------------------

void ChatWindow::SetEncoding( uint32 encoding ) {

	PRINT(("ChatWindow::SetEncoding\n"));

	LockLooper();

	// set the encoding
	useEncoding = encoding;	
	switch( encoding ) {
		case B_ISO2_CONVERSION:
			e1->SetMarked(true);
			break;
		case B_ISO5_CONVERSION:
			e2->SetMarked(true);
			break;
		case B_KOI8R_CONVERSION:
			e3->SetMarked(true);
			break;
		case B_MS_DOS_866_CONVERSION:
			e4->SetMarked(true);
			break;
		case B_MS_WINDOWS_1251_CONVERSION:
			e5->SetMarked(true);
			break;
		case B_ISO7_CONVERSION:
			e6->SetMarked(true);
			break;
		case B_SJIS_CONVERSION:
			e7->SetMarked(true);
			break;
		case B_EUC_CONVERSION:
			e8->SetMarked(true);
			break;
		case B_UNICODE_CONVERSION:
			e9->SetMarked(true);
			break;
		case B_ISO1_CONVERSION:
			e10->SetMarked(true);
			break;
		case B_MAC_ROMAN_CONVERSION:
			e11->SetMarked(true);
			break;
		case B_MS_WINDOWS_CONVERSION:
			e12->SetMarked(true);
			break;
	}
	
	// clear the textview
	textview->Clear();
	UnlockLooper();
	
	// go through all the saved entries
	mFinnishedUpdating = false;
	mManager->history->ReadAll( uin, mManager->own_uin, this );
	
}

//-----------------------------------------------------

// Adjust the size of the BTextView's text rectangles
// when the window is resized.
void ChatWindow::FrameResized(float width, float height) {

	PRINT(("ChatWindow::FrameResized\n"));

	// resize the windows themselves
	float newHeight = (height - totalUIMargin) * divider;
	textscrollview->ResizeTo( width, newHeight );
	editscrollview->ResizeTo( width, height - totalUIMargin - newHeight );
	editscrollview->MoveTo( 0, topUIMargin + midUIMargin + newHeight );
	dragger->MoveTo( 0, topUIMargin + newHeight + 1 );
	
	// cheesy hack alert!	
	if( lastWidth == width && lastHeight == height )
		;//
	lastWidth = width;
	lastHeight = height;
		
	// set the new text rects
	BRect textrect = textview->Bounds();
	textrect.InsetBy( 2.0, 2.0 );
	textview->SetTextRect( textrect );
	textrect = editview->Bounds();
	textrect.InsetBy( 2.0, 2.0 );
	editview->SetTextRect( textrect );
	
#warning not calling BWindow::FrameResized().
}

//-----------------------------------------------------

void ChatWindow::WindowActivated( bool activated ) {

	PRINT(("ChatWindow::WindowActivated\n"));
	// Let the Contact List know you activated the chatwindow.
	
	if( mManager -> LoggedIn )
		mManager -> list -> SetPersonMessage(uin, false);

	if( activated && isFlashing ) {
		BMessage* msg2 = new BMessage(GIMICQ_FLASH_OFF);
		msg2->AddInt32( "UIN", uin );
		KDeskbar::Instance()->TalkToReplicant( msg2 );	
		isFlashing = false;
	}
	
	// if that was a new message we just responded to,
	//  then take out the '√' from the titlebar
	if( activated && hasNewMessage && prefixNewMessages ) {
		SetTitle( displayName );
		hasNewMessage = false;
	}

	BWindow::WindowActivated( activated );
}

//-----------------------------------------------------

void ChatWindow::InformDeskbar(uint32 what) {
	PRINT(("ChatWindow::InformDeskbar\n"));
/*	if( closeAfterSend && !IsActive() ) {
		BMessage* msg2 = new BMessage(GIMICQ_FLASH_ON);
		msg2->AddInt32( "UIN", uin );
		msg2->AddString( "nick", displayName );
		KDeskbar::Instance()->TalkToReplicant( msg2 );
		isFlashing = true;
	}*/
	
	/* Should only inform the deskbar if it these kind of messages. */
	if(what != ICQ_TEXT_MESSAGE && what != ICQ_URL_MESSAGE)
		return;

	bool doIt = false;
	
	if( popupOnReceive )
		doIt = false;
	else if( closeAfterSend )
		doIt = true;
	else if( !IsActive() )
		doIt = true;
		
	if( doIt ) {
		BMessage* msg2 = new BMessage(GIMICQ_FLASH_ON);
		msg2->AddInt32( "UIN", uin );
		msg2->AddString( "nick", displayName );
		KDeskbar::Instance()->TalkToReplicant( msg2 );
		isFlashing = true;
	}
}

//-----------------------------------------------------

void ChatWindow::MessageReceived( BMessage *msg ) {

	PRINT(("ChatWindow::MessageReceived\n"));
	// Some variables needed for various messages
	BMessage* sendMessage;
	char bigtext[1024], bigtext2[1024];
	chatEntry newEntry;
	int32 srcLen, destLen = 1048, state = 0;
	char* text;
	time_t the_time;
	
	switch(msg->what) {

		// Fire off a GIMICQ_SEND_IM message, and clear the editview
		case GIMICQ_ATTEMPT_SEND_IM: {
			if( editview->IsEmpty() )	// don't send if nothing is there
				return;
			text = editview->GetFormattedMessage();
			UnFixLineBreaks( bigtext, text );
			srcLen = strlen( bigtext );
			convert_from_utf8( useEncoding, bigtext, &srcLen, bigtext2, &destLen, &state );
			bigtext2[destLen] = '\0';

			sendMessage = new BMessage(ICQ_TEXT_MESSAGE);
			sendMessage->AddInt32( "TO", (int32)uin );
			sendMessage->AddString( "TEXT", bigtext2 );
			sendMessage->AddInt32( "LENGTH", (int32)strlen(bigtext2) );
			mManager -> icq->PostMessage(sendMessage);
			if( closeAfterSend )
				Minimize(true);

			time(&the_time);
			msg->AddString( "TEXT", text );
			msg -> AddInt32("uin", uin);
			msg -> AddInt32("own_uin", mManager -> own_uin);
			msg -> AddInt64("time",the_time);

			AddMyStatement( text );
			mManager -> windows -> SendBuddyListMessage( msg );
			editview->SetText( "" );
			editview->ResetFontToBase();
			editview->MakeFocus(true);
			delete text;
			mManager -> sounds->PlaySound( WS_MSGSEND );
			break;
		}
			
		case ICQ_TEXT_MESSAGE:
			
			/* If the window is active when a message arrives, second arg is false -> not red in contact list.
			 * If the window is not active whena messag arrives, second arg is true -> red in contact list.
			 */
			if(mFinnishedUpdating)
				mManager -> list -> SetPersonMessage(uin, !IsActive());
			
			ProcessIncomingIM( msg );
			break;
			
		case ICQ_OFFLINE_DONE:
			AddOfflineDone();
			break;
			
		case ICQ_URL_MESSAGE:

			/* If the window is active when a message arrives, second arg is false -> not red in contact list.
			 * If the window is not active whena messag arrives, second arg is true -> red in contact list.
			 */
			if(mFinnishedUpdating)
				mManager -> list -> SetPersonMessage(uin, !IsActive());

			ProcessIncomingURL( msg );
			break;			
			
		case GIMICQ_RELOAD_PREF_SETTINGS:
			LoadPrefs();
			break;			
			
		case CHATWINDOW_SHOW_IN_THIS_WORKSPACE:
			SetWorkspaces( B_CURRENT_WORKSPACE );
			break;
			
		case CHATWINDOW_SHOW_IN_ALL_WORKSPACES:
			SetWorkspaces( B_ALL_WORKSPACES );
			break;
			
		case CHATWINDOW_POPUP_ON_RECEIVE:
			miPopupOnReceive->SetMarked( !miPopupOnReceive->IsMarked() );
			popupOnReceive = miPopupOnReceive->IsMarked();
			break;

		case CHATWINDOW_ENABLE_LINKS:
			miEnableLinks->SetMarked( !miEnableLinks->IsMarked() );
			linksEnabled = miEnableLinks->IsMarked();
			textview->SetShowLinks( linksEnabled );
			break;

		case CHATWINDOW_CLOSE_AFTER_SEND:
			break;

		case GIMICQ_ATTEMPT_SEND_URL:
			SendURL();
			break;
			
		case GIMICQ_OUTGOING_URL:
			DisplaySentURL( msg );
			break;
		
		case GIMICQ_OUTGOING_IM:
			AddMyStatement( msg->FindString("TEXT"), false );
			break;

		case GIMICQ_SHOW_MESSAGE:
			if(IsHidden() ) {// || !IsFront()) {
				Show();
			}
			break;

		case GIMICQ_LAST_MESSAGE:
			mFinnishedUpdating=true;
			break;
			
		case B_CANCEL:
		case MENU_FILE_CLOSE:
			QuitRequested();
			Lock();
			Quit();
			break;

		case GIMICQ_GET_PERSON_INFO:
			sendMessage = new BMessage(GIMICQ_OPEN_USER_WINDOW);
			sendMessage->AddInt32( "UIN", (int32)uin );	
			if( displayName[0] )
				sendMessage->AddString( "nick", displayName );
			sendMessage->AddInt32( "wtype", (int32)USER_INFO_TYPE );
			sendMessage->AddPointer( "poswindow", this );
			mManager -> windows -> SendBuddyListMessage( sendMessage );
			break;

		case GIMICQ_TRY_ADD_PERSON:
			AddPerson(msg);
			break;

		case GIMICQ_NEXT_CHAT_WINDOW:
		case GIMICQ_PREV_CHAT_WINDOW:
			msg->AddInt32( "uin", (int32)uin );
			msg->AddInt32( "wtype", (int32)USER_MESSAGE_TYPE );

		case GIMICQ_CHANGE_ENCODING:
			SetEncoding( (uint32)msg->FindInt32( "encoding" ) );
			msg->AddInt32( "UIN", (int32)uin );
			mManager -> windows -> SendBuddyListMessage( msg );
			break;

		case GIMICQ_ADD_PERSON:
			mManager -> windows -> SendBuddyListMessage( msg );
			break;
			
		case GIMICQ_UPDATE_INFO:
			if( (unsigned)msg->FindInt32("UIN") != uin )
				break;
			strcpy( displayName, msg->FindString("NICKNAME") );
			if( displayName && displayName[0] ) {
				if( prefixNewMessages && hasNewMessage ) {
					char newTitle[DISPLAY_NAME_MAX+10];
					sprintf( newTitle, "√ %s", displayName );
					SetTitle( newTitle );
				} else
					SetTitle( displayName );
			}
			break;

		case CHATWINDOW_FONT_SIZE_1:
		case CHATWINDOW_FONT_SIZE_2:
		case CHATWINDOW_FONT_SIZE_3:
		case CHATWINDOW_FONT_SIZE_4:
		case CHATWINDOW_FONT_SIZE_5:
		case CHATWINDOW_FONT_SIZE_6:
		case CHATWINDOW_FONT_SIZE_7:
		case CHATWINDOW_FONT_NORMAL:
		case CHATWINDOW_FONT_ITALIC:
		case CHATWINDOW_FONT_BOLD:
			SetStyles( msg );			
			break;
		
		case GIMICQ_CLEAR_HISTORY_MEM:
			Lock();
			textview->Clear();
			mManager -> history -> Clear(uin, mManager -> own_uin);
			Unlock();
			break;
			
		default:
			BWindow::MessageReceived(msg);
			break;
	}
}

//-----------------------------------------------------

// the cancel function
void ChatWindow::DispatchMessage( BMessage* msg, BHandler* handler ) {

	PRINT(("ChatWindow::DispatchMessage\n"));
	// the enter key does a bunch of stuff, depending on prefs...
	if( msg->what == B_KEY_DOWN && msg->HasString("bytes") ) {
		if( msg->FindString("bytes")[0] == B_ENTER ) {

			// mod+enter... depends on pref
			if( modifiers() & B_COMMAND_KEY ) {
				if( !enterIsNewline ) {
					editview->Insert("\n");
					editview->ScrollToSelection();
					return;
				}
				// not a newline insert, so make it "un-mod-ed"
				msg->ReplaceInt32( "modifiers", 0 );
			}

			// shift-enter always inserts a line break
			else if( modifiers() & B_SHIFT_KEY ) {
				editview->Insert("\n");
				editview->ScrollToSelection();
				return;
			}

			// plain old enter... depends on pref
			else {
				if( enterIsNewline ) {
					editview->Insert("\n");
					editview->ScrollToSelection();
					return;
				}
			}
		}
		
		else if( msg->FindString("bytes")[0] == B_TAB ) {
			if( (modifiers() == 0 || modifiers() == 32) && editview->IsFocus() ) {
				btView->sendButton->MakeFocus(true);
				return;
			}
		}
		
		else if( msg->FindString("bytes")[0] == B_ESCAPE ) {
			if( modifiers() == 0 || modifiers() == 32 && !menubar->IsFocus()) {
				PostMessage( new BMessage(B_QUIT_REQUESTED) );
				return;
			}
		}
	}

	// our work here is done... dispatch normally
	BWindow::DispatchMessage( msg, handler );
}

//-----------------------------------------------------

void ChatWindow::SendURL() {

	PRINT(("ChatWindow::SendURL\n"));
	BRect r( 0, 0, 256, 158 );
	mManager -> windows->MakeDialogFrame( r, this );
	SendURLWindow* su = new SendURLWindow( r, uin, displayName, useEncoding, mManager );
	mManager -> windows -> AddGeneralWindow(su);
	su->Show();
}

//-----------------------------------------------------

void ChatWindow::SendMessage() {

	PRINT(("ChatWindow::SendMessage\n"));
	char* text;
	BMessage* sendMessage;

	if( editview->IsEmpty() )	// don't send if nothing is there
		return;
	text = editview->GetFormattedMessage();
	sendMessage = new BMessage(ICQ_TEXT_MESSAGE);
	sendMessage->AddInt32( "TO", (int32)uin );
	sendMessage->AddString( "TEXT", text );
	sendMessage->AddInt32( "LENGTH", (int32)strlen(text) );
	AddMyStatement( editview->GetFormattedMessage() );
	editview->SetText( "" );
	editview->ResetFontToBase();
	mManager -> icq -> PostMessage(sendMessage);
	//btView->sendButton->MakeFocus(true);
	editview->MakeFocus(true);
	delete text;

}

//-----------------------------------------------------

bool ChatWindow::QuitRequested()
{
	PRINT(("ChatWindow::QuitRequested\n"));
	// Tell the main app that the window has been closed
	BMessage* sendMessage = new BMessage( GIMICQ_IM_WINDOW_CLOSED );
	sendMessage->AddInt32( "wtype", (int32)USER_MESSAGE_TYPE );
	sendMessage->AddInt32( "uin", (int32)uin );
	sendMessage->AddRect( "rect", Frame() );
	if(mManager -> windows)
		mManager -> windows -> SendBuddyListMessage( sendMessage );
	
	return(true);
}

//-----------------------------------------------------

void ChatWindow::MenusBeginning() {

	PRINT(("ChatWindow::MenusBeginning\n"));
	// Adjust the Edit menu... if the read-only TextView is selected, Cut, Paste and Undo
	// should be disabled. Also, the targets on Copy/SelectAll need be correct.
	
	if( textview->IsFocus() ) {
		miSelectAll->SetTarget( textview );
		miCopy->SetTarget( textview );
		miCut->SetEnabled( false );
		miPaste->SetEnabled( false );
		miUndo->SetEnabled( false );				
	}
	
	if( editview->IsFocus() ) {
		miSelectAll->SetTarget( editview );
		miCopy->SetTarget( editview );
		miCut->SetEnabled( true );
		miPaste->SetEnabled( true );
		miUndo->SetEnabled( true );	
	}	
#warning not calling MenusBeginning()

}


//-----------------------------------------------------

void ChatWindow::DisplaySentURL( BMessage* incomingIM, bool play_sound ) {

	PRINT(("ChatWindow::DisplatSentURL\n"));
	HTMLParser parse;
	styleList styles;
	rgb_color purple = {128,0,128};
	char msg[2048];
	char url[1024];
	char text[1024];

	BString text_str = _("text41");
	text_str += ": ";

	// grab the stuff from the message
	if( incomingIM->HasString("TEXT") )
		strcpy( text, incomingIM->FindString("TEXT") );
	if( incomingIM->HasString("URL") )
		strcpy( url, incomingIM->FindString("URL") );
	if( !url )
		return;

	// insert some more stuff
	sprintf( msg, "<a href=\"%s\">%s</a>", url, url );
	if( text && strlen(text) != 0 ) {
		strcat( msg, "<br>" );
		strcat( msg, text );
	}

	// insert the other person's name (blue font)
	textview->SetFontAttribute( B_BOLD_FACE, true );
	textview->SetFontColor( purple );
	textview->InsertSomeText( text_str.String() );
	textview->ResetFontToBase();
	
	// now parse and insert the HTML
	parse.Parse( msg );
	styles = parse.Styles();
	textview->AddStyledText( parse.ParsedString(), styles );
	delete styles;

	// Finalize the insertion, and update the timestamp
	textview->AddStatement();
	UpdateLastMessage();
}

//-----------------------------------------------------

void ChatWindow::ProcessIncomingIM( BMessage* incomingIM, bool play_sound ) {

	PRINT(("ChatWindow::ProcessIncomingIM\n"));
	HTMLParser parse;
	styleList styles;
	char parsedMessage[2048], finalMessage[2048];
	int32 srcLen, destLen = 2048, state = 0;

	bool empty = textview->IsEmpty();
	textview->ClearInsertStuff();

	// insert the other person's name (blue font)
	textview->SetFontAttribute( B_BOLD_FACE, true );
	textview->SetFontColor( kThat );
	textview->InsertSomeText( displayName );
	textview->InsertSomeText( ": " );
	textview->ResetFontToBase();

	// now parse and insert the HTML
	FixLineBreaks( parsedMessage, incomingIM->FindString("TEXT") );
	//parse.Parse( (char*)incomingIM->FindString("TEXT") );
	srcLen = strlen(parsedMessage);
	convert_to_utf8( useEncoding, const_cast<char*>(parsedMessage), &srcLen, finalMessage, &destLen, &state );
	finalMessage[destLen] = '\0';
	parse.Parse( finalMessage );
	styles = parse.Styles();
	textview->AddStyledText( parse.ParsedString(), styles );
	delete styles;

	// Finalize the insertion, and update the timestamp
	textview->AddStatement();
	UpdateLastMessage();

	if(play_sound) {
		// Play the sound
		if( IsMinimized() || (mManager -> prefs->ReadBool("gm_fl",true) && !IsActive()) )
			mManager -> sounds->PlaySound( WS_RIBBIT );
		else {
			if( empty )
				mManager -> sounds->PlaySound( WS_NEWMSG );
			else
				mManager -> sounds->PlaySound( WS_MSGRECEIVE );
		}
	}
	// popup the window if the user has specified that option
	Popup();
	
	// update the window title if prefixing with '√' is on
	hasNewMessage = true;
	if( prefixNewMessages && !IsActive() ) {
		char newTitle[DISPLAY_NAME_MAX+10];
		sprintf( newTitle, "√ %s", displayName );
		SetTitle( newTitle );
	}
}

//-----------------------------------------------------

void ChatWindow::ProcessIncomingURL( BMessage* incomingIM, bool play_sound ) {

	PRINT(("ChatWindow::ProcessIncomingURL\n"));
	HTMLParser parse;
	styleList styles;
	rgb_color green = {0,128,0};
	char msg[2048];
	char msg2[1024], msg3[1024];
	char* url = NULL;
	char* text = NULL;
	int32 srcLen, destLen = 1024, state = 0;
	
	BString text_str = _("text42");
	text_str += ": ";

	bool empty = textview->IsEmpty();
	textview->ClearInsertStuff();
	
	// grab the stuff from the message
	if( incomingIM->HasString("TEXT") ) {
		FixLineBreaks( msg2, const_cast<char*>( incomingIM->FindString("TEXT") ));
		srcLen = strlen(msg2);
		convert_to_utf8( useEncoding, const_cast<char*>(msg2), &srcLen, msg3, &destLen, &state );
		msg3[destLen] = '\0';
		text = msg3;
	}
	if( incomingIM->HasString("URL") )
		url = const_cast<char*>( incomingIM->FindString("URL") );
	if( !url )
		return;

	// insert some more stuff
	sprintf( msg, "<a href=\"%s\">%s</a>", url, url );
	if( text && strlen(text) != 0 ) {
		strcat( msg, "<br>" );
		strcat( msg, text );
	}

	// insert the other person's name (blue font)
	textview->SetFontAttribute( B_BOLD_FACE, true );
	textview->SetFontColor( green );
	textview->InsertSomeText( text_str.String() );
	textview->ResetFontToBase();
	
	// now parse and insert the HTML
	parse.Parse( msg );
	styles = parse.Styles();
	textview->AddStyledText( parse.ParsedString(), styles );
	delete styles;

	// Finalize the insertion, and update the timestamp
	textview->AddStatement();
	UpdateLastMessage();

	if(play_sound) {
		// Play the sound
		if( empty )
			mManager -> sounds->PlaySound( WS_NEWMSG );
		else
			mManager -> sounds->PlaySound( WS_MSGRECEIVE );
	}
	
	// popup the window if the user has specified that option
	Popup();
	
	// update the window title if prefixing with '√' is on
	hasNewMessage = true;
	if( prefixNewMessages && !IsActive() ) {
		char newTitle[DISPLAY_NAME_MAX+10];
		sprintf( newTitle, "√ %s", displayName );
		SetTitle( newTitle );
	}
}


//-----------------------------------------------------

void ChatWindow::AddMyStatement( const char* statement, bool play_sound ) {

	PRINT(("ChatWindow::AddMyStatement\n"));
	HTMLParser parse;
	styleList styles;
	char text[1024];
	textview->ClearInsertStuff();

	// insert our name (red font)
	textview->SetFontAttribute( B_BOLD_FACE, true );
	textview->SetFontColor( kThis );
	textview->InsertSomeText( mManager->p_comm->GetDisplayName() );
	textview->InsertSomeText( ": " );
	textview->ResetFontToBase();
	
	// now parse and insert the HTML
	FixLineBreaks(text, statement);
	parse.Parse( text );
	styles = parse.Styles();
	textview->AddStyledText( parse.ParsedString(), styles );
	delete styles;

	// Finalize the insertion, and update the timestamp
	textview->AddStatement();
	UpdateLastMessage();
}

//-----------------------------------------------------

void ChatWindow::ForcePopup() {

	PRINT(("ChatWindow::ForcePopup\n"));
	// Check to see if this window is already displayed in all workspaces
	// if not, move it to the current workspace		
	if( Workspaces() != B_ALL_WORKSPACES )
		SetWorkspaces( B_CURRENT_WORKSPACE );

	if(Lock()) {
		if(IsHidden() ) {// || !IsFront()) {
			LockLooper();
			PostMessage( GIMICQ_SHOW_MESSAGE );
			UnlockLooper();
		}

		if( !IsActive()  || !IsFront()) {//) {
			Activate(true);
		}
		Unlock();
	}		
}
void ChatWindow::Popup() {
	
	PRINT(("ChatWindow::Popup\n"));
	// if the window is hidden, pop it up
	if( popupOnReceive ) {
		ForcePopup();
	}
	
}

//-----------------------------------------------------

void ChatWindow::LoadPrefs() {

	PRINT(("ChatWindow::LoadPrefs\n"));
	// put in or take out the '√', depending on what the pref setting is
	prefixNewMessages = mManager -> prefs->ReadBool( "cw_p>", false );
	if( !prefixNewMessages )
		SetTitle( displayName );
	else if( hasNewMessage ) {
		char newTitle[DISPLAY_NAME_MAX+10];
		sprintf( newTitle, "√ %s", displayName );
		SetTitle( newTitle );	
	}

	closeAfterSend = mManager -> prefs->ReadBool( "cw_cs", false );

	// enable/disable enter inserts a newline
	enterIsNewline = mManager -> prefs->ReadBool( "cw_en", true );
}

//-----------------------------------------------------

void ChatWindow::SetStyles( BMessage* msg ) {
	
	PRINT(("ChatWindow::SetStyles\n"));
	int32 start, finish, attrib;
	text_run_array* curStyles = NULL;
	BFont font, font2;
	uint16 curMask;
	
	// Get the current selection, and the current styles for that selection
	editview->GetSelection( &start, &finish );
	curStyles = editview->RunArray( start, finish );
	font = curStyles->runs[0].font;
	curMask = font.Face();
		
	bool on;
		
	switch( msg->what ) {
		
		// set the font back to the normal font	
		case CHATWINDOW_FONT_NORMAL:
			font = editview->GetBaseFont();
			editview->SetFontAndColor( start, finish, &font );
			break;
		
		// make it italic
		case CHATWINDOW_FONT_ITALIC:
			if( msg->what == CHATWINDOW_FONT_ITALIC )
				attrib = B_ITALIC_FACE;

			editview->GetFontAndColor( start, &font2 );

			on = !(curMask & attrib);
			
			on = true;

				for( int i = 0; i < curStyles->count; ++i ) {
					curMask = curStyles->runs[i].font.Face();
					if( on ) curMask = ((curMask & B_REGULAR_FACE) ? 0 : curMask) | attrib;
					else curMask = (curMask == attrib) ? B_REGULAR_FACE : curMask & (~attrib);
					curStyles->runs[i].font.SetFace( curMask );
				}
				editview->SetRunArray( start, finish, curStyles );
				break;
		
		// make it bold
		case CHATWINDOW_FONT_BOLD:
			Say( "bold" );
			break;
		
		// font size change
		default:
			font.SetSize( ConvertFontSize(msg->what - CHATWINDOW_FONT_BASE) );
			editview->SetFontAndColor( start, finish, &font );
			break;
	}
		
	delete curStyles;
}

//-----------------------------------------------------

void ChatWindow::AddPerson( BMessage* msg ) {

	PRINT(("ChatWindow::AddPerson\n"));
	BMessage* sndMessage = new BMessage(GIMICQ_ADD_PERSON);
 	sndMessage->AddInt32( "UIN", (int32)uin );
 	if( displayName[0] != '\0' )
 		sndMessage->AddString( "NICKNAME", displayName );
	mManager -> windows -> SendBuddyListMessage(sndMessage);
}

//-----------------------------------------------------

void ChatWindow::UpdateLastMessage() {
	
	PRINT(("ChatWindow::UpdateLastMessage\n"));
	char theTime[100];
	
	int rnd = (int)(random() % 1979);
	if( rnd == 42 )
		lastView->SetText( "Run! The squirrels are coming!" );
	else {
		time_t now;
		time(&now);		
		strftime( theTime, 100, "%b %e, %Y (%I:%M %p)", localtime(&now) );
		lastView->SetText( theTime );
	}
}


//-----------------------------------------------------

void ChatWindow::FixLineBreaks( char* buffer, const char* input ) {

	PRINT(("ChatWindow::FixLineBreaks\n"));
	// Used to grab bytes
	char c, next;

	// position pointer for the buffer
	int i = 0;

    // Loop thru, grabbing bytes as we go
    while( (*input) != 0 ) {

	   c = (*input);
       if( c == CR ) {

         // attempt to get the next byte of input to see whether it's a
         // CR/LF pair (dos), or just a CR (mac)
         if( (next = *(input+1)) != 0 ) {

             // It's a CR/LF (dos), so the line has ended
             if( next == LF ) {
				buffer[i++] = '\n';
				input += 2;
				continue;
             }

             // Just a CR... put the next character onto the line and return it
             // Need to move the position back a byte so the first character
             // of the next line doesn't get lost!
             buffer[i++] = '\n';
             input++;
             continue;

         } else {    // couldn't get next byte; therefore the input stream is eof

			buffer[i++] = '\0';
			return;
         }
       }

       // If it's an LF, the line has ended no matter what the format.
       if( c == LF ) {

			buffer[i++] = '\n';
         	input++;
			continue;
       }

       // This byte is nothing special. Stick it onto the line.
       buffer[i++] = c;
       input++;
    }

    // apparently the input is now eof
    buffer[i++] = '\0';
}

//-----------------------------------------------------

void ChatWindow::UnFixLineBreaks( char* buffer, char* input ) {

	PRINT(("ChatWindow::UnFixLineBreaks\n"));
	DataContainer Message;
	char *i, *q;
	char value[1024];
	int l = 1;

	// parse the message, looking for newlines, and insert <br>'s instead
	//i = q = (char*)msg->FindString("message");
	i = q = input;
	if( i ) {
		while( *i ) {
			if( *i == char(10) ) {
				strncpy( value, q, l );
				value[l-1] = '\0';
				Message << value;
				Message << (unsigned char)CR;
				Message << (unsigned char)LF;
				l = 0;
				q = i+1;
			}
			++i; ++l;
		}
		if( i != q )
			Message << q;
	}
	strncpy( buffer, Message.c_ptr(), Message.length() );
	buffer[Message.length()] = '\0';
}

//-----------------------------------------------------

void ChatWindow::AddOfflineDone( bool repeat ) {

	PRINT(("ChatWindow::AddOfflineDone\n"));
	static bool didThis = false;

	BString text_str = "<< ";
	text_str += _("text43");
	text_str += " >>";

	if( (!didThis && !textview->IsEmpty()) || repeat ) {
		rgb_color color;
		color.red = 148;
		color.green = 109;
		color.blue = 76;
		textview->ClearInsertStuff();
		textview->SetFontAttribute( B_BOLD_FACE, true );
		textview->SetFontColor( color );
		textview->InsertSomeText( text_str.String() );
		textview->AddStatement();
		didThis = true;
	}
}

//=====================================================

BottomView::BottomView( BRect frame , GManager *the_manager)
		   : BView( frame, "bottom_view", B_FOLLOW_BOTTOM | B_FOLLOW_LEFT_RIGHT,
		   								  B_WILL_DRAW | B_NAVIGABLE_JUMP )
{
	PRINT(("BottomView::BottomView\n"));
	SetViewColor( 216, 216, 216 );
	mManager = the_manager;
	Locale *mLocale = mManager -> mLocale;	

	GButton *clearHistoryButton;
	BRect btnFrame = Bounds();
	btnFrame.right -= 7;
	btnFrame.left = btnFrame.right - 60;
	btnFrame.top = 7;
	btnFrame.bottom = 27;
	sendButton = new GButton( btnFrame, "send_button", _("sendbutton") , new BMessage(GIMICQ_ATTEMPT_SEND_IM),
										B_FOLLOW_RIGHT | B_FOLLOW_TOP );
	sendButton->MakeDefault( true );

	btnFrame = Bounds();
	btnFrame.left += 7;
	btnFrame.right = btnFrame.left + 90;
	btnFrame.top = 7;
	btnFrame.bottom = 27;
	clearHistoryButton = new GButton( btnFrame, "clear_button", _("clearbutton") , new BMessage(GIMICQ_CLEAR_HISTORY_MEM),
										B_FOLLOW_LEFT | B_FOLLOW_TOP , B_WILL_DRAW );
	
	AddChild( sendButton );
	AddChild( clearHistoryButton);
}

//=====================================================

PointerStringView::PointerStringView( BRect frame, const char *name, const char *text, 
					   uint32 resizingMode, uint32 flags )
				 : BStringView( frame, name, text, resizingMode, flags )
{
	PRINT(("PointerStringView::PointerStringView\n"));
}

//-----------------------------------------------------

void PointerStringView::MouseMoved( BPoint where, uint32 code, const BMessage *msg ) {

	PRINT(("PointerStringView::MouseMoved\n"));
	be_app->SetCursor(B_HAND_CURSOR);
	BStringView::MouseMoved(where, code, msg);
}

//=====================================================