#include <Debug.h>
#include "Gicq.h"

#include <Autolock.h>
#include <MessageRunner.h>

#include <stdlib.h>
#include <string.h>

#include "Date.h"
#include "GManager.h"
#include "WindowManager.h"
#include "PeopleData.h"
#include "ProtocolManager.h"
#include "Prefs.h"

//Ickle includes
#include <libicq2000/Client.h>
#include <libicq2000/events.h>

const int SOCKET_NOTHING	= 0;
const int SOCKET_READ			= 1;
const int SOCKET_WRITE		= 2;
const int SOCKET_EXCEPTION= 4;


//-------------------------- global ------------------------

int32 main_spawned( void *arg) {

	int fd;
	BMessage *m = (BMessage *)arg;
	protoICQ *icq=0;
	
	m->FindPointer( "icq", (void **)&icq );
	fd = m->FindInt32( "fd" );

	delete m;
	return icq->Main( fd ); 
}

bool global_socket_find( SocketInfo *aItem, void *aArg ) {
	int *fd = static_cast<int *>(aArg);
	return (aItem->mFd == *fd);
};

bool global_wait_for_threads( SocketInfo *aItem, void *aArg ) {
	BLocker *lock = static_cast<BLocker *>(aArg);
	PRINT( ("Wait for thread_id %d (current thread_id %d).\n", aItem->mThreadId, find_thread(NULL)) );
	PRINT( ("Socket: %d.\n", aItem->mFd) );

	status_t s;
	wait_for_thread( aItem->mThreadId, &s );

	return false;
}

//-------------------------- class -------------------------

protoICQ::protoICQ( GManager *the_manager ) : BLooper( "icq", B_LOW_PRIORITY )
{
	mManager = the_manager;
	
	mIsRunning = false;
	
	mPoller = new BMessageRunner( BMessenger( this ), new BMessage( ICQ_POLL ), 5000000 );
	mClient = new Client();
	InitFunctions();
	Run( );
}

protoICQ::~protoICQ()
{
	PRINT( ("~protoICQ\n") );
	delete mClient;
}

// This is called automagically when a BLooper is destroyed.  I re-defined it here to make sure we try to disconnect first
void protoICQ::Quit( )
{
	Logout( );

	// Let the BLooper do it's quit stuff.
	BLooper::Quit( );

}

void protoICQ::InitFunctions(){

	mClient->connected.connect(slot(this, &protoICQ::CBConnected)); 
	mClient->disconnected.connect(slot(this, &protoICQ::CBDisconnected)); 
	mClient->logger.connect(slot(this, &protoICQ::CBLogger)); 
	mClient->socket.connect(slot(this, &protoICQ::CBSocket)); 
	mClient->messageack.connect(slot(this, &protoICQ::CBMessageack)); 
	mClient->messaged.connect(slot( this, &protoICQ::CBMessaged ));
	mClient->contactlist.connect(slot( this, &protoICQ::CBContactList ));
	mClient->self_contact_userinfo_change_signal.connect(slot( this, &protoICQ::CBSelf ) );
	mClient->contact_userinfo_change_signal.connect(slot( this, &protoICQ::CBContactInfo ) );
	mClient->contact_status_change_signal.connect(slot( this, &protoICQ::CBContactStatus ) );
	mClient->search_result.connect(slot( this, &protoICQ::CBSearchResult ) );
	
}

// Time to leave.
void protoICQ::Logout( )
{

	// This causes the Disconnect callback to be called, and thus a message is sent to the app.
	mClient->setStatus( STATUS_OFFLINE );	
	mIsRunning = false;

	mMapLock.Lock();
	mSocketList.DoForEach( global_wait_for_threads, &mMapLock );
	mSocketList.MakeEmpty();
	mMapLock.Unlock();

}

int32 protoICQ::Main( int aFd ) {
	PRINT( ("Thread running for fd %d\n", aFd) );
	SocketInfo *si = NULL;
	fd_set rfds, wfds, efds;
	struct timeval tv;
	int value;
	
	mMapLock.Lock();
	si = mSocketList.FindItem(global_socket_find, &aFd);
	value = si->mValue;
	mMapLock.Unlock();

	while( mIsRunning && value != SOCKET_NOTHING) {
		
		FD_ZERO(&rfds);
		FD_ZERO(&wfds);
		FD_ZERO(&efds);

		if( value & SOCKET_READ ) {
			FD_SET(aFd, &rfds);
		}
		if( value & SOCKET_WRITE ) {
			FD_SET(aFd, &wfds);
		}
		if( value & SOCKET_EXCEPTION ) {
			FD_SET(aFd, &efds);
		}

		// should poll mClient every 5 seconds once connected
#ifdef BEOS_NETSERVER

		struct timeval *tvptr;
		tvptr = &tv;
		tv.tv_sec = 0;
		tv.tv_usec = 0;

		snooze( 20000 );
		int ret = select(aFd+1, &rfds, &wfds, &efds, tvptr);
#else
		struct timeval *tvptr;
		if (mClient->isConnected()) {
			tvptr = &tv;
			tv.tv_sec = 2;
			tv.tv_usec = 0;
		} else {
			tvptr = NULL;
		}

		int ret = select(aFd+1, &rfds, &wfds, &efds, tvptr);

#endif
		mMapLock.Lock();
		value = si->mValue;
		mMapLock.Unlock();
		if( value == SOCKET_NOTHING )
			break;

		if (ret > 0) {
			/*
			 * Care must be taken here, when iterating through the sets of
			 * file descriptors the Client::socket_cb()'s may signal
			 * Adding/Removing - modifying the sets inplace whilst we are
			 * iterating through them. Use a temporary copy for safety.
			 *
			 */

			mSocketLock.Lock();
			if ( FD_ISSET( aFd, &rfds ) ) {
				PRINT( ("\t\tSocketEvent::READ. fd: %d, thread: %d\n", aFd, find_thread(NULL) ) );
				mClient->socket_cb( aFd, SocketEvent::READ );
			}

			if ( FD_ISSET( aFd, &wfds ) ) {
				PRINT( ("\t\tSocketEvent::WRITE. fd: %d, thread: %d\n", aFd, find_thread(NULL) ) );
				mClient->socket_cb( aFd, SocketEvent::WRITE );
			}

			if ( FD_ISSET( aFd, &efds ) ) {
				PRINT( ("\t\tSocketEvent::EXCEPTION. fd: %d, thread: %d\n", aFd, find_thread(NULL) ) );
				mClient->socket_cb( aFd, SocketEvent::EXCEPTION );
			}
			mSocketLock.Unlock();
			
		}
	}
	
	PRINT( ("Thread returning. fd: %d, thread: %d\n", aFd, find_thread(NULL)) );
	mMapLock.Lock();
	delete si;
	mMapLock.Unlock();
	
	PRINT( ("Thread returned. fd: %d, thread: %d\n", aFd, find_thread(NULL)) );
	return 0;
}

// Standard BeOS thing...
void protoICQ::MessageReceived( BMessage *msg )
{
	BAutolock al( mSocketLock );
//	msg -> PrintToStream();
	switch( msg->what )
	{
		
		
		case B_QUIT_REQUESTED:
			Quit( );
			break;


		case ICQ_POLL:
			if (mClient->isConnected()) {
				mClient->Poll();
			}
			break;
		
		case ICQ_LOGIN:
		{
			PRINT( ("ICQ::LOGIN\n") ); 
			
			mClient->setUIN( msg->FindInt32( "UIN" ) );
			mClient->setPassword( msg->FindString( "PASSWORD" ) );
			mClient->setLoginServerHost( msg->FindString( "HOST" ) );
			PRINT( ("HOST: %s\n", msg->FindString( "HOST" ) ) );
			mClient->setLoginServerPort( msg->FindInt32( "PORT" ) );
			PRINT( ("PORT: %d\n", msg->FindInt32( "PORT" ) ) );
			
			mIsRunning = true;
			mClient->setStatus( STATUS_OFFLINE );
			mClient->setStatus( (Status) msg->FindInt32( "STATUS" ) );
			
			break;
		}

		case ICQ_LOGOUT:
			Logout( );
			break;

		case ICQ_STATUS_UPDATE:
			mClient->setStatus( (Status) msg -> FindInt32( "STATUS" ) );
			break;

		case ICQ_TEXT_MESSAGE:
		{
			uint32 uin = msg->FindInt32( "TO" );
			ContactRef cr = mClient->getContact( uin );
   		if( !cr.get() ) {
	   		ContactRef c( new Contact(uin) );
	   		mClient->addContact( c );
				cr = mClient->getContact( uin );
	   	}
			NormalMessageEvent *nmev = new NormalMessageEvent( cr, msg->FindString( "TEXT" ) );
			mClient->SendEvent( nmev );		
			break;
		}
		case ICQ_URL_MESSAGE:
		{
			uint32 uin = msg->FindInt32( "TO" );
			ContactRef cr = mClient->getContact( uin );
   		if( !cr.get() ) {
	   		ContactRef c( new Contact(uin) );
	   		mClient->addContact( c );
				cr = mClient->getContact( uin );
	   	}
			URLMessageEvent *umev = new URLMessageEvent( cr, msg->FindString( "TEXT" ), msg->FindString( "URL" ) );
			mClient->SendEvent( umev );
			break;
		}

		case ICQ_USER_INFO:
		case ICQ_USER_EXTINFO:
		{
			PRINT( ("User extra info request\n") );
			uint32 uin = msg->FindInt32( "UIN" );
			ContactRef cr = mClient->getContact( uin );
   		if( !cr.get() ) {
	   		ContactRef c( new Contact(uin) );
	   		mClient->addContact( c );
				cr = mClient->getContact( uin );
	   	}

   		mClient->fetchDetailContactInfo( cr );
			break;
		}
		
		case ICQ_ADD_CONTACT:
		{
			PRINT( ("ICQ_ADD_CONTACT\n") );
			ContactRef cr( new Contact(msg->FindInt32( "UIN" )) );
			mClient->addContact( cr );
			UserAddEvent *uae = new UserAddEvent( cr );
			mClient->SendEvent( uae );
			break;
		}

		case ICQ_GRANT_AUTH:
		{
			PRINT( ("ICQ_GRANT_AUTH\n") );
			ContactRef cr = mClient->getContact( msg->FindInt32( "UIN" ) );
			if( !cr.get() )
				return;
			
			PRINT( ("Sending authorization grant.\n") );
			AuthAckEvent *aaev = new AuthAckEvent( cr, true );
			mClient->SendEvent( aaev );
			break;
		}

		case ICQ_USER_SEARCH:
			mSREvent = mClient->searchForContacts( msg -> FindString( "NICKNAME" ), msg -> FindString( "FIRSTNAME" ), msg -> FindString( "LASTNAME" ),
																						 msg -> FindString( "EMAIL" ), range_NoRange, SEX_UNSPECIFIED, 0, "", "", 0, "", "", "", false );
			
			break;

		case ICQ_UPDATE_INFO:
/*
			icq_UpdateUserInfo( &mLink, msg -> FindString( "NICKNAME" ), msg -> FindString( "FIRSTNAME" ), msg -> FindString( "LASTNAME" ), msg ->FindString( "EMAIL" ) );
			if( msg->FindBool( "AUTHORIZE" ) )
				icq_UpdateAuthInfo( &mLink, 1 );
			else
				icq_UpdateAuthInfo( &mLink, 0 );
*/
			break;

#if 0
		case ICQ_UPDATE_EXTINFO:
			icq_UpdateMetaInfoSet( &mLink,
					
			icq_UpdateMetaInfoHomepage( &mLink,
					
			Send_CMD_UPDATE_EXT( msg );
			break;

		case ICQ_UPDATE_PASSWORD:

			Send_CMD_CHANGE_PASS( msg );
			break;
		case ICQ_NEW_UIN:
			Send_CMD_REG_NEW_USER( msg );
			break;
#endif
		
		default:
			BLooper::MessageReceived( msg );
			break;
	};
}

void protoICQ::sendInfo(Contact *aContact)
{
	BMessage m( ICQ_USER_INFO );

	m.AddInt32( "UIN", aContact->getUIN() );
	m.AddString( "NICKNAME", aContact->getAlias().c_str() );
	m.AddString( "FIRSTNAME", aContact->getFirstName().c_str() );
	m.AddString( "LASTNAME", aContact->getLastName().c_str() );
	m.AddString( "EMAIL", aContact->getEmail().c_str() );
	m.AddBool( "REQACK", aContact->getAuthReq() );

	Invoke( &m );
}

void protoICQ::sendExtInfo(Contact *aContact)
{
	BMessage m( ICQ_USER_EXTINFO );
	m.AddInt32( "UIN", aContact->getUIN() );
	m.AddString( "CITY", aContact->getMainHomeInfo().city.c_str() );
	m.AddString( "STATE", aContact->getMainHomeInfo().state.c_str() );
	m.AddString( "PHONENUMBER", aContact->getMainHomeInfo().phone.c_str() );
	m.AddString( "HOMEPAGE", aContact->getHomepageInfo().homepage.c_str() );
	m.AddString( "PROFILE", aContact->getAboutInfo().c_str() );
//	m.AddInt32( "COUNTRY", country_code );
	m.AddString( "COUNTRYNAME", aContact->getMainHomeInfo().getCountry().c_str() );
	m.AddInt32( "AGE", aContact->getHomepageInfo().age );
#warning look up TimeZone (gmt).
//	m.AddInt32( "TIMEZONE", ((signed char)zone>>1) );
	m.AddInt32( "SEX", aContact->getHomepageInfo().sex );
//	m.AddInt32( "ZIPCODE", zip );
	m.AddString( "ZIP", aContact->getMainHomeInfo().zip.c_str() );
	
//	m.AddString( "COUNTRYNAME", icq_GetCountryName( country_code ) );
	Invoke( &m );
}

void protoICQ::sendClientInfo(Contact *aContact)
{
	BMessage m( ICQ_CLIENT_INFO );
	m.AddInt32( "UIN", aContact->getUIN() );
	m.AddString( "NICKNAME", aContact->getAlias().c_str() );
	Invoke( &m );
}

// Ickle functions 
void protoICQ::CBConnected(ConnectedEvent *cev)
{
	PRINT( ("CBConnected\n") );
	
	BMessage m( ICQ_LOGGED_IN );
	Invoke( &m );
	
	mClient->fetchSelfDetailContactInfo();

}

void protoICQ::CBDisconnected(DisconnectedEvent *dev)
{
	PRINT( ("CBDisconnected\n") );
	mIsRunning = false;

	if(dev->getReason() == DisconnectedEvent::REQUESTED) { 
		BMessage m1( ICQ_LOGGED_OUT );
		Invoke( &m1 );
	} else { 
		BMessage m;
		switch(dev->getReason()) { 
			case DisconnectedEvent::FAILED_BADPASSWORD: 
				m.what=ICQ_BAD_PASSWORD; 
				break; 
			case DisconnectedEvent::FAILED_LOWLEVEL: 
			case DisconnectedEvent::FAILED_TURBOING: 
			case DisconnectedEvent::FAILED_MISMATCH_PASSWD: 
			case DisconnectedEvent::FAILED_UNKNOWN: 
			case DisconnectedEvent::FAILED_BADUSERNAME:
				fprintf(stderr, "ICQ_LOGIN_ERROR\n" );
				m.what=ICQ_LOGIN_ERROR; 
				break; 
		} 
		Invoke( &m ) ;
	}
}

void protoICQ::CBLogger(LogEvent *lev)
{
	PRINT( ("Logger ") );
	switch(lev->getType()) { 
		case LogEvent::INFO: 
			printf("Info: ");
			printf("%s\n", lev->getMessage().c_str());
			break; 
		case LogEvent::ERROR: 
			printf("Error: ");
			printf("%s\n", lev->getMessage().c_str());
			break; 
		case LogEvent::WARN: 
			printf("Warning: ");
			printf("%s\n", lev->getMessage().c_str());
			break; 
		case LogEvent::PACKET: 
			PRINT( ("Packet:\n") );
			PRINT( ("%s\n", lev->getMessage().c_str()) );
			break; 
		case LogEvent::DIRECTPACKET: 
			PRINT( ("DirectPacket:\n") );
			PRINT( ("%s\n", lev->getMessage().c_str()) );
			break; 
	}
	
}

void protoICQ::CBSocket(SocketEvent *sev)
{
	PRINT( ("CBSocket\n") );
	if( !sev )
		return;
		
	int fd;
	AddSocketHandleEvent *aev;
	
	fd = sev -> getSocketHandle();

	if( (aev=dynamic_cast<AddSocketHandleEvent *>(sev) ) != NULL) { 
		PRINT( ("Adding socket %d\n", fd) ); 
		BMessage *m = new BMessage();
		m->AddPointer( "icq", this );
		m->AddInt32( "fd", fd );

		mMapLock.Lock();
		
		SocketInfo *si = mSocketList.FindItem( global_socket_find, &fd );
		if( !si ) {
			si = new SocketInfo;
			si->mFd = fd;
			mSocketList.AddItem(si);
		}
		if(aev->isRead()) si->mValue |= SOCKET_READ; 
		if(aev->isWrite()) si->mValue |= SOCKET_WRITE; 
		if(aev->isException()) si->mValue |= SOCKET_EXCEPTION; 

		if( !si->mThreadId ) {
			si->mThreadId = spawn_thread( main_spawned, "icq_main", B_LOW_PRIORITY, m );
			if( si->mThreadId <= 0 ) {
				if( LockLooper() ) {
					PostMessage( ICQ_LOGOUT );
					UnlockLooper();
				}
				mMapLock.Unlock();
				return;
			}
			resume_thread( si->mThreadId );
		}
		mMapLock.Unlock();

	} else if(dynamic_cast<RemoveSocketHandleEvent *>(sev) != NULL) { 
		PRINT( ("Removing socket %d\n", fd) ); 

		mMapLock.Lock();
		SocketInfo *si = mSocketList.FindItem( global_socket_find, &fd );
		if( !si ) {
			mMapLock.Unlock();
			return;
		}
		si->mValue = SOCKET_NOTHING;
		mSocketList.RemoveItem( si );
		mMapLock.Unlock();

		PRINT( ("Removed socket %d\n", fd) ); 
	} 
}

void protoICQ::CBMessageack(MessageEvent *mev)
{
	PRINT( ("CBMessageack\n") );

#warning Need to get info. Does Ickle resend the messages?

}

void protoICQ::CBMessaged(MessageEvent *mev)
{
	PRINT( ("CBMessaged\n") );
	if( !mev )
		return;
	
	BMessage m;
	Contact *c = mev->getContact().get();
	
	NormalMessageEvent *nmev;
	URLMessageEvent *umev;
	AuthReqEvent *arev;
	AwayMessageEvent *amev;
	
	switch( mev->getType() ) {

	case MessageEvent::Normal:
		PRINT( ("MessageEvent::Normal\n") );
		if( ( nmev=dynamic_cast<NormalMessageEvent *>(mev) ) == NULL )
			return;
		m.what = ICQ_TEXT_MESSAGE;
		m.AddInt32( "FROM", c->getUIN() );
		m.AddInt32( "LENGTH", nmev->getMessage().length() );
		m.AddString( "TEXT", nmev->getMessage().c_str() );
		break;

	case MessageEvent::URL:
		PRINT( ("MessageEvent::URL\n") );
		if( ( umev=dynamic_cast<URLMessageEvent *>(mev) ) == NULL )
			return;
		m.what = ICQ_URL_MESSAGE;
		m.AddInt32( "FROM", c->getUIN() );
		m.AddInt32( "LENGTH", umev->getMessage().length() );
		m.AddString( "TEXT", umev->getMessage().c_str() );
		m.AddString( "URL", umev->getURL().c_str() );
		break;

	case MessageEvent::SMS:
		PRINT( ("MessageEvent::SMS\n") );
		return;
		break;
	case MessageEvent::SMS_Receipt:
		PRINT( ("MessageEvent::SMS_Receipt\n") );
		return;
		break;

	case MessageEvent::AuthReq:
		PRINT( ("MessageEvent::AuthReq\n") );
		if( ( arev=dynamic_cast<AuthReqEvent *>(mev) ) == NULL )
			return;
		m.what = ICQ_AUTH_REQUEST;
		m.AddInt32( "FROM", c->getUIN() );
		m.AddString( "NICKNAME", c->getAlias().c_str() );
		m.AddString( "FIRSTNAME", c->getFirstName().c_str() );
		m.AddString( "LASTNAME", c->getLastName().c_str() );
		m.AddString( "EMAIL", c->getEmail().c_str() );
		m.AddString( "REASON", arev->getMessage().c_str() );
		break;

	case MessageEvent::AuthAck:
		PRINT( ("MessageEvent::AuthAck\n") );
		return;
		break;

	case MessageEvent::AwayMessage:
		PRINT( ("MessageEvent::AwayMessage\n") );
		if( ( amev=dynamic_cast<AwayMessageEvent *>(mev) ) == NULL )
			return;
		m.what = ICQ_TEXT_MESSAGE;
		m.AddInt32( "FROM", c->getUIN() );
		m.AddInt32( "LENGTH", amev->getAwayMessage().length() );
		m.AddString( "TEXT", amev->getAwayMessage().c_str() );
		break;
	}

	Invoke( &m );
}

void protoICQ::CBContactList(ContactListEvent *clev)
{
	PRINT( ("CBContactList\n") );

	if( !clev )
		return;
	
	switch( clev->getType() ) {
	case ContactListEvent::UserAdded:
		PRINT( ("CBContactList::UserAdded\n") );
		if( dynamic_cast<UserAddedEvent *>(clev) == NULL )
			return;
		return;
		break;
		
	case ContactListEvent::UserRemoved:
		PRINT( ("CBContactList::UserRemoved\n") );
		if( dynamic_cast<UserRemovedEvent *>(clev)== NULL )
			return;
		return;
		break;
	}		
		
}

void protoICQ::CBSelf(UserInfoChangeEvent *uice)
{
	PRINT( ("CBSelf\n") );
	if( !uice )
		return;
		
	Contact *c = uice->getContact().get();
	sendClientInfo( c );
}

void 
protoICQ::CBContactInfo(UserInfoChangeEvent *uice)
{
	PRINT( ("CBContact\n") );
	
	Contact *c = uice->getContact().get();
	sendInfo( c );
	sendExtInfo( c );

}

void 
protoICQ::CBContactStatus(StatusChangeEvent *scev)
{
	PRINT( ("CBContactStatus\n") );
	BMessage m;
	if( scev->getStatus() != STATUS_OFFLINE ) {
		m.what=ICQ_USER_STATUS;
		m.AddInt32( "UIN", scev->getUIN() );
		m.AddInt32( "STATUS", scev->getStatus() );
	} else {
		m.what=ICQ_USER_OFFLINE;
		m.AddInt32( "UIN", scev->getUIN() );			
	}
	
	Invoke( &m );

}

void protoICQ::CBSearchResult(SearchResultEvent *srev)
{
	PRINT( ("CBSearchResult\n") );
	if( mSREvent != srev )
		return;
	
	if( srev->isExpired() ) {
		BMessage m( ICQ_USER_SEARCH_EXPIRED );
		Invoke( &m );
		return;
	}
	
//	ContactList& cl = srev->getContactList();
	Contact *c = srev->getLastContactAdded().get();
	
	if( c ) {
		BMessage m( ICQ_USER_SEARCH );
		m.AddInt32( "UIN", c->getUIN() );
		m.AddString( "NICKNAME", c->getAlias().c_str() );
		m.AddString( "FIRSTNAME", c->getFirstName().c_str() );
		m.AddString( "LASTNAME", c->getLastName().c_str() );
		m.AddString( "EMAIL", c->getEmail().c_str() );
		m.AddBool( "REQACK", c->getAuthReq() );
		Invoke( &m );
	}
	
	if( srev->isFinished() ){
		BMessage m( ICQ_USER_SEARCH_DONE );
		if( srev->getNumberMoreResults() )
			m.AddBool( "MORE", true );
		else
			m.AddBool( "MORE", false );			
		Invoke( &m );
	}
	
}

