#include <Debug.h>
#include "Date.h"
#include <time.h>

//---------------------------------------------------------

const int ReformYear = 1582;
const int ReformMonth = 10;
const unsigned long ReformDayNumber = 577737L;
const unsigned long EpochDayNumber = 719165L;

//---------------------------------------------------------

bool IsLeapYear(int year)
{
	if( year % 4 ) return false;			// if not divisible by 4, not leap
	if( year < ReformYear ) return true;	// before this year, all were leap
	if( year % 100 ) return true;			// by 4, but not by 100 is leap
	if( year % 400 ) return false;			// not by 100 and not by 400 not leap
	return true;
}

//---------------------------------------------------------

int DaysInMonth( int aMonth, int aYear )
{
	static const int days[13] = {0,31,28,31,30,31,30,31,31,30,31,30,31};
	if( aYear == ReformYear && aMonth == ReformMonth ) return 21;
	return days[aMonth] + (2 == aMonth && IsLeapYear(aYear));
}

//---------------------------------------------------------

unsigned long CountDays( int dYear, int dMonth, int dDay )
{
	unsigned long days;
	long year = (long)dYear - 1L;

	// get all days plus all leap years, minus non-leap years
	days = year * 365L + year / 4L - year / 100L + year / 400L;

	// the years before 1582 were all leap if divisible by 4
	if( year > ReformYear )
		days += 12;
	else {
		days += year / 100L;
		days -= year / 400L;
	}

	// get the days for the month up to the current one
	for( int i = 1; i < dMonth; ++i )
		days += DaysInMonth( i, dYear );

	// now add the current days of the month
	days += dDay;

	// now adjust for the 10 missing days (Oct 4 - Oct 15, 1582)
	if( days > ReformDayNumber ) days -= 10L;

	// return the day count
	return days;
}

//---------------------------------------------------------

time_t MakeEpoch( int dYear, int dMonth, int dDay, int dHour, int dMinute )
{
	// first get the number of days since the beginning of A.D.
	unsigned long seconds = CountDays( dYear, dMonth, dDay );
	
	// now adjust that count to refer to Jan 1, 1970 - start of the unix epoch
	seconds -= EpochDayNumber;
	
	// now multiply that by the number of seconds in the day
	seconds *= (60*60*24);
	
	// finally add on the seconds for the hour and the minute
	seconds += (60*60*dHour) + (60*dMinute);
	
	// return it
	return (time_t)seconds;
}

//---------------------------------------------------------

