#include <Debug.h>
#include "WindowManager.h"

#include <Locker.h>
#include <Screen.h>
#include <Alert.h>
#include <Autolock.h>
#include <File.h>
#include <Path.h>

#include <stdlib.h>

#include "Locale.h"
#include "BuddyList.h"
#include "PeopleData.h"
#include "MiscStuff.h"
#include "PList.h"

#include "GManager.h"
#include "ChatWindow.h"
#include "InfoWindow.h"
#include "LoginBox.h"
#include "PrefsWindow.h"
#include "ResultsWindow.h"
#include "AboutBox.h"
#include "Prefs.h"

//------------- Global functions -------------------------------------

bool global_close( IMWindowItem *item )
{
	item->window->Lock();
	item->window->Quit();
	delete item;
	return false;
}

bool global_post( IMWindowItem *item, void *arg2 )
{
	BMessage *msg = static_cast<BMessage *>(arg2);
	
	item->window->PostMessage( msg );
	return false;
}

bool global_IMfind( IMWindowItem *item, void *arg2 )
{
	IMWindowItem *temp = static_cast<IMWindowItem *>( arg2 );
	
	return ((item->uin == temp->uin) && (item->type == temp->type));
}
//-----------------------------

bool global_delete( WinPos *item )
{
	delete item;
	return false;
}

bool global_save( WinPos *item, void *arg2 )
{
	FILE *ptr = static_cast<FILE *>(arg2);
	
	fprintf( ptr, "%lu	%f	%f	%f	%f\n", item->uin, item->rect.left, item->rect.top, item->rect.right, item->rect.bottom );
	return false;
}

bool global_WPfind( WinPos *item, void *arg2 )
{
	uint32 uin = *static_cast<uint32 *>(arg2);
	return (item->uin == uin );
}

//---------------------------------------------------------------------
WindowManager::WindowManager(GManager *aManager) 
	:bList(NULL)
	,signOn(NULL)
	,resWin(NULL)
	,about(NULL)
	,mUin(0)
	,closing_all_general(false)
	,mManager( aManager )
{
	mIMWindowRect = BRect( 25, 25, 350, 280 );
	mLock = new BLocker( "window_lock" );

	mIMWindows = new PList<IMWindowItem>();
	mWinPositions = new PList<WinPos>(40);
	mGeneralWindows = new PList<BWindow>();
}
//---------------------------------------------------------------------
WindowManager::~WindowManager() {

	SaveWindowPositions();

	CloseAllGeneralWindows();
	CloseAllIMWindows();
	delete mLock;
	
	delete mIMWindows;
	delete mWinPositions;
	delete mGeneralWindows;

}

//---------------------------------------------------------------------
void WindowManager::ToggleHidden() {

	BAutolock a( mLock );

	// toggle the minimization of the Buddy List window...
	if( bList->Window() && bList->Window()->Lock() ) {
		bList->Window()->Minimize(!bList->Window()->IsMinimized());
		bList->Window()->Unlock();
	}

}

//---------------------------------------------------------------------

void WindowManager::CalcNewIMRect() {

	BAutolock a( mLock );
	BScreen screen;
	
	// offset it... put it back to top-left if it doesn't fit onscreen
	mIMWindowRect.OffsetBy( 20, 20 );
	if( !screen.Frame().Contains(mIMWindowRect) )
		mIMWindowRect = BRect( 25, 25, 350, 280 );
	
}

//---------------------------------------------------------------------

void WindowManager::UserWindowClosed( BMessage *aMsg ){

	BAutolock a( mLock );

	IMWindowItem *item, temp; 
	temp.uin = (unsigned)aMsg->FindInt32("uin");
	temp.type = (int)aMsg->FindInt32("wtype");
	
	// save the rect, if need be
	if( aMsg->HasRect("rect") )
		SetWindowPosition( temp.uin, aMsg->FindRect("rect") );
	
	// Find the window in the list
	item = mIMWindows->FindItem( global_IMfind, &temp );
	
	// Delete the record (if it was there)
	if( item ) {
		//mIMWindows.GetItem(i).window -> Lock();    // Always closed somewhere else.
		//mIMWindows.GetItem(i).window -> Quit(); 
		mIMWindows->RemoveItem( item );
	}	
}

//---------------------------------------------------------------------

void WindowManager::SwitchUserWindow( BMessage *aMsg ) {

	BAutolock a( mLock );
	IMWindowItem *item=NULL, temp;
	
	// true if we should go the the next one, false if to the prev
	bool forward = bool(aMsg->what == GIMICQ_NEXT_CHAT_WINDOW);
	
	// if the list is empty, just ignore it
	if( mIMWindows->IsEmpty() )
		return;	

	// was this sent from a user window or somewhere else?
	if( aMsg->HasString("userid") ) {

		temp.uin = (unsigned)aMsg->FindInt32("uin");
		temp.type = (int)aMsg->FindInt32("wtype");

		// Find the window in the list
		item = mIMWindows->FindItem( global_IMfind, &temp );
	}
	
	// if it wasn't there, use the first or last one
	if( !item ) {
		if( forward )
			item = mIMWindows->FirstItem();
		else
			item = mIMWindows->LastItem();

		if( item->window->Lock() ) {
			item->window->Activate();
			item->window->Unlock();
		}

		return;
	}

	// now look for the next (or previous) window
	int32 index = mIMWindows->IndexOf( item );
	item = mIMWindows->ItemAt( ++index );
	if( forward )
		if( !item )
			item = mIMWindows->FirstItem();
	else 
		if( !item )
			item = mIMWindows->LastItem();

	// activate the target window
	if( item->window->Lock() ) {
		item->window->Activate();
		item->window->Unlock();
	}

}

//---------------------------------------------------------------------


void WindowManager::OpenBuddyList( BPoint *point ) {
 
	BAutolock a( mLock );
	BScreen screen;
	BRect frame(0,0,158,280);
	BuddyWindow *bWindow = NULL;
	
	if( bList )
		bWindow = dynamic_cast<BuddyWindow *>(bList->Window());
	
	if( !bWindow ) {
		
		frame.OffsetBy( screen.Frame().Width() - frame.Width() - 10, 29 );
		bWindow = new BuddyWindow(frame, mManager);
		
		bList = new BuddyList(bWindow -> Bounds(), mManager);
	  bWindow -> AddChild(bList);

		// add shortcuts to handle prev and next commands from the buddylist
		bWindow -> AddShortcut( ',', 0, new BMessage(GIMICQ_PREV_CHAT_WINDOW), bList );
		bWindow -> AddShortcut( '.', 0, new BMessage(GIMICQ_NEXT_CHAT_WINDOW), bList );
	
		BMessage *statMsg;
	
		statMsg = new BMessage(GIMICQ_MY_STATUS_CHANGE);
		statMsg->AddInt32( "status", (int32)IS_AVAILABLE);
		bWindow -> AddShortcut( '1', 0, statMsg, bList );
		statMsg = new BMessage(GIMICQ_MY_STATUS_CHANGE);
		statMsg->AddInt32( "status", (int32)IS_FREEFORCHAT);
		bWindow -> AddShortcut( '2', 0, statMsg, bList );
		statMsg = new BMessage(GIMICQ_MY_STATUS_CHANGE);
		statMsg->AddInt32( "status", (int32)IS_AWAY );
		bWindow -> AddShortcut( '3', 0, statMsg, bList );
		statMsg = new BMessage(GIMICQ_MY_STATUS_CHANGE);
		statMsg->AddInt32( "status", (int32)IS_EXTENDEDAWAY );
		bWindow -> AddShortcut( '4', 0, statMsg, bList );	
		statMsg = new BMessage(GIMICQ_MY_STATUS_CHANGE);
		statMsg->AddInt32( "status", (int32)IS_OCCUPIED );
		bWindow -> AddShortcut( '5', 0, statMsg, bList );	
		statMsg = new BMessage(GIMICQ_MY_STATUS_CHANGE);
		statMsg->AddInt32( "status", (int32)IS_DND );
		bWindow -> AddShortcut( '6', 0, statMsg, bList );	
		statMsg = new BMessage(GIMICQ_MY_STATUS_CHANGE);
		statMsg->AddInt32( "status", (int32)IS_INVISIBLE );
		bWindow -> AddShortcut( '7', 0, statMsg, bList );

		bWindow->Show();
	} else {
		if( bWindow->Lock() ) {
			bWindow->Activate();
			bWindow->Unlock();
		}
	}

 }
 
//---------------------------------------------------------------------

void WindowManager::CloseBuddyList() {

	BAutolock a( mLock );
	BMessage *aMsg = new BMessage( B_QUIT_REQUESTED );
	aMsg->AddBool( "killapp", false );
	if( bList->Window() && bList->Window()->Lock() ) {
		bList->Window()->PostMessage( aMsg );
		bList->Window()->Unlock();
	}
}

//---------------------------------------------------------------------

void WindowManager::UserWindowOpened( BMessage *aMsg )
{
/*
	BAutolock a( mLock );

	BWindow *newWindow;

	// Get the data from the BMessage and make a new record out of it
	aMsg->FindPointer( "new_window", (void**)&newWindow );
	IMWindowItem newRecord( (unsigned)aMsg->FindInt32("uin"),
						    (int)aMsg->FindInt32("wtype"), newWindow );

	// Add that record to the list
	mIMWindows.Add( newRecord );
	
*/
}

//---------------------------------------------------------------------

BWindow *WindowManager::OpenUserWindow( BMessage *aMsg)
{
	BAutolock a( mLock );
	BWindow *userWindow = NULL, *posWindow = NULL;

	char displayName[DISPLAY_NAME_MAX];
	uint32 encoding;
	const char *str=NULL;
	IMWindowItem *item, temp;
	temp.type = (int)aMsg->FindInt32("wtype");
	temp.uin = (unsigned)aMsg->FindInt32("UIN");

	bool min = (aMsg->HasBool("minimize") && aMsg->FindBool("minimize"));
	if( aMsg->HasPointer("poswindow") )
		aMsg->FindPointer( "poswindow", (void**)&posWindow );

	if( aMsg->HasString("nick") )
		strcpy( displayName, aMsg->FindString("nick") );
	else if( mManager -> list ) {
		str = mManager -> list -> FindNameForUIN(temp.uin);
		if( str )
			strcpy( displayName, str );
		else
			sprintf( displayName, "%lu", temp.uin );
	} else
		sprintf( displayName, "%lu", temp.uin );
		
	// do the encoding stuff
	encoding = mManager -> defEncoding;
	if( aMsg->HasInt32("encoding") )
		encoding = (uint32)aMsg->FindInt32("encoding");
	else if( mManager -> list ) {
		if( !mManager -> list -> FindEncodingForUIN(temp.uin,encoding) )
			encoding = mManager -> defEncoding;
	}
	
	item = mIMWindows->FindItem( global_IMfind, &temp );
	if( item ) {
		return item->window;
	}	
	
	// Otherwise, make a new one
	switch( temp.type ) {
	
		// a message (IM) window
		case USER_MESSAGE_TYPE:
			userWindow = MakeMessageWindow( temp.uin, displayName, min, encoding );
			break;
			
		case USER_INFO_TYPE:
			userWindow = MakeInfoWindow( temp.uin, displayName, posWindow );
			break;
	}

	// show it and quit
	if( userWindow ) {
		IMWindowItem *rec = new IMWindowItem( temp.uin, temp.type, userWindow );
		mIMWindows->AddItem( rec );
	}
	return userWindow;
}

//---------------------------------------------------------------------

BWindow *WindowManager::MakeMessageWindow( uint32 aUin, const char *nick, bool min, uint32 encoding ) {

	BRect rect = mIMWindowRect;
	if( !FindWindowPosition( aUin, rect ) )
		CalcNewIMRect();
	
	BWindow *ret = new ChatWindow( aUin, nick, rect, min, encoding, mManager );
	return ret;
}

//---------------------------------------------------------------------

BWindow *WindowManager::MakeInfoWindow( uint32 aUin, const char *nick, BWindow *posWindow )
{

	BAutolock a( mLock );
	//BRect rect(0,0,330,173);
	BRect rect(0,0,340,215);

	// position it appropriately
	if( posWindow )
		MakeDialogFrame( rect, posWindow );
	else
		CenterFrame( rect );

	// make it	
	InfoWindow *ret = new InfoWindow( rect, aUin, nick, mManager );
	return ret;
}
		
//---------------------------------------------------------------------

void WindowManager::OpenSignOnWindow( BPoint *point ) {

	if(signOn == NULL) {
		BAutolock a( mLock );
		BRect frame( 0, 0, 230, 205 );
		BScreen screen;
		
		// create the signon window and show it
		frame.OffsetBy( screen.Frame().Width() - frame.Width() - 10, 29 );
		signOn = new LoginWindow( frame, mManager );
		signOn->Show();
		
	}
}

//---------------------------------------------------------------------


void WindowManager::SignOnStop() {

	BAutolock a( mLock );
	if( signOn ) {
#warning ouch... change to constant.
		signOn -> LockLooper();					// perhaps needed.
		signOn->PostMessage( 1511159 );
		signOn -> UnlockLooper();
	}
}

//---------------------------------------------------------------------

void WindowManager::CloseSignOnWindow(bool should_quit ) {

	BAutolock a( mLock );

	// lock the sign on window and close it
	if(signOn != NULL && should_quit) {
		signOn->Lock();
		signOn->Quit();
	}
	signOn = NULL;
}

//---------------------------------------------------------------------

void WindowManager::ForwardIncomingMessage( BMessage *aMsg, bool should_open ) { // if open is true a new window is opened.

	BAutolock a( mLock );

	IMWindowItem *item, temp;
	temp.uin = (unsigned)aMsg->FindInt32("UIN");
	temp.type = (int)aMsg->FindInt32("wtype");

	// Find the window in the list
	item = mIMWindows->FindItem( global_IMfind, &temp );

	// Construct a BMessenger and forward the message
	if( item ) {		
		
		//If it is a chatwindow, use the popup finction and inform the deskbar.
		if( item->type != USER_INFO_TYPE && aMsg -> what != GIMICQ_OUTGOING_URL) {
#warning check the casting!
			((ChatWindow *) item->window) -> Popup();
			((ChatWindow *) item->window) -> InformDeskbar(aMsg -> what);
		}

		// Send a message to the window.
		if( item->window->Lock() ) {
			item->window->PostMessage( aMsg );
			item->window->Unlock();
		}

	} else if( should_open && temp.type != USER_INFO_TYPE && aMsg -> what != GIMICQ_OUTGOING_URL) {
		
		aMsg->AddBool( "minimize", true );
		ChatWindow *newWindow = (ChatWindow *) OpenUserWindow( aMsg);
		newWindow -> InformDeskbar(aMsg -> what);
		//newWindow -> PostMessage( aMsg );
		newWindow -> Popup();
	}
	
}

//---------------------------------------------------------------------

void WindowManager::OpenPrefsWindow( bool globalOnly, BPoint *point ) {

	BAutolock a( mLock );
	BRect prefsRect( 0, 0, 520, 210 );

	if( bList )
		MakeDialogFrame( prefsRect, bList );
	else
		CenterFrame( prefsRect );

	if( point )
		prefsRect.OffsetTo( *point );

	// make the window
	PrefsWindow *pf = new PrefsWindow( prefsRect, mManager, globalOnly );
	AddGeneralWindow(pf);
	pf->Show();

}

//---------------------------------------------------------------------

void WindowManager::SendBuddyListMessage( BMessage *aMsg ) {

	BAutolock a(mLock);

	// Bail of there is no BuddyList.
	if(!bList)
		return;

	BMessenger *messenger = new BMessenger(bList);
	if( bList -> LockLooper() ) {
		messenger -> SendMessage( aMsg);
		bList -> UnlockLooper();
	}
	delete messenger;
}
		
//---------------------------------------------------------------------		

void WindowManager::SendSignOnMessage( BMessage *aMsg ) {
	
	BAutolock a(mLock);
	
	if(!signOn)
		return;
	
	if( signOn->Lock() ) {
		signOn->PostMessage( aMsg );
		signOn->Unlock();
	}
	
	return;
}

//---------------------------------------------------------------------

void WindowManager::BroadcastMessage( BMessage *aMsg, bool toApp ) {

	BAutolock a( mLock );
	if(toApp)
		SendBuddyListMessage( aMsg );

	// send to all the open chat windows
	mIMWindows->DoForEach( global_post, aMsg );
		
}
		
//---------------------------------------------------------------------

void WindowManager::CenterFrame( BRect& frame ) {

	BAutolock a( mLock );
	BRect centered;
	BScreen screen;	
	
	centered.left = (screen.Frame().Width() - frame.Width()) / 2;
	centered.top = (screen.Frame().Height() - frame.Height()) / 2;
	centered.right = centered.left + frame.Width();
	centered.bottom = centered.top + frame.Height();
	frame = centered;
	
}

//---------------------------------------------------------------------

void WindowManager::MakeDialogFrame( BRect& frame, BView *view ) {
	
	BAutolock a( mLock );
	BPoint point;

	// make sure there IS a view	
	if( !view ) {
		return;
	}
	
	// get it's top left corner and call the real function
	view -> Window() -> LockLooper();
	point = view->Frame().LeftTop();
	view -> Window() -> UnlockLooper();
	
	MakeDialogFrame( frame, point );
}

//---------------------------------------------------------------------

void WindowManager::MakeDialogFrame( BRect& frame, BWindow *window ) {
	
	BAutolock a( mLock );
	BPoint point;

	// make sure there IS a window	
	if( !window ) {
		return;
	}
	
	// get it's top left corner and call the real function

	point = window->Frame().LeftTop();
	
	MakeDialogFrame( frame, point );
}

//---------------------------------------------------------------------

void WindowManager::AdjustFrame( BRect& frame ) {

	BAutolock a( mLock );
	BScreen screen;
	float offx = 0, offy = 0;
	
	// if the the rect isn't totally onscreen
	if( !screen.Frame().Contains( frame ) ) {

		// offset by the amount that the box would be offscreen
		offx = frame.left + frame.Width() - screen.Frame().Width();
		offy = frame.top + frame.Height() - screen.Frame().Height();
		frame.OffsetBy( offx < 0 ? 0 : -offx - 20, offy < 0 ? 0 : -offy - 20 );
	}	
		
}

//---------------------------------------------------------------------

void WindowManager::MakeDialogFrame( BRect& frame, BPoint point ) {

	BAutolock a( mLock );
	BScreen screen;
	float offx = 0, offy = 0;	
	
	// Move the rectangle sligtly down and right relative to point
	frame.OffsetTo( point.x + 20, point.y + 40 );

	// if the the rect isn't totally onscreen
	if( !screen.Frame().Contains( frame ) ) {

		// offset by the amount that the box would be offscreen
		offx = frame.left + frame.Width() - screen.Frame().Width();
		offy = frame.top + frame.Height() - screen.Frame().Height();
		frame.OffsetBy( offx < 0 ? 0 : -offx - 20, offy < 0 ? 0 : -offy - 20 );
	}	
		
}

//---------------------------------------------------------------------

void WindowManager::OpenEmailSearchResultsWindow( BMessage *aMsg ) {

	BAutolock a( mLock );
	BRect resRect(100,100,520,330);
	// only make a new results window if it's not already open
	if( !resWin ) {
	
		// put the frame in the right spot
		if( bList )
			MakeDialogFrame( resRect, bList );
		else
			CenterFrame( resRect );

		// open the window
		resWin = new SearchResultsWindow( resRect, mManager );
		AddGeneralWindow(resWin);
		resWin->Show();
	}
	
	// set (or reset) the window
	if( resWin->Lock() ) {
		resWin->Activate();	
		resWin->PostMessage( aMsg );
		resWin->Unlock();
	}
	
}

//---------------------------------------------------------------------

void WindowManager::ForwardEmailSearchResults( BMessage *aMsg ) {

	if( resWin && resWin->Lock() ) {
		resWin->PostMessage( aMsg );
		resWin->Unlock();
	}
}

//---------------------------------------------------------------------

void WindowManager::EmailSearchResultsWindowClosed() {
	resWin = NULL;
}

//---------------------------------------------------------------------

void WindowManager::DoAbout() {

//	BRect frame( 0, 0, 225, 235 );
	BRect frame( 0, 0, 150, 100 );
	MakeDialogFrame( frame, bList );
	about = new AboutBox( frame, mManager );
	AddGeneralWindow(about);
	about->Show();
}

//---------------------------------------------------------------------

bool WindowManager::FindWindowPosition( uint32 aUin, BRect& aRect ) {

	BAutolock a( mLock );
	bool found = false;
	WinPos *item;

	item = mWinPositions->FindItem( global_WPfind, &aUin );

	if( item ) {
		aRect = item->rect;
		found = true;
	}

	return found;
}

//---------------------------------------------------------------------

void WindowManager::SetWindowPosition( uint32 aUin, BRect aRect ) {

	BAutolock a( mLock );
	WinPos *item;

	item = mWinPositions->FindItem( global_WPfind, &aUin );

	if( item ) {
		item->rect = aRect;
	} else {
		item = new WinPos();
		item->uin = aUin;
		item->rect = aRect;
		mWinPositions->AddItem( item );
	}

}
		
//---------------------------------------------------------------------

void WindowManager::SetProgramUser( uint32 aUin ) {

	BAutolock a( mLock );
	mUin = aUin;
	LoadWindowPositions();
}

//---------------------------------------------------------------------

void WindowManager::AddGeneralWindow(BWindow *w)
{

	BAutolock a( mLock );
	mGeneralWindows->AddItem( w );
	
}

void WindowManager::CloseGeneralWindow(BWindow *w, bool should_quit)
{

	if( closing_all_general )
		return;

	BAutolock a( mLock );

	mGeneralWindows->RemoveItem( w );
	if(should_quit) {
		w -> Lock();
		w -> Quit();
	}

	return;
}

void WindowManager::CloseAllGeneralWindows() {

	BAutolock a( mLock );
	
	closing_all_general = true;
	
	while( !mGeneralWindows->IsEmpty() ) {
		BWindow *w = mGeneralWindows->RemoveItem((int32)0);
		if(w) {
			w -> Lock();
			w -> Quit();			
		}
	}
	
	closing_all_general = false;
	
	return;
}

void WindowManager::CloseAllIMWindows() {

	BAutolock a( mLock );

	mIMWindows->DoForEach( global_close );
	mIMWindows->MakeEmpty();
		
	return;
}

//--------------------------------- Private ---------------------------------------

void WindowManager::LoadWindowPositions() {

	BAutolock a( mLock );
	
	mWinPositions->DoForEach( global_delete );
	mWinPositions->MakeEmpty();
	
	char filename[2048], data[512];
	char ftemp[100], tmpchar;
	WinPos *addRec;
	unsigned uuin, i = 0, j, k;
	float ftemp2;
	BRect temp;
	FILE *ptr;
	
	BPath path;
	BFile file;
	MakeFilenameFromUIN( filename, mUin, "windowpos" );
	mManager->prefs->FindConfigFile( NULL, filename, &file, B_READ_ONLY, &path );
	file.Unset();
	
	ptr = fopen( path.Path(), "r" );
	if( ptr == 0 ) {
		return;
	}

	ftemp[i] = 0;
	tmpchar = getc( ptr );
	while( tmpchar != -1 ) {
	
		i = 0;
		while( tmpchar != -1 ) {
			if( tmpchar == '\n' ) {
				data[i] = '\0';
				break;
			}
			data[i++] =	tmpchar;
			tmpchar = getc( ptr );
		}
		tmpchar = getc( ptr );
		j = k = 0;
		
		// get the uin
		k = 0;
		while( data[j] && data[j] != char(9) )
			ftemp[k++] = data[j++];
		ftemp[k] = '\0';
		uuin = atol( ftemp );

		// get the coord
		for( int l = 0; l < 4; ++l ) {
			k = 0; ++j;
			while( data[j] && data[j] != char(9) )
				ftemp[k++] = data[j++];
			ftemp[k] = '\0';
			ftemp2 = atof( ftemp );
			if( l == 0 ) temp.left = ftemp2;
			else if( l == 1 ) temp.top = ftemp2;
			else if( l == 2 ) temp.right = ftemp2;
			else if( l == 3 ) temp.bottom = ftemp2;
		}
		
		// make and add the record
		addRec = new WinPos();
		addRec->uin = uuin;
		addRec->rect = temp;
		mWinPositions->AddItem( addRec );
	}
	fclose( ptr );
}

//---------------------------------------------------------------------

void WindowManager::SaveWindowPositions() {//

	if( !mUin )
		return;
		
	BAutolock a( mLock );
#warning 2048 should be an OS constant instead.
	char filename[2048];
	FILE *ptr;
	BPath path;

	BFile file;
	MakeFilenameFromUIN( filename, mUin, "windowpos" );
	mManager->prefs->FindConfigFile( NULL, filename, &file, B_WRITE_ONLY | B_CREATE_FILE, &path );
	file.Unset();

	ptr = fopen( path.Path(), "w" );
	if( ptr == 0 ) {
		fprintf(stderr, "Failed to write the window positions: %s\n", path.Path() );
		return;
	}

	mWinPositions->DoForEach( global_save, ptr );
	
	fclose( ptr );
}

//---------------------------------------------------------------------		


