#include <Debug.h>
#include "HistoryWindow.h"

#include <Box.h>
#include <ScrollView.h>
#include <Alert.h>

#include "GButton.h"
#include "GConstants.h"
#include "FancyView.h"
#include "HTMLStuff.h"
#include "HistoryIO.h"
#include "GManager.h"
#include "ProtocolManager.h"
#include "PeopleData.h"
#include "Prefs.h"
#include "History.h"
#include "WindowManager.h"
#include "Locale.h"

HistoryWindow::HistoryWindow( unsigned int a_uin, unsigned int a_own_uin, GManager *the_manager, uint32 encoding)
	: BWindow(BRect(30,30,430,330),NULL, B_TITLED_WINDOW, 0, B_ALL_WORKSPACES) {
	
	mManager = the_manager;
	mLocale = mManager -> mLocale;
	
	uin = a_uin;
	own_uin = a_own_uin;
	nick = mManager -> list -> FindNameForUIN( uin );
	own_nick = mManager -> p_comm -> GetDisplayName();
	
	BRect w_bounds = Bounds();
	BRect rect = w_bounds;
	
	BString title = " ";
	title.Append( _("text60") );
	title.Append(":");
	title.Append( nick.String() );
	
	SetTitle( title.String() );
	
	BBox *box = new BBox(rect, NULL, B_FOLLOW_LEFT_RIGHT | B_FOLLOW_TOP_BOTTOM);

	rect = box -> Bounds();
	
	rect.left   += 20;
	rect.right  -= 35;
	rect.top    += 30;
	rect.bottom -= 50;
	BRect text_rect = rect;
	text_rect.OffsetTo(B_ORIGIN);
	text_rect.InsetBy( 2.0, 2.0 );
	textview = new FancyTextView(rect,"Textbox", mManager , text_rect, B_FOLLOW_ALL, B_WILL_DRAW);
	textview -> MakeEditable(false);
	textview->SetViewColor( 242, 242, 242 );
	rgb_color black;
	black.red = black.green = black.blue = 0;
	BFont historyFont;
	historyFont.SetSize( 11.0 );
	textview->SetBaseFontAndColor( historyFont, black );
	textview->SetStylable( true );
	if( mManager -> prefs -> ReadBool( "cw_lk", true ) ) {
		linksEnabled = true;
		textview->SetShowLinks( true );
	} else
		textview->SetShowLinks( false );

	rect.top = rect.bottom + 15;
	rect.bottom = (box -> Bounds()).bottom - 25;
	rect.right = (box -> Bounds()).right - 18;
	rect.left = rect.right - 50;
	GButton *close_button = new GButton(rect,"close_button", _("closebutton") , new BMessage(CLOSE_HISTORY_WINDOW),
		B_FOLLOW_RIGHT | B_FOLLOW_BOTTOM);
	
	rect.bottom = (box -> Bounds()).bottom - 25;
	rect.left = (box -> Bounds()).left + 18;
	rect.right = rect.left + 80;
	GButton *delete_button = new GButton(rect,"delete_button", _("deletebutton") , new BMessage(CLEAR_HISTORY_IO),
		B_FOLLOW_LEFT | B_FOLLOW_BOTTOM);

	box -> AddChild(close_button);
	box -> AddChild(delete_button);
	box -> AddChild(new BScrollView("scroll_history", textview, 
		B_FOLLOW_LEFT_RIGHT | B_FOLLOW_TOP_BOTTOM, B_FULL_UPDATE_ON_RESIZE, false, true));
	
	AddChild(box);
	
	//Adding History.
	
	// Need to start the Looper before requesting all the messages, otherwise the queue can get overloaded.
	Hide();
	Show();
	
	mManager -> history -> getHistoryIO() -> ReadAll(uin, own_uin, this);
	
	SetEncoding(encoding);
}

HistoryWindow::~HistoryWindow() {

	mManager -> windows -> CloseGeneralWindow(this);

}

void HistoryWindow::MessageReceived(BMessage *msg) {
	
	switch(msg -> what) {
	
	case CT_OUTGOING_MESSAGE:
		AddMyStatement( msg -> FindString("TEXT") , (time_t) msg -> FindInt64("time") );
		break;
	case CT_INCOMING_MESSAGE:
		ProcessIncomingIM( msg, (time_t) msg -> FindInt64("time") );
		break;
	case CT_INCOMING_URL:
		ProcessIncomingURL( msg, (time_t) msg -> FindInt64("time") );
		break;
	case CT_OUTGOING_URL:
		DisplaySentURL( msg, (time_t) msg -> FindInt64("time") );
		break;
	case CT_OFFLINE_DONE:
		break;

	case CLOSE_HISTORY_WINDOW:
		
		if(Lock()) {
			Quit();	
		}
		break;
	
	case GIMICQ_LAST_MESSAGE:
		Show();
		break;

	case CLEAR_HISTORY_IO: {
		
		//Alert...are you sure? This will empty the history on disc.

// Comment out the BAlert code and it doesnt crash.
		BAlert *alert = new BAlert("delete warning", _("text61"),
			 					 _("nobutton") , _("yesbutton") , NULL , B_WIDTH_AS_USUAL , B_OFFSET_SPACING , B_WARNING_ALERT );
		alert -> SetShortcut(0, B_ESCAPE);
		int32 button = alert -> Go();
		if(button == 0)	// If Cancel
			return;
			
#if 0
#warning Need to delete history.
		history_list.Clear();
		mManager -> history -> getHistoryIO() -> ReWrite(history_list, uin, own_uin);
#endif
		SetEncoding(useEncoding);	
		
		// Delete in memory.
		mManager -> history -> Clear(uin, own_uin);
		
		// if a chatwindow is open... clear the chatwindow.
		BMessage *chat_msg = new BMessage(GIMICQ_CLEAR_HISTORY_MEM);
		chat_msg -> AddInt32("UIN", uin);
		chat_msg -> AddInt32("wtype", USER_MESSAGE_TYPE);
		
		// Ugly... just a way to communicate with a chatwindow.
		mManager -> windows -> ForwardIncomingMessage( chat_msg, false );
		
		delete chat_msg;
		break;
	}
	default:
		BWindow::MessageReceived(msg);
		break;
	}
	
}

bool HistoryWindow::QuitRequested() {

	PRINT(("HistoryWindow::QuitRequested\n"));
	return true;

}

//-----------------------------------------------------

void HistoryWindow::SetEncoding( uint32 encoding ) {
	
	PRINT(("HistoryWindow::SetEncoding\n"));
//	bool keepGoing;
//	BMessage* msg;
//	chatEntry entry;

	Lock();
	
	// set the encoding
	useEncoding = encoding;

	// clear the textview
	textview->Clear();
	
	Unlock();
}

void HistoryWindow::DisplayDateTime(time_t display_time) {

	PRINT(("HistoryWindow::DisplayDateTime\n"));
	char strdate[20], strtime[10];

	strftime( strdate, 20, "%Y-%m-%d ", localtime(&display_time));
	strftime( strtime, 10, "%H:%M ", localtime(&display_time));

	textview->SetFontColor( kDate );
	textview->InsertSomeText( strdate );

	textview->SetFontColor( kTime );
	textview->InsertSomeText( strtime );

}

void HistoryWindow::AddMyStatement( const char* statement, time_t display_time ) {

	PRINT(("HistoryWindow::AddMystatement\n"));
	HTMLParser parse;
	styleList styles;
	char text[1024];

	textview->ClearInsertStuff();

	DisplayDateTime(display_time);

	// insert our name (red font)
	textview->SetFontAttribute( B_BOLD_FACE, true );
	textview->SetFontColor( kThis );
	textview->InsertSomeText( own_nick.String() );
	textview->InsertSomeText( ": " );
	textview->ResetFontToBase();

	// now parse and insert the HTML
	FixLineBreaks(text, statement);
	parse.Parse( text );
	//parse.Parse( statement );
	styles = parse.Styles();
	textview->AddStyledText( parse.ParsedString(), styles );
	delete styles;

	// Finalize the insertion, and update
	textview->AddStatement();
	
}

void HistoryWindow::DisplaySentURL( BMessage* incomingIM, time_t display_time ) {

	PRINT(("HistoryWindow::DisplaySentUrl\n"));
	HTMLParser parse;
	styleList styles;
	rgb_color purple = {128,0,128};
	char msg[2048];
	char url[1024];
	char text[1024];

	BString text_str = _("text41");
	text_str += ": ";

	// grab the stuff from the message
	if( incomingIM->HasString("TEXT") )
		//strcpy( text, incomingIM->FindString("TEXT") );
		FixLineBreaks( text, incomingIM->FindString("TEXT") );
	if( incomingIM->HasString("URL") )
		strcpy( url, incomingIM->FindString("URL") );
	if( !url )
		return;

	// insert some more stuff
	sprintf( msg, "<a href=\"%s\">%s</a>", url, url );
	if( text && strlen(text) != 0 ) {
		strcat( msg, "<br>" );
		strcat( msg, text );
	}

	DisplayDateTime(display_time);

	// insert the other person's name (blue font)
	textview->SetFontAttribute( B_BOLD_FACE, true );
	textview->SetFontColor( purple );
	textview->InsertSomeText( text_str.String() );
	textview->ResetFontToBase();
	
	// now parse and insert the HTML
	parse.Parse( msg );
	styles = parse.Styles();
	textview->AddStyledText( parse.ParsedString(), styles );
	delete styles;

	// Finalize the insertion, and update the timestamp
	textview->AddStatement();
}

//-----------------------------------------------------

void HistoryWindow::ProcessIncomingIM( BMessage* incomingIM, time_t display_time ) {

	PRINT(("HistoryWindow::ProcessIncomingIM\n"));
	HTMLParser parse;
	styleList styles;
	char parsedMessage[2048], finalMessage[2048];
	int32 srcLen, destLen = 2048, state = 0;

	textview->ClearInsertStuff();

	DisplayDateTime(display_time);

	// insert the other person's name (blue font)
	textview->SetFontAttribute( B_BOLD_FACE, true );
	textview->SetFontColor( kThat );
	textview->InsertSomeText( nick.String() );
	textview->InsertSomeText( ": " );
	textview->ResetFontToBase();

	// now parse and insert the HTML
	FixLineBreaks( parsedMessage, incomingIM->FindString("TEXT") );
	//parse.Parse( (char*)incomingIM->FindString("TEXT") );
	srcLen = strlen(parsedMessage);
	convert_to_utf8( useEncoding, const_cast<char*>(parsedMessage), &srcLen, finalMessage, &destLen, &state );
	finalMessage[destLen] = '\0';
	parse.Parse( finalMessage );
	styles = parse.Styles();
	textview->AddStyledText( parse.ParsedString(), styles );
	delete styles;

	// Finalize the insertion, and update the timestamp
	textview->AddStatement();

}

//-----------------------------------------------------

void HistoryWindow::ProcessIncomingURL( BMessage* incomingIM, time_t display_time ) {

	PRINT(("HistoryWindow::ProcessIncomingURL\n"));
	HTMLParser parse;
	styleList styles;
	rgb_color green = {0,128,0};
	char msg[2048];
	char msg2[1024], msg3[1024];
	char* url = NULL;
	char* text = NULL;
	int32 srcLen, destLen = 1024, state = 0;
	
	BString text_str = _("text42");
	text_str += ": ";

	textview->ClearInsertStuff();
	
	// grab the stuff from the message
	if( incomingIM->HasString("TEXT") ) {
		FixLineBreaks( msg2, const_cast<char*>( incomingIM->FindString("TEXT") ));
		srcLen = strlen(msg2);
		convert_to_utf8( useEncoding, const_cast<char*>(msg2), &srcLen, msg3, &destLen, &state );
		msg3[destLen] = '\0';
		text = msg3;
	}
	if( incomingIM->HasString("URL") )
		url = const_cast<char*>( incomingIM->FindString("URL") );
	if( !url )
		return;

	// insert some more stuff
	sprintf( msg, "<a href=\"%s\">%s</a>", url, url );
	if( text && strlen(text) != 0 ) {
		strcat( msg, "<br>" );
		strcat( msg, text );
	}

	DisplayDateTime(display_time);

	// insert the other person's name (blue font)
	textview->SetFontAttribute( B_BOLD_FACE, true );
	textview->SetFontColor( green );
	textview->InsertSomeText( text_str.String() );
	textview->ResetFontToBase();
	
	// now parse and insert the HTML
	parse.Parse( msg );
	styles = parse.Styles();
	textview->AddStyledText( parse.ParsedString(), styles );
	delete styles;

	// Finalize the insertion, and update the timestamp
	textview->AddStatement();
}

void HistoryWindow::FixLineBreaks( char* buffer, const char* input ) {

	PRINT(("HistoryWindow::FixLineBreaks\n"));
	// Used to grab bytes
	char c, next;

	// position pointer for the buffer
	int i = 0;

    // Loop thru, grabbing bytes as we go
    while( (*input) != 0 ) {

	   c = (*input);
       if( c == CR ) {

         // attempt to get the next byte of input to see whether it's a
         // CR/LF pair (dos), or just a CR (mac)
         if( (next = *(input+1)) != 0 ) {

             // It's a CR/LF (dos), so the line has ended
             if( next == LF ) {
				buffer[i++] = '\n';
				input += 2;
				continue;
             }

             // Just a CR... put the next character onto the line and return it
             // Need to move the position back a byte so the first character
             // of the next line doesn't get lost!
             buffer[i++] = '\n';
             input++;
             continue;

         } else {    // couldn't get next byte; therefore the input stream is eof

			buffer[i++] = '\0';
			return;
         }
       }

       // If it's an LF, the line has ended no matter what the format.
       if( c == LF ) {

			buffer[i++] = '\n';
         	input++;
			continue;
       }

       // This byte is nothing special. Stick it onto the line.
       buffer[i++] = c;
       input++;
    }

    // apparently the input is now eof
    buffer[i++] = '\0';
}

// Adjust the size of the BTextView's text rectangles
// when the window is resized.
void HistoryWindow::FrameResized(float width, float height) {

	PRINT(("HistoryWindow::FramResized\n"));
	// set the new text rects
	BRect textrect = textview->Bounds();
	textrect.InsetBy( 2.0, 2.0 );
	textview->SetTextRect( textrect );
}


