//-------------------------------------------------------------------------------
// Non polling tool tip class.
// Changed to TToolTip by CGSoftware 1.15.2000.
//  Major overhaul by Alan (CGSoftware) 7.31.2000
//-------------------------------------------------------------------------------
// Major portions of ShowTip() from Marco Nelissen's BubbleHelper class.


//-------------------------------------------------------------------------------
//-------------------------------------------------------------------------------

#include <be/app/Application.h>
#include <be/app/MessageFilter.h>
#include <be/app/MessageRunner.h>
#include <be/app/Messenger.h>
#include <be/interface/Region.h>
#include <be/interface/Screen.h>
#include <be/interface/TextView.h>
#include <be/interface/Window.h>
#include <be/support/Autolock.h>
#include <be/support/Locker.h>
#include <be/support/String.h>

#include "ToolTip.h"

namespace DCL
{

//-------------------------------------------------------------------------------
// Private helper class for TToolTip
//-------------------------------------------------------------------------------

#define _CHECK_TIP		'cftt'		// check for tool tip
#define TIP_INTERVAL	750000		// 3/4 of a second
#define TIP_WHERE		"deco:tt:where"

class TTipFilter : public BMessageFilter
{
	public:
			TTipFilter(TToolTip& ToolTip, BView* Who, const char* Tip);
		
		virtual
			~TTipFilter();
			
		filter_result
			Filter(BMessage *Message, BHandler **Target);
		
		void
			ShowTip(const BPoint& Where);
			
		void
			HideTip();
			
		void
			SetTip(const char* TipText);

		void
			Enabled(bool Enabled);
		
	private:

		void
			OnMouseMoved(BMessage* Message, BView* Handler);

		BMessageRunner*		FRunner;
		BMessenger*			FMessenger;
		BString 			FTip;
		BPoint				FWhere;
		bool				FEnabled;
		TToolTip&			FToolTip;
		const BView*		FView;				
		
};
//-------------------------------------------------------------------------------
//-------------------------------------------------------------------------------
TTipFilter::TTipFilter(TToolTip& ToolTip, BView* Who, const char* Tip)
		   :BMessageFilter(B_ANY_DELIVERY, B_LOCAL_SOURCE),
		    FRunner(NULL),
		    FMessenger(NULL),
		    FTip(Tip),
		    FWhere(-1, -1),
		    FEnabled(true),
		    FToolTip(ToolTip),
		    FView(Who)
{

}
//-------------------------------------------------------------------------------
filter_result TTipFilter::Filter(BMessage* Message, BHandler** Target)
{
	filter_result result = B_DISPATCH_MESSAGE;
	 
	switch (Message->what)
	{
		case B_MOUSE_MOVED:
		{
			OnMouseMoved(Message, (BView*)(*Target));
			break;
		}
		
		case _CHECK_TIP:
		{
			if ( FWhere == Message->FindPoint(TIP_WHERE) )
			{
				delete FRunner;
				FRunner = NULL;
				ShowTip(((BView*)(*Target))->ConvertToScreen(FWhere));
			}
			
			result = B_SKIP_MESSAGE;
			break;
		}
		
		// Any mouse action should hide the tip.
		case B_MOUSE_DOWN:
		case B_MOUSE_UP  :
			HideTip();
			break;
	}
	 
	return result;
}
//-------------------------------------------------------------------------------
TTipFilter::~TTipFilter()
{
	delete FRunner;
	delete FMessenger;
	FToolTip.RemoveTip(FView);
}
//-------------------------------------------------------------------------------
void 
TTipFilter::ShowTip(const BPoint &Where)
{
	FToolTip.ShowTip(Where, FTip);
}
//-------------------------------------------------------------------------------
void 
TTipFilter::HideTip()
{
	FToolTip.HideTip();
}
//-------------------------------------------------------------------------------
void TTipFilter::SetTip(const char* TipText)
{
	FTip = TipText;
}
//-------------------------------------------------------------------------------
void TTipFilter::Enabled(bool Enabled)
{
	FEnabled = Enabled;
}
//-------------------------------------------------------------------------------
void TTipFilter::OnMouseMoved(BMessage* Message, BView* Handler)
{
	if ( B_EXITED_VIEW == Message->FindInt32("be:transit") )
	{
		delete FRunner;
		FRunner = NULL;
		HideTip();
		return;
	}
	
	BPoint pt;
	Message->FindPoint("be:view_where", &pt);
	if ( pt == FWhere )
	{
		return;
	}
	else
	{
		FWhere = pt;
	}
	
	BMessage msg(_CHECK_TIP);
	msg.AddPoint(TIP_WHERE, FWhere);

	if ( NULL != FRunner )
	{
		delete FRunner;
	}
	else
	{
		if ( NULL == FMessenger )
		{
			// if this fails, nothing we can do, oh well...
			FMessenger = new BMessenger(Handler);
		}
	}
	
	if ( FMessenger )
	{
		// If this fails, oh well, nothing we can do.
		FRunner = new BMessageRunner(*FMessenger, &msg, TIP_INTERVAL, 1);
	}

	HideTip();
}
//-------------------------------------------------------------------------------



//-------------------------------------------------------------------------------
// The tip locker
static BLocker sTipLock;

// DCL TToolTip instance
static TToolTip* sToolTip = NULL;
// ------------------------------------------------------------------------------
// Clean up!
class TToolTipCleanup
{
	public:
		~TToolTipCleanup()
		{
			if ( sToolTip )
			{
				delete sToolTip;
			}
		}
};

static TToolTipCleanup sCleaner;

//-------------------------------------------------------------------------------
// Function for getting at it!

TToolTip& ToolTip()
{
	BAutolock lock(sTipLock);
	if( lock.IsLocked() )
	{
		if ( NULL == sToolTip )
		{
			sToolTip = new TToolTip;
		}
	}
	else
	{
		debugger("\n\n\tCould not lock Tip, something is very wrong.\n\n");
	}
	
    return *sToolTip;
}
//-------------------------------------------------------------------------------


//-------------------------------------------------------------------------------
// TToolTip class proper
//-------------------------------------------------------------------------------

//-------------------------------------------------------------------------------
TToolTip::TToolTip()
         :FRunCount(0),
          FEnabled(true)
{
	// You will need to create a BApplicaiton object before you can
	//  use this.
    if(!be_app_messenger.IsValid())
    {
        debugger("\n\n\tYou will need to create a BApplication object before using the TToolTip silly!\n\n");
    }

    FTipWindow = new BWindow(BRect(-100,-100,-50,-50),
                             "",
                             B_BORDERED_WINDOW_LOOK,
                             B_FLOATING_ALL_WINDOW_FEEL,
                         	 B_NOT_MOVABLE|B_AVOID_FOCUS|B_ASYNCHRONOUS_CONTROLS);
                         	 
    BTextView* tv = new BTextView(BRect(0,0,50,50),
                                  "",
                                  BRect(0,0,50,50),
                                  B_FOLLOW_ALL_SIDES,
                                  B_WILL_DRAW);
    tv->MakeEditable(false);
    tv->MakeSelectable(false);
    tv->SetWordWrap(false);
    tv->SetLowColor(255, 255, 225);
    tv->SetHighColor(0,0,0);
    
    FTipWindow->AddChild(tv);
    FTipWindow->Hide();
    FTipWindow->Show();
    FTipWindow->Lock();
    FTipWindow->Activate(false);
    FTipWindow->Unlock();
}
//-------------------------------------------------------------------------------
TToolTip::~TToolTip()
{
	// The BApplication object seems to want to clean this up for us.
	//	FTipWindow->Lock();
	//	FTipWindow->Quit();
}
//-------------------------------------------------------------------------------
void TToolTip::SetTip(BView *View, const char *Text)
{
	if (FFilterMap.find(View) == FFilterMap.end())
	{
		TTipFilter* filter = new TTipFilter(*this, View, Text);
		View->AddFilter(filter);
		filter->Enabled(FEnabled);
		FFilterMap[View] = filter;
	}
	else
	{
		FFilterMap[View]->SetTip(Text);
	}
}
//-------------------------------------------------------------------------------
void TToolTip::Enable(BView* Which, bool Enable)
{
	FFilterMap[Which]->Enabled(Enable);
}
//-------------------------------------------------------------------------------
void TToolTip::EnableAll(bool Enable)
{
	FEnabled = Enable;
	
	std::map<const BView*, TTipFilter*>::iterator iter;
	
	for( iter = FFilterMap.begin(); iter != FFilterMap.end(); iter++ )
	{
		(*iter).second->Enabled(FEnabled);
	}
}
//-------------------------------------------------------------------------------
void TToolTip::ShowTip(const BPoint &Where, const BString& Tip)
{
	BAutolock lock(FLock);
	
	if ( ! lock.IsLocked() )
	{
		debugger( "\n\n\tCould not acquire lock.\n\n" );
	}
	
	BAutolock winlock(FTipWindow);
	if( !winlock.IsLocked() )
	{
		debugger( "\n\n\tCould not acquire lock.\n\n" );
	}
	
	BTextView* tv = dynamic_cast<BTextView*>(FTipWindow->ChildAt(0));
	
	if ( NULL == tv )
	{
		debugger( "\n\n\tWe have serous problems.\n\n" );
	}
	
	tv->SetText(Tip.String());

    float height = tv->TextHeight(0,2E6);
    float width = 0.0;
    int numlines = tv->CountLines();
    int linewidth;
    
    for ( int i=0; i < numlines; i++)
    {
        if ( (linewidth = (int)tv->LineWidth(i)) > width)
        {
            width=linewidth;
        }
    }
    
    FTipWindow->ResizeTo( width + 2, height - 3 );
    tv->SetTextRect( BRect( 1, -1, width, height - 3 ) );

    BScreen screen;
    BPoint dest = Where + BPoint(0,20);
    BRect screenframe = screen.Frame();
    
    if( (dest.y + height) > (screenframe.bottom - 3) )
    {
        dest.y = dest.y - (16 + height + 8);
	}
	
    if( (dest.x + width + 4) > (screenframe.right) )
    {
        dest.x = dest.x - ( (dest.x + width + 4) - screenframe.right );
	}
	
    FTipWindow->MoveTo(dest);

    if ( FTipWindow->IsHidden() )
    {
        FTipWindow->SetWorkspaces(B_CURRENT_WORKSPACE);
		FTipWindow->Show();
    }
}
//-------------------------------------------------------------------------------
void TToolTip::HideTip()
{
	BAutolock lock(FLock);
	
	if ( ! lock.IsLocked() )
	{
		debugger( "\n\n\tCould not acquire lock.\n\n" );
	}
	
	BAutolock winlock(FTipWindow);
	if( !winlock.IsLocked() )
	{
		debugger( "\n\n\tCould not acquire lock.\n\n" );
	}
	
    if ( !FTipWindow->IsHidden() )
    {
        FTipWindow->Hide();
	}
}
//-------------------------------------------------------------------------------
void TToolTip::RemoveTip(const BView* Which)
{
	BAutolock lock(FLock);
	
	if ( ! lock.IsLocked() )
	{
		debugger( "\n\n\tCould not acquire lock.\n\n" );
	}
	FFilterMap.erase(Which);	
}
//-------------------------------------------------------------------------------


//-------------------------------------------------------------------------------

}	//namespace DCL


