// Z-Viewer, written by Zenja Solaja, December 2000
// Z-Viewer is a zoomable image viewer.

#include <InterfaceKit.h>
#include <StorageKit.h>
#include <TranslationKit.h>
#include <SupportKit.h>

#include "ZWindow.h"
#include "ZView.h"
#include "main.h"
#include "linked_list.h"
#include "ZDisplayLibs.h"
#include "ZScrollView.h"

#include <KernelKit.h>
#include "AnimateThread.h"

//==============================================
ZWindow::ZWindow(BRect screenbounds)
	:BWindow(screenbounds, "Z-Viewer 0.96",B_TITLED_WINDOW, 0)
{
	handle_thread = 0;
	
	screenbounds.OffsetTo(B_ORIGIN);
	BRect menuBarRect = screenbounds;
	menuBarRect.bottom = MENU_BAR_HEIGHT;

	PrepareMenu(menuBarRect);	
	AddChild(fmenuBar);
		
	fimage_title = NULL;
			
	screenbounds.top += MENU_BAR_HEIGHT; //Leave space for Taskbar
	zv_backview = new ZBackgroundView(screenbounds);
	AddChild(zv_backview);
	
	//PrepareBitmap("startup.jpg");
	Show();
}

//=================================================
void ZWindow::PrepareMenu(BRect menuRect)
{
	// File menu
	fmenuBar = new BMenuBar(menuRect, "MenuBar");
	BMenu *File_menu = new BMenu("File");
	BMenuItem *item;
	File_menu->AddItem(item = new BMenuItem("Open File/Directory ", new BMessage(SELECT_DIR_MSG), 'O'));
	item->SetTarget(be_app);
	File_menu->AddSeparatorItem();
	File_menu->AddItem(item = new BMenuItem("About", new BMessage(ABOUT_MSG)));
	File_menu->AddItem(item = new BMenuItem("Quit", new BMessage(B_CLOSE_REQUESTED),'Q'));
	
	// Image menu
	fimage_menu = new BMenu("Image");
	//fimage_menu->AddItem(item = new BMenuItem("Next Image", new BMessage(NEXT_IMAGE_MSG)));
	//fimage_menu->AddItem(item = new BMenuItem("Previous Image", new BMessage(PREV_IMAGE_MSG)));
	//fimage_menu->AddSeparatorItem();
	//fimage_menu->AddItem(item = new BMenuItem("First Image in List", new BMessage(FIRST_IMAGE_MSG)));
	//fimage_menu->AddItem(item = new BMenuItem("Last Image in List", new BMessage(LAST_IMAGE_MSG)));
	//fimage_menu->AddSeparatorItem();
	fimage_menu->AddItem(item = new BMenuItem("Zoom In", new BMessage(ZOOM_IN_MSG)));
	fimage_menu->AddItem(item = new BMenuItem("Zoom Out", new BMessage(ZOOM_OUT_MSG)));
	fimage_menu->AddItem(item = new BMenuItem("Activate Image Smoothing", new BMessage(BLUR_IMAGE_MSG)));
	fimage_menu->AddSeparatorItem();
	fimage_menu->AddItem(item = new BMenuItem("Full Screen", new BMessage(ZOOM_FULL_MSG),'F'));
	fimage_menu->AddItem(item = new BMenuItem("Animate", new BMessage(ANIMATION_MSG), 'A'));
		
	fmenuBar->AddItem(File_menu);
}
		
//=================================================
void ZWindow::MessageReceived(BMessage *message)
{
	switch(message->what){
		case ABOUT_MSG: {
			BAlert *alert = new BAlert("",
					"Z-Viewer v0.96, written by Zenja Solaja, Mar 2001.\n\
					\nKeyboard shortcuts (numpad also works):
					\nPage Down		-	Next image in list\
					\nPage Up		-	Previous image in list\
					\nHome			-	First image in list\
					\nEnd			-	Last image in list
					\nInsert			-	Toggle Animation\
					\n+			-	Zoom In\
					\n-			-	Zoom Out\
					\n*			-	Zoom To Fit\
					\nDel 			-	Activate image smoothing
					\nCursor keys	-	Pan Zoomed Image\
					\nEnter			-	Center Image
					\nAlt-O			-	Select new Image/Directory\
					\nESC			-	Quit", "All Right!!!", NULL, NULL, B_WIDTH_AS_USUAL, B_IDEA_ALERT);
					
			alert->Go(); 
			break;
		}
		case NEXT_IMAGE_MSG: {
			if (list_navigation->position == list_navigation->tail)
				list_navigation->position = list_navigation->head;
			else
				list_navigation->position = list_navigation->position->next;
			PrepareBitmap(list_navigation->position->contents);
			break;
		}
		case PREV_IMAGE_MSG: {
			if (list_navigation->position == list_navigation->head)
				list_navigation->position = list_navigation->tail;
			else
				list_navigation->position = list_navigation->position->prev;
			PrepareBitmap(list_navigation->position->contents);
			break;
		}
		case FIRST_IMAGE_MSG: {
			list_navigation->position = list_navigation->head;
			PrepareBitmap(list_navigation->position->contents);
			break;
		}
		case LAST_IMAGE_MSG: {
			list_navigation->position = list_navigation->tail;
			PrepareBitmap(list_navigation->position->contents);
			break;
		}
		case BLUR_IMAGE_MSG:
		{
			bitmap_view->BlurImage();
			break;
		}
		case ZOOM_IN_MSG:
		{
			bitmap_view->AnimatedZoom(ZOOM_IN);	
			break;
		}
		case ZOOM_OUT_MSG:
		{
			bitmap_view->AnimatedZoom(ZOOM_OUT);	
			break;
		}
		case ZOOM_FULL_MSG: {
			fullscreen = !fullscreen;
			BRect aRect;
			
			if (fullscreen)	{
				ZScreenBounds(&aRect);
				zv_window->MoveTo(0,0);
			}
			else {
				aRect.Set(100,100,800,600);
				zv_window->MoveTo(100,100);
			}
			zv_window->ResizeTo(aRect.Width(), aRect.Height() + MENU_BAR_HEIGHT);
			bitmap_view->ResizeView();
			bitmap_view->AdjustZoom(1.0);
			bitmap_view->Zoom();
			
			BMenuItem *aItem = fimage_menu->FindItem("Full Screen");
			aItem->SetMarked(fullscreen); 
			break;
		}
		case ANIMATION_MSG: {
			animation = !animation;
			if (animation)
				resume_thread(handle_thread);	
			else
				suspend_thread(handle_thread);
			
			BMenuItem *aItem = fimage_menu->FindItem("Animate");
			aItem->SetMarked(animation); 
			break;
		}
		default:
			BWindow::MessageReceived(message);
	}
}

//===============================================
bool ZWindow::QuitRequested()
{
	if (handle_thread)
		kill_thread(handle_thread);
	be_app->PostMessage(B_QUIT_REQUESTED);
	return (true);
}

//====================================================
void ZWindow::FrameResized(float width, float height)
{
	bitmap_view->ResizeView();
	bitmap_view->Zoom();
}

//===============================================
void ZWindow::PrepareBitmap(char *filename)
{
	BitmapView *old_view = bitmap_view;
	
	BBitmap *bitmap_image = BTranslationUtils::GetBitmapFile(filename,NULL);
	BRect bRect = bitmap_image->Bounds();
				
	BitmapView *new_bitmap_view = new BitmapView(bitmap_image, bRect, "Bitmap_Image");
	new_bitmap_view->ResizeView();
	zv_window->AddChild(new_bitmap_view);
	bitmap_view = new_bitmap_view;
			
	if (old_view)
	{
		kill_thread(handle_thread);
		zv_window->RemoveChild(old_view);
		delete old_view;
	}
	else // First image ever shown
	{
		fmenuBar->AddItem(fimage_menu); 
	}
	
	zv_listwindow->UpdateSelection();
	PrepareTitle(filename, bRect); //add title
	
	handle_thread = spawn_thread(animation_thread, "Animation Thread", B_NORMAL_PRIORITY, NULL);
	if (animation)
		resume_thread(handle_thread);
}

//===============================================
void ZWindow::PrepareTitle(char *filename, BRect size)
{
	BPath thePath(filename);
	BMenuItem *item;
	BString size_string;
	
	size_string << (int32) (size.Width()+1) << " x " << (int32) (size.Height()+1);
	
	if (fimage_title)
	{
		fmenuBar->RemoveItem(fimage_title);
		delete fimage_title;
	}
	fimage_title = new BMenu(thePath.Leaf());
	
	
	fimage_title->AddItem(item = new BMenuItem(size_string.String(), 0, 0, 0));
	fmenuBar->AddItem(fimage_title);
}