// ZViewer, written by Zenja Solaja, December 2000
// ZViewer is a zoomable image viewer.

#include <InterfaceKit.h>
#include <AppKit.h>

#include "ZView.h"
#include "main.h"
#include "ZDisplayLibs.h"

//===============================================
ZBackgroundView::ZBackgroundView(BRect frame)
	:BView(frame, "ZBackgroundVIew", B_FOLLOW_ALL, B_WILL_DRAW)
{
	SetViewColor(B_TRANSPARENT_COLOR); //Prevent screen flashes
}

//===============================================
void ZBackgroundView::Draw(BRect frame)
{
	rgb_color black = {0, 0, 0, 255};
	SetHighColor(black);
	SetLowColor(black);
	FillRect(frame, B_SOLID_HIGH);
}

//=================================================
void ZBackgroundView::PaintBackground(BRect original, BRect source)
{
	BRect screenbounds;
	ZScreenBounds(&screenbounds);
			
	float stretch, x, y;
	
	if ((original.Width()/original.Height()) > 1) // aspect > 1
		stretch = screenbounds.Width()/source.Width();
	else
		stretch = screenbounds.Height()/source.Height();
	
	x = stretch * (source.Width()-original.Width())/2;
	y = stretch * (source.Height()-original.Height())/2;
	
	//Draw vertical black bars (left and right side of screens)
	{
		BRect aRect = screenbounds;	
		aRect.top -= MENU_BAR_HEIGHT;
		aRect.right = x;
		zv_backview->Draw(aRect);  //left
		aRect.right = screenbounds.right;
		aRect.left = screenbounds.right - x;
		zv_backview->Draw(aRect);  //right
	}
	//Draw horizontal black bars (top and bottom of screens)
	{
		BRect aRect = screenbounds;	
		aRect.top -= MENU_BAR_HEIGHT;
		aRect.bottom = y;
		aRect.left = x;		// Dont repaint vertical bars since they're already painted
		aRect.right = screenbounds.right - x;
		zv_backview->Draw(aRect);  //top
		aRect.bottom = screenbounds.bottom;
		aRect.top = screenbounds.bottom - y - MENU_BAR_HEIGHT;
		zv_backview->Draw(aRect);  //bottom
	}
}

//================================================
void ZBackgroundView::MouseDown(BPoint point)
{
}	

//================================================
BitmapView::BitmapView(BBitmap* bitmap, BRect frame, const char* name,
	uint32 resizingMode, uint32 flags)
: BView(frame, name, resizingMode, flags), image(bitmap)
{
	SetViewColor(B_TRANSPARENT_COLOR); //Prevent screen flashes
	
	BRect original = image->Bounds();
	center.x = original.Width()/2;
	center.y = original.Height()/2;
	
	AdjustZoom(1.0);
}

//===============================================
BitmapView::~BitmapView()
{
	if (image)
		delete image;
}

//===============================================
void BitmapView::AdjustZoom(float new_zoom)
{
	BRect original = image->Bounds();
	BRect screenbounds;
	ZScreenBounds(&screenbounds);
	
	float x = original.Width();
	float y = original.Height();
	float a = screenbounds.Width();
	float b = screenbounds.Height();
	
	if (x / y > 1) //image aspect
	{
		if (y*a/x > b)	//scaled image outside window
			zoom_factor = b / (y*a/x);
		else 
			zoom_factor = new_zoom;
	}
	else
		zoom_factor = new_zoom;
}

//===============================================
void BitmapView::AttachedToWindow()
{
}

//===============================================
void BitmapView::Draw(BRect updateRect)
{
	Zoom();
}

//===============================================
void BitmapView::AnimatedZoom(int zoom_direction)
{
	float increment = (animation ? ANIM_ZOOM_INCREMENT : NOANIM_ZOOM_INCREMENT);
	
	int num_frames = (animation ? ANIM_ZOOM_NUM_FRAMES : NOANIM_ZOOM_NUM_FRAMES);  		
	
	if (zoom_direction == ZOOM_OUT)
		increment = -increment;
	increment *= zoom_factor;
	
	if (image)
	{
		for (int i = 0; i < num_frames; i++)
		{
			zoom_factor += increment;
			if (zoom_factor > 8) {
				zoom_factor = 8;
				break;
			}
			else if (zoom_factor < 0.05) {
				zoom_factor = 0.05;
				break;
			}
			else
			{
				Zoom();
			}
		}
	}
}	

//===============================================
void BitmapView::Zoom()
{
	BRect screenbounds;
	ZScreenBounds(&screenbounds);
	
	if (image)
	{
		float source_x, source_y, aspect;
		BRect source = image->Bounds();
		BRect original = source;
		
		aspect = source.Width() / source.Height();
		if (aspect <= 1) //This one gave me a headache
		{
			source_y = source.Height() / zoom_factor;
			source_x = screenbounds.Width() * source_y / screenbounds.Height();	
		}
		else
		{
			source_x = source.Width()/zoom_factor;
			source_y = screenbounds.Height() * source_x / screenbounds.Width();
		}
				
		//adjust center_x and center_y
		if (zoom_factor > 1)
		{
			float y_limit = original.Height()/zoom_factor/2;
			float x_limit = original.Width()/zoom_factor/2;
			if (center.y < y_limit)
				center.y = y_limit;
			else if (center.y > (original.bottom - y_limit))
				center.y = original.bottom - y_limit;
			if (center.x < x_limit)
				center.x = x_limit;
			else if (center.x > (original.right- x_limit))
				center.x = original.right - x_limit;
		}
		
		//adjust visible window		
		source.left = center.x - source_x/2;
		source.right = center.x + source_x/2;
		source.top = center.y - source_y/2;
		source.bottom = center.y + source_y/2;
		
		DrawBitmap(image, source, screenbounds);
		zv_backview->PaintBackground(original, source);
		MakeFocus();
	}
}

//===============================================
void BitmapView::ResizeView()
{
	if (image) {
		BRect screenbounds;
		ZScreenBounds(&screenbounds);
		ResizeTo(screenbounds.Width(), screenbounds.Height() + MENU_BAR_HEIGHT);
	}
}

//===============================================
void BitmapView::KeyDown(const char *bytes, int32 numbytes)
{
	BRect original = image->Bounds();
	
	BRect screenbounds;
	ZScreenBounds(&screenbounds);
	
	float y_limit = original.Height()/zoom_factor/2;
	float x_limit = original.Width()/zoom_factor/2;
		
	switch (*bytes) {
		case B_ESCAPE: {
			be_app->PostMessage(B_QUIT_REQUESTED);
			break;
		}
		case B_INSERT: {
			zv_window->PostMessage(ANIMATION_MSG);
			break;
		}
		case B_PAGE_UP: {
			zv_window->PostMessage(PREV_IMAGE_MSG);
			break;
		}
		case B_PAGE_DOWN: {
			zv_window->PostMessage(NEXT_IMAGE_MSG);
			break;
		}
		case B_HOME: {
			zv_window->PostMessage(FIRST_IMAGE_MSG); 
			break;
		}
		case B_END: {
			zv_window->PostMessage(LAST_IMAGE_MSG);
			break;
		}
		case '+': {
			AnimatedZoom(ZOOM_IN);
			break;
		}
		case '-': {
			AnimatedZoom(ZOOM_OUT);
			break;
		}
		case B_UP_ARROW: {
			if (zoom_factor > 1)
			{
				center.y -= PAN_FACTOR;
				if (center.y < y_limit)
					center.y = y_limit;
				Zoom();
			}
			break;
		}
		case B_DOWN_ARROW: {
			if (zoom_factor > 1)
			{
				center.y += PAN_FACTOR;
				if (center.y > (original.bottom - y_limit))
					center.y = original.bottom - y_limit;
				Zoom();
			}
			break;
		}
		case B_LEFT_ARROW: {
			if (zoom_factor > 1)
			{
				center.x -= PAN_FACTOR;
				if (center.x < x_limit)
					center.x = x_limit;
				Zoom();
			}
			break;
		}
		case B_RIGHT_ARROW: {
			if (zoom_factor > 1)
			{
				center.x += PAN_FACTOR;
				if (center.x > (original.right- x_limit))
					center.x = original.right - x_limit;
				Zoom();
			}
			break;
		}
		case B_ENTER: {
			center.x = original.Width()/2;
			center.y = original.Height()/2;
			Zoom();
			break;
		}
		case '*': {
			AdjustZoom(1.0);
			center.x = original.Width()/2;
			center.y = original.Height()/2;
			Zoom();
			break;
		}
		case B_DELETE: {
			zv_window->PostMessage(BLUR_IMAGE_MSG);
			break;
		}
		default:
			break;
	}
}

//==========================
void BitmapView::BlurImage()
{
	BBitmap *new_image = BlurBitmap(image);
	if (new_image)  //check if blur operation successful
	{
		delete image;
		image = new_image;
		Zoom();
	}	
}

