// AtariMemTest.cpp
// Implementation of the AtariMemTest class

// Jon Pimble, fatrat@fat-rat.com
// Last revised: 2002.05.30



#include "AtariMemTest.h"



// Main instantiation function
extern "C" _EXPORT
BScreenSaver *instantiate_screen_saver( BMessage *msg, image_id id )
{
   return new AtariMemTest( msg, id );
} 



// Constructor
AtariMemTest::AtariMemTest( BMessage *archive, image_id image ) 
          : BScreenSaver( archive, image ) 
{ 
  // Character-to-data LUT is defined by #define statements in AtariMemTest.h
  // I.e., ATFONT_CHAR_A, ATFONT_CHAR_E, etc.

  // Character data for Atari font; hardcoded to keep this simple
  //  Characters needed:   A E H I L M O P R S T X Y + 3 funky (16 in all)
  
  atfont_data = new char[ATFONT_NUM_CHARS * ATFONT_NUM_ROWS] = {
    0x00, 0x18, 0x3c, 0x66, 0x66, 0x7e, 0x66, 0x00, // ATFONT_CHAR_A
    0x00, 0x7e, 0x60, 0x7c, 0x60, 0x60, 0x7e, 0x00, // ATFONT_CHAR_E
    0x00, 0x66, 0x66, 0x7e, 0x66, 0x66, 0x66, 0x00, // ATFONT_CHAR_H
    0x00, 0x7e, 0x18, 0x18, 0x18, 0x18, 0x7e, 0x00, // ATFONT_CHAR_I

    0x00, 0x60, 0x60, 0x60, 0x60, 0x60, 0x7e, 0x00, // ATFONT_CHAR_L
    0x00, 0x63, 0x77, 0x7f, 0x6b, 0x63, 0x63, 0x00, // ATFONT_CHAR_M
    0x00, 0x3c, 0x66, 0x66, 0x66, 0x66, 0x3c, 0x00, // ATFONT_CHAR_O
    0x00, 0x7c, 0x66, 0x66, 0x7c, 0x60, 0x60, 0x00, // ATFONT_CHAR_P

    0x00, 0x7c, 0x66, 0x66, 0x7c, 0x6c, 0x66, 0x00, // ATFONT_CHAR_R
    0x00, 0x3c, 0x60, 0x3c, 0x06, 0x06, 0x3c, 0x00, // ATFONT_CHAR_S
    0x00, 0x7e, 0x18, 0x18, 0x18, 0x18, 0x18, 0x00, // ATFONT_CHAR_T
    0x00, 0x66, 0x66, 0x3c, 0x3c, 0x66, 0x66, 0x00, // ATFONT_CHAR_X

    0x00, 0x66, 0x66, 0x3c, 0x18, 0x18, 0x18, 0x00, // ATFONT_CHAR_Y
    
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // ATFONT_CHAR_SPACE
    0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, 0xC0, // ATFONT_CHAR_LT_BAR
    0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03 // ATFONT_CHAR_RT_BAR
  };
  
  
}



// Destructor
AtariMemTest::~AtariMemTest()
{
  // Free allocated memory
  delete atfont_data;
}


       
void AtariMemTest::StartConfig(BView *view) 
{ 
  BFont font;  
  font_height f_height;

  float vert_spacing, y_pos;
  
  // String views for credits in config view  
  BStringView *title_view, *sub_title_view, *version_view, *credits_view;

  // Make our background color match the one used by the screensaver
  //  configuration panel
  view->SetViewColor(ui_color(B_PANEL_BACKGROUND_COLOR));


  // Create the configuration menu
  
  // First get some info about the font we're using
  view->GetFont(&font);
  view->GetFontHeight(&f_height);
  vert_spacing = f_height.ascent + f_height.descent + f_height.leading;
  // Start a couple of lines into the view
  y_pos = vert_spacing + vert_spacing;

  // Title
  title_view=new BStringView(BRect(0, y_pos, view->Bounds().Width(),
   vert_spacing + y_pos), B_EMPTY_STRING, AT_TITLE_STRING);
  title_view->SetAlignment(B_ALIGN_CENTER);  
  view->AddChild(title_view);
  y_pos += vert_spacing;
  
  // Sub-title
  sub_title_view=new BStringView(BRect(0, y_pos, view->Bounds().Width(),
   vert_spacing + y_pos), B_EMPTY_STRING, AT_SUBTITLE_STRING);
  sub_title_view->SetAlignment(B_ALIGN_CENTER);  
  view->AddChild(sub_title_view);
  y_pos += vert_spacing;

  // Some extra spacing between this line and the next
  y_pos += vert_spacing;
  
  // Date / Version
  version_view=new BStringView(BRect(0, y_pos, view->Bounds().Width(),
   vert_spacing + y_pos), B_EMPTY_STRING, AT_VERSION_STRING);
  version_view->SetAlignment(B_ALIGN_CENTER);  
  view->AddChild(version_view);
  y_pos += vert_spacing;
 
  // Credits
  credits_view=new BStringView(BRect(0, y_pos, view->Bounds().Width(),
   vert_spacing + y_pos), B_EMPTY_STRING, AT_CREDITS_STRING);
  credits_view->SetAlignment(B_ALIGN_CENTER);  
  view->AddChild(credits_view);

}



// Setup screensaver parameters before inital drawing
status_t AtariMemTest::StartSaver( BView *view, bool preview ) 
{
  int i;
  
  // Seed for random number generator
  srand(time(NULL));
  
  
  // Time between calls to Draw(), in microseconds (1000000us == 1s)
  SetTickSize(3000000);
  
  
  // Set scaling factor, so that we draw things to the proper scaled sizes
  x_scale = (float) (view->Bounds().Width()/ATARI_WIDTH);
  y_scale = (float) (view->Bounds().Height()/ATARI_HEIGHT);

   
  // Setup variables
  rom_banks_drawn = 0;
  ram_banks_drawn = 0;
  first_loop_flag = false;
  
    
  // Set status for ROM banks
  for(i=0; i < 2; i++)
  {
    if( (rand() % ATARI_FAIL_CHANCE) == 1)
    {
      rom_status[i] = 1;
    }
    else
    {
      rom_status[i] = 0;
    }
  }

    
  // Set status for RAM banks
  for(i = 0; i < 64; i++)
  {
    if( (rand() % ATARI_FAIL_CHANCE) == 1)
    {
      ram_status[i] = 1;
    }
    else
    {
      ram_status[i] = 0;
    }
  }
    

  // Must return B_OK to tell screensaver system that we're ready to draw
  return B_OK; 
}
       
       

// Main drawing function
void AtariMemTest::Draw(BView *view, int32 frame)
{
  int i, status;
  float cursor_x, cursor_y;
  
  // If this is the first frame being drawn, blank the screen with the
  //  proper background color & draw our text
  if(frame == 0)
  {
    // We got this color value through trial and error, it matches the original
    //  Atari background color well enough
    view->SetLowColor(0x00, 0x00, 0x81);
    view->FillRect(view->Bounds(), B_SOLID_LOW);
    
    // Draw the text on the top & bottom of the screen...
    // Text is drawn in white
    view->SetHighColor(0xff, 0xff, 0xff);
   
    // MemTest title at top of screen, "MEMORY TEST", in GR.1 (2x width of
    //  normal text, 20x24 characters; start at line 2)
    cursor_x = (float) 4*(ATARI_WIDTH/20);
    cursor_y = (float) 2*(ATARI_HEIGHT/24);
    
    FontPlotter(view, ATFONT_CHAR_M, false, cursor_x, cursor_y, 2.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/20);
    FontPlotter(view, ATFONT_CHAR_E, false, cursor_x, cursor_y, 2.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/20);
    FontPlotter(view, ATFONT_CHAR_M, false, cursor_x, cursor_y, 2.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/20);
    FontPlotter(view, ATFONT_CHAR_O, false, cursor_x, cursor_y, 2.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/20);
    FontPlotter(view, ATFONT_CHAR_R, false, cursor_x, cursor_y, 2.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/20);
    FontPlotter(view, ATFONT_CHAR_Y, false, cursor_x, cursor_y, 2.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/20);
    
    // Skip a space
    cursor_x += (float) 1*(ATARI_WIDTH/20);
    
    FontPlotter(view, ATFONT_CHAR_T, false, cursor_x, cursor_y, 2.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/20);
    FontPlotter(view, ATFONT_CHAR_E, false, cursor_x, cursor_y, 2.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/20);
    FontPlotter(view, ATFONT_CHAR_S, false, cursor_x, cursor_y, 2.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/20);
    FontPlotter(view, ATFONT_CHAR_T, false, cursor_x, cursor_y, 2.0, 1.0);


    // "ROM"
    cursor_x = (float) 2*(ATARI_WIDTH/20);
    cursor_y += (float) 3*(ATARI_HEIGHT/24);
    FontPlotter(view, ATFONT_CHAR_R, false, cursor_x, cursor_y, 2.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/20);
    FontPlotter(view, ATFONT_CHAR_O, false, cursor_x, cursor_y, 2.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/20);
    FontPlotter(view, ATFONT_CHAR_M, false, cursor_x, cursor_y, 2.0, 1.0);

    
    // "RAM"
    cursor_x = (float) 2*(ATARI_WIDTH/20);
    cursor_y += (float) 5*(ATARI_HEIGHT/24);
    FontPlotter(view, ATFONT_CHAR_R, false, cursor_x, cursor_y, 2.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/20);
    FontPlotter(view, ATFONT_CHAR_A, false, cursor_x, cursor_y, 2.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/20);
    FontPlotter(view, ATFONT_CHAR_M, false, cursor_x, cursor_y, 2.0, 1.0);
    
    
    // Exit instructions at bottom of screen
    // There's some inverse video here; we'll put those in lower case to show
    //  that; also, there's two funky characters on the edge of each run of
    //  inverse video text, but we're not going to show that unless it looks
    //  funny.
    // " reset OR help TO EXIT"
    // Switch back to 'normal' Atari font: 40x24 text characters, no zoom.
    cursor_x = (float) 9*(ATARI_WIDTH/40);
    cursor_y = (float) 22*(ATARI_HEIGHT/24);
    
    FontPlotter(view, ATFONT_CHAR_RT_BAR, false, cursor_x, cursor_y, 1.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/40);
    FontPlotter(view, ATFONT_CHAR_R, true, cursor_x, cursor_y, 1.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/40);
    FontPlotter(view, ATFONT_CHAR_E, true, cursor_x, cursor_y, 1.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/40);
    FontPlotter(view, ATFONT_CHAR_S, true, cursor_x, cursor_y, 1.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/40);
    FontPlotter(view, ATFONT_CHAR_E, true, cursor_x, cursor_y, 1.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/40);
    FontPlotter(view, ATFONT_CHAR_T, true, cursor_x, cursor_y, 1.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/40);


    // Left Bar
    FontPlotter(view, ATFONT_CHAR_LT_BAR, false, cursor_x, cursor_y, 1.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/40);
    
    
    FontPlotter(view, ATFONT_CHAR_O, false, cursor_x, cursor_y, 1.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/40);
    FontPlotter(view, ATFONT_CHAR_R, false, cursor_x, cursor_y, 1.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/40);

    
    // Right bar
    FontPlotter(view, ATFONT_CHAR_RT_BAR, false, cursor_x, cursor_y, 1.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/40);
    

    FontPlotter(view, ATFONT_CHAR_H, true, cursor_x, cursor_y, 1.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/40);
    FontPlotter(view, ATFONT_CHAR_E, true, cursor_x, cursor_y, 1.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/40);
    FontPlotter(view, ATFONT_CHAR_L, true, cursor_x, cursor_y, 1.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/40);
    FontPlotter(view, ATFONT_CHAR_P, true, cursor_x, cursor_y, 1.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/40);

    
    // Left Bar
    FontPlotter(view, ATFONT_CHAR_LT_BAR, false, cursor_x, cursor_y, 1.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/40);
    

    FontPlotter(view, ATFONT_CHAR_T, false, cursor_x, cursor_y, 1.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/40);
    FontPlotter(view, ATFONT_CHAR_O, false, cursor_x, cursor_y, 1.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/40);


    // Just a space
    cursor_x += (float) 1*(ATARI_WIDTH/40);
     

    FontPlotter(view, ATFONT_CHAR_E, false, cursor_x, cursor_y, 1.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/40);
    FontPlotter(view, ATFONT_CHAR_X, false, cursor_x, cursor_y, 1.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/40);
    FontPlotter(view, ATFONT_CHAR_I, false, cursor_x, cursor_y, 1.0, 1.0);
    cursor_x += (float) 1*(ATARI_WIDTH/40);
    FontPlotter(view, ATFONT_CHAR_T, false, cursor_x, cursor_y, 1.0, 1.0);

  }

  
  // Draw the ROM banks
  switch(rom_banks_drawn)
  {
  
    // None drawn yet, draw the first one in white
    case 0:
      view->SetHighColor(0xff, 0xff, 0xff);
      
      cursor_x = (float) 4*(ATARI_WIDTH/40);
      cursor_y = (float) 7*(ATARI_HEIGHT/24);
      
      // ROM thingie is 14 characters long
      for(i=0; i < 15; i++)
      {
        FontPlotter(view, ATFONT_CHAR_SPACE, true, cursor_x, cursor_y, 1.0, 1.0);
        cursor_x += (float) 1*(ATARI_WIDTH/40);
      }
      
      rom_banks_drawn++;
      break;
    
    
    // First one is done, draw it in the proper status color, draw the
    //  second one in white
    case 1:
      // Setcolor based on status: green for 'OK' bank, red for 'bad'
      if(rom_status[0] == 0)
      {
        view->SetHighColor(0x00, 0xff, 0x00);
      }
      else
      {
        view->SetHighColor(0xff, 0x00, 0x00);
      }
      
      cursor_x = (float) 4*(ATARI_WIDTH/40);
      cursor_y = (float) 7*(ATARI_HEIGHT/24);
      
      // ROM thingie is 14 characters long
      for(i=0; i < 15; i++)
      {
        FontPlotter(view, ATFONT_CHAR_SPACE, true, cursor_x, cursor_y, 1.0, 1.0);
        cursor_x += (float) 1*(ATARI_WIDTH/40);
      }


      // Now draw the currently-being-tested bank in white
      view->SetHighColor(0xff, 0xff, 0xff);
      cursor_x += (float) 1*(ATARI_WIDTH/40);
      
      // ROM thingie is 14 characters long
      for(i=0; i < 15; i++)
      {
        FontPlotter(view, ATFONT_CHAR_SPACE, true, cursor_x, cursor_y, 1.0, 1);
        cursor_x += (float) 1*(ATARI_WIDTH/40);
      }
      
      rom_banks_drawn++;
      break;
    
    
    // Second one is done, draw it in the proper status color
    case 2:
      // Setcolor based on status: green for 'OK' bank, red for 'bad'
      if(rom_status[1] == 0)
      {
        view->SetHighColor(0x00, 0xff, 0x00);
      }
      else
      {
        view->SetHighColor(0xff, 0x00, 0x00);
      }
      
      cursor_x = (float) 20*(ATARI_WIDTH/40);
      cursor_y = (float) 7*(ATARI_HEIGHT/24);
      
      // ROM thingie is 14 characters long
      for(i=0; i < 15; i++)
      {
        FontPlotter(view, ATFONT_CHAR_SPACE, true, cursor_x, cursor_y, 1.0, 1.0);
        cursor_x += (float) 1*(ATARI_WIDTH/40);
      }

      rom_banks_drawn++;
      break;
  
  
    // Both ROM banks have been drawn - do nothing
    default:
      break;
  }


  // Draw RAM banks
  if(rom_banks_drawn > 2)
  {
    // Special case for the first one - check a flag to see if we've looped
    //  through yet; if not, we need to draw the last RAM block in its final
    //  state color.
    if(ram_banks_drawn == 0)
    {
      // Don't do this on the first loop
      if(first_loop_flag)
      {
        status = ram_status[63];
      
        // Green, it passes
        if(status == 0)
        {
          view->SetHighColor(0, 255, 0);
        }
        // Red, it failed
        else
        {
          view->SetHighColor(255, 0, 0);
        }
            
        cursor_x = (float) (4 + (2 * (63 & 0x0f) )) * (float) (ATARI_WIDTH/40);
        cursor_y = (float) (12 + (2 * ((63 & 0x30)>>4) )) * (float) (ATARI_HEIGHT/24);
    
        // Draw the block
        FontPlotter(view, ATFONT_CHAR_SPACE, true, cursor_x, cursor_y, 1.0, 1.0);

      }
      
      // Set our flag to true, we've now gone into the loop once; the next
      //  time we get to this case (ram_banks_drawn == 0), we'll need to draw
      //  that last bank (#63)
      first_loop_flag = true;
    }
    
    
    // If we're past the first one, draw the previous one in the proper status
    //  color
    if(ram_banks_drawn > 0)
    {
      status = ram_status[ram_banks_drawn - 1];
      
      // Green, it passes
      if(status == 0)
      {
        view->SetHighColor(0, 255, 0);
      }
      // Red, it failed
      else
      {
        view->SetHighColor(255, 0, 0);
      }
      
      cursor_x = (float) (4 + (2 * ((ram_banks_drawn-1) & 0x0f) )) * (float) (ATARI_WIDTH/40);
      cursor_y = (float) (12 + (2 * (((ram_banks_drawn-1) & 0x30)>>4) )) * (float) (ATARI_HEIGHT/24);
    
      // Draw the block
      FontPlotter(view, ATFONT_CHAR_SPACE, true, cursor_x, cursor_y, 1.0, 1.0);
    }
    
    // Set color to white, to show testing-in-progress
    view->SetHighColor(255, 255, 255);
    
    // Set coordinates for current bank
    cursor_x = (float) (4 + (2 * (ram_banks_drawn & 0x0f) )) * (float) (ATARI_WIDTH/40);
    cursor_y = (float) (12 + (2 * ((ram_banks_drawn & 0x30)>>4) )) * (float) (ATARI_HEIGHT/24);
    
    // Draw the block
    FontPlotter(view, ATFONT_CHAR_SPACE, true, cursor_x, cursor_y, 1.0, 1.0);
        
    
    // Advance counter to the next RAM bank
    ram_banks_drawn++;
  
    // If we've finished the last one, we go back and draw all of the RAM
    //  banks again without erasing anything on the screen; since there are
    //  64 of them in all, we can just use a mask against 0x3f to loop
    ram_banks_drawn = ram_banks_drawn & 0x3f;

  }
  
}



// Draws a 8x8 bitmap character, whose 8-byte font data is at base_ptr,
//  at the location & scale specified, in the current color
void AtariMemTest::FontPlotter(BView *view, char plot_char,
 bool reverse_video, float x_pos, float y_pos, float x_zoom, float y_zoom)
{
  unsigned char column, row;
  char test_bit, test_byte;
  BRect r;
  
  char *base_ptr = atfont_data + (plot_char*8);
  
  for(row=0; row < ATFONT_NUM_ROWS; row++)
  {
    for(column=0; column < 8; column++)
    {
      // Test current bit
      test_bit = 1;
      test_bit = test_bit << column;      
      test_byte = *base_ptr;
            
      // If it's reverse video, invert it
      if(reverse_video)
      {
        test_byte =  0xff - (unsigned char) test_byte;
      }
      
      // Now see if the bit we want is set
      test_byte = (test_byte & test_bit);
      
      // If it's set, draw a block
      if(test_byte != 0)
      {
        // Scaling factor for whole screen is used here, too (x_scale, y_scale)
        r.Set(
          (x_pos + ((7.0 - column) * x_zoom)) * x_scale,
          (y_pos + (row * y_zoom)) * y_scale,
          ((x_pos + ((7.0 - column) * x_zoom)) * x_scale) + (x_zoom * x_scale - 1.0),
          ((y_pos + (row * y_zoom)) * y_scale) + (y_zoom * y_scale - 1.0)
        );
        
        view->FillRect(r, B_SOLID_HIGH);
      }
            
    }
    // Move onto the next row in this font character
    base_ptr++;

  }
}




