/*#############################################
  #  ASIM : Simulateur Apple ][               #
  #                                           #
  #           CPU : Emulation microprocesseur # 
  #############################################*/

#include "asim.h"
#include <OS.h>
#include <KernelKit.h>

extern bool Sniff;
FILE *Fil=NULL;

thread_id ThreadCPU=0;

// CONFIG VARIABLES ---------
int Speed=1000;		// Vitesse CPU (kHz)
int LoopCycles=2000;	// Instructions / bloc
bool CMOS=false;	// 6502=false 65C02=true

// CPU REGISTERS ------------
const uint8 _flagC=1,_flagZ=2,_flagI=4,_flagD=8,_flagB=16,_flagR=32,_flagV=64,_flagN=128;
vuint8  GrA,GrX,GrY;                   // Accu. Index X,Y
vuint8  GrSP;      	                  // Stack
vuint16 GrPC;        	              // Prog. Counter 
vuint8  GrP;             	          // Status register

// CPU task Control ---------
volatile bool StopReq=false;	// Start/stop CPU
volatile bool StopAck=false;	// Handshake
volatile bool StopCPU=false;	// Final END

// Siulation timing ---------
int64 SimCycles=0;	// Cycles simulation
int64 TimCycles=0;	// Cycles temporisation
int	FasCycles=0;	// Cycles boucle interne
vint64 StartTime;	// Emulation start time

//---------------------------
bool UpIt=false;	// Mise à jour registres Généraux

#define Bopr(a) (MEM[a])
#define Bopw(a,b) { MEM[(uint16)a]=b; }

#define BUSR(a) ( ( (((a)&0xFF00)!=0xC000) )?MEM[(a)]:Busr(a))
#define BUSW Busw

//##############################################################################################

int InstrCycles[256]= {
//	0	1	2	3	4	5	6	7		8	9	A	B	C	D	E	F
	 7	,6	,99	,99	,99	,3	,5	,99		,3	,2	,2	,99	,99	,4	,6	,99	// 0
	,2	,5	,99	,99	,99	,4	,6	,99		,2	,4	,99	,99	,99	,4	,7	,99	// 1
	,6	,6	,99	,99	,3	,3	,5	,99		,4	,2	,2	,99	,4	,4	,6	,99	// 2
	,2	,5	,99	,99	,99	,4	,6	,99		,2	,4	,99	,99	,99	,4	,7	,99	// 3
	,6	,6	,99	,99	,99	,3	,5	,99		,3	,2	,2	,99	,3	,4	,6	,99	// 4
	,2	,5	,99	,99	,99	,4	,6	,99		,2	,4	,99	,99	,99	,4	,7	,99	// 5
	,6	,6	,99	,99	,99	,3	,5	,99		,4	,2	,2	,99	,5	,4	,6	,99	// 6
	,2	,5	,99	,99	,99	,4	,6	,99		,2	,4	,99	,99	,99	,4	,7	,99	// 7
	,99	,6	,99	,99	,3	,3	,3	,99		,2	,99	,2	,99	,4	,4	,4	,99	// 8
	,2	,5	,99	,99	,4	,4	,4	,99		,2	,4	,2	,99	,99	,4	,99	,99	// 9
	,2	,6	,2	,99	,3	,3	,3	,99		,2	,2	,2	,99	,4	,4	,4	,99	// A
	,2	,5	,99	,99	,4	,4	,4	,99		,2	,4	,2	,99	,4	,4	,4	,99	// B
	,2	,6	,99	,99	,3	,3	,5	,99		,2	,2	,2	,99	,4	,4	,6	,99	// C
	,2	,5	,99	,99	,99	,4	,6	,99		,2	,4	,99	,99	,99	,4	,7	,99	// D
	,2	,6	,99	,99	,3	,3	,5	,99		,2	,2	,2	,99	,4	,4	,6	,99	// E
	,2	,5	,99	,99	,99	,4	,6	,99		,2	,4	,99	,99	,99	,4	,7	,99	// F
	};

int InstrCumul[256];

const int D_ERR =0;	// Invalide
const int D_IMM =1;	// Immédiat
const int D_ABS =2;	// Absolu
const int D_ABSX=3;	// Absolu X
const int D_ABSY=4;	// Absolu Y
const int D_ZPG =5;	// Zéro page
const int D_ZPGX=6;	// Zéro page X
const int D_ZPGY=7;	// Zéro page Y
const int D_IND =8;	// Indirect
const int D_INDX=9;	// Indirect X
const int D_INDY=10;// Indirect Y
const int D_REL =11;// Relatif
const int D_IMP =12;// Implicite
const int D_ACC =13;// Accumulateur


struct SInstDebug { char *Nom; int Mode; };
SInstDebug Liste[256]={
		{"BRK",D_IMP },{"ORA",D_INDX},{"---",D_ERR },{"---",D_ERR },	{"---",D_ERR },{"ORA",D_ZPG },{"ZPG",D_ZPG },{"---",D_ERR },
/*08*/	{"PHP",D_IMP },{"ORA",D_IMM },{"ASL",D_ACC },{"---",D_ERR },	{"---",D_ERR },{"ORA",D_ABS },{"ASL",D_ABS },{"---",D_ERR },
/*10*/	{"BPL",D_REL },{"ORA",D_INDY},{"---",D_ERR },{"---",D_ERR },	{"---",D_ERR },{"ORA",D_ZPGX},{"ASL",D_ZPGX},{"---",D_ERR },
/*18*/	{"CLC",D_IMP },{"ORA",D_ABSY},{"---",D_ERR },{"---",D_ERR },	{"---",D_ERR },{"ORA",D_ABSX},{"ASL",D_ABSX},{"---",D_ERR },
/*20*/	{"JSR",D_ABS },{"AND",D_INDX},{"---",D_ERR },{"---",D_ERR },	{"BIT",D_ZPG },{"AND",D_ZPG },{"ROL",D_ZPG },{"---",D_ERR },
/*28*/	{"PLP",D_IMP },{"AND",D_IMM },{"ROL",D_ACC },{"---",D_ERR },	{"BIT",D_ABS },{"AND",D_ABS },{"ROL",D_ABS },{"---",D_ERR },
/*30*/	{"BMI",D_REL },{"AND",D_INDX},{"---",D_ERR },{"---",D_ERR },	{"---",D_ERR },{"AND",D_ZPGX},{"ROL",D_ZPGX},{"---",D_ERR },
/*38*/	{"SEC",D_IMP },{"AND",D_ABSY},{"---",D_ERR },{"---",D_ERR },	{"---",D_ERR },{"AND",D_ABSX},{"ROL",D_ABSX},{"---",D_ERR },
/*40*/	{"RTI",D_IMP },{"EOR",D_INDX},{"---",D_ERR },{"---",D_ERR },	{"---",D_ERR },{"EOR",D_ZPG },{"LSR",D_ZPG },{"---",D_ERR },
/*48*/	{"PHA",D_IMP },{"EOR",D_IMM },{"LSR",D_ACC },{"---",D_ERR },	{"JMP",D_ABS },{"EOR",D_ABS },{"LSR",D_ABS },{"---",D_ERR },
/*50*/	{"BVC",D_REL },{"EOR",D_INDY},{"---",D_ERR },{"---",D_ERR },	{"---",D_ERR },{"EOR",D_ZPGX},{"LSR",D_ZPGX},{"---",D_ERR },
/*58*/	{"CLI",D_IMP },{"EOR",D_ABSY},{"---",D_ERR },{"---",D_ERR },	{"---",D_ERR },{"EOR",D_ABSX},{"LSR",D_ABSX},{"---",D_ERR },
/*60*/	{"RTS",D_IMP },{"ADC",D_INDX},{"---",D_ERR },{"---",D_ERR },	{"---",D_ERR },{"ADC",D_ZPG },{"ROR",D_ZPG },{"---",D_ERR },
/*68*/	{"PLA",D_IMP },{"ADC",D_IMM },{"ROR",D_ACC },{"---",D_ERR },	{"JMP",D_IND },{"ADC",D_ABS },{"ROR",D_ABS },{"---",D_ERR },
/*70*/	{"BVS",D_REL },{"ADC",D_INDY},{"---",D_ERR },{"---",D_ERR },	{"---",D_ERR },{"ADC",D_ZPGX},{"ROR",D_ZPGX},{"---",D_ERR },
/*78*/	{"SEI",D_IMP },{"ADC",D_ABSY},{"---",D_ERR },{"---",D_ERR },	{"---",D_ERR },{"ADC",D_ABSX},{"ROR",D_ABSX},{"---",D_ERR },
/*80*/	{"---",D_ERR },{"STA",D_INDX},{"---",D_ERR },{"---",D_ERR },	{"STY",D_ZPG },{"STA",D_ZPG },{"STX",D_ZPG },{"---",D_ERR },
		{"DEY",D_IMP },{"---",D_ERR },{"TXA",D_IMP },{"---",D_ERR },	{"STY",D_ABS },{"STA",D_ABS },{"STX",D_ABS },{"---",D_ERR },
/*90*/	{"BCC",D_REL },{"STA",D_INDY},{"---",D_ERR },{"---",D_ERR },	{"STY",D_ZPGX},{"STA",D_ZPGX},{"STX",D_ZPGY},{"---",D_ERR },
		{"TYA",D_IMP },{"STA",D_ABSY},{"TXS",D_IMP },{"---",D_ERR },	{"---",D_ERR },{"STA",D_ABSX},{"---",D_ERR },{"---",D_ERR },
/*A0*/	{"LDY",D_IMM },{"LDA",D_INDX},{"LDX",D_IMM },{"---",D_ERR },	{"LDY",D_ZPG },{"LDA",D_ZPG },{"LDX",D_ZPG },{"---",D_ERR },
		{"TAY",D_IMP },{"LDA",D_IMM },{"TAX",D_IMP },{"---",D_ERR },	{"LDY",D_ABS },{"LDA",D_ABS },{"LDX",D_ABS },{"---",D_ERR },
/*B0*/	{"BCS",D_REL },{"LDA",D_INDY},{"---",D_ERR },{"---",D_ERR },	{"LDY",D_ZPGX},{"LDA",D_ZPGX},{"LDX",D_ZPGY},{"---",D_ERR },
		{"CLV",D_IMP },{"LDA",D_ABSY},{"TSX",D_IMP },{"---",D_ERR },	{"LDY",D_ABSX},{"LDA",D_ABSX},{"LDX",D_ABSY},{"---",D_ERR },
/*C0*/	{"CPY",D_IMM },{"CMP",D_INDX},{"---",D_ERR },{"---",D_ERR },	{"CPY",D_ZPG },{"CMP",D_ZPG },{"DEC",D_ZPG },{"---",D_ERR },
		{"INY",D_IMP },{"CMP",D_IMM },{"DEX",D_IMP },{"---",D_ERR },	{"CPY",D_ABS },{"CMP",D_ABS },{"DEC",D_ABS },{"---",D_ERR },
/*D0*/	{"BNE",D_REL },{"CMP",D_INDY},{"---",D_ERR },{"---",D_ERR },	{"---",D_ERR },{"CMP",D_ZPGX},{"DEC",D_ZPGX},{"---",D_ERR },
		{"CLD",D_IMP },{"CMP",D_ABSY},{"---",D_ERR },{"---",D_ERR },	{"---",D_ERR },{"CMP",D_ABSX},{"DEC",D_ABSX},{"---",D_ERR },
/*E0*/	{"CPX",D_IMM },{"SBC",D_INDX},{"---",D_ERR },{"---",D_ERR },	{"CPX",D_ZPG },{"SBC",D_ZPG },{"INC",D_ZPG },{"---",D_ERR },
		{"INX",D_IMP },{"SBC",D_IMM },{"NOP",D_IMP },{"---",D_ERR },	{"CPX",D_ABS },{"SBC",D_ABS },{"INC",D_ABS },{"---",D_ERR },
/*F0*/	{"BEQ",D_REL },{"SBC",D_INDY},{"---",D_ERR },{"---",D_ERR },	{"---",D_ERR },{"SBC",D_ZPGX},{"INC",D_ZPGX},{"---",D_ERR },
		{"SED",D_IMP },{"SBC",D_ABSY},{"---",D_ERR },{"---",D_ERR },	{"---",D_ERR },{"SBC",D_ABSX},{"INC",D_ABSX},{"---",D_ERR }};

void Debug_Instr(int PC)
{
 int I,J;
 uint8 A,B,C;
// 1234:12 34 45 INS
 A=Bopr(PC);
 switch (Liste[A].Mode) {
 	case D_ERR :	// Invalide
 		fprintf (Fil,"%4X:%2X       ???\n",PC,(int)A);
 		break;
	case D_IMM :	// Immédiat
		B=Bopr(PC+1);
 		fprintf (Fil,"%4X:%2X %2X    %s #$%2X\n",PC,(int)A,(int)B,Liste[A].Nom,(int)B);
 		break;
	case D_ABS :	// Absolu
		B=Bopr(PC+1); C=Bopr(PC+2);
 		fprintf (Fil,"%4X:%2X %2X %2X %s $%02X%02X\n",PC,(int)A,(int)B,(int)C,Liste[A].Nom,(int)C,(int)B);
 		break;
	case D_ABSX:	// Absolu X
		B=Bopr(PC+1); C=Bopr(PC+2);
 		fprintf (Fil,"%4X:%2X %2X %2X %s $%02X%02X,X\n",PC,(int)A,(int)B,(int)C,Liste[A].Nom,(int)C,(int)B);
 		break;
	case D_ABSY:	// Absolu Y
		B=Bopr(PC+1); C=Bopr(PC+2);
 		fprintf (Fil,"%4X:%2X %2X %2X %s $%02X%02X,Y\n",PC,(int)A,(int)B,(int)C,Liste[A].Nom,(int)C,(int)B);
 		break;
	case D_ZPG :	// Zéro page
		B=Bopr(PC+1);
 		fprintf (Fil,"%4X:%2X %2X    %s $%2X\n",PC,(int)A,(int)B,Liste[A].Nom,(int)B);
 		break;
	case D_ZPGX:	// Zéro page X
		B=Bopr(PC+1);
 		fprintf (Fil,"%4X:%2X %2X    %s $%2X,X\n",PC,(int)A,(int)B,Liste[A].Nom,(int)B);
 		break;
	case D_ZPGY:	// Zéro page Y
		B=Bopr(PC+1);
 		fprintf (Fil,"%4X:%2X %2X    %s $%2X,Y\n",PC,(int)A,(int)B,Liste[A].Nom,(int)B);
 		break;
	case D_IND:		// Indirect
		B=Bopr(PC+1); C=Bopr(PC+2);
 		fprintf (Fil,"%4X:%2X %2X %2X %s ($%02X%02X)\n",PC,(int)A,(int)B,(int)C,Liste[A].Nom,(int)C,(int)B);
 		break;
	case D_INDX:	// Indirect X
		B=Bopr(PC+1);
 		fprintf (Fil,"%4X:%2X %2X    %s ($%2X,X)\n",PC,(int)A,(int)B,Liste[A].Nom,(int)B);
 		break;
	case D_INDY:	// Indirect Y
		B=Bopr(PC+1);
 		fprintf (Fil,"%4X:%2X %2X    %s ($%2X),Y\n",PC,(int)A,(int)B,Liste[A].Nom,(int)B);
 		break;
	case D_REL :	// Relatif
		B=Bopr(PC+1);
 		fprintf (Fil,"%4X:%2X %2X    %s %04X\n",PC,(int)A,(int)B,Liste[A].Nom,(int)((int)((int8)B)+PC+2));
 		break;
	case D_IMP :	// Implicite
 		fprintf (Fil,"%4X:%2X       %s\n",PC,(int)A,Liste[A].Nom);
 		break;
 	case D_ACC :	// Accumulateur
 		fprintf (Fil,"%4X:%2X       %s\n",PC,(int)A,Liste[A].Nom);
 		break;
 	}
}
//##############################################################################################


/*
## NOTE : Not so good ideas :
## - Remplacing #define-s below by modern inline functions 
## - Using bit fields like : union {	uint8  rP;
##										struct Tflag { bool C:1; bool Z:1; bool I:1; ... } flag; };
## Resulting increased executable size and decreased speed.
##*/


/* MODES D'ADRESSAGE */


#define VAL8;	{ W=MEM[rPC++]; }
#define VAL16;	{ W=(uint16)MEM[rPC]|((uint16)((uint16)MEM[rPC+1]<<8)); rPC+=2; }

#define IMM;	{ V=MEM[rPC++]; }

#define ZPGX;	{ W=(uint8)(MEM[rPC++]+rX); }
#define ZPGY;	{ W=(uint8)(MEM[rPC++]+rY); }

#define ABSX;	{ W=((uint16)MEM[rPC]|((uint16)((uint16)MEM[rPC+1]<<8))); W+=(uint16)rX; rPC+=2; }
#define ABSY;	{ W=((uint16)MEM[rPC]|((uint16)((uint16)MEM[rPC+1]<<8))); W+=(uint16)rY; rPC+=2; }

#define IND;	{ V=MEM[rPC++]; W=(uint16)MEM[V]+(uint16)MEM[V+1]*256; }
#define INDX;	{ V=MEM[rPC++]+rX; W=(uint16)MEM[V]+(uint16)MEM[V+1]*256; }
#define INDY;	{ V=MEM[rPC++]; W=(uint16)MEM[V]+(uint16)MEM[V+1]*256; W+=rY; }

#define PUSH(a); { MEM[256+(rSP--)]=(uint8)(a); }
#define PULL(a); { rSP++; (a)=MEM[256+rSP]; }

#define SETZ(a);  { rP&=~_flagZ; if ((a)==0) rP|=_flagZ; }
#define SETZN(a); { rP&=~(_flagZ|_flagN); if (!(a)) rP|=_flagZ; if ((a)&0x80) rP|=_flagN; }

#define Signe(a) ((int16)((int8)a))

/* INSTRUCTIONS */

#define ORA { rA|=V; rP&=~(_flagZ|_flagN); if (rA==0) rP|=_flagZ; if (rA&0x80) rP|=_flagN;}
#define AND { rA&=V; rP&=~(_flagZ|_flagN); if (rA==0) rP|=_flagZ; if (rA&0x80) rP|=_flagN;}
#define EOR { rA^=V; rP&=~(_flagZ|_flagN); if (rA==0) rP|=_flagZ; if (rA&0x80) rP|=_flagN;}

#define ADC { uint16 T; if (rP&_flagD) {										\
    	T=(uint16)((rA>>4)*10+(rA&15))+(uint16)((V>>4)*10+(V&15))+((rP&_flagC)?1:0);		\
		rP&=~(_flagC|_flagV);													\
		if (T>99) rP|=_flagC;													\
		if  (((rA&128)==(V&128))&&((rA&128)!=(T&128))) rP|=_flagV;				\
		rA=(((T/10)%10)<<4)+(T%10);												\
/*		printf ("ADC décimal !\n");												*/\
	  }else{																	\
		T=(uint16)rA+(uint16)V+((rP&_flagC)?1:0);								\
		rP&=~(_flagC|_flagV);													\
		if (T>255) rP|=_flagC;													\
		if (((rA&128)==(V&128))&&((rA&128)!=(T&128))) rP|=_flagV;				\
		rA=T;																	\
 		}																		\
	rP&=~(_flagZ|_flagN); if (rA==0) rP|=_flagZ; if (rA&0x80) rP|=_flagN;		\
	}

#define SBC { uint16 T; if (rP&_flagD) { 										\
/*		printf ("SBC décimal ! rA=%X V=%X C=%i\n",(int)rA,(int)V,(int)(rP&_flagC));		*/\
		T=(uint16)((rA>>4)*10+(rA&15))-(uint16)((V>>4)*10+(V&15))-((rP&_flagC)?0:1);	\
		rP&=~(_flagC|_flagV);													\
		if  (((rA&128)!=(V&128))&&((rA&128)!=(T&128))) rP|=_flagV;				\
		if (T<100) rP|=_flagC; /*else T=100+T;*/								\
		rA=(((T/10)%10)<<4)+(T%10);												\
/*		printf ("SBC décimal ! %X C=%i\n",(int)rA,(int)(rP&_flagC));			*/\
	 }else{																		\
		T=(uint16)rA-(uint16)V-((rP&_flagC)?0:1);								\
		rP&=~(_flagC|_flagV);													\
		if (T<0x8000) rP|=_flagC;												\
		if  (((rA&128)!=(V&128))&&((rA&128)!=(T&128))) rP|=_flagV;				\
		rA=T;																	\
 		}																		\
	rP&=~(_flagZ|_flagN); if (rA==0) rP|=_flagZ; if (rA&0x80) rP|=_flagN;		\
	}

#define CMP(a) { rP&=~(_flagZ|_flagN|_flagC);						\
	if ((a)>=V) rP|=_flagC;											\
	if ((a)==V) rP|=_flagZ; if (((a)-V)&0x80) rP|=_flagN;			\
	}

#define ASL(a) { rP&=~(_flagZ|_flagN|_flagC);						\
	if (((a)&128)!=0) rP|=_flagC;									\
	(a)<<=1; if ((a)==0) rP|=_flagZ; if ((a)&0x80) rP|=_flagN;		\
	}

#define ROL(a) { bool flag=((a)&128); (a)<<=1;						\
	if (rP&_flagC) (a)++;	rP&=~(_flagZ|_flagN|_flagC); 			\
	if (flag) rP|=_flagC;											\
	if (!(a)) rP|=_flagZ; if ((a)&0x80) rP|=_flagN;					\
	}
	
#define ROR(a) { bool flag=(a)&1; (a)>>=1;							\
	if (rP&_flagC) (a)|=128; rP&=~(_flagZ|_flagN|_flagC); 			\
	if (flag) rP|=_flagC;											\
	if (!(a)) rP|=_flagZ; if ((a)&0x80) rP|=_flagN;					\
	}
	
#define LSR(a) {	rP&=~(_flagZ|_flagN|_flagC);					\
	if ((a)&1) rP|=_flagC; (a)>>=1;									\
	if (!(a)) rP|=_flagZ; if ((a)&0x80) rP|=_flagN;					\
	}

// Spécial CMOS
#define TRB {	rP&=~(_flagZ);		\
	if (!(V&rA)) rP|=_flagZ;		\
	V&=~rA;							\
	}

#define TSB {	rP&=~(_flagZ);		\
	if (!(V&rA)) rP|=_flagZ;		\
	V|=rA;							\
	}


// Invalid opcodes
#define CHAOS_1 { }
#define CHAOS_2 { rPC++; }
#define CHAOS_3 { rPC+=2; }

//##############################################################################################

void CPU(void)
{
 
// int I;
 uint8 V;
 uint16 W;
 uint8 IC;

 uint8  rA,rX,rY;            // Accu. Index X,Y
 uint8  rSP;      	                  // Stack
 uint16 rPC;                 // Prog. Counter 
 uint8  rP;                  // Status register

 rA=GrA; rX=GrX; rY=GrY;
 rPC=GrPC; rP=GrP; rSP=GrSP;
 FasCycles=0; 
 do {
/*   if (Fil) {
    	fprintf (Fil,"%02X %02X %02X|SP%02X S%02X| ",(int)rA,(int)rX,(int)rY,(int)rSP,(int)rP);
		Debug_Instr(rPC); //suspend_thread(find_thread(NULL));
		}
*/	IC=Bopr(rPC++);
	W=InstrCycles[IC];
//	InstrCumul[IC]++;
	FasCycles+=W;
	if (W==99) { printf ("---!!! Instruction invalide !!!-%2X --\n",IC); continue; }
    
	switch (IC) {
	case 0x00 : rPC++; PUSH(rPC>>8); PUSH(rPC); rP|=_flagB; PUSH(rP|_flagR);
			rP|=_flagI; W=0xFFFE; rPC=BUSR(W); rPC|=(uint16)BUSR(W+1)*256;	break;	// BRK
    case 0x01 : INDX; V=BUSR(W); ORA; 										break;	// ORA indX 
	case 0x02 : CHAOS_2; break;
	case 0x03 : CHAOS_2; break;
	case 0x04 : if (CMOS) {VAL8; V=Bopr(W); TSB; Bopw(W,V); } else CHAOS_2;	break;  // TSB zpg (CMOS)
	case 0x05 : VAL8; V=Bopr(W); ORA;										break;	// ORA zpg
	case 0x06 : VAL8; V=Bopr(W); ASL(V); Bopw(W,V);							break;	// ASL zpg
	case 0x07 : CHAOS_2; break;
	case 0x08 : PUSH(rP|_flagR|_flagB); /*Flag B present ?*/				break;	// PHP imp 
	case 0x09 : IMM; ORA;													break;	// ORA imm
	case 0x0A : ASL(rA);													break;	// ASL Acc
	case 0x0B : CHAOS_2; break;
	case 0x0C : if (CMOS) { VAL16; V=BUSR(W); TSB; BUSW(W,V); } else CHAOS_3;break;	// TSB abs (CMOS)
	case 0x0D : VAL16; V=BUSR(W); ORA;										break;	// ORA abs
	case 0x0E : VAL16; V=BUSR(W); ASL(V); BUSW(W,V);						break;	// ASL abs
	case 0x0F : CHAOS_3; break;

	case 0x10 : IMM; if (!(rP&_flagN)) rPC+=Signe(V);						break;	// BPL rel
	case 0x11 : INDY; V=BUSR(W); ORA;										break;	// ORA indY
	case 0x12 : if (CMOS) { IND; V=BUSR(W); ORA; BUSW(W,V); } else CHAOS_2;	break;	// ORA ind (CMOS)
	case 0x13 : CHAOS_2; break;
	case 0x14 : if (CMOS) {VAL8; V=Bopr(W); TRB; Bopw(W,V); } else CHAOS_2;	break;  // TRB zpg (CMOS)
	case 0x15 : ZPGX; V=Bopr(W); ORA;										break;	// ORA zpgX
	case 0x16 : ZPGX; V=Bopr(W); ASL(V); Bopw(W,V);							break;	// ASL zpgX
	case 0x17 : CHAOS_2; break;
	case 0x18 : rP&=~_flagC;												break;	// CLC imp
	case 0x19 : ABSY; V=BUSR(W); ORA;										break;	// ORA absY
	case 0x1A : if (CMOS) { rA++; SETZN(rA); } else CHAOS_1;				break;  // INC Acc (CMOS)
	case 0x1B : CHAOS_2; break;
	case 0x1C : if (CMOS) { VAL16; V=BUSR(W); TRB; BUSW(W,V); } else CHAOS_3;break;	// TRB abs (CMOS)
	case 0x1D : ABSX; V=BUSR(W); ORA;										break;	// ORA absX
	case 0x1E : ABSX; V=BUSR(W); ASL(V); BUSW(W,V);							break;	// ASL absX
	case 0x1F : CHAOS_3; break;

	case 0x20 : VAL16; rPC--; PUSH(rPC>>8); PUSH(rPC); rPC=W;				break;	// JSR
	case 0x21 : INDX; V=BUSR(W); AND;										break;	// AND indX
	case 0x22 : CHAOS_2; break;
	case 0x23 : CHAOS_2; break;
	case 0x24 : VAL8; V=Bopr(W); rP&=~(_flagZ|_flagN|_flagV); 
		if (!(rA&V)) rP|=_flagZ; if (V&0x80) rP|=_flagN;
		if (V&0x40) rP|=_flagV;												break;	// BIT zpg
	case 0x25 : VAL8; V=Bopr(W); AND;										break;	// AND zpg
	case 0x26 : VAL8; V=Bopr(W); ROL(V); Bopw(W,V);							break;	// ROL zpg
	case 0x27 : CHAOS_2; break;
	case 0x28 : PULL(rP);													break;	// PLP imp
	case 0x29 : IMM; AND;													break;	// AND imm
	case 0x2A : ROL(rA);													break;	// ROL Acc
	case 0x2B : CHAOS_2; break;
	case 0x2C : VAL16; V=BUSR(W); rP&=~(_flagZ|_flagN|_flagV); 
		if (!(rA&V)) rP|=_flagZ; if (V&0x80) rP|=_flagN;
		if (V&0x40) rP|=_flagV;												break;	// BIT abs
	case 0x2D : VAL16; V=BUSR(W); AND;										break;	// AND abs
	case 0x2E : VAL16; V=BUSR(W); ROL(V); BUSW(W,V);						break;	// ROL abs
	case 0x2F : CHAOS_3; break;

	case 0x30 : IMM; if (rP&_flagN) rPC+=Signe(V);							break;	// BMI rel
	case 0x31 : INDY; V=BUSR(W); AND;										break;	// AND indY
	case 0x32 : if (CMOS) { IND; V=BUSR(W); AND; BUSW(W,V); } else CHAOS_2;	break;	// AND ind (CMOS)
	case 0x33 : CHAOS_2; break;
	case 0x34 : if (CMOS) { ZPGX; V=Bopr(W); rP&=~(_flagZ|_flagN|_flagV); 
		if (!(rA&V)) rP|=_flagZ; if (V&0x80) rP|=_flagN;
		if (V&0x40) rP|=_flagV; } else CHAOS_2;								break;	// BIT zpgX
	case 0x35 : ZPGX; V=Bopr(W); AND;										break;	// AND zpgX
	case 0x36 : ZPGX; V=Bopr(W); ROL(V); Bopw(W,V);							break;	// ROL zpgX
	case 0x37 : CHAOS_2; break;
	case 0x38 : rP|=_flagC;													break;	// SEC imp
	case 0x39 : ABSY; V=BUSR(W); AND;										break;	// AND absY
	case 0x3A : if (CMOS) { rA--; SETZN(rA); } else CHAOS_1;				break;  // DEC Acc (CMOS)
	case 0x3B : CHAOS_2; break;
	case 0x3C :if (CMOS) { ABSX; V=BUSR(W); rP&=~(_flagZ|_flagN|_flagV); 
		if (!(rA&V)) rP|=_flagZ; if (V&0x80) rP|=_flagN;
		if (V&0x40) rP|=_flagV; } else CHAOS_3;								break;	// BIT absX
	case 0x3D : ABSX; V=BUSR(W); AND;										break;	// AND absX
	case 0x3E : ABSX; V=BUSR(W); ROL(V); BUSW(W,V);							break;	// ROL absX
	case 0x3F : CHAOS_3; break;

	case 0x40 : PULL(rP); PULL(rPC); PULL(V); rPC|=(uint16)V*256;			break;	// RTI imp
	case 0x41 : INDX; V=BUSR(W); EOR;										break;	// XOR indX
	case 0x42 : CHAOS_2; break;
	case 0x43 : CHAOS_2; break;
	case 0x44 : CHAOS_2; break;
	case 0x45 : VAL8; V=Bopr(W); EOR;										break;	// EOR zpg
	case 0x46 : VAL8; V=Bopr(W); LSR(V); Bopw(W,V);							break;	// LSR zpg
	case 0x47 : CHAOS_2; break;
	case 0x48 : PUSH(rA);													break;	// PHA imp
	case 0x49 : IMM; EOR;													break;	// EOR imm
	case 0x4A : LSR(rA);													break;	// LSR acc
	case 0x4B : CHAOS_2; break;
	case 0x4C : VAL16; rPC=W;												break;	// JMP abs
	case 0x4D : VAL16; V=BUSR(W); EOR;										break;	// EOR abs
	case 0x4E : VAL16; V=BUSR(W); LSR(V); BUSW(W,V);						break;	// LSR abs
	case 0x4F : CHAOS_3; break;

	case 0x50 : IMM; if (!(rP&_flagV)) rPC+=Signe(V);						break;	// BVC rel
	case 0x51 : INDY; V=BUSR(W); EOR;										break;	// EOR indY
	case 0x52 : if (CMOS) { IND; V=BUSR(W); EOR; BUSW(W,V); } else CHAOS_2;	break;	// EOR ind (CMOS)
	case 0x53 : CHAOS_2; break;
	case 0x54 : CHAOS_2; break;
	case 0x55 : ZPGX; V=Bopr(W); EOR;										break;	// EOR zpgX 
	case 0x56 : ZPGX; V=Bopr(W); LSR(V); Bopw(W,V);							break;	// LSR zpgX
	case 0x57 : CHAOS_2; break;
	case 0x58 : rP&=~_flagI;												break;	// CLI imp
	case 0x59 : ABSY; V=BUSR(W); EOR;										break;	// EOR absY
	case 0x5A : if (CMOS) PUSH(rY); /*else CHAOS_1;*/						break;	// PHY imp
	case 0x5B : CHAOS_2; break;
	case 0x5C : CHAOS_3; break;
	case 0x5D : ABSX; V=BUSR(W); EOR;										break;	// EOR absX
	case 0x5E : ABSX; V=BUSR(W); LSR(V); BUSW(W,V);							break;	// LSR absX
	case 0x5F : CHAOS_3; break;

	case 0x60 : PULL(rPC); PULL(V); rPC|=(uint16)V*256; rPC++;				break;	// RTS imp
	case 0x61 : INDX; V=BUSR(W); ADC;										break;	// ADC indX
	case 0x62 : CHAOS_2; break;
	case 0x63 : CHAOS_2; break;
	case 0x64 : if (CMOS) { VAL8; Bopw(W,0); } else CHAOS_2;				break;	// STZ zpg
	case 0x65 : VAL8; V=Bopr(W); ADC;										break;	// ADC zpg
	case 0x66 : VAL8; V=Bopr(W); ROR(V); Bopw(W,V);							break;	// ROR zpg
	case 0x67 : CHAOS_2; break;
	case 0x68 : PULL(rA); SETZN(rA);										break;	// PLA imp
	case 0x69 : IMM; ADC;													break;	// ADC imm
	case 0x6A : ROR(rA);													break;	// ROR acc
	case 0x6B : CHAOS_2; break;
	case 0x6C : VAL16; rPC=BUSR(W);
		if ((!CMOS)&&((W&255)==255)) { printf ("Erreur JMP indirect !!!!\n"); 
							rPC|=(uint16)BUSR(W+1-256)*256; } else
		rPC|=(uint16)BUSR(W+1)*256;											break;	// JMP ind
	case 0x6D : VAL16; V=BUSR(W); ADC;										break;	// ADC abs
	case 0x6E : VAL16; V=BUSR(W); ROR(V); BUSW(W,V);						break;	// ROR abs
	case 0x6F : CHAOS_3; break;

	case 0x70 : IMM; if (rP&_flagV) rPC+=Signe(V);							break;	// BVS rel
	case 0x71 : INDY; V=BUSR(W); ADC;										break;	// ADC indY
	case 0x72 : if (CMOS) { IND; V=BUSR(W); ADC; BUSW(W,V); } else CHAOS_2;	break;	// ADC ind (CMOS)
	case 0x73 : CHAOS_2; break;
	case 0x74 : if (CMOS) { ZPGX; Bopw(W,0); } else CHAOS_2;				break;	// STZ zpgX (CMOS)
	case 0x75 : ZPGX; V=Bopr(W); ADC;										break;	// ADC zpgX
	case 0x76 : ZPGX; V=Bopr(W); ROR(V); Bopw(W,V);							break;	// ROR zpgX
	case 0x77 : CHAOS_2; break;
	case 0x78 : rP|=_flagI;													break;	// SEI imp
	case 0x79 : ABSY; V=BUSR(W); ADC;										break;	// ADC absY
	case 0x7A : if (CMOS) { PULL(rY); SETZN(rY); } else CHAOS_1;			break;	// PLY imp (CMOS)
	case 0x7B : CHAOS_2; break;
	case 0x7C : if (CMOS) { VAL16; rPC=BUSR(W);
		rPC|=(uint16)BUSR(W+1)*256; rPC+=rX;	} else CHAOS_3;				break;	// JMP indX (CMOS)
	case 0x7D : ABSX; V=BUSR(W); ADC;										break;	// ADC absX
	case 0x7E : ABSX; V=BUSR(W); ROR(V); BUSW(W,V);							break;	// ROR absX
	case 0x7F : CHAOS_3; break;

	case 0x80 : if (CMOS) { IMM; rPC+=Signe(V); } else CHAOS_2;				break;	// BRA rel (CMOS)
	case 0x81 : INDX; BUSW(W,rA);											break;	// STA indX
	case 0x82 : CHAOS_2; break;
	case 0x83 : CHAOS_2; break;
	case 0x84 : VAL8; Bopw(W,rY);											break;	// STY zpg
	case 0x85 : VAL8; Bopw(W,rA);											break;	// STA zpg
	case 0x86 : VAL8; Bopw(W,rX);											break;	// STX zpg
	case 0x87 : CHAOS_2; break;
	case 0x88 : rY--; SETZN(rY);											break;	// DEY imp
	case 0x89 : if (CMOS) {IMM; rP&=~(_flagZ|_flagN|_flagV); 
		if (!(rA&V)) rP|=_flagZ; if (V&0x80) rP|=_flagN;
		if (V&0x40) rP|=_flagV; } else CHAOS_2;								break;	// BIT imm
	case 0x8A : rA=rX; SETZN(rA);											break;	// TXA imp
	case 0x8B : CHAOS_2; break;
	case 0x8C : VAL16; BUSW(W,rY);											break;	// STY abs
	case 0x8D : VAL16; BUSW(W,rA);											break;	// STA abs
	case 0x8E : VAL16; BUSW(W,rX);											break;	// STX abs
	case 0x8F : CHAOS_3; break;
	
	case 0x90 : IMM; if (!(rP&_flagC)) rPC+=Signe(V);						break;	// BCC rel
	case 0x91 : INDY; BUSW(W,rA);											break;	// STA indY
	case 0x92 : if (CMOS) { IND; BUSW(W,rA); } else CHAOS_2;				break;	// STA ind (CMOS)
	case 0x93 : CHAOS_2; break;
	case 0x94 : ZPGX; Bopw(W,rY);											break;	// STY zpgX 
	case 0x95 : ZPGX; Bopw(W,rA);											break;	// STA zpgX 
	case 0x96 : ZPGY; Bopw(W,rX);											break;	// STX zpgY 
	case 0x97 : CHAOS_2; break;
	case 0x98 : rA=rY; SETZN(rA);											break;	// TYA imp
	case 0x99 : ABSY; BUSW(W,rA);											break;	// STA absY
	case 0x9A : rSP=rX;														break;	// TXS imp
	case 0x9B : CHAOS_2; break;
	case 0x9C : if (CMOS) { VAL16; BUSW(W,0); } else CHAOS_3;				break;	// STZ abs (CMOS)
	case 0x9D : ABSX; BUSW(W,rA);											break;	// STA absX
	case 0x9E : if (CMOS) { ABSX; BUSW(W,0); } else CHAOS_3;					break;	// STZ absX (CMOS)
	case 0x9F : CHAOS_3; break;
	
	case 0xA0 : IMM; rY=V; SETZN(rY);										break;	// LDY imm
	case 0xA1 : INDX; rA=BUSR(W); SETZN(rA);									break;	// LDA indX 
	case 0xA2 : IMM; rX=V; SETZN(rX);										break;	// LDX imm
	case 0xA3 : CHAOS_2; break;
	case 0xA4 : VAL8; rY=Bopr(W); SETZN(rY);								break;	// LDY zpg
	case 0xA5 : VAL8; rA=Bopr(W); SETZN(rA);								break;	// LDA zpg
	case 0xA6 : VAL8; rX=Bopr(W); SETZN(rX);								break;	// LDX zpg
	case 0xA7 : CHAOS_2; break;
	case 0xA8 : rY=rA; SETZN(rY);											break;	// TAY imp
	case 0xA9 : IMM; rA=V; SETZN(rA);										break;	// LDA imm
	case 0xAA : rX=rA; SETZN(rX);											break;	// TAX imp
	case 0xAB : CHAOS_2; break;
	case 0xAC : VAL16; rY=BUSR(W); SETZN(rY);								break;	// LDY abs
	case 0xAD : VAL16; rA=BUSR(W); SETZN(rA);								break;	// LDA abs
	case 0xAE : VAL16; rX=BUSR(W); SETZN(rX);								break;	// LDX abs
	case 0xAF : CHAOS_3; break;
	  
	case 0xB0 : IMM; if (rP&_flagC) rPC+=Signe(V);							break;	// BCS rel
	case 0xB1 : INDY; rA=BUSR(W); SETZN(rA);								break;	// LDA indY
	case 0xB2 : if (CMOS) { IND; rA=BUSR(W); SETZN(rA); } else CHAOS_2;		break;	// LDA ind (CMOS)
	case 0xB3 : CHAOS_2; break;
	case 0xB4 : ZPGX; rY=Bopr(W); SETZN(rY);								break;	// LDY zpgX 
	case 0xB5 : ZPGX; rA=Bopr(W); SETZN(rA);								break;	// LDA zpgX 
	case 0xB6 : ZPGY; rX=Bopr(W); SETZN(rX);								break;	// LDX zpgY 
	case 0xB7 : CHAOS_2; break;
	case 0xB8 : rP&=~_flagV;												break;	// CLV imp
	case 0xB9 : ABSY; rA=BUSR(W); SETZN(rA);								break;	// LDA absY
	case 0xBA : rX=rSP; SETZN(rX);											break;	// TSX imp
	case 0xBB : CHAOS_2; break;
	case 0xBC : ABSX; rY=BUSR(W); SETZN(rY);								break;	// LDY absX
	case 0xBD : ABSX; rA=BUSR(W); SETZN(rA);								break;	// LDA absX
	case 0xBE : ABSY; rX=BUSR(W); SETZN(rX);								break;	// LDX absY
	case 0xBF : CHAOS_3; break;
	
	case 0xC0 : IMM; CMP(rY);												break;	// CPY imm
	case 0xC1 : INDX; V=BUSR(W); CMP(rA);									break;	// CMP indX
	case 0xC2 : CHAOS_2; break;
	case 0xC3 : CHAOS_2; break;
	case 0xC4 : VAL8; V=Bopr(W); CMP(rY);									break;	// CPY zpg
	case 0xC5 : VAL8; V=Bopr(W); CMP(rA);									break;	// CMP zpg
	case 0xC6 : VAL8; V=Bopr(W)-1; Bopw(W,V); SETZN(V);						break;	// DEC zpg
	case 0xC7 : CHAOS_2; break;
	case 0xC8 : rY++; SETZN(rY);											break;	// INY imp
	case 0xC9 : IMM; CMP(rA);												break;	// CMP imm
	case 0xCA : rX--; SETZN(rX);											break;	// DEX imp
	case 0xCB : CHAOS_2; break;
	case 0xCC : VAL16; V=BUSR(W); CMP(rY);									break;	// CPY abs
	case 0xCD : VAL16; V=BUSR(W); CMP(rA);									break;	// CMP abs
	case 0xCE : VAL16; V=BUSR(W)-1; BUSW(W,V); SETZN(V);					break;	// DEC abs
	case 0xCF : CHAOS_3; break;
	
	case 0xD0 : IMM; if (!(rP&_flagZ)) rPC+=Signe(V);						break;	// BNE rel
	case 0xD1 : INDY; V=BUSR(W); CMP(rA);									break;	// CMP indY 
	case 0xD2 : if (CMOS) {IND;V=BUSR(W); CMP(rA); BUSW(W,V); } else CHAOS_2; break;	// CMP ind (CMOS)
	case 0xD3 : CHAOS_2; break;
	case 0xD4 : CHAOS_2; break;
	case 0xD5 : ZPGX; V=Bopr(W); CMP(rA);									break;	// CMP zpgX
	case 0xD6 : ZPGX; V=Bopr(W)-1; Bopw(W,V); SETZN(V);						break;	// DEC zpgX
	case 0xD7 : CHAOS_2; break;
	case 0xD8 : rP&=~_flagD;												break;	// CLD imp
	case 0xD9 : ABSY; V=BUSR(W); CMP(rA);									break;	// CMP absY
	case 0xDA : if (CMOS) PUSH(rX); /*else CHAOS_1;*/						break;	// PHX imp
	case 0xDB : CHAOS_2; break;
	case 0xDC : CHAOS_3; break;
	case 0xDD : ABSX; V=BUSR(W); CMP(rA);									break;	// CMP absX
	case 0xDE : ABSX; V=BUSR(W)-1; BUSW(W,V); SETZN(V);						break;	// DEC absX
	case 0xDF : CHAOS_3; break;

	case 0xE0 : IMM; CMP(rX);												break;	// CPX imm
	case 0xE1 : INDX; V=BUSR(W); SBC;										break;	// SBC indX
	case 0xE2 : CHAOS_2; break;
	case 0xE3 : CHAOS_2; break;
	case 0xE4 : VAL8; V=Bopr(W); CMP(rX);									break;	// CPX zpg
	case 0xE5 : VAL8; V=Bopr(W); SBC;										break;	// SBC zpg
	case 0xE6 : VAL8; V=Bopr(W)+1; Bopw(W,V); SETZN(V); 					break;	// INC zpg
	case 0xE7 : CHAOS_2; break;
	case 0xE8 : rX++; SETZN(rX);											break;	// INX imp
	case 0xE9 : IMM; SBC(V);												break;	// SBC imm
	case 0xEA : /* NOP: The only universal assembly language instruction */	break;	// NOP imp
	case 0xEB : CHAOS_2; break;
	case 0xEC : VAL16; V=BUSR(W); CMP(rX);									break;	// CPX abs
	case 0xED : VAL16; V=BUSR(W); SBC;										break;	// SBC abs
	case 0xEE : VAL16; V=BUSR(W)+1; BUSW(W,V);  SETZN(V);					break;	// INC abs
	case 0xEF : CHAOS_3; break;

	case 0xF0 : IMM; if (rP&_flagZ) rPC+=Signe(V);							break;	// BEQ rel
	case 0xF1 : INDY; V=BUSR(W); SBC;										break;	// SBC indY
	case 0xF2 : if (CMOS) { IND; V=BUSR(W); SBC; BUSW(W,V); } else CHAOS_2;	break;	// SBC ind (CMOS)
	case 0xF3 : CHAOS_2; break;
	case 0xF4 : CHAOS_2; break;
	case 0xF5 : ZPGX; V=Bopr(W); SBC;										break;	// SBC zpgX
	case 0xF6 : ZPGX; V=Bopr(W)+1; Bopw(W,V);  SETZN(V);					break;	// INC zpgX
	case 0xF7 : CHAOS_2; break;
	case 0xF8 : rP|=_flagD;													break;	// SED imp
	case 0xF9 : ABSY; V=BUSR(W); SBC;										break;	// SBC absY
	case 0xFA : if (CMOS) { PULL(rX); SETZN(rX); } else CHAOS_1;			break;	// PLX imp (CMOS)
	case 0xFB : CHAOS_2; break;
	case 0xFC : CHAOS_3; break;
	case 0xFD : ABSX; V=BUSR(W); SBC;										break;	// SBC absX
	case 0xFE : ABSX; V=BUSR(W)+1; BUSW(W,V);  SETZN(V);					break;	// INC absX
	case 0xFF : CHAOS_3; break;

	} // switch
 	} while (FasCycles<LoopCycles);
 GrA=rA;
 GrX=rX;
 GrY=rY;
 GrPC=rPC;
 GrP=rP;
 GrSP=rSP;
 
 SimCycles+=FasCycles;
 TimCycles+=FasCycles; FasCycles=0;
}

//---------------------------------------------------------------
int32 CPUTask(void *) // Tâche d'exécution du processeur !!!!!
{
// int I,J;
 do {
 	int64 SpkCycles=TimCycles;
	CPU();
	SpeakLoop(TimCycles-SpkCycles);
	if (StopReq) {
		StopAck=true;
		do { snooze(1000); } while (StopReq);
		StopAck=false;
		}
 	while ((real_time_clock_usecs()/1000LL-StartTime)<(TimCycles/Speed)) snooze(10000);
	} while (!StopCPU);
 return 0;
}

//###############################################################

//###############################################################

//###############################################################

int SwitchCPU(bool State)
{
 if (State) { // Start & restart
	if (ThreadCPU) {
		StopReq=false;
		StopAck=false;
		StartTime=real_time_clock_usecs()/1000LL-TimCycles/Speed;
		resume_thread(ThreadCPU);
		} else return 1;
  }else{
	if (ThreadCPU) {
		resume_thread(ThreadCPU);
		StopAck=false;
		StopReq=true;
		do { snooze(500); } while (!StopAck);
		suspend_thread(ThreadCPU);
		} else return 1;
	}
 return 0;
}

//---------------------------------------------------------------
void PrintStat()
{
 int I;
 for (I=0;I<256;I++) if (InstrCycles[I]!=99)
 	{
 	printf(">%2X =%i\n",(int)I,(int)InstrCumul[I]);
 	}
 for (I=0;I<256;I++) InstrCumul[I]=0;	
}

//---------------------------------------------------------------
void InterCPU(int Type)
{
 SwitchCPU(false); // Pause
 switch(Type) {
 	case INTER_RESET:
 		GrA=GrX=GrY=0;
 		GrP=0x20;
 		GrPC=BUSR(0xFFFC)+BUSR(0xFFFD)*256;
 		//printf (" Reset Vector=%X\n",(int)GrPC);
 		GrSP=0xFF; //==0x1FF
  		break;
	case INTER_IRQ:
		break;
	case INTER_NMI:
		break;
	}
 SwitchCPU(true); // Restart
}

//---------------------------------------------------------------
void InitCPU()
{
 SimCycles=0;
 TimCycles=0;
 StartTime=real_time_clock_usecs()/1000LL;
 InterCPU(INTER_RESET);
}



