/*#############################################
  #  ASIM : Simulateur Apple ][               #
  #                                           #
  #                     Disk : Gestion disque # 
  #############################################
*/
#include <AppKit.h>
#include <StorageKit.h>
//#include <stdio.h>

#include "asim.h"
#include "asimbe.h"

// CONFIG FLAGS ---------
bool TurboDisk=true;
bool CycleDisk=false;
int SectorSkew=SKEW_DOS;

//-----------------------
SDrive Drive1,Drive2;
SDrive *pdDrive;
bool SelDrive=false;

uint8 *DiskImage1, *DiskImage2;

//############################################################## 

//17 8 13
const int GAP1=12;
const int GAP2=8;
const int GAP3=10;
const int TRACKTOTAL=GAP1+16*(14+GAP2+349+GAP3); //=6161

// TURBODISK ------------
const int64 DELAY_PHASEON =15; //OFF->ON=15
const int64 DELAY_PHASEOFF= 15000; // ON->OFF=15934
const int64 DELAY_MOTORON =100;
const int64 DELAY_MOTOROFF=10;
const int64 DELAY_READ=50;

//############################################################## 
//-------------- Interface Disk ][

void MoveHead(int Phase)
{
 if ((pdDrive->HeadPhase==((Phase-1)&3))&&(pdDrive->HeadTrack<pdDrive->MaxTrack*2-2))
	pdDrive->HeadTrack++;
 if ((pdDrive->HeadPhase==((Phase+1)&3))&&(pdDrive->HeadTrack>0))
	pdDrive->HeadTrack--;
 //printf ("Movetrack %i\n",pdDrive->HeadTrack);
 pdDrive->UpTrack=true;

}

void IODisk(int A)
{
 switch(A) {
	case 0:  // Phase 0 OFF
		pdDrive->PH0=false;
		if (TurboDisk) TimCycles-=DELAY_PHASEOFF;
		break;
	case 1:  // Phase 0 ON
		MoveHead(0);
		pdDrive->HeadPhase=0;
		pdDrive->PH0=true;
		if (TurboDisk) TimCycles-=DELAY_PHASEON;
		break;
	case 2:  // Phase 1 OFF
		pdDrive->PH1=false;
		if (TurboDisk) TimCycles-=DELAY_PHASEOFF;
		break;
	case 3:  // Phase 1 ON
		MoveHead(1);
		pdDrive->HeadPhase=1;
		pdDrive->PH1=true;
		if (TurboDisk) TimCycles-=DELAY_PHASEON;
		break;
	case 4:  // Phase 2 OFF
		pdDrive->PH2=false;
		if (TurboDisk) TimCycles-=DELAY_PHASEOFF;
		break;
	case 5:  // Phase 2 ON
		MoveHead(2);
		pdDrive->HeadPhase=2;
		pdDrive->PH2=true;
		if (TurboDisk) TimCycles-=DELAY_PHASEON;
		break;
	case 6:  // Phase 3 OFF
		pdDrive->PH3=false;
		if (TurboDisk) TimCycles-=DELAY_PHASEOFF;
		break;
	case 7:  // Phase 3 ON
		MoveHead(3);
		pdDrive->HeadPhase=3;
		pdDrive->PH3=true;
		if (TurboDisk) TimCycles-=DELAY_PHASEON;
		break;
	case 8:  // Motor OFF
		pdDrive->MotorON=false;
		if (TurboDisk) TimCycles-=DELAY_MOTOROFF;
		pdDrive->UpMotor=true;
		break;
	case 9:  // Motor ON
		pdDrive->MotorON=true;
		if (TurboDisk) TimCycles-=DELAY_MOTORON;
		pdDrive->UpMotor=true;
		break;
	case 10: // Engage drive 1
		pdDrive=&Drive1;
		SelDrive=false;
		break;
	case 11: // Engage drive 2
		pdDrive=&Drive2;
		SelDrive=true;
		break;
 	}
 if ((TurboDisk)&&(TimCycles<0)) TimCycles=0;
}
//---------------------------------------------
void  IOWrDisk(uint8 A,uint8 V)
{
 A&=15;
 
 //printf ("DISK Write %X Time=%Li\n",A,(int64)(SimCycles+FasCycles));

 if (A<12) { IODisk(A); return; }
 switch(A) {
 	case 0x0C: // Strobe data latch for I/O
 		printf ("disk write \n");
 		if ((pdDrive->LatchInOut)&&(!pdDrive->WriteProtect)) { // write data
			if (!pdDrive->Image) return ;
 			printf ("Ecriture disque %X\n",(int)pdDrive->Latch);
 			pdDrive->Image[pdDrive->HeadTrack/2*pdDrive->TrackLen+pdDrive->Turn]=pdDrive->Latch;
 			pdDrive->Turn++; if (pdDrive->Turn>=pdDrive->TrackLen) pdDrive->Turn-=pdDrive->TrackLen;
 			if (TurboDisk) { TimCycles-=DELAY_READ; if (TimCycles<0) TimCycles=0; }
 			}
		pdDrive->UpWrite=true;
		pdDrive->UpdateFile=true;
 		break;
 	case 0x0D: // Load data latch
 		if (pdDrive->LatchInOut) pdDrive->Latch=V;
 		break;
 	case 0x0E: // Prepare latch for input
 		pdDrive->LatchInOut=false;
 		break;
 	case 0x0F: // Prepare latch for output
 		pdDrive->LatchInOut=true;
 		break;
	}
}

//---------------------------------------------
uint8 IORdDisk(uint8 A)
{
 uint8 V=0;
 A&=15;
 //printf ("DISK Read %X Time=%Li\n",A,(int64)(SimCycles+FasCycles));
 if (A<12) { IODisk(A); return 0; }
 switch(A) {
 	case 0x0C: // Strobe data latch for I/O
 		if (TurboDisk) { TimCycles-=DELAY_READ; if (TimCycles<0) TimCycles=0; }
 		if (!pdDrive->Image) return 0;	
 		if (!pdDrive->LatchInOut) { // read data
			V=pdDrive->Image[pdDrive->HeadTrack/2*pdDrive->TrackLen+pdDrive->Turn];
			pdDrive->Turn++; if (pdDrive->Turn>=pdDrive->TrackLen) pdDrive->Turn-=pdDrive->TrackLen;
			pdDrive->UpRead=true;
			return V; 	
 		  }else{ // write data
			if (pdDrive->WriteProtect) return 0;
 		//	printf ("Ecriture disque %X\n",(int)pdDrive->Latch);
 			pdDrive->Image[pdDrive->HeadTrack/2*pdDrive->TrackLen+pdDrive->Turn]=pdDrive->Latch;
 			pdDrive->Turn++; if (pdDrive->Turn>=pdDrive->TrackLen) pdDrive->Turn-=pdDrive->TrackLen;
			pdDrive->UpWrite=true;
			pdDrive->UpdateFile=true;
 			}	
 		break;
 	case 0x0D: // Load data latch
 		if (pdDrive->LatchInOut) pdDrive->Latch=V;
 		break;
 	case 0x0E: // Prepare latch for input
 		pdDrive->LatchInOut=false;
 		return pdDrive->WriteProtect?0x80:0; 
 	case 0x0F: // Prepare latch for output
 		pdDrive->LatchInOut=true;
 		break;
	}
 return 0;
}

//############################################################## 
//-------------- Fichiers images
// 4ns/bit , 0.2sec/rotation -> 50000 bits/track=6250bytes/track
// GAP1 | Adrs 0 | GAP2 | Data | GAP3 
//      | Adrs 1 | GAP2 | Data | GAP3 ...
//   ...| Adrs 15| GAP2 | Data | GAP3, final GAP3 overlaps initial GAP1 

uint8 Nibble62[64] = {
	0x96,0x97,0x9A,0x9B,0x9D,0x9E,0x9F,0xA6,0xA7,0xAB,0xAC,0xAD,0xAE,0xAF,0xB2,0xB3,
	0xB4,0xB5,0xB6,0xB7,0xB9,0xBA,0xBB,0xBC,0xBD,0xBE,0xBF,0xCB,0xCD,0xCE,0xCF,0xD3,
	0xD6,0xD7,0xD9,0xDA,0xDB,0xDC,0xDD,0xDE,0xDF,0xE5,0xE6,0xE7,0xE9,0xEA,0xEB,0xEC,
	0xED,0xEE,0xEF,0xF2,0xF3,0xF4,0xF5,0xF6,0xF7,0xF9,0xFA,0xFB,0xFC,0xFD,0xFE,0xFF};

uint8 DeNibble62[128];

/* DO logical order  0 1 2 3 4 5 6 7 8 9 A B C D E F */
/*    physical order 0 D B 9 7 5 3 1 E C A 8 6 4 2 F */

/* PO logical order  0 E D C B A 9 8 7 6 5 4 3 2 1 F */
/*    physical order 0 2 4 6 8 A C E 1 3 5 7 9 B D F */


// DOS ORDER
// Physical :	0 1 2 3 4 5 6 7 8 9 A B C D E F
// Logical  :	0 7 E 6 D 5 C 4 B 3 A 2 9 1 8 F

uint8 SkewArray[3][16]={
	{0x0,0x7,0xE,0x6,0xD,0x5,0xC,0x4,0xB,0x3,0xA,0x2,0x9,0x1,0x8,0xF},
	{0x0,0x8,0x1,0x9,0x2,0xA,0x3,0xB,0x4,0xC,0x5,0xD,0x6,0xE,0x7,0xF},
	{0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0}};

inline uint HiByte44( uint V) { return ((V&0xAA)>>1)|0xAA; }
inline uint LoByte44( uint V) { return (V&0x55)|0xAA; }
inline uint Pack44(uint U,uint V) { return ((U&0x55)<<1)|(V&0x55); }

//############################################################## 
// Format DSK : 256*16*35=143360

int SDrive::ReadDSKImage(BFile *Fil)
{
 int I,J,K;
 int T;
 off_t Size;
 uint8 Tdo[16][256];
 uint8 *Track;
 
 Fil->GetSize(&Size);
 if (Size%(256*16)!=0) return 1;
 if (Size/16/256<35) return 1;

 TrackLen=TRACKTOTAL;
 MaxTrack=Size/16/256;
 ImageType=DSK_IMAGE;
 Turn=0;
 Skew=SectorSkew;

 if (Image) delete(Image);
 Image=new uint8[MaxTrack*TRACKTOTAL];
 if (!Image) { printf ("Erreur alloc \n"); return 1; }

 printf ("Fichier DSK reconnu\n");

 for (T=0;T<MaxTrack;T++) {
	K=0;
	Track=&Image[T*TRACKTOTAL];
	//printf ("Track %i\n",(int)T);
	Fil->Read(Tdo,256*16);
	// ------ GAP 1	
	for (J=0;J<GAP1;J++) Track[K++]=0xFF;
	for (I=0;I<16;I++) {
		// ------ ADRESS FIELD
		Track[K++]=0xD5; // prologue
		Track[K++]=0xAA;
		Track[K++]=0x96;
		Track[K++]=HiByte44(0xFE);
		Track[K++]=LoByte44(0xFE);
		Track[K++]=HiByte44(T);
		Track[K++]=LoByte44(T);
		Track[K++]=HiByte44(I);
		Track[K++]=LoByte44(I);
		Track[K++]=HiByte44(0xFE^T^I);
		Track[K++]=LoByte44(0xFE^T^I); // checksum
		Track[K++]=0xDE; // epilogue
		Track[K++]=0xAA;
		Track[K++]=0xEB;
		
		// ------ GAP 2
		for (J=0;J<GAP2;J++) Track[K++]=0xFF;	
		// ------ DATA SECTOR
		Track[K++]=0xD5; // prologue
		Track[K++]=0xAA;
		Track[K++]=0xAD;
		{
		uint8 Nib[343],*Ptr;
		uint8 *Sec=Tdo[SkewArray[Skew][I]];
		uint8 A,B;
		Ptr=Nib;
		for(J=0;J<0x56;J++) {
			A=       ((Sec[(J+0xAC)&255]&1)*2)|((Sec[(J+0xAC)&255]&2)/2);
			A=(A<<2)|((Sec[(J+0x56)&255]&1)*2)|((Sec[(J+0x56)&255]&2)/2);
			A=(A<<2)|((Sec[(J     )&255]&1)*2)|((Sec[(J     )&255]&2)/2);
			(*Ptr++)=A;
			}
		Ptr[-2]&=0x0F; // 3F
		Ptr[-1]&=0x0F; // 3F
		for (J=0;J<256;J++) (*Ptr++)=Sec[J]/4;
		for (B=0,J=0;J<342;J++) {
			A=Nib[J];
			Track[K++]=Nibble62[B^A];
			B=A;
			}
		Track[K++]=Nibble62[B];
		}
		Track[K++]=0xDE; // epilogue
		Track[K++]=0xAA;
		Track[K++]=0xEB;
		// ------ GAP 3
		for (J=0;J<GAP3;J++) Track[K++]=0xFF; 		
		}
	}
 return 0;
}
//----------
int SDrive::WriteDSKImage(BFile *Fil)
{
 int I,J,K;
 int T,S;
 uint8 Tdo[16][256];
 uint8 *Track;

 for (T=0;T<MaxTrack;T++) {
 	Track=&Image[T*TrackLen];
 	printf ("Track %i\n",(int)T);
	S=0; // Okaou
	for (I=0;I<TrackLen;I++) {
		J=I+1; if (J>=TrackLen) J-=TrackLen;
		if (Track[I]==255) continue; // Gap ...
		if ((Track[I]==0xD5)&&(Track[J]==0xAA)) {
			J=I+2; if (J>=TrackLen) J-=TrackLen;
			if (Track[J]==0x96) { // ADRESS FIELD
				J=I+5; if (J>=TrackLen) J-=TrackLen;
				K=I+6; if (K>=TrackLen) K-=TrackLen;
				S=Pack44(Track[J],Track[K]); // Piste !!!!	
				if (T!=S) printf ("Sauvegarde DSK :Erreur numéro piste :%i!=%i !!!\n",S,T);
				J=I+7; if (J>=TrackLen) J-=TrackLen;
				K=I+8; if (K>=TrackLen) K-=TrackLen;
				S=Pack44(Track[J],Track[K]); // Secteur !!!!
				if (S>15) printf ("Sauvegarde DSK :Erreur numéro de secteur délirant :%i\n",S);
				S&=15;
				I+=14;
				//printf ("T=%i S=%i\n",(int)T,(int)S);
				} else
			if (Track[J]==0xAD) { // DATA FIELD
				//printf ("T=%i S=%i DATA field\n",(int)T,(int)S);
				
				uint8 Nib[343];
				uint8 *Sec=Tdo[SkewArray[Skew][S]];
				uint8 A,B;
				B=0;
				for (J=0,K=I+3;J<343;J++,K++) {
					if (K>=TrackLen) K-=TrackLen;
					A=B; B=DeNibble62[Track[K]-128];
					if (B==255) printf ("ERREUR DENIBBELISATION\n");
					Nib[J]=B=A^B;
					}
				if (B) printf ("Erreur checksum T=%i S=%i\n",T,S);
				for (J=0;J<0x56;J++) {
					Sec[J     ]=((Nib[J]&1)*2 )|((Nib[J]&2 )/2 );
					Sec[J+0x56]=((Nib[J]&4)/2 )|((Nib[J]&8 )/8 );
					if (J<0x54) Sec[J+0xAC]=((Nib[J]&16)/8)|((Nib[J]&32)/32);
					}
				for (J=0;J<256;J++) Sec[J]|=Nib[0x56+J]*4;
				I+=349;
				}
			continue;
			}
		if ((Track[I]==0xDE)&&(Track[J]==0xAA)) {
			J=I+2; if (J>=TrackLen) J-=TrackLen;
			if (Track[J]!=0xEB) printf ("Sauvegarde DSK:Erreur epilogue\n");
			I+=3;
			continue;
			}
		//printf ("Sauvegarde DSK: Erreur de lecture\n");
		}			
	Fil->Write(Tdo,256*16); 
	}
 ImageType=0;
 if (Image) delete(Image); Image=NULL;
 MaxTrack=0;
 return 0;
}
//----------
int DetectDSKImage(BFile *Fil)
{
 off_t Size;
 Fil->GetSize(&Size);
 if ((Size%(256*16))!=0) return 1;
 if ((Size/16/256)<35) return 1;
 return 0;
}

//############################################################## 
// Format NIB : 6656*35=232960

int SDrive::ReadNIBImage(BFile *Fil)
{
 off_t Size;
 int T;

 Fil->GetSize(&Size);
 if (Size!=232960) return 1;
 if (Image) delete(Image);

 MaxTrack=Size/6656;
 TrackLen=6656;
 ImageType=NIB_IMAGE;
 Turn=0;
 Skew=SKEW_NIBBLE;
 
 Image=new uint8[MaxTrack*6656]; 
 if (!Image) { printf ("Erreur alloc \n"); return 1; }
 printf ("Fichier NIB reconnu\n");
 
 for (T=0;T<MaxTrack;T++) {
	Fil->Read(&Image[T*6656],6656); 
	}
 return 0;
}
//----------
int SDrive::WriteNIBImage(BFile *Fil)
{
 off_t Size;
 int T;
 for (T=0;T<MaxTrack;T++) {
	Fil->Write(&Image[T*6656],6656); 
	}
 ImageType=0;
 if (Image) delete Image; Image=NULL;
 MaxTrack=0;
 return 0;
}
//----------
int DetectNIBImage(BFile *Fil)
{
 off_t Size;
 Fil->GetSize(&Size);
 if (Size!=232960) return 1;
 return 0;
}
//############################################################## 

void SDrive::Init(int I)	// Construx
{
 No=I;
 PH0=PH1=PH2=PH3=false;
 HeadTrack=0;
 HeadPhase=0;
 MotorON=false;
 WriteProtect=true;
 LatchInOut=false;
 Latch=255;
 Turn=0;
 ImageType=0;
 MaxTrack=0;
 TrackLen=0;
 Image=NULL;
 NbImages=0;
 PathList=NULL;
 NoImage=0;
 UpdateFile=false;
 UpMotor=UpTrack=UpRead=UpWrite=false;
}
//############################################################## 

int SDrive::ReadImage(BPath *Path)
{
 BFile Fil(Path->Path(),B_READ_ONLY);
 Fil.Seek(0,SEEK_SET);
 if (ReadDSKImage(&Fil)) 
 if (ReadNIBImage(&Fil)) { return 1; }
 UpdateFile=false;
 WriteProtect=false; // Achtung !
 return 0;
}
//---------------------------------
int SDrive::CloseImage(BPath *Path)
{
 if (!Path) return 1;
 if (!Image) return 1;
 if (!UpdateFile) {
	ImageType=0;
	if (Image) delete Image; Image=NULL;
	MaxTrack=0;
 	return 0;
 	}
 
 BFile Fil(Path->Path(),B_WRITE_ONLY);
// Fil.Seek(0,SEEK_SET);

 switch(ImageType) {
 	case DSK_IMAGE: WriteDSKImage(&Fil); break;
 	case NIB_IMAGE:	WriteNIBImage(&Fil); break;
 	}
 return 0;
}
//---------------------------------
int DetectImage(entry_ref *Ref)
{
 int RV;
 BFile Fil(Ref,B_READ_ONLY);
 RV=DetectDSKImage(&Fil); if (!RV) return DSK_IMAGE;
 RV=DetectNIBImage(&Fil); if (!RV) return NIB_IMAGE;
 return 0;
}

//##############################################################

int SDrive::FreePlaylist()
{
 int I;
 if (PathList) CloseImage(PathList[NoImage]);
 for (I=0;I<NbImages;I++) delete PathList[I];
 if (PathList) delete PathList;
 PathList=NULL;
 return 0;
}
//---------------------------------
int SDrive::LoadPlaylist(BMessage *msg)
{
 entry_ref ER;
 BEntry Entry;
 BPath *bpPath;
 int I,J;
 uint8 Sum[100];
 I=J=0;
 while (!msg->FindRef("refs",I,&ER)) {
	if (Sum[I]=DetectImage(&ER)) J++;
	I++;
	if (I>99) break;
	}
 NbImages=J;
 PathList=new BPath*[NbImages]; //entry_ref*[NbImages];
 I=J=0;
 while (!msg->FindRef("refs",I,&ER)) {
	//printf ("%i:%s\n",(int)I,(char *)ER.name);
	if (Sum[I]) {
		Entry.SetTo(&ER);
		bpPath=new BPath();
		Entry.GetPath(bpPath);
		PathList[J]=bpPath; J++; }
	I++;
	}

 ReadImage(PathList[0]);
 NoImage=0; 
 UpdateFile=false;
}
//##############################################################
//##############################################################

//##############################################################
//##############################################################

//##############################################################
//##############################################################

/*++++++++++++++++++++++++++++++++++++++++++
  +  LedView                               +
  ++++++++++++++++++++++++++++++++++++++++++*/
LedView::LedView(int I,BRect frame,const char *name) : BView(frame,name,B_FOLLOW_NONE,B_WILL_DRAW)
{
 Num=I; State=0;
}

void LedView::Draw(BRect brUp)
{
 winAsim->Lock();
 switch (State) {
	case 0: SetHighColor(0,0,0);	break;
	case 1: SetHighColor(255,0,0);	break;
	case 2: SetHighColor(0,255,0);	break;
	}
 FillEllipse(BRect(0,0,4,4));
 winAsim->Unlock();
}
/*++++++++++++++++++++++++++++++++++++++++++
  +  TrackView                             +
  ++++++++++++++++++++++++++++++++++++++++++*/
TrackView::TrackView(int I,BRect frame,const char *name) : BView(frame,name,B_FOLLOW_NONE,B_WILL_DRAW)
{
 Num=I; Track=0;
}
//------------------------------------------
void TrackView::TrackMax(int M) { ResizeTo(M,3); }
//------------------------------------------
void TrackView::Draw(BRect brUp)
{
 winAsim->Lock();
 SetHighColor(0,0,255);
 StrokeLine(BPoint(Track/2,0),BPoint(Track/2,3));
 winAsim->Unlock();
}

/*++++++++++++++++++++++++++++++++++++++++++
  +  DiskView                              +
  ++++++++++++++++++++++++++++++++++++++++++*/
DiskView::DiskView(int I,BRect frame,const char *name) : BView(frame,name,B_FOLLOW_ALL,0/*B_WILL_DRAW*/)
{
 char Tex[40];
 Num=I;
 BRect r=Bounds();
 r.bottom=29;
 sprintf (Tex,"Disk %i",I);
 bbFile=new BButton(r,Tex,Tex,new BMessage((Num==2)?msgDisk2:msgDisk1));
 AddChild(bbFile); bbFile->AttachedToWindow();

 lvLed=new LedView(I,BRect(7,18,11,22),"Led");
 lvLed->State=0;
 bbFile->AddChild(lvLed);
 tvTrack=new TrackView(I,BRect(7,7,8,10),"Track");
 tvTrack->Track=0;

 bbFile->AddChild(tvTrack);

 meMenu=NULL;
 mfField=NULL;
}
//------------------------------------------
void DiskView::Draw(BRect brUp){ }
//------------------------------------------
void DiskView::SetEnabled(bool enabled)
{
 bbFile->SetEnabled(enabled);
}
//------------------------------------------
void DiskView::SetTrack(int Track)
{
 tvTrack->Track=Track;
 tvTrack->Invalidate();
}
//------------------------------------------
void DiskView::SetLed(int Light)
{
 lvLed->State=Light;
 lvLed->Invalidate();
}
//------------------------------------------
void DiskView::SetupAll()
{
 SDrive *sdDrive=((Num==1)?&Drive1:&Drive2);
 int I;

 if (mfField) { RemoveChild(mfField);
				delete mfField; 
 				}
 
 if (sdDrive->NbImages) {
 	BMenuItem *miBlop;
	meMenu=new BMenu("---");
	for (I=0;I<sdDrive->NbImages;I++) {
	 	BMessage *bmMsg=new BMessage(msgFileBar);
		bmMsg->AddInt16("noimage",I);
  		meMenu->AddItem(miBlop=new BMenuItem(sdDrive->PathList[I]->Leaf(),bmMsg));
 		if (I==sdDrive->NoImage) miBlop->SetMarked(true);
 		}
	meMenu->SetRadioMode(true);
	mfField= new BMenuField(BRect(1,29,79,47),":","<>",meMenu); 
	mfField->SetDivider(0);
	mfField->ResizeToPreferred();
	AddChild(mfField);
	meMenu->SetTargetForItems(this);
	meMenu->Superitem()->SetLabel(sdDrive->PathList[sdDrive->NoImage]->Leaf());
	}
 tvTrack->TrackMax(sdDrive->MaxTrack);
}
//------------------------------------------
void DiskView::MessageReceived(BMessage *msg)
{
 switch(msg->what) {
	case msgFileBar:
		SDrive *sdDrive=((Num==1)?&Drive1:&Drive2);
		int16 I=msg->FindInt16("noimage");
		if (I!=sdDrive->NoImage) {
			if (ThreadCPU) suspend_thread(ThreadCPU);
			winAsim->Lock();
			meMenu->ItemAt(I)->SetMarked(true);
			if (!sdDrive->PathList) return;
			if (sdDrive->PathList[sdDrive->NoImage]) sdDrive->CloseImage(sdDrive->PathList[sdDrive->NoImage]);
			sdDrive->ReadImage(sdDrive->PathList[I]);
			sdDrive->NoImage=I;
			meMenu->Superitem()->SetLabel(sdDrive->PathList[I]->Leaf());
			winAsim->Unlock();
		
			if (ThreadCPU) resume_thread(ThreadCPU);
			}
		break;
	}
}
//##############################################################
//##############################################################
void InitDisk()
{
 SetIOHooks(6,IOWrDisk,IORdDisk);
 int I;
 for (I=0;I<128;I++) DeNibble62[I]=255; // Init dénibbelisation
 for (I=0;I<64;I++) DeNibble62[Nibble62[I]-128]=I;

 pdDrive=&Drive1;
 SelDrive=false;
 Drive1.Init(1);
 Drive2.Init(2);
}
//---------------------------------
void ClearDisk()
{
 Drive1.FreePlaylist();
 Drive2.FreePlaylist();
}
