/**
 * Mupen64 - pure_interp.c
 * Copyright (C) 2002 Hacktarux
 *
 * Mupen64 homepage: http://mupen64.emulation64.com
 * email address: hacktarux@yahoo.fr
 * 
 * If you want to contribute to the project please contact
 * me first (maybe someone is already making what you are
 * planning to do).
 *
 *
 * This program is free software; you can redistribute it and/
 * or modify it under the terms of the GNU General Public Li-
 * cence as published by the Free Software Foundation; either
 * version 2 of the Licence, or any later version.
 *
 * This program is distributed in the hope that it will be use-
 * ful, but WITHOUT ANY WARRANTY; without even the implied war-
 * ranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public Licence for more details.
 *
 * You should have received a copy of the GNU General Public
 * Licence along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *
**/

#include <stdio.h>
#include <stdlib.h>
#include "r4300.h"
#include "../memory/memory.h"
#include "macros.h"
#include "interupt.h"

unsigned long interp_addr;
unsigned long op;
static long skip;

static void prefetch();

static void (*interp_ops[64])(void);

static void NI()
{
   printf("NI:%x\n", (unsigned int)op);
   stop=1;
}

static void SLL()
{
   rrd = (unsigned long)(rrt) << rsa;
   sign_extended(rrd);
   interp_addr+=4;
}

static void SRL()
{
   rrd = (unsigned long)rrt >> rsa;
   sign_extended(rrd);
   interp_addr+=4;
}

static void SRA()
{
   rrd = (signed long)rrt >> rsa;
   sign_extended(rrd);
   interp_addr+=4;
}

static void SLLV()
{
   rrd = (unsigned long)(rrt) << (rrs&0x1F);
   sign_extended(rrd);
   interp_addr+=4;
}

static void SRLV()
{
   rrd = (unsigned long)rrt >> (rrs & 0x1F);
   sign_extended(rrd);
   interp_addr+=4;
}

static void JR()
{
   local_rs = irs;
   interp_addr+=4;
   delay_slot=1;
   update_system();
   prefetch();
   interp_ops[((op >> 26) & 0x3F)]();
   delay_slot=0;
   if (!skip_jump)
     interp_addr = local_rs;
   else skip_jump = 0;
}

static void MFHI()
{
   rrd = hi;
   interp_addr+=4;
}

static void MTHI()
{
   hi = rrs;
   interp_addr+=4;
}

static void MFLO()
{
   rrd = lo;
   interp_addr+=4;
}

static void MTLO()
{
   lo = rrs;
   interp_addr+=4;
}

static void MULT()
{
   unsigned long long int temp;
   temp = rrs * rrt;
   hi = temp >> 32;
   sign_extended(hi);
   lo = temp & 0xFFFFFFFF;
   sign_extended(lo);
   interp_addr+=4;
}

static void MULTU()
{
   unsigned long long int temp;
   temp = (rrs & 0xFFFFFFFF) * (rrt & 0xFFFFFFFF);
   hi = temp >> 32;
   sign_extended(hi);
   lo = temp & 0xFFFFFFFF;
   sign_extended(lo);
   interp_addr+=4;
}

static void DIV()
{
   if (rrt)
     {
	lo = rrs / rrt;
	hi = rrs % rrt;
     }
   else printf("div\n");
   interp_addr+=4;
}

static void DMULTU()
{
   unsigned long long int op1, op2, op3, op4;
   unsigned long long int result1, result2, result3, result4;
   unsigned long long int temp1, temp2, temp3, temp4;
   
   op1 = rrs & 0xFFFFFFFF;
   op2 = (rrs >> 32) & 0xFFFFFFFF;
   op3 = rrt & 0xFFFFFFFF;
   op4 = (rrt >> 32) & 0xFFFFFFFF;
   
   temp1 = op1 * op3;
   temp2 = (temp1 >> 32) + op1 * op4;
   temp3 = op2 * op3;
   temp4 = (temp1 >> 32) + op2 * op4;
   
   result1 = temp1 & 0xFFFFFFFF;
   result2 = (temp2 & 0xFFFFFFFF) + (temp3 & 0xFFFFFFFF);
   result3 = (result2 >> 32) + (temp2 >> 32) + (temp4 & 0xFFFFFFFF);
   result4 = (result3 >> 32) + (temp4 >> 32);
   result2 &= 0xFFFFFFFF;
   result3 &= 0xFFFFFFFF;
   
   lo = result1 | (result2 << 32);
   hi = result3 | (result4 << 32);
   interp_addr+=4;
}

static void DDIVU()
{
   if (rrt)
     {
	lo = (unsigned long long int)rrs / (unsigned long long int)rrt;
	hi = (unsigned long long int)rrs % (unsigned long long int)rrt;
     }
   else printf("ddivu\n");
   interp_addr+=4;
}

static void ADD()
{
   rrd = rrs + rrt;
   sign_extended(rrd);
   interp_addr+=4;
}

static void ADDU()
{
   rrd = rrs + rrt;
   sign_extended(rrd);
   interp_addr+=4;
}

static void SUBU()
{
   rrd = rrs - rrt;
   sign_extended(rrd);
   interp_addr+=4;
}

static void AND()
{
   rrd = rrs & rrt;
   interp_addr+=4;
}

static void OR()
{
   rrd = rrs | rrt;
   interp_addr+=4;
}

static void XOR()
{
   rrd = rrs ^ rrt;
   interp_addr+=4;
}

static void SLT()
{
   if (rrs < rrt) rrd = 1;
   else rrd = 0;
   interp_addr+=4;
}

static void SLTU()
{
   if ((rrs&0xFFFFFFFFFFFFFFFLL) < (rrt&0xFFFFFFFFFFFFFFFLL))
     rrd = 1;
   else rrd = 0;
   interp_addr+=4;
}

static void DSLL32()
{
   rrd = rrt << (32+rsa);
   interp_addr+=4;
}

static void DSRA32()
{
   rrd = (signed long long int)rrt >> (32+rsa);
   interp_addr+=4;
}

static void (*interp_special[64])(void) =
{
   SLL , NI   , SRL , SRA , SLLV  , NI    , SRLV, NI    ,
   JR  , NI   , NI  , NI  , NI    , NI    , NI  , NI    ,
   MFHI, MTHI , MFLO, MTLO, NI    , NI    , NI  , NI    ,
   MULT, MULTU, DIV , NI  , NI    , DMULTU, NI  , DDIVU ,
   ADD , ADDU , NI  , SUBU, AND   , OR    , XOR , NI    ,
   NI  , NI   , SLT , SLTU, NI    , NI    , NI  , NI    ,
   NI  , NI   , NI  , NI  , NI    , NI    , NI  , NI    ,
   NI  , NI   , NI  , NI  , DSLL32, NI    , NI  , DSRA32
};

static void BLTZ()
{
   short local_immediate = iimmediate;
   local_rs = irs;
   if ((interp_addr + (local_immediate+1)*4) == interp_addr)
     if (sp_register.halt)
       if (local_rs < 0)
     {
	interp_addr+=4;
	prefetch();
	interp_addr-=4;
	if (op == 0)
	  {
	     skip = next_interupt - Count;
	     if (skip >= 3) 
	       {
		  Count += (skip & 0xFFFFFFFC);
		  return;
	       }
	  }
	prefetch();
     }
   interp_addr+=4;
   delay_slot=1;
   update_system();
   prefetch();
   interp_ops[((op >> 26) & 0x3F)]();
   delay_slot=0;
   if (local_rs < 0)
     if (!skip_jump)
       interp_addr += (local_immediate-1)*4;
   if (skip_jump) skip_jump=0;
}

static void BGEZ()
{
   short local_immediate = iimmediate;
   local_rs = irs;
   if ((interp_addr + (local_immediate+1)*4) == interp_addr)
     if (sp_register.halt)
       if (local_rs >= 0)
     {
	interp_addr+=4;
	prefetch();
	interp_addr-=4;
	if (op == 0)
	  {
	     skip = next_interupt - Count;
	     if (skip >= 3) 
	       {
		  Count += (skip & 0xFFFFFFFC);
		  return;
	       }
	  }
	prefetch();
     }
   interp_addr+=4;
   delay_slot=1;
   update_system();
   prefetch();
   interp_ops[((op >> 26) & 0x3F)]();
   delay_slot=0;
   if (local_rs >= 0)
     if (!skip_jump)
       interp_addr += (local_immediate-1)*4;
   if (skip_jump) skip_jump=0;
}

static void BGEZL()
{
   short local_immediate = iimmediate;
   if ((interp_addr + (local_immediate+1)*4) == interp_addr)
     if (sp_register.halt)
       if (irs >= 0)
     {
	interp_addr+=4;
	prefetch();
	interp_addr-=4;
	if (op == 0)
	  {
	     skip = next_interupt - Count;
	     if (skip >= 3) 
	       {
		  Count += (skip & 0xFFFFFFFC);
		  return;
	       }
	  }
	prefetch();
     }
   if (irs >= 0)
     {
	interp_addr+=4;
	delay_slot=1;
	update_system();
	prefetch();
	interp_ops[((op >> 26) & 0x3F)]();
	delay_slot=0;
	if (!skip_jump)
	  interp_addr += (local_immediate-1)*4;
	else skip_jump = 0;
     }
   else interp_addr+=8;
}

static void BGEZAL()
{
   short local_immediate = iimmediate;
   local_rs = irs;
   reg[31]=interp_addr+8;
   if((&irs)!=(reg+31))
     {
	if ((interp_addr + (local_immediate+1)*4) == interp_addr)
	  if (sp_register.halt)
	    if (local_rs >= 0)
	  {
	     interp_addr+=4;
	     prefetch();
	     interp_addr-=4;
	     if (op == 0)
	       {
		  skip = next_interupt - Count;
		  if (skip >= 3) 
		    {
		       Count += (skip & 0xFFFFFFFC);
		       return;
		    }
	       }
	     prefetch();
	  }
	interp_addr+=4;
	delay_slot=1;
	update_system();
	prefetch();
	interp_ops[((op >> 26) & 0x3F)]();
	delay_slot=0;
	if(local_rs >= 0)
	  {
	     if (!skip_jump)
	       interp_addr += (local_immediate-1)*4;
	     else skip_jump = 0;
	  }
     }
   else printf("erreur dans bgezal\n");
}

static void (*interp_regimm[32])(void) =
{
   BLTZ, BGEZ  , NI, BGEZL, NI, NI, NI, NI,
   NI  , NI    , NI, NI   , NI, NI, NI, NI,
   NI  , BGEZAL, NI, NI   , NI, NI, NI, NI,
   NI  , NI    , NI, NI   , NI, NI, NI, NI
};

static void TLBR()
{
   int index;
   index = Index & 0x1F;
   PageMask = tlb_e[index].mask << 13;
   EntryHi = ((tlb_e[index].vpn2 << 13) | (tlb_e[index].g << 12)
     | tlb_e[index].asid);
   EntryLo1 = (tlb_e[index].pfn_even << 6) | (tlb_e[index].c_even << 3)
     | (tlb_e[index].d_even << 2) | (tlb_e[index].v_even << 1)
       | tlb_e[index].g;
   EntryLo0 = (tlb_e[index].pfn_odd << 6) | (tlb_e[index].c_odd << 3)
     | (tlb_e[index].d_odd << 2) | (tlb_e[index].v_odd << 1)
       | tlb_e[index].g;
   interp_addr+=4;
}

static void TLBWI()
{
   tlb_e[Index&0x3F].g = (EntryLo0 & EntryLo1 & 1);
   tlb_e[Index&0x3F].pfn_even = (EntryLo0 & 0x3FFFFFC0) >> 6;
   tlb_e[Index&0x3F].pfn_odd = (EntryLo1 & 0x3FFFFFC0) >> 6;
   tlb_e[Index&0x3F].c_even = (EntryLo0 & 0x38) >> 3;
   tlb_e[Index&0x3F].c_odd = (EntryLo1 & 0x38) >> 3;
   tlb_e[Index&0x3F].d_even = (EntryLo0 & 0x4) >> 2;
   tlb_e[Index&0x3F].d_odd = (EntryLo1 & 0x4) >> 2;
   tlb_e[Index&0x3F].v_even = (EntryLo0 & 0x2) >> 1;
   tlb_e[Index&0x3F].v_odd = (EntryLo1 & 0x2) >> 1;
   tlb_e[Index&0x3F].asid = (EntryHi & 0xFF);
   tlb_e[Index&0x3F].vpn2 = (EntryHi & 0xFFFFE000) >> 13;
   tlb_e[Index&0x3F].r = (EntryHi & 0xC000000000000000LL) >> 62;
   tlb_e[Index&0x3F].mask = (PageMask & 0x1FFE000) >> 13;
   switch(tlb_e[Index&0x3F].mask)
     {
      case 0x000:
	tlb_e[Index&0x3F].check_parity_mask = 0x00001000;
	break;
      case 0x003:
	tlb_e[Index&0x3F].check_parity_mask = 0x00004000;
	break;
      case 0x00F:
	tlb_e[Index&0x3F].check_parity_mask = 0x00010000;
	break;
      case 0x03F:
	tlb_e[Index&0x3F].check_parity_mask = 0x00040000;
	break;
      case 0x0FF:
	tlb_e[Index&0x3F].check_parity_mask = 0x00100000;
	break;
      case 0x3FF:
	tlb_e[Index&0x3F].check_parity_mask = 0x00400000;
	break;
      case 0xFFF:
	tlb_e[Index&0x3F].check_parity_mask = 0x01000000;
	break;
     }
   interp_addr+=4;
}

static void TLBP()
{
   int i;
   Index = 0x80000000;
   for (i=0; i<32; i++)
     {
	if ((tlb_e[i].vpn2 == ((EntryHi & 0xFFFFE000) >> 13)*2) &&
	    ((tlb_e[i].g) ||
	     (tlb_e[i].asid == (EntryHi & 0xFF))))
	  {
	     Index = i;
	     break;
	  }
     }
   interp_addr+=4;
}

static void ERET()
{
   if (Status & 0x4)
     {
	printf("erreur dans ERET\n");
	stop=1;
     }
   else
     {
	unsigned long local_addr = EPC;
	interp_addr = EPC;
	Status &= 0xFFFFFFFD;
	update_system();
	prefetch();
	interp_ops[((op >> 26) & 0x3F)]();
	if (interp_addr == (local_addr+8)) Count--;
     }
   llbit = 0;
}

static void (*interp_tlb[64])(void) =
{
   NI  , TLBR, TLBWI, NI, NI, NI, NI, NI,
   TLBP, NI  , NI   , NI, NI, NI, NI, NI,
   NI  , NI  , NI   , NI, NI, NI, NI, NI,
   ERET, NI  , NI   , NI, NI, NI, NI, NI,
   NI  , NI  , NI   , NI, NI, NI, NI, NI,
   NI  , NI  , NI   , NI, NI, NI, NI, NI,
   NI  , NI  , NI   , NI, NI, NI, NI, NI,
   NI  , NI  , NI   , NI, NI, NI, NI, NI
};

static void MFC0()
{
   switch(PC->f.r.nrd)
     {
      case 1:
	printf("lecture de Random\n");
	stop=1;
      default:
	rrt = rrd & 0xFFFFFFFF;
	sign_extended(rrt);
     }
   interp_addr+=4;
}

static void MTC0()
{
   switch(PC->f.r.nrd)
     {
      case 0:    // Index
	Index = rrt & 0x000000008000003F;
	if ((Index & 0x3F) > 31) 
	  {
	     printf ("il y a plus de 32 TLB\n");
	     stop=1;
	  }
	break;
      case 1:    // Random
	break;
      case 2:    // EntryLo0
	EntryLo0 = rrt & 0x3FFFFFFF;
	break;
      case 3:    // EntryLo1
 	EntryLo1 = rrt & 0x3FFFFFFF;
	break;
      case 4:    // Context
	Context = rrt & 0xFFFFFFFFFF800000LL;
	break;
      case 5:    // PageMask
	PageMask = rrt & 0x01FFE000;
	break;
      case 6:    // Wired
	printf("criture dans Wired\n");
	stop=1;
	Wired = rrt;
	Random = 31;
	break;
      case 8:    // BadVAddr
	break;
      case 9:    // Count
	if (Compare > Count) printf("criture dans Count non vrifie\n");
	debug_count += Count;
	Count = rrt & 0xFFFFFFFF;
	debug_count -= Count;
	break;
      case 10:   // EntryHi
	EntryHi = rrt & 0xC00000FFFFFFE0FFLL;
	break;
      case 11:   // Compare
	if (rrt > Count)
	  {
	     if (rrt < next_interupt)
	       {
		  interupt_type=1;
		  save_next_interupt = next_interupt;
		  next_interupt = (unsigned long)Compare-1;
	       }
	  }
	Compare = rrt;
	Cause = Cause & 0xFFFFFFFFFFFF7FFFLL; //Timer interupt is clear
	break;
      case 12:   // Status
	if((rrt & 0x04000000) != (Status & 0x04000000))
	  {
	     if (rrt & 0x04000000)
	       {
		  int i;
		  for (i=0; i<32; i++)
		    {
		       if (reg_cop1_fgr_32[i])
			 {
			    printf("erreur de conversion 32 vers 64 bits\n");
			    stop=1;
			 }
		    }
	       }
	     else
	       {
		  int i;
		  for (i=0; i<32; i++)
		    {
		       if (reg_cop1_fgr_64[i])
			 {
			    printf("erreur de conversion 64 vers 32 bits\n");
			    stop=1;
			 }
		    }
	       }
	  }
	Status = rrt;
	if (rrt & 0xFF01)
	  {
	     interp_addr+=4;
	     interupt();
	     interp_addr-=4;
	  }
	break;
      case 13:   // Cause
	if (rrt!=0)
	  {
	     printf("criture dans Cause\n");
	     stop = 1;
	  }
	else Cause = rrt;
	break;
      case 14:   // EPC
	EPC = rrt;
	break;
      case 15:  // PRevID
	break;
      case 0x1B: // CacheErr
	break;
      case 0x1C: // TagLo
	TagLo = rrt & 0x0FFFFFC0;
	break;
      case 0x1D: // TagHi
	TagHi =0;
	break;
      default:
	printf("ecriture dans cop0 inconnue  PC=%x !!!\n", (unsigned int)(PC->addr));
	stop=1;
     }
   interp_addr+=4;
}

static void TLB()
{
   interp_tlb[(op & 0x3F)]();
}

static void (*interp_cop0[32])(void) =
{
   MFC0, NI, NI, NI, MTC0, NI, NI, NI,
   NI  , NI, NI, NI, NI  , NI, NI, NI,
   TLB , NI, NI, NI, NI  , NI, NI, NI,
   NI  , NI, NI, NI, NI  , NI, NI, NI
};

static void ADD_S()
{
   *reg_cop1_simple[cffd] = *reg_cop1_simple[cffs] +
     *reg_cop1_simple[cfft];
   interp_addr+=4;
}

static void MUL_S()
{
   *reg_cop1_simple[cffd] = *reg_cop1_simple[cffs] *
     *reg_cop1_simple[cfft];
   interp_addr+=4;
}

static void DIV_S()
{
   if (!*reg_cop1_simple[cfft])
     printf("div_s\n");
   *reg_cop1_simple[cffd] = *reg_cop1_simple[cffs] /
     *reg_cop1_simple[cfft];
   interp_addr+=4;
}

static void CVT_D_S()
{
   double d;
   d = *reg_cop1_simple[cffs];
   *reg_cop1_double[cffd] = d;
   interp_addr+=4;
}

static void (*interp_cop1_s[64])(void) =
{
   ADD_S, NI     , MUL_S, DIV_S, NI, NI, NI, NI,
   NI   , NI     , NI   , NI   , NI, NI, NI, NI,
   NI   , NI     , NI   , NI   , NI, NI, NI, NI,
   NI   , NI     , NI   , NI   , NI, NI, NI, NI,
   NI   , CVT_D_S, NI   , NI   , NI, NI, NI, NI,
   NI   , NI     , NI   , NI   , NI, NI, NI, NI,
   NI   , NI     , NI   , NI   , NI, NI, NI, NI,
   NI   , NI     , NI   , NI   , NI, NI, NI, NI
};

static void ADD_D()
{
   *reg_cop1_double[cffd] = *reg_cop1_double[cffs] +
     *reg_cop1_double[cfft];
   interp_addr+=4;
}

static void TRUNC_W_D()
{
   if (Status & 0x04000000)
     {
	printf("TRUNC_W_D 64 bits\n");
	stop=1;
     }
   else
     reg_cop1_fgr_32[cffd] = *reg_cop1_double[cffs];
   interp_addr+=4;
}

static void (*interp_cop1_d[64])(void) =
{
   ADD_D, NI, NI, NI, NI, NI       , NI, NI,
   NI   , NI, NI, NI, NI, TRUNC_W_D, NI, NI,
   NI   , NI, NI, NI, NI, NI       , NI, NI,
   NI   , NI, NI, NI, NI, NI       , NI, NI,
   NI   , NI, NI, NI, NI, NI       , NI, NI,
   NI   , NI, NI, NI, NI, NI       , NI, NI,
   NI   , NI, NI, NI, NI, NI       , NI, NI,
   NI   , NI, NI, NI, NI, NI       , NI, NI
};

static void CVT_S_W()
{
   if (Status & 0x04000000)
     {
	printf("CVT.S.W 64 bits\n");
	stop=1;
     }
   else
     {
	float f;
	f = reg_cop1_fgr_32[cffs];
	*reg_cop1_simple[cffd] = f;
     }
   interp_addr+=4;
}

static void (*interp_cop1_w[64])(void) =
{
   NI     , NI, NI, NI, NI, NI, NI, NI,
   NI     , NI, NI, NI, NI, NI, NI, NI,
   NI     , NI, NI, NI, NI, NI, NI, NI,
   NI     , NI, NI, NI, NI, NI, NI, NI,
   CVT_S_W, NI, NI, NI, NI, NI, NI, NI,
   NI     , NI, NI, NI, NI, NI, NI, NI,
   NI     , NI, NI, NI, NI, NI, NI, NI,
   NI     , NI, NI, NI, NI, NI, NI, NI
};

static void MFC1()
{
   if (Status & 0x04000000)
     {
	printf("MFC1 64 bits\n");
	stop=1;
     }
   else
     {
	rrt = reg_cop1_fgr_32[rfs];
	sign_extended(rrt);
     }
   interp_addr+=4;
}

static void CFC1()
{
   if (rfs==31)
     {
	rrt = FCR31;
	sign_extended(rrt);
     }
   if (rfs==0)
     {
	rrt = FCR0;
	sign_extended(rrt);
     }
   interp_addr+=4;
}

static void MTC1()
{
   if (Status & 0x04000000)
     {
	printf("MTC1 64 bits\n");
	stop=1;
     }
   else
     {
	reg_cop1_fgr_32[rfs] = rrt & 0xFFFFFFFF;
     }
   interp_addr+=4;
}

static void CTC1()
{
   if (rfs==31)
     FCR31 = rrt & 0xFFFFFFFF;
   interp_addr+=4;
}

static void S()
{
   interp_cop1_s[(op & 0x3F)]();
}

static void D()
{
   interp_cop1_d[(op & 0x3F)]();
}

static void W()
{
   interp_cop1_w[(op & 0x3F)]();
}

static void (*interp_cop1[32])(void) =
{
   MFC1, NI, CFC1, NI, MTC1, NI, CTC1, NI,
   NI  , NI, NI  , NI, NI  , NI, NI  , NI,
   S   , D , NI  , NI, W   , NI, NI  , NI,
   NI  , NI, NI  , NI, NI  , NI, NI  , NI
};

static void SPECIAL()
{
   interp_special[(op & 0x3F)]();
}

static void REGIMM()
{
   interp_regimm[((op >> 16) & 0x1F)]();
}

static void J()
{
   unsigned long naddr = (PC->f.j.inst_index<<2) | (interp_addr & 0xF0000000);
   if (naddr == interp_addr)
     if (sp_register.halt)
     {
	interp_addr+=4;
	prefetch();
	interp_addr-=4;
	if (op == 0)
	  {
	     skip = next_interupt - Count;
	     if (skip >= 3) 
	       {
		  Count += (skip & 0xFFFFFFFC);
		  return;
	       }
	  }
	prefetch();
     }
   interp_addr+=4;
   delay_slot=1;
   update_system();
   prefetch();
   interp_ops[((op >> 26) & 0x3F)]();
   delay_slot=0;
   if (!skip_jump)
     interp_addr = naddr;
   else skip_jump = 0;
}

static void JAL()
{
   unsigned long naddr = (PC->f.j.inst_index<<2) | (interp_addr & 0xF0000000);
   if (naddr == interp_addr)
     if (sp_register.halt)
     {
	interp_addr+=4;
	prefetch();
	interp_addr-=4;
	if (op == 0)
	  {
	     skip = next_interupt - Count;
	     if (skip >= 3) 
	       {
		  Count += (skip & 0xFFFFFFFC);
		  return;
	       }
	  }
	prefetch();
     }
   interp_addr+=4;
   delay_slot=1;
   update_system();
   prefetch();
   interp_ops[((op >> 26) & 0x3F)]();
   delay_slot=0;
   if (!skip_jump)
     {
	reg[31]=interp_addr;
	sign_extended(reg[31]);
	
	interp_addr = naddr;
     }
   else skip_jump = 0;
}

static void BEQ()
{
   short local_immediate = iimmediate;
   local_rs = irs;
   local_rt = irt;
   if ((interp_addr + (local_immediate+1)*4) == interp_addr)
     if (sp_register.halt)
       if (local_rs == local_rt)
     {
	interp_addr+=4;
	prefetch();
	interp_addr-=4;
	if (op == 0)
	  {
	     skip = next_interupt - Count;
	     if (skip >= 3) 
	       {
		  Count += (skip & 0xFFFFFFFC);
		  return;
	       }
	  }
	prefetch();
     }
   interp_addr+=4;
   delay_slot=1;
   update_system();
   prefetch();
   interp_ops[((op >> 26) & 0x3F)]();
   delay_slot=0;
   if (local_rs == local_rt)
     if (!skip_jump)
       interp_addr += (local_immediate-1)*4;
   if (skip_jump) skip_jump=0;
}

static void BNE()
{
   short local_immediate = iimmediate;
   local_rs = irs;
   local_rt = irt;
   if ((interp_addr + (local_immediate+1)*4) == interp_addr)
     if (sp_register.halt)
       if (local_rs != local_rt)
     {
	interp_addr+=4;
	prefetch();
	interp_addr-=4;
	if (op == 0)
	  {
	     skip = next_interupt - Count;
	     if (skip >= 3) 
	       {
		  Count += (skip & 0xFFFFFFFC);
		  return;
	       }
	  }
	prefetch();
     }
   interp_addr+=4;
   delay_slot=1;
   update_system();
   prefetch();
   interp_ops[((op >> 26) & 0x3F)]();
   delay_slot=0;
   if (local_rs != local_rt)
     if (!skip_jump)
       interp_addr += (local_immediate-1)*4;
   if (skip_jump) skip_jump=0;
}

static void BLEZ()
{
   short local_immediate = iimmediate;
   local_rs = irs;
   if ((interp_addr + (local_immediate+1)*4) == interp_addr)
     if (sp_register.halt)
       if (local_rs <= 0)
     {
	interp_addr+=4;
	prefetch();
	interp_addr-=4;
	if (op == 0)
	  {
	     skip = next_interupt - Count;
	     if (skip >= 3) 
	       {
		  Count += (skip & 0xFFFFFFFC);
		  return;
	       }
	  }
	prefetch();
     }
   interp_addr+=4;
   delay_slot=1;
   update_system();
   prefetch();
   interp_ops[((op >> 26) & 0x3F)]();
   delay_slot=0;
   if (local_rs <= 0)
     if (!skip_jump)
       interp_addr += (local_immediate-1)*4;
   if (skip_jump) skip_jump=0;
}

static void BGTZ()
{
   short local_immediate = iimmediate;
   local_rs = irs;
   if ((interp_addr + (local_immediate+1)*4) == interp_addr)
     if (sp_register.halt)
       if (local_rs > 0)
     {
	interp_addr+=4;
	prefetch();
	interp_addr-=4;
	if (op == 0)
	  {
	     skip = next_interupt - Count;
	     if (skip >= 3) 
	       {
		  Count += (skip & 0xFFFFFFFC);
		  return;
	       }
	  }
	prefetch();
     }
   interp_addr+=4;
   delay_slot=1;
   update_system();
   prefetch();
   interp_ops[((op >> 26) & 0x3F)]();
   delay_slot=0;
   if (local_rs > 0)
     if (!skip_jump)
       interp_addr += (local_immediate-1)*4;
   if (skip_jump) skip_jump=0;
}

static void ADDI()
{
   irt = irs + iimmediate;
   sign_extended(irt);
   interp_addr+=4;
}

static void ADDIU()
{
   irt = irs + iimmediate;
   sign_extended(irt);
   interp_addr+=4;
}

static void SLTI()
{
   if (irs < iimmediate) irt = 1;
   else irt = 0;
   interp_addr+=4;
}

static void SLTIU()
{
   if ((irs & 0x7FFFFFFFFFFFFFFFLL) < iimmediate) irt = 1;
   else irt = 0;
   interp_addr+=4;
}

static void ANDI()
{
   irt = irs & (iimmediate & 0xFFFF);
   interp_addr+=4;
}

static void ORI()
{
   irt = irs | (unsigned short)iimmediate;
   interp_addr+=4;
}

static void XORI()
{
   irt = irs ^ (iimmediate & 0xFFFF);
   interp_addr+=4;
}

static void LUI()
{
   irt = iimmediate << 16;
   sign_extended(irt);
   interp_addr+=4;
}

static void COP0()
{
   interp_cop0[((op >> 21) & 0x1F)]();
}

static void COP1()
{
   interp_cop1[((op >> 21) & 0x1F)]();
}

static void BEQL()
{
   short local_immediate = iimmediate;
   if ((interp_addr + (local_immediate+1)*4) == interp_addr)
     if (sp_register.halt)
       if (irs == irt)
     {
	interp_addr+=4;
	prefetch();
	interp_addr-=4;
	if (op == 0)
	  {
	     skip = next_interupt - Count;
	     if (skip >= 3) 
	       {
		  Count += (skip & 0xFFFFFFFC);
		  return;
	       }
	  }
	prefetch();
     }
   if (irs == irt)
     {
	interp_addr+=4;
	delay_slot=1;
	update_system();
	prefetch();
	interp_ops[((op >> 26) & 0x3F)]();
	delay_slot=0;
	if (!skip_jump)
	  interp_addr += (local_immediate-1)*4;
	else skip_jump = 0;
     }
   else interp_addr+=8;
}

static void BNEL()
{
   short local_immediate = iimmediate;
   if ((interp_addr + (local_immediate+1)*4) == interp_addr)
     if (sp_register.halt)
       if (irs != irt)
     {
	interp_addr+=4;
	prefetch();
	interp_addr-=4;
	if (op == 0)
	  {
	     skip = next_interupt - Count;
	     if (skip >= 3) 
	       {
		  Count += (skip & 0xFFFFFFFC);
		  return;
	       }
	  }
	prefetch();
     }
   if (irs != irt)
     {
	interp_addr+=4;
	delay_slot=1;
	update_system();
	prefetch();
	interp_ops[((op >> 26) & 0x3F)]();
	delay_slot=0;
	if (!skip_jump)
	  interp_addr += (local_immediate-1)*4;
	else skip_jump = 0;
     }
   else interp_addr+=8;
}

static void BLEZL()
{
   short local_immediate = iimmediate;
   if ((interp_addr + (local_immediate+1)*4) == interp_addr)
     if (sp_register.halt)
       if (irs <= 0)
     {
	interp_addr+=4;
	prefetch();
	interp_addr-=4;
	if (op == 0)
	  {
	     skip = next_interupt - Count;
	     if (skip >= 3) 
	       {
		  Count += (skip & 0xFFFFFFFC);
		  return;
	       }
	  }
	prefetch();
     }
   if (irs <= 0)
     {
	interp_addr+=4;
	delay_slot=1;
	update_system();
	prefetch();
	interp_ops[((op >> 26) & 0x3F)]();
	delay_slot=0;
	if (!skip_jump)
	  interp_addr += (local_immediate-1)*4;
	else skip_jump = 0;
     }
   else interp_addr+=8;
}

static void LB()
{
   interp_addr+=4;
   address = iimmediate + irs;
   rdword = &irt;
   read_byte_in_memory();
   sign_extendedb(irt);
}

static void LWL()
{
   unsigned long long int word;
   interp_addr+=4;
   switch ((iimmediate + irs) & 3)
     {
      case 0:
	address = iimmediate + irs;
	rdword = &lsrt;
	read_word_in_memory();
	break;
      case 1:
	address = (iimmediate + irs) & 0xFFFFFFFC;
	rdword = &word;
	read_word_in_memory();
	lsrt = (lsrt & 0xFF) | (word << 8);
	break;
      case 2:
	address = (iimmediate + irs) & 0xFFFFFFFC;
	rdword = &word;
	read_word_in_memory();
	lsrt = (lsrt & 0xFFFF) | (word << 16);
	break;
      case 3:
	address = (iimmediate + irs) & 0xFFFFFFFC;
	rdword = &word;
	read_word_in_memory();
	lsrt = (lsrt & 0xFFFFFF) | (word << 24);
	break;
     }
   sign_extended(lsrt);
}

static void LW()
{
   address = iimmediate + irs;
   rdword = &irt;
   interp_addr+=4;
   read_word_in_memory();
   sign_extended(irt);
}

static void LBU()
{
   interp_addr+=4;
   address = iimmediate + irs;
   rdword = &irt;
   read_byte_in_memory();
}

static void LHU()
{
   interp_addr+=4;
   address = iimmediate + irs;
   rdword = &irt;
   read_hword_in_memory();
}

static void LWR()
{
   unsigned long long int word;
   interp_addr+=4;
   switch ((iimmediate + irs) & 3)
     {
      case 0:
	address = (iimmediate + irs) & 0xFFFFFFFC;
	rdword = &word;
	read_word_in_memory();
	lsrt = (lsrt & 0xFFFFFF00) | (word >> 24);
	break;
      case 1:
	address = (iimmediate + irs) & 0xFFFFFFFC;
	rdword = &word;
	read_word_in_memory();
	lsrt = (lsrt & 0xFFFF0000) | (word >> 16);
	break;
      case 2:
	address = (iimmediate + irs) & 0xFFFFFFFC;
	rdword = &word;
	read_word_in_memory();
	lsrt = (lsrt & 0xFF000000) | (word >> 8);
	break;
      case 3:
	address = (iimmediate + irs) & 0xFFFFFFFC;
	rdword = &lsrt;
	read_word_in_memory();
	sign_extended(lsrt);
     }
}

static void SB()
{
   interp_addr+=4;
   address = iimmediate + irs;
   byte = (unsigned char)(irt & 0xFF);
   write_byte_in_memory();
}

static void SH()
{
   interp_addr+=4;
   address = iimmediate + irs;
   hword = (unsigned short)(irt & 0xFFFF);
   write_hword_in_memory();
}

static void SWL()
{
   unsigned long long int old_word;
   interp_addr+=4;
   switch ((iimmediate + irs) & 3)
     {
      case 0:
	address = (iimmediate + irs) & 0xFFFFFFFC;
	word = (unsigned long)lsrt;
	write_word_in_memory();
	break;
      case 1:
	address = (iimmediate + irs) & 0xFFFFFFFC;
	rdword = (unsigned long long int *)&old_word;
	read_word_in_memory();
	word = ((unsigned long)lsrt >> 8) | (old_word & 0xFF000000);
	write_word_in_memory();
	break;
      case 2:
	address = (iimmediate + irs) & 0xFFFFFFFC;
	rdword = (unsigned long long int *)&old_word;
	read_word_in_memory();
	word = ((unsigned long)lsrt >> 16) | (old_word & 0xFFFF0000);
	write_word_in_memory();
	break;
      case 3:
	address = iimmediate + irs;
	byte = (unsigned char)(lsrt >> 24);
	write_byte_in_memory();
	break;
     }
}

static void SW()
{
   interp_addr+=4;
   address = iimmediate + irs;
   word = (unsigned long)(irt & 0xFFFFFFFF);
   write_word_in_memory();
}

static void SWR()
{
   unsigned long long int old_word;
   interp_addr+=4;
   switch ((iimmediate + irs) & 3)
     {
      case 0:
	address = iimmediate + irs;
	byte = (unsigned char)(lsrt << 24);
	write_byte_in_memory();
	break;
      case 1:
	address = (iimmediate + irs) & 0xFFFFFFFC;
	rdword = (unsigned long long int *)&old_word;
	read_word_in_memory();
	word = ((unsigned long)lsrt << 16) | (old_word & 0x0000FFFF);
	write_word_in_memory();
	break;
      case 2:
	address = (iimmediate + irs) & 0xFFFFFFFC;
	rdword = (unsigned long long int *)&old_word;
	read_word_in_memory();
	word = ((unsigned long)lsrt << 8) | (old_word & 0x000000FF);
	write_word_in_memory();
	break;
      case 3:
	address = (iimmediate + irs) & 0xFFFFFFFC;
	word = (unsigned long)lsrt;
	write_word_in_memory();
	break;
     }
}

static void CACHE()
{
   interp_addr+=4;
}

static void LWC1()
{
   interp_addr+=4;
   if (Status & 0x04000000)
     {
	printf("lwc 64 bits\n");
	stop=1;
     }
   else
     {
	unsigned long long int temp;
	address = lfoffset+reg[lfbase];
	rdword = &temp;
	read_word_in_memory();
	reg_cop1_fgr_32[lfft] = *rdword & 0xFFFFFFFF;
     }
}

static void LD()
{
   interp_addr+=4;
   address = iimmediate + irs;
   rdword = &irt;
   read_dword_in_memory();
}

static void SD()
{
   interp_addr+=4;
   address = iimmediate + irs;
   dword = irt;
   write_dword_in_memory();
}

static void (*interp_ops[64])(void) =
{
   SPECIAL, REGIMM, J   , JAL  , BEQ , BNE , BLEZ , BGTZ ,
   ADDI   , ADDIU , SLTI, SLTIU, ANDI, ORI , XORI , LUI  ,
   COP0   , COP1  , NI  , NI   , BEQL, BNEL, BLEZL, NI   ,
   NI     , NI    , NI  , NI   , NI  , NI  , NI   , NI   ,
   LB     , NI    , LWL , LW   , LBU , LHU , LWR  , NI   ,
   SB     , SH    , SWL , SW   , NI  , NI  , SWR  , CACHE,
   NI     , LWC1  , NI  , NI   , NI  , NI  , NI   , LD   ,
   NI     , NI    , NI  , NI   , NI  , NI  , NI   , SD
};

static void prefetch()
{
   if ((interp_addr >= 0x80000000) && (interp_addr < 0xc0000000))
     {
	if ((interp_addr >= 0xa4000000) && (interp_addr < 0xa4001000))
	  {
	     op = SP_DMEM[(interp_addr&0xFFF)/4];
	     prefetch_opcode(op);
	  }
	else if ((interp_addr >= 0x80000000) && (interp_addr < 0x80400000))
	  {
	     op = rdram[(interp_addr&0xFFFFFF)/4];
	     prefetch_opcode(op);
	  }
	else
	  {
	     printf("execution  l'addresse :%x\n", (int)interp_addr);
	     stop=1;
	  }
     }
   else
     {
	printf("execute instruction dans l'espace virtuel\n");
	stop=1;
     }
}

void pure_interpreter()
{
   interp_addr = 0xa4000040;
   stop=0;
   PC = malloc(sizeof(precomp_instr));
   while (!stop)
     {
	prefetch();
	interp_ops[((op >> 26) & 0x3F)]();
	update_system();
     }
   PC->addr = interp_addr;
}
