//============================================================================
//
//   SSSS    tt          lll  lll       
//  SS  SS   tt           ll   ll        
//  SS     tttttt  eeee   ll   ll   aaaa 
//   SSSS    tt   ee  ee  ll   ll      aa
//      SS   tt   eeeeee  ll   ll   aaaaa  --  "An Atari 2600 VCS Emulator"
//  SS  SS   tt   ee      ll   ll  aa  aa
//   SSSS     ttt  eeeee llll llll  aaaaa
//
// Copyright (c) 1995-1998 by Bradford W. Mott
//
// See the file "license" for information on usage and redistribution of
// this file, and for a DISCLAIMER OF ALL WARRANTIES.
//
// $Id: mainX11.cxx,v 1.4 1998/08/29 16:08:30 bwmott Exp $
//============================================================================

// TODO:
// - filetypes
// - using overlays
// - fix colors
// - need icons
// - nice rescaling
// - network game
// - add attributes in tracker (?)

// Done:
// - an info view/window for the BFilePanel/BMenu -> gives info on the selected game
// - make the emulator a replicant
// - STellaView positionning rework
// - fixed drawing optimization (less rect drawn)
// - save filepanel directory
// - draw the focus state in the BFilePanel
// - separate the emulator thread and the drawing thread
// - fix the BDragger's color
// - multiple players (keyboard)
// - in PAL, switch to 50Hz
// - fix the deadlock 
// - support for all controls
// - lightgun
// - Set the default controler
// - Directwindow


#include <assert.h>
#include <fstream.h>
#include <iostream.h>
#include <strstream.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <errno.h>

#include "mainBeOS.h"

#include "bspf.hxx"
#include "Console.hxx"
#include "DefProps.hxx"
#include "Event.hxx"
#include "MediaSrc.hxx"
#include "PropsSet.hxx"
#include "SndUnix.hxx"
#include "System.hxx"


// ********************************************************************

int main()
{
	// Sanity check
	assert(is_computer_on());

	// We can run the app now!
	STellaApplication app;
	app.Run();
	return 0;
}

// ********************************************************************
#pragma mark -


const char *STellaApplication::fApplicationSignature = "application/x-vnd.ma.stella";
 
STellaApplication::STellaApplication()
	: BApplication(fApplicationSignature),
	fStellaWindow(NULL)
{
}

STellaApplication::~STellaApplication()
{
}

void STellaApplication::AboutRequested(void)
{
	BAlert *theAlert = new BAlert("About Stella"B_UTF8_ELLIPSIS,
        "Stella\n"
        "An Atari 2600 VCS Emulator\n"
        "Version 1.2.1\n\n"
		"Copyright " B_UTF8_COPYRIGHT " 1995-1998 by Bradford W. Mott\n"
		"BeOS port by Mathias Agopian\n"
        "e-mail : mathias.agopian@teaser.fr\n\n",
        "OK", NULL, NULL, B_WIDTH_AS_USUAL);

	BTextView *theText = theAlert->TextView();
	if (theText)
	{
		theText->SetStylable(true);
		theText->Select(0, strlen("Stella"));
		BFont ourFont;
		theText->SetFontAndColor(be_bold_font);
		theText->GetFontAndColor(2, &ourFont, NULL);
		ourFont.SetSize(30);
		uint32 code = ourFont.FamilyAndStyle();
		ourFont.SetFamilyAndStyle("Baskerville","Italic");
		if (code == ourFont.FamilyAndStyle())
		{ // something went wrong. Try another name (it's scary, I know)
			ourFont.SetFamilyAndStyle("Baskerville BT","Italic");
		}
		theText->SetFontAndColor(&ourFont);
	}
	theAlert->Go();   
}

void STellaApplication::ArgvReceived(int32 argc, char **argv)
{
	// Argv received: Construct a B_REFS_RECEIVED message and call RefsReceived()
	BMessage refs(B_REFS_RECEIVED);
	for (int i=1 ; i<argc ; i++)
	{
		entry_ref ref;
		if (get_ref_for_path(argv[i], &ref) == B_OK)
			refs.AddRef("refs", &ref);
	}

	RefsReceived(&refs);
}

void STellaApplication::RefsReceived(BMessage *msg)
{
	// Process refs
	uint32 type; 
	int32 count; 
	entry_ref ref;

	if (msg->GetInfo("refs", &type, &count) != B_OK)	return;
	if (type != B_REF_TYPE)								return;
	if (msg->FindRef("refs", &ref) != B_OK )			return;
		
	// Begin the show!
	if (fStellaWindow == NULL)
		open_stella_window();
	
	// If a tracker window opened me, get a messenger from it.
	if (fStellaWindow->Lock())
	{
		fStellaWindow->fRef = ref;
		if (msg->HasMessenger("TrackerViewToken"))
		{
			BMessenger m;
			msg->FindMessenger("TrackerViewToken", &m);
			fStellaWindow->fTrackerMessenger = m;
		}
		fStellaWindow->Unlock();
	}

	// Tell the STellaView to load begin the game	
	fStellaWindow->PostMessage(msg, fStellaWindow->fStellaView);
}

void STellaApplication::ReadyToRun()
{
	if (fStellaWindow == NULL)
		open_stella_window();
}

void STellaApplication::open_stella_window()
{
	// Figure out the desired size of the window
	BRect screen = BScreen().Frame();
	BRect r(0, 0, 640-1, 400-1);
	r.OffsetTo(	(screen.Width()-r.Width()) * 0.5f,
				(screen.Height()-r.Height()) * 0.5f);
	(fStellaWindow = new STellaWindow(r))->Show();
	fStellaWindow->DisplayMainMenu();
}

// ********************************************************************
#pragma mark -

STellaWindow::STellaWindow(BRect r)
	:	BDirectWindow(r, "No game", B_TITLED_WINDOW, 0, B_QUIT_ON_WINDOW_CLOSE | B_ASYNCHRONOUS_CONTROLS),
		fOldFrame(r),
		fStellaView(NULL),
		fKeyMenuBar(NULL),
		fFilePanel(NULL),
		fIsPaused(false),
		fInfoWindow(NULL),
		fInfoView(NULL)
{
	// Menus
	fKeyMenuBar = new BMenuBar(Bounds(), "StellaMenuBar");

	BMenuItem *menuitem;
	BMessage *mes;

	fFileMenu = new BMenu("File");
		fOpenSubmenu = new BMenu("Open" B_UTF8_ELLIPSIS);
		menuitem = new BMenuItem(fOpenSubmenu, new BMessage(MSG_OPEN));
		menuitem->SetShortcut('O', B_COMMAND_KEY);
		fFileMenu->AddItem(menuitem);
		fFileMenu->AddItem(fPreviousMenuItem = new BMenuItem("Previous game", new BMessage(MSG_PREVMSG), B_UP_ARROW));
		fFileMenu->AddItem(fNextMenuItem = new BMenuItem("Next game", new BMessage(MSG_NEXTMSG), B_DOWN_ARROW));
		fFileMenu->AddSeparatorItem();				// --------------
		fFileMenu->AddItem(fInfoMenuItem = new BMenuItem("Game Information" B_UTF8_ELLIPSIS, new BMessage(MSG_GAMEINFO), 'I'));
		fFileMenu->AddSeparatorItem();				// --------------
		menuitem = new BMenuItem("About STella" B_UTF8_ELLIPSIS, new BMessage(B_ABOUT_REQUESTED), 'A');
		menuitem->SetTarget(be_app);
		fFileMenu->AddItem(menuitem);
		fFileMenu->AddSeparatorItem();				// --------------
		fFileMenu->AddItem(new BMenuItem("Quit", new BMessage(B_QUIT_REQUESTED), 'Q'));
		fKeyMenuBar->AddItem(fFileMenu);

	fViewMenu = new BMenu("Options");
		mes = new BMessage(MSG_RESOLUTION);
		mes->AddFloat("factor", 1.0f);
		fViewMenu->AddItem(new BMenuItem("Small size", mes, 'S'));
		mes = new BMessage(MSG_RESOLUTION);
		mes->AddFloat("factor", 2.0f);
		fViewMenu->AddItem(new BMenuItem("Medium size", mes, 'M'));
		mes = new BMessage(MSG_RESOLUTION);
		mes->AddFloat("factor", 3.0f);
		fViewMenu->AddItem(new BMenuItem("Large size", mes, 'L'));
		fViewMenu->AddItem(new BMenuItem("Full Screen", new BMessage(MSG_FULLSCREEN), 'F'));
		fViewMenu->AddSeparatorItem();				// --------------
		fViewMenu->AddItem(fPauseMenuItem = new BMenuItem("Pause Game", new BMessage(MSG_PAUSE_GAME), 'P'));
		fViewMenu->AddSeparatorItem();				// --------------
		fViewMenu->AddItem(fFpsMenuItem = new BMenuItem("Show Frame rate", new BMessage(MSG_SHOW_FPS)));
		fViewMenu->AddSeparatorItem();				// --------------
		fViewMenu->AddItem(fDirectWindowMenuItem = new BMenuItem("Use frame buffer", new BMessage(MSG_DIRECT_WINDOW)));
		fKeyMenuBar->AddItem(fViewMenu);

	(fLeftControlMenu = new BMenu("Left Controller"))->SetRadioMode(true);
		mes = new BMessage(MSG_SELECT_CONTROLLER);
		mes->AddInt32("ctrl", STellaView::JOYSTICK);		mes->AddInt32("pos", 0);
		fLeftControlMenu->AddItem(new BMenuItem("Joystick", mes));

		mes = new BMessage(MSG_SELECT_CONTROLLER);
		mes->AddInt32("ctrl", STellaView::BOOSTER_GRIP);	mes->AddInt32("pos", 0);
		fLeftControlMenu->AddItem(new BMenuItem("Booster-Grip", mes));

		mes = new BMessage(MSG_SELECT_CONTROLLER);
		mes->AddInt32("ctrl", STellaView::DRIVING);			mes->AddInt32("pos", 0);
		fLeftControlMenu->AddItem(new BMenuItem("Driving Controller", mes));

		mes = new BMessage(MSG_SELECT_CONTROLLER);
		mes->AddInt32("ctrl", STellaView::PADDLE);			mes->AddInt32("pos", 0);
		fLeftControlMenu->AddItem(new BMenuItem("Paddle", mes));

		mes = new BMessage(MSG_SELECT_CONTROLLER);
		mes->AddInt32("ctrl", STellaView::KEYBOARD);		mes->AddInt32("pos", 0);
		fLeftControlMenu->AddItem(new BMenuItem("Keyboard", mes));

		mes = new BMessage(MSG_SELECT_CONTROLLER);
		mes->AddInt32("ctrl", STellaView::LIGHTGUN);		mes->AddInt32("pos", 0);
		fLeftControlMenu->AddItem(new BMenuItem("Light-gun", mes));

		mes = new BMessage(MSG_SELECT_CONTROLLER);
		mes->AddInt32("ctrl", STellaView::MINDLINK);		mes->AddInt32("pos", 0);
		fLeftControlMenu->AddItem(new BMenuItem("Mind link", mes));

		fLeftControlMenu->ItemAt(STellaView::JOYSTICK)->SetMarked(true);
		fKeyMenuBar->AddItem(fLeftControlMenu);

	(fRightControlMenu = new BMenu("Right Controller"))->SetRadioMode(true);
		mes = new BMessage(MSG_SELECT_CONTROLLER);
		mes->AddInt32("ctrl", STellaView::JOYSTICK);		mes->AddInt32("pos", 1);
		fRightControlMenu->AddItem(new BMenuItem("Joystick", mes));

		mes = new BMessage(MSG_SELECT_CONTROLLER);
		mes->AddInt32("ctrl", STellaView::BOOSTER_GRIP);	mes->AddInt32("pos", 1);
		fRightControlMenu->AddItem(new BMenuItem("Booster-Grip", mes));

		mes = new BMessage(MSG_SELECT_CONTROLLER);
		mes->AddInt32("ctrl", STellaView::DRIVING);			mes->AddInt32("pos", 1);
		fRightControlMenu->AddItem(new BMenuItem("Driving Controller", mes));

		mes = new BMessage(MSG_SELECT_CONTROLLER);
		mes->AddInt32("ctrl", STellaView::PADDLE);			mes->AddInt32("pos", 1);
		fRightControlMenu->AddItem(new BMenuItem("Paddle", mes));

		mes = new BMessage(MSG_SELECT_CONTROLLER);
		mes->AddInt32("ctrl", STellaView::KEYBOARD);		mes->AddInt32("pos", 1);
		fRightControlMenu->AddItem(new BMenuItem("Keyboard", mes));

		fRightControlMenu->ItemAt(STellaView::JOYSTICK)->SetMarked(true);
		fKeyMenuBar->AddItem(fRightControlMenu);

	// Disable View and Control menu when no game loaded
	fViewMenu->SetEnabled(false);

	// Add the menu to the window
	AddChild(fKeyMenuBar);
	fKeyMenuBarHeight = fKeyMenuBar->Bounds().Height()+1.0f;
	ResizeBy(0, fKeyMenuBarHeight);

	// A black background view
	BRect rect = Bounds();
	rect.top += fKeyMenuBarHeight;
	fBackGround = new BView(rect, "background", B_FOLLOW_ALL, B_WILL_DRAW);
	fBackGround->SetViewColor(0,0,0);
	AddChild(fBackGround);

		// Add the game view
		fBackGround->AddChild(fStellaView = new STellaView(Bounds()));
		fStellaView->MakeFocus();
		fStellaView->SetHOffset((int)fKeyMenuBarHeight);
	
	// Shortcut
	AddShortcut('F', B_COMMAND_KEY, new BMessage(MSG_FULLSCREEN));
	AddShortcut('P', B_COMMAND_KEY, new BMessage(MSG_PAUSE_GAME));

	// FilePanel
	fFilePanel = new STellaFilePanel();
	
	// Set window's limits
	float minw, maxw, minh, maxh;
	GetSizeLimits(&minw, &maxw, &minh, &maxh);
	SetSizeLimits(160.0f, maxw, 100.0f+fKeyMenuBarHeight, maxh);
}

STellaWindow::~STellaWindow()
{
	delete fFilePanel;
}

bool STellaWindow::QuitRequested(void)
{
	be_app->PostMessage(B_QUIT_REQUESTED);
	return true;
}

void STellaWindow::MenusBeginning(void)
{
	if (fStellaView)
	{
		// Pause item
		fPauseMenuItem->SetMarked(fIsPaused);

		// FPS item
		fFpsMenuItem->SetMarked(fStellaView->IsFpsShowing());

		// Direct Window
		fDirectWindowMenuItem->SetMarked(fStellaView->GetDirectWindow());

		// Game info
		game_info dummy;	
		fInfoMenuItem->SetEnabled(fStellaView->GetGameInfo(&dummy) == B_OK);	
	}

	// Update Open menu
	BMenuItem *item;
	while ( (item = fOpenSubmenu->RemoveItem((int32)0)) )
		delete item;

	BMessage refList;
	be_roster->GetRecentDocuments(&refList, 10, NULL, STellaApplication::fApplicationSignature);
	uint32 type; 
	int32 count; 
	entry_ref ref;
	if ((refList.GetInfo("refs", &type, &count) == B_OK) && (type == B_REF_TYPE))
	{
		for (int i=0 ; i<count ; i++)
		{
			if (refList.FindRef("refs", i, &ref) == B_OK)
			{
				char name[B_FILE_NAME_LENGTH];
				BEntry entry(&ref);
				entry.GetName(name);
				BMessage *refs = new BMessage(B_REFS_RECEIVED);
				refs->AddRef("refs", &ref);
				BMenuItem *item = new BMenuItem(name, refs);
				item->SetTarget(be_app);
				fOpenSubmenu->AddItem(item);
			}
		}
	}
	
	// Prev/next item
	bool b = fTrackerMessenger.IsValid();
	fPreviousMenuItem->SetEnabled(b);
	fNextMenuItem->SetEnabled(b);

	BDirectWindow::MenusBeginning();
}

void STellaWindow::MenusEnded(void)
{
	BDirectWindow::MenusEnded();
}

void STellaWindow::DirectConnected(direct_buffer_info *info)
{
	fStellaView->DirectConnected(info);
	BDirectWindow::DirectConnected(info);
}


void STellaWindow::MessageReceived(BMessage *msg)
{
	float f;

	switch(msg->what)
	{
		case MSG_OPEN:
			fFilePanel->Show();
			break;

		case MSG_FULLSCREEN:
			{ // Full Screen mode toggle
				bool fullScreen = !(IsFullScreen());
				if (SetFullScreen(fullScreen) == B_OK)
				{ // in fullscreen mode, hide the cursor
					if (fullScreen)		be_app->HideCursor();
					else				be_app->ShowCursor();
				}
			}
			if ((SupportsWindowMode() == false) && (IsFullScreen() == false) && (fStellaView->GetDirectWindow()))
			{ // The video card does not support DirectWindow AND Window mode
				PostMessage(MSG_DIRECT_WINDOW);
			}
			break;

		case MSG_DIRECT_WINDOW:
			if (fStellaView->GetDirectWindow() == false)
			{ // Terminate immidiately the current drawing before switching mode
				UpdateIfNeeded();
				fStellaView->SetDirectWindow(true);
			}
			else
			{ // Back to normal mode, then terminate immidiately the current drawing
				fStellaView->SetDirectWindow(false);
				UpdateIfNeeded();
			}
			if ((SupportsWindowMode() == false) && (IsFullScreen() == false) && (fStellaView->GetDirectWindow()))
			{ // The video card does not support DirectWindow AND Window mode
				PostMessage(MSG_FULLSCREEN);
			}
			break;

		case MSG_PAUSE_GAME:
			if (fStellaView->IsPaused())
				fStellaView->ResumeGame();
			else
				fStellaView->PauseGame();
			fIsPaused = fStellaView->IsPaused();
			break;

		case MSG_SHOW_FPS:
			fStellaView->ShowFps(!fStellaView->IsFpsShowing());
			SetDisplaySize();
			break;

		case MSG_RESOLUTION:
			if ((msg->FindFloat("factor", &f) == B_OK))
			{
				fStellaView->ResizeToConsole(f);
				SetDisplaySize(); // Accomodate all elements for this width
			}
			break;

		case MSG_GAMEINFO:
			ShowInfoWindow();
			break;

		case MSG_PREVMSG:
		case MSG_NEXTMSG:
		{
			entry_ref nextRef = fRef;
			if (GetTrackerWindowFile(&nextRef, (msg->what == MSG_NEXTMSG)))
			{
				BMessage refs(B_REFS_RECEIVED);
				refs.AddRef("refs", &nextRef);
				be_app->PostMessage(&refs); // Must pass through the BApplication
				SetTrackerSelection(&nextRef);
			}
			break;
		}

		case MSG_GAME_STARTED:
			Play();
			break;

		case MSG_SELECT_CONTROLLER:
		{
			const int32 controller = msg->FindInt32("ctrl");
			if (controller == STellaView::MINDLINK)
			{
				(new BAlert("",
					"\nYour mind is now linked to Stella :-p",
			        "Argh", NULL, NULL, B_WIDTH_AS_USUAL, B_WARNING_ALERT))->Go();
				fLeftControlMenu->ItemAt(STellaView::MINDLINK)->SetEnabled(false);
				break;
			}
			const int32 position = msg->FindInt32("pos");
			if (position == 0)	fStellaView->SetLeftController((STellaView::controller_t)controller);
			else				fStellaView->SetRightController((STellaView::controller_t)controller);
			break;
		}

		default:
			BDirectWindow::MessageReceived(msg);
			break;
	}
}


void STellaWindow::Play()
{
	// Set the windows Title
	game_info gameInfo;
	if (fStellaView->GetGameInfo(&gameInfo) == B_OK)
	{
		SetTitle(gameInfo.name.String());
		SetDisplaySize();
		
		if ((fInfoView) && (fInfoView->LockLooper()))
		{
			fInfoView->Update(&gameInfo);
			fInfoView->UnlockLooper();			
		}

		// Enable the 'View' menu item
		fViewMenu->SetEnabled(true);
	}
	
	// Display the menu correctly
	DisplayMainMenu();
	
	// Set the default controler
	fLeftControlMenu->ItemAt(fStellaView->GetLeftController())->SetMarked(true);
	fRightControlMenu->ItemAt(fStellaView->GetRightController())->SetMarked(true);
}


void STellaWindow::ShowInfoWindow()
{
	if ((fInfoWindow) && (fInfoWindow->Lock()))
	{
		game_info info;	
		if (fStellaView->GetGameInfo(&info) == B_OK)
			fInfoView->Update(&info);
		if (fInfoWindow->IsHidden())	fInfoWindow->Show();
		else							fInfoWindow->Activate();
		fInfoWindow->Unlock();
	}
	else
	{
		// Info window
		BRect r(0,0,16,16);
		fInfoView = new STellaGameInfoView(r);
		fInfoView->ResizeToPreferred();
		r = fInfoView->Bounds();
		r.OffsetTo(100, 60);
		fInfoWindow = new BWindow(r, 	"Game Informations",
										B_FLOATING_WINDOW_LOOK, 
										B_FLOATING_SUBSET_WINDOW_FEEL,
										B_NOT_V_RESIZABLE);
		fInfoWindow->AddToSubset(this);
		float minw, maxw, minh, maxh;
		fInfoWindow->GetSizeLimits(&minw, &maxw, &minh, &maxh);
		fInfoWindow->SetSizeLimits(r.Width(), maxw, r.Height(), maxh);
		fInfoWindow->AddChild(fInfoView);

		game_info info;	
		if (fStellaView->GetGameInfo(&info) == B_OK)
			fInfoView->Update(&info);

		fInfoWindow->Show();
	}
}


void STellaWindow::WindowActivated(bool active)
{
	if (fStellaView)
	{
		if (active == false)
		{ // Pause the game
			fStellaView->PauseGame();
		}
		else
		{ // Continue the game if it is suspended
			if (fIsPaused == false)
				fStellaView->ResumeGame();
		}
	}
	BDirectWindow::WindowActivated(active);
}


void STellaWindow::DisplayMainMenu(void)
{
	if (Lock())
	{
		if (IsFullScreen())
		{ // Remove main menu
			BScreen s(this);
			if (fKeyMenuBar->IsHidden() == false)
			{
				fKeyMenuBar->Hide();
				fKeyMenuBar->SetEnabled(false);
				fBackGround->ResizeTo(s.Frame().Width(), s.Frame().Height());
				fBackGround->MoveTo(B_ORIGIN);
			}
			fStellaView->SetEventMask(B_POINTER_EVENTS, B_NO_POINTER_HISTORY);
		}
		else
		{ // Add main menu
			fStellaView->SetEventMask(0);
			if (fKeyMenuBar->IsHidden() == true)
			{
				fKeyMenuBar->Show();
				fKeyMenuBar->SetEnabled(true);
				fBackGround->ResizeTo(Frame().Width(), Frame().Height()-fKeyMenuBarHeight);
				fBackGround->MoveTo(0, fKeyMenuBarHeight);
			}
		}
		Unlock();
	}
}

void STellaWindow::SetDisplaySize(void)
{
	if (IsFullScreen() == false)
	{
		float w, h;
		fStellaView->GetPreferredSize(&w, &h);
		ResizeTo(w, fKeyMenuBarHeight + h);
		if ((w == Bounds().Width()) && (fKeyMenuBarHeight + h == Bounds().Height()))
			FrameResized(w, fKeyMenuBarHeight + h);
	}
}

void STellaWindow::FrameResized(float neww, float newh)
{
	BDirectWindow::FrameResized(neww, newh);	
	DisplayMainMenu();	
	fStellaView->ResizeToFitWindow();
}

bool STellaWindow::GetTrackerWindowFile(entry_ref *ref, bool next) const
{
	if (fTrackerMessenger.IsValid() == false)
		return false;
	
	//	Ask the tracker what the next/prev file in the window is.
	//	Continue asking for the next reference until a valid 
	//	file is found.

	entry_ref nextRef = *ref;
	bool foundRef = false;
	while (!foundRef)
	{
		BMessage spc((next ? ('snxt') : ('sprv')));
		spc.AddString("property", "Entry");
		spc.AddRef("data", &nextRef);

		BMessage request(B_GET_PROPERTY);
		request.AddSpecifier(&spc);

		BMessage reply;
		if (fTrackerMessenger.SendMessage(&request, &reply) != B_OK)
			return false;

		if (reply.FindRef("result", &nextRef) != B_OK)
			return false;

		BNode node(&nextRef);
		if (node.InitCheck() != B_OK) 
			return false;

		foundRef = true;	
	}

	*ref = nextRef;
	return foundRef;
}

void STellaWindow::SetTrackerSelection(entry_ref *ref)
{
	if (fTrackerMessenger.IsValid() == false)
		return;

	BMessage setsel(B_SET_PROPERTY);
	setsel.AddSpecifier("Selection");
	setsel.AddRef("data", ref);
	fTrackerMessenger.SendMessage(&setsel);
}


// ********************************************************************
#pragma mark -

STellaView::STellaView(BRect frame)
	:	BView(frame, "STella_view", B_FOLLOW_NONE, B_WILL_DRAW | B_FULL_UPDATE_ON_RESIZE),
		BLocker("DirectRectEngine"),
		fNbRectangles(0),
		fRectangles(NULL),
		fConsole(NULL),
		fQuitIndicator(false),
		fPauseIndicator(false),
		fDesiredPaddle(0),
		fPropertiesSet(NULL),
		fFramePerSecond(60),
		fShowFPS(false),
		fFpsDisplayHeight(20.0f),
		fIsReplicant(false),
		fFocusView(NULL),
		fCurrentFrameBufferIndex(0),
		fProducerFrameBuffer(NULL),
		fDrawSemID(-1),
		fLeftController(JOYSTICK),
		fRightController(JOYSTICK),
		fFrameDuration(1000000/60),
		fDirectWindowMode(false),
		fEventDirectDraw(false),
		fEventDraw(false),
		fConnected(false),
		fClipList(NULL),
		fNumClipRects(0),
		fOriginX(0),
		fOriginY(0)
{
	fConsummerFrameBuffer[0] = NULL;
	fConsummerFrameBuffer[1] = NULL;
	fWidth = Bounds().IntegerWidth()+1;
	fHeight = Bounds().IntegerHeight()+1;

//	BBitmap *bitmap;
//	bitmap = new BBitmap(BRect(0,0,159,199), B_BITMAP_WILL_OVERLAY , B_YCbCr422);
//	printf("B_YCbCr422: %s\n", strerror(bitmap->InitCheck()));
//	delete bitmap;
//	bitmap = new BBitmap(BRect(0,0,159,199), B_BITMAP_WILL_OVERLAY , B_RGB32_LITTLE);
//	printf("B_RGB32_LITTLE: %s\n", strerror(bitmap->InitCheck()));
//	delete bitmap;
}

STellaView::STellaView(BMessage *archive)
	: 	BView(archive),
		BLocker("DirectRectEngine"),
		fNbRectangles(0),
		fRectangles(NULL),
		fConsole(NULL),
		fQuitIndicator(false),
		fPauseIndicator(false),
		fDesiredPaddle(0),
		fPropertiesSet(NULL),
		fFramePerSecond(60),
		fShowFPS(false),
		fFpsDisplayHeight(20.0f),
		fIsReplicant(true),
		fFocusView(NULL),
		fCurrentFrameBufferIndex(0),
		fProducerFrameBuffer(NULL),
		fDrawSemID(-1),
		fLeftController(JOYSTICK),
		fRightController(JOYSTICK),
		fFrameDuration(1000000/60),
		fDirectWindowMode(false),
		fConnected(false),
		fClipList(NULL),
		fNumClipRects(0),
		fOriginX(0),
		fOriginY(0)
{
	fConsummerFrameBuffer[0] = NULL;
	fConsummerFrameBuffer[1] = NULL;
	fWidth = Bounds().IntegerWidth()+1;
	fHeight = Bounds().IntegerHeight()+1;

	SetFlags((Flags() | B_NAVIGABLE) & ~B_FRAME_EVENTS);
	SetResizingMode(B_FOLLOW_NONE);
	const char *path;
	if (archive->FindString("game", &path) == B_OK)
	{
		InsertCartridge(path);
		ResizeToPreferred();
		ResumeGame();
	}
	archive->FindInt32("lctrl", &fLeftController);
	archive->FindInt32("rctrl", &fRightController);
}

BArchivable *STellaView::Instantiate(BMessage *archive)
{
	if (validate_instantiation(archive, "STellaView")) 
		return new STellaView(archive); 
	return NULL;
}

status_t STellaView::Archive(BMessage *archive, bool deep) const
{
	BView::Archive(archive, deep);
	archive->AddString("class", "STellaView");
	archive->AddString("add_on", STellaApplication::fApplicationSignature);
	archive->AddBool("be:unique_replicant", false);
	archive->AddBool("be:load_each_time", true);
	archive->AddBool("be:unload_on_delete", true);
	archive->AddString("game", fGameName.String());
	archive->AddInt32("lctrl", fLeftController);
	archive->AddInt32("rctrl", fRightController);
	return B_OK;
}

STellaView::~STellaView()
{
	RemoveCartridge();
	delete fPropertiesSet;
	fPropertiesSet = NULL;
}

void STellaView::RemoveCartridge(void)
{
	// If a console was opened, then close it!
	if (fConsole)
	{
		// here, stop the game.
		fSound.reset();
		fSound.mute(true);

		// Tell the thread to quit
		fQuitIndicator = true;
		status_t result;

		suspend_thread(fComputeThreadID);
		wait_for_thread(fComputeThreadID, &result); // This thread is safe to quit
		kill_thread(fDrawThreadID); // send a SIGKILL

		// delete the console
		delete_sem(fDrawSemID);
		delete [] fConsummerFrameBuffer[0]; fConsummerFrameBuffer[0] = NULL;
		delete [] fConsummerFrameBuffer[1]; fConsummerFrameBuffer[1] = NULL;
		delete [] fProducerFrameBuffer; fProducerFrameBuffer = NULL;
		delete fConsole; fConsole = NULL;
		delete [] fRectangles; fRectangles = NULL;
	}
}

status_t STellaView::InsertCartridge(const char *file)
{
	// Open the cartridge image and read it in
	ifstream in(file);
	if (!in)
	{
		BString err;
		err << "ERROR: Couldn't open\n" << file << "\n";
		err << "(" << strerror(errno) << ")\n";
		(new BAlert("Stella", err.String(), "Bummer!", NULL, NULL, B_WIDTH_AS_USUAL, B_STOP_ALERT))->Go();
		return errno;
	}

	// Save the path of the game
	fGameName = file;

	// Create a properties set for us to use and set it up
	if (fPropertiesSet == NULL)
	{
		fPropertiesSet = new PropertiesSet("Cartridge.Name");
		setupProperties(*fPropertiesSet);	
	}

	// If a console was opened, then close it!
	RemoveCartridge();

	uInt8* image = new uInt8[512 * 1024];
	in.read((char *)image, 512 * 1024);
	uInt32 size = in.gcount();
	in.close();
	
	// Get just the filename of the file containing the ROM image
	BPath p(file);
	const char* filename = p.Leaf();
	
	// Create the 2600 game console
	fConsole = new Console(image, size, filename, fEvent, *fPropertiesSet, fSound);

	// Free the image since we don't need it any longer
	delete [] image;
	image = NULL;

	// The palette
	const uInt32* palette = fConsole->mediaSource().palette();
	for (int i=0 ; i<256 ; i++)
	{
		fPalette[i].red = (palette[i] & 0x00ff0000) >> 16;
		fPalette[i].green = (palette[i] & 0x0000ff00) >> 8;
		fPalette[i].blue = (palette[i] & 0x000000ff);
		fPalette[i].alpha = 255;
	}
	
//	const uInt32 palette[] = {
//		0x000000,0x757575,0x9a9a9a,0xb5b5b5,0xcbcbcb,0xdedede,0xefefef,0xffffff,
//		0x000000,0x756e00,0x9a9200,0xb5ac00,0xcbc100,0xded300,0xefe300,0xfff100,
//		0x000000,0x755e00,0x9a7c00,0xb59200,0xcba400,0xdeb300,0xefc000,0xffcd00,
//		0x000000,0x753f00,0x9a5300,0xb56100,0xcb6d00,0xde7800,0xef8100,0xff8900,
//		0x000000,0x75004b,0x9a0064,0xb50075,0xcb0084,0xde0090,0xef009b,0xff00a5,
//		0x000000,0x75006d,0x9a0090,0xb500a9,0xcb00be,0xde00cf,0xef00df,0xff00ed,
//		0x000000,0x650075,0x86009a,0x9d00b5,0xb100cb,0xc100de,0xd000ef,0xdd00ff,
//		0x000000,0x520075,0x6c009a,0x7f00b5,0x8f00cb,0x9c00de,0xa800ef,0xb300ff,
//		0x000000,0x380075,0x4a009a,0x5700b5,0x6200cb,0x6b00de,0x7300ef,0x7b00ff,
//		0x000000,0x003375,0x00439a,0x004fb5,0x0059cb,0x0061de,0x0069ef,0x006fff,
//		0x000000,0x005275,0x006d9a,0x0080b5,0x008fcb,0x009dde,0x00a9ef,0x00b4ff,
//		0x000000,0x006a75,0x008c9a,0x00a5b5,0x00b9cb,0x00cade,0x00d9ef,0x00e7ff,
//		0x000000,0x007563,0x009a83,0x00b59a,0x00cbac,0x00debd,0x00efcb,0x00ffd8,
//		0x000000,0x007538,0x009a4a,0x00b557,0x00cb61,0x00de6a,0x00ef72,0x00ff7a,
//		0x000000,0x477500,0x5e9a00,0x6fb500,0x7dcb00,0x88de00,0x93ef00,0x9cff00,
//		0x000000,0x5e7500,0x7c9a00,0x92b500,0xa4cb00,0xb3de00,0xc1ef00,0xcdff00
//	} ;
//	for (int i=0 ; i<128 ; i++)
//	{
//		fPalette[i*2].red = (palette[i] & 0x00ff0000) >> 16;
//		fPalette[i*2].green = (palette[i] & 0x0000ff00) >> 8;
//		fPalette[i*2].blue = (palette[i] & 0x000000ff);
//		fPalette[i*2].alpha = 255;
//		fPalette[i*2+1] = fPalette[i*2];
//	}

	// Switch the console ON !
	fEvent.set(Event::ConsoleOn, 1);
	fEvent.set(Event::ConsoleOff, 0);

	// Get the desired width and height of the display
	fWidth = fConsole->mediaSource().width();
	fHeight = fConsole->mediaSource().height();	

	delete [] fConsummerFrameBuffer[0]; fConsummerFrameBuffer[0] = NULL;
	delete [] fConsummerFrameBuffer[1]; fConsummerFrameBuffer[1] = NULL;
	delete [] fProducerFrameBuffer; fProducerFrameBuffer = NULL;

	fCurrentFrameBufferIndex = 0;
	const int frame_buffer_size = fWidth*fHeight;
	fConsummerFrameBuffer[0] = new uint8[frame_buffer_size];
	fConsummerFrameBuffer[1] = new uint8[frame_buffer_size];
	fProducerFrameBuffer = new uint8[frame_buffer_size];
	memset(fConsummerFrameBuffer[0], 0, frame_buffer_size);
	memset(fConsummerFrameBuffer[1], 0, frame_buffer_size);
	memset(fProducerFrameBuffer, 0, frame_buffer_size);
	fDrawSemID = create_sem(1, "STella_draw_sem");

	delete [] fRectangles; fRectangles = NULL;
	fRectangles = new rectangle_t[frame_buffer_size];

	// Get informations on the game
	get_game_info();

	// Set the controllers
	BString leftController(fConsole->properties().get("Controller.Left").c_str());
	BString rightController(fConsole->properties().get("Controller.Right").c_str());

	if (leftController == "Booster-Grip")	SetLeftController(BOOSTER_GRIP);
	else if (leftController == "Driving")	SetLeftController(DRIVING);
	else if (leftController == "Keyboard")	SetLeftController(KEYBOARD);
	else if (leftController == "Paddles")	SetLeftController(PADDLE);
	else if (leftController == "Lightgun")	SetLeftController(LIGHTGUN);
	else 									SetLeftController(JOYSTICK);

	if (rightController == "Booster-Grip")	SetRightController(BOOSTER_GRIP);
	else if (rightController == "Driving")	SetRightController(DRIVING);
	else if (rightController == "Keyboard")	SetRightController(KEYBOARD);
	else if (rightController == "Paddles")	SetRightController(PADDLE);
	else 									SetRightController(JOYSTICK);

	// Set frame rate
	fFrameDuration = 1000000/60;
	if ((fGameInfo.format == "PAL") || (fGameInfo.format == "SECAM"))
		fFrameDuration = 1000000/50;
	
	// spawn threads (the draw thread must have a lower priority)
	fQuitIndicator = false;
	fPauseIndicator = true; // ResumeGame() will be called later

	// The computing thread must have a priority greater than the display thread
	fComputeThreadID = spawn_thread(_ComputeThread, "STella_compute_thread", B_URGENT_DISPLAY_PRIORITY, this);
	fDrawThreadID = spawn_thread(_DrawThread, "STella_draw_thread", B_DISPLAY_PRIORITY, this);

	// No error
	return B_OK;
}


//  Setup the properties set by loading builtin defaults and then a
//  set of user specific ones from the file "stella.pro"
//  @param set The properties set to setup

void STellaView::setupProperties(PropertiesSet& set)
{
	// Try to load the file "stella.pro"
	BPath path;
	find_directory(B_COMMON_ETC_DIRECTORY, &path);
	path.Append("stella.pro");
	string filename = path.Path();
	
	// See if we can open the file $HOME/.stella.pro
	ifstream stream(filename.c_str());
	if (stream)
	{ // File was opened so load properties from it
		set.load(stream, &Console::defaultProperties());
	}
	else
	{ // Couldn't open the file so use the builtin properties file
		strstream builtin;
		for (const char** p = defaultPropertiesFile(); *p != 0; ++p)
			builtin << *p << endl;		
		set.load(builtin, &Console::defaultProperties());
	}
}

void STellaView::MessageReceived(BMessage *m)
{
	switch(m->what)
	{
		case B_SIMPLE_DATA:		dropped_files(m);	break;
		case B_REFS_RECEIVED:	RefsReceived(m);	break;
		default:
			return BView::MessageReceived(m);
	}
}

void STellaView::RefsReceived(BMessage *msg)
{
	// Process refs
	uint32 type; 
	int32 count; 
	entry_ref ref;
	if (msg->GetInfo("refs", &type, &count) != B_OK)	return;
	if (type != B_REF_TYPE)								return;
	if (msg->FindRef("refs", &ref) != B_OK )			return;
	be_roster->AddToRecentDocuments(&ref, STellaApplication::fApplicationSignature);
	BPath path;
	BEntry entry(&ref);
	entry.GetPath(&path);	// Get a pointer to the file which contains the cartridge ROM
	InsertCartridge(path.Path());
	if (Window()->IsActive())
		ResumeGame();
	Window()->PostMessage(STellaWindow::MSG_GAME_STARTED);
}


void STellaView::dropped_files(const BMessage *msg)
{
	uint32 type; 
	int32 count; 
	entry_ref ref;

	if (msg->GetInfo("refs", &type, &count) != B_OK)
		return;
	
	if (type != B_REF_TYPE)
		return;

	BMessage refs(*msg);
	refs.what = B_REFS_RECEIVED;
	RefsReceived(&refs);
}

void STellaView::MouseDown(BPoint)
{
	if (fLeftController == PADDLE)
	{
		fEvent.set(Event::PaddleZeroFire, 1);
		fEvent.set(Event::PaddleOneFire, 1);
	}
	else if (fLeftController == LIGHTGUN)
	{
		fEvent.set(Event::LightGunZeroFire, 1);
	}

	if (fRightController == PADDLE)
	{
		fEvent.set(Event::PaddleTwoFire, 1);
		fEvent.set(Event::PaddleThreeFire, 1);
	}
}

void STellaView::MouseUp(BPoint)
{
	if (fLeftController == PADDLE)
	{
		fEvent.set(Event::PaddleZeroFire, 0);
		fEvent.set(Event::PaddleOneFire, 0);
	}
	else if (fLeftController == LIGHTGUN)
	{
		fEvent.set(Event::LightGunZeroFire, 0);
	}

	if (fRightController == PADDLE)
	{
		fEvent.set(Event::PaddleTwoFire, 0);
		fEvent.set(Event::PaddleThreeFire, 0);
	}
}

void STellaView::MouseMoved(BPoint point, uint32, const BMessage *)
{
	if (fLeftController == PADDLE)
	{
		const float r = 2.0f * (0.5f - point.x/Bounds().Width());
		uint32 resistance = (uint32)(520000.0f + r * 350000.0f);
    	fEvent.set(Event::PaddleZeroResistance, resistance);
    	fEvent.set(Event::PaddleOneResistance, resistance);
	}
	else if (fLeftController == LIGHTGUN)
	{
		uint32 x = ((uint32)point.x * fWidth)/Bounds().IntegerWidth();
		uint32 y = ((uint32)point.y * fHeight)/Bounds().IntegerHeight();
		fEvent.set(Event::LightGunZeroX, x);
		fEvent.set(Event::LightGunZeroY, y);
	}

	if (fRightController == PADDLE)
	{
		const float r = 2.0f * (0.5f - point.x/Bounds().Width());
		uint32 resistance = (uint32)(520000.0f + r * 350000.0f);
    	fEvent.set(Event::PaddleTwoResistance, resistance);
    	fEvent.set(Event::PaddleThreeResistance, resistance);
	}
}

void STellaView::KeyDown(const char *bytes, int32 numBytes)
{
	handle_key(bytes, numBytes, 1);
	BView::KeyDown(bytes, numBytes);
}

void STellaView::KeyUp(const char *bytes, int32 numBytes)
{
	handle_key(bytes, numBytes, 0);
	BView::KeyUp(bytes, numBytes);
}


void STellaView::handle_key(const char *bytes, int32, int32 value)
{
	Event& event = fEvent;
	BMessage *m = Window()->CurrentMessage();
	int32 key = m->FindInt32("key");

	if (m->HasInt32("be:key_repeat") == true)
		return; // We don't want repeated keys

	if (bytes[0] != B_FUNCTION_KEY)
	{ 
		// Left Controller
		if ((fLeftController == JOYSTICK) || (fLeftController == BOOSTER_GRIP))
		{
			if ((key == 40) || (bytes[0] == B_UP_ARROW))			event.set(Event::JoystickZeroUp, value);	// Left UP
			else if ((key == 60) || (bytes[0] == B_LEFT_ARROW))		event.set(Event::JoystickZeroLeft, value);	// Left LEFT
			else if ((key == 61) || (bytes[0] == B_DOWN_ARROW))		event.set(Event::JoystickZeroDown, value);	// Left DOWN
			else if ((key == 62) || (bytes[0] == B_RIGHT_ARROW))	event.set(Event::JoystickZeroRight, value);	// Left RIGHT
			else if ((bytes[0] == B_TAB) || (bytes[0] == B_SPACE))	event.set(Event::JoystickZeroFire, value);	// Left FIRE
			else if (fLeftController == BOOSTER_GRIP)
			{	if ((key == 18) || (key == 76))				event.set(Event::BoosterGripZeroTrigger, value);	// Left Trigger
				else if ((key == 19) || (key == 77))		event.set(Event::BoosterGripZeroBooster, value);	// Left Booster
			}
		}
		else if (fLeftController == DRIVING)
		{
			if ((bytes[0] == B_TAB) || (bytes[0] == B_SPACE))
			{	event.set(Event::JoystickZeroFire, value);	// Left FIRE
				event.set(Event::DrivingZeroFire, value);
			}
			else if ((key == 60) || (bytes[0] == B_LEFT_ARROW))
			{	event.set(Event::JoystickZeroLeft, value);	// Left LEFT
			}
			else if ((key == 62) || (bytes[0] == B_RIGHT_ARROW))
			{	event.set(Event::JoystickZeroRight, value);	// Left RIGHT
			}
		}
		else if (fLeftController == PADDLE)
		{
			if ((bytes[0] == B_TAB) || (bytes[0] == B_SPACE))	event.set(Event::PaddleZeroFire, value);	// Left FIRE
		}
		else if (fLeftController == LIGHTGUN)
		{
			if ((bytes[0] == B_TAB) || (bytes[0] == B_SPACE))
				event.set(Event::LightGunZeroFire, value);	// Left FIRE
		}
		else if (fLeftController == KEYBOARD)
		{
			switch (key)
			{
				case 18:	event.set(Event::KeyboardZero1, value);		break;
				case 19:	event.set(Event::KeyboardZero2, value);		break;
				case 20:	event.set(Event::KeyboardZero3, value);		break;
				case 39:	event.set(Event::KeyboardZero4, value);		break;
				case 40:	event.set(Event::KeyboardZero5, value);		break;
				case 41:	event.set(Event::KeyboardZero6, value);		break;
				case 60:	event.set(Event::KeyboardZero7, value);		break;
				case 61:	event.set(Event::KeyboardZero8, value);		break;
				case 62:	event.set(Event::KeyboardZero9, value);		break;
				case 76:	event.set(Event::KeyboardZeroStar, value);	break;
				case 77:	event.set(Event::KeyboardZero0, value);		break;
				case 78:	event.set(Event::KeyboardZeroPound, value);	break;
			}
		}
	
		// Right Controller
		if ((fRightController == JOYSTICK) || (fRightController == BOOSTER_GRIP))
		{
			if (key == 47)			event.set(Event::JoystickOneUp, value);		// Right UP
			else if (key == 67)		event.set(Event::JoystickOneLeft, value);	// Right LEFT
			else if (key == 68)		event.set(Event::JoystickOneDown, value);	// Right DOWN
			else if (key == 69)		event.set(Event::JoystickOneRight, value);	// Right RIGHT
			else if (key == 66)		event.set(Event::JoystickOneFire, value);	// Right FIRE
			else if (fRightController == BOOSTER_GRIP)
			{	if (key == 81)		event.set(Event::BoosterGripOneTrigger, value);	// Left Trigger
				else if (key == 82)	event.set(Event::BoosterGripOneBooster, value);	// Left Booster
			}
		}
		else if (fRightController == DRIVING)
		{
			if (key == 66)
			{	event.set(Event::JoystickOneFire, value);	// Right FIRE
				event.set(Event::DrivingOneFire, value);
			}
			else if (key == 67)
			{	event.set(Event::JoystickOneLeft, value);	// Right LEFT
			}
			else if (key == 69)
			{	event.set(Event::JoystickOneRight, value);	// Right RIGHT
			}
		}
		else if (fRightController == PADDLE)
		{
			if (key == 66)		event.set(Event::PaddleZeroFire, value);	// Right FIRE
		}
		else if (fRightController == KEYBOARD)
		{
			switch (key)
			{
				case 25:	event.set(Event::KeyboardOne1, value);		break;
				case 26:	event.set(Event::KeyboardOne2, value);		break;
				case 27:	event.set(Event::KeyboardOne3, value);		break;
				case 46:	event.set(Event::KeyboardOne4, value);		break;
				case 47:	event.set(Event::KeyboardOne5, value);		break;
				case 48:	event.set(Event::KeyboardOne6, value);		break;
				case 67:	event.set(Event::KeyboardOne7, value);		break;
				case 68:	event.set(Event::KeyboardOne8, value);		break;
				case 69:	event.set(Event::KeyboardOne9, value);		break;
				case 83:	event.set(Event::KeyboardOneStar, value);	break;
				case 84:	event.set(Event::KeyboardOne0, value);		break;
				case 85:	event.set(Event::KeyboardOnePound, value);	break;
			}
		}
	}
	else	// Function Key
	{
		switch (key)
		{
			case B_F1_KEY:	// Select game
				event.set(Event::ConsoleSelect, value);
				break;
			case B_F2_KEY:	// Game Reset
				event.set(Event::ConsoleReset, value);
				break;
		}
		if (value == 1)
		{ // On KeyDown only
			switch (key)
			{
				case B_F3_KEY:	// Color TV
					event.set(Event::ConsoleColor, 		1);
					event.set(Event::ConsoleBlackWhite, 0);
					break;
				case B_F4_KEY:	// Black & White TV
					event.set(Event::ConsoleColor, 		0);
					event.set(Event::ConsoleBlackWhite, 1);
					break;	
				case B_F5_KEY: // Left Player Difficulty B
					event.set(Event::ConsoleLeftDifficultyA, 0);
					event.set(Event::ConsoleLeftDifficultyB, 1);
					break;
				case B_F6_KEY: // Left Player Difficulty A
					event.set(Event::ConsoleLeftDifficultyA, 1);
					event.set(Event::ConsoleLeftDifficultyB, 0);
					break;
				case B_F7_KEY: // Right Player Difficulty B
					event.set(Event::ConsoleRightDifficultyA, 0);
					event.set(Event::ConsoleRightDifficultyB, 1);
					break;
				case B_F8_KEY: // Right Player Difficulty A
					event.set(Event::ConsoleRightDifficultyA, 1);
					event.set(Event::ConsoleRightDifficultyB, 0);
					break;
				case B_F12_KEY:
					Window()->PostMessage(STellaWindow::MSG_FULLSCREEN);
					break;
			}
		}
	}
}

void STellaView::AttachedToWindow()
{
	SetViewColor(B_TRANSPARENT_32_BIT);
}

void STellaView::SetFocusView(MFocusView *view)
{
	fFocusView = view;
}

void STellaView::MakeFocus(bool focus)
{
	if ((fFocusView) && (Flags() & B_NAVIGABLE))
	{ // Tell the focus view that the focus changed
		if (fFocusView->LockLooper())
		{
			fFocusView->FocusChanging(focus);
			fFocusView->UnlockLooper();
		}
	}
	BView::MakeFocus(focus);
}

void STellaView::PauseGame()
{
	if (fPauseIndicator == false)
	{
		fSound.mute(true);
		fPauseIndicator = true;
	}
}

void STellaView::ResumeGame()
{
	if (fPauseIndicator == true)
	{
		fNumberOfFrames = 0;
		fTimeFrameRate = system_time();
		fPauseIndicator = false;
		resume_thread(fDrawThreadID);
		resume_thread(fComputeThreadID);
		fSound.mute(false);
	}
}

void STellaView::Draw(BRect frame)
{
	// We must redraw the whole thing

	if (fConsole == NULL)
	{ // Nothing to do if there is no game
		SetHighColor(0,0,0);
		FillRect(frame);
		return;
	}

	// Draw FPS
	DrawFPS();

	// We must redraw the whole screen, here.
	if ((fDirectWindowMode == false) || (fConnected == false))
	{
		UpdateScreen(true);
		DrawScreen(false);
	}
	else
	{
		if (Lock())
		{ // Update the screen
			fEventDraw = true;
			if (fEventDirectDraw == true)
			{	
				fEventDirectDraw = false;
				fEventDraw = false;
				UpdateScreen(true);
				DrawScreen(true);
			}
			Unlock();
		}
	}
}

void STellaView::FrameResized(float w, float h)
{
	BView::FrameResized(w, h);
}

void STellaView::DrawFPS()
{
	if (fShowFPS)
	{
		BRect r;
		const float f = (Bounds().Width()+1.0f) / (fWidth*2);
		SetHighColor(0,0,0);
		SetLowColor(HighColor());
		r.Set(Bounds().left, fHeight*f, Bounds().right, Bounds().bottom);
		FillRect(r);
		SetHighColor(255,255,255);
		BString fps("FPS: ");
		fps << fFramePerSecond;
		DrawString(fps.String(), BPoint(8, fHeight*f+fFpsDisplayHeight-4.0f));
	}
}

void STellaView::DrawScreen(bool directmode)
{
	BRect r;
	if (directmode == false)
	{
		const float f = (Bounds().Width()+1.0f) / (fWidth*2);	
		for (int i=0 ; i<fNbRectangles ; i++)
		{ // Draw all the rectangles
			r = fRectangles[i].r;
			r.left *= f; 
			r.top *= f; 
			(r.right *= f)--; 
			(r.bottom *= f)--;
			const rgb_color c = (fPauseIndicator) ? (tint_color(fRectangles[i].c, 1.3f)) : (fRectangles[i].c);
			SetHighColor(c);
			FillRect(r);
		}
		Window()->Flush();
	}
	else
	{
		const float f = (float)(fDirectWidth) / (float)(fWidth*2);
		for (int i=0 ; i<fNbRectangles ; i++)
		{ // Draw all the rectangles
			r = fRectangles[i].r;
			r.left *= f; 
			r.top *= f; 
			(r.right *= f)--; 
			(r.bottom *= f)--;
			const rgb_color c = (fPauseIndicator) ? (tint_color(fRectangles[i].c, 1.3f)) : (fRectangles[i].c);
			DirectFillRect(r, c);
		}
	}
}

void STellaView::UpdateScreen(bool redraw_entire_frame)
{
	// When called from Draw(), don't swap the frame buffer
	if (redraw_entire_frame == false)
	{
		fCurrentFrameBufferIndex = 1-fCurrentFrameBufferIndex;
	
		// Lock the drawing thread while we copy the data for it
		if (acquire_sem(fDrawSemID) != B_OK)
			return;
		
		// Copy the data
		memcpy(fConsummerFrameBuffer[fCurrentFrameBufferIndex], fProducerFrameBuffer, fWidth*fHeight);
	
		// Unlock the drawing thread
		if (release_sem(fDrawSemID) != B_OK)
			return;
	}
	
	// Get the frame buffer
	uInt8* currentFrame = fConsummerFrameBuffer[fCurrentFrameBufferIndex];
	uInt8* previousFrame = fConsummerFrameBuffer[1-fCurrentFrameBufferIndex];

	// Alias the fNbRectangles variable
	int& rindex = fNbRectangles;
	rindex = 0;

	struct Rectangle
	{
		int color;
		int x, y, width, height;
	} rectangles[2][160];
	
	// This array represents the rectangles that need displaying
	// on the current scanline we're processing
	Rectangle* currentRectangles = rectangles[0];
	
	// This array represents the rectangles that are still active
	// from the previous scanlines we have processed
	Rectangle* activeRectangles = rectangles[1];
	
	// Indicates the number of active rectangles
	int activeCount = 0;

	// This update procedure requires fWidth to be a multiple of four.
	// This is validated when the properties are loaded.
	for (int y=0; y<(int)fHeight ; y++)
	{
		// Indicates the number of current rectangles
		int currentCount = 0;
		
		// Look at four pixels at a time to see if anything has changed
		const uInt32 *current = (const uInt32 *)currentFrame;
		const uInt32 *previous = (const uInt32 *)previousFrame;
		
		for (int x=0 ; x<(int)fWidth ; x+=4, current++, previous++)
		{
			// Has something changed in this set of four pixels?
			uInt32 lc, lp;
			if (((lc=*current) != (lp=*previous)) || redraw_entire_frame)
			{
				// Look at each of the bytes that make up the uInt32
				lp = B_HOST_TO_LENDIAN_INT32(lp);
				lc = B_HOST_TO_LENDIAN_INT32(lc);

				for (int i=0; i<4; i++)
				{
					const int p = lp & 0xFF;
					const int c = lc & 0xFF;
					lp>>=8;
					lc>>=8;
					
					// See if this pixel has changed
					if (redraw_entire_frame || (c != p))
					{
						// Can we extend a rectangle or do we have to create a new one?
						if ((currentCount != 0) &&
							(currentRectangles[currentCount - 1].color == c) &&
							((currentRectangles[currentCount - 1].x + currentRectangles[currentCount - 1].width) == (x + i)))
						{
							currentRectangles[currentCount - 1].width++;
						}
						else
						{
							currentRectangles[currentCount].x = x + i;
							currentRectangles[currentCount].y = y;
							currentRectangles[currentCount].width = 1;
							currentRectangles[currentCount].height = 1;
							currentRectangles[currentCount].color = c;
							currentCount++;
						}
					}
				}
			}
		}
		
		// Merge the active and current rectangles flushing any that are of no use
		int activeIndex = 0;
		for (int t=0; (t<currentCount) && (activeIndex<activeCount) ; )
		{
			Rectangle& current = currentRectangles[t];
			Rectangle& active = activeRectangles[activeIndex];

			if ((current.x == active.x) &&
				(current.width == active.width) &&
				(current.color == active.color))
			{ // Can we merge the current rectangle with an active one?
				current.y = active.y;
				current.height = active.height + 1;
				activeIndex++;
				t++;
			}
			else 
			{ // Can't merge current and active rectangles
				if (current.x+current.width > active.x)
				{ // Is it possible for this active rectangle to be merged?
					// No - Flush the active rectangle
					fRectangles[rindex].r.right = (fRectangles[rindex].r.left = active.x * 2) + active.width * 2;
					fRectangles[rindex].r.bottom = (fRectangles[rindex].r.top = active.y) + active.height;				
					fRectangles[rindex].c = fPalette[active.color];
					rindex++;
					activeIndex++;
				}

				if (current.x+current.width <= active.x+active.width)
				{
					t++;
				}
			}
		}

		// Flush any remaining active rectangles
		for (int s=activeIndex ; s<activeCount ; s++)
		{
			Rectangle& active = activeRectangles[s];	
			fRectangles[rindex].r.right = (fRectangles[rindex].r.left = active.x * 2) + active.width * 2;
			fRectangles[rindex].r.bottom = (fRectangles[rindex].r.top = active.y) + active.height;				
			fRectangles[rindex].c = fPalette[active.color];
			rindex++;
		}
		
		// We can now make the current rectangles into the active rectangles
		Rectangle* tmp = currentRectangles;
		currentRectangles = activeRectangles;
		activeRectangles = tmp;
		activeCount = currentCount;
		
		currentFrame += fWidth;
		previousFrame += fWidth;
	}
	
	// Flush any rectangles that are still active
	for (int t=0 ; t<activeCount ; t++)
	{
		Rectangle& active = activeRectangles[t];
		fRectangles[rindex].r.right = (fRectangles[rindex].r.left = active.x * 2) + active.width * 2;
		fRectangles[rindex].r.bottom = (fRectangles[rindex].r.top = active.y) + active.height;				
		fRectangles[rindex].c = fPalette[active.color];
		rindex++;
	}
}



void STellaView::ShowFps(bool show)
{
	fShowFPS = show;
}

bool STellaView::IsFpsShowing()
{
	return fShowFPS;
}


int32 STellaView::DrawThread(void)
{
	// Make a full invalidate the 1st time
	LockLooper();
	Draw(Bounds());
	UnlockLooper();

	bigtime_t wait_until = 0;
	fTimeFrameRate = system_time();
	fNumberOfFrames = 0;

	while (fQuitIndicator == false)
	{
		if (fPauseIndicator == true)
		{
			const bool currentMode = fDirectWindowMode;
			if ((currentMode ? (Lock()) : (LockLooper())) == true)
			{ // redraw the screen in pause mode
				UpdateScreen(true);
				DrawScreen(currentMode);
				(currentMode ? (Unlock()) : (UnlockLooper()));
			}
			suspend_thread(find_thread(NULL));
			if ((currentMode ? (Lock()) : (LockLooper())) == true)
			{ // redraw the screen in normal mode
				UpdateScreen(true);
				DrawScreen(currentMode);
				(currentMode ? (Unlock()) : (UnlockLooper()));
			}
		}
	
		// Wait a little
		snooze_until(wait_until, B_SYSTEM_TIMEBASE);
		wait_until = system_time() + (bigtime_t)fFrameDuration;


		if (fDirectWindowMode == false)
		{ // We have to lock the window, because BView drawing will occur
			if (LockLooper() == true)
			{ // Update the screen
				UpdateScreen(false);
				DrawScreen(false);
				UnlockLooper();
			}
		}
		else
		{ // We must lock the Direct Engine, because DirectConnected can also call it.
			// However, we don't need to lock the window, because DirectWindow mecanism has nothing to do with that.
			if (Lock())
			{ // Update the screen
				if (fConnected == true)
				{ // We are connected, update the screen
					UpdateScreen(false);
					DrawScreen(true);	
				}
				Unlock();
			}
		}

		if (LockLooper() == true)
		{ // Draw the FPS
			DrawFPS();
			Sync();
			UnlockLooper();
		}

		// Calculate the frame rate
		fFramePerSecond = (fNumberOfFrames++ / ((system_time()-fTimeFrameRate)/1000000.0f));
	}
	return B_OK;
}


int32 STellaView::ComputeThread(void)
{
	MediaSource& mediaSource = fConsole->mediaSource();
	const int size = fWidth*fHeight;	
	bigtime_t wait_until = 0;

	while (fQuitIndicator == false)
	{	
		if (fPauseIndicator == true)
			suspend_thread(find_thread(NULL));

		// Wait 1/60th second
		snooze_until(wait_until, B_SYSTEM_TIMEBASE);
		wait_until = system_time() + (bigtime_t)fFrameDuration;

		// Draw the frame and handle events
		fConsole->mediaSource().update();

		// Lock the drawing thread while we compute the data
		if (acquire_sem(fDrawSemID) != B_OK)
			break;

		memcpy(fProducerFrameBuffer, mediaSource.currentFrameBuffer(), size);

		// Unlock the drawing thread, because the data is ready
		if (release_sem(fDrawSemID) != B_OK)
			break;
	}

	return B_OK;
}

void STellaView::GetPreferredSize(float *width, float *height)
{ // Calculates the height for STellaView width
	const float dh = (fShowFPS) ? fFpsDisplayHeight : (0.0f);
	const float hw = (float)fHeight/((float)fWidth*2.0f);
	*width = Bounds().Width();
	*height = Bounds().Width()*hw + dh;
}


void STellaView::ResizeToConsole(float f)
{
	float w = fConsole->mediaSource().width() * 2.0f * f;
	float h = fConsole->mediaSource().height() * f;
	ResizeTo(w-1.0f, h-1.0f); // Resize the view to the desired width (height is not important)
}

void STellaView::ResizeToFitWindow()
{
	// Center the view in its parent
	const float dh = (fShowFPS) ? (fFpsDisplayHeight) : (0.0f);
	const float w = Parent()->Bounds().Width()+1.0f;
	const float h = Parent()->Bounds().Height()+1.0f;
	const float hw = (float)fHeight/((float)fWidth*2.0f);	
	float width = w;
	float height = w*hw + dh;
	if (height > h)
	{
		height = h;
		width = (h-dh)/hw;
	}

	// Resize and move the view
	ResizeTo(width-1.0f, height-1.0f);
	MoveTo(0.5f*(w-width), 0.5f*(h-height));
}

status_t STellaView::GetGameInfo(game_info *gameInfo)
{
	if (fConsole == NULL)
		return B_ERROR;
	*gameInfo = fGameInfo;
	return B_OK;
}

void STellaView::get_game_info()
{
	fGameInfo.name 			= fConsole->properties().get("Cartridge.Name").c_str();
	fGameInfo.manufacturer	= fConsole->properties().get("Cartridge.Manufacturer").c_str();
	fGameInfo.rarity 		= fConsole->properties().get("Cartridge.Rarity").c_str();
	fGameInfo.note			= fConsole->properties().get("Cartridge.Note").c_str();
	fGameInfo.format		= fConsole->properties().get("Display.Format").c_str();
	fGameInfo.width 		= atoi(fConsole->properties().get("Display.Width").c_str());
	fGameInfo.height 		= atoi(fConsole->properties().get("Display.Height").c_str());
	if (fGameInfo.width == 0)	fGameInfo.width = 160;
	if (fGameInfo.height == 0)	fGameInfo.height = 210;

/*
"Cartridge.MD5"
"Cartridge.Name"
"Cartridge.Manufacturer"
"Cartridge.ModelNo"
"Cartridge.Rarity"
"Cartridge.Note"
"Cartridge.Type"
"Controller.Left"
"Controller.Right"
"Display.XStart"
"Display.YStart"
"Display.Width"
"Display.Height"
"Display.FrameRate"
"Display.Format"
"Console.LeftDifficulty"
"Console.RightDifficulty"
"Console.TelevisionType"
*/
}

// ********************************************************************

void STellaView::SetHOffset(int h)
{
	 fMenuBarHeight = h;
}

void STellaView::DirectConnected(direct_buffer_info *info)
{
	if (Lock())
	{
		uint32 state = info->buffer_state & (B_BUFFER_MOVED | B_BUFFER_RESET | B_BUFFER_RESIZED | B_CLIPPING_MODIFIED);

		switch(info->buffer_state & B_DIRECT_MODE_MASK)
		{
			case B_DIRECT_START:
				fConnected = true;

			case B_DIRECT_MODIFY:
				fWindowBits =	(uint32 *)(	(uint8 *)info->bits +
											(info->window_bounds.top * info->bytes_per_row) +
											(info->window_bounds.left * (info->bits_per_pixel/8)) );
		
				fBytesPerLine	= info->bytes_per_row;
				fNbBitsPerPixel	= info->bits_per_pixel;
				fColorSpace		= info->pixel_format;

				// copy the new clipping region
				fOldClipBounds = fClipBounds;
				fClipBounds.left	= info->clip_bounds.left	- info->window_bounds.left;
				fClipBounds.top		= info->clip_bounds.top		- info->window_bounds.top;
				fClipBounds.right	= info->clip_bounds.right	- info->window_bounds.left	+ 1;
				fClipBounds.bottom	= info->clip_bounds.bottom	- info->window_bounds.top	+ 1;
				fNumClipRects = info->clip_list_count;
				delete [] fClipList;
				fClipList = new clipping_rect[fNumClipRects];
				for (int i=0 ; i<fNumClipRects ; i++)
				{ // Save the clipping rects and adjust to the window's coordinate space
					fClipList[i].left	= info->clip_list[i].left	- info->window_bounds.left;
					fClipList[i].top	= info->clip_list[i].top	- info->window_bounds.top;
					fClipList[i].right	= info->clip_list[i].right	- info->window_bounds.left	+ 1;
					fClipList[i].bottom	= info->clip_list[i].bottom	- info->window_bounds.top	+ 1;
				}

				if ((fConsole) && (state & B_BUFFER_RESIZED))
				{ // Center the view in its parent
					const int dh = (int)((fShowFPS) ? (fFpsDisplayHeight) : (0));
					bool isfullscreen = ((BDirectWindow *)(Window()))->IsFullScreen();
					int dh2 = (isfullscreen ? 0 : fMenuBarHeight);
					const int w = info->window_bounds.right - info->window_bounds.left + 1;
					const int h = info->window_bounds.bottom - info->window_bounds.top + 1 - dh2;
					const float hw = fHeight / (fWidth * 2.0f);
					fDirectWidth = w;
					fDirectHeight = (int)((float)w*hw + dh);
					if (fDirectHeight > h)
					{
						fDirectHeight = h;
						fDirectWidth = (int)(((float)(h-dh))/hw);
					}
					fOriginX = (w-fDirectWidth) / 2;
					fOriginY = (h-fDirectHeight) / 2 + dh2;
				}

				fEventDirectDraw = true;
				if ((fDirectWindowMode == true) && (state != B_BUFFER_MOVED) && fEventDraw)
				{
					fEventDirectDraw = false;
					fEventDraw = false;
					if (state == (B_BUFFER_MOVED | B_CLIPPING_MODIFIED))
					{ // In this case, we should redraw only if we can see _more_ things on screen (old > new)
						if (fNumClipRects == 1)
						{ // This is an easy case, the more interresting
							if ((fOldClipBounds.left <= fClipBounds.left) &&
								(fOldClipBounds.right >= fClipBounds.right) &&
								(fOldClipBounds.top <= fClipBounds.top) &&
								(fOldClipBounds.bottom >= fClipBounds.bottom))
							{ // Less things visible, so we've nothing to do!						
								break;
							}
						}
					}
					UpdateScreen(true);
					DrawScreen(true);
				}

				break;

			case B_DIRECT_STOP:
				delete [] fClipList;
				fClipList = NULL;
				fConnected = false;
				break;
		}
		Unlock();
	}
}

void STellaView::DirectFillRect(BRect r, rgb_color c)
{
	if (fConnected)
	{ // Verify that the Rect is valid and visible
		const int left		= max_c(fClipBounds.left,	fOriginX + (int)(r.left	+ 0.5f));
		const int top		= max_c(fClipBounds.top,	fOriginY + (int)(r.top	+ 0.5f));
		const int right		= min_c(fClipBounds.right,	fOriginX + (int)(r.right + 0.5f) + 1);
		const int bottom	= min_c(fClipBounds.bottom,	fOriginY + (int)(r.bottom+ 0.5f) + 1);

		if ((right > left) && (bottom > top))
		{ 	// Now find the color to poke in the frame buffer
			uint32 color32=0;
			uint16 color16=0;
			uint8 color8=0;
			switch (fColorSpace)
			{
				case B_RGB32:
				case B_RGBA32:	color32 = B_HOST_TO_LENDIAN_INT32((c.alpha<<24)|(c.red<<16)|(c.green<<8)|(c.blue));				break;
				case B_RGB16:	color16 = B_HOST_TO_LENDIAN_INT16((c.blue>>3)|((c.green<<3)&0x7e0)|((c.red<<8)&0xf800));		break;
				case B_RGB15:	color16 = B_HOST_TO_LENDIAN_INT16((c.blue>>3)|((c.green<<2)&0x3e0)|((c.red<<7)&0x7c00)|0x8000);	break;
				case B_RGBA15:	color16 = B_HOST_TO_LENDIAN_INT16((c.blue>>3)|((c.green<<2)&0x3e0)|((c.red<<7)&0x7c00));		break;
				case B_RGB32_BIG:
				case B_RGBA32_BIG:	color32 = B_HOST_TO_BENDIAN_INT32((c.alpha<<24)|(c.red<<16)|(c.green<<8)|(c.blue));				break;
				case B_RGB16_BIG:	color16 = B_HOST_TO_LENDIAN_INT16((c.blue>>3)|((c.green<<3)&0x7e0)|((c.red<<8)&0xf800));		break;
				case B_RGB15_BIG:	color16 = B_HOST_TO_LENDIAN_INT16((c.blue>>3)|((c.green<<2)&0x3e0)|((c.red<<7)&0x7c00)|0x8000);	break;
				case B_RGBA15_BIG:	color16 = B_HOST_TO_LENDIAN_INT16((c.blue>>3)|((c.green<<2)&0x3e0)|((c.red<<7)&0x7c00));		break;
				case B_CMAP8: {
					const color_map *map = system_colors();
					const uint16 val = (c.blue>>3)|((c.green<<2)&0x3e0)|((c.red<<7)&0x7c00);
					color8 = map->index_map[val];
					} break;
				default:
					return;
			}

			switch (fNbBitsPerPixel)
			{ // Select the right pixel format
				case 32:
					for (int i=0 ; i<fNumClipRects ; i++)
					{ // clip this rect
						int l = max_c(fClipList[i].left, left);
						int t = max_c(fClipList[i].top, top);
						int r = min_c(fClipList[i].right, right);
						int b = min_c(fClipList[i].bottom, bottom);
						const int h = b-t;
						const int w = r-l;
						if ((w>0) && (h>0))
							fill_rect_32(l, t, w, h, color32);
					}
					break;
				case 16:
					for (int i=0 ; i<fNumClipRects ; i++)
					{ // clip this rect
						int l = max_c(fClipList[i].left, left);
						int t = max_c(fClipList[i].top, top);
						int r = min_c(fClipList[i].right, right);
						int b = min_c(fClipList[i].bottom, bottom);
						const int h = b-t;
						const int w = r-l;
						if ((w>0) && (h>0))
							fill_rect_16(l, t, w, h, color16);
					}
					break;
				case 8:
					for (int i=0 ; i<fNumClipRects ; i++)
					{ // clip this rect
						int l = max_c(fClipList[i].left, left);
						int t = max_c(fClipList[i].top, top);
						int r = min_c(fClipList[i].right, right);
						int b = min_c(fClipList[i].bottom, bottom);
						const int h = b-t;
						const int w = r-l;
						if ((w>0) && (h>0))
							fill_rect_8(l, t, w, h, color8);
					}
					break;
			}
		}
	}
}


void STellaView::fill_rect_32(int x, int y, int w, int h, uint32 color)
{
	uint32 *frame = (uint32 *)((uint8 *)fWindowBits + y*fBytesPerLine + x*4);
	const int offset = fBytesPerLine/4 - w;
	while (h--)
	{
		int curw = w;
		while (curw--)
			*frame++ = color;
		frame += offset;
	}
}

void STellaView::fill_rect_16(int x, int y, int w, int h, uint16 color)
{
	uint16 *frame = (uint16 *)((uint8 *)fWindowBits + y*fBytesPerLine + x*2);
	const int offset = fBytesPerLine/2 - w;
	while (h--)
	{
		int curw = w;
		while (curw--)
			*frame++ = color;
		frame += offset;
	}
}

void STellaView::fill_rect_8(int x, int y, int w, int h, uint8 color)
{
	uint8 *frame = (uint8 *)fWindowBits + y*fBytesPerLine + x;
	if (w == fBytesPerLine)
	{ // Ultra rapid case
		memset(frame, color, w*h);
		return;
	}

	while (h--)
	{
		memset(frame, color, w);
		frame += fBytesPerLine;
	}
}

// ********************************************************************
#pragma mark -

STellaFilePanel::STellaFilePanel()
	: BFilePanel()
{
	LoadPanelDirectory();
	if (Window()->Lock())
	{
		Window()->SetTitle("Load a game");
		BView *background = Window()->ChildAt(0);

		float dh = -100.0f;

		BView *v;
		v = background->FindView("PoseView");	v->ResizeBy(0, dh);
		v = background->FindView("VScrollBar");	v->ResizeBy(0, dh);
		v = background->FindView("HScrollBar");	v->MoveBy(0, dh);
		v = background->FindView("CountVw");	v->MoveBy(0, dh);
		
		// The FocusView
		BRect r;
		r.left	= v->Frame().left;
		r.top	= v->Frame().bottom + 13.0f;
		r.right	= r.left + 159.0f + 4.0f;
		r.bottom= background->Bounds().bottom - 10.0f;
		r.InsetBy(-2, -2);
		MFocusView *focusview = new MFocusView(r, NULL, B_FOLLOW_BOTTOM | B_FOLLOW_LEFT);
		background->AddChild(focusview);

			// A BBox
			r.OffsetTo(B_ORIGIN);
			r.InsetBy(2, 2);
			BBox *b = new BBox(r, NULL, B_FOLLOW_LEFT | B_FOLLOW_TOP);
			focusview->AddChild(b);
	
				// A black background view
				r.OffsetTo(B_ORIGIN);
				r.InsetBy(2,2);
				v = new BView(r, "black", B_FOLLOW_LEFT | B_FOLLOW_TOP, B_WILL_DRAW);
				v->SetViewColor(0,0,0);
				b->AddChild(v);
	
					// STellaView
					r.OffsetTo(B_ORIGIN);
					fStellaView = new STellaView(r);
					fStellaView->SetFlags(fStellaView->Flags() | B_NAVIGABLE);
					fStellaView->SetResizingMode(B_FOLLOW_LEFT | B_FOLLOW_TOP);
					fStellaView->SetFocusView(focusview);
					v->AddChild(fStellaView);

			// The info view container
			r.left = focusview->Frame().right + 8.0f;
			r.top = background->FindView("CountVw")->Frame().bottom + 13.0f;
			r.right = background->FindView("VScrollBar")->Frame().right;
			r.bottom = background->FindView("default button")->Frame().top - 8.0f;
			b = new BBox(r, NULL, B_FOLLOW_LEFT_RIGHT | B_FOLLOW_BOTTOM, B_WILL_DRAW | B_FRAME_EVENTS);
			background->AddChild(b);
				
				// the info view
				r.OffsetTo(B_ORIGIN);
				r.InsetBy(4,4);
				fStellaInfoView = new STellaGameInfoView(r);
				b->AddChild(fStellaInfoView);

		float minw, maxw, minh, maxh;
		Window()->GetSizeLimits(&minw, &maxw, &minh, &maxh);
		Window()->SetSizeLimits(minw+8.0f, maxw, minh-dh, maxh);

		Window()->Unlock();
	}
}

STellaFilePanel::~STellaFilePanel()
{
	SavePanelDirectory();
}

void STellaFilePanel::SelectionChanged(void)
{
	entry_ref ref;
	Rewind();
	if (GetNextSelectedRef(&ref) == B_OK)
	{
		BEntry entry(&ref);
		if (entry.IsDirectory() == false)
		{
			if (fOldRef == ref)
				return;

			BPath path;
			entry.GetPath(&path);	// Get a pointer to the file which contains the cartridge ROM
			fStellaView->InsertCartridge(path.Path());
			fStellaView->ResizeToPreferred();
			fStellaView->MoveTo(	0.5f * (fStellaView->Parent()->Bounds().Width() - fStellaView->Bounds().Width()),			
									0.5f * (fStellaView->Parent()->Bounds().Height() - fStellaView->Bounds().Height()) );
			fStellaView->ResumeGame();
			game_info info;	
			if (fStellaView->GetGameInfo(&info) == B_OK)
				fStellaInfoView->Update(&info);
			fOldRef = ref;
		}
	}
}

void STellaFilePanel::WasHidden()
{
	fStellaView->RemoveCartridge();
}

void STellaFilePanel::LoadPanelDirectory()
{
	char directory[B_FILE_NAME_LENGTH];
	app_info appInfo;
	be_app->GetAppInfo(&appInfo);
	BNode node(&(appInfo.ref));
	if (node.ReadAttr("dir", B_STRING_TYPE, 0, directory, B_FILE_NAME_LENGTH) >= 0)
		SetPanelDirectory(directory);
}

void STellaFilePanel::SavePanelDirectory()
{
	entry_ref ref;
	GetPanelDirectory(&ref);
	BEntry entry(&ref);
	BPath directory;
	entry.GetPath(&directory);	
	app_info appInfo;
	be_app->GetAppInfo(&appInfo);
	BNode node(&(appInfo.ref));
	node.WriteAttr("dir", B_STRING_TYPE, 0, directory.Path(), strlen(directory.Path())+1);
}

// ********************************************************************
#pragma mark -


STellaGameInfoView::STellaGameInfoView(BRect r)
	: BView(r, "gameinfo", B_FOLLOW_ALL, B_WILL_DRAW | B_FRAME_EVENTS),
	fInitOk(false)
{
	fTruncatedStrings[0] = NULL;
	fInputTitles[0] =        "Title:  ";
	fInputTitles[1] = "Manufacturer:  ";
	fInputTitles[2] =       "Rarity:  ";
	fInputTitles[3] =     "Comments:  ";
	fInputTitles[4] =       "Format:  ";
	fInputTitles[5] =   "Resolution:  ";

	SetFont(be_plain_font);
	font_height fontHeight;
	GetFontHeight(&fontHeight);
	fTextHeight = fontHeight.ascent + fontHeight.descent + fontHeight.leading;
}

STellaGameInfoView::~STellaGameInfoView()
{
	free(fTruncatedStrings[0]);
}

void STellaGameInfoView::AttachedToWindow(void)
{
	SetViewColor(B_TRANSPARENT_32_BIT);
	font_height fontHeight;
	GetFontHeight(&fontHeight);
	const float totalHeight = 6*fTextHeight;
	fTextVPos = fontHeight.ascent + (fontHeight.leading + Bounds().Height() - totalHeight)*0.5f;
	fTextHPos = StringWidth(fInputTitles[1])+4.0f;
}

void STellaGameInfoView::Draw(BRect frame)
{
	if (Parent())	SetHighColor(Parent()->ViewColor());
	else			SetHighColor(ui_color(B_PANEL_BACKGROUND_COLOR));
	SetLowColor(HighColor());
	FillRect(frame);

	SetHighColor(0, 0, 220);
	for (int i=0 ; i<6 ; i++)
		DrawTitle(i, fInputTitles[i]);

	if (fInitOk)
	{
		SetHighColor(220,0,0);
		DrawText(0, fTruncatedStrings[0]);
		SetHighColor(0,0,0);
		for (int i=1 ; i<6 ; i++)
			DrawText(i, fTruncatedStrings[i]);
	}
}

void STellaGameInfoView::FrameResized(float, float)
{
	if (fInitOk)
	{
		const float maxWidth = Bounds().Width()-fTextHPos;
		BFont f;
		GetFont(&f);
		f.GetTruncatedStrings(fInputStrings, 6, B_TRUNCATE_END, maxWidth, fTruncatedStrings);
		BRect r = Bounds();
		r.left = fTextHPos;
		Invalidate(r);
		Window()->UpdateIfNeeded();
	}
}

void STellaGameInfoView::DrawTitle(int line, const char *s)
{
	MovePenTo(fTextHPos-StringWidth(s), fTextVPos+line*fTextHeight);
	DrawString(s);
}

void STellaGameInfoView::DrawText(int line, const char *s)
{
	MovePenTo(fTextHPos, fTextVPos+line*fTextHeight);
	DrawString(s);
}

void STellaGameInfoView::Update(const game_info *info)
{
	fGameInfo = *info;

	fResolution.SetTo(B_EMPTY_STRING);
	fResolution << fGameInfo.width << "x" << fGameInfo.height;
	fInputStrings[0] = fGameInfo.name.String();
	fInputStrings[1] = fGameInfo.manufacturer.String();
	fInputStrings[2] = fGameInfo.rarity.String();
	fInputStrings[3] = fGameInfo.note.String();
	fInputStrings[4] = fGameInfo.format.String();
	fInputStrings[5] = fResolution.String();

	free(fTruncatedStrings[0]);
	int bufferLength = 0;
	for (int i=0 ; i<6 ; i++)
		bufferLength += (strlen(fInputStrings[i])+3);
	char *tmpBuffer = (char *)malloc(bufferLength);
	for (int i=0, bufferLength=0 ; i<6 ; i++)
	{
		fTruncatedStrings[i] = tmpBuffer + bufferLength;
		bufferLength += (strlen(fInputStrings[i])+3);
	}

	float maxWidth = Bounds().Width()-fTextHPos;
	BFont f;
	GetFont(&f);
	f.GetTruncatedStrings(fInputStrings, 6, B_TRUNCATE_END, maxWidth, fTruncatedStrings);

	fInitOk = true;
	FrameResized(0, 0);
}

void STellaGameInfoView::GetPreferredSize(float *width, float *height)
{
	*height = (6+1) * fTextHeight;
	*width = (*height) * 3;
}


// ********************************************************************
#pragma mark -

MFocusView::MFocusView(BRect r, const char *name, uint32 resizingMode)
	: BView(r, name, resizingMode, B_WILL_DRAW),
	fIsFocus(false)
{
	fLowColor.red = 255;
	fLowColor.green = 255;
	fLowColor.blue = 255;
	fLowColor.alpha = 255;
}

MFocusView::~MFocusView()
{
}

void MFocusView::AttachedToWindow()
{
	if (Parent())
		fLowColor = Parent()->ViewColor();
	SetViewColor(B_TRANSPARENT_32_BIT);
}

void MFocusView::Draw(BRect)
{
	BRect r = Bounds();
	r.InsetBy(1,1);
	SetHighColor(fLowColor);
	FillRect(r);
	draw_focus(fIsFocus);
}

void MFocusView::WindowActivated(bool activated)
{
	fWindowActivated = activated;
	draw_focus(fIsFocus);
	Flush();
}

void MFocusView::FocusChanging(bool focus)
{
	fIsFocus = focus;
	draw_focus(fIsFocus);
	Flush();
}

void MFocusView::draw_focus(bool focus)
{
	SetHighColor(fLowColor);
	if ((focus) && (fWindowActivated))
		SetHighColor(keyboard_navigation_color());
	StrokeRect(Bounds());
}

